/*
 * Copyright (c) 2024-2025, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*!*****************************************************************************
 *  @file       PWMTimerWFF3.h
 *  @brief      PWM driver implementation for Wi-Fi F3 devices
 *
 *  # Overview #
 * The general PWM API should be used in application code, i.e. PWM_open()
 * should be used instead of PWMTimerWFF3_open(). The board file will define the device
 * specific config, and casting in the general API will ensure that the correct
 * device specific functions are called.
 *
 * # General Behavior #
 * Before using PWM on WFF3:
 *   - The Timer HW is configured and system dependencies (for example IOs,
 *     power, etc.) are set by calling PWM_open().
 *
 * # Error handling #
 * If unsupported arguments are provided to an API returning an error code, the
 * PWM configuration will *not* be updated and PWM will stay in the mode it
 * was already configured to.
 *
 * # Power Management #
 * The SimpleLink power management framework will try to put the device into the most
 * power efficient mode whenever possible. Please see the technical reference
 * manual for further details on each power mode.
 *
 *  The PWMTimerWFF3.h driver is not explicitly setting a power constraint when the
 *  PWM is running to prevent standby as this is assumed to be done in the
 *  underlying GPTimer driver.
 *  The following statements are valid:
 *    - After PWM_open(): The device is still allowed to enter Sleep. When the
 *                        device is active the underlying GPTimer peripheral will
 *                        be enabled and clocked.
 *    - After PWM_start(): The device can only go to Idle power mode since the
 *                         system clock is needed for PWM operation:
 *    - After PWM_stop():  Conditions are equal as for after PWM_open
 *    - After PWM_close(): The underlying GPTimer is turned off and the device
 *                         is allowed to go to sleep.
 *
 *  # Accuracy #
 *  The PWM output period and duty cycle are limited by the underlying timer.
 *  For GPT counter widths see @ref ti_drivers_GPTimerWFF3_PeripheralProperties
 *  "GPT peripheral properties".
 *
 *  The frequency of the underlying timer counter can be divided by configuring
 *  the preScalerDivision element of the @ref PWMTimerWFF3_HwAttrs struct.
 *  This configuration can be used to extend the PWM signal period but will
 *  decrease the accuracy.
 *  The minimum obtainable PWM signal frequency is dependent on the width of
 *  the counter for the GPT peripheral used and the configured preScalerDivision
 *  value. When using either the GPT0 or the GPT1 peripheral with the default
 *  preScalerDivision value of 1, the minimal frequency is
 *  40MHz / (2^32-1) = 0.0093Hz (107.374sec).
 *
 *  When using high output frequencies the duty cycle resolution is reduced
 *  correspondingly. For a 20MHz PWM only a 0%/50%/100% duty is available as
 *  the timer uses only counts 0 and 1.
 *  Similarly for a 10MHz period the duty cycle will be limited to a 12.5%
 *  resolution.
 *
 *  @note The PWM signals are generated by the GPT peripheral using the
 *  timer clock which is dependent on the system clock.
 *
 *  # Limitations #
 *  - The PWM output can currently not be synchronized with other PWM outputs.
 *  - The PWM driver does not support updating duty and period using DMA.
 *  - Attempts to change both the period and duty cycle at the same time
 *    for an active PWM signal, can cause pulses to be too long or short if
 *    the change is applied close to the end of the current counter cycle.
 *    This does not apply to changing only the period or only the duty cycle,
 *    as any of these separate changes will take effect on the next counter cycle.
 *  # PWM usage #
 *
 *  ## Basic PWM output ##
 *  The below example will output a 8MHz PWM signal with 50% duty cycle.
 *  @code
 *  PWM_Handle pwmHandle;
 *  PWM_Params params;
 *
 *  PWM_Params_init(&params);
 *  params.idleLevel      = PWM_IDLE_LOW;
 *  params.periodUnits    = PWM_PERIOD_HZ;
 *  params.periodValue    = 8e6;
 *  params.dutyUnits      = PWM_DUTY_FRACTION;
 *  params.dutyValue      = PWM_DUTY_FRACTION_MAX / 2;
 *
 *  pwmHandle = PWM_open(CONFIG_PWM0, &params);
 *  if(pwmHandle == NULL) {
 *    Log_error0("Failed to open PWM");
 *    Task_exit();
 *  }
 *  PWM_start(pwmHandle);
 *  @endcode
 *
 *
 *******************************************************************************
 */
#ifndef ti_drivers_pwm_PWMTimerWFF3__include
#define ti_drivers_pwm_PWMTimerWFF3__include

#include <stdint.h>
#include <stdbool.h>

#include <ti/drivers/PWM.h>
#include <ti/drivers/timer/GPTimerWFF3.h>

#ifdef __cplusplus
extern "C" {
#endif

/* PWM function table pointer */
extern const PWM_FxnTable PWMTimerWFF3_fxnTable;

/*!
 *  @brief  PWMTimerWFF3 Hardware attributes
 *
 *  These fields are used by the driver to set up the underlying GPTimer
 *  driver statically.
 */
typedef struct PWMTimerWFF3_HwAttrs
{
    uint8_t gpTimerInstance;    /*!< GPTimer unit index (0, 1) */
    uint16_t preScalerDivision; /*!< GPTimer prescaler divider */
} PWMTimerWFF3_HwAttrs;

/*!
 *  @brief  PWMTimerWFF3 Object
 *
 * These fields are used by the driver to store and modify PWM configuration
 * during run-time.
 * The application must not edit any member variables of this structure.
 * Appplications should also not access member variables of this structure
 * as backwards compatibility is not guaranteed.
 */
typedef struct PWMTimerWFF3_Object
{
    uint32_t periodValue;           /*!< Current period value in unit */
    uint32_t periodCounts;          /*!< Current period in raw timer counts */
    uint32_t dutyValue;             /*!< Current duty cycle value in unit */
    uint32_t dutyCounts;            /*!< Current duty in raw timer counts */
    GPTimerWFF3_Handle hTimer;      /*!< Handle to underlying GPTimer peripheral */
    PWM_Period_Units periodUnit;    /*!< Current period unit */
    PWM_Duty_Units dutyUnit;        /*!< Current duty cycle unit */
    GPTimerWFF3_ChannelNo chNumber; /*!< GPTimer peripheral channel number selected for PWM output signal */
    PWM_IdleLevel idleLevel;        /*!< PWM idle level when stopped / not started */
    bool isOpen;                    /*!< open flag used to check if PWM is opened */
    bool isRunning;                 /*!< running flag, set if the output is active */
} PWMTimerWFF3_Object;

#ifdef __cplusplus
}
#endif
#endif /* ti_drivers_pwm_PWMTimerWFF3__include */
