/*
 * Copyright (c) 2024-2025, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 *  ======== ECCParamsXXF3HSM.c ========
 *
 *  This file contains structure definitions for various ECC curves for use
 *  on CC27XX devices.
 */

#include <stdbool.h>
#include <stdint.h>
#include <string.h>

#include <ti/drivers/cryptoutils/ecc/ECCParams.h>
#include <ti/drivers/cryptoutils/cryptokey/CryptoKey.h>
#include <ti/drivers/cryptoutils/utils/CryptoUtils.h>

#define WORD_LENGTH_HSM 32

/*
 * NIST P256 (secp256r1) curve params in little endian format.
 * byte[0] is the least significant byte of the curve parameter.
 */
const ECC_NISTP256_Param ECC_NISTP256_generatorX = {
    .byte = {0x96, 0xc2, 0x98, 0xd8, 0x45, 0x39, 0xa1, 0xf4, 0xa0, 0x33, 0xeb, 0x2d, 0x81, 0x7d, 0x03, 0x77,
             0xf2, 0x40, 0xa4, 0x63, 0xe5, 0xe6, 0xbc, 0xf8, 0x47, 0x42, 0x2c, 0xe1, 0xf2, 0xd1, 0x17, 0x6b}};

const ECC_NISTP256_Param ECC_NISTP256_generatorY = {
    .byte = {0xf5, 0x51, 0xbf, 0x37, 0x68, 0x40, 0xb6, 0xcb, 0xce, 0x5e, 0x31, 0x6b, 0x57, 0x33, 0xce, 0x2b,
             0x16, 0x9e, 0x0f, 0x7c, 0x4a, 0xeb, 0xe7, 0x8e, 0x9b, 0x7f, 0x1a, 0xfe, 0xe2, 0x42, 0xe3, 0x4f}};

const ECC_NISTP256_Param ECC_NISTP256_prime = {
    .byte = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff}};

const ECC_NISTP256_Param ECC_NISTP256_a = {.byte = {0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                                    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff}};

const ECC_NISTP256_Param ECC_NISTP256_b = {.byte = {0x4b, 0x60, 0xd2, 0x27, 0x3e, 0x3c, 0xce, 0x3b, 0xf6, 0xb0, 0x53,
                                                    0xcc, 0xb0, 0x06, 0x1d, 0x65, 0xbc, 0x86, 0x98, 0x76, 0x55, 0xbd,
                                                    0xeb, 0xb3, 0xe7, 0x93, 0x3a, 0xaa, 0xd8, 0x35, 0xc6, 0x5a}};

const ECC_NISTP256_Param ECC_NISTP256_order = {
    .byte = {0x51, 0x25, 0x63, 0xfc, 0xc2, 0xca, 0xb9, 0xf3, 0x84, 0x9e, 0x17, 0xa7, 0xad, 0xfa, 0xe6, 0xbc,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff}};

/* Invert of square of Montgomery constant, k = 2^(256) mod p, p is the curve prime */
const ECC_NISTP256_Param ECC_NISTP256_k_mont = {
    .byte = {0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
             0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00}};

/* Converted to Montgomery domain by modular multiplication of ECC_NISTP256_a with ECC_NISTP224_k_mont */
const ECC_NISTP256_Param ECC_NISTP256_a_mont = {
    .byte = {0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x03, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xfc, 0xff, 0xff, 0xff}};

/* Converted to Montgomery domain by modular multiplication of ECC_NISTP256_b with ECC_NISTP224_k_mont */
const ECC_NISTP256_Param ECC_NISTP256_b_mont = {
    .byte = {0xdf, 0xbd, 0xc4, 0x29, 0x62, 0xdf, 0x9c, 0xd8, 0x90, 0x30, 0x84, 0x78, 0xcd, 0x05, 0xf0, 0xac,
             0xd6, 0x2e, 0x21, 0xf7, 0xab, 0x20, 0xa2, 0xe5, 0x34, 0x48, 0x87, 0x04, 0x1d, 0x06, 0x30, 0xdc}};

const ECCParams_CurveParams ECCParams_NISTP256 = {.curveType  = ECCParams_CURVE_TYPE_SHORT_WEIERSTRASS_AN3,
                                                  .length     = ECCParams_NISTP256_LENGTH,
                                                  .prime      = ECC_NISTP256_prime.byte,
                                                  .order      = ECC_NISTP256_order.byte,
                                                  .a          = ECC_NISTP256_a.byte,
                                                  .b          = ECC_NISTP256_b.byte,
                                                  .generatorX = ECC_NISTP256_generatorX.byte,
                                                  .generatorY = ECC_NISTP256_generatorY.byte,
                                                  .cofactor   = 1};

/*
 * NIST P192 curve params in little endian format.
 * byte[0] is the least significant byte of the curve parameter.
 */
const ECC_NISTP192_Param ECC_NISTP192_generatorX = {.byte = {0x12, 0x10, 0xff, 0x82, 0xfd, 0x0a, 0xff, 0xf4,
                                                             0x00, 0x88, 0xa1, 0x43, 0xeb, 0x20, 0xbf, 0x7c,
                                                             0xf6, 0x90, 0x30, 0xb0, 0x0e, 0xa8, 0x8d, 0x18}};

const ECC_NISTP192_Param ECC_NISTP192_generatorY = {.byte = {0x11, 0x48, 0x79, 0x1e, 0xa1, 0x77, 0xf9, 0x73,
                                                             0xd5, 0xcd, 0x24, 0x6b, 0xed, 0x11, 0x10, 0x63,
                                                             0x78, 0xda, 0xc8, 0xff, 0x95, 0x2b, 0x19, 0x07}};

const ECC_NISTP192_Param ECC_NISTP192_prime = {.byte = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                        0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

const ECC_NISTP192_Param ECC_NISTP192_a = {.byte = {0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                    0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

const ECC_NISTP192_Param ECC_NISTP192_b = {.byte = {0xb1, 0xb9, 0x46, 0xc1, 0xec, 0xde, 0xb8, 0xfe,
                                                    0x49, 0x30, 0x24, 0x72, 0xab, 0xe9, 0xa7, 0x0f,
                                                    0xe7, 0x80, 0x9c, 0xe5, 0x19, 0x05, 0x21, 0x64}};

const ECC_NISTP192_Param ECC_NISTP192_order = {.byte = {0x31, 0x28, 0xd2, 0xb4, 0xb1, 0xc9, 0x6b, 0x14,
                                                        0x36, 0xf8, 0xde, 0x99, 0xff, 0xff, 0xff, 0xff,
                                                        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

const ECCParams_CurveParams ECCParams_NISTP192 = {.curveType  = ECCParams_CURVE_TYPE_SHORT_WEIERSTRASS_AN3,
                                                  .length     = ECCParams_NISTP192_LENGTH,
                                                  .prime      = ECC_NISTP192_prime.byte,
                                                  .order      = ECC_NISTP192_order.byte,
                                                  .a          = ECC_NISTP192_a.byte,
                                                  .b          = ECC_NISTP192_b.byte,
                                                  .generatorX = ECC_NISTP192_generatorX.byte,
                                                  .generatorY = ECC_NISTP192_generatorY.byte,
                                                  .cofactor   = 1};

/*
 * NIST P224 curve params in little endian format.
 * byte[0] is the least significant byte of the curve parameter.
 */
const ECC_NISTP224_Param ECC_NISTP224_generatorX = {.byte = {0x21, 0x1d, 0x5c, 0x11, 0xd6, 0x80, 0x32, 0x34, 0x22, 0x11,
                                                             0xc2, 0x56, 0xd3, 0xc1, 0x03, 0x4a, 0xb9, 0x90, 0x13, 0x32,
                                                             0x7f, 0xbf, 0xb4, 0x6b, 0xbd, 0x0c, 0x0e, 0xb7}};

const ECC_NISTP224_Param ECC_NISTP224_generatorY = {.byte = {0x34, 0x7e, 0x00, 0x85, 0x99, 0x81, 0xd5, 0x44, 0x64, 0x47,
                                                             0x07, 0x5a, 0xa0, 0x75, 0x43, 0xcd, 0xe6, 0xdf, 0x22, 0x4c,
                                                             0xfb, 0x23, 0xf7, 0xb5, 0x88, 0x63, 0x37, 0xbd}};

const ECC_NISTP224_Param ECC_NISTP224_prime = {.byte = {0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                                        0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

const ECC_NISTP224_Param ECC_NISTP224_a = {.byte = {0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                    0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

const ECC_NISTP224_Param ECC_NISTP224_b = {.byte = {0xb4, 0xff, 0x55, 0x23, 0x43, 0x39, 0x0b, 0x27, 0xba, 0xd8,
                                                    0xbf, 0xd7, 0xb7, 0xb0, 0x44, 0x50, 0x56, 0x32, 0x41, 0xf5,
                                                    0xab, 0xb3, 0x04, 0x0c, 0x85, 0x0a, 0x05, 0xb4}};

const ECC_NISTP224_Param ECC_NISTP224_order = {.byte = {0x3d, 0x2a, 0x5c, 0x5c, 0x45, 0x29, 0xdd, 0x13, 0x3e, 0xf0,
                                                        0xb8, 0xe0, 0xa2, 0x16, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

const ECCParams_CurveParams ECCParams_NISTP224 = {.curveType  = ECCParams_CURVE_TYPE_SHORT_WEIERSTRASS_AN3,
                                                  .length     = ECCParams_NISTP224_LENGTH,
                                                  .prime      = ECC_NISTP224_prime.byte,
                                                  .order      = ECC_NISTP224_order.byte,
                                                  .a          = ECC_NISTP224_a.byte,
                                                  .b          = ECC_NISTP224_b.byte,
                                                  .generatorX = ECC_NISTP224_generatorX.byte,
                                                  .generatorY = ECC_NISTP224_generatorY.byte,
                                                  .cofactor   = 1};

/*
 * NIST P384 curve params in little endian format.
 * byte[0] is the least significant byte of the curve parameter.
 */
const ECC_NISTP384_Param ECC_NISTP384_generatorX = {
    .byte = {0xb7, 0x0a, 0x76, 0x72, 0x38, 0x5e, 0x54, 0x3a, 0x6c, 0x29, 0x55, 0xbf, 0x5d, 0xf2, 0x02, 0x55,
             0x38, 0x2a, 0x54, 0x82, 0xe0, 0x41, 0xf7, 0x59, 0x98, 0x9b, 0xa7, 0x8b, 0x62, 0x3b, 0x1d, 0x6e,
             0x74, 0xad, 0x20, 0xf3, 0x1e, 0xc7, 0xb1, 0x8e, 0x37, 0x05, 0x8b, 0xbe, 0x22, 0xca, 0x87, 0xaa}};

const ECC_NISTP384_Param ECC_NISTP384_generatorY = {
    .byte = {0x5f, 0x0e, 0xea, 0x90, 0x7c, 0x1d, 0x43, 0x7a, 0x9d, 0x81, 0x7e, 0x1d, 0xce, 0xb1, 0x60, 0x0a,
             0xc0, 0xb8, 0xf0, 0xb5, 0x13, 0x31, 0xda, 0xe9, 0x7c, 0x14, 0x9a, 0x28, 0xbd, 0x1d, 0xf4, 0xf8,
             0x29, 0xdc, 0x92, 0x92, 0xbf, 0x98, 0x9e, 0x5d, 0x6f, 0x2c, 0x26, 0x96, 0x4a, 0xde, 0x17, 0x36}};

const ECC_NISTP384_Param ECC_NISTP384_prime = {
    .byte = {0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
             0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

const ECC_NISTP384_Param ECC_NISTP384_a = {
    .byte = {0xfc, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
             0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

const ECC_NISTP384_Param ECC_NISTP384_b = {
    .byte = {0xef, 0x2a, 0xec, 0xd3, 0xed, 0xc8, 0x85, 0x2a, 0x9d, 0xd1, 0x2e, 0x8a, 0x8d, 0x39, 0x56, 0xc6,
             0x5a, 0x87, 0x13, 0x50, 0x8f, 0x08, 0x14, 0x03, 0x12, 0x41, 0x81, 0xfe, 0x6e, 0x9c, 0x1d, 0x18,
             0x19, 0x2d, 0xf8, 0xe3, 0x6b, 0x05, 0x8e, 0x98, 0xe4, 0xe7, 0x3e, 0xe2, 0xa7, 0x2f, 0x31, 0xb3}};

const ECC_NISTP384_Param ECC_NISTP384_order = {
    .byte = {0x73, 0x29, 0xc5, 0xcc, 0x6a, 0x19, 0xec, 0xec, 0x7a, 0xa7, 0xb0, 0x48, 0xb2, 0x0d, 0x1a, 0x58,
             0xdf, 0x2d, 0x37, 0xf4, 0x81, 0x4d, 0x63, 0xc7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

const ECCParams_CurveParams ECCParams_NISTP384 = {.curveType  = ECCParams_CURVE_TYPE_SHORT_WEIERSTRASS_AN3,
                                                  .length     = ECCParams_NISTP384_LENGTH,
                                                  .prime      = ECC_NISTP384_prime.byte,
                                                  .order      = ECC_NISTP384_order.byte,
                                                  .a          = ECC_NISTP384_a.byte,
                                                  .b          = ECC_NISTP384_b.byte,
                                                  .generatorX = ECC_NISTP384_generatorX.byte,
                                                  .generatorY = ECC_NISTP384_generatorY.byte,
                                                  .cofactor   = 1};

/*
 * NIST P521 curve params in little endian format.
 * byte[0] is the least significant byte of the curve parameter.
 * Note that the length value in the ECCParams_NISTP521 struct will
 * specify 66 bytes, but the individual curve parameters are all 68 bytes.
 * This is required for the HSM, which expects inputs loaded to the PK array to
 * have a length divisible by 4.
 */
const ECC_NISTP521_Param ECC_NISTP521_generatorX = {
    .byte = {0x66, 0xbd, 0xe5, 0xc2, 0x31, 0x7e, 0x7e, 0xf9, 0x9b, 0x42, 0x6a, 0x85, 0xc1, 0xb3, 0x48, 0x33, 0xde,
             0xa8, 0xff, 0xa2, 0x27, 0xc1, 0x1d, 0xfe, 0x28, 0x59, 0xe7, 0xef, 0x77, 0x5e, 0x4b, 0xa1, 0xba, 0x3d,
             0x4d, 0x6b, 0x60, 0xaf, 0x28, 0xf8, 0x21, 0xb5, 0x3f, 0x05, 0x39, 0x81, 0x64, 0x9c, 0x42, 0xb4, 0x95,
             0x23, 0x66, 0xcb, 0x3e, 0x9e, 0xcd, 0xe9, 0x04, 0x04, 0xb7, 0x06, 0x8e, 0x85, 0xc6, 0x00, 0x00, 0x00}};

const ECC_NISTP521_Param ECC_NISTP521_generatorY = {
    .byte = {0x50, 0x66, 0xd1, 0x9f, 0x76, 0x94, 0xbe, 0x88, 0x40, 0xc2, 0x72, 0xa2, 0x86, 0x70, 0x3c, 0x35, 0x61,
             0x07, 0xad, 0x3f, 0x01, 0xb9, 0x50, 0xc5, 0x40, 0x26, 0xf4, 0x5e, 0x99, 0x72, 0xee, 0x97, 0x2c, 0x66,
             0x3e, 0x27, 0x17, 0xbd, 0xaf, 0x17, 0x68, 0x44, 0x9b, 0x57, 0x49, 0x44, 0xf5, 0x98, 0xd9, 0x1b, 0x7d,
             0x2c, 0xb4, 0x5f, 0x8a, 0x5c, 0x04, 0xc0, 0x3b, 0x9a, 0x78, 0x6a, 0x29, 0x39, 0x18, 0x01, 0x00, 0x00}};

const ECC_NISTP521_Param ECC_NISTP521_prime = {
    .byte = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00}};

const ECC_NISTP521_Param ECC_NISTP521_a = {
    .byte = {0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00}};

const ECC_NISTP521_Param ECC_NISTP521_b = {
    .byte = {0x00, 0x3f, 0x50, 0x6b, 0xd4, 0x1f, 0x45, 0xef, 0xf1, 0x34, 0x2c, 0x3d, 0x88, 0xdf, 0x73, 0x35, 0x07,
             0xbf, 0xb1, 0x3b, 0xbd, 0xc0, 0x52, 0x16, 0x7b, 0x93, 0x7e, 0xec, 0x51, 0x39, 0x19, 0x56, 0xe1, 0x09,
             0xf1, 0x8e, 0x91, 0x89, 0xb4, 0xb8, 0xf3, 0x15, 0xb3, 0x99, 0x5b, 0x72, 0xda, 0xa2, 0xee, 0x40, 0x85,
             0xb6, 0xa0, 0x21, 0x9a, 0x92, 0x1f, 0x9a, 0x1c, 0x8e, 0x61, 0xb9, 0x3e, 0x95, 0x51, 0x00, 0x00, 0x00}};

const ECC_NISTP521_Param ECC_NISTP521_order = {
    .byte = {0x09, 0x64, 0x38, 0x91, 0x1e, 0xb7, 0x6f, 0xbb, 0xae, 0x47, 0x9c, 0x89, 0xb8, 0xc9, 0xb5, 0x3b, 0xd0,
             0xa5, 0x09, 0xf7, 0x48, 0x01, 0xcc, 0x7f, 0x6b, 0x96, 0x2f, 0xbf, 0x83, 0x87, 0x86, 0x51, 0xfa, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00}};

const ECCParams_CurveParams ECCParams_NISTP521 = {.curveType  = ECCParams_CURVE_TYPE_SHORT_WEIERSTRASS_AN3,
                                                  .length     = ECCParams_NISTP521_LENGTH,
                                                  .prime      = ECC_NISTP521_prime.byte,
                                                  .order      = ECC_NISTP521_order.byte,
                                                  .a          = ECC_NISTP521_a.byte,
                                                  .b          = ECC_NISTP521_b.byte,
                                                  .generatorX = ECC_NISTP521_generatorX.byte,
                                                  .generatorY = ECC_NISTP521_generatorY.byte,
                                                  .cofactor   = 1};

/*
 * BrainpoolP256R1 curve params in little endian format.
 * byte[0] is the least significant byte of the curve parameter.
 */
const ECC_BrainpoolP256R1_Param ECC_BrainpoolP256R1_generatorX = {
    .byte = {0x62, 0x32, 0xCE, 0x9A, 0xBD, 0x53, 0x44, 0x3A, 0xC2, 0x23, 0xBD, 0xE3, 0xE1, 0x27, 0xDE, 0xB9,
             0xAF, 0xB7, 0x81, 0xFC, 0x2F, 0x48, 0x4B, 0x2C, 0xCB, 0x57, 0x7E, 0xCB, 0xB9, 0xAE, 0xD2, 0x8B}};

const ECC_BrainpoolP256R1_Param ECC_BrainpoolP256R1_generatorY = {
    .byte = {0x97, 0x69, 0x04, 0x2F, 0xC7, 0x54, 0x1D, 0x5C, 0x54, 0x8E, 0xED, 0x2D, 0x13, 0x45, 0x77, 0xC2,
             0xC9, 0x1D, 0x61, 0x14, 0x1A, 0x46, 0xF8, 0x97, 0xFD, 0xC4, 0xDA, 0xC3, 0x35, 0xF8, 0x7E, 0x54}};

const ECC_BrainpoolP256R1_Param ECC_BrainpoolP256R1_prime = {
    .byte = {0x77, 0x53, 0x6E, 0x1F, 0x1D, 0x48, 0x13, 0x20, 0x28, 0x20, 0x26, 0xD5, 0x23, 0xF6, 0x3B, 0x6E,
             0x72, 0x8D, 0x83, 0x9D, 0x90, 0x0A, 0x66, 0x3E, 0xBC, 0xA9, 0xEE, 0xA1, 0xDB, 0x57, 0xFB, 0xA9}};

const ECC_BrainpoolP256R1_Param ECC_BrainpoolP256R1_a = {
    .byte = {0xD9, 0xB5, 0x30, 0xF3, 0x44, 0x4B, 0x4A, 0xE9, 0x6C, 0x5C, 0xDC, 0x26, 0xC1, 0x55, 0x80, 0xFB,
             0xE7, 0xFF, 0x7A, 0x41, 0x30, 0x75, 0xF6, 0xEE, 0x57, 0x30, 0x2C, 0xFC, 0x75, 0x09, 0x5A, 0x7D}};

const ECC_BrainpoolP256R1_Param ECC_BrainpoolP256R1_b = {
    .byte = {0xB6, 0x07, 0x8C, 0xFF, 0x18, 0xDC, 0xCC, 0x6B, 0xCE, 0xE1, 0xF7, 0x5C, 0x29, 0x16, 0x84, 0x95,
             0xBF, 0x7C, 0xD7, 0xBB, 0xD9, 0xB5, 0x30, 0xF3, 0x44, 0x4B, 0x4A, 0xE9, 0x6C, 0x5C, 0xDC, 0x26}};

const ECC_BrainpoolP256R1_Param ECC_BrainpoolP256R1_order = {
    .byte = {0xA7, 0x56, 0x48, 0x97, 0x82, 0x0E, 0x1E, 0x90, 0xF7, 0xA6, 0x61, 0xB5, 0xA3, 0x7A, 0x39, 0x8C,
             0x71, 0x8D, 0x83, 0x9D, 0x90, 0x0A, 0x66, 0x3E, 0xBC, 0xA9, 0xEE, 0xA1, 0xDB, 0x57, 0xFB, 0xA9}};

const ECCParams_CurveParams ECCParams_BrainpoolP256R1 = {.curveType  = ECCParams_CURVE_TYPE_SHORT_WEIERSTRASS_AN3,
                                                         .length     = ECCParams_BrainpoolP256R1_LENGTH,
                                                         .prime      = ECC_BrainpoolP256R1_prime.byte,
                                                         .order      = ECC_BrainpoolP256R1_order.byte,
                                                         .a          = ECC_BrainpoolP256R1_a.byte,
                                                         .b          = ECC_BrainpoolP256R1_b.byte,
                                                         .generatorX = ECC_BrainpoolP256R1_generatorX.byte,
                                                         .generatorY = ECC_BrainpoolP256R1_generatorY.byte,
                                                         .cofactor   = 1};

/*
 * BrainpoolP384R1 curve params in little endian format.
 * byte[0] is the least significant byte of the curve parameter.
 */
const ECC_BrainpoolP384R1_Param ECC_BrainpoolP384R1_generatorX = {
    .byte = {0x1E, 0xAF, 0xD4, 0x47, 0xE2, 0xB2, 0x87, 0xEF, 0xAA, 0x46, 0xD6, 0x36, 0x34, 0xE0, 0x26, 0xE8,
             0xE8, 0x10, 0xBD, 0x0C, 0xFE, 0xCA, 0x7F, 0xDB, 0xE3, 0x4F, 0xF1, 0x7E, 0xE7, 0xA3, 0x47, 0x88,
             0x6B, 0x3F, 0xC1, 0xB7, 0x81, 0x3A, 0xA6, 0xA2, 0xFF, 0x45, 0xCF, 0x68, 0xF0, 0x64, 0x1C, 0x1D}};

const ECC_BrainpoolP384R1_Param ECC_BrainpoolP384R1_generatorY = {
    .byte = {0x15, 0x53, 0x3C, 0x26, 0x41, 0x03, 0x82, 0x42, 0x11, 0x81, 0x91, 0x77, 0x21, 0x46, 0x46, 0x0E,
             0x28, 0x29, 0x91, 0xF9, 0x4F, 0x05, 0x9C, 0xE1, 0x64, 0x58, 0xEC, 0xFE, 0x29, 0x0B, 0xB7, 0x62,
             0x52, 0xD5, 0xCF, 0x95, 0x8E, 0xEB, 0xB1, 0x5C, 0xA4, 0xC2, 0xF9, 0x20, 0x75, 0x1D, 0xBE, 0x8A}};

const ECC_BrainpoolP384R1_Param ECC_BrainpoolP384R1_prime = {
    .byte = {0x53, 0xEC, 0x07, 0x31, 0x13, 0x00, 0x47, 0x87, 0x71, 0x1A, 0x1D, 0x90, 0x29, 0xA7, 0xD3, 0xAC,
             0x23, 0x11, 0xB7, 0x7F, 0x19, 0xDA, 0xB1, 0x12, 0xB4, 0x56, 0x54, 0xED, 0x09, 0x71, 0x2F, 0x15,
             0xDF, 0x41, 0xE6, 0x50, 0x7E, 0x6F, 0x5D, 0x0F, 0x28, 0x6D, 0x38, 0xA3, 0x82, 0x1E, 0xB9, 0x8C}};

const ECC_BrainpoolP384R1_Param ECC_BrainpoolP384R1_a = {
    .byte = {0x26, 0x28, 0xCE, 0x22, 0xDD, 0xC7, 0xA8, 0x04, 0xEB, 0xD4, 0x3A, 0x50, 0x4A, 0x81, 0xA5, 0x8A,
             0x0F, 0xF9, 0x91, 0xBA, 0xEF, 0x65, 0x91, 0x13, 0x87, 0x27, 0xB2, 0x4F, 0x8E, 0xA2, 0xBE, 0xC2,
             0xA0, 0xAF, 0x05, 0xCE, 0x0A, 0x08, 0x72, 0x3C, 0x0C, 0x15, 0x8C, 0x3D, 0xC6, 0x82, 0xC3, 0x7B}};

const ECC_BrainpoolP384R1_Param ECC_BrainpoolP384R1_b = {
    .byte = {0x11, 0x4C, 0x50, 0xFA, 0x96, 0x86, 0xB7, 0x3A, 0x94, 0xC9, 0xDB, 0x95, 0x02, 0x39, 0xB4, 0x7C,
             0xD5, 0x62, 0xEB, 0x3E, 0xA5, 0x0E, 0x88, 0x2E, 0xA6, 0xD2, 0xDC, 0x07, 0xE1, 0x7D, 0xB7, 0x2F,
             0x7C, 0x44, 0xF0, 0x16, 0x54, 0xB5, 0x39, 0x8B, 0x26, 0x28, 0xCE, 0x22, 0xDD, 0xC7, 0xA8, 0x04}};

const ECC_BrainpoolP384R1_Param ECC_BrainpoolP384R1_order = {
    .byte = {0x65, 0x65, 0x04, 0xE9, 0x02, 0x32, 0x88, 0x3B, 0x10, 0xC3, 0x7F, 0x6B, 0xAF, 0xB6, 0x3A, 0xCF,
             0xA7, 0x25, 0x04, 0xAC, 0x6C, 0x6E, 0x16, 0x1F, 0xB3, 0x56, 0x54, 0xED, 0x09, 0x71, 0x2F, 0x15,
             0xDF, 0x41, 0xE6, 0x50, 0x7E, 0x6F, 0x5D, 0x0F, 0x28, 0x6D, 0x38, 0xA3, 0x82, 0x1E, 0xB9, 0x8C}};

const ECCParams_CurveParams ECCParams_BrainpoolP384R1 = {.curveType  = ECCParams_CURVE_TYPE_SHORT_WEIERSTRASS_AN3,
                                                         .length     = ECCParams_BrainpoolP384R1_LENGTH,
                                                         .prime      = ECC_BrainpoolP384R1_prime.byte,
                                                         .order      = ECC_BrainpoolP384R1_order.byte,
                                                         .a          = ECC_BrainpoolP384R1_a.byte,
                                                         .b          = ECC_BrainpoolP384R1_b.byte,
                                                         .generatorX = ECC_BrainpoolP384R1_generatorX.byte,
                                                         .generatorY = ECC_BrainpoolP384R1_generatorY.byte,
                                                         .cofactor   = 1};

/*
 * BrainpoolP512R1 curve params in little endian format.
 * byte[0] is the least significant byte of the curve parameter.
 */
const ECC_BrainpoolP512R1_Param ECC_BrainpoolP512R1_generatorX = {
    .byte = {0x22, 0xF8, 0xB9, 0xBC, 0x09, 0x22, 0x35, 0x8B, 0x68, 0x5E, 0x6A, 0x40, 0x47, 0x50, 0x6D, 0x7C,
             0x5F, 0x7D, 0xB9, 0x93, 0x7B, 0x68, 0xD1, 0x50, 0x8D, 0xD4, 0xD0, 0xE2, 0x78, 0x1F, 0x3B, 0xFF,
             0x8E, 0x09, 0xD0, 0xF4, 0xEE, 0x62, 0x3B, 0xB4, 0xC1, 0x16, 0xD9, 0xB5, 0x70, 0x9F, 0xED, 0x85,
             0x93, 0x6A, 0x4C, 0x9C, 0x2E, 0x32, 0x21, 0x5A, 0x64, 0xD9, 0x2E, 0xD8, 0xBD, 0xE4, 0xAE, 0x81}};

const ECC_BrainpoolP512R1_Param ECC_BrainpoolP512R1_generatorY = {
    .byte = {0x92, 0x08, 0xD8, 0x3A, 0x0F, 0x1E, 0xCD, 0x78, 0x06, 0x54, 0xF0, 0xA8, 0x2F, 0x2B, 0xCA, 0xD1,
             0xAE, 0x63, 0x27, 0x8A, 0xD8, 0x4B, 0xCA, 0x5B, 0x5E, 0x48, 0x5F, 0x4A, 0x49, 0xDE, 0xDC, 0xB2,
             0x11, 0x81, 0x1F, 0x88, 0x5B, 0xC5, 0x00, 0xA0, 0x1A, 0x7B, 0xA5, 0x24, 0x00, 0xF7, 0x09, 0xF2,
             0xFD, 0x22, 0x78, 0xCF, 0xA9, 0xBF, 0xEA, 0xC0, 0xEC, 0x32, 0x63, 0x56, 0x5D, 0x38, 0xDE, 0x7D}};

const ECC_BrainpoolP512R1_Param ECC_BrainpoolP512R1_prime = {
    .byte = {0xF3, 0x48, 0x3A, 0x58, 0x56, 0x60, 0xAA, 0x28, 0x85, 0xC6, 0x82, 0x2D, 0x2F, 0xFF, 0x81, 0x28,
             0xE6, 0x80, 0xA3, 0xE6, 0x2A, 0xA1, 0xCD, 0xAE, 0x42, 0x68, 0xC6, 0x9B, 0x00, 0x9B, 0x4D, 0x7D,
             0x71, 0x08, 0x33, 0x70, 0xCA, 0x9C, 0x63, 0xD6, 0x0E, 0xD2, 0xC9, 0xB3, 0xB3, 0x8D, 0x30, 0xCB,
             0x07, 0xFC, 0xC9, 0x33, 0xAE, 0xE6, 0xD4, 0x3F, 0x8B, 0xC4, 0xE9, 0xDB, 0xB8, 0x9D, 0xDD, 0xAA}};

const ECC_BrainpoolP512R1_Param ECC_BrainpoolP512R1_a = {
    .byte = {0xCA, 0x94, 0xFC, 0x77, 0x4D, 0xAC, 0xC1, 0xE7, 0xB9, 0xC7, 0xF2, 0x2B, 0xA7, 0x17, 0x11, 0x7F,
             0xB5, 0xC8, 0x9A, 0x8B, 0xC9, 0xF1, 0x2E, 0x0A, 0xA1, 0x3A, 0x25, 0xA8, 0x5A, 0x5D, 0xED, 0x2D,
             0xBC, 0x63, 0x98, 0xEA, 0xCA, 0x41, 0x34, 0xA8, 0x10, 0x16, 0xF9, 0x3D, 0x8D, 0xDD, 0xCB, 0x94,
             0xC5, 0x4C, 0x23, 0xAC, 0x45, 0x71, 0x32, 0xE2, 0x89, 0x3B, 0x60, 0x8B, 0x31, 0xA3, 0x30, 0x78}};

const ECC_BrainpoolP512R1_Param ECC_BrainpoolP512R1_b = {
    .byte = {0x23, 0xF7, 0x16, 0x80, 0x63, 0xBD, 0x09, 0x28, 0xDD, 0xE5, 0xBA, 0x5E, 0xB7, 0x50, 0x40, 0x98,
             0x67, 0x3E, 0x08, 0xDC, 0xCA, 0x94, 0xFC, 0x77, 0x4D, 0xAC, 0xC1, 0xE7, 0xB9, 0xC7, 0xF2, 0x2B,
             0xA7, 0x17, 0x11, 0x7F, 0xB5, 0xC8, 0x9A, 0x8B, 0xC9, 0xF1, 0x2E, 0x0A, 0xA1, 0x3A, 0x25, 0xA8,
             0x5A, 0x5D, 0xED, 0x2D, 0xBC, 0x63, 0x98, 0xEA, 0xCA, 0x41, 0x34, 0xA8, 0x10, 0x16, 0xF9, 0x3D}};

const ECC_BrainpoolP512R1_Param ECC_BrainpoolP512R1_order = {
    .byte = {0x69, 0x00, 0xA9, 0x9C, 0x82, 0x96, 0x87, 0xB5, 0xDD, 0xDA, 0x5D, 0x08, 0x81, 0xD3, 0xB1, 0x1D,
             0x47, 0x10, 0xAC, 0x7F, 0x19, 0x61, 0x86, 0x41, 0x19, 0x26, 0xA9, 0x4C, 0x41, 0x5C, 0x3E, 0x55,
             0x70, 0x08, 0x33, 0x70, 0xCA, 0x9C, 0x63, 0xD6, 0x0E, 0xD2, 0xC9, 0xB3, 0xB3, 0x8D, 0x30, 0xCB,
             0x07, 0xFC, 0xC9, 0x33, 0xAE, 0xE6, 0xD4, 0x3F, 0x8B, 0xC4, 0xE9, 0xDB, 0xB8, 0x9D, 0xDD, 0xAA}};

const ECCParams_CurveParams ECCParams_BrainpoolP512R1 = {.curveType  = ECCParams_CURVE_TYPE_SHORT_WEIERSTRASS_AN3,
                                                         .length     = ECCParams_BrainpoolP512R1_LENGTH,
                                                         .prime      = ECC_BrainpoolP512R1_prime.byte,
                                                         .order      = ECC_BrainpoolP512R1_order.byte,
                                                         .a          = ECC_BrainpoolP512R1_a.byte,
                                                         .b          = ECC_BrainpoolP512R1_b.byte,
                                                         .generatorX = ECC_BrainpoolP512R1_generatorX.byte,
                                                         .generatorY = ECC_BrainpoolP512R1_generatorY.byte,
                                                         .cofactor   = 1};

/*
 * Curve25519 curve params in little endian format.
 * byte[0] is the least significant byte of the curve parameter.
 */
const ECC_Curve25519_Param ECC_Curve25519_generatorX = {
    .byte = {0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}};

const ECC_Curve25519_Param ECC_Curve25519_generatorY = {
    .byte = {0xd9, 0xd3, 0xce, 0x7e, 0xa2, 0xc5, 0xe9, 0x29, 0xb2, 0x61, 0x7c, 0x6d, 0x7e, 0x4d, 0x3d, 0x92,
             0x4c, 0xd1, 0x48, 0x77, 0x2c, 0xdd, 0x1e, 0xe0, 0xb4, 0x86, 0xa0, 0xb8, 0xa1, 0x19, 0xae, 0x20}};

const ECC_Curve25519_Param ECC_Curve25519_prime = {
    .byte = {0xed, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f}};

const ECC_Curve25519_Param ECC_Curve25519_a = {
    .byte = {0x06, 0x6d, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}};

const ECC_Curve25519_Param ECC_Curve25519_b = {
    .byte = {0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}};

const ECC_Curve25519_Param ECC_Curve25519_order = {
    .byte = {0xed, 0xd3, 0xf5, 0x5c, 0x1a, 0x63, 0x12, 0x58, 0xd6, 0x9c, 0xf7, 0xa2, 0xde, 0xf9, 0xde, 0x14,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10}};

const ECCParams_CurveParams ECCParams_Curve25519 = {.curveType  = ECCParams_CURVE_TYPE_MONTGOMERY,
                                                    .length     = ECCParams_CURVE25519_LENGTH,
                                                    .prime      = ECC_Curve25519_prime.byte,
                                                    .order      = ECC_Curve25519_order.byte,
                                                    .a          = ECC_Curve25519_a.byte,
                                                    .b          = ECC_Curve25519_b.byte,
                                                    .generatorX = ECC_Curve25519_generatorX.byte,
                                                    .generatorY = ECC_Curve25519_generatorY.byte,
                                                    .cofactor   = 1};

/*
 * Ed25519 curve params in little endian format.
 * byte[0] is the least significant byte of the curve parameter.
 */
const ECC_Curve25519_Param ECC_Ed25519_generatorX = {
    .byte = {0x1a, 0xd5, 0x25, 0x8f, 0x60, 0x2d, 0x56, 0xc9, 0xb2, 0xa7, 0x25, 0x95, 0x60, 0xc7, 0x2c, 0x69,
             0x5c, 0xdc, 0xd6, 0xfd, 0x31, 0xe2, 0xa4, 0xc0, 0xfe, 0x53, 0x6e, 0xcd, 0xd3, 0x36, 0x69, 0x21}};

const ECC_Curve25519_Param ECC_Ed25519_generatorY = {
    .byte = {0x58, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66,
             0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66}};

const ECC_Curve25519_Param ECC_Ed25519_prime = {
    .byte = {0xed, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
             0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f}};

const ECC_Curve25519_Param ECC_Ed25519_order = {
    .byte = {0xed, 0xd3, 0xf5, 0x5c, 0x1a, 0x63, 0x12, 0x58, 0xd6, 0x9c, 0xf7, 0xa2, 0xde, 0xf9, 0xde, 0x14,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10}};

const ECC_Curve25519_Param ECC_Ed25519_a = {.byte = {0xec, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                     0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                                                     0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f}};

const ECC_Curve25519_Param ECC_Ed25519_d = {.byte = {0xa3, 0x78, 0x59, 0x13, 0xca, 0x4d, 0xeb, 0x75, 0xab, 0xd8, 0x41,
                                                     0x41, 0x4d, 0x0a, 0x70, 0x00, 0x98, 0xe8, 0x79, 0x77, 0x79, 0x40,
                                                     0xc7, 0x8c, 0x73, 0xfe, 0x6f, 0x2b, 0xee, 0x6c, 0x03, 0x52}};

const ECCParams_CurveParams ECCParams_Ed25519 = {.curveType  = ECCParams_CURVE_TYPE_EDWARDS,
                                                 .length     = 32,
                                                 .prime      = ECC_Ed25519_prime.byte,
                                                 .order      = ECC_Ed25519_order.byte,
                                                 .a          = ECC_Ed25519_a.byte,
                                                 .b          = ECC_Ed25519_d.byte,
                                                 .generatorX = ECC_Ed25519_generatorX.byte,
                                                 .generatorY = ECC_Ed25519_generatorY.byte,
                                                 .cofactor   = 8};

/*
 *  ======== ECCParams_getUncompressedGeneratorPoint ========
 */
int_fast16_t ECCParams_getUncompressedGeneratorPoint(const ECCParams_CurveParams *curveParams,
                                                     uint8_t *buffer,
                                                     size_t length)
{
    size_t paramLength = curveParams->length;
    size_t pointLength = (paramLength * 2) + 1;

    if (length < pointLength)
    {
        return ECCParams_STATUS_ERROR;
    }

    /* Reverse and concatenate x and y */
    CryptoUtils_reverseCopy(curveParams->generatorX, buffer + 1, paramLength);

    CryptoUtils_reverseCopy(curveParams->generatorY, buffer + 1 + paramLength, paramLength);

    buffer[0] = 0x04;

    memset(buffer + pointLength, 0, length - pointLength);

    return ECCParams_STATUS_SUCCESS;
}
