/******************************************************************************
*  Filename:       hw_dcan.h
*
*  Description:    Defines and prototypes for the DCAN peripheral.
*
*  Copyright (c) 2023-2025, Texas Instruments Incorporated
*  All rights reserved.
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions are met:
*
*  1) Redistributions of source code must retain the above copyright notice,
*     this list of conditions and the following disclaimer.
*
*  2) Redistributions in binary form must reproduce the above copyright notice,
*     this list of conditions and the following disclaimer in the documentation
*     and/or other materials provided with the distribution.
*
*  3) Neither the name of the ORGANIZATION nor the names of its contributors may
*     be used to endorse or promote products derived from this software without
*     specific prior written permission.
*
  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
*  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
*  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
*  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
*  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
*  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
*  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
*  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
*  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
*  POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************/
#ifndef __HW_DCAN_H__
#define __HW_DCAN_H__
/*-------------------------------------------------------------------------------

This section defines the register offsets of the DCAN component

--------------------------------------------------------------------------------- */

//MCAN Core Release Register
#define DCAN_O_CREL                                                  0x00000000U

//MCAN Endian Register
#define DCAN_O_ENDN                                                  0x00000004U

//This register is only writable if bits CCCR
#define DCAN_O_DBTP                                                  0x0000000CU

//Write access to the Test Register has to be enabled by setting bit CCCR
#define DCAN_O_TEST                                                  0x00000010U

//MCAN RAM Watchdog
#define DCAN_O_RWD                                                   0x00000014U

//MCAN CC Control Register
#define DCAN_O_CCCR                                                  0x00000018U

//This register is only writable if bits CCCR
#define DCAN_O_NBTP                                                  0x0000001CU

//MCAN Timestamp Counter Configuration
#define DCAN_O_TSCC                                                  0x00000020U

//MCAN Timestamp Counter Value
#define DCAN_O_TSCV                                                  0x00000024U

//MCAN Timeout Counter Configuration
#define DCAN_O_TOCC                                                  0x00000028U

//MCAN Timeout Counter Value
#define DCAN_O_TOCV                                                  0x0000002CU

//MCAN Error Counter Register
#define DCAN_O_ECR                                                   0x00000040U

//MCAN Protocol Status Register
#define DCAN_O_PSR                                                   0x00000044U

//MCAN Transmitter Delay Compensation Register
#define DCAN_O_TDCR                                                  0x00000048U

//The flags are set when one of the listed conditions is detected (edge-sensitive)
#define DCAN_O_IR                                                    0x00000050U

//MCAN Interrupt Enable
#define DCAN_O_IE                                                    0x00000054U

//The Interrupt Line Select register assigns an interrupt generated by a specific interrupt flag from the Interrupt Register to one of the two module interrupt lines
#define DCAN_O_ILS                                                   0x00000058U

//MCAN Interrupt Line Enable
#define DCAN_O_ILE                                                   0x0000005CU

//MCAN Global Filter Configuration
#define DCAN_O_GFC                                                   0x00000080U

//MCAN Standard ID Filter Configuration
#define DCAN_O_SIDFC                                                 0x00000084U

//MCAN Extended ID Filter Configuration
#define DCAN_O_XIDFC                                                 0x00000088U

//MCAN Extended ID and Mask
#define DCAN_O_XIDAM                                                 0x00000090U

//This register is updated every time a Message ID filter element configured to generate a priority event matches
#define DCAN_O_HPMS                                                  0x00000094U

//MCAN New Data 1
#define DCAN_O_NDAT1                                                 0x00000098U

//MCAN New Data 2
#define DCAN_O_NDAT2                                                 0x0000009CU

//MCAN Rx FIFO 0 Configuration
#define DCAN_O_RXF0C                                                 0x000000A0U

//MCAN Rx FIFO 0 Status
#define DCAN_O_RXF0S                                                 0x000000A4U

//MCAN Rx FIFO 0 Acknowledge
#define DCAN_O_RXF0A                                                 0x000000A8U

//MCAN Rx Buffer Configuration
#define DCAN_O_RXBC                                                  0x000000ACU

//MCAN Rx FIFO 1 Configuration
#define DCAN_O_RXF1C                                                 0x000000B0U

//MCAN Rx FIFO 1 Status
#define DCAN_O_RXF1S                                                 0x000000B4U

//MCAN Rx FIFO 1 Acknowledge
#define DCAN_O_RXF1A                                                 0x000000B8U

//Configures the number of data bytes belonging to an Rx Buffer / Rx FIFO element
#define DCAN_O_RXESC                                                 0x000000BCU

//MCAN Tx Buffer Configuration
#define DCAN_O_TXBC                                                  0x000000C0U

//The Tx FIFO/Queue status is related to the pending Tx requests listed in register TXBRP
#define DCAN_O_TXFQS                                                 0x000000C4U

//Configures the number of data bytes belonging to a Tx Buffer element
#define DCAN_O_TXESC                                                 0x000000C8U

//MCAN Tx Buffer Request Pending
#define DCAN_O_TXBRPAP                                               0x000000CCU

//MCAN Tx Buffer Add Request
#define DCAN_O_TXBAR                                                 0x000000D0U

//MCAN Tx Buffer Cancellation Request
#define DCAN_O_TXBCR                                                 0x000000D4U

//MCAN Tx Buffer Transmission Occurred
#define DCAN_O_TXBTO                                                 0x000000D8U

//MCAN Tx Buffer Cancellation Finished
#define DCAN_O_TXBCF                                                 0x000000DCU

//MCAN Tx Buffer Transmission Interrupt Enable
#define DCAN_O_TXTIE                                                 0x000000E0U

//MCAN Tx Buffer Cancellation Finished Interrupt Enable
#define DCAN_O_TXBCIE                                                0x000000E4U

//MCAN Tx Event FIFO Configuration
#define DCAN_O_TXEFC                                                 0x000000F0U

//MCAN Tx Event FIFO Status
#define DCAN_O_TXEFS                                                 0x000000F4U

//MCAN Tx Event FIFO Acknowledge
#define DCAN_O_TXEFA                                                 0x000000F8U

//MCAN Subsystem Revision Register
#define DCAN_O_SSPID                                                 0x00000200U

//MCAN Subsystem Control Register
#define DCAN_O_SSCTL                                                 0x00000204U

//MCAN Subsystem Status Register
#define DCAN_O_SSSTA                                                 0x00000208U

//MCAN Subsystem Interrupt Clear Shadow Register
#define DCAN_O_SSICS                                                 0x0000020CU

//MCAN Subsystem Interrupt Raw Status Register
#define DCAN_O_SSIRS                                                 0x00000210U

//MCAN Subsystem Interrupt Enable Clear Shadow Register
#define DCAN_O_SSIECS                                                0x00000214U

//MCAN Subsystem Interrupt Enable Register
#define DCAN_O_SSIE                                                  0x00000218U

//MCAN Subsystem Masked Interrupt Status
#define DCAN_O_SSIES                                                 0x0000021CU

//MCAN Subsystem End of Interrupt
#define DCAN_O_SSEOI                                                 0x00000220U

//MCAN Subsystem External Timestamp Prescaler 0
#define DCAN_O_EXTTSPS                                               0x00000224U

//MCAN Subsystem External Timestamp Unserviced Interrupts Counter
#define DCAN_O_EXTTSUSI                                              0x00000228U

//MCAN Error Aggregator Revision Register
#define DCAN_O_ERRREV                                                0x00000400U

//Each error detection and correction (EDC) controller has a bank of error registers (offsets 0x10 - 0x3B) associated with it
#define DCAN_O_ERRVEC                                                0x00000408U

//MCAN Error Misc Status
#define DCAN_O_ERRSTA                                                0x0000040CU

//This register is accessed through the ECC aggregator via an internal serial bus
#define DCAN_O_ERRWRAPREV                                            0x00000410U

//This register is accessed through the ECC aggregator via an internal serial bus
#define DCAN_O_ERRCTL                                                0x00000414U

//This register is accessed through the ECC aggregator via an internal serial bus
#define DCAN_O_ERRCTL1                                               0x00000418U

//This register is accessed through the ECC aggregator via an internal serial bus
#define DCAN_O_ERRCTL2                                               0x0000041CU

//This register is accessed through the ECC aggregator via an internal serial bus
#define DCAN_O_ERRSTA1                                               0x00000420U

//This register is accessed through the ECC aggregator via an internal serial bus
#define DCAN_O_ERRSTA2                                               0x00000424U

//This register is accessed through the ECC aggregator via an internal serial bus
#define DCAN_O_ERRSTA3                                               0x00000428U

//MCAN Single Error Corrected End of Interrupt Register
#define DCAN_O_SECEOI                                                0x0000043CU

//MCAN Single Error Corrected Interrupt Status Register
#define DCAN_O_SECSTA                                                0x00000440U

//MCAN Single Error Corrected Interrupt Enable Set Register
#define DCAN_O_SECENSET                                              0x00000480U

//MCAN Single Error Corrected Interrupt Enable Clear Register
#define DCAN_O_SECENCLR                                              0x000004C0U

//MCAN Double Error Detected End of Interrupt Register
#define DCAN_O_DEDEOI                                                0x0000053CU

//MCAN Double Error Detected Interrupt Status Register
#define DCAN_O_DEDSTA                                                0x00000540U

//MCAN Double Error Detected Interrupt Enable Set Register
#define DCAN_O_DEDENSET                                              0x00000580U

//MCAN Double Error Detected Interrupt Enable Clear Register
#define DCAN_O_DEDENCLR                                              0x000005C0U

//MCAN Error Aggregator Enable Set Register
#define DCAN_O_AGGRENSET                                             0x00000600U

//MCAN Error Aggregator Enable Clear Register
#define DCAN_O_AGGRENCLR                                             0x00000604U

//MCAN Error Aggregator Status Set Register
#define DCAN_O_AGGRSTASET                                            0x00000608U

//MCAN Error Aggregator Status Clear Register
#define DCAN_O_AGGRSTACLR                                            0x0000060CU

//This register identifies the peripheral and its exact version
#define DCAN_O_DESC                                                  0x00000800U

//Interrupt Mask
#define DCAN_O_IMASK0                                                0x00000844U

//Raw interrupt status
#define DCAN_O_RIS0                                                  0x00000848U

//Masked interrupt status
#define DCAN_O_MIS0                                                  0x0000084CU

//Interrupt set
#define DCAN_O_ISET0                                                 0x00000850U

//Interrupt clear
#define DCAN_O_ICLR0                                                 0x00000854U

//Interrupt Mask
#define DCAN_O_IMASK1                                                0x00000868U

//Raw interrupt status
#define DCAN_O_RIS1                                                  0x0000086CU

//Masked interrupt status
#define DCAN_O_MIS1                                                  0x00000870U

//Interrupt set
#define DCAN_O_ISET1                                                 0x00000874U

//Interrupt clear
#define DCAN_O_ICLR1                                                 0x00000878U

//Needs to go to the Management aperture once available
#define DCAN_O_CLKDIV                                                0x00000904U

//MCANSS clock stop control MMR
#define DCAN_O_CLKCTL                                                0x00000908U

//MCANSS clock stop status register to indicate status of clock stop mechanism
#define DCAN_O_CLKSTA                                                0x0000090CU

//MCANSS fixed DMA0 control and configuration register
#define DCAN_O_DMA0CTL                                               0x00000924U

//MCANSS fixed DMA1 control and configuration register
#define DCAN_O_DMA1CTL                                               0x0000092CU

//Rx buffer (index x) base address
#define DCAN_O_TTOFE0                                                0x00000938U

//Rx buffer (index x+1) base address
#define DCAN_O_TTOFE1                                                0x00000948U

//Rx Buffer two-to-one DMA mode, hardware NDAT1 value register
#define DCAN_O_TTONDAT1                                              0x00000950U

//SRAM
#define DCAN_O_SRAM                                                  0x00001000U

//Clock Configuration MMR for **DCAN**
#define DCAN_O_CLKCFG                                                0x00002000U



/*-----------------------------------REGISTER------------------------------------
    Register name:       CREL
    Offset name:         DCAN_O_CREL
    Relative address:    0x0
    Description:         MCAN Core Release Register
    Default Value:       0x32380608

        Field:           DAY
        From..to bits:   0...7
        DefaultValue:    0x8
        Access type:     read-only
        Description:     Time Stamp Day. Two digits, BCD-coded.

*/
#define DCAN_CREL_DAY_W                                              8U
#define DCAN_CREL_DAY_M                                              0x000000FFU
#define DCAN_CREL_DAY_S                                              0U
/*

        Field:           MON
        From..to bits:   8...15
        DefaultValue:    0x6
        Access type:     read-only
        Description:     Time Stamp Month. Two digits, BCD-coded.

*/
#define DCAN_CREL_MON_W                                              8U
#define DCAN_CREL_MON_M                                              0x0000FF00U
#define DCAN_CREL_MON_S                                              8U
/*

        Field:           YEAR
        From..to bits:   16...19
        DefaultValue:    0x8
        Access type:     read-only
        Description:     Time Stamp Year. One digit, BCD-coded.

*/
#define DCAN_CREL_YEAR_W                                             4U
#define DCAN_CREL_YEAR_M                                             0x000F0000U
#define DCAN_CREL_YEAR_S                                             16U
/*

        Field:           SUBSTEP
        From..to bits:   20...23
        DefaultValue:    0x3
        Access type:     read-only
        Description:     Sub-Step of Core Release. One digit, BCD-coded.

*/
#define DCAN_CREL_SUBSTEP_W                                          4U
#define DCAN_CREL_SUBSTEP_M                                          0x00F00000U
#define DCAN_CREL_SUBSTEP_S                                          20U
/*

        Field:           STEP
        From..to bits:   24...27
        DefaultValue:    0x2
        Access type:     read-only
        Description:     Step of Core Release. One digit, BCD-coded.

*/
#define DCAN_CREL_STEP_W                                             4U
#define DCAN_CREL_STEP_M                                             0x0F000000U
#define DCAN_CREL_STEP_S                                             24U
/*

        Field:           REL
        From..to bits:   28...31
        DefaultValue:    0x3
        Access type:     read-only
        Description:     Core Release. One digit, BCD-coded.

*/
#define DCAN_CREL_REL_W                                              4U
#define DCAN_CREL_REL_M                                              0xF0000000U
#define DCAN_CREL_REL_S                                              28U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ENDN
    Offset name:         DCAN_O_ENDN
    Relative address:    0x4
    Description:         MCAN Endian Register
    Default Value:       0x87654321

        Field:           ETV
        From..to bits:   0...31
        DefaultValue:    0x87654321
        Access type:     read-only
        Description:     Endianess Test Value. Reading the constant value maintained in this register allows software to determine the endianess of the host CPU.

*/
#define DCAN_ENDN_ETV_W                                              32U
#define DCAN_ENDN_ETV_M                                              0xFFFFFFFFU
#define DCAN_ENDN_ETV_S                                              0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       DBTP
    Offset name:         DCAN_O_DBTP
    Relative address:    0xC
    Description:         This register is only writable if bits CCCR.CCE and CCCR.INIT are set. The CAN bit time may be programmed in the range of 4 to 49 time quanta. The CAN time quantum may be programmed in the range of 1 to 32 m_can_cclk periods. tq = (DBRP + 1) mtq.

                        DTSEG1 is the sum of Prop_Seg and Phase_Seg1. DTSEG2 is Phase_Seg2.

                        Therefore the length of the bit time is (programmed values) (DTSEG1 + DTSEG2 + 3) tq or (functional values) (Sync_Seg + Prop_Seg + Phase_Seg1 + Phase_Seg2) tq.

                        The Information Processing Time (IPT) is zero, meaning the data for the next bit is available at the first clock edge after the sample point.
    Default Value:       0x00000A33

        Field:           DSJW
        From..to bits:   0...3
        DefaultValue:    0x3
        Access type:     read-write
        Description:     Data Resynchronization Jump Width. Valid values are 0 to 15. The actual interpretation by the hardware of this value is such that one more than the value programmed here is used.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_DBTP_DSJW_W                                             4U
#define DCAN_DBTP_DSJW_M                                             0x0000000FU
#define DCAN_DBTP_DSJW_S                                             0U
/*

        Field:           DTSEG2
        From..to bits:   4...7
        DefaultValue:    0x3
        Access type:     read-write
        Description:     Data Time Segment After Sample Point. Valid values are 0 to 15. The actual interpretation by the hardware of this value is such that one more than the programmed value is used.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_DBTP_DTSEG2_W                                           4U
#define DCAN_DBTP_DTSEG2_M                                           0x000000F0U
#define DCAN_DBTP_DTSEG2_S                                           4U
/*

        Field:           DTSEG1
        From..to bits:   8...12
        DefaultValue:    0xA
        Access type:     read-write
        Description:     Data Time Segment Before Sample Point. Valid values are 0 to 31. The actual interpretation by the hardware of this value is such that one more than the programmed value is used.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_DBTP_DTSEG1_W                                           5U
#define DCAN_DBTP_DTSEG1_M                                           0x00001F00U
#define DCAN_DBTP_DTSEG1_S                                           8U
/*

        Field:           DBRP
        From..to bits:   16...20
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Data Bit Rate Prescaler. The value by which the oscillator frequency is divided for generating the bit time quanta. The bit time is built up from a multiple of this quanta. Valid values for the Bit Rate Prescaler are 0 to 31. The actual interpretation by the hardware of this value is such that one more than the value programmed here is used.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_DBTP_DBRP_W                                             5U
#define DCAN_DBTP_DBRP_M                                             0x001F0000U
#define DCAN_DBTP_DBRP_S                                             16U
/*

        Field:           TDC
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmitter Delay Compensation
                           0  Transmitter Delay Compensation disabled
                           1  Transmitter Delay Compensation enabled

                         +I107

*/
#define DCAN_DBTP_TDC                                                0x00800000U
#define DCAN_DBTP_TDC_M                                              0x00800000U
#define DCAN_DBTP_TDC_S                                              23U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TEST
    Offset name:         DCAN_O_TEST
    Relative address:    0x10
    Description:         Write access to the Test Register has to be enabled by setting bit CCCR.TEST to '1'. All Test Register functions are set to their reset values when bit CCCR.TEST is reset.

                        Loop Back Mode and software control of the internal CAN TX pin are hardware test modes. Programming of
                        TX ? "00" may disturb the message transfer on the CAN bus.
    Default Value:       0x00000000

        Field:           LBCK
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Loop Back Mode
                           0  Reset value, Loop Back Mode is disabled
                           1  Loop Back Mode is enabled

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TEST_LBCK                                               0x00000010U
#define DCAN_TEST_LBCK_M                                             0x00000010U
#define DCAN_TEST_LBCK_S                                             4U
/*

        Field:           TX
        From..to bits:   5...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Control of Transmit Pin
                           00  CAN TX pin controlled by the CAN Core, updated at the end of the CAN bit time
                           01  Sample Point can be monitored at CAN TX pin
                           10  Dominant ('0') level at CAN TX pin
                           11  Recessive ('1') at CAN TX pin

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TEST_TX_W                                               2U
#define DCAN_TEST_TX_M                                               0x00000060U
#define DCAN_TEST_TX_S                                               5U
/*

        Field:           RX
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Receive Pin. Monitors the actual value of the CAN receive pin.
                           0  The CAN bus is dominant (CAN RX pin = '0')
                           1  The CAN bus is recessive (CAN RX pin = '1')

*/
#define DCAN_TEST_RX                                                 0x00000080U
#define DCAN_TEST_RX_M                                               0x00000080U
#define DCAN_TEST_RX_S                                               7U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RWD
    Offset name:         DCAN_O_RWD
    Relative address:    0x14
    Description:         MCAN RAM Watchdog
    Default Value:       0x00000000

        Field:           WDC
        From..to bits:   0...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Watchdog Configuration. Start value of the Message RAM Watchdog Counter. With the reset value of "00" the counter is disabled.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RWD_WDC_W                                               8U
#define DCAN_RWD_WDC_M                                               0x000000FFU
#define DCAN_RWD_WDC_S                                               0U
/*

        Field:           WDV
        From..to bits:   8...15
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Watchdog Value. Actual Message RAM Watchdog Counter Value.

                         The RAM Watchdog monitors the READY output of the Message RAM. A Message RAM access via the MCAN's Generic Commander Interface starts the Message RAM Watchdog Counter with the value configured by the WDC field. The counter is reloaded with WDC when the Message RAM signals successful completion by activating its READY output. In case there is no response from the Message RAM until the counter has counted down to zero, the counter stops and interrupt flag MCAN_IR.WDI is set. The RAM Watchdog Counter is clocked by the host (system) clock.

*/
#define DCAN_RWD_WDV_W                                               8U
#define DCAN_RWD_WDV_M                                               0x0000FF00U
#define DCAN_RWD_WDV_S                                               8U


/*-----------------------------------REGISTER------------------------------------
    Register name:       CCCR
    Offset name:         DCAN_O_CCCR
    Relative address:    0x18
    Description:         MCAN CC Control Register
    Default Value:       0x00000001

        Field:           INIT
        From..to bits:   0...0
        DefaultValue:    0x1
        Access type:     read-write
        Description:     Initialization
                           0  Normal Operation
                           1  Initialization is started
                         Note: Due to the synchronization mechanism between the two clock domains, there may be a delay until the value written to INIT can be read back. Therefore the programmer has to assure that the previous value written to INIT has been accepted by reading INIT before setting INIT to a new value.

*/
#define DCAN_CCCR_INIT                                               0x00000001U
#define DCAN_CCCR_INIT_M                                             0x00000001U
#define DCAN_CCCR_INIT_S                                             0U
/*

        Field:           CCE
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Configuration Change Enable
                           0  The CPU has no write access to the protected configuration registers
                           1  The CPU has write access to the protected configuration registers (while CCCR.INIT = '1')

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_CCCR_CCE                                                0x00000002U
#define DCAN_CCCR_CCE_M                                              0x00000002U
#define DCAN_CCCR_CCE_S                                              1U
/*

        Field:           ASM
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Restricted Operation Mode. Bit ASM can only be set by SW when both CCE and INIT are set to '1'. The bit can be reset by SW at any time.
                           0  Normal CAN operation
                           1  Restricted Operation Mode active

                         Qualified Write 1 to Set is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_CCCR_ASM                                                0x00000004U
#define DCAN_CCCR_ASM_M                                              0x00000004U
#define DCAN_CCCR_ASM_S                                              2U
/*

        Field:           CSA
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Clock Stop Acknowledge
                           0  No clock stop acknowledged
                           1  MCAN may be set in power down by stopping the Host and CAN clocks

*/
#define DCAN_CCCR_CSA                                                0x00000008U
#define DCAN_CCCR_CSA_M                                              0x00000008U
#define DCAN_CCCR_CSA_S                                              3U
/*

        Field:           CSR
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Clock Stop Request
                           0  No clock stop is requested
                           1  Clock stop requested. When clock stop is requested, first INIT and then CSA will be set after all pending transfer requests have been completed and the CAN bus reached idle.

*/
#define DCAN_CCCR_CSR                                                0x00000010U
#define DCAN_CCCR_CSR_M                                              0x00000010U
#define DCAN_CCCR_CSR_S                                              4U
/*

        Field:           MON
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Bus Monitoring Mode. Bit MON can only be set by SW when both CCE and INIT are set to '1'. The bit can be reset by SW at any time.
                           0  Bus Monitoring Mode is disabled
                           1  Bus Monitoring Mode is enabled

                         Qualified Write 1 to Set is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_CCCR_MON                                                0x00000020U
#define DCAN_CCCR_MON_M                                              0x00000020U
#define DCAN_CCCR_MON_S                                              5U
/*

        Field:           DAR
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Disable Automatic Retransmission
                           0  Automatic retransmission of messages not transmitted successfully enabled
                           1  Automatic retransmission disabled

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_CCCR_DAR                                                0x00000040U
#define DCAN_CCCR_DAR_M                                              0x00000040U
#define DCAN_CCCR_DAR_S                                              6U
/*

        Field:           TEST
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Test Mode Enable
                           0  Normal operation, register TEST holds reset values
                           1  Test Mode, write access to register TEST enabled

                         Qualified Write 1 to Set is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_CCCR_TEST                                               0x00000080U
#define DCAN_CCCR_TEST_M                                             0x00000080U
#define DCAN_CCCR_TEST_S                                             7U
/*

        Field:           FDOE
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Flexible Datarate Operation Enable
                           0  FD operation disabled
                           1  FD operation enabled

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_CCCR_FDOE                                               0x00000100U
#define DCAN_CCCR_FDOE_M                                             0x00000100U
#define DCAN_CCCR_FDOE_S                                             8U
/*

        Field:           BRSE
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Bit Rate Switch Enable
                           0  Bit rate switching for transmissions disabled
                           1  Bit rate switching for transmissions enabled
                         Note: When CAN FD operation is disabled FDOE = '0', BRSE is not evaluated.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_CCCR_BRSE                                               0x00000200U
#define DCAN_CCCR_BRSE_M                                             0x00000200U
#define DCAN_CCCR_BRSE_S                                             9U
/*

        Field:           PXHD
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Protocol Exception Handling Disable
                           0  Protocol exception handling enabled
                           1  Protocol exception handling disabled
                         Note: When protocol exception handling is disabled, the MCAN will transmit an error frame when it detects a protocol exception condition.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_CCCR_PXHD                                               0x00001000U
#define DCAN_CCCR_PXHD_M                                             0x00001000U
#define DCAN_CCCR_PXHD_S                                             12U
/*

        Field:           EFBI
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Edge Filtering during Bus Integration
                           0  Edge filtering disabled
                           1  Two consecutive dominant tq required to detect an edge for hard synchronization

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_CCCR_EFBI                                               0x00002000U
#define DCAN_CCCR_EFBI_M                                             0x00002000U
#define DCAN_CCCR_EFBI_S                                             13U
/*

        Field:           TXP
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmit Pause. If this bit is set, the MCAN pauses for two CAN bit times before starting the next transmission after itself has successfully transmitted a frame.
                           0  Transmit pause disabled
                           1  Transmit pause enabled

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_CCCR_TXP                                                0x00004000U
#define DCAN_CCCR_TXP_M                                              0x00004000U
#define DCAN_CCCR_TXP_S                                              14U
/*

        Field:           NISO
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Non ISO Operation. If this bit is set, the MCAN uses the CAN FD frame format as specified by the Bosch CAN FD Specification V1.0.
                           0  CAN FD frame format according to ISO 11898-1:2015
                           1  CAN FD frame format according to Bosch CAN FD Specification V1.0

*/
#define DCAN_CCCR_NISO                                               0x00008000U
#define DCAN_CCCR_NISO_M                                             0x00008000U
#define DCAN_CCCR_NISO_S                                             15U


/*-----------------------------------REGISTER------------------------------------
    Register name:       NBTP
    Offset name:         DCAN_O_NBTP
    Relative address:    0x1C
    Description:         This register is only writable if bits CCCR.CCE and CCCR.INIT are set. The CAN bit time may be programmed in the range of 4 to 385 time quanta. The CAN time quantum may be programmed in the range of 1 to 512 m_can_cclk periods. tq = (NBRP + 1) mtq.

                        NTSEG1 is the sum of Prop_Seg and Phase_Seg1. NTSEG2 is Phase_Seg2.

                        Therefore the length of the bit time is (programmed values) (NTSEG1 + NTSEG2 + 3) tq or (functional values) (Sync_Seg + Prop_Seg + Phase_Seg1 + Phase_Seg2) tq.

                        The Information Processing Time (IPT) is zero, meaning the data for the next bit is available at the first clock edge after the sample point.

                        Note: With a CAN clock of 8 MHz, the reset value of 0x06000A03 configures the MCAN for a bit rate of 500 kBit/s.
    Default Value:       0x06000A03

        Field:           NTSEG2
        From..to bits:   0...6
        DefaultValue:    0x3
        Access type:     read-write
        Description:     Nominal Time Segment After Sample Point. Valid values are 1 to 127. The actual interpretation by the hardware of this value is such that one more than the programmed value is used.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_NBTP_NTSEG2_W                                           7U
#define DCAN_NBTP_NTSEG2_M                                           0x0000007FU
#define DCAN_NBTP_NTSEG2_S                                           0U
/*

        Field:           NTSEG1
        From..to bits:   8...15
        DefaultValue:    0xA
        Access type:     read-write
        Description:     Nominal Time Segment Before Sample Point. Valid values are 1 to 255. The actual interpretation by the hardware of this value is such that one more than the programmed value is used.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_NBTP_NTSEG1_W                                           8U
#define DCAN_NBTP_NTSEG1_M                                           0x0000FF00U
#define DCAN_NBTP_NTSEG1_S                                           8U
/*

        Field:           NBRP
        From..to bits:   16...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Nominal Bit Rate Prescaler. The value by which the oscillator frequency is divided for generating the bit time quanta. The bit time is built up from a multiple of this quanta. Valid values for the Bit Rate Prescaler are 0 to 511. The actual interpretation by the hardware of this value is such that one more than the value programmed here is used.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_NBTP_NBRP_W                                             9U
#define DCAN_NBTP_NBRP_M                                             0x01FF0000U
#define DCAN_NBTP_NBRP_S                                             16U
/*

        Field:           NSJW
        From..to bits:   25...31
        DefaultValue:    0x3
        Access type:     read-write
        Description:     Nominal (Re)Synchronization Jump Width. Valid values are 0 to 127. The actual interpretation by the hardware of this value is such that one more than the value programmed here is used.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_NBTP_NSJW_W                                             7U
#define DCAN_NBTP_NSJW_M                                             0xFE000000U
#define DCAN_NBTP_NSJW_S                                             25U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TSCC
    Offset name:         DCAN_O_TSCC
    Relative address:    0x20
    Description:         MCAN Timestamp Counter Configuration
    Default Value:       0x00000000

        Field:           TSS
        From..to bits:   0...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Timestamp Select
                           00  Timestamp counter value always 0x0000
                           01  Timestamp counter value incremented according to TCP
                           10  External timestamp counter value used
                           11  Same as "00"

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TSCC_TSS_W                                              2U
#define DCAN_TSCC_TSS_M                                              0x00000003U
#define DCAN_TSCC_TSS_S                                              0U
/*

        Field:           TCP
        From..to bits:   16...19
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Timestamp Counter Prescaler. Configures the timestamp and timeout counters time unit in multiples of CAN bit times. Valid values are 0 to 15. The actual interpretation by the hardware of this value is such that one more than the value programmed here is used.

                         Note: With CAN FD an external counter is required for timestamp generation (TSS = "10").

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TSCC_TCP_W                                              4U
#define DCAN_TSCC_TCP_M                                              0x000F0000U
#define DCAN_TSCC_TCP_S                                              16U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TSCV
    Offset name:         DCAN_O_TSCV
    Relative address:    0x24
    Description:         MCAN Timestamp Counter Value
    Default Value:       0x00000000

        Field:           TSC
        From..to bits:   0...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Timestamp Counter. The internal/external Timestamp Counter value is captured on start of frame (both Rx and Tx). When TSCC.TSS = "01", the Timestamp Counter is incremented in multiples of CAN bit times, (1...16), depending on the configuration of TSCC.TCP. A wrap around sets interrupt flag IR.TSW. Write access resets the counter to zero. When TSCC.TSS = "10", TSC reflects the External Timestamp Counter value, and a write access has no impact.

                         Note: A "wrap around" is a change of the Timestamp Counter value from non-zero to zero not
                         caused by write access to MCAN_TSCV.

*/
#define DCAN_TSCV_TSC_W                                              16U
#define DCAN_TSCV_TSC_M                                              0x0000FFFFU
#define DCAN_TSCV_TSC_S                                              0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TOCC
    Offset name:         DCAN_O_TOCC
    Relative address:    0x28
    Description:         MCAN Timeout Counter Configuration
    Default Value:       0xFFFF0000

        Field:           ETOC
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Enable Timeout Counter
                           0  Timeout Counter disabled
                           1  Timeout Counter enabled

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TOCC_ETOC                                               0x00000001U
#define DCAN_TOCC_ETOC_M                                             0x00000001U
#define DCAN_TOCC_ETOC_S                                             0U
/*

        Field:           TOS
        From..to bits:   1...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Timeout Select. When operating in Continuous mode, a write to TOCV presets the counter to the value configured by TOCC.TOP and continues down-counting. When the Timeout Counter is controlled by one of the FIFOs, an empty FIFO presets the counter to the value configured by TOCC.TOP. Down-counting is started when the first FIFO element is stored.
                           00  Continuous operation
                           01  Timeout controlled by Tx Event FIFO
                           10  Timeout controlled by Rx FIFO 0
                           11  Timeout controlled by Rx FIFO 1

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TOCC_TOS_W                                              2U
#define DCAN_TOCC_TOS_M                                              0x00000006U
#define DCAN_TOCC_TOS_S                                              1U
/*

        Field:           TOP
        From..to bits:   16...31
        DefaultValue:    0xFFFF
        Access type:     read-write
        Description:     Timeout Period. Start value of the Timeout Counter (down-counter). Configures the Timeout Period.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TOCC_TOP_W                                              16U
#define DCAN_TOCC_TOP_M                                              0xFFFF0000U
#define DCAN_TOCC_TOP_S                                              16U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TOCV
    Offset name:         DCAN_O_TOCV
    Relative address:    0x2C
    Description:         MCAN Timeout Counter Value
    Default Value:       0x0000FFFF

        Field:           TOC
        From..to bits:   0...15
        DefaultValue:    0xFFFF
        Access type:     read-write
        Description:     Timeout Counter. The Timeout Counter is decremented in multiples of CAN bit times, (1...16), depending on the configuration of TSCC.TCP. When decremented to zero, interrupt flag IR.TOO is set and the Timeout Counter is stopped. Start and reset/restart conditions are configured via TOCC.TOS.

*/
#define DCAN_TOCV_TOC_W                                              16U
#define DCAN_TOCV_TOC_M                                              0x0000FFFFU
#define DCAN_TOCV_TOC_S                                              0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ECR
    Offset name:         DCAN_O_ECR
    Relative address:    0x40
    Description:         MCAN Error Counter Register
    Default Value:       0x00000000

        Field:           TEC
        From..to bits:   0...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmit Error Counter. Actual state of the Transmit Error Counter, values between 0 and 255.

                         Note: When CCCR.ASM is set, the CAN protocol controller does not increment TEC and REC when a CAN protocol error is detected, but CEL is still incremented.

*/
#define DCAN_ECR_TEC_W                                               8U
#define DCAN_ECR_TEC_M                                               0x000000FFU
#define DCAN_ECR_TEC_S                                               0U
/*

        Field:           REC
        From..to bits:   8...14
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Receive Error Counter. Actual state of the Receive Error Counter, values between 0 and 127.

                         Note: When CCCR.ASM is set, the CAN protocol controller does not increment TEC and REC when a CAN protocol error is detected, but CEL is still incremented.

*/
#define DCAN_ECR_REC_W                                               7U
#define DCAN_ECR_REC_M                                               0x00007F00U
#define DCAN_ECR_REC_S                                               8U
/*

        Field:           RP
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Receive Error Passive
                           0  The Receive Error Counter is below the error passive level of 128
                           1  The Receive Error Counter has reached the error passive level of 128

*/
#define DCAN_ECR_RP                                                  0x00008000U
#define DCAN_ECR_RP_M                                                0x00008000U
#define DCAN_ECR_RP_S                                                15U
/*

        Field:           CEL
        From..to bits:   16...23
        DefaultValue:    0x0
        Access type:     read-only
        Description:     CAN Error Logging. The counter is incremented each time when a CAN protocol error causes the Transmit Error Counter or the Receive Error Counter to be incremented. It is reset by read access to CEL. The counter stops at 0xFF; the next increment of TEC or REC sets interrupt flag IR.ELO.

                         Note: When CCCR.ASM is set, the CAN protocol controller does not increment TEC and REC when a CAN protocol error is detected, but CEL is still incremented.

*/
#define DCAN_ECR_CEL_W                                               8U
#define DCAN_ECR_CEL_M                                               0x00FF0000U
#define DCAN_ECR_CEL_S                                               16U


/*-----------------------------------REGISTER------------------------------------
    Register name:       PSR
    Offset name:         DCAN_O_PSR
    Relative address:    0x44
    Description:         MCAN Protocol Status Register
    Default Value:       0x00000707

        Field:           LEC
        From..to bits:   0...2
        DefaultValue:    0x7
        Access type:     read-only
        Description:     Last Error Code. The LEC indicates the type of the last error to occur on the CAN bus. This field will be cleared to '0' when a message has been transferred (reception or transmission) without error.
                           0  No Error: No error occurred since LEC has been reset by successful reception or transmission.
                           1  Stuff Error: More than 5 equal bits in a sequence have occurred in a part of a received message where this is not allowed.
                           2  Form Error: A fixed format part of a received frame has the wrong format.
                           3  AckError: The message transmitted by the MCAN was not acknowledged by another node.
                           4  Bit1Error: During the transmission of a message (with the exception of the arbitration field), the device wanted to send a recessive level (bit of logical value '1'), but the monitored bus value was dominant.
                           5  Bit0Error: During the transmission of a message (or acknowledge bit, or active error flag, or overload flag), the device wanted to send a dominant level (data or identifier bit logical value '0'), but the monitored bus value was recessive. During Bus_Off recovery this status is set each time a sequence of 11 recessive bits has been monitored. This enables the CPU to monitor the proceeding of the Bus_Off recovery sequence (indicating the bus is not stuck at dominant or continuously disturbed).
                           6  CRCError: The CRC check sum of a received message was incorrect. The CRC of an incoming message does not match with the CRC calculated from the received data.
                           7  NoChange: Any read access to the Protocol Status Register re-initializes the LEC to '7'. When the LEC shows the value '7', no CAN bus event was detected since the last CPU read access to the Protocol Status Register.

                         Note: When a frame in CAN FD format has reached the data phase with BRS flag set, the next CAN event (error or valid frame) will be shown in DLEC instead of LEC. An error in a fixed stuff bit of a CAN FD CRC sequence will be shown as a Form Error, not Stuff Error. Note: The Bus_Off recovery sequence (see ISO 11898-1:2015) cannot be shortened by setting or resetting CCCR.INIT. If the device goes Bus_Off, it will set CCCR.INIT of its own accord, stopping all bus activities. Once CCCR.INIT has been cleared by the CPU, the device will then wait for 129 occurrences of Bus Idle (129 * 11 consecutive recessive bits) before resuming normal operation. At the end of the Bus_Off recovery sequence, the Error Management Counters will be reset. During the waiting time after the resetting of CCCR.INIT, each time a sequence of 11 recessive bits has been monitored, a Bit0Error code is written to PSR.LEC, enabling the CPU to readily check up whether the CAN bus is stuck at dominant or continuously disturbed and to monitor the Bus_Off recovery sequence. ECR.REC is used to count these sequences.

*/
#define DCAN_PSR_LEC_W                                               3U
#define DCAN_PSR_LEC_M                                               0x00000007U
#define DCAN_PSR_LEC_S                                               0U
/*

        Field:           ACT
        From..to bits:   3...4
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Node Activity.  Monitors the module's CAN communication state.
                           00  Synchronizing - node is synchronizing on CAN communication
                           01  Idle - node is neither receiver nor transmitter
                           10  Receiver - node is operating as receiver
                           11  Transmitter - node is operating as transmitter

                         Note: ACT is set to "00" by a Protocol Exception Event.

*/
#define DCAN_PSR_ACT_W                                               2U
#define DCAN_PSR_ACT_M                                               0x00000018U
#define DCAN_PSR_ACT_S                                               3U
/*

        Field:           EP
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Error Passive
                           0  The M_CAN is in the Error_Active state. It normally takes part in bus communication and sends an active error flag when an error has been detected
                           1  The M_CAN is in the Error_Passive state

*/
#define DCAN_PSR_EP                                                  0x00000020U
#define DCAN_PSR_EP_M                                                0x00000020U
#define DCAN_PSR_EP_S                                                5U
/*

        Field:           EW
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Warning Status
                           0  Both error counters are below the Error_Warning limit of 96
                           1  At least one of error counter has reached the Error_Warning limit of 96

*/
#define DCAN_PSR_EW                                                  0x00000040U
#define DCAN_PSR_EW_M                                                0x00000040U
#define DCAN_PSR_EW_S                                                6U
/*

        Field:           BO
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Bus_Off Status
                           0  The M_CAN is not Bus_Off
                           1  The M_CAN is in Bus_Off state

*/
#define DCAN_PSR_BO                                                  0x00000080U
#define DCAN_PSR_BO_M                                                0x00000080U
#define DCAN_PSR_BO_S                                                7U
/*

        Field:           DLEC
        From..to bits:   8...10
        DefaultValue:    0x7
        Access type:     read-only
        Description:     Data Phase Last Error Code. Type of last error that occurred in the data phase of a CAN FD format frame with its BRS flag set. Coding is the same as for LEC. This field will be cleared to zero when a CAN FD format frame with its BRS flag set has been transferred (reception or transmission) without error.

*/
#define DCAN_PSR_DLEC_W                                              3U
#define DCAN_PSR_DLEC_M                                              0x00000700U
#define DCAN_PSR_DLEC_S                                              8U
/*

        Field:           RESI
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-only
        Description:     ESI Flag of Last Received CAN FD Message. This bit is set together with RFDF, independent of acceptance filtering.
                           0  Last received CAN FD message did not have its ESI flag set
                           1  Last received CAN FD message had its ESI flag set

*/
#define DCAN_PSR_RESI                                                0x00000800U
#define DCAN_PSR_RESI_M                                              0x00000800U
#define DCAN_PSR_RESI_S                                              11U
/*

        Field:           RBRS
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-only
        Description:     BRS Flag of Last Received CAN FD Message. This bit is set together with RFDF, independent of acceptance filtering.
                           0  Last received CAN FD message did not have its BRS flag set
                           1  Last received CAN FD message had its BRS flag set

*/
#define DCAN_PSR_RBRS                                                0x00001000U
#define DCAN_PSR_RBRS_M                                              0x00001000U
#define DCAN_PSR_RBRS_S                                              12U
/*

        Field:           RFDF
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Received a CAN FD Message.  This bit is set independent of acceptance filtering.
                           0  Since this bit was reset by the CPU, no CAN FD message has been received
                           1  Message in CAN FD format with FDF flag set has been received

*/
#define DCAN_PSR_RFDF                                                0x00002000U
#define DCAN_PSR_RFDF_M                                              0x00002000U
#define DCAN_PSR_RFDF_S                                              13U
/*

        Field:           PXE
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Protocol Exception Event
                           0  No protocol exception event occurred since last read access
                           1  Protocol exception event occurred

*/
#define DCAN_PSR_PXE                                                 0x00004000U
#define DCAN_PSR_PXE_M                                               0x00004000U
#define DCAN_PSR_PXE_S                                               14U
/*

        Field:           TDCV
        From..to bits:   16...22
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmitter Delay Compensation Value. Position of the secondary sample point, defined by the sum of the measured delay from the internal CAN TX signal to the internal CAN RX signal and TDCR.TDCO. The SSP position is, in the data phase, the number of mtq between the start of the transmitted bit and the secondary sample point. Valid values are 0 to 127 mtq.

*/
#define DCAN_PSR_TDCV_W                                              7U
#define DCAN_PSR_TDCV_M                                              0x007F0000U
#define DCAN_PSR_TDCV_S                                              16U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TDCR
    Offset name:         DCAN_O_TDCR
    Relative address:    0x48
    Description:         MCAN Transmitter Delay Compensation Register
    Default Value:       0x00000000

        Field:           TDCF
        From..to bits:   0...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmitter Delay Compensation Filter Window Length. Defines the minimum value for the SSP position, dominant edges on the internal CAN RX signal that would result in an earlier SSP position are ignored for transmitter delay measurement. The feature is enabled when TDCF is configured to a value greater than TDCO. Valid values are 0 to 127 mtq.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TDCR_TDCF_W                                             7U
#define DCAN_TDCR_TDCF_M                                             0x0000007FU
#define DCAN_TDCR_TDCF_S                                             0U
/*

        Field:           TDCO
        From..to bits:   8...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmitter Delay Compensation Offset. Offset value defining the distance between the measured delay from the internal CAN TX signal to the internal CAN RX signal and the secondary sample point. Valid values are 0 to 127 mtq.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TDCR_TDCO_W                                             7U
#define DCAN_TDCR_TDCO_M                                             0x00007F00U
#define DCAN_TDCR_TDCO_S                                             8U


/*-----------------------------------REGISTER------------------------------------
    Register name:       IR
    Offset name:         DCAN_O_IR
    Relative address:    0x50
    Description:         The flags are set when one of the listed conditions is detected (edge-sensitive). The flags remain set until the Host clears them. Aflag is cleared by writing a '1' to the corresponding bit position. Writing a '0' has no effect. Ahard reset will clear the register. The configuration of IE controls whether an interrupt is generated. The configuration of ILS controls on which interrupt line an interrupt is signalled.
    Default Value:       0x80000000

        Field:           RF0N
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 New Message
                           0  No new message written to Rx FIFO 0
                           1  New message written to Rx FIFO 0

*/
#define DCAN_IR_RF0N                                                 0x00000001U
#define DCAN_IR_RF0N_M                                               0x00000001U
#define DCAN_IR_RF0N_S                                               0U
/*

        Field:           RF0W
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Watermark Reached
                           0  Rx FIFO 0 fill level below watermark
                           1  Rx FIFO 0 fill level reached watermark

*/
#define DCAN_IR_RF0W                                                 0x00000002U
#define DCAN_IR_RF0W_M                                               0x00000002U
#define DCAN_IR_RF0W_S                                               1U
/*

        Field:           RF0F
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Full
                           0  Rx FIFO 0 not full
                           1  Rx FIFO 0 full

*/
#define DCAN_IR_RF0F                                                 0x00000004U
#define DCAN_IR_RF0F_M                                               0x00000004U
#define DCAN_IR_RF0F_S                                               2U
/*

        Field:           RF0L
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Message Lost
                           0  No Rx FIFO 0 message lost
                           1  Rx FIFO 0 message lost, also set after write attempt to Rx FIFO 0 of size zero

*/
#define DCAN_IR_RF0L                                                 0x00000008U
#define DCAN_IR_RF0L_M                                               0x00000008U
#define DCAN_IR_RF0L_S                                               3U
/*

        Field:           RF1N
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 New Message
                           0  No new message written to Rx FIFO 1
                           1  New message written to Rx FIFO 1

*/
#define DCAN_IR_RF1N                                                 0x00000010U
#define DCAN_IR_RF1N_M                                               0x00000010U
#define DCAN_IR_RF1N_S                                               4U
/*

        Field:           RF1W
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Watermark Reached
                           0  Rx FIFO 1 fill level below watermark
                           1  Rx FIFO 1 fill level reached watermark

*/
#define DCAN_IR_RF1W                                                 0x00000020U
#define DCAN_IR_RF1W_M                                               0x00000020U
#define DCAN_IR_RF1W_S                                               5U
/*

        Field:           RF1F
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Full
                           0  Rx FIFO 1 not full
                           1  Rx FIFO 1 full

*/
#define DCAN_IR_RF1F                                                 0x00000040U
#define DCAN_IR_RF1F_M                                               0x00000040U
#define DCAN_IR_RF1F_S                                               6U
/*

        Field:           RF1L
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Message Lost
                           0  No Rx FIFO 1 message lost
                           1  Rx FIFO 1 message lost, also set after write attempt to Rx FIFO 1 of size zero

*/
#define DCAN_IR_RF1L                                                 0x00000080U
#define DCAN_IR_RF1L_M                                               0x00000080U
#define DCAN_IR_RF1L_S                                               7U
/*

        Field:           HPM
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     High Priority Message
                           0  No high priority message received
                           1  High priority message received

*/
#define DCAN_IR_HPM                                                  0x00000100U
#define DCAN_IR_HPM_M                                                0x00000100U
#define DCAN_IR_HPM_S                                                8U
/*

        Field:           TC
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Completed
                           0  No transmission completed
                           1  Transmission completed

*/
#define DCAN_IR_TC                                                   0x00000200U
#define DCAN_IR_TC_M                                                 0x00000200U
#define DCAN_IR_TC_S                                                 9U
/*

        Field:           TCF
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Cancellation Finished
                           0  No transmission cancellation finished
                           1  Transmission cancellation finished

*/
#define DCAN_IR_TCF                                                  0x00000400U
#define DCAN_IR_TCF_M                                                0x00000400U
#define DCAN_IR_TCF_S                                                10U
/*

        Field:           TFE
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx FIFO Empty
                           0  Tx FIFO non-empty
                           1  Tx FIFO empty

*/
#define DCAN_IR_TFE                                                  0x00000800U
#define DCAN_IR_TFE_M                                                0x00000800U
#define DCAN_IR_TFE_S                                                11U
/*

        Field:           TEFN
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO New Entry
                           0  Tx Event FIFO unchanged
                           1  Tx Handler wrote Tx Event FIFO element

*/
#define DCAN_IR_TEFN                                                 0x00001000U
#define DCAN_IR_TEFN_M                                               0x00001000U
#define DCAN_IR_TEFN_S                                               12U
/*

        Field:           TEFW
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO Watermark Reached
                           0  Tx Event FIFO fill level below watermark
                           1  Tx Event FIFO fill level reached watermark

*/
#define DCAN_IR_TEFW                                                 0x00002000U
#define DCAN_IR_TEFW_M                                               0x00002000U
#define DCAN_IR_TEFW_S                                               13U
/*

        Field:           TEFF
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO Full
                           0  Tx Event FIFO not full
                           1  Tx Event FIFO full

*/
#define DCAN_IR_TEFF                                                 0x00004000U
#define DCAN_IR_TEFF_M                                               0x00004000U
#define DCAN_IR_TEFF_S                                               14U
/*

        Field:           TEFL
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO Element Lost
                           0  No Tx Event FIFO element lost
                           1  Tx Event FIFO element lost, also set after write attempt to Tx Event FIFO of size zero

*/
#define DCAN_IR_TEFL                                                 0x00008000U
#define DCAN_IR_TEFL_M                                               0x00008000U
#define DCAN_IR_TEFL_S                                               15U
/*

        Field:           TSW
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Timestamp Wraparound
                           0  No timestamp counter wrap-around
                           1  Timestamp counter wrapped around

*/
#define DCAN_IR_TSW                                                  0x00010000U
#define DCAN_IR_TSW_M                                                0x00010000U
#define DCAN_IR_TSW_S                                                16U
/*

        Field:           MRAF
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message RAM Access Failure.  The flag is set, when the Rx Handler:
                           - has not completed acceptance filtering or storage of an accepted message until the arbitration field of the following message has been received. In this case acceptance filtering or message storage is aborted and the Rx Handler starts processing of the following message.
                           - was not able to write a message to the Message RAM. In this case message storage is aborted.

                         In both cases the FIFO put index is not updated resp. the New Data flag for a dedicated Rx Buffer is not set, a partly stored message is overwritten when the next message is stored to this location.

                         The flag is also set when the Tx Handler was not able to read a message from the Message RAM in time. In this case message transmission is aborted. In case of a Tx Handler access failure the MCAN is switched into Restricted Operation Mode. To leave Restricted Operation Mode, the Host CPU has to reset CCCR.ASM.
                           0  No Message RAM access failure occurred
                           1  Message RAM access failure occurred

*/
#define DCAN_IR_MRAF                                                 0x00020000U
#define DCAN_IR_MRAF_M                                               0x00020000U
#define DCAN_IR_MRAF_S                                               17U
/*

        Field:           TOO
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Timeout Occurred
                           0  No timeout
                           1  Timeout reached

*/
#define DCAN_IR_TOO                                                  0x00040000U
#define DCAN_IR_TOO_M                                                0x00040000U
#define DCAN_IR_TOO_S                                                18U
/*

        Field:           DRX
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message Stored to Dedicated Rx Buffer. The flag is set whenever a received message has been stored into a dedicated Rx Buffer.
                           0  No Rx Buffer updated
                           1  At least one received message stored into an Rx Buffer

*/
#define DCAN_IR_DRX                                                  0x00080000U
#define DCAN_IR_DRX_M                                                0x00080000U
#define DCAN_IR_DRX_S                                                19U
/*

        Field:           BEU
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Bit Error Uncorrected. Message RAM bit error detected, uncorrected. This bit is set when a double bit error is detected by the ECC aggregator attached to the Message RAM. An uncorrected Message RAM bit error sets CCCR.INIT to '1'. This is done to avoid transmission of corrupted data.
                           0  No bit error detected when reading from Message RAM
                           1  Bit error detected, uncorrected (e.g. parity logic)

*/
#define DCAN_IR_BEU                                                  0x00200000U
#define DCAN_IR_BEU_M                                                0x00200000U
#define DCAN_IR_BEU_S                                                21U
/*

        Field:           ELO
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Error Logging Overflow
                           0  CAN Error Logging Counter did not overflow
                           1  Overflow of CAN Error Logging Counter occurred

*/
#define DCAN_IR_ELO                                                  0x00400000U
#define DCAN_IR_ELO_M                                                0x00400000U
#define DCAN_IR_ELO_S                                                22U
/*

        Field:           EP
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Error Passive
                           0  Error_Passive status unchanged
                           1  Error_Passive status changed

*/
#define DCAN_IR_EP                                                   0x00800000U
#define DCAN_IR_EP_M                                                 0x00800000U
#define DCAN_IR_EP_S                                                 23U
/*

        Field:           EW
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Warning Status
                           0  Error_Warning status unchanged
                           1  Error_Warning status changed

*/
#define DCAN_IR_EW                                                   0x01000000U
#define DCAN_IR_EW_M                                                 0x01000000U
#define DCAN_IR_EW_S                                                 24U
/*

        Field:           BO
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Bus_Off Status
                           0  Bus_Off status unchanged
                           1  Bus_Off status changed

*/
#define DCAN_IR_BO                                                   0x02000000U
#define DCAN_IR_BO_M                                                 0x02000000U
#define DCAN_IR_BO_S                                                 25U
/*

        Field:           WDI
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Watchdog Interrupt
                           0  No Message RAM Watchdog event occurred
                           1  Message RAM Watchdog event due to missing READY

*/
#define DCAN_IR_WDI                                                  0x04000000U
#define DCAN_IR_WDI_M                                                0x04000000U
#define DCAN_IR_WDI_S                                                26U
/*

        Field:           PEA
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Protocol Error in Arbitration Phase (Nominal Bit Time is used)
                           0  No protocol error in arbitration phase
                           1  Protocol error in arbitration phase detected (PSR.LEC ? 0,7)

*/
#define DCAN_IR_PEA                                                  0x08000000U
#define DCAN_IR_PEA_M                                                0x08000000U
#define DCAN_IR_PEA_S                                                27U
/*

        Field:           PED
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Protocol Error in Data Phase (Data Bit Time is used)
                           0  No protocol error in data phase
                           1  Protocol error in data phase detected (PSR.DLEC ? 0,7)

*/
#define DCAN_IR_PED                                                  0x10000000U
#define DCAN_IR_PED_M                                                0x10000000U
#define DCAN_IR_PED_S                                                28U
/*

        Field:           ARA
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Access to Reserved Address
                           0  No access to reserved address occurred
                           1  Access to reserved address occurred

*/
#define DCAN_IR_ARA                                                  0x20000000U
#define DCAN_IR_ARA_M                                                0x20000000U
#define DCAN_IR_ARA_S                                                29U


/*-----------------------------------REGISTER------------------------------------
    Register name:       IE
    Offset name:         DCAN_O_IE
    Relative address:    0x54
    Description:         MCAN Interrupt Enable
    Default Value:       0x00000000

        Field:           RF0NE
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 New Message Enable

*/
#define DCAN_IE_RF0NE                                                0x00000001U
#define DCAN_IE_RF0NE_M                                              0x00000001U
#define DCAN_IE_RF0NE_S                                              0U
/*

        Field:           RF0WE
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Watermark Reached Enable

*/
#define DCAN_IE_RF0WE                                                0x00000002U
#define DCAN_IE_RF0WE_M                                              0x00000002U
#define DCAN_IE_RF0WE_S                                              1U
/*

        Field:           RF0FE
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Full Enable

*/
#define DCAN_IE_RF0FE                                                0x00000004U
#define DCAN_IE_RF0FE_M                                              0x00000004U
#define DCAN_IE_RF0FE_S                                              2U
/*

        Field:           RF0LE
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Message Lost Enable

*/
#define DCAN_IE_RF0LE                                                0x00000008U
#define DCAN_IE_RF0LE_M                                              0x00000008U
#define DCAN_IE_RF0LE_S                                              3U
/*

        Field:           RF1NE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 New Message Enable

*/
#define DCAN_IE_RF1NE                                                0x00000010U
#define DCAN_IE_RF1NE_M                                              0x00000010U
#define DCAN_IE_RF1NE_S                                              4U
/*

        Field:           RF1WE
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Watermark Reached Enable

*/
#define DCAN_IE_RF1WE                                                0x00000020U
#define DCAN_IE_RF1WE_M                                              0x00000020U
#define DCAN_IE_RF1WE_S                                              5U
/*

        Field:           RF1FE
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Full Enable

*/
#define DCAN_IE_RF1FE                                                0x00000040U
#define DCAN_IE_RF1FE_M                                              0x00000040U
#define DCAN_IE_RF1FE_S                                              6U
/*

        Field:           RF1LE
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Message Lost Enable

*/
#define DCAN_IE_RF1LE                                                0x00000080U
#define DCAN_IE_RF1LE_M                                              0x00000080U
#define DCAN_IE_RF1LE_S                                              7U
/*

        Field:           HPME
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     High Priority Message Enable

*/
#define DCAN_IE_HPME                                                 0x00000100U
#define DCAN_IE_HPME_M                                               0x00000100U
#define DCAN_IE_HPME_S                                               8U
/*

        Field:           TCE
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Completed Enable

*/
#define DCAN_IE_TCE                                                  0x00000200U
#define DCAN_IE_TCE_M                                                0x00000200U
#define DCAN_IE_TCE_S                                                9U
/*

        Field:           TCFE
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Cancellation Finished Enable

*/
#define DCAN_IE_TCFE                                                 0x00000400U
#define DCAN_IE_TCFE_M                                               0x00000400U
#define DCAN_IE_TCFE_S                                               10U
/*

        Field:           TFEE
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx FIFO Empty Enable

*/
#define DCAN_IE_TFEE                                                 0x00000800U
#define DCAN_IE_TFEE_M                                               0x00000800U
#define DCAN_IE_TFEE_S                                               11U
/*

        Field:           TEFNE
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO New Entry Enable

*/
#define DCAN_IE_TEFNE                                                0x00001000U
#define DCAN_IE_TEFNE_M                                              0x00001000U
#define DCAN_IE_TEFNE_S                                              12U
/*

        Field:           TEFWE
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO Watermark Reached Enable

*/
#define DCAN_IE_TEFWE                                                0x00002000U
#define DCAN_IE_TEFWE_M                                              0x00002000U
#define DCAN_IE_TEFWE_S                                              13U
/*

        Field:           TEFFE
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO Full Enable

*/
#define DCAN_IE_TEFFE                                                0x00004000U
#define DCAN_IE_TEFFE_M                                              0x00004000U
#define DCAN_IE_TEFFE_S                                              14U
/*

        Field:           TEFLE
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO Element Lost Enable

*/
#define DCAN_IE_TEFLE                                                0x00008000U
#define DCAN_IE_TEFLE_M                                              0x00008000U
#define DCAN_IE_TEFLE_S                                              15U
/*

        Field:           TSWE
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Timestamp Wraparound Enable

*/
#define DCAN_IE_TSWE                                                 0x00010000U
#define DCAN_IE_TSWE_M                                               0x00010000U
#define DCAN_IE_TSWE_S                                               16U
/*

        Field:           MRAFE
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message RAM Access Failure Enable

*/
#define DCAN_IE_MRAFE                                                0x00020000U
#define DCAN_IE_MRAFE_M                                              0x00020000U
#define DCAN_IE_MRAFE_S                                              17U
/*

        Field:           TOOE
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Timeout Occurred Enable

*/
#define DCAN_IE_TOOE                                                 0x00040000U
#define DCAN_IE_TOOE_M                                               0x00040000U
#define DCAN_IE_TOOE_S                                               18U
/*

        Field:           DRXE
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message Stored to Dedicated Rx Buffer Enable

*/
#define DCAN_IE_DRXE                                                 0x00080000U
#define DCAN_IE_DRXE_M                                               0x00080000U
#define DCAN_IE_DRXE_S                                               19U
/*

        Field:           BEUE
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Bit Error Uncorrected Enable

*/
#define DCAN_IE_BEUE                                                 0x00200000U
#define DCAN_IE_BEUE_M                                               0x00200000U
#define DCAN_IE_BEUE_S                                               21U
/*

        Field:           ELOE
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Error Logging Overflow Enable

*/
#define DCAN_IE_ELOE                                                 0x00400000U
#define DCAN_IE_ELOE_M                                               0x00400000U
#define DCAN_IE_ELOE_S                                               22U
/*

        Field:           EPE
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Error Passive Enable

*/
#define DCAN_IE_EPE                                                  0x00800000U
#define DCAN_IE_EPE_M                                                0x00800000U
#define DCAN_IE_EPE_S                                                23U
/*

        Field:           EWE
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Warning Status Enable

*/
#define DCAN_IE_EWE                                                  0x01000000U
#define DCAN_IE_EWE_M                                                0x01000000U
#define DCAN_IE_EWE_S                                                24U
/*

        Field:           BOE
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Bus_Off Status Enable

*/
#define DCAN_IE_BOE                                                  0x02000000U
#define DCAN_IE_BOE_M                                                0x02000000U
#define DCAN_IE_BOE_S                                                25U
/*

        Field:           WDIE
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Watchdog Interrupt Enable

*/
#define DCAN_IE_WDIE                                                 0x04000000U
#define DCAN_IE_WDIE_M                                               0x04000000U
#define DCAN_IE_WDIE_S                                               26U
/*

        Field:           PEAE
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Protocol Error in Arbitration Phase Enable

*/
#define DCAN_IE_PEAE                                                 0x08000000U
#define DCAN_IE_PEAE_M                                               0x08000000U
#define DCAN_IE_PEAE_S                                               27U
/*

        Field:           PEDE
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Protocol Error in Data Phase Enable

*/
#define DCAN_IE_PEDE                                                 0x10000000U
#define DCAN_IE_PEDE_M                                               0x10000000U
#define DCAN_IE_PEDE_S                                               28U
/*

        Field:           ARAE
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Access to Reserved Address Enable

*/
#define DCAN_IE_ARAE                                                 0x20000000U
#define DCAN_IE_ARAE_M                                               0x20000000U
#define DCAN_IE_ARAE_S                                               29U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ILS
    Offset name:         DCAN_O_ILS
    Relative address:    0x58
    Description:         The Interrupt Line Select register assigns an interrupt generated by a specific interrupt flag from the Interrupt Register to one of the two module interrupt lines. For interrupt generation the respective interrupt line has to be enabled via ILE.EINT0 and ILE.EINT1.
    Default Value:       0x00000000

        Field:           RF0NL
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 New Message Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_RF0NL                                               0x00000001U
#define DCAN_ILS_RF0NL_M                                             0x00000001U
#define DCAN_ILS_RF0NL_S                                             0U
/*

        Field:           RF0WL
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Watermark Reached Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_RF0WL                                               0x00000002U
#define DCAN_ILS_RF0WL_M                                             0x00000002U
#define DCAN_ILS_RF0WL_S                                             1U
/*

        Field:           RF0FL
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Full Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_RF0FL                                               0x00000004U
#define DCAN_ILS_RF0FL_M                                             0x00000004U
#define DCAN_ILS_RF0FL_S                                             2U
/*

        Field:           RF0LL
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Message Lost Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_RF0LL                                               0x00000008U
#define DCAN_ILS_RF0LL_M                                             0x00000008U
#define DCAN_ILS_RF0LL_S                                             3U
/*

        Field:           RF1NL
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 New Message Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_RF1NL                                               0x00000010U
#define DCAN_ILS_RF1NL_M                                             0x00000010U
#define DCAN_ILS_RF1NL_S                                             4U
/*

        Field:           RF1WL
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Watermark Reached Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_RF1WL                                               0x00000020U
#define DCAN_ILS_RF1WL_M                                             0x00000020U
#define DCAN_ILS_RF1WL_S                                             5U
/*

        Field:           RF1FL
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Full Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_RF1FL                                               0x00000040U
#define DCAN_ILS_RF1FL_M                                             0x00000040U
#define DCAN_ILS_RF1FL_S                                             6U
/*

        Field:           RF1LL
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Message Lost Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_RF1LL                                               0x00000080U
#define DCAN_ILS_RF1LL_M                                             0x00000080U
#define DCAN_ILS_RF1LL_S                                             7U
/*

        Field:           HPML
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     High Priority Message Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_HPML                                                0x00000100U
#define DCAN_ILS_HPML_M                                              0x00000100U
#define DCAN_ILS_HPML_S                                              8U
/*

        Field:           TCL
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Completed Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_TCL                                                 0x00000200U
#define DCAN_ILS_TCL_M                                               0x00000200U
#define DCAN_ILS_TCL_S                                               9U
/*

        Field:           TCFL
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Cancellation Finished Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_TCFL                                                0x00000400U
#define DCAN_ILS_TCFL_M                                              0x00000400U
#define DCAN_ILS_TCFL_S                                              10U
/*

        Field:           TFEL
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx FIFO Empty Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_TFEL                                                0x00000800U
#define DCAN_ILS_TFEL_M                                              0x00000800U
#define DCAN_ILS_TFEL_S                                              11U
/*

        Field:           TEFNL
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO New Entry Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_TEFNL                                               0x00001000U
#define DCAN_ILS_TEFNL_M                                             0x00001000U
#define DCAN_ILS_TEFNL_S                                             12U
/*

        Field:           TEFWL
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO Watermark Reached Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_TEFWL                                               0x00002000U
#define DCAN_ILS_TEFWL_M                                             0x00002000U
#define DCAN_ILS_TEFWL_S                                             13U
/*

        Field:           TEFFL
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO Full Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_TEFFL                                               0x00004000U
#define DCAN_ILS_TEFFL_M                                             0x00004000U
#define DCAN_ILS_TEFFL_S                                             14U
/*

        Field:           TEFLL
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Event FIFO Element Lost Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_TEFLL                                               0x00008000U
#define DCAN_ILS_TEFLL_M                                             0x00008000U
#define DCAN_ILS_TEFLL_S                                             15U
/*

        Field:           TSWL
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Timestamp Wraparound Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_TSWL                                                0x00010000U
#define DCAN_ILS_TSWL_M                                              0x00010000U
#define DCAN_ILS_TSWL_S                                              16U
/*

        Field:           MRAFL
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message RAM Access Failure Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_MRAFL                                               0x00020000U
#define DCAN_ILS_MRAFL_M                                             0x00020000U
#define DCAN_ILS_MRAFL_S                                             17U
/*

        Field:           TOOL
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Timeout Occurred Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_TOOL                                                0x00040000U
#define DCAN_ILS_TOOL_M                                              0x00040000U
#define DCAN_ILS_TOOL_S                                              18U
/*

        Field:           DRXL
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message Stored to Dedicated Rx Buffer Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_DRXL                                                0x00080000U
#define DCAN_ILS_DRXL_M                                              0x00080000U
#define DCAN_ILS_DRXL_S                                              19U
/*

        Field:           BECL
        From..to bits:   20...20
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Bit Error Corrected Line

                         A separate interrupt line reserved for corrected bit errors is provided via the MCAN_ERROR_REGS. It advised for the user to use these registers and leave the MCAN_IE.BECE bit cleared to '0' (disabled), thereby relegating this bit to not applicable.

*/
#define DCAN_ILS_BECL                                                0x00100000U
#define DCAN_ILS_BECL_M                                              0x00100000U
#define DCAN_ILS_BECL_S                                              20U
/*

        Field:           BEUL
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Bit Error Uncorrected Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_BEUL                                                0x00200000U
#define DCAN_ILS_BEUL_M                                              0x00200000U
#define DCAN_ILS_BEUL_S                                              21U
/*

        Field:           ELOL
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Error Logging Overflow Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_ELOL                                                0x00400000U
#define DCAN_ILS_ELOL_M                                              0x00400000U
#define DCAN_ILS_ELOL_S                                              22U
/*

        Field:           EPL
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Error Passive Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_EPL                                                 0x00800000U
#define DCAN_ILS_EPL_M                                               0x00800000U
#define DCAN_ILS_EPL_S                                               23U
/*

        Field:           EWL
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Warning Status Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_EWL                                                 0x01000000U
#define DCAN_ILS_EWL_M                                               0x01000000U
#define DCAN_ILS_EWL_S                                               24U
/*

        Field:           BOL
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Bus_Off Status Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_BOL                                                 0x02000000U
#define DCAN_ILS_BOL_M                                               0x02000000U
#define DCAN_ILS_BOL_S                                               25U
/*

        Field:           WDIL
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Watchdog Interrupt Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_WDIL                                                0x04000000U
#define DCAN_ILS_WDIL_M                                              0x04000000U
#define DCAN_ILS_WDIL_S                                              26U
/*

        Field:           PEAL
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Protocol Error in Arbitration Phase Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_PEAL                                                0x08000000U
#define DCAN_ILS_PEAL_M                                              0x08000000U
#define DCAN_ILS_PEAL_S                                              27U
/*

        Field:           PEDL
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Protocol Error in Data Phase Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_PEDL                                                0x10000000U
#define DCAN_ILS_PEDL_M                                              0x10000000U
#define DCAN_ILS_PEDL_S                                              28U
/*

        Field:           ARAL
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Access to Reserved Address Line
                           0  Interrupt source is assigned to Interrupt Line 0
                           1  Interrupt source is assigned to Interrupt Line 1

*/
#define DCAN_ILS_ARAL                                                0x20000000U
#define DCAN_ILS_ARAL_M                                              0x20000000U
#define DCAN_ILS_ARAL_S                                              29U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ILE
    Offset name:         DCAN_O_ILE
    Relative address:    0x5C
    Description:         MCAN Interrupt Line Enable
    Default Value:       0x00000000

        Field:           EINT0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Enable Interrupt Line 0
                           0  Interrupt Line 0 is disabled
                           1  Interrupt Line 0 is enabled

*/
#define DCAN_ILE_EINT0                                               0x00000001U
#define DCAN_ILE_EINT0_M                                             0x00000001U
#define DCAN_ILE_EINT0_S                                             0U
/*

        Field:           EINT1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Enable Interrupt Line 1
                           0  Interrupt Line 1 is disabled
                           1  Interrupt Line 1 is enabled

*/
#define DCAN_ILE_EINT1                                               0x00000002U
#define DCAN_ILE_EINT1_M                                             0x00000002U
#define DCAN_ILE_EINT1_S                                             1U


/*-----------------------------------REGISTER------------------------------------
    Register name:       GFC
    Offset name:         DCAN_O_GFC
    Relative address:    0x80
    Description:         MCAN Global Filter Configuration
    Default Value:       0x00000000

        Field:           RRFE
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Reject Remote Frames Extended
                           0  Filter remote frames with 29-bit extended IDs
                           1  Reject all remote frames with 29-bit extended IDs

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_GFC_RRFE                                                0x00000001U
#define DCAN_GFC_RRFE_M                                              0x00000001U
#define DCAN_GFC_RRFE_S                                              0U
/*

        Field:           RRFS
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Reject Remote Frames Standard
                           0  Filter remote frames with 11-bit standard IDs
                           1  Reject all remote frames with 11-bit standard IDs

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_GFC_RRFS                                                0x00000002U
#define DCAN_GFC_RRFS_M                                              0x00000002U
#define DCAN_GFC_RRFS_S                                              1U
/*

        Field:           ANFE
        From..to bits:   2...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Accept Non-matching Frames Extended. Defines how received messages with 29-bit IDs that do not match any element of the filter list are treated.
                           00  Accept in Rx FIFO 0
                           01  Accept in Rx FIFO 1
                           10  Reject
                           11  Reject

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_GFC_ANFE_W                                              2U
#define DCAN_GFC_ANFE_M                                              0x0000000CU
#define DCAN_GFC_ANFE_S                                              2U
/*

        Field:           ANFS
        From..to bits:   4...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Accept Non-matching Frames Standard. Defines how received messages with 11-bit IDs that do not match any element of the filter list are treated.
                           00  Accept in Rx FIFO 0
                           01  Accept in Rx FIFO 1
                           10  Reject
                           11  Reject

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_GFC_ANFS_W                                              2U
#define DCAN_GFC_ANFS_M                                              0x00000030U
#define DCAN_GFC_ANFS_S                                              4U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SIDFC
    Offset name:         DCAN_O_SIDFC
    Relative address:    0x84
    Description:         MCAN Standard ID Filter Configuration
    Default Value:       0x00000000

        Field:           FLSSA
        From..to bits:   2...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Filter List Standard Start Address. Start address of standard Message ID filter list (32-bit word address).

*/
#define DCAN_SIDFC_FLSSA_W                                           14U
#define DCAN_SIDFC_FLSSA_M                                           0x0000FFFCU
#define DCAN_SIDFC_FLSSA_S                                           2U
/*

        Field:           LSS
        From..to bits:   16...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     List Size Standard
                           0        No standard Message ID filter
                           1-128  Number of standard Message ID filter elements
                           >128   Values greater than 128 are interpreted as 128

*/
#define DCAN_SIDFC_LSS_W                                             8U
#define DCAN_SIDFC_LSS_M                                             0x00FF0000U
#define DCAN_SIDFC_LSS_S                                             16U


/*-----------------------------------REGISTER------------------------------------
    Register name:       XIDFC
    Offset name:         DCAN_O_XIDFC
    Relative address:    0x88
    Description:         MCAN Extended ID Filter Configuration
    Default Value:       0x00000000

        Field:           FLESA
        From..to bits:   2...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Filter List Extended Start Address. Start address of extended Message ID filter list (32-bit word address).

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_XIDFC_FLESA_W                                           14U
#define DCAN_XIDFC_FLESA_M                                           0x0000FFFCU
#define DCAN_XIDFC_FLESA_S                                           2U
/*

        Field:           LSE
        From..to bits:   16...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     List Size Extended
                           0     No extended Message ID filter
                           1-64 Number of extended Message ID filter elements
                           >64  Values greater than 64 are interpreted as 64

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_XIDFC_LSE_W                                             7U
#define DCAN_XIDFC_LSE_M                                             0x007F0000U
#define DCAN_XIDFC_LSE_S                                             16U


/*-----------------------------------REGISTER------------------------------------
    Register name:       XIDAM
    Offset name:         DCAN_O_XIDAM
    Relative address:    0x90
    Description:         MCAN Extended ID and Mask
    Default Value:       0x1FFFFFFF

        Field:           EIDM
        From..to bits:   0...28
        DefaultValue:    0x1FFFFFFF
        Access type:     read-write
        Description:     Extended ID Mask. For acceptance filtering of extended frames the Extended ID AND Mask is ANDed with the Message ID of a received frame. Intended for masking of 29-bit IDs in SAE J1939. With the reset value of all bits set to one the mask is not active.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_XIDAM_EIDM_W                                            29U
#define DCAN_XIDAM_EIDM_M                                            0x1FFFFFFFU
#define DCAN_XIDAM_EIDM_S                                            0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       HPMS
    Offset name:         DCAN_O_HPMS
    Relative address:    0x94
    Description:         This register is updated every time a Message ID filter element configured to generate a priority event matches. This can be used to monitor the status of incoming high priority messages and to enable fast access to these messages.
    Default Value:       0x00000000

        Field:           BIDX
        From..to bits:   0...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Buffer Index. Index of Rx FIFO element to which the message was stored. Only valid when MSI(1) = '1'.

*/
#define DCAN_HPMS_BIDX_W                                             6U
#define DCAN_HPMS_BIDX_M                                             0x0000003FU
#define DCAN_HPMS_BIDX_S                                             0U
/*

        Field:           MSI
        From..to bits:   6...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Message Storage Indicator
                           00  No FIFO selected
                           01  FIFO message lost
                           10  Message stored in FIFO 0
                           11  Message stored in FIFO 1

*/
#define DCAN_HPMS_MSI_W                                              2U
#define DCAN_HPMS_MSI_M                                              0x000000C0U
#define DCAN_HPMS_MSI_S                                              6U
/*

        Field:           FIDX
        From..to bits:   8...14
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Filter Index. Index of matching filter element. Range is 0 to SIDFC.LSS - 1 resp. XIDFC.LSE - 1.

*/
#define DCAN_HPMS_FIDX_W                                             7U
#define DCAN_HPMS_FIDX_M                                             0x00007F00U
#define DCAN_HPMS_FIDX_S                                             8U
/*

        Field:           FLST
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Filter List. Indicates the filter list of the matching filter element.
                           0  Standard Filter List
                           1  Extended Filter List

*/
#define DCAN_HPMS_FLST                                               0x00008000U
#define DCAN_HPMS_FLST_M                                             0x00008000U
#define DCAN_HPMS_FLST_S                                             15U


/*-----------------------------------REGISTER------------------------------------
    Register name:       NDAT1
    Offset name:         DCAN_O_NDAT1
    Relative address:    0x98
    Description:         MCAN New Data 1
    Default Value:       0x00000000

        Field:           ND0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 0
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND0                                               0x00000001U
#define DCAN_NDAT1_ND0_M                                             0x00000001U
#define DCAN_NDAT1_ND0_S                                             0U
/*

        Field:           ND1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 1
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND1                                               0x00000002U
#define DCAN_NDAT1_ND1_M                                             0x00000002U
#define DCAN_NDAT1_ND1_S                                             1U
/*

        Field:           ND2
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 2
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND2                                               0x00000004U
#define DCAN_NDAT1_ND2_M                                             0x00000004U
#define DCAN_NDAT1_ND2_S                                             2U
/*

        Field:           ND3
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 3
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND3                                               0x00000008U
#define DCAN_NDAT1_ND3_M                                             0x00000008U
#define DCAN_NDAT1_ND3_S                                             3U
/*

        Field:           ND4
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 4
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND4                                               0x00000010U
#define DCAN_NDAT1_ND4_M                                             0x00000010U
#define DCAN_NDAT1_ND4_S                                             4U
/*

        Field:           ND5
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 5
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND5                                               0x00000020U
#define DCAN_NDAT1_ND5_M                                             0x00000020U
#define DCAN_NDAT1_ND5_S                                             5U
/*

        Field:           ND6
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 6
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND6                                               0x00000040U
#define DCAN_NDAT1_ND6_M                                             0x00000040U
#define DCAN_NDAT1_ND6_S                                             6U
/*

        Field:           ND7
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 7
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND7                                               0x00000080U
#define DCAN_NDAT1_ND7_M                                             0x00000080U
#define DCAN_NDAT1_ND7_S                                             7U
/*

        Field:           ND8
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 8
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND8                                               0x00000100U
#define DCAN_NDAT1_ND8_M                                             0x00000100U
#define DCAN_NDAT1_ND8_S                                             8U
/*

        Field:           ND9
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 9
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND9                                               0x00000200U
#define DCAN_NDAT1_ND9_M                                             0x00000200U
#define DCAN_NDAT1_ND9_S                                             9U
/*

        Field:           ND10
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 10
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND10                                              0x00000400U
#define DCAN_NDAT1_ND10_M                                            0x00000400U
#define DCAN_NDAT1_ND10_S                                            10U
/*

        Field:           ND11
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 11
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND11                                              0x00000800U
#define DCAN_NDAT1_ND11_M                                            0x00000800U
#define DCAN_NDAT1_ND11_S                                            11U
/*

        Field:           ND12
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 12
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND12                                              0x00001000U
#define DCAN_NDAT1_ND12_M                                            0x00001000U
#define DCAN_NDAT1_ND12_S                                            12U
/*

        Field:           ND13
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 13
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND13                                              0x00002000U
#define DCAN_NDAT1_ND13_M                                            0x00002000U
#define DCAN_NDAT1_ND13_S                                            13U
/*

        Field:           ND14
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 14
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND14                                              0x00004000U
#define DCAN_NDAT1_ND14_M                                            0x00004000U
#define DCAN_NDAT1_ND14_S                                            14U
/*

        Field:           ND15
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 15
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND15                                              0x00008000U
#define DCAN_NDAT1_ND15_M                                            0x00008000U
#define DCAN_NDAT1_ND15_S                                            15U
/*

        Field:           ND16
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 16
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND16                                              0x00010000U
#define DCAN_NDAT1_ND16_M                                            0x00010000U
#define DCAN_NDAT1_ND16_S                                            16U
/*

        Field:           ND17
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 17
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND17                                              0x00020000U
#define DCAN_NDAT1_ND17_M                                            0x00020000U
#define DCAN_NDAT1_ND17_S                                            17U
/*

        Field:           ND18
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 18
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND18                                              0x00040000U
#define DCAN_NDAT1_ND18_M                                            0x00040000U
#define DCAN_NDAT1_ND18_S                                            18U
/*

        Field:           ND19
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 19
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND19                                              0x00080000U
#define DCAN_NDAT1_ND19_M                                            0x00080000U
#define DCAN_NDAT1_ND19_S                                            19U
/*

        Field:           ND20
        From..to bits:   20...20
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 20
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND20                                              0x00100000U
#define DCAN_NDAT1_ND20_M                                            0x00100000U
#define DCAN_NDAT1_ND20_S                                            20U
/*

        Field:           ND21
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 21
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND21                                              0x00200000U
#define DCAN_NDAT1_ND21_M                                            0x00200000U
#define DCAN_NDAT1_ND21_S                                            21U
/*

        Field:           ND22
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 22
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND22                                              0x00400000U
#define DCAN_NDAT1_ND22_M                                            0x00400000U
#define DCAN_NDAT1_ND22_S                                            22U
/*

        Field:           ND23
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 23
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND23                                              0x00800000U
#define DCAN_NDAT1_ND23_M                                            0x00800000U
#define DCAN_NDAT1_ND23_S                                            23U
/*

        Field:           ND24
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 24
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND24                                              0x01000000U
#define DCAN_NDAT1_ND24_M                                            0x01000000U
#define DCAN_NDAT1_ND24_S                                            24U
/*

        Field:           ND25
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 25
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND25                                              0x02000000U
#define DCAN_NDAT1_ND25_M                                            0x02000000U
#define DCAN_NDAT1_ND25_S                                            25U
/*

        Field:           ND26
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 26
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND26                                              0x04000000U
#define DCAN_NDAT1_ND26_M                                            0x04000000U
#define DCAN_NDAT1_ND26_S                                            26U
/*

        Field:           ND27
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 27
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND27                                              0x08000000U
#define DCAN_NDAT1_ND27_M                                            0x08000000U
#define DCAN_NDAT1_ND27_S                                            27U
/*

        Field:           ND28
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 28
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND28                                              0x10000000U
#define DCAN_NDAT1_ND28_M                                            0x10000000U
#define DCAN_NDAT1_ND28_S                                            28U
/*

        Field:           ND29
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 29
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND29                                              0x20000000U
#define DCAN_NDAT1_ND29_M                                            0x20000000U
#define DCAN_NDAT1_ND29_S                                            29U
/*

        Field:           ND30
        From..to bits:   30...30
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 30
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND30                                              0x40000000U
#define DCAN_NDAT1_ND30_M                                            0x40000000U
#define DCAN_NDAT1_ND30_S                                            30U
/*

        Field:           ND31
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 31
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT1_ND31                                              0x80000000U
#define DCAN_NDAT1_ND31_M                                            0x80000000U
#define DCAN_NDAT1_ND31_S                                            31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       NDAT2
    Offset name:         DCAN_O_NDAT2
    Relative address:    0x9C
    Description:         MCAN New Data 2
    Default Value:       0x00000000

        Field:           ND32
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 32
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND32                                              0x00000001U
#define DCAN_NDAT2_ND32_M                                            0x00000001U
#define DCAN_NDAT2_ND32_S                                            0U
/*

        Field:           ND33
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 33
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND33                                              0x00000002U
#define DCAN_NDAT2_ND33_M                                            0x00000002U
#define DCAN_NDAT2_ND33_S                                            1U
/*

        Field:           ND34
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 34
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND34                                              0x00000004U
#define DCAN_NDAT2_ND34_M                                            0x00000004U
#define DCAN_NDAT2_ND34_S                                            2U
/*

        Field:           ND35
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 35
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND35                                              0x00000008U
#define DCAN_NDAT2_ND35_M                                            0x00000008U
#define DCAN_NDAT2_ND35_S                                            3U
/*

        Field:           ND36
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 36
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND36                                              0x00000010U
#define DCAN_NDAT2_ND36_M                                            0x00000010U
#define DCAN_NDAT2_ND36_S                                            4U
/*

        Field:           ND37
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 37
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND37                                              0x00000020U
#define DCAN_NDAT2_ND37_M                                            0x00000020U
#define DCAN_NDAT2_ND37_S                                            5U
/*

        Field:           ND38
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 38
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND38                                              0x00000040U
#define DCAN_NDAT2_ND38_M                                            0x00000040U
#define DCAN_NDAT2_ND38_S                                            6U
/*

        Field:           ND39
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 39
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND39                                              0x00000080U
#define DCAN_NDAT2_ND39_M                                            0x00000080U
#define DCAN_NDAT2_ND39_S                                            7U
/*

        Field:           ND40
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 40
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND40                                              0x00000100U
#define DCAN_NDAT2_ND40_M                                            0x00000100U
#define DCAN_NDAT2_ND40_S                                            8U
/*

        Field:           ND41
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 41
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND41                                              0x00000200U
#define DCAN_NDAT2_ND41_M                                            0x00000200U
#define DCAN_NDAT2_ND41_S                                            9U
/*

        Field:           ND42
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 42
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND42                                              0x00000400U
#define DCAN_NDAT2_ND42_M                                            0x00000400U
#define DCAN_NDAT2_ND42_S                                            10U
/*

        Field:           ND43
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 43
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND43                                              0x00000800U
#define DCAN_NDAT2_ND43_M                                            0x00000800U
#define DCAN_NDAT2_ND43_S                                            11U
/*

        Field:           ND44
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 44
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND44                                              0x00001000U
#define DCAN_NDAT2_ND44_M                                            0x00001000U
#define DCAN_NDAT2_ND44_S                                            12U
/*

        Field:           ND45
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 45
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND45                                              0x00002000U
#define DCAN_NDAT2_ND45_M                                            0x00002000U
#define DCAN_NDAT2_ND45_S                                            13U
/*

        Field:           ND46
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 46
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND46                                              0x00004000U
#define DCAN_NDAT2_ND46_M                                            0x00004000U
#define DCAN_NDAT2_ND46_S                                            14U
/*

        Field:           ND47
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 47
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND47                                              0x00008000U
#define DCAN_NDAT2_ND47_M                                            0x00008000U
#define DCAN_NDAT2_ND47_S                                            15U
/*

        Field:           ND48
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 48
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND48                                              0x00010000U
#define DCAN_NDAT2_ND48_M                                            0x00010000U
#define DCAN_NDAT2_ND48_S                                            16U
/*

        Field:           ND49
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 49
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND49                                              0x00020000U
#define DCAN_NDAT2_ND49_M                                            0x00020000U
#define DCAN_NDAT2_ND49_S                                            17U
/*

        Field:           ND50
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 50
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND50                                              0x00040000U
#define DCAN_NDAT2_ND50_M                                            0x00040000U
#define DCAN_NDAT2_ND50_S                                            18U
/*

        Field:           ND51
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 51
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND51                                              0x00080000U
#define DCAN_NDAT2_ND51_M                                            0x00080000U
#define DCAN_NDAT2_ND51_S                                            19U
/*

        Field:           ND52
        From..to bits:   20...20
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 52
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND52                                              0x00100000U
#define DCAN_NDAT2_ND52_M                                            0x00100000U
#define DCAN_NDAT2_ND52_S                                            20U
/*

        Field:           ND53
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 53
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND53                                              0x00200000U
#define DCAN_NDAT2_ND53_M                                            0x00200000U
#define DCAN_NDAT2_ND53_S                                            21U
/*

        Field:           ND54
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 54
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND54                                              0x00400000U
#define DCAN_NDAT2_ND54_M                                            0x00400000U
#define DCAN_NDAT2_ND54_S                                            22U
/*

        Field:           ND55
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 55
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND55                                              0x00800000U
#define DCAN_NDAT2_ND55_M                                            0x00800000U
#define DCAN_NDAT2_ND55_S                                            23U
/*

        Field:           ND56
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 56
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND56                                              0x01000000U
#define DCAN_NDAT2_ND56_M                                            0x01000000U
#define DCAN_NDAT2_ND56_S                                            24U
/*

        Field:           ND57
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 57
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND57                                              0x02000000U
#define DCAN_NDAT2_ND57_M                                            0x02000000U
#define DCAN_NDAT2_ND57_S                                            25U
/*

        Field:           ND58
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 58
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND58                                              0x04000000U
#define DCAN_NDAT2_ND58_M                                            0x04000000U
#define DCAN_NDAT2_ND58_S                                            26U
/*

        Field:           ND59
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 59
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND59                                              0x08000000U
#define DCAN_NDAT2_ND59_M                                            0x08000000U
#define DCAN_NDAT2_ND59_S                                            27U
/*

        Field:           ND60
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 60
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND60                                              0x10000000U
#define DCAN_NDAT2_ND60_M                                            0x10000000U
#define DCAN_NDAT2_ND60_S                                            28U
/*

        Field:           ND61
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 61
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND61                                              0x20000000U
#define DCAN_NDAT2_ND61_M                                            0x20000000U
#define DCAN_NDAT2_ND61_S                                            29U
/*

        Field:           ND62
        From..to bits:   30...30
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 62
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND62                                              0x40000000U
#define DCAN_NDAT2_ND62_M                                            0x40000000U
#define DCAN_NDAT2_ND62_S                                            30U
/*

        Field:           ND63
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     New Data RX Buffer 63
                           0  Rx Buffer not updated
                           1  Rx Buffer updated from new message

*/
#define DCAN_NDAT2_ND63                                              0x80000000U
#define DCAN_NDAT2_ND63_M                                            0x80000000U
#define DCAN_NDAT2_ND63_S                                            31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RXF0C
    Offset name:         DCAN_O_RXF0C
    Relative address:    0xA0
    Description:         MCAN Rx FIFO 0 Configuration
    Default Value:       0x00000000

        Field:           F0SA
        From..to bits:   2...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Start Address. Start address of Rx FIFO 0 in Message RAM (32-bit word address).

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RXF0C_F0SA_W                                            14U
#define DCAN_RXF0C_F0SA_M                                            0x0000FFFCU
#define DCAN_RXF0C_F0SA_S                                            2U
/*

        Field:           F0S
        From..to bits:   16...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Size. The Rx FIFO 0 elements are indexed from 0 to F0S-1.
                           0      No Rx FIFO 0
                           1-64  Number of Rx FIFO 0 elements
                           >64   Values greater than 64 are interpreted as 64

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RXF0C_F0S_W                                             7U
#define DCAN_RXF0C_F0S_M                                             0x007F0000U
#define DCAN_RXF0C_F0S_S                                             16U
/*

        Field:           F0WM
        From..to bits:   24...30
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Watermark
                           0      Watermark interrupt disabled
                           1-64  Level for Rx FIFO 0 watermark interrupt (IR.RF0W)
                           >64   Watermark interrupt disabled

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RXF0C_F0WM_W                                            7U
#define DCAN_RXF0C_F0WM_M                                            0x7F000000U
#define DCAN_RXF0C_F0WM_S                                            24U
/*

        Field:           F0OM
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     FIFO 0 Operation Mode. FIFO 0 can be operated in blocking or in overwrite mode.
                           0  FIFO 0 blocking mode
                           1  FIFO 0 overwrite mode

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RXF0C_F0OM                                              0x80000000U
#define DCAN_RXF0C_F0OM_M                                            0x80000000U
#define DCAN_RXF0C_F0OM_S                                            31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RXF0S
    Offset name:         DCAN_O_RXF0S
    Relative address:    0xA4
    Description:         MCAN Rx FIFO 0 Status
    Default Value:       0x00000000

        Field:           F0FL
        From..to bits:   0...6
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Rx FIFO 0 Fill Level. Number of elements stored in Rx FIFO 0, range 0 to 64.

*/
#define DCAN_RXF0S_F0FL_W                                            7U
#define DCAN_RXF0S_F0FL_M                                            0x0000007FU
#define DCAN_RXF0S_F0FL_S                                            0U
/*

        Field:           F0GI
        From..to bits:   8...13
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Rx FIFO 0 Get Index. Rx FIFO 0 read index pointer, range 0 to 63.

*/
#define DCAN_RXF0S_F0GI_W                                            6U
#define DCAN_RXF0S_F0GI_M                                            0x00003F00U
#define DCAN_RXF0S_F0GI_S                                            8U
/*

        Field:           F0PI
        From..to bits:   16...21
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Rx FIFO 0 Put Index. Rx FIFO 0 write index pointer, range 0 to 63.

*/
#define DCAN_RXF0S_F0PI_W                                            6U
#define DCAN_RXF0S_F0PI_M                                            0x003F0000U
#define DCAN_RXF0S_F0PI_S                                            16U
/*

        Field:           F0F
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Rx FIFO 0 Full
                           0  Rx FIFO 0 not full
                           1  Rx FIFO 0 full

*/
#define DCAN_RXF0S_F0F                                               0x01000000U
#define DCAN_RXF0S_F0F_M                                             0x01000000U
#define DCAN_RXF0S_F0F_S                                             24U
/*

        Field:           RF0L
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Rx FIFO 0 Message Lost. This bit is a copy of interrupt flag IR.RF0L. When IR.RF0L is reset, this bit is also reset.
                           0  No Rx FIFO 0 message lost
                           1  Rx FIFO 0 message lost, also set after write attempt to Rx FIFO 0 of size zero

                         Note: Overwriting the oldest message when RXF0C.F0OM = '1' will not set this flag.

*/
#define DCAN_RXF0S_RF0L                                              0x02000000U
#define DCAN_RXF0S_RF0L_M                                            0x02000000U
#define DCAN_RXF0S_RF0L_S                                            25U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RXF0A
    Offset name:         DCAN_O_RXF0A
    Relative address:    0xA8
    Description:         MCAN Rx FIFO 0 Acknowledge
    Default Value:       0x00000000

        Field:           F0AI
        From..to bits:   0...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Acknowledge Index. After the Host has read a message or a sequence of messages from Rx FIFO 0 it has to write the buffer index of the last element read from Rx FIFO 0 to F0AI. This will set the Rx FIFO 0 Get Index RXF0S.F0GI to F0AI + 1 and update the FIFO 0 Fill Level RXF0S.F0FL.

*/
#define DCAN_RXF0A_F0AI_W                                            6U
#define DCAN_RXF0A_F0AI_M                                            0x0000003FU
#define DCAN_RXF0A_F0AI_S                                            0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RXBC
    Offset name:         DCAN_O_RXBC
    Relative address:    0xAC
    Description:         MCAN Rx Buffer Configuration
    Default Value:       0x00000000

        Field:           RBSA
        From..to bits:   2...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx Buffer Start Address. Configures the start address of the Rx Buffers section in the Message RAM (32-bit word address).

                         +I466

*/
#define DCAN_RXBC_RBSA_W                                             14U
#define DCAN_RXBC_RBSA_M                                             0x0000FFFCU
#define DCAN_RXBC_RBSA_S                                             2U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RXF1C
    Offset name:         DCAN_O_RXF1C
    Relative address:    0xB0
    Description:         MCAN Rx FIFO 1 Configuration
    Default Value:       0x00000000

        Field:           F1SA
        From..to bits:   2...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Start Address Start address of Rx FIFO 1 in Message RAM (32-bit word address).

*/
#define DCAN_RXF1C_F1SA_W                                            14U
#define DCAN_RXF1C_F1SA_M                                            0x0000FFFCU
#define DCAN_RXF1C_F1SA_S                                            2U
/*

        Field:           F1S
        From..to bits:   16...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Size. The Rx FIFO 1 elements are indexed from 0 to F1S - 1.
                           0      No Rx FIFO 1
                           1-64  Number of Rx FIFO 1 elements
                           >64   Values greater than 64 are interpreted as 64

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RXF1C_F1S_W                                             7U
#define DCAN_RXF1C_F1S_M                                             0x007F0000U
#define DCAN_RXF1C_F1S_S                                             16U
/*

        Field:           F1WM
        From..to bits:   24...30
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Watermark
                           0      Watermark interrupt disabled
                           1-64  Level for Rx FIFO 1 watermark interrupt (IR.RF1W)
                           >64   Watermark interrupt disabled

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RXF1C_F1WM_W                                            7U
#define DCAN_RXF1C_F1WM_M                                            0x7F000000U
#define DCAN_RXF1C_F1WM_S                                            24U
/*

        Field:           F1OM
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     FIFO 1 Operation Mode. FIFO 1 can be operated in blocking or in overwrite mode.
                           0  FIFO 1 blocking mode
                           1  FIFO 1 overwrite mode

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RXF1C_F1OM                                              0x80000000U
#define DCAN_RXF1C_F1OM_M                                            0x80000000U
#define DCAN_RXF1C_F1OM_S                                            31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RXF1S
    Offset name:         DCAN_O_RXF1S
    Relative address:    0xB4
    Description:         MCAN Rx FIFO 1 Status
    Default Value:       0x00000000

        Field:           F1FL
        From..to bits:   0...6
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Rx FIFO 1 Fill Level. Number of elements stored in Rx FIFO 1, range 0 to 64.

*/
#define DCAN_RXF1S_F1FL_W                                            7U
#define DCAN_RXF1S_F1FL_M                                            0x0000007FU
#define DCAN_RXF1S_F1FL_S                                            0U
/*

        Field:           F1GI
        From..to bits:   8...13
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Rx FIFO 1 Get Index. Rx FIFO 1 read index pointer, range 0 to 63.

*/
#define DCAN_RXF1S_F1GI_W                                            6U
#define DCAN_RXF1S_F1GI_M                                            0x00003F00U
#define DCAN_RXF1S_F1GI_S                                            8U
/*

        Field:           F1PI
        From..to bits:   16...21
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Rx FIFO 1 Put Index. Rx FIFO 1 write index pointer, range 0 to 63.

*/
#define DCAN_RXF1S_F1PI_W                                            6U
#define DCAN_RXF1S_F1PI_M                                            0x003F0000U
#define DCAN_RXF1S_F1PI_S                                            16U
/*

        Field:           F1F
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Rx FIFO 1 Full
                           0  Rx FIFO 1 not full
                           1  Rx FIFO 1 full

*/
#define DCAN_RXF1S_F1F                                               0x01000000U
#define DCAN_RXF1S_F1F_M                                             0x01000000U
#define DCAN_RXF1S_F1F_S                                             24U
/*

        Field:           RF1L
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Rx FIFO 1 Message Lost. This bit is a copy of interrupt flag IR.RF1L. When IR.RF1L is reset, this bit is also reset.
                           0  No Rx FIFO 1 message lost
                           1  Rx FIFO 1 message lost, also set after write attempt to Rx FIFO 1 of size zero

                         Note: Overwriting the oldest message when RXF1C.F1OM = '1' will not set this flag.

*/
#define DCAN_RXF1S_RF1L                                              0x02000000U
#define DCAN_RXF1S_RF1L_M                                            0x02000000U
#define DCAN_RXF1S_RF1L_S                                            25U
/*

        Field:           DMS
        From..to bits:   30...31
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Debug Message Status
                           00  Idle state, wait for reception of debug messages, DMA request is cleared
                           01  Debug message A received
                           10  Debug messages A, B received
                           11  Debug messages A, B, C received, DMA request is set

*/
#define DCAN_RXF1S_DMS_W                                             2U
#define DCAN_RXF1S_DMS_M                                             0xC0000000U
#define DCAN_RXF1S_DMS_S                                             30U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RXF1A
    Offset name:         DCAN_O_RXF1A
    Relative address:    0xB8
    Description:         MCAN Rx FIFO 1 Acknowledge
    Default Value:       0x00000000

        Field:           F1AI
        From..to bits:   0...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Acknowledge Index. After the Host has read a message or a sequence of messages from Rx FIFO 1 it has to write the buffer index of the last element read from Rx FIFO 1 to F1AI. This will set the Rx FIFO 1 Get Index RXF1S.F1GI to F1AI + 1 and update the FIFO 1 Fill Level RXF1S.F1FL.

*/
#define DCAN_RXF1A_F1AI_W                                            6U
#define DCAN_RXF1A_F1AI_M                                            0x0000003FU
#define DCAN_RXF1A_F1AI_S                                            0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RXESC
    Offset name:         DCAN_O_RXESC
    Relative address:    0xBC
    Description:         Configures the number of data bytes belonging to an Rx Buffer / Rx FIFO element. Data field sizes >8 bytes are intended for CAN FD operation only.
    Default Value:       0x00000000

        Field:           F0DS
        From..to bits:   0...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 0 Data Field Size
                           000  8 byte data field
                           001  12 byte data field
                           010  16 byte data field
                           011  20 byte data field
                           100  24 byte data field
                           101  32 byte data field
                           110  48 byte data field
                           111  64 byte data field

                         Note: In case the data field size of an accepted CAN frame exceeds the data field size configured for the matching Rx Buffer or Rx FIFO, only the number of bytes as configured by RXESC are stored to the Rx Buffer resp. Rx FIFO element. The rest of the frame's data field is ignored.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RXESC_F0DS_W                                            3U
#define DCAN_RXESC_F0DS_M                                            0x00000007U
#define DCAN_RXESC_F0DS_S                                            0U
/*

        Field:           F1DS
        From..to bits:   4...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx FIFO 1 Data Field Size
                           000  8 byte data field
                           001  12 byte data field
                           010  16 byte data field
                           011  20 byte data field
                           100  24 byte data field
                           101  32 byte data field
                           110  48 byte data field
                           111  64 byte data field

                         Note: In case the data field size of an accepted CAN frame exceeds the data field size configured for the matching Rx Buffer or Rx FIFO, only the number of bytes as configured by RXESC are stored to the Rx Buffer resp. Rx FIFO element. The rest of the frame's data field is ignored.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RXESC_F1DS_W                                            3U
#define DCAN_RXESC_F1DS_M                                            0x00000070U
#define DCAN_RXESC_F1DS_S                                            4U
/*

        Field:           RBDS
        From..to bits:   8...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Rx Buffer Data Field Size
                           000  8 byte data field
                           001  12 byte data field
                           010  16 byte data field
                           011  20 byte data field
                           100  24 byte data field
                           101  32 byte data field
                           110  48 byte data field
                           111  64 byte data field

                         Note: In case the data field size of an accepted CAN frame exceeds the data field size configured for the matching Rx Buffer or Rx FIFO, only the number of bytes as configured by RXESC are stored to the Rx Buffer resp. Rx FIFO element. The rest of the frame's data field is ignored.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_RXESC_RBDS_W                                            3U
#define DCAN_RXESC_RBDS_M                                            0x00000700U
#define DCAN_RXESC_RBDS_S                                            8U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXBC
    Offset name:         DCAN_O_TXBC
    Relative address:    0xC0
    Description:         MCAN Tx Buffer Configuration
    Default Value:       0x00000000

        Field:           TBSA
        From..to bits:   2...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Buffers Start Address. Start address of Tx Buffers section in Message RAM (32-bit word address).

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TXBC_TBSA_W                                             14U
#define DCAN_TXBC_TBSA_M                                             0x0000FFFCU
#define DCAN_TXBC_TBSA_S                                             2U
/*

        Field:           NDTB
        From..to bits:   16...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Number of Dedicated Transmit Buffers
                           0      No Dedicated Tx Buffers
                           1-32  Number of Dedicated Tx Buffers
                           >32   Values greater than 32 are interpreted as 32

                         Note: Be aware that the sum of TFQS and NDTB may be not greater than 32. There is no check
                         for erroneous configurations. The Tx Buffers section in the Message RAM starts with the
                         dedicated Tx Buffers.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TXBC_NDTB_W                                             6U
#define DCAN_TXBC_NDTB_M                                             0x003F0000U
#define DCAN_TXBC_NDTB_S                                             16U
/*

        Field:           TFQS
        From..to bits:   24...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmit FIFO/Queue Size
                           0      No Tx FIFO/Queue
                           1-32  Number of Tx Buffers used for Tx FIFO/Queue
                           >32   Values greater than 32 are interpreted as 32

                         Note: Be aware that the sum of TFQS and NDTB may be not greater than 32. There is no check
                         for erroneous configurations. The Tx Buffers section in the Message RAM starts with the
                         dedicated Tx Buffers.

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TXBC_TFQS_W                                             6U
#define DCAN_TXBC_TFQS_M                                             0x3F000000U
#define DCAN_TXBC_TFQS_S                                             24U
/*

        Field:           TFQM
        From..to bits:   30...30
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx FIFO/Queue Mode
                           0  Tx FIFO operation
                           1  Tx Queue operation

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TXBC_TFQM                                               0x40000000U
#define DCAN_TXBC_TFQM_M                                             0x40000000U
#define DCAN_TXBC_TFQM_S                                             30U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXFQS
    Offset name:         DCAN_O_TXFQS
    Relative address:    0xC4
    Description:         The Tx FIFO/Queue status is related to the pending Tx requests listed in register TXBRP. Therefore the effect of Add/Cancellation requests may be delayed due to a running Tx scan (TXBRP not yet updated).
    Default Value:       0x00000000

        Field:           TFFL
        From..to bits:   0...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Tx FIFO Free Level.  Number of consecutive free Tx FIFO elements starting from TFGI, range 0 to 32. Read as zero when Tx Queue operation is configured (TXBC.TFQM = '1').

*/
#define DCAN_TXFQS_TFFL_W                                            6U
#define DCAN_TXFQS_TFFL_M                                            0x0000003FU
#define DCAN_TXFQS_TFFL_S                                            0U
/*

        Field:           TFGI
        From..to bits:   8...12
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Tx FIFO Get Index. Tx FIFO read index pointer, range 0 to 31. Read as zero when Tx Queue operation is configured (TXBC.TFQM = '1').

                         Note: In case of mixed configurations where dedicated Tx Buffers are combined with a Tx FIFO or a Tx Queue, the Put and Get Indices indicate the number of the Tx Buffer starting with the first dedicated Tx Buffers. Example: For a configuration of 12 dedicated Tx Buffers and a Tx FIFO of 20 Buffers a Put Index of 15 points to the fourth buffer of the Tx FIFO.

*/
#define DCAN_TXFQS_TFGI_W                                            5U
#define DCAN_TXFQS_TFGI_M                                            0x00001F00U
#define DCAN_TXFQS_TFGI_S                                            8U
/*

        Field:           TFQP
        From..to bits:   16...20
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Tx FIFO/Queue Put Index. Tx FIFO/Queue write index pointer, range 0 to 31.

                         Note: In case of mixed configurations where dedicated Tx Buffers are combined with a Tx FIFO or a Tx Queue, the Put and Get Indices indicate the number of the Tx Buffer starting with the first dedicated Tx Buffers. Example: For a configuration of 12 dedicated Tx Buffers and a Tx FIFO of 20 Buffers a Put Index of 15 points to the fourth buffer of the Tx FIFO.

*/
#define DCAN_TXFQS_TFQP_W                                            5U
#define DCAN_TXFQS_TFQP_M                                            0x001F0000U
#define DCAN_TXFQS_TFQP_S                                            16U
/*

        Field:           TFQF
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Tx FIFO/Queue Full
                           0  Tx FIFO/Queue not full
                           1  Tx FIFO/Queue full

*/
#define DCAN_TXFQS_TFQF                                              0x00200000U
#define DCAN_TXFQS_TFQF_M                                            0x00200000U
#define DCAN_TXFQS_TFQF_S                                            21U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXESC
    Offset name:         DCAN_O_TXESC
    Relative address:    0xC8
    Description:         Configures the number of data bytes belonging to a Tx Buffer element. Data field sizes > 8 bytes are intended for CAN FD operation only.
    Default Value:       0x00000000

        Field:           TBDS
        From..to bits:   0...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Tx Buffer Data Field Size
                           000  8 byte data field
                           001  12 byte data field
                           010  16 byte data field
                           011  20 byte data field
                           100  24 byte data field
                           101  32 byte data field
                           110  48 byte data field
                           111  64 byte data field

                         Note: In case the data length code DLC of a Tx Buffer element is configured to a value higher than the Tx Buffer data field size TXESC.TBDS, the bytes not defined by the Tx Buffer are transmitted as "0xCC" (padding bytes).

                         Qualified Write is possible only with CCCR.CCE='1' and CCCR.INIT='1'.

*/
#define DCAN_TXESC_TBDS_W                                            3U
#define DCAN_TXESC_TBDS_M                                            0x00000007U
#define DCAN_TXESC_TBDS_S                                            0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXBRPAP
    Offset name:         DCAN_O_TXBRPAP
    Relative address:    0xCC
    Description:         MCAN Tx Buffer Request Pending
    Default Value:       0x00000000

        Field:           TRP0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 0.

                         Each Tx Buffer has its own Transmission Request Pending bit. The bits are set via register TXBAR. The bits are reset after a requested transmission has completed or has been cancelled via register TXBCR.

                         TXBRP bits are set only for those Tx Buffers configured via TXBC. After a TXBRP bit has been set, a Tx scan is started to check for the pending Tx request with the highest priority (Tx Buffer with lowest Message ID).

                         A cancellation request resets the corresponding transmission request pending bit of register TXBRP. In case a transmission has already been started when a cancellation is requested, this is done at the end of the transmission, regardless whether the transmission was successful or not. The cancellation request bits are reset directly after the corresponding TXBRP bit has been reset.

                         After a cancellation has been requested, a finished cancellation is signalled via TXBCF
                         - after successful transmission together with the corresponding TXBTO bit
                         - when the transmission has not yet been started at the point of cancellation
                         - when the transmission has been aborted due to lost arbitration
                         - when an error occurred during frame transmission

                         In DAR mode all transmissions are automatically cancelled if they are not successful. The corresponding TXBCF bit is set for all unsuccessful transmissions.
                           0  No transmission request pending
                           1  Transmission request pending

                         Note: TXBRP bits which are set while a Tx scan is in progress are not considered during this particular Tx scan. In case a cancellation is requested for such a Tx Buffer, this Add Request is cancelled immediately, the corresponding TXBRP bit is reset.

*/
#define DCAN_TXBRPAP_TRP0                                            0x00000001U
#define DCAN_TXBRPAP_TRP0_M                                          0x00000001U
#define DCAN_TXBRPAP_TRP0_S                                          0U
/*

        Field:           TRP1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 1. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP1                                            0x00000002U
#define DCAN_TXBRPAP_TRP1_M                                          0x00000002U
#define DCAN_TXBRPAP_TRP1_S                                          1U
/*

        Field:           TRP2
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 2. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP2                                            0x00000004U
#define DCAN_TXBRPAP_TRP2_M                                          0x00000004U
#define DCAN_TXBRPAP_TRP2_S                                          2U
/*

        Field:           TRP3
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 3. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP3                                            0x00000008U
#define DCAN_TXBRPAP_TRP3_M                                          0x00000008U
#define DCAN_TXBRPAP_TRP3_S                                          3U
/*

        Field:           TRP4
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 4. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP4                                            0x00000010U
#define DCAN_TXBRPAP_TRP4_M                                          0x00000010U
#define DCAN_TXBRPAP_TRP4_S                                          4U
/*

        Field:           TRP5
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 5. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP5                                            0x00000020U
#define DCAN_TXBRPAP_TRP5_M                                          0x00000020U
#define DCAN_TXBRPAP_TRP5_S                                          5U
/*

        Field:           TRP6
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 6. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP6                                            0x00000040U
#define DCAN_TXBRPAP_TRP6_M                                          0x00000040U
#define DCAN_TXBRPAP_TRP6_S                                          6U
/*

        Field:           TRP7
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 7. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP7                                            0x00000080U
#define DCAN_TXBRPAP_TRP7_M                                          0x00000080U
#define DCAN_TXBRPAP_TRP7_S                                          7U
/*

        Field:           TRP8
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 8. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP8                                            0x00000100U
#define DCAN_TXBRPAP_TRP8_M                                          0x00000100U
#define DCAN_TXBRPAP_TRP8_S                                          8U
/*

        Field:           TRP9
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 9. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP9                                            0x00000200U
#define DCAN_TXBRPAP_TRP9_M                                          0x00000200U
#define DCAN_TXBRPAP_TRP9_S                                          9U
/*

        Field:           TRP10
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 10. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP10                                           0x00000400U
#define DCAN_TXBRPAP_TRP10_M                                         0x00000400U
#define DCAN_TXBRPAP_TRP10_S                                         10U
/*

        Field:           TRP11
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 11. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP11                                           0x00000800U
#define DCAN_TXBRPAP_TRP11_M                                         0x00000800U
#define DCAN_TXBRPAP_TRP11_S                                         11U
/*

        Field:           TRP12
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 12. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP12                                           0x00001000U
#define DCAN_TXBRPAP_TRP12_M                                         0x00001000U
#define DCAN_TXBRPAP_TRP12_S                                         12U
/*

        Field:           TRP13
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 13. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP13                                           0x00002000U
#define DCAN_TXBRPAP_TRP13_M                                         0x00002000U
#define DCAN_TXBRPAP_TRP13_S                                         13U
/*

        Field:           TRP14
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 14. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP14                                           0x00004000U
#define DCAN_TXBRPAP_TRP14_M                                         0x00004000U
#define DCAN_TXBRPAP_TRP14_S                                         14U
/*

        Field:           TRP15
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 15. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP15                                           0x00008000U
#define DCAN_TXBRPAP_TRP15_M                                         0x00008000U
#define DCAN_TXBRPAP_TRP15_S                                         15U
/*

        Field:           TRP16
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 16. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP16                                           0x00010000U
#define DCAN_TXBRPAP_TRP16_M                                         0x00010000U
#define DCAN_TXBRPAP_TRP16_S                                         16U
/*

        Field:           TRP17
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 17. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP17                                           0x00020000U
#define DCAN_TXBRPAP_TRP17_M                                         0x00020000U
#define DCAN_TXBRPAP_TRP17_S                                         17U
/*

        Field:           TRP18
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 18. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP18                                           0x00040000U
#define DCAN_TXBRPAP_TRP18_M                                         0x00040000U
#define DCAN_TXBRPAP_TRP18_S                                         18U
/*

        Field:           TRP19
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 19. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP19                                           0x00080000U
#define DCAN_TXBRPAP_TRP19_M                                         0x00080000U
#define DCAN_TXBRPAP_TRP19_S                                         19U
/*

        Field:           TRP20
        From..to bits:   20...20
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 20. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP20                                           0x00100000U
#define DCAN_TXBRPAP_TRP20_M                                         0x00100000U
#define DCAN_TXBRPAP_TRP20_S                                         20U
/*

        Field:           TRP21
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 21. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP21                                           0x00200000U
#define DCAN_TXBRPAP_TRP21_M                                         0x00200000U
#define DCAN_TXBRPAP_TRP21_S                                         21U
/*

        Field:           TRP22
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 22. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP22                                           0x00400000U
#define DCAN_TXBRPAP_TRP22_M                                         0x00400000U
#define DCAN_TXBRPAP_TRP22_S                                         22U
/*

        Field:           TRP23
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 23. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP23                                           0x00800000U
#define DCAN_TXBRPAP_TRP23_M                                         0x00800000U
#define DCAN_TXBRPAP_TRP23_S                                         23U
/*

        Field:           TRP24
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 24. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP24                                           0x01000000U
#define DCAN_TXBRPAP_TRP24_M                                         0x01000000U
#define DCAN_TXBRPAP_TRP24_S                                         24U
/*

        Field:           TRP25
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 25. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP25                                           0x02000000U
#define DCAN_TXBRPAP_TRP25_M                                         0x02000000U
#define DCAN_TXBRPAP_TRP25_S                                         25U
/*

        Field:           TRP26
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 26. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP26                                           0x04000000U
#define DCAN_TXBRPAP_TRP26_M                                         0x04000000U
#define DCAN_TXBRPAP_TRP26_S                                         26U
/*

        Field:           TRP27
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 27. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP27                                           0x08000000U
#define DCAN_TXBRPAP_TRP27_M                                         0x08000000U
#define DCAN_TXBRPAP_TRP27_S                                         27U
/*

        Field:           TRP28
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 28. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP28                                           0x10000000U
#define DCAN_TXBRPAP_TRP28_M                                         0x10000000U
#define DCAN_TXBRPAP_TRP28_S                                         28U
/*

        Field:           TRP29
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 29. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP29                                           0x20000000U
#define DCAN_TXBRPAP_TRP29_M                                         0x20000000U
#define DCAN_TXBRPAP_TRP29_S                                         29U
/*

        Field:           TRP30
        From..to bits:   30...30
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 30. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP30                                           0x40000000U
#define DCAN_TXBRPAP_TRP30_M                                         0x40000000U
#define DCAN_TXBRPAP_TRP30_S                                         30U
/*

        Field:           TRP31
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Request Pending 31. See description for bit 0.

*/
#define DCAN_TXBRPAP_TRP31                                           0x80000000U
#define DCAN_TXBRPAP_TRP31_M                                         0x80000000U
#define DCAN_TXBRPAP_TRP31_S                                         31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXBAR
    Offset name:         DCAN_O_TXBAR
    Relative address:    0xD0
    Description:         MCAN Tx Buffer Add Request
    Default Value:       0x00000000

        Field:           AR0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 0.

                         Each Tx Buffer has its own Add Request bit. Writing a '1' will set the corresponding Add Request bit; writing a '0' has no impact. This enables the Host to set transmission requests for multiple Tx Buffers with one write to TXBAR. TXBAR bits are set only for those Tx Buffers configured via TXBC. When no Tx scan is running, the bits are reset immediately, else the bits remain set until the Tx scan process has completed.
                           0  No transmission request added
                           1  Transmission requested added

                         Note: If an add request is applied for a Tx Buffer with pending transmission request (corresponding TXBRP bit already set), this add request is ignored.

                         Qualified Write is possible only with CCCR.CCE='0'

*/
#define DCAN_TXBAR_AR0                                               0x00000001U
#define DCAN_TXBAR_AR0_M                                             0x00000001U
#define DCAN_TXBAR_AR0_S                                             0U
/*

        Field:           AR1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 1. See description for bit 0.

*/
#define DCAN_TXBAR_AR1                                               0x00000002U
#define DCAN_TXBAR_AR1_M                                             0x00000002U
#define DCAN_TXBAR_AR1_S                                             1U
/*

        Field:           AR2
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 2. See description for bit 0.

*/
#define DCAN_TXBAR_AR2                                               0x00000004U
#define DCAN_TXBAR_AR2_M                                             0x00000004U
#define DCAN_TXBAR_AR2_S                                             2U
/*

        Field:           AR3
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 3. See description for bit 0.

*/
#define DCAN_TXBAR_AR3                                               0x00000008U
#define DCAN_TXBAR_AR3_M                                             0x00000008U
#define DCAN_TXBAR_AR3_S                                             3U
/*

        Field:           AR4
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 4. See description for bit 0.

*/
#define DCAN_TXBAR_AR4                                               0x00000010U
#define DCAN_TXBAR_AR4_M                                             0x00000010U
#define DCAN_TXBAR_AR4_S                                             4U
/*

        Field:           AR5
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 5. See description for bit 0.

*/
#define DCAN_TXBAR_AR5                                               0x00000020U
#define DCAN_TXBAR_AR5_M                                             0x00000020U
#define DCAN_TXBAR_AR5_S                                             5U
/*

        Field:           AR6
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 6. See description for bit 0.

*/
#define DCAN_TXBAR_AR6                                               0x00000040U
#define DCAN_TXBAR_AR6_M                                             0x00000040U
#define DCAN_TXBAR_AR6_S                                             6U
/*

        Field:           AR7
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 7. See description for bit 0.

*/
#define DCAN_TXBAR_AR7                                               0x00000080U
#define DCAN_TXBAR_AR7_M                                             0x00000080U
#define DCAN_TXBAR_AR7_S                                             7U
/*

        Field:           AR8
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 8. See description for bit 0.

*/
#define DCAN_TXBAR_AR8                                               0x00000100U
#define DCAN_TXBAR_AR8_M                                             0x00000100U
#define DCAN_TXBAR_AR8_S                                             8U
/*

        Field:           AR9
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 9. See description for bit 0.

*/
#define DCAN_TXBAR_AR9                                               0x00000200U
#define DCAN_TXBAR_AR9_M                                             0x00000200U
#define DCAN_TXBAR_AR9_S                                             9U
/*

        Field:           AR10
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 10. See description for bit 0.

*/
#define DCAN_TXBAR_AR10                                              0x00000400U
#define DCAN_TXBAR_AR10_M                                            0x00000400U
#define DCAN_TXBAR_AR10_S                                            10U
/*

        Field:           AR11
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 11. See description for bit 0.

*/
#define DCAN_TXBAR_AR11                                              0x00000800U
#define DCAN_TXBAR_AR11_M                                            0x00000800U
#define DCAN_TXBAR_AR11_S                                            11U
/*

        Field:           AR12
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 12. See description for bit 0.

*/
#define DCAN_TXBAR_AR12                                              0x00001000U
#define DCAN_TXBAR_AR12_M                                            0x00001000U
#define DCAN_TXBAR_AR12_S                                            12U
/*

        Field:           AR13
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 13. See description for bit 0.

*/
#define DCAN_TXBAR_AR13                                              0x00002000U
#define DCAN_TXBAR_AR13_M                                            0x00002000U
#define DCAN_TXBAR_AR13_S                                            13U
/*

        Field:           AR14
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 14. See description for bit 0.

*/
#define DCAN_TXBAR_AR14                                              0x00004000U
#define DCAN_TXBAR_AR14_M                                            0x00004000U
#define DCAN_TXBAR_AR14_S                                            14U
/*

        Field:           AR15
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 15. See description for bit 0.

*/
#define DCAN_TXBAR_AR15                                              0x00008000U
#define DCAN_TXBAR_AR15_M                                            0x00008000U
#define DCAN_TXBAR_AR15_S                                            15U
/*

        Field:           AR16
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 16. See description for bit 0.

*/
#define DCAN_TXBAR_AR16                                              0x00010000U
#define DCAN_TXBAR_AR16_M                                            0x00010000U
#define DCAN_TXBAR_AR16_S                                            16U
/*

        Field:           AR17
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 17. See description for bit 0.

*/
#define DCAN_TXBAR_AR17                                              0x00020000U
#define DCAN_TXBAR_AR17_M                                            0x00020000U
#define DCAN_TXBAR_AR17_S                                            17U
/*

        Field:           AR18
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 18. See description for bit 0.

*/
#define DCAN_TXBAR_AR18                                              0x00040000U
#define DCAN_TXBAR_AR18_M                                            0x00040000U
#define DCAN_TXBAR_AR18_S                                            18U
/*

        Field:           AR19
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 19. See description for bit 0.

*/
#define DCAN_TXBAR_AR19                                              0x00080000U
#define DCAN_TXBAR_AR19_M                                            0x00080000U
#define DCAN_TXBAR_AR19_S                                            19U
/*

        Field:           AR20
        From..to bits:   20...20
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 20. See description for bit 0.

*/
#define DCAN_TXBAR_AR20                                              0x00100000U
#define DCAN_TXBAR_AR20_M                                            0x00100000U
#define DCAN_TXBAR_AR20_S                                            20U
/*

        Field:           AR21
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 21. See description for bit 0.

*/
#define DCAN_TXBAR_AR21                                              0x00200000U
#define DCAN_TXBAR_AR21_M                                            0x00200000U
#define DCAN_TXBAR_AR21_S                                            21U
/*

        Field:           AR22
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 22. See description for bit 0.

*/
#define DCAN_TXBAR_AR22                                              0x00400000U
#define DCAN_TXBAR_AR22_M                                            0x00400000U
#define DCAN_TXBAR_AR22_S                                            22U
/*

        Field:           AR23
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 23. See description for bit 0.

*/
#define DCAN_TXBAR_AR23                                              0x00800000U
#define DCAN_TXBAR_AR23_M                                            0x00800000U
#define DCAN_TXBAR_AR23_S                                            23U
/*

        Field:           AR24
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 24. See description for bit 0.

*/
#define DCAN_TXBAR_AR24                                              0x01000000U
#define DCAN_TXBAR_AR24_M                                            0x01000000U
#define DCAN_TXBAR_AR24_S                                            24U
/*

        Field:           AR25
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 25. See description for bit 0.

*/
#define DCAN_TXBAR_AR25                                              0x02000000U
#define DCAN_TXBAR_AR25_M                                            0x02000000U
#define DCAN_TXBAR_AR25_S                                            25U
/*

        Field:           AR26
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 26. See description for bit 0.

*/
#define DCAN_TXBAR_AR26                                              0x04000000U
#define DCAN_TXBAR_AR26_M                                            0x04000000U
#define DCAN_TXBAR_AR26_S                                            26U
/*

        Field:           AR27
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 27. See description for bit 0.

*/
#define DCAN_TXBAR_AR27                                              0x08000000U
#define DCAN_TXBAR_AR27_M                                            0x08000000U
#define DCAN_TXBAR_AR27_S                                            27U
/*

        Field:           AR28
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 28. See description for bit 0.

*/
#define DCAN_TXBAR_AR28                                              0x10000000U
#define DCAN_TXBAR_AR28_M                                            0x10000000U
#define DCAN_TXBAR_AR28_S                                            28U
/*

        Field:           AR29
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 29. See description for bit 0.

*/
#define DCAN_TXBAR_AR29                                              0x20000000U
#define DCAN_TXBAR_AR29_M                                            0x20000000U
#define DCAN_TXBAR_AR29_S                                            29U
/*

        Field:           AR30
        From..to bits:   30...30
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 30. See description for bit 0.

*/
#define DCAN_TXBAR_AR30                                              0x40000000U
#define DCAN_TXBAR_AR30_M                                            0x40000000U
#define DCAN_TXBAR_AR30_S                                            30U
/*

        Field:           AR31
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Add Request 31. See description for bit 0.

*/
#define DCAN_TXBAR_AR31                                              0x80000000U
#define DCAN_TXBAR_AR31_M                                            0x80000000U
#define DCAN_TXBAR_AR31_S                                            31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXBCR
    Offset name:         DCAN_O_TXBCR
    Relative address:    0xD4
    Description:         MCAN Tx Buffer Cancellation Request
    Default Value:       0x00000000

        Field:           CR0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 0.

                         Each Tx Buffer has its own Cancellation Request bit. Writing a '1' will set the corresponding Cancellation Request bit; writing a '0' has no impact. This enables the Host to set cancellation requests for multiple Tx Buffers with one write to TXBCR. TXBCR bits are set only for those Tx Buffers configured via TXBC. The bits remain set until the corresponding bit of TXBRP is reset.
                           0  No cancellation pending
                           1  Cancellation pending

                         Qualified Write is possible only with CCCR.CCE='0'

*/
#define DCAN_TXBCR_CR0                                               0x00000001U
#define DCAN_TXBCR_CR0_M                                             0x00000001U
#define DCAN_TXBCR_CR0_S                                             0U
/*

        Field:           CR1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 1. See description for bit 0.

*/
#define DCAN_TXBCR_CR1                                               0x00000002U
#define DCAN_TXBCR_CR1_M                                             0x00000002U
#define DCAN_TXBCR_CR1_S                                             1U
/*

        Field:           CR2
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 2. See description for bit 0.

*/
#define DCAN_TXBCR_CR2                                               0x00000004U
#define DCAN_TXBCR_CR2_M                                             0x00000004U
#define DCAN_TXBCR_CR2_S                                             2U
/*

        Field:           CR3
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 3. See description for bit 0.

*/
#define DCAN_TXBCR_CR3                                               0x00000008U
#define DCAN_TXBCR_CR3_M                                             0x00000008U
#define DCAN_TXBCR_CR3_S                                             3U
/*

        Field:           CR4
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 4. See description for bit 0.

*/
#define DCAN_TXBCR_CR4                                               0x00000010U
#define DCAN_TXBCR_CR4_M                                             0x00000010U
#define DCAN_TXBCR_CR4_S                                             4U
/*

        Field:           CR5
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 5. See description for bit 0.

*/
#define DCAN_TXBCR_CR5                                               0x00000020U
#define DCAN_TXBCR_CR5_M                                             0x00000020U
#define DCAN_TXBCR_CR5_S                                             5U
/*

        Field:           CR6
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 6. See description for bit 0.

*/
#define DCAN_TXBCR_CR6                                               0x00000040U
#define DCAN_TXBCR_CR6_M                                             0x00000040U
#define DCAN_TXBCR_CR6_S                                             6U
/*

        Field:           CR7
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 7. See description for bit 0.

*/
#define DCAN_TXBCR_CR7                                               0x00000080U
#define DCAN_TXBCR_CR7_M                                             0x00000080U
#define DCAN_TXBCR_CR7_S                                             7U
/*

        Field:           CR8
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 8. See description for bit 0.

*/
#define DCAN_TXBCR_CR8                                               0x00000100U
#define DCAN_TXBCR_CR8_M                                             0x00000100U
#define DCAN_TXBCR_CR8_S                                             8U
/*

        Field:           CR9
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 9. See description for bit 0.

*/
#define DCAN_TXBCR_CR9                                               0x00000200U
#define DCAN_TXBCR_CR9_M                                             0x00000200U
#define DCAN_TXBCR_CR9_S                                             9U
/*

        Field:           CR10
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 10. See description for bit 0.

*/
#define DCAN_TXBCR_CR10                                              0x00000400U
#define DCAN_TXBCR_CR10_M                                            0x00000400U
#define DCAN_TXBCR_CR10_S                                            10U
/*

        Field:           CR11
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 11. See description for bit 0.

*/
#define DCAN_TXBCR_CR11                                              0x00000800U
#define DCAN_TXBCR_CR11_M                                            0x00000800U
#define DCAN_TXBCR_CR11_S                                            11U
/*

        Field:           CR12
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 12. See description for bit 0.

*/
#define DCAN_TXBCR_CR12                                              0x00001000U
#define DCAN_TXBCR_CR12_M                                            0x00001000U
#define DCAN_TXBCR_CR12_S                                            12U
/*

        Field:           CR13
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 13. See description for bit 0.

*/
#define DCAN_TXBCR_CR13                                              0x00002000U
#define DCAN_TXBCR_CR13_M                                            0x00002000U
#define DCAN_TXBCR_CR13_S                                            13U
/*

        Field:           CR14
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 14. See description for bit 0.

*/
#define DCAN_TXBCR_CR14                                              0x00004000U
#define DCAN_TXBCR_CR14_M                                            0x00004000U
#define DCAN_TXBCR_CR14_S                                            14U
/*

        Field:           CR15
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 15. See description for bit 0.

*/
#define DCAN_TXBCR_CR15                                              0x00008000U
#define DCAN_TXBCR_CR15_M                                            0x00008000U
#define DCAN_TXBCR_CR15_S                                            15U
/*

        Field:           CR16
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 16. See description for bit 0.

*/
#define DCAN_TXBCR_CR16                                              0x00010000U
#define DCAN_TXBCR_CR16_M                                            0x00010000U
#define DCAN_TXBCR_CR16_S                                            16U
/*

        Field:           CR17
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 17. See description for bit 0.

*/
#define DCAN_TXBCR_CR17                                              0x00020000U
#define DCAN_TXBCR_CR17_M                                            0x00020000U
#define DCAN_TXBCR_CR17_S                                            17U
/*

        Field:           CR18
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 18. See description for bit 0.

*/
#define DCAN_TXBCR_CR18                                              0x00040000U
#define DCAN_TXBCR_CR18_M                                            0x00040000U
#define DCAN_TXBCR_CR18_S                                            18U
/*

        Field:           CR19
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 19. See description for bit 0.

*/
#define DCAN_TXBCR_CR19                                              0x00080000U
#define DCAN_TXBCR_CR19_M                                            0x00080000U
#define DCAN_TXBCR_CR19_S                                            19U
/*

        Field:           CR20
        From..to bits:   20...20
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 20. See description for bit 0.

*/
#define DCAN_TXBCR_CR20                                              0x00100000U
#define DCAN_TXBCR_CR20_M                                            0x00100000U
#define DCAN_TXBCR_CR20_S                                            20U
/*

        Field:           CR21
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 21. See description for bit 0.

*/
#define DCAN_TXBCR_CR21                                              0x00200000U
#define DCAN_TXBCR_CR21_M                                            0x00200000U
#define DCAN_TXBCR_CR21_S                                            21U
/*

        Field:           CR22
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 22. See description for bit 0.

*/
#define DCAN_TXBCR_CR22                                              0x00400000U
#define DCAN_TXBCR_CR22_M                                            0x00400000U
#define DCAN_TXBCR_CR22_S                                            22U
/*

        Field:           CR23
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 23. See description for bit 0.

*/
#define DCAN_TXBCR_CR23                                              0x00800000U
#define DCAN_TXBCR_CR23_M                                            0x00800000U
#define DCAN_TXBCR_CR23_S                                            23U
/*

        Field:           CR24
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 24. See description for bit 0.

*/
#define DCAN_TXBCR_CR24                                              0x01000000U
#define DCAN_TXBCR_CR24_M                                            0x01000000U
#define DCAN_TXBCR_CR24_S                                            24U
/*

        Field:           CR25
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 25. See description for bit 0.

*/
#define DCAN_TXBCR_CR25                                              0x02000000U
#define DCAN_TXBCR_CR25_M                                            0x02000000U
#define DCAN_TXBCR_CR25_S                                            25U
/*

        Field:           CR26
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 26. See description for bit 0.

*/
#define DCAN_TXBCR_CR26                                              0x04000000U
#define DCAN_TXBCR_CR26_M                                            0x04000000U
#define DCAN_TXBCR_CR26_S                                            26U
/*

        Field:           CR27
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 27. See description for bit 0.

*/
#define DCAN_TXBCR_CR27                                              0x08000000U
#define DCAN_TXBCR_CR27_M                                            0x08000000U
#define DCAN_TXBCR_CR27_S                                            27U
/*

        Field:           CR28
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 28. See description for bit 0.

*/
#define DCAN_TXBCR_CR28                                              0x10000000U
#define DCAN_TXBCR_CR28_M                                            0x10000000U
#define DCAN_TXBCR_CR28_S                                            28U
/*

        Field:           CR29
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 29. See description for bit 0.

*/
#define DCAN_TXBCR_CR29                                              0x20000000U
#define DCAN_TXBCR_CR29_M                                            0x20000000U
#define DCAN_TXBCR_CR29_S                                            29U
/*

        Field:           CR30
        From..to bits:   30...30
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 30. See description for bit 0.

*/
#define DCAN_TXBCR_CR30                                              0x40000000U
#define DCAN_TXBCR_CR30_M                                            0x40000000U
#define DCAN_TXBCR_CR30_S                                            30U
/*

        Field:           CR31
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Request 31. See description for bit 0.

*/
#define DCAN_TXBCR_CR31                                              0x80000000U
#define DCAN_TXBCR_CR31_M                                            0x80000000U
#define DCAN_TXBCR_CR31_S                                            31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXBTO
    Offset name:         DCAN_O_TXBTO
    Relative address:    0xD8
    Description:         MCAN Tx Buffer Transmission Occurred
    Default Value:       0x00000000

        Field:           TO0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 0.

                         Each Tx Buffer has its own Transmission Occurred bit. The bits are set when the corresponding TXBRP bit is cleared after a successful transmission. The bits are reset when a new transmission is requested by writing a '1' to the corresponding bit of register TXBAR.
                           0  No transmission occurred
                           1  Transmission occurred

*/
#define DCAN_TXBTO_TO0                                               0x00000001U
#define DCAN_TXBTO_TO0_M                                             0x00000001U
#define DCAN_TXBTO_TO0_S                                             0U
/*

        Field:           TO1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 1. See description for bit 0.

*/
#define DCAN_TXBTO_TO1                                               0x00000002U
#define DCAN_TXBTO_TO1_M                                             0x00000002U
#define DCAN_TXBTO_TO1_S                                             1U
/*

        Field:           TO2
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 2. See description for bit 0.

*/
#define DCAN_TXBTO_TO2                                               0x00000004U
#define DCAN_TXBTO_TO2_M                                             0x00000004U
#define DCAN_TXBTO_TO2_S                                             2U
/*

        Field:           TO3
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 3. See description for bit 0.

*/
#define DCAN_TXBTO_TO3                                               0x00000008U
#define DCAN_TXBTO_TO3_M                                             0x00000008U
#define DCAN_TXBTO_TO3_S                                             3U
/*

        Field:           TO4
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 4. See description for bit 0.

*/
#define DCAN_TXBTO_TO4                                               0x00000010U
#define DCAN_TXBTO_TO4_M                                             0x00000010U
#define DCAN_TXBTO_TO4_S                                             4U
/*

        Field:           TO5
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 5. See description for bit 0.

*/
#define DCAN_TXBTO_TO5                                               0x00000020U
#define DCAN_TXBTO_TO5_M                                             0x00000020U
#define DCAN_TXBTO_TO5_S                                             5U
/*

        Field:           TO6
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 6. See description for bit 0.

*/
#define DCAN_TXBTO_TO6                                               0x00000040U
#define DCAN_TXBTO_TO6_M                                             0x00000040U
#define DCAN_TXBTO_TO6_S                                             6U
/*

        Field:           TO7
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 7. See description for bit 0.

*/
#define DCAN_TXBTO_TO7                                               0x00000080U
#define DCAN_TXBTO_TO7_M                                             0x00000080U
#define DCAN_TXBTO_TO7_S                                             7U
/*

        Field:           TO8
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 8. See description for bit 0.

*/
#define DCAN_TXBTO_TO8                                               0x00000100U
#define DCAN_TXBTO_TO8_M                                             0x00000100U
#define DCAN_TXBTO_TO8_S                                             8U
/*

        Field:           TO9
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 9. See description for bit 0.

*/
#define DCAN_TXBTO_TO9                                               0x00000200U
#define DCAN_TXBTO_TO9_M                                             0x00000200U
#define DCAN_TXBTO_TO9_S                                             9U
/*

        Field:           TO10
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 10. See description for bit 0.

*/
#define DCAN_TXBTO_TO10                                              0x00000400U
#define DCAN_TXBTO_TO10_M                                            0x00000400U
#define DCAN_TXBTO_TO10_S                                            10U
/*

        Field:           TO11
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 11. See description for bit 0.

*/
#define DCAN_TXBTO_TO11                                              0x00000800U
#define DCAN_TXBTO_TO11_M                                            0x00000800U
#define DCAN_TXBTO_TO11_S                                            11U
/*

        Field:           TO12
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 12. See description for bit 0.

*/
#define DCAN_TXBTO_TO12                                              0x00001000U
#define DCAN_TXBTO_TO12_M                                            0x00001000U
#define DCAN_TXBTO_TO12_S                                            12U
/*

        Field:           TO13
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 13. See description for bit 0.

*/
#define DCAN_TXBTO_TO13                                              0x00002000U
#define DCAN_TXBTO_TO13_M                                            0x00002000U
#define DCAN_TXBTO_TO13_S                                            13U
/*

        Field:           TO14
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 14. See description for bit 0.

*/
#define DCAN_TXBTO_TO14                                              0x00004000U
#define DCAN_TXBTO_TO14_M                                            0x00004000U
#define DCAN_TXBTO_TO14_S                                            14U
/*

        Field:           TO15
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 15. See description for bit 0.

*/
#define DCAN_TXBTO_TO15                                              0x00008000U
#define DCAN_TXBTO_TO15_M                                            0x00008000U
#define DCAN_TXBTO_TO15_S                                            15U
/*

        Field:           TO16
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 16. See description for bit 0.

*/
#define DCAN_TXBTO_TO16                                              0x00010000U
#define DCAN_TXBTO_TO16_M                                            0x00010000U
#define DCAN_TXBTO_TO16_S                                            16U
/*

        Field:           TO17
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 17. See description for bit 0.

*/
#define DCAN_TXBTO_TO17                                              0x00020000U
#define DCAN_TXBTO_TO17_M                                            0x00020000U
#define DCAN_TXBTO_TO17_S                                            17U
/*

        Field:           TO18
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 18. See description for bit 0.

*/
#define DCAN_TXBTO_TO18                                              0x00040000U
#define DCAN_TXBTO_TO18_M                                            0x00040000U
#define DCAN_TXBTO_TO18_S                                            18U
/*

        Field:           TO19
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 19. See description for bit 0.

*/
#define DCAN_TXBTO_TO19                                              0x00080000U
#define DCAN_TXBTO_TO19_M                                            0x00080000U
#define DCAN_TXBTO_TO19_S                                            19U
/*

        Field:           TO20
        From..to bits:   20...20
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 20. See description for bit 0.

*/
#define DCAN_TXBTO_TO20                                              0x00100000U
#define DCAN_TXBTO_TO20_M                                            0x00100000U
#define DCAN_TXBTO_TO20_S                                            20U
/*

        Field:           TO21
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 21. See description for bit 0.

*/
#define DCAN_TXBTO_TO21                                              0x00200000U
#define DCAN_TXBTO_TO21_M                                            0x00200000U
#define DCAN_TXBTO_TO21_S                                            21U
/*

        Field:           TO22
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 22. See description for bit 0.

*/
#define DCAN_TXBTO_TO22                                              0x00400000U
#define DCAN_TXBTO_TO22_M                                            0x00400000U
#define DCAN_TXBTO_TO22_S                                            22U
/*

        Field:           TO23
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 23. See description for bit 0.

*/
#define DCAN_TXBTO_TO23                                              0x00800000U
#define DCAN_TXBTO_TO23_M                                            0x00800000U
#define DCAN_TXBTO_TO23_S                                            23U
/*

        Field:           TO24
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 24. See description for bit 0.

*/
#define DCAN_TXBTO_TO24                                              0x01000000U
#define DCAN_TXBTO_TO24_M                                            0x01000000U
#define DCAN_TXBTO_TO24_S                                            24U
/*

        Field:           TO25
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 25. See description for bit 0.

*/
#define DCAN_TXBTO_TO25                                              0x02000000U
#define DCAN_TXBTO_TO25_M                                            0x02000000U
#define DCAN_TXBTO_TO25_S                                            25U
/*

        Field:           TO26
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 26. See description for bit 0.

*/
#define DCAN_TXBTO_TO26                                              0x04000000U
#define DCAN_TXBTO_TO26_M                                            0x04000000U
#define DCAN_TXBTO_TO26_S                                            26U
/*

        Field:           TO27
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 27. See description for bit 0.

*/
#define DCAN_TXBTO_TO27                                              0x08000000U
#define DCAN_TXBTO_TO27_M                                            0x08000000U
#define DCAN_TXBTO_TO27_S                                            27U
/*

        Field:           TO28
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 28. See description for bit 0.

*/
#define DCAN_TXBTO_TO28                                              0x10000000U
#define DCAN_TXBTO_TO28_M                                            0x10000000U
#define DCAN_TXBTO_TO28_S                                            28U
/*

        Field:           TO29
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 29. See description for bit 0.

*/
#define DCAN_TXBTO_TO29                                              0x20000000U
#define DCAN_TXBTO_TO29_M                                            0x20000000U
#define DCAN_TXBTO_TO29_S                                            29U
/*

        Field:           TO30
        From..to bits:   30...30
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 30. See description for bit 0.

*/
#define DCAN_TXBTO_TO30                                              0x40000000U
#define DCAN_TXBTO_TO30_M                                            0x40000000U
#define DCAN_TXBTO_TO30_S                                            30U
/*

        Field:           TO31
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Transmission Occurred 31. See description for bit 0.

*/
#define DCAN_TXBTO_TO31                                              0x80000000U
#define DCAN_TXBTO_TO31_M                                            0x80000000U
#define DCAN_TXBTO_TO31_S                                            31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXBCF
    Offset name:         DCAN_O_TXBCF
    Relative address:    0xDC
    Description:         MCAN Tx Buffer Cancellation Finished
    Default Value:       0x00000000

        Field:           CF0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 0.

                         Each Tx Buffer has its own Cancellation Finished bit. The bits are set when the corresponding TXBRP bit is cleared after a cancellation was requested via TXBCR. In case the corresponding TXBRP bit was not set at the point of cancellation, CF is set immediately. The bits are reset when a new transmission is requested by writing a '1' to the corresponding bit of register TXBAR.
                           0  No transmit buffer cancellation
                           1  Transmit buffer cancellation finished

*/
#define DCAN_TXBCF_CF0                                               0x00000001U
#define DCAN_TXBCF_CF0_M                                             0x00000001U
#define DCAN_TXBCF_CF0_S                                             0U
/*

        Field:           CF1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 1. See description for bit 0.

*/
#define DCAN_TXBCF_CF1                                               0x00000002U
#define DCAN_TXBCF_CF1_M                                             0x00000002U
#define DCAN_TXBCF_CF1_S                                             1U
/*

        Field:           CF2
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 2. See description for bit 0.

*/
#define DCAN_TXBCF_CF2                                               0x00000004U
#define DCAN_TXBCF_CF2_M                                             0x00000004U
#define DCAN_TXBCF_CF2_S                                             2U
/*

        Field:           CF3
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 3. See description for bit 0.

*/
#define DCAN_TXBCF_CF3                                               0x00000008U
#define DCAN_TXBCF_CF3_M                                             0x00000008U
#define DCAN_TXBCF_CF3_S                                             3U
/*

        Field:           CF4
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 4. See description for bit 0.

*/
#define DCAN_TXBCF_CF4                                               0x00000010U
#define DCAN_TXBCF_CF4_M                                             0x00000010U
#define DCAN_TXBCF_CF4_S                                             4U
/*

        Field:           CF5
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 5. See description for bit 0.

*/
#define DCAN_TXBCF_CF5                                               0x00000020U
#define DCAN_TXBCF_CF5_M                                             0x00000020U
#define DCAN_TXBCF_CF5_S                                             5U
/*

        Field:           CF6
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 6. See description for bit 0.

*/
#define DCAN_TXBCF_CF6                                               0x00000040U
#define DCAN_TXBCF_CF6_M                                             0x00000040U
#define DCAN_TXBCF_CF6_S                                             6U
/*

        Field:           CF7
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 7. See description for bit 0.

*/
#define DCAN_TXBCF_CF7                                               0x00000080U
#define DCAN_TXBCF_CF7_M                                             0x00000080U
#define DCAN_TXBCF_CF7_S                                             7U
/*

        Field:           CF8
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 8. See description for bit 0.

*/
#define DCAN_TXBCF_CF8                                               0x00000100U
#define DCAN_TXBCF_CF8_M                                             0x00000100U
#define DCAN_TXBCF_CF8_S                                             8U
/*

        Field:           CF9
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 9. See description for bit 0.

*/
#define DCAN_TXBCF_CF9                                               0x00000200U
#define DCAN_TXBCF_CF9_M                                             0x00000200U
#define DCAN_TXBCF_CF9_S                                             9U
/*

        Field:           CF10
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 10. See description for bit 0.

*/
#define DCAN_TXBCF_CF10                                              0x00000400U
#define DCAN_TXBCF_CF10_M                                            0x00000400U
#define DCAN_TXBCF_CF10_S                                            10U
/*

        Field:           CF11
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 11. See description for bit 0.

*/
#define DCAN_TXBCF_CF11                                              0x00000800U
#define DCAN_TXBCF_CF11_M                                            0x00000800U
#define DCAN_TXBCF_CF11_S                                            11U
/*

        Field:           CF12
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 12. See description for bit 0.

*/
#define DCAN_TXBCF_CF12                                              0x00001000U
#define DCAN_TXBCF_CF12_M                                            0x00001000U
#define DCAN_TXBCF_CF12_S                                            12U
/*

        Field:           CF13
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 13. See description for bit 0.

*/
#define DCAN_TXBCF_CF13                                              0x00002000U
#define DCAN_TXBCF_CF13_M                                            0x00002000U
#define DCAN_TXBCF_CF13_S                                            13U
/*

        Field:           CF14
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 14. See description for bit 0.

*/
#define DCAN_TXBCF_CF14                                              0x00004000U
#define DCAN_TXBCF_CF14_M                                            0x00004000U
#define DCAN_TXBCF_CF14_S                                            14U
/*

        Field:           CF15
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 15. See description for bit 0.

*/
#define DCAN_TXBCF_CF15                                              0x00008000U
#define DCAN_TXBCF_CF15_M                                            0x00008000U
#define DCAN_TXBCF_CF15_S                                            15U
/*

        Field:           CF16
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 16. See description for bit 0.

*/
#define DCAN_TXBCF_CF16                                              0x00010000U
#define DCAN_TXBCF_CF16_M                                            0x00010000U
#define DCAN_TXBCF_CF16_S                                            16U
/*

        Field:           CF17
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 17. See description for bit 0.

*/
#define DCAN_TXBCF_CF17                                              0x00020000U
#define DCAN_TXBCF_CF17_M                                            0x00020000U
#define DCAN_TXBCF_CF17_S                                            17U
/*

        Field:           CF18
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 18. See description for bit 0.

*/
#define DCAN_TXBCF_CF18                                              0x00040000U
#define DCAN_TXBCF_CF18_M                                            0x00040000U
#define DCAN_TXBCF_CF18_S                                            18U
/*

        Field:           CF19
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 19. See description for bit 0.

*/
#define DCAN_TXBCF_CF19                                              0x00080000U
#define DCAN_TXBCF_CF19_M                                            0x00080000U
#define DCAN_TXBCF_CF19_S                                            19U
/*

        Field:           CF20
        From..to bits:   20...20
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 20. See description for bit 0.

*/
#define DCAN_TXBCF_CF20                                              0x00100000U
#define DCAN_TXBCF_CF20_M                                            0x00100000U
#define DCAN_TXBCF_CF20_S                                            20U
/*

        Field:           CF21
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 21. See description for bit 0.

*/
#define DCAN_TXBCF_CF21                                              0x00200000U
#define DCAN_TXBCF_CF21_M                                            0x00200000U
#define DCAN_TXBCF_CF21_S                                            21U
/*

        Field:           CF22
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 22. See description for bit 0.

*/
#define DCAN_TXBCF_CF22                                              0x00400000U
#define DCAN_TXBCF_CF22_M                                            0x00400000U
#define DCAN_TXBCF_CF22_S                                            22U
/*

        Field:           CF23
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 23. See description for bit 0.

*/
#define DCAN_TXBCF_CF23                                              0x00800000U
#define DCAN_TXBCF_CF23_M                                            0x00800000U
#define DCAN_TXBCF_CF23_S                                            23U
/*

        Field:           CF24
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 24. See description for bit 0.

*/
#define DCAN_TXBCF_CF24                                              0x01000000U
#define DCAN_TXBCF_CF24_M                                            0x01000000U
#define DCAN_TXBCF_CF24_S                                            24U
/*

        Field:           CF25
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 25. See description for bit 0.

*/
#define DCAN_TXBCF_CF25                                              0x02000000U
#define DCAN_TXBCF_CF25_M                                            0x02000000U
#define DCAN_TXBCF_CF25_S                                            25U
/*

        Field:           CF26
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 26. See description for bit 0.

*/
#define DCAN_TXBCF_CF26                                              0x04000000U
#define DCAN_TXBCF_CF26_M                                            0x04000000U
#define DCAN_TXBCF_CF26_S                                            26U
/*

        Field:           CF27
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 27. See description for bit 0.

*/
#define DCAN_TXBCF_CF27                                              0x08000000U
#define DCAN_TXBCF_CF27_M                                            0x08000000U
#define DCAN_TXBCF_CF27_S                                            27U
/*

        Field:           CF28
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 28. See description for bit 0.

*/
#define DCAN_TXBCF_CF28                                              0x10000000U
#define DCAN_TXBCF_CF28_M                                            0x10000000U
#define DCAN_TXBCF_CF28_S                                            28U
/*

        Field:           CF29
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 29. See description for bit 0.

*/
#define DCAN_TXBCF_CF29                                              0x20000000U
#define DCAN_TXBCF_CF29_M                                            0x20000000U
#define DCAN_TXBCF_CF29_S                                            29U
/*

        Field:           CF30
        From..to bits:   30...30
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 30. See description for bit 0.

*/
#define DCAN_TXBCF_CF30                                              0x40000000U
#define DCAN_TXBCF_CF30_M                                            0x40000000U
#define DCAN_TXBCF_CF30_S                                            30U
/*

        Field:           CF31
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Cancellation Finished 31. See description for bit 0.

*/
#define DCAN_TXBCF_CF31                                              0x80000000U
#define DCAN_TXBCF_CF31_M                                            0x80000000U
#define DCAN_TXBCF_CF31_S                                            31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXTIE
    Offset name:         DCAN_O_TXTIE
    Relative address:    0xE0
    Description:         MCAN Tx Buffer Transmission Interrupt Enable
    Default Value:       0x00000000

        Field:           TIE0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 0. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE0                                              0x00000001U
#define DCAN_TXTIE_TIE0_M                                            0x00000001U
#define DCAN_TXTIE_TIE0_S                                            0U
/*

        Field:           TIE1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 1. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE1                                              0x00000002U
#define DCAN_TXTIE_TIE1_M                                            0x00000002U
#define DCAN_TXTIE_TIE1_S                                            1U
/*

        Field:           TIE2
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 2. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE2                                              0x00000004U
#define DCAN_TXTIE_TIE2_M                                            0x00000004U
#define DCAN_TXTIE_TIE2_S                                            2U
/*

        Field:           TIE3
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 3. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE3                                              0x00000008U
#define DCAN_TXTIE_TIE3_M                                            0x00000008U
#define DCAN_TXTIE_TIE3_S                                            3U
/*

        Field:           TIE4
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 4. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE4                                              0x00000010U
#define DCAN_TXTIE_TIE4_M                                            0x00000010U
#define DCAN_TXTIE_TIE4_S                                            4U
/*

        Field:           TIE5
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 5. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE5                                              0x00000020U
#define DCAN_TXTIE_TIE5_M                                            0x00000020U
#define DCAN_TXTIE_TIE5_S                                            5U
/*

        Field:           TIE6
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 6. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE6                                              0x00000040U
#define DCAN_TXTIE_TIE6_M                                            0x00000040U
#define DCAN_TXTIE_TIE6_S                                            6U
/*

        Field:           TIE7
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 7. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE7                                              0x00000080U
#define DCAN_TXTIE_TIE7_M                                            0x00000080U
#define DCAN_TXTIE_TIE7_S                                            7U
/*

        Field:           TIE8
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 8. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE8                                              0x00000100U
#define DCAN_TXTIE_TIE8_M                                            0x00000100U
#define DCAN_TXTIE_TIE8_S                                            8U
/*

        Field:           TIE9
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 9. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE9                                              0x00000200U
#define DCAN_TXTIE_TIE9_M                                            0x00000200U
#define DCAN_TXTIE_TIE9_S                                            9U
/*

        Field:           TIE10
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 10. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE10                                             0x00000400U
#define DCAN_TXTIE_TIE10_M                                           0x00000400U
#define DCAN_TXTIE_TIE10_S                                           10U
/*

        Field:           TIE11
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 11. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE11                                             0x00000800U
#define DCAN_TXTIE_TIE11_M                                           0x00000800U
#define DCAN_TXTIE_TIE11_S                                           11U
/*

        Field:           TIE12
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 12. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE12                                             0x00001000U
#define DCAN_TXTIE_TIE12_M                                           0x00001000U
#define DCAN_TXTIE_TIE12_S                                           12U
/*

        Field:           TIE13
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 13. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE13                                             0x00002000U
#define DCAN_TXTIE_TIE13_M                                           0x00002000U
#define DCAN_TXTIE_TIE13_S                                           13U
/*

        Field:           TIE14
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 14. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE14                                             0x00004000U
#define DCAN_TXTIE_TIE14_M                                           0x00004000U
#define DCAN_TXTIE_TIE14_S                                           14U
/*

        Field:           TIE15
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 15. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE15                                             0x00008000U
#define DCAN_TXTIE_TIE15_M                                           0x00008000U
#define DCAN_TXTIE_TIE15_S                                           15U
/*

        Field:           TIE16
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 16. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE16                                             0x00010000U
#define DCAN_TXTIE_TIE16_M                                           0x00010000U
#define DCAN_TXTIE_TIE16_S                                           16U
/*

        Field:           TIE17
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 17. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE17                                             0x00020000U
#define DCAN_TXTIE_TIE17_M                                           0x00020000U
#define DCAN_TXTIE_TIE17_S                                           17U
/*

        Field:           TIE18
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 18. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE18                                             0x00040000U
#define DCAN_TXTIE_TIE18_M                                           0x00040000U
#define DCAN_TXTIE_TIE18_S                                           18U
/*

        Field:           TIE19
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 19. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE19                                             0x00080000U
#define DCAN_TXTIE_TIE19_M                                           0x00080000U
#define DCAN_TXTIE_TIE19_S                                           19U
/*

        Field:           TIE20
        From..to bits:   20...20
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 20. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE20                                             0x00100000U
#define DCAN_TXTIE_TIE20_M                                           0x00100000U
#define DCAN_TXTIE_TIE20_S                                           20U
/*

        Field:           TIE21
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 21. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE21                                             0x00200000U
#define DCAN_TXTIE_TIE21_M                                           0x00200000U
#define DCAN_TXTIE_TIE21_S                                           21U
/*

        Field:           TIE22
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 22. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE22                                             0x00400000U
#define DCAN_TXTIE_TIE22_M                                           0x00400000U
#define DCAN_TXTIE_TIE22_S                                           22U
/*

        Field:           TIE23
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 23. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE23                                             0x00800000U
#define DCAN_TXTIE_TIE23_M                                           0x00800000U
#define DCAN_TXTIE_TIE23_S                                           23U
/*

        Field:           TIE24
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 24. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE24                                             0x01000000U
#define DCAN_TXTIE_TIE24_M                                           0x01000000U
#define DCAN_TXTIE_TIE24_S                                           24U
/*

        Field:           TIE25
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 25. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE25                                             0x02000000U
#define DCAN_TXTIE_TIE25_M                                           0x02000000U
#define DCAN_TXTIE_TIE25_S                                           25U
/*

        Field:           TIE26
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 26. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE26                                             0x04000000U
#define DCAN_TXTIE_TIE26_M                                           0x04000000U
#define DCAN_TXTIE_TIE26_S                                           26U
/*

        Field:           TIE27
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 27. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE27                                             0x08000000U
#define DCAN_TXTIE_TIE27_M                                           0x08000000U
#define DCAN_TXTIE_TIE27_S                                           27U
/*

        Field:           TIE28
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 28. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE28                                             0x10000000U
#define DCAN_TXTIE_TIE28_M                                           0x10000000U
#define DCAN_TXTIE_TIE28_S                                           28U
/*

        Field:           TIE29
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 29. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE29                                             0x20000000U
#define DCAN_TXTIE_TIE29_M                                           0x20000000U
#define DCAN_TXTIE_TIE29_S                                           29U
/*

        Field:           TIE30
        From..to bits:   30...30
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 30. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE30                                             0x40000000U
#define DCAN_TXTIE_TIE30_M                                           0x40000000U
#define DCAN_TXTIE_TIE30_S                                           30U
/*

        Field:           TIE31
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Transmission Interrupt Enable 31. Each Tx Buffer has its own Transmission Interrupt Enable bit.
                           0  Transmission interrupt disabled
                           1  Transmission interrupt enable

*/
#define DCAN_TXTIE_TIE31                                             0x80000000U
#define DCAN_TXTIE_TIE31_M                                           0x80000000U
#define DCAN_TXTIE_TIE31_S                                           31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXBCIE
    Offset name:         DCAN_O_TXBCIE
    Relative address:    0xE4
    Description:         MCAN Tx Buffer Cancellation Finished Interrupt Enable
    Default Value:       0x00000000

        Field:           CFIE0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 0. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE0                                            0x00000001U
#define DCAN_TXBCIE_CFIE0_M                                          0x00000001U
#define DCAN_TXBCIE_CFIE0_S                                          0U
/*

        Field:           CFIE1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 1. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE1                                            0x00000002U
#define DCAN_TXBCIE_CFIE1_M                                          0x00000002U
#define DCAN_TXBCIE_CFIE1_S                                          1U
/*

        Field:           CFIE2
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 2. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE2                                            0x00000004U
#define DCAN_TXBCIE_CFIE2_M                                          0x00000004U
#define DCAN_TXBCIE_CFIE2_S                                          2U
/*

        Field:           CFIE3
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 3. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE3                                            0x00000008U
#define DCAN_TXBCIE_CFIE3_M                                          0x00000008U
#define DCAN_TXBCIE_CFIE3_S                                          3U
/*

        Field:           CFIE4
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 4. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE4                                            0x00000010U
#define DCAN_TXBCIE_CFIE4_M                                          0x00000010U
#define DCAN_TXBCIE_CFIE4_S                                          4U
/*

        Field:           CFIE5
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 5. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE5                                            0x00000020U
#define DCAN_TXBCIE_CFIE5_M                                          0x00000020U
#define DCAN_TXBCIE_CFIE5_S                                          5U
/*

        Field:           CFIE6
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 6. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE6                                            0x00000040U
#define DCAN_TXBCIE_CFIE6_M                                          0x00000040U
#define DCAN_TXBCIE_CFIE6_S                                          6U
/*

        Field:           CFIE7
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 7. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE7                                            0x00000080U
#define DCAN_TXBCIE_CFIE7_M                                          0x00000080U
#define DCAN_TXBCIE_CFIE7_S                                          7U
/*

        Field:           CFIE8
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 8. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE8                                            0x00000100U
#define DCAN_TXBCIE_CFIE8_M                                          0x00000100U
#define DCAN_TXBCIE_CFIE8_S                                          8U
/*

        Field:           CFIE9
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 9. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE9                                            0x00000200U
#define DCAN_TXBCIE_CFIE9_M                                          0x00000200U
#define DCAN_TXBCIE_CFIE9_S                                          9U
/*

        Field:           CFIE10
        From..to bits:   10...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 10. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE10                                           0x00000400U
#define DCAN_TXBCIE_CFIE10_M                                         0x00000400U
#define DCAN_TXBCIE_CFIE10_S                                         10U
/*

        Field:           CFIE11
        From..to bits:   11...11
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 11. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE11                                           0x00000800U
#define DCAN_TXBCIE_CFIE11_M                                         0x00000800U
#define DCAN_TXBCIE_CFIE11_S                                         11U
/*

        Field:           CFIE12
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 12. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE12                                           0x00001000U
#define DCAN_TXBCIE_CFIE12_M                                         0x00001000U
#define DCAN_TXBCIE_CFIE12_S                                         12U
/*

        Field:           CFIE13
        From..to bits:   13...13
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 13. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE13                                           0x00002000U
#define DCAN_TXBCIE_CFIE13_M                                         0x00002000U
#define DCAN_TXBCIE_CFIE13_S                                         13U
/*

        Field:           CFIE14
        From..to bits:   14...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 14. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE14                                           0x00004000U
#define DCAN_TXBCIE_CFIE14_M                                         0x00004000U
#define DCAN_TXBCIE_CFIE14_S                                         14U
/*

        Field:           CFIE15
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 15. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE15                                           0x00008000U
#define DCAN_TXBCIE_CFIE15_M                                         0x00008000U
#define DCAN_TXBCIE_CFIE15_S                                         15U
/*

        Field:           CFIE16
        From..to bits:   16...16
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 16. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE16                                           0x00010000U
#define DCAN_TXBCIE_CFIE16_M                                         0x00010000U
#define DCAN_TXBCIE_CFIE16_S                                         16U
/*

        Field:           CFIE17
        From..to bits:   17...17
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 17. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE17                                           0x00020000U
#define DCAN_TXBCIE_CFIE17_M                                         0x00020000U
#define DCAN_TXBCIE_CFIE17_S                                         17U
/*

        Field:           CFIE18
        From..to bits:   18...18
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 18. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE18                                           0x00040000U
#define DCAN_TXBCIE_CFIE18_M                                         0x00040000U
#define DCAN_TXBCIE_CFIE18_S                                         18U
/*

        Field:           CFIE19
        From..to bits:   19...19
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 19. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE19                                           0x00080000U
#define DCAN_TXBCIE_CFIE19_M                                         0x00080000U
#define DCAN_TXBCIE_CFIE19_S                                         19U
/*

        Field:           CFIE20
        From..to bits:   20...20
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 20. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE20                                           0x00100000U
#define DCAN_TXBCIE_CFIE20_M                                         0x00100000U
#define DCAN_TXBCIE_CFIE20_S                                         20U
/*

        Field:           CFIE21
        From..to bits:   21...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 21. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE21                                           0x00200000U
#define DCAN_TXBCIE_CFIE21_M                                         0x00200000U
#define DCAN_TXBCIE_CFIE21_S                                         21U
/*

        Field:           CFIE22
        From..to bits:   22...22
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 22. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE22                                           0x00400000U
#define DCAN_TXBCIE_CFIE22_M                                         0x00400000U
#define DCAN_TXBCIE_CFIE22_S                                         22U
/*

        Field:           CFIE23
        From..to bits:   23...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 23. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE23                                           0x00800000U
#define DCAN_TXBCIE_CFIE23_M                                         0x00800000U
#define DCAN_TXBCIE_CFIE23_S                                         23U
/*

        Field:           CFIE24
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 24. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE24                                           0x01000000U
#define DCAN_TXBCIE_CFIE24_M                                         0x01000000U
#define DCAN_TXBCIE_CFIE24_S                                         24U
/*

        Field:           CFIE25
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 25. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE25                                           0x02000000U
#define DCAN_TXBCIE_CFIE25_M                                         0x02000000U
#define DCAN_TXBCIE_CFIE25_S                                         25U
/*

        Field:           CFIE26
        From..to bits:   26...26
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 26. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE26                                           0x04000000U
#define DCAN_TXBCIE_CFIE26_M                                         0x04000000U
#define DCAN_TXBCIE_CFIE26_S                                         26U
/*

        Field:           CFIE27
        From..to bits:   27...27
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 27. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE27                                           0x08000000U
#define DCAN_TXBCIE_CFIE27_M                                         0x08000000U
#define DCAN_TXBCIE_CFIE27_S                                         27U
/*

        Field:           CFIE28
        From..to bits:   28...28
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 28. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE28                                           0x10000000U
#define DCAN_TXBCIE_CFIE28_M                                         0x10000000U
#define DCAN_TXBCIE_CFIE28_S                                         28U
/*

        Field:           CFIE29
        From..to bits:   29...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 29. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE29                                           0x20000000U
#define DCAN_TXBCIE_CFIE29_M                                         0x20000000U
#define DCAN_TXBCIE_CFIE29_S                                         29U
/*

        Field:           CFIE30
        From..to bits:   30...30
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 30. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE30                                           0x40000000U
#define DCAN_TXBCIE_CFIE30_M                                         0x40000000U
#define DCAN_TXBCIE_CFIE30_S                                         30U
/*

        Field:           CFIE31
        From..to bits:   31...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Cancellation Finished Interrupt Enable 31. Each Tx Buffer has its own Cancellation Finished Interrupt Enable bit.
                           0  Cancellation finished interrupt disabled
                           1  Cancellation finished interrupt enabled

*/
#define DCAN_TXBCIE_CFIE31                                           0x80000000U
#define DCAN_TXBCIE_CFIE31_M                                         0x80000000U
#define DCAN_TXBCIE_CFIE31_S                                         31U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXEFC
    Offset name:         DCAN_O_TXEFC
    Relative address:    0xF0
    Description:         MCAN Tx Event FIFO Configuration
    Default Value:       0x00000000

        Field:           EFSA
        From..to bits:   2...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Event FIFO Start Address. Start address of Tx Event FIFO in Message RAM (32-bit word address).

*/
#define DCAN_TXEFC_EFSA_W                                            14U
#define DCAN_TXEFC_EFSA_M                                            0x0000FFFCU
#define DCAN_TXEFC_EFSA_S                                            2U
/*

        Field:           EFS
        From..to bits:   16...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Event FIFO Size. The Tx Event FIFO elements are indexed from 0 to EFS - 1.
                           0      Tx Event FIFO disabled
                           1-32  Number of Tx Event FIFO elements
                           >32   Values greater than 32 are interpreted as 32

*/
#define DCAN_TXEFC_EFS_W                                             6U
#define DCAN_TXEFC_EFS_M                                             0x003F0000U
#define DCAN_TXEFC_EFS_S                                             16U
/*

        Field:           EFWM
        From..to bits:   24...29
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Event FIFO Watermark
                           0      Watermark interrupt disabled
                           1-32  Level for Tx Event FIFO watermark interrupt (IR.TEFW)
                           >32   Watermark interrupt disabled

*/
#define DCAN_TXEFC_EFWM_W                                            6U
#define DCAN_TXEFC_EFWM_M                                            0x3F000000U
#define DCAN_TXEFC_EFWM_S                                            24U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXEFS
    Offset name:         DCAN_O_TXEFS
    Relative address:    0xF4
    Description:         MCAN Tx Event FIFO Status
    Default Value:       0x00000000

        Field:           EFFL
        From..to bits:   0...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Event FIFO Fill Level. Number of elements stored in Tx Event FIFO, range 0 to 32.

*/
#define DCAN_TXEFS_EFFL_W                                            6U
#define DCAN_TXEFS_EFFL_M                                            0x0000003FU
#define DCAN_TXEFS_EFFL_S                                            0U
/*

        Field:           EFGI
        From..to bits:   8...12
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Event FIFO Get Index. Tx Event FIFO read index pointer, range 0 to 31.

*/
#define DCAN_TXEFS_EFGI_W                                            5U
#define DCAN_TXEFS_EFGI_M                                            0x00001F00U
#define DCAN_TXEFS_EFGI_S                                            8U
/*

        Field:           EFPI
        From..to bits:   16...20
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Event FIFO Put Index.Tx Event FIFO write index pointer, range 0 to 31.

*/
#define DCAN_TXEFS_EFPI_W                                            5U
#define DCAN_TXEFS_EFPI_M                                            0x001F0000U
#define DCAN_TXEFS_EFPI_S                                            16U
/*

        Field:           EFF
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Event FIFO Full
                           0  Tx Event FIFO not full
                           1  Tx Event FIFO full

*/
#define DCAN_TXEFS_EFF                                               0x01000000U
#define DCAN_TXEFS_EFF_M                                             0x01000000U
#define DCAN_TXEFS_EFF_S                                             24U
/*

        Field:           TEFL
        From..to bits:   25...25
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Tx Event FIFO Element Lost. This bit is a copy of interrupt flag IR.TEFL. When IR.TEFL is reset, this bit is also reset.
                           0  No Tx Event FIFO element lost
                           1  Tx Event FIFO element lost, also set after write attempt to Tx Event FIFO of size zero.

*/
#define DCAN_TXEFS_TEFL                                              0x02000000U
#define DCAN_TXEFS_TEFL_M                                            0x02000000U
#define DCAN_TXEFS_TEFL_S                                            25U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TXEFA
    Offset name:         DCAN_O_TXEFA
    Relative address:    0xF8
    Description:         MCAN Tx Event FIFO Acknowledge
    Default Value:       0x00000000

        Field:           EFAI
        From..to bits:   0...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Event FIFO Acknowledge Index. After the Host has read an element or a sequence of elements from the Tx Event FIFO it has to write the index of the last element read from Tx Event FIFO to EFAI. This will set the Tx Event FIFO Get Index TXEFS.EFGI to EFAI + 1 and update the Event FIFO Fill Level TXEFS.EFFL.

*/
#define DCAN_TXEFA_EFAI_W                                            5U
#define DCAN_TXEFA_EFAI_M                                            0x0000001FU
#define DCAN_TXEFA_EFAI_S                                            0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SSPID
    Offset name:         DCAN_O_SSPID
    Relative address:    0x200
    Description:         MCAN Subsystem Revision Register
    Default Value:       0x68E04901

        Field:           MINOR
        From..to bits:   0...5
        DefaultValue:    0x1
        Access type:     read-only
        Description:     Minor Revision of the MCAN Subsystem

*/
#define DCAN_SSPID_MINOR_W                                           6U
#define DCAN_SSPID_MINOR_M                                           0x0000003FU
#define DCAN_SSPID_MINOR_S                                           0U
/*

        Field:           CUSTOM
        From..to bits:   6...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Custom Value

*/
#define DCAN_SSPID_CUSTOM_W                                          2U
#define DCAN_SSPID_CUSTOM_M                                          0x000000C0U
#define DCAN_SSPID_CUSTOM_S                                          6U
/*

        Field:           MAJOR
        From..to bits:   8...10
        DefaultValue:    0x1
        Access type:     read-only
        Description:     Major Revision of the MCAN Subsystem

*/
#define DCAN_SSPID_MAJOR_W                                           3U
#define DCAN_SSPID_MAJOR_M                                           0x00000700U
#define DCAN_SSPID_MAJOR_S                                           8U
/*

        Field:           RTL
        From..to bits:   11...15
        DefaultValue:    0x9
        Access type:     read-only
        Description:     RTL revision. Will vary depending on release

*/
#define DCAN_SSPID_RTL_W                                             5U
#define DCAN_SSPID_RTL_M                                             0x0000F800U
#define DCAN_SSPID_RTL_S                                             11U
/*

        Field:           MODULEID
        From..to bits:   16...27
        DefaultValue:    0x8E0
        Access type:     read-only
        Description:     Module Identification Number

*/
#define DCAN_SSPID_MODULEID_W                                        12U
#define DCAN_SSPID_MODULEID_M                                        0x0FFF0000U
#define DCAN_SSPID_MODULEID_S                                        16U
/*

        Field:           BU
        From..to bits:   28...29
        DefaultValue:    0x2
        Access type:     read-only
        Description:     Business Unit: 0x2 = Processors

*/
#define DCAN_SSPID_BU_W                                              2U
#define DCAN_SSPID_BU_M                                              0x30000000U
#define DCAN_SSPID_BU_S                                              28U
/*

        Field:           SCHEME
        From..to bits:   30...31
        DefaultValue:    0x1
        Access type:     read-only
        Description:     PID Register Scheme

*/
#define DCAN_SSPID_SCHEME_W                                          2U
#define DCAN_SSPID_SCHEME_M                                          0xC0000000U
#define DCAN_SSPID_SCHEME_S                                          30U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SSCTL
    Offset name:         DCAN_O_SSCTL
    Relative address:    0x204
    Description:         MCAN Subsystem Control Register
    Default Value:       0x00000008

        Field:           DBGSF
        From..to bits:   3...3
        DefaultValue:    0x1
        Access type:     read-write
        Description:     Debug Suspend Free Bit. Enables debug suspend.
                           0  Disable debug suspend
                           1  Enable debug suspend

*/
#define DCAN_SSCTL_DBGSF                                             0x00000008U
#define DCAN_SSCTL_DBGSF_M                                           0x00000008U
#define DCAN_SSCTL_DBGSF_S                                           3U
/*

        Field:           WUREQEN
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Wakeup Request Enable. Enables the MCANSS to wakeup on CAN RXD activity.
                           0  Disable wakeup request
                           1  Enables wakeup request

*/
#define DCAN_SSCTL_WUREQEN                                           0x00000010U
#define DCAN_SSCTL_WUREQEN_M                                         0x00000010U
#define DCAN_SSCTL_WUREQEN_S                                         4U
/*

        Field:           AUTOWU
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Automatic Wakeup Enable. Enables the MCANSS to automatically clear the MCAN CCCR.INIT bit, fully waking the MCAN up, on an enabled wakeup request.
                           0  Disable the automatic write to CCCR.INIT
                           1  Enable the automatic write to CCCR.INIT

*/
#define DCAN_SSCTL_AUTOWU                                            0x00000020U
#define DCAN_SSCTL_AUTOWU_M                                          0x00000020U
#define DCAN_SSCTL_AUTOWU_S                                          5U
/*

        Field:           EXTTSCNTEN
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     External Timestamp Counter Enable.
                           0  External timestamp counter disabled
                           1  External timestamp counter enabled

*/
#define DCAN_SSCTL_EXTTSCNTEN                                        0x00000040U
#define DCAN_SSCTL_EXTTSCNTEN_M                                      0x00000040U
#define DCAN_SSCTL_EXTTSCNTEN_S                                      6U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SSSTA
    Offset name:         DCAN_O_SSSTA
    Relative address:    0x208
    Description:         MCAN Subsystem Status Register
    Default Value:       0x00000000

        Field:           RESET
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Soft Reset Status.
                           0  Not in reset
                           1  Reset is in progress

*/
#define DCAN_SSSTA_RESET                                             0x00000001U
#define DCAN_SSSTA_RESET_M                                           0x00000001U
#define DCAN_SSSTA_RESET_S                                           0U
/*

        Field:           MEMINITSTA
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Memory Initialization Done.
                           0  Message RAM initialization is in progress
                           1  Message RAM is initialized for use

*/
#define DCAN_SSSTA_MEMINITSTA                                        0x00000002U
#define DCAN_SSSTA_MEMINITSTA_M                                      0x00000002U
#define DCAN_SSSTA_MEMINITSTA_S                                      1U
/*

        Field:           ENFDOE
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Flexible Datarate Operation Enable. Determines whether CAN FD operation may be enabled via the MCAN core CCCR.FDOE bit (bit 8) or if only standard CAN operation is possible with this instance of the MCAN.
                           0  MCAN is only capable of standard CAN communication
                           1  MCAN may be configured to perform CAN FD communication

*/
#define DCAN_SSSTA_ENFDOE                                            0x00000004U
#define DCAN_SSSTA_ENFDOE_M                                          0x00000004U
#define DCAN_SSSTA_ENFDOE_S                                          2U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SSICS
    Offset name:         DCAN_O_SSICS
    Relative address:    0x20C
    Description:         MCAN Subsystem Interrupt Clear Shadow Register
    Default Value:       0x00000000

        Field:           TSCNTOVFL
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     External Timestamp Counter Overflow Interrupt Status Clear. Reads always return a 0.
                           0  Write of '0' has no effect
                           1  Write of '1' clears the MCANSS_IRS.EXT_TS_CNTR_OVFL bit

*/
#define DCAN_SSICS_TSCNTOVFL                                         0x00000001U
#define DCAN_SSICS_TSCNTOVFL_M                                       0x00000001U
#define DCAN_SSICS_TSCNTOVFL_S                                       0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SSIRS
    Offset name:         DCAN_O_SSIRS
    Relative address:    0x210
    Description:         MCAN Subsystem Interrupt Raw Status Register
    Default Value:       0x00000000

        Field:           TSCNTOVFL
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     External Timestamp Counter Overflow Interrupt Status. This bit is set by HW or by a SW write of '1'. To clear, use the MCANSS_ICS.EXT_TS_CNTR_OVFL bit.
                           0  External timestamp counter has not overflowed
                           1  External timestamp counter has overflowed

                         When this bit is set to '1' by HW or SW, the MCANSS_EXT_TS_UNSERVICED_INTR_CNTR.EXT_TS_INTR_CNTR bit field will increment by 1.

*/
#define DCAN_SSIRS_TSCNTOVFL                                         0x00000001U
#define DCAN_SSIRS_TSCNTOVFL_M                                       0x00000001U
#define DCAN_SSIRS_TSCNTOVFL_S                                       0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SSIECS
    Offset name:         DCAN_O_SSIECS
    Relative address:    0x214
    Description:         MCAN Subsystem Interrupt Enable Clear Shadow Register
    Default Value:       0x00000000

        Field:           TSCNTOVFL
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     External Timestamp Counter Overflow Interrupt Enable Clear. Reads always return a 0.
                           0  Write of '0' has no effect
                           1  Write of '1' clears the MCANSS_IES.EXT_TS_CNTR_OVFL bit

*/
#define DCAN_SSIECS_TSCNTOVFL                                        0x00000001U
#define DCAN_SSIECS_TSCNTOVFL_M                                      0x00000001U
#define DCAN_SSIECS_TSCNTOVFL_S                                      0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SSIE
    Offset name:         DCAN_O_SSIE
    Relative address:    0x218
    Description:         MCAN Subsystem Interrupt Enable Register
    Default Value:       0x00000000

        Field:           TSCNTOVFL
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     External Timestamp Counter Overflow Interrupt Enable. A write of '0' has no effect. A write of '1' sets the MCANSS_IES.EXT_TS_CNTR_OVFL bit.

*/
#define DCAN_SSIE_TSCNTOVFL                                          0x00000001U
#define DCAN_SSIE_TSCNTOVFL_M                                        0x00000001U
#define DCAN_SSIE_TSCNTOVFL_S                                        0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SSIES
    Offset name:         DCAN_O_SSIES
    Relative address:    0x21C
    Description:         MCAN Subsystem Masked Interrupt Status. It is the logical AND of IRS and IE for the respective bits.
    Default Value:       0x00000000

        Field:           TSCNTOVFL
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     External Timestamp Counter Overflow masked interrupt status.
                           0  External timestamp counter overflow interrupt is cleared
                           1  External timestamp counter overflow interrupt is set

*/
#define DCAN_SSIES_TSCNTOVFL                                         0x00000001U
#define DCAN_SSIES_TSCNTOVFL_M                                       0x00000001U
#define DCAN_SSIES_TSCNTOVFL_S                                       0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SSEOI
    Offset name:         DCAN_O_SSEOI
    Relative address:    0x220
    Description:         MCAN Subsystem End of Interrupt
    Default Value:       0x00000000

        Field:           EOI
        From..to bits:   0...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     End of Interrupt. A write to this register will clear the associated interrupt. If the unserviced interrupt counter is > 1, another interrupt is generated.
                           0x00  External TS Interrupt is cleared
                           0x01  MCAN(0) interrupt is cleared
                           0x02  MCAN(1) interrupt is cleared
                           Other writes are ignored.

*/
#define DCAN_SSEOI_EOI_W                                             8U
#define DCAN_SSEOI_EOI_M                                             0x000000FFU
#define DCAN_SSEOI_EOI_S                                             0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       EXTTSPS
    Offset name:         DCAN_O_EXTTSPS
    Relative address:    0x224
    Description:         MCAN Subsystem External Timestamp Prescaler 0
    Default Value:       0x00000000

        Field:           PRESCALER
        From..to bits:   0...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     External Timestamp Prescaler Reload Value. The external timestamp count rate is the host (system) clock rate divided by this value, except in the case of 0. A zero value in this bit field will act identically to a value of 0x000001.

*/
#define DCAN_EXTTSPS_PRESCALER_W                                     24U
#define DCAN_EXTTSPS_PRESCALER_M                                     0x00FFFFFFU
#define DCAN_EXTTSPS_PRESCALER_S                                     0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       EXTTSUSI
    Offset name:         DCAN_O_EXTTSUSI
    Relative address:    0x228
    Description:         MCAN Subsystem External Timestamp Unserviced Interrupts Counter
    Default Value:       0x00000000

        Field:           INTRCNT
        From..to bits:   0...4
        DefaultValue:    0x0
        Access type:     read-only
        Description:     External Timestamp Counter Unserviced Rollover Interrupts. If this value is > 1, an MCANSS_EOI write of '1' to bit 0 will issue another interrupt.

                         The status of this bit field is affected by the MCANSS_IRS.EXT_TS_CNTR_OVFL bit field.

*/
#define DCAN_EXTTSUSI_INTRCNT_W                                      5U
#define DCAN_EXTTSUSI_INTRCNT_M                                      0x0000001FU
#define DCAN_EXTTSUSI_INTRCNT_S                                      0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ERRREV
    Offset name:         DCAN_O_ERRREV
    Relative address:    0x400
    Description:         MCAN Error Aggregator Revision Register
    Default Value:       0x66A0EA00

        Field:           REVMIN
        From..to bits:   0...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Minor Revision of the Error Aggregator

*/
#define DCAN_ERRREV_REVMIN_W                                         6U
#define DCAN_ERRREV_REVMIN_M                                         0x0000003FU
#define DCAN_ERRREV_REVMIN_S                                         0U
/*

        Field:           REVCUSTOM
        From..to bits:   6...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Custom Revision of the Error Aggregator

*/
#define DCAN_ERRREV_REVCUSTOM_W                                      2U
#define DCAN_ERRREV_REVCUSTOM_M                                      0x000000C0U
#define DCAN_ERRREV_REVCUSTOM_S                                      6U
/*

        Field:           REVMAJ
        From..to bits:   8...10
        DefaultValue:    0x2
        Access type:     read-only
        Description:     Major Revision of the Error Aggregator

*/
#define DCAN_ERRREV_REVMAJ_W                                         3U
#define DCAN_ERRREV_REVMAJ_M                                         0x00000700U
#define DCAN_ERRREV_REVMAJ_S                                         8U
/*

        Field:           REVRTL
        From..to bits:   11...15
        DefaultValue:    0x1D
        Access type:     read-only
        Description:     RTL revision. Will vary depending on release

*/
#define DCAN_ERRREV_REVRTL_W                                         5U
#define DCAN_ERRREV_REVRTL_M                                         0x0000F800U
#define DCAN_ERRREV_REVRTL_S                                         11U
/*

        Field:           MODULEID
        From..to bits:   16...27
        DefaultValue:    0x6A0
        Access type:     read-only
        Description:     Module Identification Number

*/
#define DCAN_ERRREV_MODULEID_W                                       12U
#define DCAN_ERRREV_MODULEID_M                                       0x0FFF0000U
#define DCAN_ERRREV_MODULEID_S                                       16U
/*

        Field:           BU
        From..to bits:   28...29
        DefaultValue:    0x2
        Access type:     read-only
        Description:     Business Unit: 0x2 = Processors

*/
#define DCAN_ERRREV_BU_W                                             2U
#define DCAN_ERRREV_BU_M                                             0x30000000U
#define DCAN_ERRREV_BU_S                                             28U
/*

        Field:           SCHEME
        From..to bits:   30...31
        DefaultValue:    0x1
        Access type:     read-only
        Description:     PID Register Scheme

*/
#define DCAN_ERRREV_SCHEME_W                                         2U
#define DCAN_ERRREV_SCHEME_M                                         0xC0000000U
#define DCAN_ERRREV_SCHEME_S                                         30U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ERRVEC
    Offset name:         DCAN_O_ERRVEC
    Relative address:    0x408
    Description:         Each error detection and correction (EDC) controller has a bank of error registers (offsets 0x10 - 0x3B) associated with it. These registers are accessed via an internal serial bus (SVBUS). To access them through the ECC aggregator the controller ID desired must be written to the ECC_VECTOR field, together with the RD_SVBUS trigger and RD_SVBUS_ADDRESS bit field. This initiates the serial read which consummates by setting the RD_SVBUS_DONE bit. At this point the addressed register may be read by a normal CPU read of the appropriate offset address.
    Default Value:       0x00000000

        Field:           ECCVEC
        From..to bits:   0...10
        DefaultValue:    0x0
        Access type:     read-write
        Description:     ECC RAM ID. Each error detection and correction (EDC) controller has a bank of error registers (offsets 0x10 - 0x3B) associated with it. These registers are accessed via an internal serial bus (SVBUS). To access them through the ECC aggregator the controller ID desired must be written to the ECC_VECTOR field, together with the RD_SVBUS trigger and RD_SVBUS_ADDRESS bit field. This initiates the serial read which consummates by setting the RD_SVBUS_DONE bit. At this point the addressed register may be read by a normal CPU read of the appropriate offset address.
                           0x000  Message RAM ECC controller is selected
                           Others  Reserved (do not use)

                         Subsequent writes through the SVBUS (offsets 0x10 - 0x3B) have a delayed completion. To avoid conflicts, perform a read back of a register within this range after writing.

*/
#define DCAN_ERRVEC_ECCVEC_W                                         11U
#define DCAN_ERRVEC_ECCVEC_M                                         0x000007FFU
#define DCAN_ERRVEC_ECCVEC_S                                         0U
/*

        Field:           RDSVBUS
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Read Trigger

*/
#define DCAN_ERRVEC_RDSVBUS                                          0x00008000U
#define DCAN_ERRVEC_RDSVBUS_M                                        0x00008000U
#define DCAN_ERRVEC_RDSVBUS_S                                        15U
/*

        Field:           RDSVBUSA
        From..to bits:   16...23
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Read Address Offset

*/
#define DCAN_ERRVEC_RDSVBUSA_W                                       8U
#define DCAN_ERRVEC_RDSVBUSA_M                                       0x00FF0000U
#define DCAN_ERRVEC_RDSVBUSA_S                                       16U
/*

        Field:           SVBUSDONE
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Read Completion Flag

*/
#define DCAN_ERRVEC_SVBUSDONE                                        0x01000000U
#define DCAN_ERRVEC_SVBUSDONE_M                                      0x01000000U
#define DCAN_ERRVEC_SVBUSDONE_S                                      24U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ERRSTA
    Offset name:         DCAN_O_ERRSTA
    Relative address:    0x40C
    Description:         MCAN Error Misc Status
    Default Value:       0x00000002

        Field:           NUMRAMS
        From..to bits:   0...10
        DefaultValue:    0x2
        Access type:     read-only
        Description:     Number of RAMs. Number of ECC RAMs serviced by the aggregator.

*/
#define DCAN_ERRSTA_NUMRAMS_W                                        11U
#define DCAN_ERRSTA_NUMRAMS_M                                        0x000007FFU
#define DCAN_ERRSTA_NUMRAMS_S                                        0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ERRWRAPREV
    Offset name:         DCAN_O_ERRWRAPREV
    Relative address:    0x410
    Description:         This register is accessed through the ECC aggregator via an internal serial bus. To access, the appropriate procedure must be first followed in the MCAN ECC Vector Register.
    Default Value:       0x66A46A02

        Field:           REVMIN
        From..to bits:   0...5
        DefaultValue:    0x2
        Access type:     read-only
        Description:     Minor Revision of the Error Aggregator

*/
#define DCAN_ERRWRAPREV_REVMIN_W                                     6U
#define DCAN_ERRWRAPREV_REVMIN_M                                     0x0000003FU
#define DCAN_ERRWRAPREV_REVMIN_S                                     0U
/*

        Field:           REVCUSTOM
        From..to bits:   6...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Custom Revision of the Error Aggregator

*/
#define DCAN_ERRWRAPREV_REVCUSTOM_W                                  2U
#define DCAN_ERRWRAPREV_REVCUSTOM_M                                  0x000000C0U
#define DCAN_ERRWRAPREV_REVCUSTOM_S                                  6U
/*

        Field:           REVMAJ
        From..to bits:   8...10
        DefaultValue:    0x2
        Access type:     read-only
        Description:     Major Revision of the Error Aggregator

*/
#define DCAN_ERRWRAPREV_REVMAJ_W                                     3U
#define DCAN_ERRWRAPREV_REVMAJ_M                                     0x00000700U
#define DCAN_ERRWRAPREV_REVMAJ_S                                     8U
/*

        Field:           REVRTL
        From..to bits:   11...15
        DefaultValue:    0xD
        Access type:     read-only
        Description:     RTL revision. Will vary depending on release

*/
#define DCAN_ERRWRAPREV_REVRTL_W                                     5U
#define DCAN_ERRWRAPREV_REVRTL_M                                     0x0000F800U
#define DCAN_ERRWRAPREV_REVRTL_S                                     11U
/*

        Field:           MODULEID
        From..to bits:   16...27
        DefaultValue:    0x6A4
        Access type:     read-only
        Description:     Module Identification Number

*/
#define DCAN_ERRWRAPREV_MODULEID_W                                   12U
#define DCAN_ERRWRAPREV_MODULEID_M                                   0x0FFF0000U
#define DCAN_ERRWRAPREV_MODULEID_S                                   16U
/*

        Field:           BU
        From..to bits:   28...29
        DefaultValue:    0x2
        Access type:     read-only
        Description:     Business Unit: 0x2 = Processors

*/
#define DCAN_ERRWRAPREV_BU_W                                         2U
#define DCAN_ERRWRAPREV_BU_M                                         0x30000000U
#define DCAN_ERRWRAPREV_BU_S                                         28U
/*

        Field:           SCHEME
        From..to bits:   30...31
        DefaultValue:    0x1
        Access type:     read-only
        Description:     PID Register Scheme

*/
#define DCAN_ERRWRAPREV_SCHEME_W                                     2U
#define DCAN_ERRWRAPREV_SCHEME_M                                     0xC0000000U
#define DCAN_ERRWRAPREV_SCHEME_S                                     30U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ERRCTL
    Offset name:         DCAN_O_ERRCTL
    Relative address:    0x414
    Description:         This register is accessed through the ECC aggregator via an internal serial bus. To access, the appropriate procedure must be first followed in the MCAN ECC Vector Register.
    Default Value:       0x00000187

        Field:           ECCEN
        From..to bits:   0...0
        DefaultValue:    0x1
        Access type:     read-write
        Description:     Enable ECC Generation

*/
#define DCAN_ERRCTL_ECCEN                                            0x00000001U
#define DCAN_ERRCTL_ECCEN_M                                          0x00000001U
#define DCAN_ERRCTL_ECCEN_S                                          0U
/*

        Field:           ECCCHECK
        From..to bits:   1...1
        DefaultValue:    0x1
        Access type:     read-write
        Description:     Enable ECC Check. ECC is completely bypassed if both ECC_ENABLE and ECC_CHECK are '0'.

*/
#define DCAN_ERRCTL_ECCCHECK                                         0x00000002U
#define DCAN_ERRCTL_ECCCHECK_M                                       0x00000002U
#define DCAN_ERRCTL_ECCCHECK_S                                       1U
/*

        Field:           ENRMW
        From..to bits:   2...2
        DefaultValue:    0x1
        Access type:     read-write
        Description:     Enable read-modify-write on partial word writes

*/
#define DCAN_ERRCTL_ENRMW                                            0x00000004U
#define DCAN_ERRCTL_ENRMW_M                                          0x00000004U
#define DCAN_ERRCTL_ENRMW_S                                          2U
/*

        Field:           FRCSEC
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Force single-bit error. Cleared on a writeback or the cycle following the error if ERROR_ONCE is asserted. For write through mode, this applies to writes as well as reads. MCANERR_ERR_CTRL1 and MCANERR_ERR_CTRL2 should be configured prior to setting this bit.

*/
#define DCAN_ERRCTL_FRCSEC                                           0x00000008U
#define DCAN_ERRCTL_FRCSEC_M                                         0x00000008U
#define DCAN_ERRCTL_FRCSEC_S                                         3U
/*

        Field:           FRCDED
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Force double-bit error. Cleared the cycle following the error if ERROR_ONCE is asserted. For write through mode, this applies to writes as well as reads. MCANERR_ERR_CTRL1 and MCANERR_ERR_CTRL2 should be configured prior to setting this bit.

*/
#define DCAN_ERRCTL_FRCDED                                           0x00000010U
#define DCAN_ERRCTL_FRCDED_M                                         0x00000010U
#define DCAN_ERRCTL_FRCDED_S                                         4U
/*

        Field:           FRCNROW
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Enable single/double-bit error on the next RAM read, regardless of the MCANERR_ERR_CTRL1.ECC_ROW setting. For write through mode, this applies to writes as well as reads.

*/
#define DCAN_ERRCTL_FRCNROW                                          0x00000020U
#define DCAN_ERRCTL_FRCNROW_M                                        0x00000020U
#define DCAN_ERRCTL_FRCNROW_S                                        5U
/*

        Field:           ERRONCE
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     If this bit is set, the FORCE_SEC/FORCE_DED will inject an error to the specified row only once. The FORCE_SEC bit will be cleared once a writeback happens. If writeback is not enabled, this error will be cleared the cycle following the read when the data is corrected. For double-bit errors, the FORCE_DED bit will be cleared the cycle following the double-bit error. Any subsequent reads will not force an error.

*/
#define DCAN_ERRCTL_ERRONCE                                          0x00000040U
#define DCAN_ERRCTL_ERRONCE_M                                        0x00000040U
#define DCAN_ERRCTL_ERRONCE_S                                        6U
/*

        Field:           CHECKPAR
        From..to bits:   7...7
        DefaultValue:    0x1
        Access type:     read-write
        Description:     Enables parity checking on internal data

*/
#define DCAN_ERRCTL_CHECKPAR                                         0x00000080U
#define DCAN_ERRCTL_CHECKPAR_M                                       0x00000080U
#define DCAN_ERRCTL_CHECKPAR_S                                       7U
/*

        Field:           CHECKSVBTO
        From..to bits:   8...8
        DefaultValue:    0x1
        Access type:     read-write
        Description:     Enables Serial VBUS timeout mechanism

*/
#define DCAN_ERRCTL_CHECKSVBTO                                       0x00000100U
#define DCAN_ERRCTL_CHECKSVBTO_M                                     0x00000100U
#define DCAN_ERRCTL_CHECKSVBTO_S                                     8U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ERRCTL1
    Offset name:         DCAN_O_ERRCTL1
    Relative address:    0x418
    Description:         This register is accessed through the ECC aggregator via an internal serial bus. To access, the appropriate procedure must be first followed in the MCAN ECC Vector Register.
    Default Value:       0x00000000

        Field:           ECCROW
        From..to bits:   0...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Row address where FORCE_SEC or FORCE_DED needs to be applied. This is ignored if FORCE_N_ROW is set.

*/
#define DCAN_ERRCTL1_ECCROW_W                                        32U
#define DCAN_ERRCTL1_ECCROW_M                                        0xFFFFFFFFU
#define DCAN_ERRCTL1_ECCROW_S                                        0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ERRCTL2
    Offset name:         DCAN_O_ERRCTL2
    Relative address:    0x41C
    Description:         This register is accessed through the ECC aggregator via an internal serial bus. To access, the appropriate procedure must be first followed in the MCAN ECC Vector Register.
    Default Value:       0x00000000

        Field:           ECCB1
        From..to bits:   0...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Column/Data bit that needs to be flipped when FORCE_SEC or FORCE_DED is set

*/
#define DCAN_ERRCTL2_ECCB1_W                                         16U
#define DCAN_ERRCTL2_ECCB1_M                                         0x0000FFFFU
#define DCAN_ERRCTL2_ECCB1_S                                         0U
/*

        Field:           ECCB2
        From..to bits:   16...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Second column/data bit that needs to be flipped when FORCE_DED is set

*/
#define DCAN_ERRCTL2_ECCB2_W                                         16U
#define DCAN_ERRCTL2_ECCB2_M                                         0xFFFF0000U
#define DCAN_ERRCTL2_ECCB2_S                                         16U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ERRSTA1
    Offset name:         DCAN_O_ERRSTA1
    Relative address:    0x420
    Description:         This register is accessed through the ECC aggregator via an internal serial bus. To access, the appropriate procedure must be first followed in the MCAN ECC Vector Register.
    Default Value:       0x00000000

        Field:           ECCSEC
        From..to bits:   0...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Single Bit Error Corrected Status. A 2-bit saturating counter of the number of SEC errors that have occurred since last cleared.

                           0  No single-bit error detected
                           1  One single-bit error was detected and corrected
                           2  Two single-bit errors were detected and corrected
                           3  Three single-bit errors were detected and corrected

                         A write of a non-zero value to this bit field increments it by the value provided.

*/
#define DCAN_ERRSTA1_ECCSEC_W                                        2U
#define DCAN_ERRSTA1_ECCSEC_M                                        0x00000003U
#define DCAN_ERRSTA1_ECCSEC_S                                        0U
/*

        Field:           ECCDED
        From..to bits:   2...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Double Bit Error Detected Status. A 2-bit saturating counter of the number of DED errors that have occurred since last cleared.

                           0  No double-bit error detected
                           1  One double-bit error was detected
                           2  Two double-bit errors were detected
                           3  Three double-bit errors were detected

                         A write of a non-zero value to this bit field increments it by the value provided.

*/
#define DCAN_ERRSTA1_ECCDED_W                                        2U
#define DCAN_ERRSTA1_ECCDED_M                                        0x0000000CU
#define DCAN_ERRSTA1_ECCDED_S                                        2U
/*

        Field:           ECCOTHER
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     SEC While Writeback Error Status
                           0  No SEC error while writeback pending
                           1  Indicates that successive single-bit errors have occurred while a writeback is still pending

*/
#define DCAN_ERRSTA1_ECCOTHER                                        0x00000010U
#define DCAN_ERRSTA1_ECCOTHER_M                                      0x00000010U
#define DCAN_ERRSTA1_ECCOTHER_S                                      4U
/*

        Field:           PARERR
        From..to bits:   5...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Parity Error Status. A 2-bit saturating counter of the number of parity errors that have occurred since last cleared.

                           0  No parity error detected
                           1  One parity error was detected
                           2  Two parity errors were detected
                           3  Three parity errors were detected

                         A write of a non-zero value to this bit field increments it by the value provided.

*/
#define DCAN_ERRSTA1_PARERR_W                                        2U
#define DCAN_ERRSTA1_PARERR_M                                        0x00000060U
#define DCAN_ERRSTA1_PARERR_S                                        5U
/*

        Field:           CTLERR
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Control Register Error. A bit field in the control register is in an ambiguous state. This means that the redundancy registers have detected a state where not all values are the same and has defaulted to the reset state. S/W needs to re-write these registers to a known state. A write of 1 will set this interrupt flag.

*/
#define DCAN_ERRSTA1_CTLERR                                          0x00000080U
#define DCAN_ERRSTA1_CTLERR_M                                        0x00000080U
#define DCAN_ERRSTA1_CTLERR_S                                        7U
/*

        Field:           CLRECCSEC
        From..to bits:   8...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Clear ECC_SEC. A write of a non-zero value to this bit field decrements the ECC_SEC bit field by the value provided.

*/
#define DCAN_ERRSTA1_CLRECCSEC_W                                     2U
#define DCAN_ERRSTA1_CLRECCSEC_M                                     0x00000300U
#define DCAN_ERRSTA1_CLRECCSEC_S                                     8U
/*

        Field:           CLRECCDED
        From..to bits:   10...11
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Clear ECC_DED. A write of a non-zero value to this bit field decrements the ECC_DED bit field by the value provided.

*/
#define DCAN_ERRSTA1_CLRECCDED_W                                     2U
#define DCAN_ERRSTA1_CLRECCDED_M                                     0x00000C00U
#define DCAN_ERRSTA1_CLRECCDED_S                                     10U
/*

        Field:           CLRECCOT
        From..to bits:   12...12
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Writing a '1' clears the ECC_OTHER bit.

*/
#define DCAN_ERRSTA1_CLRECCOT                                        0x00001000U
#define DCAN_ERRSTA1_CLRECCOT_M                                      0x00001000U
#define DCAN_ERRSTA1_CLRECCOT_S                                      12U
/*

        Field:           CLRPARERR
        From..to bits:   13...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Clear Parity Error. A write of a non-zero value to this bit field decrements the PARITY_ERROR bit field by the value provided.

*/
#define DCAN_ERRSTA1_CLRPARERR_W                                     2U
#define DCAN_ERRSTA1_CLRPARERR_M                                     0x00006000U
#define DCAN_ERRSTA1_CLRPARERR_S                                     13U
/*

        Field:           CLRCTLERR
        From..to bits:   15...15
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Writing a '1' clears the CTRL_REG_ERROR bit

*/
#define DCAN_ERRSTA1_CLRCTLERR                                       0x00008000U
#define DCAN_ERRSTA1_CLRCTLERR_M                                     0x00008000U
#define DCAN_ERRSTA1_CLRCTLERR_S                                     15U
/*

        Field:           ECCB1
        From..to bits:   16...31
        DefaultValue:    0x0
        Access type:     read-only
        Description:     ECC Error Bit Position. Indicates the bit position in the RAM data that is in error on an SEC error. Only valid on an SEC error.
                           0  Bit 0 is in error
                           1  Bit 1 is in error
                           2  Bit 2 is in error
                           3  Bit 3 is in error
                           ...
                           31 Bit 31 is in error
                           >32 Invalid

*/
#define DCAN_ERRSTA1_ECCB1_W                                         16U
#define DCAN_ERRSTA1_ECCB1_M                                         0xFFFF0000U
#define DCAN_ERRSTA1_ECCB1_S                                         16U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ERRSTA2
    Offset name:         DCAN_O_ERRSTA2
    Relative address:    0x424
    Description:         This register is accessed through the ECC aggregator via an internal serial bus. To access, the appropriate procedure must be first followed in the MCAN ECC Vector Register.
    Default Value:       0x00000000

        Field:           ECCROW
        From..to bits:   0...31
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Indicates the row address where the single or double-bit error occurred. This value is address offset/4.

*/
#define DCAN_ERRSTA2_ECCROW_W                                        32U
#define DCAN_ERRSTA2_ECCROW_M                                        0xFFFFFFFFU
#define DCAN_ERRSTA2_ECCROW_S                                        0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ERRSTA3
    Offset name:         DCAN_O_ERRSTA3
    Relative address:    0x428
    Description:         This register is accessed through the ECC aggregator via an internal serial bus. To access, the appropriate procedure must be first followed in the MCAN ECC Vector Register.
    Default Value:       0x00000000

        Field:           WBPEND
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Delayed Write Back Pending Status
                           0  No write back pending
                           1  An ECC data correction write back is pending

*/
#define DCAN_ERRSTA3_WBPEND                                          0x00000001U
#define DCAN_ERRSTA3_WBPEND_M                                        0x00000001U
#define DCAN_ERRSTA3_WBPEND_S                                        0U
/*

        Field:           SVBUSTO
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Serial VBUS Timeout Flag. Write 1 to set.

*/
#define DCAN_ERRSTA3_SVBUSTO                                         0x00000002U
#define DCAN_ERRSTA3_SVBUSTO_M                                       0x00000002U
#define DCAN_ERRSTA3_SVBUSTO_S                                       1U
/*

        Field:           CLRSVBTO
        From..to bits:   9...9
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Write 1 to clear the Serial VBUS Timeout Flag

*/
#define DCAN_ERRSTA3_CLRSVBTO                                        0x00000200U
#define DCAN_ERRSTA3_CLRSVBTO_M                                      0x00000200U
#define DCAN_ERRSTA3_CLRSVBTO_S                                      9U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SECEOI
    Offset name:         DCAN_O_SECEOI
    Relative address:    0x43C
    Description:         MCAN Single Error Corrected End of Interrupt Register
    Default Value:       0x00000000

        Field:           EOIWR
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Write to this register indicates that software has acknowledged the pending interrupt and the next interrupt can be sent to the host.

                         Note that a write to the  MCANERR_ERR_STAT1.CLR_ECC_SEC goes through the SVBUS and has a delayed completion. To avoid an additional interrupt, read the MCANERR_ERR_STAT1 register back prior to writing to this bit field.

*/
#define DCAN_SECEOI_EOIWR                                            0x00000001U
#define DCAN_SECEOI_EOIWR_M                                          0x00000001U
#define DCAN_SECEOI_EOIWR_S                                          0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SECSTA
    Offset name:         DCAN_O_SECSTA
    Relative address:    0x440
    Description:         MCAN Single Error Corrected Interrupt Status Register
    Default Value:       0x00000000

        Field:           MGSPEND
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message RAM SEC Interrupt Pending
                           0  No SEC interrupt is pending
                           1  SEC interrupt is pending

*/
#define DCAN_SECSTA_MGSPEND                                          0x00000001U
#define DCAN_SECSTA_MGSPEND_M                                        0x00000001U
#define DCAN_SECSTA_MGSPEND_S                                        0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SECENSET
    Offset name:         DCAN_O_SECENSET
    Relative address:    0x480
    Description:         MCAN Single Error Corrected Interrupt Enable Set Register
    Default Value:       0x00000000

        Field:           MSGENSET
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message RAM SEC Interrupt Pending Enable Set. Writing a 1 to this bit enables the Message RAM SEC error interrupts. Writing a 0 has no effect. Reads return the corresponding enable bit's current value.

*/
#define DCAN_SECENSET_MSGENSET                                       0x00000001U
#define DCAN_SECENSET_MSGENSET_M                                     0x00000001U
#define DCAN_SECENSET_MSGENSET_S                                     0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SECENCLR
    Offset name:         DCAN_O_SECENCLR
    Relative address:    0x4C0
    Description:         MCAN Single Error Corrected Interrupt Enable Clear Register
    Default Value:       0x00000000

        Field:           MSGENCLR
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message RAM SEC Interrupt Pending Enable Clear. Writing a 1 to this bit disables the Message RAM SEC error interrupts. Writing a 0 has no effect. Reads return the corresponding enable bit's current value.

*/
#define DCAN_SECENCLR_MSGENCLR                                       0x00000001U
#define DCAN_SECENCLR_MSGENCLR_M                                     0x00000001U
#define DCAN_SECENCLR_MSGENCLR_S                                     0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       DEDEOI
    Offset name:         DCAN_O_DEDEOI
    Relative address:    0x53C
    Description:         MCAN Double Error Detected End of Interrupt Register
    Default Value:       0x00000000

        Field:           EOIWR
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Write to this register indicates that software has acknowledged the pending interrupt and the next interrupt can be sent to the host.

                         Note that a write to the  MCANERR_ERR_STAT1.CLR_ECC_DED goes through the SVBUS and has a delayed completion. To avoid an additional interrupt, read the MCANERR_ERR_STAT1 register back prior to writing to this bit field.

*/
#define DCAN_DEDEOI_EOIWR                                            0x00000001U
#define DCAN_DEDEOI_EOIWR_M                                          0x00000001U
#define DCAN_DEDEOI_EOIWR_S                                          0U


/*-----------------------------------REGISTER------------------------------------
    Register name:       DEDSTA
    Offset name:         DCAN_O_DEDSTA
    Relative address:    0x540
    Description:         MCAN Double Error Detected Interrupt Status Register
    Default Value:       0x00000000

        Field:           MSGPEND
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message RAM DED Interrupt Pending
                           0  No DED interrupt is pending
                           1  DED interrupt is pending

*/
#define DCAN_DEDSTA_MSGPEND                                          0x00000001U
#define DCAN_DEDSTA_MSGPEND_M                                        0x00000001U
#define DCAN_DEDSTA_MSGPEND_S                                        0U
/*

        Field:           TXREQPEND
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     TX Empty DMA Request Parity Interrupt Pending
                           0  No parity error interrupt is pending
                           1  A parity error interrupt is pending

*/
#define DCAN_DEDSTA_TXREQPEND                                        0x00000002U
#define DCAN_DEDSTA_TXREQPEND_M                                      0x00000002U
#define DCAN_DEDSTA_TXREQPEND_S                                      1U


/*-----------------------------------REGISTER------------------------------------
    Register name:       DEDENSET
    Offset name:         DCAN_O_DEDENSET
    Relative address:    0x580
    Description:         MCAN Double Error Detected Interrupt Enable Set Register
    Default Value:       0x00000000

        Field:           MSGENSET
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message RAM DED Interrupt Pending Enable Set. Writing a 1 to this bit enables the Message RAM DED error interrupts. Writing a 0 has no effect. Reads return the corresponding enable bit's current value.

*/
#define DCAN_DEDENSET_MSGENSET                                       0x00000001U
#define DCAN_DEDENSET_MSGENSET_M                                     0x00000001U
#define DCAN_DEDENSET_MSGENSET_S                                     0U
/*

        Field:           TXREQENSET
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     TX Empty DMA Request Parity Interrupt Pending Enable Set. Writing a 1 to this bit enables the TX empty DMA request parity error interrupts. Writing a 0 has no effect. Reads return the corresponding enable bit's current value.

*/
#define DCAN_DEDENSET_TXREQENSET                                     0x00000002U
#define DCAN_DEDENSET_TXREQENSET_M                                   0x00000002U
#define DCAN_DEDENSET_TXREQENSET_S                                   1U


/*-----------------------------------REGISTER------------------------------------
    Register name:       DEDENCLR
    Offset name:         DCAN_O_DEDENCLR
    Relative address:    0x5C0
    Description:         MCAN Double Error Detected Interrupt Enable Clear Register
    Default Value:       0x00000000

        Field:           MSGENCLR
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Message RAM DED Interrupt Pending Enable Clear. Writing a 1 to this bit disables the Message RAM DED error interrupts. Writing a 0 has no effect. Reads return the corresponding enable bit's current value.

*/
#define DCAN_DEDENCLR_MSGENCLR                                       0x00000001U
#define DCAN_DEDENCLR_MSGENCLR_M                                     0x00000001U
#define DCAN_DEDENCLR_MSGENCLR_S                                     0U
/*

        Field:           TXREQENCLR
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     TX Empty DMA Request Parity Interrupt Pending Enable Clear. Writing a 1 to this bit disables the TX empty DMA request parity error interrupts. Writing a 0 has no effect. Reads return the corresponding enable bit's current value.

*/
#define DCAN_DEDENCLR_TXREQENCLR                                     0x00000002U
#define DCAN_DEDENCLR_TXREQENCLR_M                                   0x00000002U
#define DCAN_DEDENCLR_TXREQENCLR_S                                   1U


/*-----------------------------------REGISTER------------------------------------
    Register name:       AGGRENSET
    Offset name:         DCAN_O_AGGRENSET
    Relative address:    0x600
    Description:         MCAN Error Aggregator Enable Set Register
    Default Value:       0x00000000

        Field:           PARITY
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Write 1 to enable parity errors. Reads return the corresponding enable bit's current value.

*/
#define DCAN_AGGRENSET_PARITY                                        0x00000001U
#define DCAN_AGGRENSET_PARITY_M                                      0x00000001U
#define DCAN_AGGRENSET_PARITY_S                                      0U
/*

        Field:           TIMEOUT
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Write 1 to enable timeout errors. Reads return the corresponding enable bit's current value.

*/
#define DCAN_AGGRENSET_TIMEOUT                                       0x00000002U
#define DCAN_AGGRENSET_TIMEOUT_M                                     0x00000002U
#define DCAN_AGGRENSET_TIMEOUT_S                                     1U


/*-----------------------------------REGISTER------------------------------------
    Register name:       AGGRENCLR
    Offset name:         DCAN_O_AGGRENCLR
    Relative address:    0x604
    Description:         MCAN Error Aggregator Enable Clear Register
    Default Value:       0x00000000

        Field:           PARITY
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Write 1 to disable parity errors. Reads return the corresponding enable bit's current value.

*/
#define DCAN_AGGRENCLR_PARITY                                        0x00000001U
#define DCAN_AGGRENCLR_PARITY_M                                      0x00000001U
#define DCAN_AGGRENCLR_PARITY_S                                      0U
/*

        Field:           TIMEOUT
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Write 1 to disable timeout errors. Reads return the corresponding enable bit's current value.

*/
#define DCAN_AGGRENCLR_TIMEOUT                                       0x00000002U
#define DCAN_AGGRENCLR_TIMEOUT_M                                     0x00000002U
#define DCAN_AGGRENCLR_TIMEOUT_S                                     1U


/*-----------------------------------REGISTER------------------------------------
    Register name:       AGGRSTASET
    Offset name:         DCAN_O_AGGRSTASET
    Relative address:    0x608
    Description:         MCAN Error Aggregator Status Set Register
    Default Value:       0x00000000

        Field:           PARITY
        From..to bits:   0...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Aggregator Parity Error Status

                         2-bit saturating counter of the number of parity errors that have occurred since last cleared.
                           0  No parity errors have occurred
                           1  One parity error has occurred
                           2  Two parity errors have occurred
                           3  Three parity errors have occurred

                         A write of a non-zero value to this bit field increments it by the value provided.

*/
#define DCAN_AGGRSTASET_PARITY_W                                     2U
#define DCAN_AGGRSTASET_PARITY_M                                     0x00000003U
#define DCAN_AGGRSTASET_PARITY_S                                     0U
/*

        Field:           TIMEOUT
        From..to bits:   2...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Aggregator Serial VBUS Timeout Error Status

                         2-bit saturating counter of the number of SVBUS timeout errors that have occurred since last cleared.
                           0  No timeout errors have occurred
                           1  One timeout error has occurred
                           2  Two timeout errors have occurred
                           3  Three timeout errors have occurred

                         A write of a non-zero value to this bit field increments it by the value provided.

*/
#define DCAN_AGGRSTASET_TIMEOUT_W                                    2U
#define DCAN_AGGRSTASET_TIMEOUT_M                                    0x0000000CU
#define DCAN_AGGRSTASET_TIMEOUT_S                                    2U


/*-----------------------------------REGISTER------------------------------------
    Register name:       AGGRSTACLR
    Offset name:         DCAN_O_AGGRSTACLR
    Relative address:    0x60C
    Description:         MCAN Error Aggregator Status Clear Register
    Default Value:       0x00000000

        Field:           PARITY
        From..to bits:   0...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Aggregator Parity Error Status

                         2-bit saturating counter of the number of parity errors that have occurred since last cleared.
                           0  No parity errors have occurred
                           1  One parity error has occurred
                           2  Two parity errors have occurred
                           3  Three parity errors have occurred

                         A write of a non-zero value to this bit field decrements it by the value provided.

*/
#define DCAN_AGGRSTACLR_PARITY_W                                     2U
#define DCAN_AGGRSTACLR_PARITY_M                                     0x00000003U
#define DCAN_AGGRSTACLR_PARITY_S                                     0U
/*

        Field:           TIMEOUT
        From..to bits:   2...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Aggregator Serial VBUS Timeout Error Status

                         2-bit saturating counter of the number of SVBUS timeout errors that have occurred since last cleared.
                           0  No timeout errors have occurred
                           1  One timeout error has occurred
                           2  Two timeout errors have occurred
                           3  Three timeout errors have occurred

                         A write of a non-zero value to this bit field decrements it by the value provided.

*/
#define DCAN_AGGRSTACLR_TIMEOUT_W                                    2U
#define DCAN_AGGRSTACLR_TIMEOUT_M                                    0x0000000CU
#define DCAN_AGGRSTACLR_TIMEOUT_S                                    2U


/*-----------------------------------REGISTER------------------------------------
    Register name:       DESC
    Offset name:         DCAN_O_DESC
    Relative address:    0x800
    Description:         This register identifies the peripheral and its exact version.
    Default Value:       0x00000000

        Field:           MINREV
        From..to bits:   0...3
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Minor rev of the IP

        ENUMs:
            MINIMUM:                     Smallest value
            MAXIMUM:                     Highest possible value
*/
#define DCAN_DESC_MINREV_W                                           4U
#define DCAN_DESC_MINREV_M                                           0x0000000FU
#define DCAN_DESC_MINREV_S                                           0U
#define DCAN_DESC_MINREV_MINIMUM                                     0x00000000U
#define DCAN_DESC_MINREV_MAXIMUM                                     0x0000000FU
/*

        Field:           MAJREV
        From..to bits:   4...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Major rev of the IP

        ENUMs:
            MINIMUM:                     Smallest value
            MAXIMUM:                     Highest possible value
*/
#define DCAN_DESC_MAJREV_W                                           4U
#define DCAN_DESC_MAJREV_M                                           0x000000F0U
#define DCAN_DESC_MAJREV_S                                           4U
#define DCAN_DESC_MAJREV_MINIMUM                                     0x00000000U
#define DCAN_DESC_MAJREV_MAXIMUM                                     0x000000F0U
/*

        Field:           FEATUREVER
        From..to bits:   12...15
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Feature Set for the module *instance*

        ENUMs:
            VERSION_0:                   MCAN module with CAN-FD mode enabled
                         <<Internal Note: This is an in-IP paper spin variant. How does this map to the SYS_MCAN_ENABLE_FD choice value?>>
            VERSION_1:                   MCAN module with CAN-FD mode disabled
                         <<Internal Note: This is an in-IP paper spin variant. How does this map to the SYS_MCAN_ENABLE_FD choice value?>>
*/
#define DCAN_DESC_FEATUREVER_W                                       4U
#define DCAN_DESC_FEATUREVER_M                                       0x0000F000U
#define DCAN_DESC_FEATUREVER_S                                       12U
#define DCAN_DESC_FEATUREVER_VERSION_0                               0x00000000U
#define DCAN_DESC_FEATUREVER_VERSION_1                               0x00001000U
/*

        Field:           MODULEID
        From..to bits:   16...31
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Module identification contains a unique peripheral identification number. The assignments are maintained in a central database for all of the platform modules to ensure uniqueness.

        ENUMs:
            MINIMUM:                     Smallest value
            MAXIMUM:                     Highest possible value
*/
#define DCAN_DESC_MODULEID_W                                         16U
#define DCAN_DESC_MODULEID_M                                         0xFFFF0000U
#define DCAN_DESC_MODULEID_S                                         16U
#define DCAN_DESC_MODULEID_MINIMUM                                   0x00000000U
#define DCAN_DESC_MODULEID_MAXIMUM                                   0xFFFF0000U


/*-----------------------------------REGISTER------------------------------------
    Register name:       IMASK0
    Offset name:         DCAN_O_IMASK0
    Relative address:    0x844
    Description:         Interrupt Mask. If a bit is set, then corresponding interrupt is un-masked. Un-masking the interrupt causes the raw interrupt to be visible in IIDX, as well as MIS.
    Default Value:       0x00000000

        Field:           INTL0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask channel0 Event

        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrupt Mask
*/
#define DCAN_IMASK0_INTL0                                            0x00000001U
#define DCAN_IMASK0_INTL0_M                                          0x00000001U
#define DCAN_IMASK0_INTL0_S                                          0U
#define DCAN_IMASK0_INTL0_CLR                                        0x00000000U
#define DCAN_IMASK0_INTL0_SET                                        0x00000001U
/*

        Field:           INTL1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.

        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK0_INTL1                                            0x00000002U
#define DCAN_IMASK0_INTL1_M                                          0x00000002U
#define DCAN_IMASK0_INTL1_S                                          1U
#define DCAN_IMASK0_INTL1_CLR                                        0x00000000U
#define DCAN_IMASK0_INTL1_SET                                        0x00000002U
/*

        Field:           SEC
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.

        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK0_SEC                                              0x00000004U
#define DCAN_IMASK0_SEC_M                                            0x00000004U
#define DCAN_IMASK0_SEC_S                                            2U
#define DCAN_IMASK0_SEC_CLR                                          0x00000000U
#define DCAN_IMASK0_SEC_SET                                          0x00000004U
/*

        Field:           DED
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.

        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK0_DED                                              0x00000008U
#define DCAN_IMASK0_DED_M                                            0x00000008U
#define DCAN_IMASK0_DED_S                                            3U
#define DCAN_IMASK0_DED_CLR                                          0x00000000U
#define DCAN_IMASK0_DED_SET                                          0x00000008U
/*

        Field:           TSORWAKE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.

        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK0_TSORWAKE                                         0x00000010U
#define DCAN_IMASK0_TSORWAKE_M                                       0x00000010U
#define DCAN_IMASK0_TSORWAKE_S                                       4U
#define DCAN_IMASK0_TSORWAKE_CLR                                     0x00000000U
#define DCAN_IMASK0_TSORWAKE_SET                                     0x00000010U
/*

        Field:           FE2
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.

        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK0_FE2                                              0x00000020U
#define DCAN_IMASK0_FE2_M                                            0x00000020U
#define DCAN_IMASK0_FE2_S                                            5U
#define DCAN_IMASK0_FE2_CLR                                          0x00000000U
#define DCAN_IMASK0_FE2_SET                                          0x00000020U
/*

        Field:           DMADONE0
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.

        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK0_DMADONE0                                         0x00000040U
#define DCAN_IMASK0_DMADONE0_M                                       0x00000040U
#define DCAN_IMASK0_DMADONE0_S                                       6U
#define DCAN_IMASK0_DMADONE0_CLR                                     0x00000000U
#define DCAN_IMASK0_DMADONE0_SET                                     0x00000040U
/*

        Field:           DMADONE1
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.

        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK0_DMADONE1                                         0x00000080U
#define DCAN_IMASK0_DMADONE1_M                                       0x00000080U
#define DCAN_IMASK0_DMADONE1_S                                       7U
#define DCAN_IMASK0_DMADONE1_CLR                                     0x00000000U
#define DCAN_IMASK0_DMADONE1_SET                                     0x00000080U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RIS0
    Offset name:         DCAN_O_RIS0
    Relative address:    0x848
    Description:         Raw interrupt status. Reflects all pending interrupts, regardless of masking. The RIS0 register allows the user to implement a poll scheme. A flag set in this register can be cleared by writing 1 to the ICLR register bit even if the corresponding IMASK bit is not enabled.
    Default Value:       0x00000000

        Field:           INTL0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS0_INTL0                                              0x00000001U
#define DCAN_RIS0_INTL0_M                                            0x00000001U
#define DCAN_RIS0_INTL0_S                                            0U
#define DCAN_RIS0_INTL0_CLR                                          0x00000000U
#define DCAN_RIS0_INTL0_SET                                          0x00000001U
/*

        Field:           INTL1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT1.
                         This bit is set to 1 when an event is received on EVENT1 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when the captured time value is read from the CH1CAPT register.

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS0_INTL1                                              0x00000002U
#define DCAN_RIS0_INTL1_M                                            0x00000002U
#define DCAN_RIS0_INTL1_S                                            1U
#define DCAN_RIS0_INTL1_CLR                                          0x00000000U
#define DCAN_RIS0_INTL1_SET                                          0x00000002U
/*

        Field:           SEC
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS0_SEC                                                0x00000004U
#define DCAN_RIS0_SEC_M                                              0x00000004U
#define DCAN_RIS0_SEC_S                                              2U
#define DCAN_RIS0_SEC_CLR                                            0x00000000U
#define DCAN_RIS0_SEC_SET                                            0x00000004U
/*

        Field:           DED
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS0_DED                                                0x00000008U
#define DCAN_RIS0_DED_M                                              0x00000008U
#define DCAN_RIS0_DED_S                                              3U
#define DCAN_RIS0_DED_CLR                                            0x00000000U
#define DCAN_RIS0_DED_SET                                            0x00000008U
/*

        Field:           TSORWAKE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS0_TSORWAKE                                           0x00000010U
#define DCAN_RIS0_TSORWAKE_M                                         0x00000010U
#define DCAN_RIS0_TSORWAKE_S                                         4U
#define DCAN_RIS0_TSORWAKE_CLR                                       0x00000000U
#define DCAN_RIS0_TSORWAKE_SET                                       0x00000010U
/*

        Field:           FE2
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS0_FE2                                                0x00000020U
#define DCAN_RIS0_FE2_M                                              0x00000020U
#define DCAN_RIS0_FE2_S                                              5U
#define DCAN_RIS0_FE2_CLR                                            0x00000000U
#define DCAN_RIS0_FE2_SET                                            0x00000020U
/*

        Field:           DMADONE0
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS0_DMADONE0                                           0x00000040U
#define DCAN_RIS0_DMADONE0_M                                         0x00000040U
#define DCAN_RIS0_DMADONE0_S                                         6U
#define DCAN_RIS0_DMADONE0_CLR                                       0x00000000U
#define DCAN_RIS0_DMADONE0_SET                                       0x00000040U
/*

        Field:           DMADONE1
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS0_DMADONE1                                           0x00000080U
#define DCAN_RIS0_DMADONE1_M                                         0x00000080U
#define DCAN_RIS0_DMADONE1_S                                         7U
#define DCAN_RIS0_DMADONE1_CLR                                       0x00000000U
#define DCAN_RIS0_DMADONE1_SET                                       0x00000080U


/*-----------------------------------REGISTER------------------------------------
    Register name:       MIS0
    Offset name:         DCAN_O_MIS0
    Relative address:    0x84C
    Description:         Masked interrupt status. This is an AND of the IMASK and RIS registers.
    Default Value:       0x00000000

        Field:           INTL0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS0_INTL0                                              0x00000001U
#define DCAN_MIS0_INTL0_M                                            0x00000001U
#define DCAN_MIS0_INTL0_S                                            0U
#define DCAN_MIS0_INTL0_CLR                                          0x00000000U
#define DCAN_MIS0_INTL0_SET                                          0x00000001U
/*

        Field:           INTL1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT1

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS0_INTL1                                              0x00000002U
#define DCAN_MIS0_INTL1_M                                            0x00000002U
#define DCAN_MIS0_INTL1_S                                            1U
#define DCAN_MIS0_INTL1_CLR                                          0x00000000U
#define DCAN_MIS0_INTL1_SET                                          0x00000002U
/*

        Field:           SEC
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS0_SEC                                                0x00000004U
#define DCAN_MIS0_SEC_M                                              0x00000004U
#define DCAN_MIS0_SEC_S                                              2U
#define DCAN_MIS0_SEC_CLR                                            0x00000000U
#define DCAN_MIS0_SEC_SET                                            0x00000004U
/*

        Field:           DED
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS0_DED                                                0x00000008U
#define DCAN_MIS0_DED_M                                              0x00000008U
#define DCAN_MIS0_DED_S                                              3U
#define DCAN_MIS0_DED_CLR                                            0x00000000U
#define DCAN_MIS0_DED_SET                                            0x00000008U
/*

        Field:           TSORWAKE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS0_TSORWAKE                                           0x00000010U
#define DCAN_MIS0_TSORWAKE_M                                         0x00000010U
#define DCAN_MIS0_TSORWAKE_S                                         4U
#define DCAN_MIS0_TSORWAKE_CLR                                       0x00000000U
#define DCAN_MIS0_TSORWAKE_SET                                       0x00000010U
/*

        Field:           FE2
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS0_FE2                                                0x00000020U
#define DCAN_MIS0_FE2_M                                              0x00000020U
#define DCAN_MIS0_FE2_S                                              5U
#define DCAN_MIS0_FE2_CLR                                            0x00000000U
#define DCAN_MIS0_FE2_SET                                            0x00000020U
/*

        Field:           DMADONE0
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS0_DMADONE0                                           0x00000040U
#define DCAN_MIS0_DMADONE0_M                                         0x00000040U
#define DCAN_MIS0_DMADONE0_S                                         6U
#define DCAN_MIS0_DMADONE0_CLR                                       0x00000000U
#define DCAN_MIS0_DMADONE0_SET                                       0x00000040U
/*

        Field:           DMADONE1
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS0_DMADONE1                                           0x00000080U
#define DCAN_MIS0_DMADONE1_M                                         0x00000080U
#define DCAN_MIS0_DMADONE1_S                                         7U
#define DCAN_MIS0_DMADONE1_CLR                                       0x00000000U
#define DCAN_MIS0_DMADONE1_SET                                       0x00000080U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ISET0
    Offset name:         DCAN_O_ISET0
    Relative address:    0x850
    Description:         Interrupt set. Allows interrupts to be set by software (useful in diagnostics and safety checks). Writing a 1 to a bit in ISET0 will set the event and therefore the related RIS bit also gets set. If the interrupt is enabled through the mask, then the corresponding MIS bit is also set.
    Default Value:       0x00000000

        Field:           INTL0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET0_INTL0                                             0x00000001U
#define DCAN_ISET0_INTL0_M                                           0x00000001U
#define DCAN_ISET0_INTL0_S                                           0U
#define DCAN_ISET0_INTL0_NO_EFFECT                                   0x00000000U
#define DCAN_ISET0_INTL0_SET                                         0x00000001U
/*

        Field:           INTL1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT1 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET0_INTL1                                             0x00000002U
#define DCAN_ISET0_INTL1_M                                           0x00000002U
#define DCAN_ISET0_INTL1_S                                           1U
#define DCAN_ISET0_INTL1_NO_EFFECT                                   0x00000000U
#define DCAN_ISET0_INTL1_SET                                         0x00000002U
/*

        Field:           SEC
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET0_SEC                                               0x00000004U
#define DCAN_ISET0_SEC_M                                             0x00000004U
#define DCAN_ISET0_SEC_S                                             2U
#define DCAN_ISET0_SEC_NO_EFFECT                                     0x00000000U
#define DCAN_ISET0_SEC_SET                                           0x00000004U
/*

        Field:           DED
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET0_DED                                               0x00000008U
#define DCAN_ISET0_DED_M                                             0x00000008U
#define DCAN_ISET0_DED_S                                             3U
#define DCAN_ISET0_DED_NO_EFFECT                                     0x00000000U
#define DCAN_ISET0_DED_SET                                           0x00000008U
/*

        Field:           TSORWAKE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET0_TSORWAKE                                          0x00000010U
#define DCAN_ISET0_TSORWAKE_M                                        0x00000010U
#define DCAN_ISET0_TSORWAKE_S                                        4U
#define DCAN_ISET0_TSORWAKE_NO_EFFECT                                0x00000000U
#define DCAN_ISET0_TSORWAKE_SET                                      0x00000010U
/*

        Field:           FE2
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET0_FE2                                               0x00000020U
#define DCAN_ISET0_FE2_M                                             0x00000020U
#define DCAN_ISET0_FE2_S                                             5U
#define DCAN_ISET0_FE2_NO_EFFECT                                     0x00000000U
#define DCAN_ISET0_FE2_SET                                           0x00000020U
/*

        Field:           DMADONE0
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET0_DMADONE0                                          0x00000040U
#define DCAN_ISET0_DMADONE0_M                                        0x00000040U
#define DCAN_ISET0_DMADONE0_S                                        6U
#define DCAN_ISET0_DMADONE0_NO_EFFECT                                0x00000000U
#define DCAN_ISET0_DMADONE0_SET                                      0x00000040U
/*

        Field:           DMADONE1
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET0_DMADONE1                                          0x00000080U
#define DCAN_ISET0_DMADONE1_M                                        0x00000080U
#define DCAN_ISET0_DMADONE1_S                                        7U
#define DCAN_ISET0_DMADONE1_NO_EFFECT                                0x00000000U
#define DCAN_ISET0_DMADONE1_SET                                      0x00000080U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ICLR0
    Offset name:         DCAN_O_ICLR0
    Relative address:    0x854
    Description:         Interrupt clear. Write a 1 to clear corresponding Interrupt.
    Default Value:       0x00000000

        Field:           INTL0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR0_INTL0                                             0x00000001U
#define DCAN_ICLR0_INTL0_M                                           0x00000001U
#define DCAN_ICLR0_INTL0_S                                           0U
#define DCAN_ICLR0_INTL0_NO_EFFECT                                   0x00000000U
#define DCAN_ICLR0_INTL0_CLR                                         0x00000001U
/*

        Field:           INTL1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT1 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR0_INTL1                                             0x00000002U
#define DCAN_ICLR0_INTL1_M                                           0x00000002U
#define DCAN_ICLR0_INTL1_S                                           1U
#define DCAN_ICLR0_INTL1_NO_EFFECT                                   0x00000000U
#define DCAN_ICLR0_INTL1_CLR                                         0x00000002U
/*

        Field:           SEC
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR0_SEC                                               0x00000004U
#define DCAN_ICLR0_SEC_M                                             0x00000004U
#define DCAN_ICLR0_SEC_S                                             2U
#define DCAN_ICLR0_SEC_NO_EFFECT                                     0x00000000U
#define DCAN_ICLR0_SEC_CLR                                           0x00000004U
/*

        Field:           DED
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR0_DED                                               0x00000008U
#define DCAN_ICLR0_DED_M                                             0x00000008U
#define DCAN_ICLR0_DED_S                                             3U
#define DCAN_ICLR0_DED_NO_EFFECT                                     0x00000000U
#define DCAN_ICLR0_DED_CLR                                           0x00000008U
/*

        Field:           TSORWAKE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR0_TSORWAKE                                          0x00000010U
#define DCAN_ICLR0_TSORWAKE_M                                        0x00000010U
#define DCAN_ICLR0_TSORWAKE_S                                        4U
#define DCAN_ICLR0_TSORWAKE_NO_EFFECT                                0x00000000U
#define DCAN_ICLR0_TSORWAKE_CLR                                      0x00000010U
/*

        Field:           FE2
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR0_FE2                                               0x00000020U
#define DCAN_ICLR0_FE2_M                                             0x00000020U
#define DCAN_ICLR0_FE2_S                                             5U
#define DCAN_ICLR0_FE2_NO_EFFECT                                     0x00000000U
#define DCAN_ICLR0_FE2_CLR                                           0x00000020U
/*

        Field:           DMADONE0
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR0_DMADONE0                                          0x00000040U
#define DCAN_ICLR0_DMADONE0_M                                        0x00000040U
#define DCAN_ICLR0_DMADONE0_S                                        6U
#define DCAN_ICLR0_DMADONE0_NO_EFFECT                                0x00000000U
#define DCAN_ICLR0_DMADONE0_CLR                                      0x00000040U
/*

        Field:           DMADONE1
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR0_DMADONE1                                          0x00000080U
#define DCAN_ICLR0_DMADONE1_M                                        0x00000080U
#define DCAN_ICLR0_DMADONE1_S                                        7U
#define DCAN_ICLR0_DMADONE1_NO_EFFECT                                0x00000000U
#define DCAN_ICLR0_DMADONE1_CLR                                      0x00000080U


/*-----------------------------------REGISTER------------------------------------
    Register name:       IMASK1
    Offset name:         DCAN_O_IMASK1
    Relative address:    0x868
    Description:         Interrupt Mask. If a bit is set, then corresponding interrupt is un-masked. Un-masking the interrupt causes the raw interrupt to be visible in IIDX, as well as MIS.
    Default Value:       0x00000000

        Field:           INTL0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask channel0 Event

        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK1_INTL0                                            0x00000001U
#define DCAN_IMASK1_INTL0_M                                          0x00000001U
#define DCAN_IMASK1_INTL0_S                                          0U
#define DCAN_IMASK1_INTL0_CLR                                        0x00000000U
#define DCAN_IMASK1_INTL0_SET                                        0x00000001U
/*

        Field:           INTL1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.


        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK1_INTL1                                            0x00000002U
#define DCAN_IMASK1_INTL1_M                                          0x00000002U
#define DCAN_IMASK1_INTL1_S                                          1U
#define DCAN_IMASK1_INTL1_CLR                                        0x00000000U
#define DCAN_IMASK1_INTL1_SET                                        0x00000002U
/*

        Field:           SEC
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.


        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK1_SEC                                              0x00000004U
#define DCAN_IMASK1_SEC_M                                            0x00000004U
#define DCAN_IMASK1_SEC_S                                            2U
#define DCAN_IMASK1_SEC_CLR                                          0x00000000U
#define DCAN_IMASK1_SEC_SET                                          0x00000004U
/*

        Field:           DED
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.


        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK1_DED                                              0x00000008U
#define DCAN_IMASK1_DED_M                                            0x00000008U
#define DCAN_IMASK1_DED_S                                            3U
#define DCAN_IMASK1_DED_CLR                                          0x00000000U
#define DCAN_IMASK1_DED_SET                                          0x00000008U
/*

        Field:           TSORWAKE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.


        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK1_TSORWAKE                                         0x00000010U
#define DCAN_IMASK1_TSORWAKE_M                                       0x00000010U
#define DCAN_IMASK1_TSORWAKE_S                                       4U
#define DCAN_IMASK1_TSORWAKE_CLR                                     0x00000000U
#define DCAN_IMASK1_TSORWAKE_SET                                     0x00000010U
/*

        Field:           FE2
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.


        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK1_FE2                                              0x00000020U
#define DCAN_IMASK1_FE2_M                                            0x00000020U
#define DCAN_IMASK1_FE2_S                                            5U
#define DCAN_IMASK1_FE2_CLR                                          0x00000000U
#define DCAN_IMASK1_FE2_SET                                          0x00000020U
/*

        Field:           DMADONE0
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.


        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK1_DMADONE0                                         0x00000040U
#define DCAN_IMASK1_DMADONE0_M                                       0x00000040U
#define DCAN_IMASK1_DMADONE0_S                                       6U
#define DCAN_IMASK1_DMADONE0_CLR                                     0x00000000U
#define DCAN_IMASK1_DMADONE0_SET                                     0x00000040U
/*

        Field:           DMADONE1
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Mask Channel1 Event.


        ENUMs:
            CLR:                         Clear Interrupt Mask
            SET:                         Set Interrrupt Mask
*/
#define DCAN_IMASK1_DMADONE1                                         0x00000080U
#define DCAN_IMASK1_DMADONE1_M                                       0x00000080U
#define DCAN_IMASK1_DMADONE1_S                                       7U
#define DCAN_IMASK1_DMADONE1_CLR                                     0x00000000U
#define DCAN_IMASK1_DMADONE1_SET                                     0x00000080U


/*-----------------------------------REGISTER------------------------------------
    Register name:       RIS1
    Offset name:         DCAN_O_RIS1
    Relative address:    0x86C
    Description:         Raw interrupt status. Reflects all pending interrupts, regardless of masking. The RIS0 register allows the user to implement a poll scheme. A flag set in this register can be cleared by writing 1 to the ICLR register bit even if the corresponding IMASK bit is not enabled.
    Default Value:       0x00000000

        Field:           INTL0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS1_INTL0                                              0x00000001U
#define DCAN_RIS1_INTL0_M                                            0x00000001U
#define DCAN_RIS1_INTL0_S                                            0U
#define DCAN_RIS1_INTL0_CLR                                          0x00000000U
#define DCAN_RIS1_INTL0_SET                                          0x00000001U
/*

        Field:           INTL1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT1.
                         This bit is set to 1 when an event is received on EVENT1 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when the captured time value is read from the CH1CAPT register.

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS1_INTL1                                              0x00000002U
#define DCAN_RIS1_INTL1_M                                            0x00000002U
#define DCAN_RIS1_INTL1_S                                            1U
#define DCAN_RIS1_INTL1_CLR                                          0x00000000U
#define DCAN_RIS1_INTL1_SET                                          0x00000002U
/*

        Field:           SEC
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS1_SEC                                                0x00000004U
#define DCAN_RIS1_SEC_M                                              0x00000004U
#define DCAN_RIS1_SEC_S                                              2U
#define DCAN_RIS1_SEC_CLR                                            0x00000000U
#define DCAN_RIS1_SEC_SET                                            0x00000004U
/*

        Field:           DED
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS1_DED                                                0x00000008U
#define DCAN_RIS1_DED_M                                              0x00000008U
#define DCAN_RIS1_DED_S                                              3U
#define DCAN_RIS1_DED_CLR                                            0x00000000U
#define DCAN_RIS1_DED_SET                                            0x00000008U
/*

        Field:           TSORWAKE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS1_TSORWAKE                                           0x00000010U
#define DCAN_RIS1_TSORWAKE_M                                         0x00000010U
#define DCAN_RIS1_TSORWAKE_S                                         4U
#define DCAN_RIS1_TSORWAKE_CLR                                       0x00000000U
#define DCAN_RIS1_TSORWAKE_SET                                       0x00000010U
/*

        Field:           FE2
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS1_FE2                                                0x00000020U
#define DCAN_RIS1_FE2_M                                              0x00000020U
#define DCAN_RIS1_FE2_S                                              5U
#define DCAN_RIS1_FE2_CLR                                            0x00000000U
#define DCAN_RIS1_FE2_SET                                            0x00000020U
/*

        Field:           DMADONE0
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS1_DMADONE0                                           0x00000040U
#define DCAN_RIS1_DMADONE0_M                                         0x00000040U
#define DCAN_RIS1_DMADONE0_S                                         6U
#define DCAN_RIS1_DMADONE0_CLR                                       0x00000000U
#define DCAN_RIS1_DMADONE0_SET                                       0x00000040U
/*

        Field:           DMADONE1
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Raw interrupt status for EVENT0.
                         This bit is set to 1 when an event is received on EVENT0 channel. when the corresponding bit in ICLR is set to 1, this bit will be cleared.
                         This bit is also cleared when a new compare value is written in CH0CMP register

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_RIS1_DMADONE1                                           0x00000080U
#define DCAN_RIS1_DMADONE1_M                                         0x00000080U
#define DCAN_RIS1_DMADONE1_S                                         7U
#define DCAN_RIS1_DMADONE1_CLR                                       0x00000000U
#define DCAN_RIS1_DMADONE1_SET                                       0x00000080U


/*-----------------------------------REGISTER------------------------------------
    Register name:       MIS1
    Offset name:         DCAN_O_MIS1
    Relative address:    0x870
    Description:         Masked interrupt status. This is an AND of the IMASK and RIS registers.
    Default Value:       0x00000000

        Field:           INTL0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS1_INTL0                                              0x00000001U
#define DCAN_MIS1_INTL0_M                                            0x00000001U
#define DCAN_MIS1_INTL0_S                                            0U
#define DCAN_MIS1_INTL0_CLR                                          0x00000000U
#define DCAN_MIS1_INTL0_SET                                          0x00000001U
/*

        Field:           INTL1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT1

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS1_INTL1                                              0x00000002U
#define DCAN_MIS1_INTL1_M                                            0x00000002U
#define DCAN_MIS1_INTL1_S                                            1U
#define DCAN_MIS1_INTL1_CLR                                          0x00000000U
#define DCAN_MIS1_INTL1_SET                                          0x00000002U
/*

        Field:           SEC
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS1_SEC                                                0x00000004U
#define DCAN_MIS1_SEC_M                                              0x00000004U
#define DCAN_MIS1_SEC_S                                              2U
#define DCAN_MIS1_SEC_CLR                                            0x00000000U
#define DCAN_MIS1_SEC_SET                                            0x00000004U
/*

        Field:           DED
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS1_DED                                                0x00000008U
#define DCAN_MIS1_DED_M                                              0x00000008U
#define DCAN_MIS1_DED_S                                              3U
#define DCAN_MIS1_DED_CLR                                            0x00000000U
#define DCAN_MIS1_DED_SET                                            0x00000008U
/*

        Field:           TSORWAKE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS1_TSORWAKE                                           0x00000010U
#define DCAN_MIS1_TSORWAKE_M                                         0x00000010U
#define DCAN_MIS1_TSORWAKE_S                                         4U
#define DCAN_MIS1_TSORWAKE_CLR                                       0x00000000U
#define DCAN_MIS1_TSORWAKE_SET                                       0x00000010U
/*

        Field:           FE2
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS1_FE2                                                0x00000020U
#define DCAN_MIS1_FE2_M                                              0x00000020U
#define DCAN_MIS1_FE2_S                                              5U
#define DCAN_MIS1_FE2_CLR                                            0x00000000U
#define DCAN_MIS1_FE2_SET                                            0x00000020U
/*

        Field:           DMADONE0
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS1_DMADONE0                                           0x00000040U
#define DCAN_MIS1_DMADONE0_M                                         0x00000040U
#define DCAN_MIS1_DMADONE0_S                                         6U
#define DCAN_MIS1_DMADONE0_CLR                                       0x00000000U
#define DCAN_MIS1_DMADONE0_SET                                       0x00000040U
/*

        Field:           DMADONE1
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Mask interrupt status for EVENT0

        ENUMs:
            CLR:                         Interrupt did not occur
            SET:                         Interrupt occured
*/
#define DCAN_MIS1_DMADONE1                                           0x00000080U
#define DCAN_MIS1_DMADONE1_M                                         0x00000080U
#define DCAN_MIS1_DMADONE1_S                                         7U
#define DCAN_MIS1_DMADONE1_CLR                                       0x00000000U
#define DCAN_MIS1_DMADONE1_SET                                       0x00000080U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ISET1
    Offset name:         DCAN_O_ISET1
    Relative address:    0x874
    Description:         Interrupt set. Allows interrupts to be set by software (useful in diagnostics and safety checks). Writing a 1 to a bit in ISET0 will set the event and therefore the related RIS bit also gets set. If the interrupt is enabled through the mask, then the corresponding MIS bit is also set.
    Default Value:       0x00000000

        Field:           INTL0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET1_INTL0                                             0x00000001U
#define DCAN_ISET1_INTL0_M                                           0x00000001U
#define DCAN_ISET1_INTL0_S                                           0U
#define DCAN_ISET1_INTL0_NO_EFFECT                                   0x00000000U
#define DCAN_ISET1_INTL0_SET                                         0x00000001U
/*

        Field:           INTL1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT1 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET1_INTL1                                             0x00000002U
#define DCAN_ISET1_INTL1_M                                           0x00000002U
#define DCAN_ISET1_INTL1_S                                           1U
#define DCAN_ISET1_INTL1_NO_EFFECT                                   0x00000000U
#define DCAN_ISET1_INTL1_SET                                         0x00000002U
/*

        Field:           SEC
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET1_SEC                                               0x00000004U
#define DCAN_ISET1_SEC_M                                             0x00000004U
#define DCAN_ISET1_SEC_S                                             2U
#define DCAN_ISET1_SEC_NO_EFFECT                                     0x00000000U
#define DCAN_ISET1_SEC_SET                                           0x00000004U
/*

        Field:           DED
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET1_DED                                               0x00000008U
#define DCAN_ISET1_DED_M                                             0x00000008U
#define DCAN_ISET1_DED_S                                             3U
#define DCAN_ISET1_DED_NO_EFFECT                                     0x00000000U
#define DCAN_ISET1_DED_SET                                           0x00000008U
/*

        Field:           TSORWAKE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET1_TSORWAKE                                          0x00000010U
#define DCAN_ISET1_TSORWAKE_M                                        0x00000010U
#define DCAN_ISET1_TSORWAKE_S                                        4U
#define DCAN_ISET1_TSORWAKE_NO_EFFECT                                0x00000000U
#define DCAN_ISET1_TSORWAKE_SET                                      0x00000010U
/*

        Field:           FE2
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET1_FE2                                               0x00000020U
#define DCAN_ISET1_FE2_M                                             0x00000020U
#define DCAN_ISET1_FE2_S                                             5U
#define DCAN_ISET1_FE2_NO_EFFECT                                     0x00000000U
#define DCAN_ISET1_FE2_SET                                           0x00000020U
/*

        Field:           DMADONE0
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET1_DMADONE0                                          0x00000040U
#define DCAN_ISET1_DMADONE0_M                                        0x00000040U
#define DCAN_ISET1_DMADONE0_S                                        6U
#define DCAN_ISET1_DMADONE0_NO_EFFECT                                0x00000000U
#define DCAN_ISET1_DMADONE0_SET                                      0x00000040U
/*

        Field:           DMADONE1
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Sets EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            SET:                         Sets interrupt
*/
#define DCAN_ISET1_DMADONE1                                          0x00000080U
#define DCAN_ISET1_DMADONE1_M                                        0x00000080U
#define DCAN_ISET1_DMADONE1_S                                        7U
#define DCAN_ISET1_DMADONE1_NO_EFFECT                                0x00000000U
#define DCAN_ISET1_DMADONE1_SET                                      0x00000080U


/*-----------------------------------REGISTER------------------------------------
    Register name:       ICLR1
    Offset name:         DCAN_O_ICLR1
    Relative address:    0x878
    Description:         Interrupt clear. Write a 1 to clear corresponding Interrupt.
    Default Value:       0x00000000

        Field:           INTL0
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR1_INTL0                                             0x00000001U
#define DCAN_ICLR1_INTL0_M                                           0x00000001U
#define DCAN_ICLR1_INTL0_S                                           0U
#define DCAN_ICLR1_INTL0_NO_EFFECT                                   0x00000000U
#define DCAN_ICLR1_INTL0_CLR                                         0x00000001U
/*

        Field:           INTL1
        From..to bits:   1...1
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT1 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR1_INTL1                                             0x00000002U
#define DCAN_ICLR1_INTL1_M                                           0x00000002U
#define DCAN_ICLR1_INTL1_S                                           1U
#define DCAN_ICLR1_INTL1_NO_EFFECT                                   0x00000000U
#define DCAN_ICLR1_INTL1_CLR                                         0x00000002U
/*

        Field:           SEC
        From..to bits:   2...2
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR1_SEC                                               0x00000004U
#define DCAN_ICLR1_SEC_M                                             0x00000004U
#define DCAN_ICLR1_SEC_S                                             2U
#define DCAN_ICLR1_SEC_NO_EFFECT                                     0x00000000U
#define DCAN_ICLR1_SEC_CLR                                           0x00000004U
/*

        Field:           DED
        From..to bits:   3...3
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR1_DED                                               0x00000008U
#define DCAN_ICLR1_DED_M                                             0x00000008U
#define DCAN_ICLR1_DED_S                                             3U
#define DCAN_ICLR1_DED_NO_EFFECT                                     0x00000000U
#define DCAN_ICLR1_DED_CLR                                           0x00000008U
/*

        Field:           TSORWAKE
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR1_TSORWAKE                                          0x00000010U
#define DCAN_ICLR1_TSORWAKE_M                                        0x00000010U
#define DCAN_ICLR1_TSORWAKE_S                                        4U
#define DCAN_ICLR1_TSORWAKE_NO_EFFECT                                0x00000000U
#define DCAN_ICLR1_TSORWAKE_CLR                                      0x00000010U
/*

        Field:           FE2
        From..to bits:   5...5
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR1_FE2                                               0x00000020U
#define DCAN_ICLR1_FE2_M                                             0x00000020U
#define DCAN_ICLR1_FE2_S                                             5U
#define DCAN_ICLR1_FE2_NO_EFFECT                                     0x00000000U
#define DCAN_ICLR1_FE2_CLR                                           0x00000020U
/*

        Field:           DMADONE0
        From..to bits:   6...6
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR1_DMADONE0                                          0x00000040U
#define DCAN_ICLR1_DMADONE0_M                                        0x00000040U
#define DCAN_ICLR1_DMADONE0_S                                        6U
#define DCAN_ICLR1_DMADONE0_NO_EFFECT                                0x00000000U
#define DCAN_ICLR1_DMADONE0_CLR                                      0x00000040U
/*

        Field:           DMADONE1
        From..to bits:   7...7
        DefaultValue:    0x0
        Access type:     write-only
        Description:     Clears EVENT0 in RIS

        ENUMs:
            NO_EFFECT:                   Writing 0 has no effect
            CLR:                         Clears the Event
*/
#define DCAN_ICLR1_DMADONE1                                          0x00000080U
#define DCAN_ICLR1_DMADONE1_M                                        0x00000080U
#define DCAN_ICLR1_DMADONE1_S                                        7U
#define DCAN_ICLR1_DMADONE1_NO_EFFECT                                0x00000000U
#define DCAN_ICLR1_DMADONE1_CLR                                      0x00000080U


/*-----------------------------------REGISTER------------------------------------
    Register name:       CLKDIV
    Offset name:         DCAN_O_CLKDIV
    Relative address:    0x904
    Description:         Needs to go to the Management aperture once available
    Default Value:       0x00000000

        Field:           RATIO
        From..to bits:   0...1
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Clock divide ratio specification. Enables configuring clock divide settings for the MCAN functional clock input to the MCAN-SS.

        ENUMs:
            DIV_BY_1_:                   Divides input clock by 1
            DIV_BY_2_:                   Divides input clock by 2
            DIV_BY_4_:                   Divides input clock by 4
*/
#define DCAN_CLKDIV_RATIO_W                                          2U
#define DCAN_CLKDIV_RATIO_M                                          0x00000003U
#define DCAN_CLKDIV_RATIO_S                                          0U
#define DCAN_CLKDIV_RATIO_DIV_BY_1_                                  0x00000000U
#define DCAN_CLKDIV_RATIO_DIV_BY_2_                                  0x00000001U
#define DCAN_CLKDIV_RATIO_DIV_BY_4_                                  0x00000002U


/*-----------------------------------REGISTER------------------------------------
    Register name:       CLKCTL
    Offset name:         DCAN_O_CLKCTL
    Relative address:    0x908
    Description:         MCANSS clock stop control MMR.
                        <Internal note> Bus clock for the Dragon wrapper MMRs (including this MMR) is not gated by this register.
    Default Value:       0x00000000

        Field:           STOPREQ
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     This bit is used to enable/disable MCAN clock (both host clock and functional clock) gating request.

                         Note: This bit can be reset by HW by Clock-Stop Wake-up via CAN RX Activity. See spec for more details.

        ENUMs:
            DISABLE:                     Disable MCAN-SS clock stop request
            ENABLE:                      Enable MCAN-SS clock stop request
*/
#define DCAN_CLKCTL_STOPREQ                                          0x00000001U
#define DCAN_CLKCTL_STOPREQ_M                                        0x00000001U
#define DCAN_CLKCTL_STOPREQ_S                                        0U
#define DCAN_CLKCTL_STOPREQ_DISABLE                                  0x00000000U
#define DCAN_CLKCTL_STOPREQ_ENABLE                                   0x00000001U
/*

        Field:           WUINTEN
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     This bit controls enabling or disabling the MCAN IP clock stop wakeup interrupt (when MCANSS_CTRL.WAKEUPREQEN wakeup request is enabled to wakeup MCAN IP upon CAN RXD activity)

        ENUMs:
            ENABLE:                      Enable MCAN IP clock stop wakeup interrupt
            DISABLE:                     Disable MCAN IP clock stop wakeup interrupt
*/
#define DCAN_CLKCTL_WUINTEN                                          0x00000010U
#define DCAN_CLKCTL_WUINTEN_M                                        0x00000010U
#define DCAN_CLKCTL_WUINTEN_S                                        4U
#define DCAN_CLKCTL_WUINTEN_ENABLE                                   0x00000010U
#define DCAN_CLKCTL_WUINTEN_DISABLE                                  0x00000000U
/*

        Field:           WUGLTFLTEN
        From..to bits:   8...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Setting this bit enables the glitch filter on MCAN RXD input, which wakes up the MCAN controller to exit clock gating.

        ENUMs:
            DISABLE:                     Disable glitch filter enable on RXD input when MCAN is in clock stop mode (waiting for event on RXD input for clock stop wakeup).
            ENABLE:                      Enable glitch filter enable on RXD input when MCAN is in clock stop mode (waiting for event on RXD input for clock stop wakeup).
*/
#define DCAN_CLKCTL_WUGLTFLTEN                                       0x00000100U
#define DCAN_CLKCTL_WUGLTFLTEN_M                                     0x00000100U
#define DCAN_CLKCTL_WUGLTFLTEN_S                                     8U
#define DCAN_CLKCTL_WUGLTFLTEN_DISABLE                               0x00000000U
#define DCAN_CLKCTL_WUGLTFLTEN_ENABLE                                0x00000100U


/*-----------------------------------REGISTER------------------------------------
    Register name:       CLKSTA
    Offset name:         DCAN_O_CLKSTA
    Relative address:    0x90C
    Description:         MCANSS clock stop status register to indicate status of clock stop mechanism
    Default Value:       0x00000000

        Field:           STPACKSTA
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-only
        Description:     Clock stop acknowledge status from MCAN IP

        ENUMs:
            RESET:                       No clock stop acknowledged.
            SET:                         MCAN-SS may be clock gated by stopping both the CAN host and functional clocks.
*/
#define DCAN_CLKSTA_STPACKSTA                                        0x00000001U
#define DCAN_CLKSTA_STPACKSTA_M                                      0x00000001U
#define DCAN_CLKSTA_STPACKSTA_S                                      0U
#define DCAN_CLKSTA_STPACKSTA_RESET                                  0x00000000U
#define DCAN_CLKSTA_STPACKSTA_SET                                    0x00000001U
/*

        Field:           STPREQHWOV
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-only
        Description:     MCANSS clock stop HW override status bit.

                         This bit indicates when the MCANSS_CLKCTL.STOPREQ bit has been cleared by HW when a clock-stop wake-up event via CAN RX activity is triggered.

        ENUMs:
            RESET:                       MCANSS_CLKCTL.STOPREQ bit has not been cleared by HW.
            SET:                         MCANSS_CLKCTL.STOPREQ bit has been cleared by HW.
*/
#define DCAN_CLKSTA_STPREQHWOV                                       0x00000010U
#define DCAN_CLKSTA_STPREQHWOV_M                                     0x00000010U
#define DCAN_CLKSTA_STPREQHWOV_S                                     4U
#define DCAN_CLKSTA_STPREQHWOV_RESET                                 0x00000000U
#define DCAN_CLKSTA_STPREQHWOV_SET                                   0x00000010U


/*-----------------------------------REGISTER------------------------------------
    Register name:       DMA0CTL
    Offset name:         DCAN_O_DMA0CTL
    Relative address:    0x924
    Description:         MCANSS fixed DMA0 control and configuration register
    Default Value:       0x00000000

        Field:           TRIGEN
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     DMA_TRIG_EN is used to enable/disable MCAN RX, TX triggers to MCANSS fixed DMA channel.
                         <Note to design> check if this bit is needed depending on if similar functionality is enabled in the EXT_DMA aperture.

        ENUMs:
            DISABLE:                     MCANSS fixed DMA channel trigger is disabled.
            ENABLE:                      MCANSS fixed DMA channel trigger is enabled.
*/
#define DCAN_DMA0CTL_TRIGEN                                          0x00000001U
#define DCAN_DMA0CTL_TRIGEN_M                                        0x00000001U
#define DCAN_DMA0CTL_TRIGEN_S                                        0U
#define DCAN_DMA0CTL_TRIGEN_DISABLE                                  0x00000000U
#define DCAN_DMA0CTL_TRIGEN_ENABLE                                   0x00000001U
/*

        Field:           TRIGSEL
        From..to bits:   2...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     DMA trigger select bits used to select between MCAN TX one-to-one mapping, MCAN TX multi-to-one round robin mapping and MCAN Rx one-to-one mapping options

        ENUMs:
            TX_OTO_TRIG:                 MCAN TX Buffer one-to-one Tx BRP (buffer request pending) trigger to DMA channel select
            RX_OTO_TRIG:                 MCAN RX Buffer one-to-one Tx BRP (buffer request pending) trigger to DMA channel select
            TX_MTO_TRIG:                 MCAN TX Buffer multi-to-one round robin, Tx BRP (buffer request pending) triggers to DMA channel select
            RX_TTO_TRIG:                 Rx buffer two-to-one DMA trigger
*/
#define DCAN_DMA0CTL_TRIGSEL_W                                       2U
#define DCAN_DMA0CTL_TRIGSEL_M                                       0x0000000CU
#define DCAN_DMA0CTL_TRIGSEL_S                                       2U
#define DCAN_DMA0CTL_TRIGSEL_TX_OTO_TRIG                             0x00000000U
#define DCAN_DMA0CTL_TRIGSEL_RX_OTO_TRIG                             0x00000008U
#define DCAN_DMA0CTL_TRIGSEL_TX_MTO_TRIG                             0x00000004U
#define DCAN_DMA0CTL_TRIGSEL_RX_TTO_TRIG                             0x0000000CU
/*

        Field:           BRPOTOSEL
        From..to bits:   4...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     TX_BRP_OTO_SEL is used to select the MCAN TX buffer request pending (BRP) signal mapped to trigger fixed MCANSS DMA channel trigger

        ENUMs:
            TX_BRP_0:                    TX Buffer Request Pending 0
            TX_BRP_2:                    TX Buffer Request Pending 2
            TX_BRP_3:                    TX Buffer Request Pending 3
            TX_BRP_4:                    TX Buffer Request Pending 4
            TX_BRP_5:                    TX Buffer Request Pending 5
            TX_BRP_6:                    TX Buffer Request Pending 6
            TX_BRP_7:                    TX Buffer Request Pending 7
            TX_BRP_8:                    TX Buffer Request Pending 8
            TX_BRP_9:                    TX Buffer Request Pending 9
            TX_BRP_10:                   TX Buffer Request Pending 10
            TX_BRP_11:                   TX Buffer Request Pending 11
            TX_BRP_12:                   TX Buffer Request Pending 12
            TX_BRP_13:                   TX Buffer Request Pending 13
            TX_BRP_14:                   TX Buffer Request Pending 14
            TX_BRP_15:                   TX Buffer Request Pending 15
            TX_BRP_16:                   TX Buffer Request Pending 16
            TX_BRP_1:                    TX Buffer Request Pending 1
            TX_BRP_17:                   TX Buffer Request Pending 17
            TX_BRP_18:                   TX Buffer Request Pending 18
            TX_BRP_19:                   TX Buffer Request Pending 19
            TX_BRP_20:                   TX Buffer Request Pending 20
            TX_BRP_21:                   TX Buffer Request Pending 21
            TX_BRP_22:                   TX Buffer Request Pending 22
            TX_BRP_23:                   TX Buffer Request Pending 23
            TX_BRP_24:                   TX Buffer Request Pending 24
            TX_BRP_25:                   TX Buffer Request Pending 25
            TX_BRP_26:                   TX Buffer Request Pending 26
            TX_BRP_27:                   TX Buffer Request Pending 27
            TX_BRP_28:                   TX Buffer Request Pending 28
            TX_BRP_29:                   TX Buffer Request Pending 29
            TX_BRP_30:                   TX Buffer Request Pending 30
            TX_BRP_31:                   TX Buffer Request Pending 31
*/
#define DCAN_DMA0CTL_BRPOTOSEL_W                                     5U
#define DCAN_DMA0CTL_BRPOTOSEL_M                                     0x000001F0U
#define DCAN_DMA0CTL_BRPOTOSEL_S                                     4U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_0                              0x00000000U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_2                              0x00000020U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_3                              0x00000030U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_4                              0x00000040U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_5                              0x00000050U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_6                              0x00000060U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_7                              0x00000070U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_8                              0x00000080U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_9                              0x00000090U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_10                             0x000000A0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_11                             0x000000B0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_12                             0x000000C0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_13                             0x000000D0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_14                             0x000000E0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_15                             0x000000F0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_16                             0x00000100U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_1                              0x00000010U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_17                             0x00000110U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_18                             0x00000120U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_19                             0x00000130U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_20                             0x00000140U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_21                             0x00000150U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_22                             0x00000160U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_23                             0x00000170U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_24                             0x00000180U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_25                             0x00000190U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_26                             0x000001A0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_27                             0x000001B0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_28                             0x000001C0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_29                             0x000001D0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_30                             0x000001E0U
#define DCAN_DMA0CTL_BRPOTOSEL_TX_BRP_31                             0x000001F0U
/*

        Field:           BRPMTOOFST
        From..to bits:   10...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     TX_BRP_MTO_OFFST selects the Tx buffer offset number for the multi-to-one round robin DMA trigger mode.

        ENUMs:
            TX_BRP_0:                    TX Buffer Request Pending 0
            TX_BRP_1:                    TX Buffer Request Pending 1
            TX_BRP_2:                    TX Buffer Request Pending 2
            TX_BRP_3:                    TX Buffer Request Pending 3
            TX_BRP_4:                    TX Buffer Request Pending 4
            TX_BRP_5:                    TX Buffer Request Pending 5
            TX_BRP_6:                    TX Buffer Request Pending 6
            TX_BRP_7:                    TX Buffer Request Pending 7
            TX_BRP_8:                    TX Buffer Request Pending 8
            TX_BRP_9:                    TX Buffer Request Pending 9
            TX_BRP_10:                   TX Buffer Request Pending 10
            TX_BRP_11:                   TX Buffer Request Pending 11
            TX_BRP_12:                   TX Buffer Request Pending 12
            TX_BRP_13:                   TX Buffer Request Pending 13
            TX_BRP_14:                   TX Buffer Request Pending 14
            TX_BRP_15:                   TX Buffer Request Pending 15
            TX_BRP_16:                   TX Buffer Request Pending 16
            TX_BRP_17:                   TX Buffer Request Pending 17
            TX_BRP_18:                   TX Buffer Request Pending 18
            TX_BRP_19:                   TX Buffer Request Pending 19
            TX_BRP_20:                   TX Buffer Request Pending 20
            TX_BRP_21:                   TX Buffer Request Pending 21
            TX_BRP_22:                   TX Buffer Request Pending 22
            TX_BRP_23:                   TX Buffer Request Pending 23
            TX_BRP_24:                   TX Buffer Request Pending 24
            TX_BRP_25:                   TX Buffer Request Pending 25
            TX_BRP_26:                   TX Buffer Request Pending 26
            TX_BRP_27:                   TX Buffer Request Pending 27
            TX_BRP_28:                   TX Buffer Request Pending 28
            TX_BRP_29:                   TX Buffer Request Pending 29
            TX_BRP_30:                   TX Buffer Request Pending 30
            TX_BRP_31:                   TX Buffer Request Pending 31
*/
#define DCAN_DMA0CTL_BRPMTOOFST_W                                    5U
#define DCAN_DMA0CTL_BRPMTOOFST_M                                    0x00007C00U
#define DCAN_DMA0CTL_BRPMTOOFST_S                                    10U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_0                             0x00000000U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_1                             0x00000400U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_2                             0x00000800U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_3                             0x00000C00U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_4                             0x00001000U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_5                             0x00001400U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_6                             0x00001800U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_7                             0x00001C00U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_8                             0x00002000U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_9                             0x00002400U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_10                            0x00002800U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_11                            0x00002C00U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_12                            0x00003000U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_13                            0x00003400U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_14                            0x00003800U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_15                            0x00003C00U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_16                            0x00004000U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_17                            0x00004400U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_18                            0x00004800U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_19                            0x00004C00U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_20                            0x00005000U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_21                            0x00005400U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_22                            0x00005800U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_23                            0x00005C00U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_24                            0x00006000U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_25                            0x00006400U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_26                            0x00006800U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_27                            0x00006C00U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_28                            0x00007000U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_29                            0x00007400U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_30                            0x00007800U
#define DCAN_DMA0CTL_BRPMTOOFST_TX_BRP_31                            0x00007C00U
/*

        Field:           BRPMTONUM
        From..to bits:   16...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Number of TX buffer request pending (BRP) signals for multi-to-one DMA trigger mapping sequence, starting from the buffer offset number selected by TX_BRP_MTO_OFFST bits

        ENUMs:
            MIN:                         Min number for TX BRP multi-to-one DMA trigger mapping sequence is 2
            MAX:                         Max number for TX BRP multi-to-one DMA trigger mapping sequence is 32
*/
#define DCAN_DMA0CTL_BRPMTONUM_W                                     6U
#define DCAN_DMA0CTL_BRPMTONUM_M                                     0x003F0000U
#define DCAN_DMA0CTL_BRPMTONUM_S                                     16U
#define DCAN_DMA0CTL_BRPMTONUM_MIN                                   0x00020000U
#define DCAN_DMA0CTL_BRPMTONUM_MAX                                   0x00200000U
/*

        Field:           FEOTOSEL
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     RX_FE_OTO_SEL is used to select the MCAN RX buffer filter event signal mapped to trigger fixed MCANSS DMA channel trigger

        ENUMs:
            FE_0:                        Filter Event 0
            FE_1:                        Filter Event 1
*/
#define DCAN_DMA0CTL_FEOTOSEL                                        0x01000000U
#define DCAN_DMA0CTL_FEOTOSEL_M                                      0x01000000U
#define DCAN_DMA0CTL_FEOTOSEL_S                                      24U
#define DCAN_DMA0CTL_FEOTOSEL_FE_0                                   0x00000000U
#define DCAN_DMA0CTL_FEOTOSEL_FE_1                                   0x01000000U
/*

        Field:           BUFTTOOFST
        From..to bits:   27...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Indicates the Rx-buffer (index x) to be mapped to FE_0 (FE001) and automatically maps and Rx buffer (index x+1) to FE_1 (FE010)
                         Valid range: Rxbuffer (0) to Rxbuffer (30)

        ENUMs:
            MIN:                         Minimum index value: 0
            MAX:                         Maximum index value: 30

                         Note: RX_FE_TTO_SEL Rx buffer index selection for FE01 cannot be 31, as Rxbuffer (index +1) maps to 32, which requires updating NDAT1 and NDAT2 value, which is not supported.
*/
#define DCAN_DMA0CTL_BUFTTOOFST_W                                    5U
#define DCAN_DMA0CTL_BUFTTOOFST_M                                    0xF8000000U
#define DCAN_DMA0CTL_BUFTTOOFST_S                                    27U
#define DCAN_DMA0CTL_BUFTTOOFST_MIN                                  0x00000000U
#define DCAN_DMA0CTL_BUFTTOOFST_MAX                                  0xF0000000U


/*-----------------------------------REGISTER------------------------------------
    Register name:       DMA1CTL
    Offset name:         DCAN_O_DMA1CTL
    Relative address:    0x92C
    Description:         MCANSS fixed DMA1 control and configuration register
    Default Value:       0x00000000

        Field:           TRIGEN
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     DMA_TRIG_EN is used to enable/disable MCAN RX, TX triggers to MCANSS fixed DMA channel.
                         <Note to design> check if this bit is needed depending on if similar functionality is enabled in the EXT_DMA aperture.

        ENUMs:
            DISABLE:                     MCANSS fixed DMA channel trigger is disabled.
            ENABLE:                      MCANSS fixed DMA channel trigger is enabled.
*/
#define DCAN_DMA1CTL_TRIGEN                                          0x00000001U
#define DCAN_DMA1CTL_TRIGEN_M                                        0x00000001U
#define DCAN_DMA1CTL_TRIGEN_S                                        0U
#define DCAN_DMA1CTL_TRIGEN_DISABLE                                  0x00000000U
#define DCAN_DMA1CTL_TRIGEN_ENABLE                                   0x00000001U
/*

        Field:           TRIGSEL
        From..to bits:   2...3
        DefaultValue:    0x0
        Access type:     read-write
        Description:     DMA trigger select bits used to select between MCAN TX one-to-one mapping, MCAN TX multi-to-one round robin mapping and MCAN Rx one-to-one mapping options

        ENUMs:
            TX_OTO_TRIG:                 MCAN TX Buffer one-to-one Tx BRP (buffer request pending) trigger to DMA channel select
            RX_OTO_TRIG:                 MCAN RX Buffer one-to-one Tx BRP (buffer request pending) trigger to DMA channel select
            TX_MTO_TRIG:                 MCAN TX Buffer multi-to-one round robin Tx BRP (buffer request pending) triggers to DMA channel select
            RX_TTO_TRIG:                 Rx buffer two-to-one DMA trigger
*/
#define DCAN_DMA1CTL_TRIGSEL_W                                       2U
#define DCAN_DMA1CTL_TRIGSEL_M                                       0x0000000CU
#define DCAN_DMA1CTL_TRIGSEL_S                                       2U
#define DCAN_DMA1CTL_TRIGSEL_TX_OTO_TRIG                             0x00000000U
#define DCAN_DMA1CTL_TRIGSEL_RX_OTO_TRIG                             0x00000008U
#define DCAN_DMA1CTL_TRIGSEL_TX_MTO_TRIG                             0x00000004U
#define DCAN_DMA1CTL_TRIGSEL_RX_TTO_TRIG                             0x0000000CU
/*

        Field:           BRPOTOSEL
        From..to bits:   4...8
        DefaultValue:    0x0
        Access type:     read-write
        Description:     TX_BRP_OTO_SEL is used to select the MCAN TX buffer request pending (BRP) signal mapped to trigger fixed MCANSS DMA channel trigger

        ENUMs:
            TX_BRP_0:                    TX Buffer Request Pending 0
            TX_BRP_2:                    TX Buffer Request Pending 2
            TX_BRP_3:                    TX Buffer Request Pending 3
            TX_BRP_4:                    TX Buffer Request Pending 4
            TX_BRP_5:                    TX Buffer Request Pending 5
            TX_BRP_6:                    TX Buffer Request Pending 6
            TX_BRP_7:                    TX Buffer Request Pending 7
            TX_BRP_8:                    TX Buffer Request Pending 8
            TX_BRP_9:                    TX Buffer Request Pending 9
            TX_BRP_10:                   TX Buffer Request Pending 10
            TX_BRP_11:                   TX Buffer Request Pending 11
            TX_BRP_12:                   TX Buffer Request Pending 12
            TX_BRP_13:                   TX Buffer Request Pending 13
            TX_BRP_14:                   TX Buffer Request Pending 14
            TX_BRP_15:                   TX Buffer Request Pending 15
            TX_BRP_16:                   TX Buffer Request Pending 16
            TX_BRP_1:                    TX Buffer Request Pending 1
            TX_BRP_17:                   TX Buffer Request Pending 17
            TX_BRP_18:                   TX Buffer Request Pending 18
            TX_BRP_19:                   TX Buffer Request Pending 19
            TX_BRP_20:                   TX Buffer Request Pending 20
            TX_BRP_21:                   TX Buffer Request Pending 21
            TX_BRP_22:                   TX Buffer Request Pending 22
            TX_BRP_23:                   TX Buffer Request Pending 23
            TX_BRP_24:                   TX Buffer Request Pending 24
            TX_BRP_25:                   TX Buffer Request Pending 25
            TX_BRP_26:                   TX Buffer Request Pending 26
            TX_BRP_27:                   TX Buffer Request Pending 27
            TX_BRP_28:                   TX Buffer Request Pending 28
            TX_BRP_29:                   TX Buffer Request Pending 29
            TX_BRP_30:                   TX Buffer Request Pending 30
            TX_BRP_31:                   TX Buffer Request Pending 31
*/
#define DCAN_DMA1CTL_BRPOTOSEL_W                                     5U
#define DCAN_DMA1CTL_BRPOTOSEL_M                                     0x000001F0U
#define DCAN_DMA1CTL_BRPOTOSEL_S                                     4U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_0                              0x00000000U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_2                              0x00000020U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_3                              0x00000030U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_4                              0x00000040U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_5                              0x00000050U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_6                              0x00000060U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_7                              0x00000070U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_8                              0x00000080U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_9                              0x00000090U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_10                             0x000000A0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_11                             0x000000B0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_12                             0x000000C0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_13                             0x000000D0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_14                             0x000000E0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_15                             0x000000F0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_16                             0x00000100U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_1                              0x00000010U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_17                             0x00000110U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_18                             0x00000120U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_19                             0x00000130U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_20                             0x00000140U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_21                             0x00000150U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_22                             0x00000160U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_23                             0x00000170U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_24                             0x00000180U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_25                             0x00000190U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_26                             0x000001A0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_27                             0x000001B0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_28                             0x000001C0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_29                             0x000001D0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_30                             0x000001E0U
#define DCAN_DMA1CTL_BRPOTOSEL_TX_BRP_31                             0x000001F0U
/*

        Field:           BRPMTOOFST
        From..to bits:   10...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     TX_BRP_MTO_OFFST selects the Tx buffer offset number for the multi-to-one round robin DMA trigger mode.


        ENUMs:
            TX_BRP_0:                    TX Buffer Request Pending 0
            TX_BRP_1:                    TX Buffer Request Pending 1
            TX_BRP_2:                    TX Buffer Request Pending 2
            TX_BRP_3:                    TX Buffer Request Pending 3
            TX_BRP_4:                    TX Buffer Request Pending 4
            TX_BRP_5:                    TX Buffer Request Pending 5
            TX_BRP_6:                    TX Buffer Request Pending 6
            TX_BRP_7:                    TX Buffer Request Pending 7
            TX_BRP_8:                    TX Buffer Request Pending 8
            TX_BRP_9:                    TX Buffer Request Pending 9
            TX_BRP_10:                   TX Buffer Request Pending 10
            TX_BRP_11:                   TX Buffer Request Pending 11
            TX_BRP_12:                   TX Buffer Request Pending 12
            TX_BRP_13:                   TX Buffer Request Pending 13
            TX_BRP_14:                   TX Buffer Request Pending 14
            TX_BRP_15:                   TX Buffer Request Pending 15
            TX_BRP_16:                   TX Buffer Request Pending 16
            TX_BRP_17:                   TX Buffer Request Pending 17
            TX_BRP_18:                   TX Buffer Request Pending 18
            TX_BRP_19:                   TX Buffer Request Pending 19
            TX_BRP_20:                   TX Buffer Request Pending 20
            TX_BRP_21:                   TX Buffer Request Pending 21
            TX_BRP_22:                   TX Buffer Request Pending 22
            TX_BRP_23:                   TX Buffer Request Pending 23
            TX_BRP_24:                   TX Buffer Request Pending 24
            TX_BRP_25:                   TX Buffer Request Pending 25
            TX_BRP_26:                   TX Buffer Request Pending 26
            TX_BRP_27:                   TX Buffer Request Pending 27
            TX_BRP_28:                   TX Buffer Request Pending 28
            TX_BRP_29:                   TX Buffer Request Pending 29
            TX_BRP_30:                   TX Buffer Request Pending 30
            TX_BRP_31:                   TX Buffer Request Pending 31
*/
#define DCAN_DMA1CTL_BRPMTOOFST_W                                    5U
#define DCAN_DMA1CTL_BRPMTOOFST_M                                    0x00007C00U
#define DCAN_DMA1CTL_BRPMTOOFST_S                                    10U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_0                             0x00000000U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_1                             0x00000400U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_2                             0x00000800U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_3                             0x00000C00U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_4                             0x00001000U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_5                             0x00001400U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_6                             0x00001800U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_7                             0x00001C00U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_8                             0x00002000U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_9                             0x00002400U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_10                            0x00002800U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_11                            0x00002C00U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_12                            0x00003000U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_13                            0x00003400U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_14                            0x00003800U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_15                            0x00003C00U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_16                            0x00004000U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_17                            0x00004400U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_18                            0x00004800U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_19                            0x00004C00U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_20                            0x00005000U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_21                            0x00005400U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_22                            0x00005800U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_23                            0x00005C00U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_24                            0x00006000U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_25                            0x00006400U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_26                            0x00006800U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_27                            0x00006C00U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_28                            0x00007000U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_29                            0x00007400U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_30                            0x00007800U
#define DCAN_DMA1CTL_BRPMTOOFST_TX_BRP_31                            0x00007C00U
/*

        Field:           BRPMTONUM
        From..to bits:   16...21
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Number of TX buffer request pending (BRP) signals for multi-to-one DMA trigger mapping sequence, starting from the buffer offset number selected by TX_BRP_MTO_OFFST bits

        ENUMs:
            MIN:                         Min number for TX BRP multi-to-one DMA trigger mapping sequence is 2
            MAX:                         Max number for TX BRP multi-to-one DMA trigger mapping sequence is 32
*/
#define DCAN_DMA1CTL_BRPMTONUM_W                                     6U
#define DCAN_DMA1CTL_BRPMTONUM_M                                     0x003F0000U
#define DCAN_DMA1CTL_BRPMTONUM_S                                     16U
#define DCAN_DMA1CTL_BRPMTONUM_MIN                                   0x00020000U
#define DCAN_DMA1CTL_BRPMTONUM_MAX                                   0x00200000U
/*

        Field:           FEOTOSEL
        From..to bits:   24...24
        DefaultValue:    0x0
        Access type:     read-write
        Description:     RX_FE_OTO_SEL is used to select the MCAN RX buffer filter event signal mapped to trigger fixed MCANSS DMA channel trigger

        ENUMs:
            FE_0:                        Filter Event 0
            FE_1:                        Filter Event 1
*/
#define DCAN_DMA1CTL_FEOTOSEL                                        0x01000000U
#define DCAN_DMA1CTL_FEOTOSEL_M                                      0x01000000U
#define DCAN_DMA1CTL_FEOTOSEL_S                                      24U
#define DCAN_DMA1CTL_FEOTOSEL_FE_0                                   0x00000000U
#define DCAN_DMA1CTL_FEOTOSEL_FE_1                                   0x01000000U
/*

        Field:           BUFTTOOFST
        From..to bits:   27...31
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Indicates the Rx-buffer (index x) to be mapped to FE_0 (FE001) and automatically maps and Rx buffer (index x+1) to FE_1 (FE010)
                         Valid range: Rxbuffer (0) to Rxbuffer (30)

        ENUMs:
            MIN:                         Minimum index value: 0
            MAX:                         Maximum index value: 30

                         Note: RX_FE_TTO_SEL Rx buffer index selection for FE01 cannot be 31, as Rxbuffer (index +1) maps to 32, which requires updating NDAT1 and NDAT2 value, which is not supported.
*/
#define DCAN_DMA1CTL_BUFTTOOFST_W                                    5U
#define DCAN_DMA1CTL_BUFTTOOFST_M                                    0xF8000000U
#define DCAN_DMA1CTL_BUFTTOOFST_S                                    27U
#define DCAN_DMA1CTL_BUFTTOOFST_MIN                                  0x00000000U
#define DCAN_DMA1CTL_BUFTTOOFST_MAX                                  0xF0000000U


/*-----------------------------------REGISTER------------------------------------
    Register name:       TTOFE0
    Offset name:         DCAN_O_TTOFE0
    Relative address:    0x938
    Description:         Rx buffer (index x) base address.
                        <Internal: Absolute address within MCAN IP: 0x7938>

                        Applicable to Rx buffer DMA two-to-one mode mapped to FE001 trigger:
                        >> LS bits 0:1 in this MMR are reserved and read as '0' as the MCAN SRAM is 4 byte data addressable.
                        >> Index x is selected using MCANSS_DMAn_CTL.RX_FE_TTO_SEL bits.
    Default Value:       0x00000000

        Field:           BASEADDR
        From..to bits:   2...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     FE0 Rx Buf x Base address (14:2).
                         Address should be computed based on the 14-bit RBSA (Rx buffer start address) + offset (depending on Rx buffer element index value and data length code (DLC) for all the buffer elements before the Rx buffer element (x))

        ENUMs:
            MIN:                         Min address offset within MCANSS SRAM: 0x0
            MAX:                         Max address offset within MCANSS SRAM: 0x1fff
*/
#define DCAN_TTOFE0_BASEADDR_W                                       13U
#define DCAN_TTOFE0_BASEADDR_M                                       0x00007FFCU
#define DCAN_TTOFE0_BASEADDR_S                                       2U
#define DCAN_TTOFE0_BASEADDR_MIN                                     0x00000000U
#define DCAN_TTOFE0_BASEADDR_MAX                                     0x00007FFCU


/*-----------------------------------REGISTER------------------------------------
    Register name:       TTOFE1
    Offset name:         DCAN_O_TTOFE1
    Relative address:    0x948
    Description:         Rx buffer (index x+1) base address
                        <Internal: Absolute address within MCAN IP: 0x7948>

                        Applicable to Rx buffer DMA two-to-one mode mapped to FE010 trigger:
                        >> LS bits 0:1 in this MMR are reserved and read as '0' as the MCAN SRAM is 4 byte data addressable.
                        >> Index x is selected using MCANSS_DMAn_CTL.RX_FE_TTO_SEL bits.
    Default Value:       0x00000000

        Field:           BASEADDR
        From..to bits:   2...14
        DefaultValue:    0x0
        Access type:     read-write
        Description:     FE010 Rx Buf x Base adddress (14:2).
                         Address should be compited based on the 14-bit RBSA (Rx buffer start address) + offset (depending on Rx buffer element index value and data length code (DLC) for all the buffer elements before the Rx buffer element (x+1))

        ENUMs:
            MIN:                         Min address offset within MCANSS SRAM: 0x0
            MAX:                         Max address offset within MCANSS SRAM: 0x1fff
*/
#define DCAN_TTOFE1_BASEADDR_W                                       13U
#define DCAN_TTOFE1_BASEADDR_M                                       0x00007FFCU
#define DCAN_TTOFE1_BASEADDR_S                                       2U
#define DCAN_TTOFE1_BASEADDR_MIN                                     0x00000000U
#define DCAN_TTOFE1_BASEADDR_MAX                                     0x00007FFCU


/*-----------------------------------REGISTER------------------------------------
    Register name:       TTONDAT1
    Offset name:         DCAN_O_TTONDAT1
    Relative address:    0x950
    Description:         Rx Buffer two-to-one DMA mode, hardware NDAT1 value register.
                        The address of this register is programmed as the DMA source address register for moving NDAT1 value during DMA operation.

                        This register is automatically updated on the fly depending on FE001/FE010 (Rxbuf(x)/Rxbuf(x+1)) ongoing transfer.
    Default Value:       0x00000000

        Field:           NDAT1VAL
        From..to bits:   0...31
        DefaultValue:    0x0
        Access type:     read-only
        Description:     NDAT1 value to be programmed onto MCAN.NDAT1 MMR.
                         Automatically updated by HW.

        ENUMs:
            MIN:                         Min value = 0x0 (not bits set)
            MAX:                         max value = (bit 31 set) = 0x80000000
*/
#define DCAN_TTONDAT1_NDAT1VAL_W                                     32U
#define DCAN_TTONDAT1_NDAT1VAL_M                                     0xFFFFFFFFU
#define DCAN_TTONDAT1_NDAT1VAL_S                                     0U
#define DCAN_TTONDAT1_NDAT1VAL_MIN                                   0x00000000U
#define DCAN_TTONDAT1_NDAT1VAL_MAX                                   0x80000000U


/*-----------------------------------REGISTER------------------------------------
    Register name:       SRAM
    Offset name:         DCAN_O_SRAM
    Relative address:    0x1000
    Description:
    Default Value:       0x00000000

*/

/*-----------------------------------REGISTER------------------------------------
    Register name:       CLKCFG
    Offset name:         DCAN_O_CLKCFG
    Relative address:    0x2000
    Description:         Clock Configuration MMR for **DCAN**
    Default Value:       0x00000000

        Field:           CLKEN
        From..to bits:   0...0
        DefaultValue:    0x0
        Access type:     read-write
        Description:     0: **DCAN** clock disabled
                              1: **DCAN** clock enabled

*/
#define DCAN_CLKCFG_CLKEN                                            0x00000001U
#define DCAN_CLKCFG_CLKEN_M                                          0x00000001U
#define DCAN_CLKCFG_CLKEN_S                                          0U
/*

        Field:           RAMEN
        From..to bits:   4...4
        DefaultValue:    0x0
        Access type:     read-write
        Description:     Ram Enable Paper spin option.

*/
#define DCAN_CLKCFG_RAMEN                                            0x00000010U
#define DCAN_CLKCFG_RAMEN_M                                          0x00000010U
#define DCAN_CLKCFG_RAMEN_S                                          4U
/*

        Field:           CLKSEL
        From..to bits:   5...6
        DefaultValue:    0x0
        Access type:     read-write
        Description:     DCAN clock selection

        ENUMs:
            NOCLOCK:                     No clock is selected
            HOST_DIV2_CLK:               80Mhz host clock is selected.
            HFXT:                        HFXT is selected as source
            HOST_DIV2_PSWL_CLK:          Un-swallowed clock 80Mhz is selected.
*/
#define DCAN_CLKCFG_CLKSEL_W                                         2U
#define DCAN_CLKCFG_CLKSEL_M                                         0x00000060U
#define DCAN_CLKCFG_CLKSEL_S                                         5U
#define DCAN_CLKCFG_CLKSEL_NOCLOCK                                   0x00000000U
#define DCAN_CLKCFG_CLKSEL_HOST_DIV2_CLK                             0x00000020U
#define DCAN_CLKCFG_CLKSEL_HFXT                                      0x00000040U
#define DCAN_CLKCFG_CLKSEL_HOST_DIV2_PSWL_CLK                        0x00000060U

#endif /* __HW_DCAN_H__*/
