/*
 * Copyright (c) 2023-2024, Texas Instruments Incorporated - https://www.ti.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 *  ======== I2CWFF3.syscfg.js ========
 */

"use strict";

/* get Common /ti/drivers utility functions */
let Common = system.getScript("/ti/drivers/Common.js");

let intPriority = Common.newIntPri()[0];
intPriority.displayName = "Interrupt Priority";
intPriority.name = "interruptPriority";

let swiPriority = Common.newSwiPri();
swiPriority.displayName = "Software Interrupt Priority";
swiPriority.name = "softwareInterruptPriority";

/*
 *  ======== devSpecific ========
 *  Device-specific extensions to be added to base I2C configuration
 */
let devSpecific = {
    config: [
        intPriority,
        swiPriority
    ],

    templates: {
        boardc : "/ti/drivers/i2c/I2CWFF3.Board.c.xdt",
        boardh : "/ti/drivers/i2c/I2C.Board.h.xdt"
    },

    /* PIN instances */
    moduleInstances: moduleInstances,

    _getPinResources: _getPinResources,

    /* override generic supported bit rates with device-specific ones */
    _supportedBitRates: [100, 400, 1000]
};

/*
 *  ======== _getPinResources ========
 */
function _getPinResources(inst)
{
    let pin;

    if (inst.i2c) {
        let sclPin = inst.i2c.sclPin.$solution.devicePinName;
        let sdaPin = inst.i2c.sdaPin.$solution.devicePinName;
        pin = "\nSCL: " + sclPin + "\nSDA: " + sdaPin;

        if (inst.$hardware && inst.$hardware.displayName) {
            pin += "\n" + inst.$hardware.displayName;
        }
    }

    return (pin);
}

/*
 *  ======== moduleInstances ========
 *  returns PIN instances
 */
function moduleInstances(inst)
{
    let shortName = inst.$name.replace("CONFIG_", "");
    let instances = [
        {
            name: "sdaPinInstance",
            displayName: "I2C SDA Pin Configuration While Pin Is Not In Use",
            moduleName: "/ti/drivers/GPIO",
            collapsed: true,
            requiredArgs: {
                parentInterfaceName: "i2c",
                parentSignalName: "sdaPin",
                parentSignalDisplayName: "SDA"
            },
            args: {
                $name: "CONFIG_GPIO_" + shortName + "_SDA",
                mode: "Input",
                pull: "Pull Up"
            }
        },
        {
            name: "sclPinInstance",
            displayName: "I2C CLK Pin Configuration While Pin Is Not In Use",
            moduleName: "/ti/drivers/GPIO",
            collapsed: true,
            requiredArgs: {
                parentInterfaceName: "i2c",
                parentSignalName: "sclPin",
                parentSignalDisplayName: "SCL"
            },
            args: {
                $name: "CONFIG_GPIO_" + shortName + "_SCL",
                mode: "Input",
                pull: "Pull Up"
            }
        }
    ];

    return instances;
}

/*
 *  ======== pinmuxRequirements ========
 *  Return peripheral pin requirements as a function of config
 *  Called on instantiation and every config change.
 *
 *  $super is needed to call the generic module's functions
 */
function pinmuxRequirements(inst, $super)
{
    let result = $super.pinmuxRequirements ? $super.pinmuxRequirements(inst) : [];
    result[0].resources[0].interfaceNames[0] = "DATA";
    result[0].resources[1].interfaceNames[0] = "CLK";
    return (result);
}

/*
 *  ======== extend ========
 *  Extends a base exports object to include any device specifics
 *
 *  This function is invoked by the generic I2C module to
 *  allow us to augment/override as needed for the WFF3 devices
 */
function extend(base)
{
    /* override base pinmuxRequirements */
    devSpecific.pinmuxRequirements = function (inst) {
        return pinmuxRequirements(inst, base);
    };

    /* display which driver implementation can be used */
    base = Common.addImplementationConfig(base, "I2C", null,
        [{name: "I2CWFF3"}], null);

    /* merge and overwrite base module attributes */
    let result = Object.assign({}, base, devSpecific);

    /* concatenate device-specific configs */
    result.config = base.config.concat(devSpecific.config);

    return (result);
}

/*
 *  ======== exports ========
 *  Export device-specific extensions to base exports
 */
exports = {
    extend: extend
};
