/* testvectors_dhdsa_domain.c
 *
 * Description: DH/DSA domain parameters for test vectors of DH and DSA.
 */

/* -------------------------------------------------------------------------- */
/*                                                                            */
/*   Module        : DDK-130_bsd                                              */
/*   Version       : 4.1.1                                                    */
/*   Configuration : DDK_EIP130_BSD                                           */
/*                                                                            */
/*   Date          : 2023-Mar-10                                              */
/*                                                                            */
/* Copyright (c) 2014-2023 by Rambus, Inc. and/or its subsidiaries.           */
/*                                                                            */
/* Redistribution and use in source and binary forms, with or without         */
/* modification, are permitted provided that the following conditions are     */
/* met:                                                                       */
/*                                                                            */
/* 1. Redistributions of source code must retain the above copyright          */
/* notice, this list of conditions and the following disclaimer.              */
/*                                                                            */
/* 2. Redistributions in binary form must reproduce the above copyright       */
/* notice, this list of conditions and the following disclaimer in the        */
/* documentation and/or other materials provided with the distribution.       */
/*                                                                            */
/* 3. Neither the name of the copyright holder nor the names of its           */
/* contributors may be used to endorse or promote products derived from       */
/* this software without specific prior written permission.                   */
/*                                                                            */
/* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS        */
/* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT          */
/* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR      */
/* A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT       */
/* HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,     */
/* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT           */
/* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,      */
/* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY      */
/* THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT        */
/* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE      */
/* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.       */
/* -------------------------------------------------------------------------- */

#include <third_party/hsmddk/include/Kit/DriverFramework/Basic_Defs_API/incl/basic_defs.h>

#include "testvectors_dhdsa_domain.h"


// mod = L=1024, N=160
static const uint8_t DhDsaDomain_L1024N160_P[] =
{
    0x91, 0x2F, 0xFA, 0x30, 0x83, 0x08, 0xBF, 0x0D, 0xC1, 0x2B, 0x51, 0x58, 0xEC, 0x9F, 0x70, 0x44,
    0xF1, 0x20, 0x4D, 0x1A, 0xE8, 0x31, 0x47, 0x08, 0x0E, 0x52, 0x52, 0x2A, 0xEE, 0xFC, 0x03, 0xD8,
    0x40, 0x52, 0x56, 0xD1, 0x80, 0x6E, 0x3E, 0x82, 0x03, 0x1F, 0x72, 0x95, 0x94, 0xDA, 0xAC, 0xCC,
    0x69, 0x8C, 0x13, 0x19, 0xED, 0xBA, 0xD6, 0x54, 0x25, 0x27, 0x11, 0x9F, 0xD1, 0xB6, 0x4D, 0xE1,
    0xAA, 0x42, 0xE6, 0xF5, 0xF6, 0x51, 0x28, 0xED, 0x67, 0xE8, 0x7F, 0x2A, 0xC5, 0x11, 0x71, 0xDE,
    0x3A, 0x93, 0xA0, 0xEE, 0xDE, 0xF2, 0xBA, 0x69, 0x90, 0x46, 0xA4, 0x19, 0xE9, 0x48, 0x4D, 0xB1,
    0x55, 0x79, 0x53, 0x18, 0xEF, 0x52, 0x1B, 0xD0, 0xC6, 0xC0, 0xB0, 0xED, 0x25, 0x13, 0x1D, 0x16,
    0xB2, 0x1D, 0x77, 0x79, 0xA6, 0x9E, 0x11, 0x85, 0x56, 0x7F, 0xE1, 0x25, 0xB4, 0x8C, 0x54, 0x1B,
};
static const uint8_t DhDsaDomain_L1024N160_Q[] =
{
    0xA1, 0xF8, 0x6D, 0x57, 0xA1, 0xF9, 0x50, 0xF7, 0xC3, 0xCD, 0xD1, 0x74, 0xF2, 0xE7, 0xDA, 0x9B,
    0x00, 0x5F, 0x8C, 0x6F,
};
static const uint8_t DhDsaDomain_L1024N160_G[] =
{
    0x16, 0x53, 0x1A, 0xF9, 0xA7, 0xC3, 0x92, 0x26, 0x42, 0x1A, 0x07, 0xCE, 0x0C, 0x08, 0xA1, 0xE3,
    0x98, 0x04, 0xAC, 0x22, 0x34, 0xC6, 0xF7, 0x2E, 0xBF, 0x85, 0xBC, 0xB9, 0x0B, 0x3F, 0x09, 0x48,
    0xC6, 0x5F, 0x24, 0xB5, 0xBB, 0x20, 0x88, 0xBA, 0x77, 0x9D, 0xE8, 0xEC, 0x60, 0xBE, 0x85, 0x80,
    0x9D, 0x5E, 0xF7, 0x81, 0xFD, 0x46, 0x16, 0xC2, 0x6C, 0x07, 0x40, 0x4B, 0xC0, 0x6C, 0x33, 0x26,
    0xA0, 0xB0, 0xAF, 0xCB, 0xF5, 0x9D, 0xB9, 0x76, 0x7C, 0x18, 0x0F, 0x13, 0x43, 0x81, 0xD4, 0x6D,
    0x9D, 0x92, 0xAC, 0x87, 0x44, 0x14, 0xAB, 0x45, 0xA8, 0x28, 0xF4, 0x0A, 0x80, 0x81, 0x0F, 0x2B,
    0x63, 0x74, 0xAD, 0x3A, 0xED, 0x54, 0x97, 0xDC, 0x00, 0x29, 0x9C, 0x5F, 0x16, 0x4A, 0x6E, 0x0C,
    0x0B, 0x4C, 0x0F, 0x41, 0x2B, 0x55, 0x82, 0xCC, 0xB1, 0x92, 0xB0, 0x46, 0x23, 0x64, 0x79, 0xF6,
};
static const TestVector_DhDsaDomain_Rec_t DhDsaDomain_L1024N160 =
{
    1024, 160,
    DhDsaDomain_L1024N160_P, sizeof(DhDsaDomain_L1024N160_P),
    DhDsaDomain_L1024N160_Q, sizeof(DhDsaDomain_L1024N160_Q),
    DhDsaDomain_L1024N160_G, sizeof(DhDsaDomain_L1024N160_G),
};
TestVector_DhDsaDomain_t DhDsaDomain_L1024N160_p = &DhDsaDomain_L1024N160;


// mod = L=2048, N=224
static const uint8_t DhDsaDomain_L2048N224_P[] =
{
    0xC6, 0xFD, 0x3F, 0x3D, 0x5A, 0x95, 0x9D, 0x0C, 0x68, 0x57, 0xF0, 0x24, 0xD9, 0xB7, 0x37, 0x56,
    0xEA, 0x1E, 0x6F, 0x51, 0x1A, 0x68, 0xB6, 0xEC, 0x76, 0x4C, 0x6D, 0x49, 0xE1, 0xC7, 0xE4, 0xE8,
    0xAA, 0xC6, 0xD4, 0x5A, 0x0B, 0xE5, 0x19, 0x73, 0xFA, 0x11, 0xCA, 0x54, 0xCE, 0xBC, 0x13, 0xAA,
    0x2B, 0x0D, 0x0F, 0xFD, 0x86, 0xD7, 0x9F, 0xA0, 0xCF, 0x61, 0x3C, 0x3B, 0x35, 0xF2, 0x0D, 0x61,
    0x2F, 0xD9, 0x23, 0xEF, 0xA2, 0xC6, 0x8B, 0xB1, 0xE4, 0xAA, 0xC0, 0x23, 0x9A, 0x39, 0xB0, 0x05,
    0xBD, 0x3E, 0x2E, 0xD6, 0xD9, 0x6B, 0xE2, 0x07, 0xAF, 0x3C, 0xD6, 0x32, 0x25, 0x9C, 0x60, 0x9F,
    0x31, 0xDF, 0x2A, 0x49, 0x1F, 0x67, 0x33, 0x9D, 0x4B, 0x48, 0xA4, 0x82, 0x9F, 0x44, 0x6C, 0xE4,
    0xC7, 0x7C, 0x56, 0xFB, 0xE6, 0x92, 0x62, 0xF4, 0xF6, 0x32, 0x9B, 0xC8, 0x2F, 0xCD, 0xD3, 0xC2,
    0xCB, 0x85, 0xAC, 0xF5, 0x32, 0xDF, 0x6F, 0x19, 0xB2, 0x56, 0xD8, 0x3E, 0x7C, 0x47, 0x4C, 0x26,
    0xA1, 0x20, 0x6E, 0xC4, 0x0F, 0x24, 0x4E, 0xBD, 0x3B, 0x2E, 0x0D, 0x3E, 0xFE, 0x6A, 0x0A, 0xFA,
    0x08, 0x29, 0xC1, 0x99, 0x18, 0x9C, 0x6D, 0xBE, 0x49, 0xB1, 0x9F, 0xCC, 0xF4, 0xD8, 0x36, 0x7C,
    0xDA, 0xF4, 0xF3, 0x40, 0x94, 0x46, 0xD9, 0x7D, 0x75, 0x2B, 0xCC, 0x6A, 0xBB, 0x93, 0x2F, 0x52,
    0x35, 0x38, 0x74, 0xFF, 0xF8, 0x1D, 0x8D, 0xE0, 0x00, 0x53, 0x96, 0x9A, 0xFE, 0xA2, 0xEF, 0xD8,
    0x43, 0xE4, 0x3F, 0xAF, 0x5A, 0x02, 0x5B, 0x89, 0x51, 0xE6, 0xEC, 0x86, 0xEB, 0xC3, 0xA5, 0xA9,
    0xFE, 0xDC, 0x01, 0xB1, 0x7F, 0x4D, 0x88, 0xC1, 0xDB, 0x8C, 0xF3, 0x0B, 0x0B, 0x1C, 0x1F, 0x99,
    0xED, 0x3D, 0x28, 0x59, 0x2D, 0x7A, 0x0A, 0xF4, 0xF4, 0xB3, 0x30, 0xA8, 0x24, 0xC3, 0x14, 0x77,
};
static const uint8_t DhDsaDomain_L2048N224_Q[] =
{
    0xCF, 0x52, 0x18, 0x68, 0xA1, 0x40, 0x57, 0xC7, 0x04, 0xBE, 0x79, 0x92, 0x2E, 0xF6, 0x88, 0x61,
    0x6F, 0x26, 0x4A, 0xE7, 0xFB, 0x8C, 0x78, 0x09, 0x26, 0xA8, 0xFB, 0x8B,
};
static const uint8_t DhDsaDomain_L2048N224_G[] =
{
    0x32, 0x99, 0x24, 0x50, 0xE9, 0x3C, 0x0E, 0x81, 0xE4, 0x9E, 0x94, 0xD6, 0x41, 0xE6, 0x91, 0xD2,
    0xE0, 0xB9, 0xBA, 0xEF, 0x3C, 0xB6, 0x1B, 0x9F, 0x49, 0x08, 0xA0, 0x6B, 0x0B, 0x2D, 0xD0, 0x87,
    0xD9, 0x52, 0x30, 0xC6, 0x0E, 0x35, 0xFE, 0xFF, 0xBA, 0x8E, 0x8A, 0x79, 0xAE, 0x1C, 0xC6, 0x25,
    0xF0, 0xDD, 0xF1, 0x8A, 0xD3, 0x74, 0x4E, 0xD7, 0x85, 0x57, 0xB7, 0x15, 0x18, 0x29, 0x07, 0x99,
    0xE1, 0x52, 0xF6, 0x43, 0xE1, 0xB2, 0x43, 0xA8, 0x1F, 0x87, 0x4D, 0x0A, 0xFC, 0x33, 0xBC, 0x2D,
    0x24, 0x99, 0x7E, 0xDB, 0xB6, 0x57, 0xAF, 0x8E, 0x92, 0x15, 0xFD, 0x8D, 0x8F, 0xA2, 0x7B, 0x77,
    0x5F, 0x83, 0xE7, 0x25, 0x50, 0x4F, 0xD8, 0x17, 0xEA, 0xAF, 0x05, 0x3E, 0x41, 0x00, 0x4A, 0xAB,
    0x0A, 0x85, 0x42, 0x7A, 0x11, 0xDF, 0xE1, 0xFF, 0x02, 0xFA, 0x56, 0x3F, 0x92, 0x89, 0x03, 0x39,
    0xAD, 0x64, 0x8D, 0xF7, 0xE6, 0x89, 0x5E, 0x7A, 0x48, 0xE1, 0xF4, 0xFA, 0x13, 0x44, 0x89, 0x7A,
    0xD3, 0xCE, 0xEE, 0xB9, 0x55, 0xA1, 0x9F, 0xED, 0x97, 0xB7, 0x90, 0xE4, 0x6F, 0x0B, 0x1D, 0x01,
    0xB4, 0xAB, 0x80, 0xBF, 0x1E, 0x3C, 0xE2, 0x8D, 0x33, 0x65, 0x00, 0xAA, 0x06, 0xF5, 0x2E, 0xE4,
    0xCF, 0xC2, 0xB3, 0x5A, 0x33, 0x04, 0xE5, 0xAD, 0xF0, 0x44, 0x6D, 0x0E, 0x93, 0xB4, 0x25, 0xC2,
    0xFC, 0xE4, 0x95, 0x80, 0x83, 0x71, 0x40, 0x67, 0xF5, 0x9A, 0x3B, 0x86, 0x19, 0x37, 0x02, 0x18,
    0x0C, 0xD9, 0x69, 0x51, 0xCD, 0x1F, 0x8D, 0x59, 0xE3, 0xB8, 0x4D, 0x59, 0xB8, 0x34, 0x06, 0x1F,
    0x85, 0x3D, 0x15, 0xBC, 0xE7, 0x2C, 0x33, 0x65, 0x35, 0x5E, 0x35, 0x40, 0x7C, 0x0C, 0x8A, 0x51,
    0x7D, 0xDB, 0xFA, 0x61, 0x2C, 0x67, 0x14, 0xCA, 0x3F, 0xDC, 0x9E, 0x52, 0x52, 0xED, 0x66, 0x86,
};
static const TestVector_DhDsaDomain_Rec_t DhDsaDomain_L2048N224 =
{
    2048, 224,
    DhDsaDomain_L2048N224_P, sizeof(DhDsaDomain_L2048N224_P),
    DhDsaDomain_L2048N224_Q, sizeof(DhDsaDomain_L2048N224_Q),
    DhDsaDomain_L2048N224_G, sizeof(DhDsaDomain_L2048N224_G),
};
TestVector_DhDsaDomain_t DhDsaDomain_L2048N224_p = &DhDsaDomain_L2048N224;


// mod = L=2048, N=256
static const uint8_t DhDsaDomain_L2048N256_P[] =
{
    0x9C, 0x9F, 0x2B, 0x96, 0x9A, 0x88, 0xA3, 0x8C, 0x9F, 0xB7, 0x9F, 0x59, 0xED, 0x25, 0xAD, 0x25,
    0x97, 0x87, 0x55, 0x45, 0x1D, 0xD4, 0x74, 0x31, 0x7B, 0x56, 0x04, 0x6E, 0x1A, 0xF0, 0xF8, 0x32,
    0xC7, 0x59, 0x7D, 0x3B, 0xC4, 0x20, 0xD4, 0x98, 0xDB, 0x82, 0x8F, 0x2F, 0x7F, 0x71, 0xD5, 0x3C,
    0x79, 0x4C, 0xF5, 0xCD, 0x11, 0x93, 0x6A, 0x22, 0x4E, 0x92, 0x54, 0xB4, 0x0C, 0x5D, 0x69, 0x71,
    0x9A, 0xBF, 0x54, 0xFB, 0x61, 0xAC, 0xD5, 0xEA, 0x70, 0xCD, 0xEF, 0x6C, 0x1A, 0xB3, 0x8E, 0x71,
    0x4A, 0xC7, 0x13, 0x04, 0xA3, 0xA5, 0x9E, 0x5C, 0x7A, 0x71, 0x26, 0xD4, 0x0E, 0x63, 0xE6, 0x87,
    0xE8, 0x4D, 0x24, 0xD5, 0x89, 0x36, 0x6D, 0x26, 0x22, 0xB9, 0xE5, 0x65, 0x88, 0xEF, 0x61, 0x0D,
    0x7B, 0x05, 0x0D, 0x2D, 0x85, 0x0C, 0x93, 0x19, 0x78, 0x0A, 0xB9, 0xE7, 0x7A, 0x27, 0x3E, 0x79,
    0x80, 0x13, 0xD8, 0xCE, 0x6A, 0xF5, 0xF4, 0x16, 0x31, 0xF8, 0x9A, 0xC1, 0x33, 0xDA, 0x3F, 0x85,
    0xE1, 0x4A, 0x3E, 0xB1, 0x9F, 0x25, 0x1B, 0x7A, 0x92, 0xD7, 0xA5, 0x6F, 0xB8, 0xA5, 0x3C, 0x96,
    0xCA, 0xAA, 0xF9, 0x99, 0xD0, 0x7F, 0x94, 0x01, 0xEE, 0xA9, 0xF1, 0x0F, 0x1E, 0x28, 0x58, 0xD0,
    0x62, 0x92, 0x30, 0x59, 0xE9, 0x3A, 0x30, 0x55, 0x44, 0x2F, 0xA3, 0x6E, 0x3C, 0xDC, 0xEB, 0x83,
    0x07, 0x00, 0x29, 0x11, 0x51, 0x32, 0xAF, 0xE6, 0x58, 0x65, 0xE7, 0xEE, 0x6D, 0x96, 0xEE, 0x7F,
    0x1D, 0xED, 0x65, 0x36, 0x37, 0xBF, 0x9B, 0xDA, 0xFF, 0xFE, 0x31, 0xC9, 0xD4, 0x5E, 0x46, 0x26,
    0xB2, 0xE5, 0xAF, 0xEC, 0x4C, 0x7A, 0x8B, 0xF1, 0x6E, 0xC1, 0x3C, 0xB7, 0xE9, 0x7B, 0xD4, 0x2D,
    0xF3, 0x40, 0xB1, 0xBA, 0x7A, 0x7D, 0xC2, 0x9B, 0xC3, 0xAF, 0xD4, 0xFD, 0xB2, 0x36, 0x4C, 0xD7,
};
static const uint8_t DhDsaDomain_L2048N256_Q[] =
{
    0x99, 0x46, 0x2C, 0x57, 0x30, 0x6C, 0xB5, 0x9B, 0x9F, 0x45, 0xCB, 0x97, 0xF9, 0xA7, 0xC7, 0x07,
    0x42, 0x76, 0x09, 0x39, 0xF5, 0x26, 0xCF, 0xCE, 0x3D, 0x97, 0xEC, 0x85, 0x9A, 0x80, 0x5F, 0x19,
};
static const uint8_t DhDsaDomain_L2048N256_G[] =
{
    0x76, 0xF6, 0xCB, 0x4D, 0x8D, 0xA1, 0x6E, 0x98, 0x4D, 0x97, 0x73, 0xDF, 0x30, 0xA9, 0x07, 0x05,
    0xE7, 0xFB, 0x70, 0xD0, 0x8C, 0x20, 0x45, 0x10, 0xF4, 0xB0, 0x0B, 0xEF, 0x48, 0x53, 0x5D, 0xAD,
    0x50, 0x28, 0xBF, 0x89, 0x5D, 0xC9, 0x01, 0xEE, 0xCC, 0x8C, 0x9D, 0xF2, 0x68, 0x49, 0x8D, 0xB4,
    0x2C, 0x93, 0xE2, 0xFA, 0x63, 0x94, 0x54, 0x74, 0x2A, 0x1D, 0xC0, 0xA7, 0xCB, 0x57, 0x43, 0xE9,
    0x26, 0xDE, 0x6A, 0x2F, 0x99, 0x2E, 0x17, 0xD2, 0x3A, 0x92, 0x72, 0x90, 0x8C, 0x3E, 0x6B, 0xCC,
    0x7C, 0xCB, 0xE6, 0xB5, 0x5B, 0xA2, 0xBC, 0xDE, 0x41, 0xD1, 0x01, 0x4E, 0xB8, 0x4F, 0xDE, 0x53,
    0x52, 0xA7, 0xFA, 0x0A, 0xAE, 0xC4, 0x95, 0x07, 0x8E, 0x23, 0x89, 0x4E, 0x9B, 0xFD, 0x8C, 0x9B,
    0xDE, 0x8A, 0xEC, 0x85, 0x98, 0xED, 0xE9, 0x97, 0x98, 0x16, 0x71, 0x2E, 0x26, 0xE1, 0xF2, 0x28,
    0xD6, 0x5F, 0x5A, 0x7A, 0xB9, 0x15, 0x4F, 0xCB, 0x3F, 0x4C, 0xE1, 0x9D, 0x77, 0x83, 0x48, 0x0C,
    0x17, 0xCE, 0x29, 0xC8, 0xA7, 0x97, 0x45, 0xC3, 0x3D, 0xE0, 0xD9, 0xD7, 0xBE, 0x89, 0x02, 0xE3,
    0x6A, 0x10, 0x34, 0x4C, 0xB0, 0x86, 0xC9, 0x9D, 0x8A, 0x02, 0xC4, 0x99, 0x31, 0x6A, 0x61, 0x95,
    0x21, 0xEB, 0x50, 0x60, 0x04, 0x65, 0x1D, 0x9A, 0x4E, 0x6D, 0x4E, 0xA0, 0xA4, 0xE3, 0x84, 0xAF,
    0x40, 0xE7, 0x0B, 0xDF, 0xC6, 0x7F, 0xCB, 0x92, 0x92, 0xE7, 0xBC, 0x04, 0x34, 0xBB, 0xB7, 0xA0,
    0x37, 0xC8, 0xC3, 0x6A, 0xE5, 0x79, 0x7C, 0x2F, 0xD8, 0x35, 0x78, 0xD4, 0x33, 0x58, 0x33, 0x8F,
    0x24, 0xE2, 0xB6, 0x69, 0x12, 0xE0, 0x62, 0xFE, 0x21, 0x05, 0x97, 0x19, 0x26, 0xBF, 0xC7, 0xD4,
    0x59, 0xDB, 0xD0, 0x3B, 0x1D, 0x5A, 0xDD, 0x55, 0x40, 0x54, 0xB7, 0x03, 0x89, 0x0B, 0xF9, 0x81,
};
static const TestVector_DhDsaDomain_Rec_t DhDsaDomain_L2048N256 =
{
    2048, 256,
    DhDsaDomain_L2048N256_P, sizeof(DhDsaDomain_L2048N256_P),
    DhDsaDomain_L2048N256_Q, sizeof(DhDsaDomain_L2048N256_Q),
    DhDsaDomain_L2048N256_G, sizeof(DhDsaDomain_L2048N256_G),
};
TestVector_DhDsaDomain_t DhDsaDomain_L2048N256_p = &DhDsaDomain_L2048N256;


// mod = L=3072, N=256
static const uint8_t DhDsaDomain_L3072N256_P[] =
{
    0x9E, 0x99, 0x34, 0x18, 0xCA, 0x6E, 0xDF, 0xEA, 0x2B, 0x91, 0x06, 0x17, 0x1B, 0xC2, 0xD1, 0xA8,
    0xD7, 0x2B, 0x4E, 0x00, 0x27, 0x3E, 0x3F, 0x53, 0xAE, 0x4E, 0x34, 0x76, 0xEC, 0xE3, 0x7A, 0xB9,
    0x7F, 0x57, 0x1C, 0xA5, 0x09, 0x82, 0xD7, 0xCA, 0xAE, 0xE0, 0x8B, 0x1F, 0x86, 0x65, 0x81, 0xC9,
    0xAA, 0x4F, 0x15, 0xE8, 0xE4, 0xEF, 0xE0, 0x47, 0x9C, 0xD0, 0xE1, 0x91, 0xE5, 0x6D, 0x78, 0xF5,
    0x0B, 0xDE, 0x87, 0x4F, 0x26, 0x3A, 0xAA, 0xAD, 0xB1, 0x37, 0x00, 0xE0, 0x57, 0x58, 0x4B, 0x69,
    0x27, 0xAE, 0x6E, 0x42, 0x95, 0xFF, 0x5B, 0x5F, 0x5B, 0x23, 0x7A, 0x6A, 0x30, 0x0C, 0x9E, 0x41,
    0xC0, 0x05, 0x9F, 0x52, 0x1C, 0xFE, 0x14, 0xCE, 0x96, 0xD2, 0x73, 0x51, 0xBF, 0x99, 0xEB, 0x1D,
    0x63, 0x50, 0x6B, 0x49, 0x0E, 0x48, 0x5D, 0x13, 0x30, 0xD8, 0x2A, 0xB4, 0xB0, 0x11, 0x3E, 0xA8,
    0x5F, 0x87, 0x24, 0xC9, 0x01, 0xA9, 0x4A, 0xF4, 0x06, 0xF8, 0xFE, 0xD1, 0xC5, 0x9A, 0xF3, 0xEF,
    0x10, 0x18, 0x9B, 0x07, 0xEC, 0x20, 0x8E, 0xF5, 0xE1, 0x51, 0x06, 0x8E, 0x88, 0xEF, 0x25, 0x17,
    0xBC, 0x52, 0x18, 0xBA, 0xC8, 0x74, 0x88, 0x07, 0x92, 0xF8, 0xE3, 0x75, 0xEC, 0xE6, 0x02, 0x55,
    0x23, 0x62, 0x11, 0x61, 0x8F, 0xE5, 0x38, 0xB6, 0xBA, 0x9A, 0x98, 0x7A, 0x04, 0x84, 0xC5, 0x93,
    0x11, 0x53, 0x4E, 0x85, 0x3B, 0x8A, 0xE6, 0x92, 0x8A, 0xEC, 0xE7, 0x53, 0x8D, 0x59, 0xCD, 0x82,
    0x4B, 0x74, 0xD5, 0x88, 0x72, 0xD3, 0xDC, 0xF5, 0x82, 0x2D, 0x19, 0x9B, 0x79, 0xC2, 0xEB, 0x39,
    0x87, 0x2F, 0xEB, 0x64, 0xBA, 0xDA, 0x3A, 0x9B, 0x17, 0x53, 0xE7, 0x09, 0x36, 0xC2, 0x8D, 0x79,
    0x33, 0xFB, 0x8F, 0x07, 0xF2, 0x1B, 0xA0, 0x75, 0xED, 0xA5, 0xFA, 0xFE, 0xDF, 0x0A, 0x0D, 0x4F,
    0x46, 0xF4, 0x4D, 0xD8, 0x7B, 0x6C, 0x37, 0xB5, 0xFD, 0x84, 0xC1, 0xA2, 0x00, 0xE3, 0x06, 0xD7,
    0x6F, 0x55, 0x96, 0x7E, 0x4C, 0xDB, 0x23, 0x9A, 0x4B, 0x00, 0x9F, 0xCE, 0x16, 0xC7, 0xBC, 0x59,
    0x2D, 0xD0, 0x72, 0xF7, 0x78, 0x49, 0xC5, 0x88, 0x49, 0x64, 0xF8, 0xB0, 0x4F, 0xDC, 0x73, 0x2F,
    0x31, 0x53, 0xA6, 0xEA, 0xC8, 0x50, 0x6A, 0x47, 0x2E, 0x51, 0xFC, 0xE2, 0x90, 0xFC, 0x2B, 0xE3,
    0x57, 0xFE, 0x00, 0xA4, 0xE9, 0xC9, 0xB7, 0x6D, 0x54, 0x6F, 0x04, 0xDE, 0x51, 0x12, 0x00, 0xEF,
    0xCC, 0x22, 0xD3, 0x87, 0x3B, 0x36, 0x40, 0xDF, 0xF0, 0x69, 0xA8, 0x18, 0x2E, 0x8D, 0xF8, 0xF1,
    0xA4, 0x10, 0x03, 0x79, 0x24, 0x1E, 0x62, 0x1F, 0x2D, 0xD2, 0x1E, 0xAE, 0x9B, 0x99, 0x19, 0xF7,
    0xFC, 0x06, 0x0F, 0xD6, 0x6B, 0x91, 0x26, 0xE8, 0xF0, 0xA7, 0x6E, 0x67, 0x7B, 0xB5, 0xAE, 0xE7,
};
static const uint8_t DhDsaDomain_L3072N256_Q[] =
{
    0x9F, 0x12, 0xC7, 0x98, 0xCF, 0xA7, 0xBA, 0x15, 0xB8, 0x2F, 0x54, 0x2C, 0x31, 0xB2, 0x28, 0x3D,
    0xAA, 0xF0, 0x6F, 0x04, 0x15, 0x21, 0x59, 0x93, 0xB3, 0x10, 0x4B, 0xF1, 0xC2, 0x35, 0x56, 0x4D,
};
static const uint8_t DhDsaDomain_L3072N256_G[] =
{
    0x1E, 0xAD, 0x70, 0x87, 0x00, 0x2D, 0x68, 0xC7, 0x0A, 0x65, 0xA8, 0x55, 0xEC, 0xEE, 0x5C, 0x2F,
    0x73, 0x1E, 0x62, 0xEC, 0xAF, 0xDB, 0xB8, 0x84, 0x64, 0xC3, 0x4B, 0x9C, 0xAB, 0x26, 0x8E, 0x89,
    0x4B, 0x94, 0xE6, 0x81, 0x5E, 0x2E, 0x15, 0x33, 0x7C, 0xBE, 0x50, 0x81, 0x71, 0xB4, 0xB9, 0xF5,
    0x3E, 0xF3, 0x42, 0xC0, 0x36, 0xF8, 0x8F, 0xBC, 0x12, 0xFC, 0x7C, 0x9C, 0x84, 0x75, 0xEB, 0x30,
    0xA8, 0xFE, 0xED, 0xB6, 0xF0, 0x94, 0xFE, 0x66, 0x71, 0xEA, 0x43, 0xCE, 0x87, 0xD0, 0xAA, 0x82,
    0x8A, 0xE3, 0x38, 0xAB, 0x10, 0x8B, 0x7C, 0x11, 0xC5, 0x32, 0x70, 0xD0, 0x76, 0x6B, 0x4C, 0x56,
    0x45, 0x28, 0xDA, 0x02, 0x2A, 0xB5, 0x0B, 0xD1, 0xDB, 0x89, 0x82, 0xD7, 0x12, 0xAA, 0x96, 0x2D,
    0x30, 0x68, 0xC3, 0xE9, 0xA5, 0x13, 0xE0, 0x29, 0xE3, 0x3E, 0x09, 0x77, 0xB2, 0x70, 0xC0, 0x15,
    0xB2, 0x6B, 0x20, 0x89, 0xCF, 0x17, 0x40, 0xAA, 0x41, 0xA9, 0x59, 0xD4, 0xBB, 0x17, 0x77, 0x2A,
    0x7A, 0xDD, 0xDA, 0x42, 0x16, 0x6A, 0x0E, 0xF4, 0xC8, 0xBA, 0x32, 0xAD, 0xA7, 0x43, 0x79, 0xB6,
    0xF2, 0x88, 0xC6, 0xE2, 0x28, 0xBA, 0x12, 0x2E, 0xBA, 0x8D, 0xA2, 0x83, 0x2A, 0x4E, 0x9A, 0xCC,
    0xA5, 0x43, 0x39, 0xF9, 0x26, 0xE5, 0xF5, 0x2F, 0x97, 0xB1, 0xDF, 0x57, 0xFA, 0xC0, 0x27, 0xC8,
    0xE4, 0x04, 0xB2, 0xB2, 0xF5, 0xCB, 0x57, 0xA0, 0x4F, 0x8B, 0xCE, 0xD1, 0xD1, 0x8B, 0xC0, 0xC5,
    0x47, 0xEA, 0x41, 0x52, 0x3F, 0x61, 0x12, 0x58, 0x25, 0x0F, 0xDB, 0x5B, 0x75, 0x34, 0x7A, 0xE1,
    0x3F, 0x0A, 0x35, 0x82, 0x7F, 0xDD, 0xBE, 0x82, 0x67, 0xF5, 0x8F, 0x08, 0xAC, 0xD2, 0x74, 0x27,
    0xB0, 0xE0, 0x0B, 0x19, 0x42, 0x3C, 0xFB, 0xB8, 0x23, 0x3E, 0xAC, 0x82, 0x85, 0xDD, 0x4A, 0xEB,
    0x7C, 0x0D, 0x20, 0x5F, 0xE4, 0x60, 0x5A, 0xF6, 0xC1, 0x78, 0x39, 0xFD, 0x59, 0x2E, 0xE5, 0xBD,
    0xAD, 0x52, 0x7B, 0xD1, 0x9A, 0x50, 0x56, 0x6A, 0xBB, 0x38, 0xA2, 0xF7, 0x0C, 0x23, 0x00, 0x59,
    0x92, 0x69, 0xA3, 0xA5, 0x1F, 0xDF, 0xE7, 0xD5, 0x3A, 0x64, 0xEE, 0xBE, 0x46, 0xF1, 0x03, 0xE9,
    0x2B, 0xD9, 0xF6, 0x4A, 0x13, 0x1A, 0xBA, 0x73, 0x26, 0xA8, 0x48, 0x71, 0x74, 0x44, 0x52, 0x98,
    0x79, 0x80, 0xB5, 0x39, 0x49, 0x16, 0x05, 0xA1, 0x9B, 0xD7, 0x41, 0x64, 0xFC, 0x21, 0xBC, 0x9F,
    0x8A, 0x17, 0x1C, 0x4F, 0x55, 0x87, 0xC5, 0xFF, 0xC2, 0x8C, 0x19, 0xCC, 0x4A, 0x5F, 0x02, 0x04,
    0xB7, 0x09, 0x2B, 0xAB, 0x6A, 0x74, 0xF4, 0xDF, 0x44, 0xEE, 0x25, 0x7F, 0x94, 0xE3, 0x4E, 0x5A,
    0x69, 0x8F, 0x80, 0xEA, 0xB3, 0x2D, 0xC8, 0xC7, 0x27, 0xC3, 0x2C, 0x3E, 0x33, 0xF6, 0x72, 0x5D,
};
static const TestVector_DhDsaDomain_Rec_t DhDsaDomain_L3072N256 =
{
    3072, 256,
    DhDsaDomain_L3072N256_P, sizeof(DhDsaDomain_L3072N256_P),
    DhDsaDomain_L3072N256_Q, sizeof(DhDsaDomain_L3072N256_Q),
    DhDsaDomain_L3072N256_G, sizeof(DhDsaDomain_L3072N256_G),
};
TestVector_DhDsaDomain_t DhDsaDomain_L3072N256_p = &DhDsaDomain_L3072N256;


/* end of file testvectors_dhdsa_domain.c */
