/*
 * Copyright (c) 2023-2025 Texas Instruments Incorporated - http://www.ti.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 *  ======== AESGCMXXF3HSM.syscfg.js ========
 */

"use strict";

/* get Common /ti/drivers utility functions */
let Common = system.getScript("/ti/drivers/Common.js");

let deviceId = system.deviceData.deviceId;

let intPriority = Common.newIntPri()[0];
intPriority.name = "interruptPriority";
intPriority.displayName = "Interrupt Priority";
intPriority.description = "Crypto peripheral interrupt priority";

/*
*  ======== getLibs ========
*  Argument to the /ti/utils/build/GenLibs.cmd.xdt template
*/
function getLibs(mod)
{
   /* Get device information from GenLibs */
   let GenLibs = system.getScript("/ti/utils/build/GenLibs");

   let libGroup = {
        name: "/third_party/hsmddk",
        deps: [],
        libs: [],
        allowDuplicates: true
    };

    if (!system.modules["/ti/utils/TrustZone"]) {
        if (deviceId.match(/CC27.{3}1/)) {
            libGroup.libs.push(GenLibs.libPath("third_party/hsmddk", "hsmddk_cc27xxx10_its.a"));
        } else if (deviceId.match(/CC27.{3}2/)) {
            libGroup.libs.push(GenLibs.libPath("third_party/hsmddk", "hsmddk_cc27xxx20_its.a"));
        } else if (deviceId.match(/CC35/)) {
            libGroup.libs.push(GenLibs.libPath("third_party/hsmddk", "hsmddk_cc35xx_its.a"));
        }
    }

   return (libGroup);
}

/*
 *  ======== devSpecific ========
 *  Device-specific extensions to be added to base AESGCM configuration
 */
let devSpecific = {
    config: [
        intPriority
    ],

    templates : {
        boardc: "/ti/drivers/aesgcm/AESGCMXXF3HSM.Board.c.xdt",
        boardh: "/ti/drivers/aesgcm/AESGCM.Board.h.xdt"
    }
};

/*
 *  ======== extend ========
 *  Extends a base exports object to include any device specifics
 */
function extend(base)
{
    /* display which driver implementation can be used */
    base = Common.addImplementationConfig(base, "AESGCM", null,
        [{name: "AESGCMLPF3"}], null);

    if (deviceId.match(/CC27/) || deviceId.match(/CC35/)) {
        devSpecific["templates"]["/ti/utils/build/GenLibs.cmd.xdt"] = {modName: "/ti/drivers/AESGCM", getLibs: getLibs};
    }

    /* merge and overwrite base module attributes */
    return (Object.assign({}, base, devSpecific));
}

/*
 *  ======== exports ========
 *  Export device-specific extensions to base exports
 */
exports = {
    /* required function, called by base TRNG module */
    extend: extend
};
