import json
from os import getcwd
from os.path import isfile, join, dirname
from sys import exit
import shutil
from pathlib import Path
import xml.etree.ElementTree as ET
import os


def update_target_setup_json(cwd, devices_info):
    target_setup_json_path = join(cwd, "cloudagent", "target_setup.json")
    origin_target_setup_json_path = join(cwd, "cloudagent", "target_setup.json.origin")

    if isfile(origin_target_setup_json_path):
        print("Overriding target_setup.json from origin")
        shutil.copy(origin_target_setup_json_path, target_setup_json_path)

    if isfile(target_setup_json_path):
        print("Creating target_setup.json backup from origin")
        shutil.copy(target_setup_json_path, origin_target_setup_json_path)
    else:
        print(f"{target_setup_json_path} not found !")
        return False

    target_setup_json_content = json.load(open(target_setup_json_path, "r"))
    devices_id_list = [d["id"] for d in target_setup_json_content["devices"]]
    devices_list = [d["device"] for d in devices_info]
    for device in devices_list:
        if device not in devices_id_list:
            target_setup_json_content["devices"].append(
                {
                    "id": device,
                    "description": f"SimpleLink {device}",
                    "xmlFile": device,
                    "connectionIndices": [
                        5,
                        7,
                        8,
                        9,
                        10,
                        11,
                        12,
                        13,
                        14,
                        15,
                        16,
                        17,
                        18,
                        19,
                    ],
                }
            )

    json.dump(target_setup_json_content, open(target_setup_json_path, "w"), indent=2)

    return True


def update_FlashDLLDB_xml(cwd, devices_info):
    internal_configuration = """
    <FlashDLL name="Flash{DEVICE}" needFlashProperties="true" needFlashUtility="true">
        <interface>JTAG</interface>
        <targetType>TMS470REX</targetType>
        <targetType>TMS470R33X</targetType>
        <partnum beginsWith="{DEVICE}" />
    </FlashDLL>
    
    """

    external_configuration = """
    <FlashDLL name="Flash{DEVICE}" needFlashProperties="true" needFlashUtility="true">
        <interface>JTAG</interface>
        <targetType>CS_DAP_0</targetType>
        <partnum beginsWith="{DEVICE}" />
    </FlashDLL>   
    
    """

    flash_dll_db_xml_path = join(cwd, "DebugServer", "bin", "FlashDLLDB.xml")
    origin_flash_dll_db_xml_path = join(
        cwd, "DebugServer", "bin", "FlashDLLDB.xml.origin"
    )

    if isfile(origin_flash_dll_db_xml_path):
        print("Overriding FlashDLLDB.xml from origin")
        shutil.copy(origin_flash_dll_db_xml_path, flash_dll_db_xml_path)

    if isfile(flash_dll_db_xml_path):
        print("Creating FlashDLLDB.xml backup from origin")
        shutil.copy(flash_dll_db_xml_path, origin_flash_dll_db_xml_path)
    else:
        print(f"{flash_dll_db_xml_path} not found !")
        return False

    devices_list = [d["device"] for d in devices_info]
    devices_is_external = [d["is_external"] for d in devices_info]
    flash_devices_list = [f"Flash{device}" for device in devices_list]

    tree = ET.parse(flash_dll_db_xml_path)
    root = tree.getroot()

    child_attribute_names = [
        child.attrib.get("name") for child in root if "name" in child.attrib
    ]

    for device, flash_device, is_external in zip(
        devices_list, flash_devices_list, devices_is_external
    ):
        if flash_device not in child_attribute_names:
            configuration = (
                external_configuration if is_external else internal_configuration
            )
            flash_dll = ET.fromstring(configuration.format(DEVICE=device))
            root.insert(0, flash_dll)

    ET.indent(tree, space="\t", level=0)
    tree.write(flash_dll_db_xml_path)

    return True


def update_FlashPropertiesTranslator_xml(cwd, devices_info):
    configuration = """
    <FlashProperties name="{DEVICE}">
        <partnum beginsWith="{DEVICE}" />
    </FlashProperties>
    """

    flash_prop_tran_xml_path = join(
        cwd, "DebugServer", "propertyDB", "FlashPropertiesTranslator.xml"
    )
    origin_flash_prop_tran_xml_path = join(
        cwd, "DebugServer", "propertyDB", "FlashPropertiesTranslator.xml.origin"
    )

    if isfile(origin_flash_prop_tran_xml_path):
        print("Overriding FlashPropertiesTranslators.xml from origin")
        shutil.copy(origin_flash_prop_tran_xml_path, flash_prop_tran_xml_path)

    if isfile(flash_prop_tran_xml_path):
        print("Creating FlashPropertiesTranslator.xml backup from origin")
        shutil.copy(flash_prop_tran_xml_path, origin_flash_prop_tran_xml_path)
    else:
        print(f"{flash_prop_tran_xml_path} not found !")
        return False

    tree = ET.parse(flash_prop_tran_xml_path)
    root = tree.getroot()

    devices_list = [d["device"] for d in devices_info]
    child_attribute_names = [
        child.attrib.get("name") for child in root if "name" in child.attrib
    ]

    for device in devices_list:
        if device not in child_attribute_names:
            flash_properties = ET.fromstring(configuration.format(DEVICE=device))
            root.insert(0, flash_properties)

    ET.indent(tree, space="\t", level=0)
    tree.write(flash_prop_tran_xml_path, encoding="utf-8", xml_declaration=True)

    return True


def touch_timestamp(cwd):
    timestamp = join(cwd, "common", "targetdb", "timestamp")

    if not isfile(timestamp):
        print(f"{timestamp} not found !")
        return False

    print(f"Touch and go of {timestamp}")
    with open(timestamp, "r") as f:
        timestamp_content = int(f.read())
    with open(timestamp, "w") as f:
        f.write(str(timestamp_content + 1))

    return True


def get_devices_info(cwd):
    cc35xx_release_path = join(cwd, ".metadata", "cc35xx_release.json")
    cc35xx_release_content = json.load(open(cc35xx_release_path, "r"))
    return cc35xx_release_content


if __name__ == "__main__":
    cwd = getcwd()
    ccs_base_path = Path(cwd).parents[2]
    devices_info = get_devices_info(ccs_base_path)
    update_target_setup_json(ccs_base_path, devices_info)
    update_FlashDLLDB_xml(ccs_base_path, devices_info)
    update_FlashPropertiesTranslator_xml(ccs_base_path, devices_info)
    touch_timestamp(ccs_base_path)
