/******************************************************************************

@file  app_bas.c

@brief This file contains the Battery Service application functionality.

Group: WCS, BTS
Target Device: cc23xx

******************************************************************************

 Copyright (c) 2022-2026, Texas Instruments Incorporated
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 *  Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

 *  Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

 *  Neither the name of Texas Instruments Incorporated nor the names of
    its contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

******************************************************************************


*****************************************************************************/

//*****************************************************************************
//! Includes
//*****************************************************************************
#include <string.h>
#include "ti/ble/app_util/common/util.h"
#include <ti/ble/profiles/battery/battery_profile.h>
#include "ti/ble/app_util/framework/bleapputil_api.h"
#include "ti/ble/app_util/menu/menu_module.h"
#include <app_main.h>

//*****************************************************************************
//! Defines
//*****************************************************************************
// Notification enabled value
#define BAS_CCC_UPDATE_NOTIFICATION_ENABLED         1

// Default battery level
#define BAS_DEFAUALT_BAT_LEVEL                      100

// Level Interval
#define MSEC_PER_MINUTE                             60000
#define BAS_LEVEL_DEFAUALT_INTERVAL                 1                                                  // Battery Level interval in minutes
#define BAS_LEVEL_DEFAUALT_INTERVAL_MSEC            (BAS_LEVEL_DEFAUALT_INTERVAL * MSEC_PER_MINUTE)    // Battery Level interval in milliseconds

//*****************************************************************************
//! TYPEDEF
//*****************************************************************************

//*****************************************************************************
//!  LOCAL VARIABLES
//*****************************************************************************

// Battery level interval in milliseconds
static uint16_t ba_battLevelIntMesc = BAS_LEVEL_DEFAUALT_INTERVAL_MSEC;

// Battery level clock
static Clock_Struct ba_battLevelClk;

// Battery level
static uint8_t ba_levelVal = 0;

//*****************************************************************************
//!LOCAL FUNCTIONS
//*****************************************************************************
static bStatus_t BA_setBattLevel( uint8_t newBatteryLevel);
static void BA_onBattLevelCCCUpdateCB( uint16_t connHandle, uint16_t pValue );
static void BA_BattLevelClkTimeout( char *pData );
static void BA_clockCB( uint32_t arg );

//*****************************************************************************
//!GLOBALS
//*****************************************************************************

//*****************************************************************************
//!APPLICATION CALLBACK
//*****************************************************************************
// Battery application callback function for incoming data
static BAP_cb_t ba_profileCB =
{
 BA_onBattLevelCCCUpdateCB,
};

//*****************************************************************************
//! Functions
//*****************************************************************************

/*********************************************************************
 * @fn      BA_onBattLevelCCCUpdateCB
 *
 * @brief   Callback from Battery profile indicating CCC update for Battery Level characteristic
 *
 * @param   connHandle - connection message was received on
 * @param   pValue - pointer to the CCC value
 *
 * @return  None
 */
static void BA_onBattLevelCCCUpdateCB( uint16_t connHandle, uint16_t pValue )
{
 if ( pValue == GATT_CLIENT_CFG_NOTIFY )
  {
    MenuModule_printf( APP_MENU_PROFILE_STATUS_LINE1, 0,
                       "BAS app: Battery Level characteristic - connectionHandle: "
                       MENU_MODULE_COLOR_YELLOW "%d " MENU_MODULE_COLOR_RESET
                       "Notifications enabled", connHandle );
  }
  else
  {
    MenuModule_printf( APP_MENU_PROFILE_STATUS_LINE1, 0,
                       "BAS app: Battery Level characteristic - connectionHandle: "
                       MENU_MODULE_COLOR_YELLOW "%d " MENU_MODULE_COLOR_RESET
                       "Notifications disabled", connHandle );
  }
}

/*********************************************************************
 * @fn      BA_start
 *
 * @brief   This function is called after stack initialization,
 *          the purpose of this function is to initialize and
 *          register the Battery Service profile.
 *
 * @param   None
 *
 * @return  SUCCESS or stack call status
 */
bStatus_t BA_start( void )
{
  bStatus_t status = SUCCESS;
  uint8_t batteryLevelValue = BAS_DEFAUALT_BAT_LEVEL;

  // Initialize BA Profile
  status = BAP_start( &ba_profileCB );

  if ( status == SUCCESS )
  {
      // Example of update the battery level
      status = BA_setBattLevel( batteryLevelValue );

      if ( status == SUCCESS )
      {
          // Create and start BA level timer
          Util_constructClock( &ba_battLevelClk, BA_clockCB,
                               ba_battLevelIntMesc, ba_battLevelIntMesc,
                               TRUE, (uint32)BA_BattLevelClkTimeout );

          MenuModule_printf( APP_MENU_PROFILE_STATUS_LINE, 0, "Battery Service start" );
      }
  }

  // Return status value
  return ( status );
}

/*********************************************************************
 * @fn      BA_setBattLevel
 *
 * @brief   This function updates the battery level.
 *
 * @param   newBatteryLevel - battery level.
 * @return  SUCCESS or stack call status
 */
static bStatus_t BA_setBattLevel( uint8_t newBatteryLevel )
{
  bStatus_t status = SUCCESS;
  status = BAP_setBattLevel( newBatteryLevel );

  if ( status == SUCCESS )
  {
    ba_levelVal = newBatteryLevel;
    MenuModule_printf( APP_MENU_PROFILE_STATUS_LINE2, 0,
                       "Battery app: battery level has been updated to: "
                       MENU_MODULE_COLOR_YELLOW "%d%%" MENU_MODULE_COLOR_RESET, newBatteryLevel);
  }
  else
  {
    MenuModule_printf( APP_MENU_PROFILE_STATUS_LINE2, 0,
                       "Battery app: Failed to update battery level - error code: "
                       MENU_MODULE_COLOR_RED "%d " MENU_MODULE_COLOR_RESET,
                       status );
  }

  return ( status );
}

/*********************************************************************
 * @fn      BA_BattLevelClkTimeout
 *
 * @brief   This function is triggered when the battery level clock expires
 *
 * @param   pData - pointer to data
 *
 * @return  None
 */
static void BA_BattLevelClkTimeout( char *pData )
{
  // Get current level value
  uint8_t curBattlevelVal = ba_levelVal;

  if ( curBattlevelVal == BAS_LEVEL_MIN )
  {
      curBattlevelVal = BAS_LEVEL_MAX;
  }
  else
  {
      curBattlevelVal--;
  }

  // Update the level value
  BA_setBattLevel( curBattlevelVal );
}

/*********************************************************************
 * @fn      BA_clockCB
 *
 * @brief   Callback function for clock module
 *
 * @param   arg - argument passed to callback function.
 *
 * @return  none
 */
static void BA_clockCB( uint32_t arg )
{
  BLEAppUtil_invokeFunctionNoData( ( InvokeFromBLEAppUtilContext_t ) arg );
}

/*********************************************************************
*********************************************************************/
