/******************************************************************************

@file  app_extctrl_l2cap_coc.c

@brief This file parse and process the messages comes form the external control module
 dispatcher module, and build the events from the app_l2cap_coc.c application and
 send it to the external control dispatcher module back.

Group: WCS, BTS
Target Device: cc23xx

******************************************************************************

 Copyright (c) 2024-2026, Texas Instruments Incorporated
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 *  Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

 *  Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

 *  Neither the name of Texas Instruments Incorporated nor the names of
    its contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

******************************************************************************


*****************************************************************************/


/*********************************************************************
 * INCLUDES
 */
#if defined (BLE_V41_FEATURES) && (BLE_V41_FEATURES & L2CAP_COC_CFG)
#include "ti/ble/app_util/framework/bleapputil_extctrl_dispatcher.h"
#include "app_extctrl_l2cap_coc.h"
#include <string.h>

/*********************************************************************
 * PROTOTYPES
 */
static void L2CAPCOCExtCtr_sendErrEvt(uint8_t, uint8_t);
static void L2CAPCOCExtCtr_commandParser(uint8_t*);
static void L2CAPCOCExtCtr_eventHandler(BLEAppUtil_eventHandlerType_e, uint32, BLEAppUtil_msgHdr_t*);
static void L2CAPCOCExtCtrl_extHostEvtHandler(uint8_t*, uint16_t);

/*********************************************************************
 * CONSTANTS
 */
#define L2CAPCOC_CMD_CREATE_PSM       0x00
#define L2CAPCOC_CMD_CLOSE_PSM        0x01
#define L2CAPCOC_CMD_CONNECT_REQ      0x02
#define L2CAPCOC_CMD_DISCONNECT_REQ   0x03
#define L2CAPCOC_CMD_SEND_SDU         0x04

/*********************************************************************
 * TYPEDEFS
 */

/*********************************************************************
 * GLOBALS
 */
// The external control host event handler
static ExtCtrlHost_eventHandler_t gExtHostEvtHandler = NULL;

/*********************************************************************
 * FUNCTIONS
 */

/*********************************************************************
 * @fn      L2CAPCOCExtCtr_commandParser
 *
 * @brief   This function parse the received host message, and call the
 *          the relevant API with the data.
 *
 * @param   pData - pointer to the data message.
 *
 * @return  None
 */
static void L2CAPCOCExtCtr_commandParser(uint8_t *pData)
{
  if (pData != NULL)
  {
    appMsg_t* appMsg = (appMsg_t*)pData;
    bStatus_t status = SUCCESS;

    switch (appMsg->cmdOp)
    {
      case L2CAPCOC_CMD_CREATE_PSM:
      {
        AppExtCtrlL2capCreatePsm_t *pCmd = (AppExtCtrlL2capCreatePsm_t *)appMsg->pData;
        status = L2CAPCOC_createPsm(pCmd->psm);
        break;
      }
      case L2CAPCOC_CMD_CLOSE_PSM:
      {
        AppExtCtrlL2capClosePsm_t *pCmd = (AppExtCtrlL2capClosePsm_t *)appMsg->pData;
        status = L2CAPCOC_closePsm(pCmd->psm);
        break;
      }
      case L2CAPCOC_CMD_CONNECT_REQ:
      {
        AppExtCtrlL2capConnectReq_t *pCmd = (AppExtCtrlL2capConnectReq_t *)appMsg->pData;

        L2capCoc_connectReqCmdParams_t params;
        params.connHandle = pCmd->connHandle;
        params.psm = pCmd->psm;
        params.peerPsm = pCmd->peerPsm;

        status = L2CAPCOC_connectReq(&params);
        break;
      }
      case L2CAPCOC_CMD_DISCONNECT_REQ:
      {
        AppExtCtrlL2capDisconnectReq_t *pCmd = (AppExtCtrlL2capDisconnectReq_t *)appMsg->pData;
        status = L2CAPCOC_disconnectReq(pCmd->CID);
        break;
      }
      case L2CAPCOC_CMD_SEND_SDU:
      {
        AppExtCtrlL2capSendSdu_t *pCmd = (AppExtCtrlL2capSendSdu_t *)appMsg->pData;

        L2capCoc_sendSduCmdParams_t *pParams = ICall_malloc(sizeof(L2capCoc_sendSduCmdParams_t) + pCmd->len);
        pParams->connHandle = pCmd->connHandle;
        pParams->CID = pCmd->CID;
        pParams->len = pCmd->len;
        memcpy(pParams->pPayload, pCmd->pPayload, pCmd->len);

        status = L2CAPCOC_sendSDU(pParams);
        // Free the resources
        ICall_free(pParams);
        break;
      }

      default:
      {
        status = FAILURE;
        break;
      }
    }

    if (status != SUCCESS)
    {
      L2CAPCOCExtCtr_sendErrEvt(status, appMsg->cmdOp);
    }
  }
}

/*********************************************************************
 * @fn      L2CAPCOCExtCtr_sendErrEvt
 *
 * @brief   This function sends the error event to the external control.
 *
 * @param   errCode - the error code
 * @param   cmdOp   - the command op-code that raised the error
 *
 * @return  None
 */
static void L2CAPCOCExtCtr_sendErrEvt(uint8_t errCode, uint8_t cmdOp)
{
  // Create error event structure
  AppExtCtrlErrorEvent_t errEvt;
  errEvt.event        = APP_EXTCTRL_FAILURE;
  errEvt.cmdOp        = cmdOp;
  errEvt.appSpecifier = APP_SPECIFIER_L2CAPCOC;
  errEvt.errCode      = errCode;

  // Send error event
  L2CAPCOCExtCtrl_extHostEvtHandler((uint8_t *)&errEvt, sizeof(AppExtCtrlErrorEvent_t));
}

/*********************************************************************
 * @fn      L2CAPCOCExtCtr_dataTypeEvtHandler
 *
 * @brief   This function handles events from type @ref BLEAPPUTIL_L2CAP_DATA_TYPE
 *
 * @param   event     - message event.
 * @param   pDataPkt  - pointer to message data.
 *
 * @return  None
 */
static void L2CAPCOCExtCtr_dataTypeEvtHandler(uint32 event, l2capDataEvent_t *pDataPkt)
{

  if (pDataPkt != NULL)
  {
    l2capPacket_t pkt = pDataPkt->pkt;

    // Create event structure
    AppExtCtrlL2capCocDataEvent_t *pPktDataEvt = NULL;

    // Allocate the necessary size
    pPktDataEvt = (AppExtCtrlL2capCocDataEvent_t *) ICall_malloc(sizeof(AppExtCtrlL2capCocDataEvent_t) + pkt.len);

    if (pPktDataEvt != NULL)
    {
      pPktDataEvt->event      = APP_EXTCTRL_L2CAP_DATA_RECEIVED;
      // Copy the connection handle
      pPktDataEvt->connHandle = pkt.connHandle;
      // Copy the CID
      pPktDataEvt->CID        = pkt.CID;
      // Copy the length of the data
      pPktDataEvt->len        = pkt.len;
      // Copy the address
      memcpy(pPktDataEvt->pData, pkt.pPayload , pkt.len);

      // Forward the event
      L2CAPCOCExtCtrl_extHostEvtHandler((uint8_t *)pPktDataEvt, sizeof(AppExtCtrlL2capCocDataEvent_t) + pPktDataEvt->len);
      // Free the resources
      ICall_free(pPktDataEvt);
    }
  }
}

/*********************************************************************
 * @fn      L2CAPCOCExtCtr_channelEstEvtHandler
 *
 * @brief   This function handles events from type @ref BLEAPPUTIL_L2CAP_CHANNEL_ESTABLISHED_EVT
 *
 * @param   evt - the event @ref L2CAP_CHANNEL_ESTABLISHED.
 *
 * @return  None
 */
static void L2CAPCOCExtCtr_channelEstEvtHandler(l2capChannelEstEvt_t evt)
{
  // Create event structure
  AppExtCtrlL2capChannelEstEvent_t pktDataEvt;

  pktDataEvt.event                 = APP_EXTCTRL_L2CAP_CHANNEL_ESTABLISHED;
  pktDataEvt.result                = evt.result;
  pktDataEvt.CID                   = evt.CID;
  pktDataEvt.psm                   = evt.info.psm;
  pktDataEvt.mtu                   = evt.info.mtu;
  pktDataEvt.mps                   = evt.info.mps;
  pktDataEvt.credits               = evt.info.credits;
  pktDataEvt.peerCID               = evt.info.peerCID;
  pktDataEvt.peerMtu               = evt.info.peerMtu;
  pktDataEvt.peerMps               = evt.info.peerMps;
  pktDataEvt.peerCredits           = evt.info.peerCredits;
  pktDataEvt.peerCreditThreshold   = evt.info.peerCreditThreshold;

  // Forward the event
  L2CAPCOCExtCtrl_extHostEvtHandler((uint8_t *)&pktDataEvt, sizeof(AppExtCtrlL2capChannelEstEvent_t));
}

/*********************************************************************
 * @fn      L2CAPCOCExtCtr_channelTermEvtHandler
 *
 * @brief   This function handles events from type @ref BLEAPPUTIL_L2CAP_CHANNEL_TERMINATED_EVT
 *
 * @param   evt - the event @ref L2CAP_CHANNEL_TERMINATED_EVT.
 *
 * @return  None
 */
static void L2CAPCOCExtCtr_channelTermEvtHandler(l2capChannelTermEvt_t evt)
{
  // Create event structure
  AppExtCtrlL2capChannelTermEvent_t pktDataEvt;

  pktDataEvt.event      = APP_EXTCTRL_L2CAP_CHANNEL_TERMINATED;

  pktDataEvt.CID        = evt.CID;
  pktDataEvt.peerCID    = evt.peerCID;
  pktDataEvt.reason     = evt.reason;

  // Forward the event
  L2CAPCOCExtCtrl_extHostEvtHandler((uint8_t *)&pktDataEvt, sizeof(AppExtCtrlL2capChannelTermEvent_t));

}

/*********************************************************************
 * @fn      L2CAPCOCExtCtr_channelOutOfCreditEvtHandler
 *
 * @brief   This function handles events from type @ref BLEAPPUTIL_L2CAP_OUT_OF_CREDIT_EVT
 *
 * @param   evt - the event @ref L2CAP_OUT_OF_CREDIT_EVT.
 *
 * @return  None
 */
static void L2CAPCOCExtCtr_channelOutOfCreditEvtHandler(l2capCreditEvt_t evt)
{
  // Create event structure
  AppExtCtrlL2capCreditEvent_t pktDataEvt;

  pktDataEvt.event      = APP_EXTCTRL_L2CAP_OUT_OF_CREDIT;

  pktDataEvt.CID        = evt.CID;
  pktDataEvt.peerCID    = evt.peerCID;
  pktDataEvt.credits    = evt.credits;

  // Forward the event
  L2CAPCOCExtCtrl_extHostEvtHandler((uint8_t *)&pktDataEvt, sizeof(AppExtCtrlL2capCreditEvent_t));

}

/*********************************************************************
 * @fn      L2CAPCOCExtCtr_signalTypeEvtHandler
 *
 * @brief   This function handles events from type @ref BLEAPPUTIL_L2CAP_SIGNAL_TYPE
 *
 * @param   event     - message event.
 * @param   pMsgData  - pointer to message data.
 *
 * @return  None
 */
static void L2CAPCOCExtCtr_signalTypeEvtHandler(uint32 event, BLEAppUtil_msgHdr_t *pMsgData)
{
  if (pMsgData != NULL)
  {
    l2capSignalEvent_t *pL2capSignalEvt = (l2capSignalEvent_t *)pMsgData;
    switch (event)
    {
      case BLEAPPUTIL_L2CAP_CHANNEL_ESTABLISHED_EVT:
      {
        L2CAPCOCExtCtr_channelEstEvtHandler((pL2capSignalEvt->cmd).channelEstEvt);
        break;
      }

      case BLEAPPUTIL_L2CAP_CHANNEL_TERMINATED_EVT:
      {
        L2CAPCOCExtCtr_channelTermEvtHandler((pL2capSignalEvt->cmd).channelTermEvt);
        break;
      }

      case BLEAPPUTIL_L2CAP_OUT_OF_CREDIT_EVT:
      {
        L2CAPCOCExtCtr_channelOutOfCreditEvtHandler((pL2capSignalEvt->cmd).creditEvt);
        break;
      }

      default:
      {
        break;
      }
    }
  }
}

/*********************************************************************
 * @fn      L2CAPCOCExtCtr_eventHandler
 *
 * @brief   This function handles the events raised from the application that
 * this module registered to.
 *
 * @param   eventType - the type of the events @ref BLEAppUtil_eventHandlerType_e.
 * @param   event     - message event.
 * @param   pMsgData  - pointer to message data.
 *
 * @return  None
 */
static void L2CAPCOCExtCtr_eventHandler(BLEAppUtil_eventHandlerType_e eventType, uint32 event, BLEAppUtil_msgHdr_t *pMsgData)
{

  if (pMsgData != NULL)
  {
    switch (eventType)
    {
      case BLEAPPUTIL_L2CAP_SIGNAL_TYPE:
      {
        L2CAPCOCExtCtr_signalTypeEvtHandler(event, pMsgData);
        break;
      }

      case BLEAPPUTIL_L2CAP_DATA_TYPE:
      {
        L2CAPCOCExtCtr_dataTypeEvtHandler(event, (l2capDataEvent_t *)pMsgData);
        break;
      }

      default :
      {
        break;
      }
    }
  }
}

/*********************************************************************
 * @fn      L2CAPCOCExtCtrl_extHostEvtHandler
 *
 * @brief   The purpose of this function is to forward the event to the external
 *          control host event handler that the external control returned once
 *          this module registered to it.
 *
 * @param   pData    - pointer to the event data
 * @param   dataLen  - data length.
 *
 * @return  None
 */
static void L2CAPCOCExtCtrl_extHostEvtHandler(uint8_t *pData, uint16_t dataLen)
{
  if (gExtHostEvtHandler != NULL && pData != NULL)
  {
    gExtHostEvtHandler(pData, dataLen);
  }
}

/*********************************************************************
 * @fn      L2CAPCOCExtCtrl_start
 *
 * @brief   This function is called after stack initialization,
 *          the purpose of this function is to initialize and
 *          register the specific message handler of the connection module
 *          to the external control dispatcher, and register the call back
 *          event handler function to the l2cap_coc application.
 *
 * @return  SUCCESS/FAILURE
 */
bStatus_t L2CAPCOCExtCtrl_start(void)
{
  bStatus_t status = SUCCESS;

  // Register to the Dispatcher module
  gExtHostEvtHandler = Dispatcher_registerMsgHandler(APP_SPECIFIER_L2CAPCOC,
                                                     &L2CAPCOCExtCtr_commandParser,
                                                     APP_CAP_L2CAPCOC);

  // If the registration succeed, register event handler call back to the l2cap_coc application
  if (gExtHostEvtHandler != NULL)
  {
    L2CAPCOC_registerEvtHandler(&L2CAPCOCExtCtr_eventHandler);
  }

  return status;
}

#endif // BLE_V41_FEATURES & L2CAP_COC_CFG
