"""
Copyright (C) 2024, Texas Instruments Incorporated

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

    Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the
    distribution.

    Neither the name of Texas Instruments Incorporated nor the names of
    its contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""

import logging
import json
from copy import copy
import typer
from queue import Queue
from importlib.metadata import version
from tilogger.interface import LogOutputABC, LogPacket, LoggerCliCtx
from tilogger.tracedb import Opcode


logger = logging.getLogger("toReplayFile Output")

REPLAY_FILE_FORMAT_VERSION = "1.0.0"


class ToReplayFileOutput(LogOutputABC):
    """This class stores notified log packets to a specified JSON file."""

    def __init__(self, replayFile, verbose):
        self.replayFile = replayFile
        self.verbose = verbose
        self._packetWrittenCount = 0
        self._packetLostCount = 0
        self._queue: Queue = Queue()

    def start(self):
        metadata = {
            "properties": {
                "tilogger": version("tilogger"),
                "tilogger-to-replayfile-output": version("tilogger-to-replayfile-output"),
                "format_version": REPLAY_FILE_FORMAT_VERSION,
            }
        }
        # Add intial metadata to replay file
        with open(self.replayFile, "w", encoding="utf-8") as file:
            json.dump(metadata, file)
            file.write("\n")

        # Append log packets from internal queue to replay file as a JSON object
        while True:
            packet = self._queue.get()
            with open(self.replayFile, "a", encoding="utf-8") as file:
                try:
                    json.dump(packet, file, cls=LogPacketEncoder)
                    file.write("\n")
                    if self.verbose:
                        self._packetWrittenCount += 1
                        print(f"ToReplayFile: {self._packetWrittenCount} packets written to {self.replayFile}")
                except Exception as e:
                    self._packetLostCount += 1
                    logger.warning(f"ToReplayFile: Warning, packet not written to file - {e}")

    def notify_packet(self, packet: LogPacket) -> None:
        self._queue.put(packet)


# Function that adds a command to a typer instance via decorator
def output_factory_cli(app: typer.Typer):
    @app.command(name="to-replayfile")
    def output_factory_hcb(
        ctx: typer.Context,
        file: str = typer.Option(..., help="Replay file path (JSON file)"),
        verbose: bool = typer.Option(
            False, "--verbose", show_default=False, help="Print updates in terminal while writing to file"
        ),
    ):
        """Store log outputs to a replay file.

        You need to specify path to a JSON file.
        A replay file will be generated by storing the captured LogPackets in the
        specified JSON file. If a path to an existing JSON file is provided, the
        file will be overwritten.

        A invocation storing an ITM stream parsed with the
        provided ELF file to replayfile.json:

            tilogger --elf path/to/elf.out itm COM6 12000000 to-replayfile --file path/to/replayfile.json


        To replay the log events, use

            tilogger from-replayfile <FILE> <OUTPUT>
        """

        state = ctx.ensure_object(LoggerCliCtx)
        return ToReplayFileOutput(replayFile=file, verbose=verbose)


### JSON Helpers ###


# Used to make the logPacket JSON serializable
class LogPacketEncoder(json.JSONEncoder):
    def default(self, o):
        log = copy(vars(o))  # copy to avoid pass by reference
        log["level"] = log["level"].value
        log["opcode"] = Opcode.REPLAY_FILE.value  # change to REPLAY_FILE, to avoid ELF file handling
        log.pop("trace_db")  # remove tracedb, data already processed
        log.pop("data")  # remove data, data already processed

        return log
