/*
 * Copyright (c) 2001-2019, Arm Limited and Contributors. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */


#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include <limits.h>

/* mbedtls lib */
#include "mbedtls/cipher.h"
#include "mbedtls/timing.h"

/* CC pal */
#include "mbedtls_cc_util_asset_prov.h"
#include "mbedtls_cc_mng.h"
#include "cc_pal_types.h"

/* local */
#include "run_integration_pal_log.h"
#include "run_integration_test.h"
#include "run_integration_helper.h"
#include "run_integration_otp.h"

#ifdef RUNIT_PIE_ENABLED
/* include sbrom data file to determine whether we are running system flows */
#include "bsv_integration_data_def.h"
#endif /* RUNIT_PIE_ENABLED */

#ifndef RUNIT_ASSET_PROV_SYS_FLOW_ENABLE
#define RUNIT_ASSET_PROV_SYS_FLOW_ENABLE 0
#endif

#ifndef RUNIT_ASSET_PROV_SKIP_TEST
#define RUNIT_ASSET_PROV_SKIP_TEST 0
#endif

/************************************************************
 *
 * static function prototypes
 *
 ************************************************************/
static RunItError_t runIt_assetProv(void);

/************************************************************
 *
 * static functions
 *
 ************************************************************/
#if ! RUNIT_ASSET_PROV_SYS_FLOW_ENABLE
static RunItError_t runIt_assetProv(void)
{
    RunItError_t rc = RUNIT_ERROR__OK;

    static const uint32_t ASSET_ID = 0x19952006;

    static uint32_t OTP_SECURE_2_HBK128_VALUES[] = {

        /*  [0x00-0x07]: 256bit Device root key (HUK) */
        0x01020408, 0x01010101, 0x01020408, 0x02020202, 0x01020408, 0x04040404, 0x01020408, 0x08080808,
        /*  [0x08-0x0B]: 128bit ICV Provosioning secret (Kpicv) */
        0xEEEEAAAA, 0x11115555, 0x11115555, 0xEEEEAAAA,
        /*  [0x0C-0x0F]: 128bit ICV Provosioning secret (Kceicv) */
        0x00000000, 0x00000000, 0x00000000, 0x00000000,
        /*  [0x10]: manufacturer programmed flag */
        0x3C0040E0,
        /*  [0x11-0x14]:  HBK0 */
        0x21F7B953, 0xFE56A24D, 0x4E3EC4EF, 0xC21F3C40,
        /*  [0x15-0x18]: HBK1 */
        0x2FF17BDA, 0xE93EB506, 0x9AA54F0B, 0x4ADE752B,
        /*  [0x19-0x1C]: 128bit OEM provisioning secret (Kcp) */
        0x0000AAAA,  0x11115555, 0x11115555, 0x0000AAAA,
        /*  [0x1D-0x20]: 128bit OEM code encryption secret (Kce) */
        0x00000000, 0x00000000, 0x00000000, 0x00000000,
        /*  [0x21]: OEM programmed flag */
        0x00005838,
        /*  [0x22-0x23]: Hbk0 trusted FW min version */
        0x00000001, 0x00000000,
        /*  [0x24-0x26]: Hbk1 trusted FW min version */
        0x00000003, 0x00000000, 0x00000000,
        /*  [0x27]: general purpose configuration flag */
        0x00000000,
        /*  [0x28-0x2B] - 128b DCU lock mask */
        0x55551111, 0xAAAAEEEE, 0x55551111, 0xAAAAEEEE,

    };

    static const uint8_t ASSET_PKG[] = {
        0x74, 0x65, 0x73, 0x41, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xb8, 0x4b, 0xce, 0x45, 0x9d, 0x6b, 0x4d, 0xc2, 0xa2, 0x7d, 0xb3, 0x92,
        0x47, 0x1a, 0x40, 0xc5, 0x43, 0x17, 0xad, 0x30, 0xeb, 0xa2, 0x76, 0xa3, 0xb8, 0x50, 0xc3, 0x6e,
        0xbc, 0x1f, 0x8a, 0x87, 0xf7, 0x66, 0x91, 0x77, 0x8c, 0x18, 0xae, 0x25, 0x5e, 0x86, 0x6e, 0x38,
        0x96, 0xdc, 0x85, 0x1b, 0xc1, 0x65, 0xd0, 0x14, 0x1b, 0x64, 0xf9, 0xb5, 0xa9, 0x1a, 0xbf, 0x22,
        0x50, 0xab, 0xd9, 0xba, 0x4e, 0xad, 0x5e, 0x2f, 0x8b, 0xcb, 0x1c, 0x77, 0xde, 0xcf, 0xf1, 0x95,
        0xc1, 0x21, 0x90, 0xf4, 0x90, 0x12, 0x0f, 0xeb, 0x05, 0x70, 0xea, 0x5b, 0x64, 0xf9, 0x9b, 0xb3,
        0x98, 0x08, 0x3b, 0xb5, 0xf3, 0x09, 0x06, 0x12, 0x5e, 0x41, 0x31, 0x37, 0x9a, 0xcc, 0x59, 0x21,
        0x2f, 0x24, 0x6b, 0x37, 0x0a, 0x0d, 0x68, 0x92, 0x6c, 0x6d, 0xfc, 0xdc, 0xe5, 0x72, 0xfc, 0x79,
        0xf3, 0x98, 0x7c, 0x7f, 0x6f, 0x4d, 0xbc, 0xe8, 0x31, 0xe8, 0x03, 0x26, 0x3e, 0x51, 0x97, 0x80,
        0x15, 0xb0, 0xad, 0x52, 0x00, 0x1d, 0x0f, 0x89, 0x9b, 0xa7, 0x7a, 0xa2, 0xa8, 0x39, 0x4c, 0x39,
        0x76, 0xdc, 0x13, 0x79, 0x85, 0x92, 0x5c, 0xc2, 0x07, 0xce, 0xeb, 0x79, 0xbe, 0x91, 0xf8, 0x6f,
        0xcc, 0xee, 0x61, 0x6e, 0xcc, 0xbe, 0xa0, 0x18, 0xc9, 0x58, 0x5d, 0x49, 0xf0, 0x6a, 0x18, 0x82,
        0xf9, 0x03, 0x9f, 0x8b, 0x44, 0x74, 0xd4, 0x99, 0xb6, 0x27, 0xa9, 0x9b, 0x3a, 0x9c, 0x86, 0x21,
        0x51, 0x55, 0xbf, 0xbe, 0x17, 0xe0, 0x91, 0x61, 0xcd, 0x28, 0x2e, 0x2c, 0x2a, 0xb7, 0xb8, 0xc4,
        0x1e, 0x3f, 0x41, 0x42, 0x2e, 0x61, 0xda, 0x4b, 0x2f, 0x02, 0x32, 0x88, 0x57, 0xfe, 0x48, 0x82,
        0x18, 0x62, 0x39, 0xfa, 0x56, 0xc4, 0x17, 0x72, 0x24, 0xee, 0xfe, 0xe2, 0x8a, 0x29, 0x67, 0x1a,
        0xd7, 0x51, 0xb0, 0xc3, 0xc1, 0xa4, 0xc9, 0x07, 0xd9, 0xb5, 0x28, 0x90, 0xf0, 0xd2, 0x7a, 0x4f,
        0x47, 0x5b, 0x08, 0x0e, 0x79, 0x5a, 0xbf, 0xf0, 0x7d, 0x6a, 0x03, 0x08, 0xd4, 0x81, 0x61, 0xae,
    };


    static const uint8_t EXP_ASSET_BIN[] = {
        0x10, 0x00, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
    };


    uint32_t *pPkg = NULL;
    uint32_t *pBuf = NULL;

    RunItPtr pkgPtr;
    RunItPtr bufPtr;

    size_t assetLen = CC_ASSET_PROV_MAX_ASSET_PKG_SIZE;

    const char* TEST_NAME = "Asset Provisioning";
    RUNIT_SUB_TEST_START(TEST_NAME);

    ALLOC32_AND_COPY(pkgPtr, pPkg, ASSET_PKG, sizeof(ASSET_PKG));
    ALLOC32(bufPtr, pBuf, CC_ASSET_PROV_MAX_ASSET_PKG_SIZE);

    RUNIT_ASSERT(runIt_burnOtp((unsigned int *)OTP_SECURE_2_HBK128_VALUES, CC_MNG_LCS_SEC_ENABLED) == RUNIT_ERROR__OK);

    RUNIT_ASSERT_API(mbedtls_util_asset_pkg_unpack(ASSET_PROV_KEY_TYPE_KCP,
                                                   ASSET_ID,
                                                   pPkg,
                                                   sizeof(ASSET_PKG),
                                                   pBuf,
                                                   &assetLen) == CC_OK);

    RUNIT_ASSERT(memcmp(pBuf, EXP_ASSET_BIN, assetLen) == 0);

bail:
    FREE_IF_NOT_NULL(pkgPtr);
    FREE_IF_NOT_NULL(bufPtr);

    RUNIT_SUB_TEST_RESULT_W_PARAMS(TEST_NAME, "ASSET[%"PRIu32"B]", (uint32_t)256);
    return rc;
}
#else
static RunItError_t runIt_assetProv(void)
{
    RunItError_t rc = RUNIT_ERROR__OK;

#if ! RUNIT_ASSET_PROV_SKIP_TEST
    uint32_t lcs = 0xffffffff;

    static const uint32_t ASSET_ID = RUNIT_ASSET_PROV_ASSET_ID;

    static const uint8_t ASSET_PKG[] = { RUNIT_ASSET_PROV_PKG };

    static const uint8_t EXP_ASSET_BIN[] = { RUNIT_ASSET_PROV_PKG_BIN };

    uint32_t *pPkg = NULL;
    uint32_t *pBuf = NULL;

    RunItPtr pkgPtr;
    RunItPtr bufPtr;

    size_t assetLen;

    const char* TEST_NAME = "Asset Provisioning";
    RUNIT_SUB_TEST_START(TEST_NAME);

    ALLOC32_AND_COPY(pkgPtr, pPkg, ASSET_PKG, sizeof(ASSET_PKG));
    ALLOC32(bufPtr, pBuf, CC_ASSET_PROV_MAX_ASSET_PKG_SIZE);

    RUNIT_ASSERT(runIt_getLcs(&lcs) == RUNIT_ERROR__OK);
    RUNIT_PRINT_DBG("lcs[%"PRIu32"]\n", (uint32_t)lcs);

    RUNIT_ASSERT_API(mbedtls_util_asset_pkg_unpack(ASSET_PROV_KEY_TYPE_KPICV,
                                                   ASSET_ID,
                                                   pPkg,
                                                   sizeof(ASSET_PKG),
                                                   pBuf,
                                                   &assetLen) == CC_OK);

    RUNIT_ASSERT(memcmp(pBuf, EXP_ASSET_BIN, assetLen) == 0);

bail:
    FREE_IF_NOT_NULL(pkgPtr);
    FREE_IF_NOT_NULL(bufPtr);

    RUNIT_SUB_TEST_RESULT_W_PARAMS(TEST_NAME, "LCS[%"PRIu32"] ASSET[%"PRIu32"B]", (uint32_t)lcs, (uint32_t)256);
#endif /* RUNIT_ASSET_PROV_SKIP_TEST */
    return rc;
}

#endif /* RUNIT_ASSET_PROV_SYS_FLOW_ENABLE */
/************************************************************
 *
 * public functions
 *
 ************************************************************/
RunItError_t runIt_assetProvTest(void)
{
    RunItError_t rc = RUNIT_ERROR__OK;


#if !RUNIT_ASSET_PROV_SKIP_TEST
    const char* TEST_NAME = "AssetProv";
    RUNIT_TEST_START(TEST_NAME);

    RUNIT_ASSERT(runIt_assetProv() == RUNIT_ERROR__OK);

bail:

    RUNIT_TEST_RESULT(TEST_NAME);
#endif /* !RUNIT_ASSET_PROV_SKIP_TEST */

    (void)runIt_assetProv;

    return rc;
}
