/* testvectors_ecc_curves.c
 *
 * Description: ECC curves for test vectors of ECDH and ECDSA
 */

/* -------------------------------------------------------------------------- */
/*                                                                            */
/*   Module        : DDK-130_bsd                                              */
/*   Version       : 4.1.1                                                    */
/*   Configuration : DDK_EIP130_BSD                                           */
/*                                                                            */
/*   Date          : 2023-Mar-10                                              */
/*                                                                            */
/* Copyright (c) 2014-2023 by Rambus, Inc. and/or its subsidiaries.           */
/*                                                                            */
/* Redistribution and use in source and binary forms, with or without         */
/* modification, are permitted provided that the following conditions are     */
/* met:                                                                       */
/*                                                                            */
/* 1. Redistributions of source code must retain the above copyright          */
/* notice, this list of conditions and the following disclaimer.              */
/*                                                                            */
/* 2. Redistributions in binary form must reproduce the above copyright       */
/* notice, this list of conditions and the following disclaimer in the        */
/* documentation and/or other materials provided with the distribution.       */
/*                                                                            */
/* 3. Neither the name of the copyright holder nor the names of its           */
/* contributors may be used to endorse or promote products derived from       */
/* this software without specific prior written permission.                   */
/*                                                                            */
/* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS        */
/* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT          */
/* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR      */
/* A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT       */
/* HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,     */
/* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT           */
/* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,      */
/* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY      */
/* THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT        */
/* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE      */
/* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.       */
/* -------------------------------------------------------------------------- */

#include <third_party/hsmddk/include/Config/cs_eip130.h>

#include <third_party/hsmddk/include/Kit/DriverFramework/Basic_Defs_API/incl/basic_defs.h>
#include "testvectors_ecc_curves.h"


/* Elliptic Curves */
#ifdef REMOVE_IFDEF_WHEN_USE_IS_REQUIRED
/* SEC P-128r1 curve (p = 2^128 - 2^97 - 1) */
static const uint8_t ECurve_SEC_P128r1_p[] =
{
    0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};
static const uint8_t ECurve_SEC_P128r1_a[] =
{
    0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC
};
static const uint8_t ECurve_SEC_P128r1_b[] =
{
    0xE8, 0x75, 0x79, 0xC1, 0x10, 0x79, 0xF4, 0x3D,
    0xD8, 0x24, 0x99, 0x3C, 0x2C, 0xEE, 0x5E, 0xD3
};
static const uint8_t ECurve_SEC_P128r1_Gx[] =
{
    0x16, 0x1F, 0xF7, 0x52, 0x8B, 0x89, 0x9B, 0x2D,
    0x0C, 0x28, 0x60, 0x7C, 0xA5, 0x2C, 0x5B, 0x86
};
static const uint8_t ECurve_SEC_P128r1_Gy[] =
{
    0xCF, 0x5A, 0xC8, 0x39, 0x5B, 0xAF, 0xEB, 0x13,
    0xC0, 0x2D, 0xA2, 0x92, 0xDD, 0xED, 0x7A, 0x83
};
static const uint8_t ECurve_SEC_P128r1_n[] =
{
    0xFF, 0xFF, 0xFF, 0xFE, 0x00, 0x00, 0x00, 0x00,
    0x75, 0xA3, 0x0D, 0x1B, 0x90, 0x38, 0xA1, 0x15
};
const TestVector_ECC_Curve_Rec_t ECurve_SEC_P128r1 =
{
    TV_ECC_FAMILY_NIST_P,
    128,
    ECurve_SEC_P128r1_p,  sizeof(ECurve_SEC_P128r1_p),
    ECurve_SEC_P128r1_a,  sizeof(ECurve_SEC_P128r1_a),
    ECurve_SEC_P128r1_b,  sizeof(ECurve_SEC_P128r1_b),
    ECurve_SEC_P128r1_Gx, sizeof(ECurve_SEC_P128r1_Gx),
    ECurve_SEC_P128r1_Gy, sizeof(ECurve_SEC_P128r1_Gy),
    ECurve_SEC_P128r1_n,  sizeof(ECurve_SEC_P128r1_n),
    1
};
TestVector_ECC_Curve_t ECC_Curve_SEC_P128r1 = &ECurve_SEC_P128r1;
#endif

#ifdef REMOVE_IFDEF_WHEN_USE_IS_REQUIRED
/* SEC P-128r2 curve (p = 2^128 - 2^97 - 1) */
static const uint8_t ECurve_SEC_P128r2_p[] =
{
    0xFF, 0xFF, 0xFF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};
static const uint8_t ECurve_SEC_P128r2_a[] =
{
    0xD6, 0x03, 0x19, 0x98, 0xD1, 0xB3, 0xBB, 0xFE,
    0xBF, 0x59, 0xCC, 0x9B, 0xBF, 0xF9, 0xAE, 0xE1
};
static const uint8_t ECurve_SEC_P128r2_b[] =
{
    0x5E, 0xEE, 0xFC, 0xA3, 0x80, 0xD0, 0x29, 0x19,
    0xDC, 0x2C, 0x65, 0x58, 0xBB, 0x6D, 0x8A, 0x5D
};
static const uint8_t ECurve_SEC_P128r2_Gx[] =
{
    0x7B, 0x6A, 0xA5, 0xD8, 0x5E, 0x57, 0x29, 0x83,
    0xE6, 0xFB, 0x32, 0xA7, 0xCD, 0xEB, 0xC1, 0x40
};
static const uint8_t ECurve_SEC_P128r2_Gy[] =
{
    0x27, 0xB6, 0x91, 0x6A, 0x89, 0x4D, 0x3A, 0xEE,
    0x71, 0x06, 0xFE, 0x80, 0x5F, 0xC3, 0x4B, 0x44
};
static const uint8_t ECurve_SEC_P128r2_n[] =
{
    0x3F, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0xFF, 0xFF,
    0xBE, 0x00, 0x24, 0x72, 0x06, 0x13, 0xB5, 0xA3
};
const TestVector_ECC_Curve_Rec_t ECurve_SEC_P128r2 =
{
    TV_ECC_FAMILY_NIST_P,
    128,
    ECurve_SEC_P128r2_p,  sizeof(ECurve_SEC_P128r2_p),
    ECurve_SEC_P128r2_a,  sizeof(ECurve_SEC_P128r2_a),
    ECurve_SEC_P128r2_b,  sizeof(ECurve_SEC_P128r2_b),
    ECurve_SEC_P128r2_Gx, sizeof(ECurve_SEC_P128r2_Gx),
    ECurve_SEC_P128r2_Gy, sizeof(ECurve_SEC_P128r2_Gy),
    ECurve_SEC_P128r2_n,  sizeof(ECurve_SEC_P128r2_n),
    4
};
TestVector_ECC_Curve_t ECC_Curve_SEC_P128r2 = &ECurve_SEC_P128r2;
#endif

#ifdef REMOVE_IFDEF_WHEN_USE_IS_REQUIRED
/* SEC P-160r1 curve (p = 2^160 - 2^31 - 1) */
static const uint8_t ECurve_SEC_P160r1_p[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0x7F, 0xFF, 0xFF, 0xFF
};
static const uint8_t ECurve_SEC_P160r1_a[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0x7F, 0xFF, 0xFF, 0xFC
};
static const uint8_t ECurve_SEC_P160r1_b[] =
{
    0x1C, 0x97, 0xBE, 0xFC, 0x54, 0xBD, 0x7A, 0x8B,
    0x65, 0xAC, 0xF8, 0x9F, 0x81, 0xD4, 0xD4, 0xAD,
    0xC5, 0x65, 0xFA, 0x45
};
static const uint8_t ECurve_SEC_P160r1_Gx[] =
{
    0x4A, 0x96, 0xB5, 0x68, 0x8E, 0xF5, 0x73, 0x28,
    0x46, 0x64, 0x69, 0x89, 0x68, 0xC3, 0x8B, 0xB9,
    0x13, 0xCB, 0xFC, 0x82
};
static const uint8_t ECurve_SEC_P160r1_Gy[] =
{
    0x23, 0xA6, 0x28, 0x55, 0x31, 0x68, 0x94, 0x7D,
    0x59, 0xDC, 0xC9, 0x12, 0x04, 0x23, 0x51, 0x37,
    0x7A, 0xC5, 0xFB, 0x32
};
static const uint8_t ECurve_SEC_P160r1_n[] =
{
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x01, 0xF4, 0xC8, 0xF9, 0x27, 0xAE,
    0xD3, 0xCA, 0x75, 0x22, 0x57
};
const TestVector_ECC_Curve_Rec_t ECurve_SEC_P160r1 =
{
    TV_ECC_FAMILY_NIST_P,
    160,
    ECurve_SEC_P160r1_p,  sizeof(ECurve_SEC_P160r1_p),
    ECurve_SEC_P160r1_a,  sizeof(ECurve_SEC_P160r1_a),
    ECurve_SEC_P160r1_b,  sizeof(ECurve_SEC_P160r1_b),
    ECurve_SEC_P160r1_Gx, sizeof(ECurve_SEC_P160r1_Gx),
    ECurve_SEC_P160r1_Gy, sizeof(ECurve_SEC_P160r1_Gy),
    ECurve_SEC_P160r1_n,  sizeof(ECurve_SEC_P160r1_n),
    1
};
TestVector_ECC_Curve_t ECC_Curve_SEC_P160r1 = &ECurve_SEC_P160r1;
#endif

/* NIST P-192 curve (p = 2^192 - 2^64 - 1) */
static const uint8_t ECurve_NIST_P192_p[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};
static const uint8_t ECurve_NIST_P192_a[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC
};
static const uint8_t ECurve_NIST_P192_b[] =
{
    0x64, 0x21, 0x05, 0x19, 0xE5, 0x9C, 0x80, 0xE7,
    0x0F, 0xA7, 0xE9, 0xAB, 0x72, 0x24, 0x30, 0x49,
    0xFE, 0xB8, 0xDE, 0xEC, 0xC1, 0x46, 0xB9, 0xB1
};
static const uint8_t ECurve_NIST_P192_Gx[] =
{
    0x18, 0x8D, 0xA8, 0x0E, 0xB0, 0x30, 0x90, 0xF6,
    0x7C, 0xBF, 0x20, 0xEB, 0x43, 0xA1, 0x88, 0x00,
    0xF4, 0xFF, 0x0A, 0xFD, 0x82, 0xFF, 0x10, 0x12
};
static const uint8_t ECurve_NIST_P192_Gy[] =
{
    0x07, 0x19, 0x2B, 0x95, 0xFF, 0xC8, 0xDA, 0x78,
    0x63, 0x10, 0x11, 0xED, 0x6B, 0x24, 0xCD, 0xD5,
    0x73, 0xF9, 0x77, 0xA1, 0x1E, 0x79, 0x48, 0x11
};
static const uint8_t ECurve_NIST_P192_n[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0x99, 0xDE, 0xF8, 0x36,
    0x14, 0x6B, 0xC9, 0xB1, 0xB4, 0xD2, 0x28, 0x31
};
const TestVector_ECC_Curve_Rec_t ECurve_NIST_P192 =
{
    TV_ECC_FAMILY_NIST_P,
    192,
    ECurve_NIST_P192_p,  sizeof(ECurve_NIST_P192_p),
    ECurve_NIST_P192_a,  sizeof(ECurve_NIST_P192_a),
    ECurve_NIST_P192_b,  sizeof(ECurve_NIST_P192_b),
    ECurve_NIST_P192_Gx, sizeof(ECurve_NIST_P192_Gx),
    ECurve_NIST_P192_Gy, sizeof(ECurve_NIST_P192_Gy),
    ECurve_NIST_P192_n,  sizeof(ECurve_NIST_P192_n),
    1
};
TestVector_ECC_Curve_t ECC_Curve_NIST_P192 = &ECurve_NIST_P192;

/* NIST P-224 curve (p = 2^224 - 2^96 + 1) */
static const uint8_t ECurve_NIST_P224_p[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01
};
static const uint8_t ECurve_NIST_P224_a[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFE
};
static const uint8_t ECurve_NIST_P224_b[] =
{
    0xB4, 0x05, 0x0A, 0x85, 0x0C, 0x04, 0xB3, 0xAB,
    0xF5, 0x41, 0x32, 0x56, 0x50, 0x44, 0xB0, 0xB7,
    0xD7, 0xBF, 0xD8, 0xBA, 0x27, 0x0B, 0x39, 0x43,
    0x23, 0x55, 0xFF, 0xB4
};
static const uint8_t ECurve_NIST_P224_Gx[] =
{
    0xB7, 0x0E, 0x0C, 0xBD, 0x6B, 0xB4, 0xBF, 0x7F,
    0x32, 0x13, 0x90, 0xB9, 0x4A, 0x03, 0xC1, 0xD3,
    0x56, 0xC2, 0x11, 0x22, 0x34, 0x32, 0x80, 0xD6,
    0x11, 0x5C, 0x1D, 0x21
};
static const uint8_t ECurve_NIST_P224_Gy[] =
{
    0xBD, 0x37, 0x63, 0x88, 0xB5, 0xF7, 0x23, 0xFB,
    0x4C, 0x22, 0xDF, 0xE6, 0xCD, 0x43, 0x75, 0xA0,
    0x5A, 0x07, 0x47, 0x64, 0x44, 0xD5, 0x81, 0x99,
    0x85, 0x00, 0x7E, 0x34
};
static const uint8_t ECurve_NIST_P224_n[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x16, 0xA2,
    0xE0, 0xB8, 0xF0, 0x3E, 0x13, 0xDD, 0x29, 0x45,
    0x5C, 0x5C, 0x2A, 0x3D
};
const TestVector_ECC_Curve_Rec_t ECurve_NIST_P224 =
{
    TV_ECC_FAMILY_NIST_P,
    224,
    ECurve_NIST_P224_p,  sizeof(ECurve_NIST_P224_p),
    ECurve_NIST_P224_a,  sizeof(ECurve_NIST_P224_a),
    ECurve_NIST_P224_b,  sizeof(ECurve_NIST_P224_b),
    ECurve_NIST_P224_Gx, sizeof(ECurve_NIST_P224_Gx),
    ECurve_NIST_P224_Gy, sizeof(ECurve_NIST_P224_Gy),
    ECurve_NIST_P224_n,  sizeof(ECurve_NIST_P224_n),
    1
};
TestVector_ECC_Curve_t ECC_Curve_NIST_P224 = &ECurve_NIST_P224;


/* NIST P-256 curve (p = 2^224(2^32-1) + 2^192 + 2^96 - 1) */
static const uint8_t ECurve_NIST_P256_p[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};
static const uint8_t ECurve_NIST_P256_a[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFC
};
static const uint8_t ECurve_NIST_P256_b[] =
{
    0x5A, 0xC6, 0x35, 0xD8, 0xAA, 0x3A, 0x93, 0xE7,
    0xB3, 0xEB, 0xBD, 0x55, 0x76, 0x98, 0x86, 0xBC,
    0x65, 0x1D, 0x06, 0xB0, 0xCC, 0x53, 0xB0, 0xF6,
    0x3B, 0xCE, 0x3C, 0x3E, 0x27, 0xD2, 0x60, 0x4B
};
static const uint8_t ECurve_NIST_P256_Gx[] =
{
    0x6B, 0x17, 0xD1, 0xF2, 0xE1, 0x2C, 0x42, 0x47,
    0xF8, 0xBC, 0xE6, 0xE5, 0x63, 0xA4, 0x40, 0xF2,
    0x77, 0x03, 0x7D, 0x81, 0x2D, 0xEB, 0x33, 0xA0,
    0xF4, 0xA1, 0x39, 0x45, 0xD8, 0x98, 0xC2, 0x96
};
static const uint8_t ECurve_NIST_P256_Gy[] =
{
    0x4F, 0xE3, 0x42, 0xE2, 0xFE, 0x1A, 0x7F, 0x9B,
    0x8E, 0xE7, 0xEB, 0x4A, 0x7C, 0x0F, 0x9E, 0x16,
    0x2B, 0xCE, 0x33, 0x57, 0x6B, 0x31, 0x5E, 0xCE,
    0xCB, 0xB6, 0x40, 0x68, 0x37, 0xBF, 0x51, 0xF5
};
static const uint8_t ECurve_NIST_P256_n[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xBC, 0xE6, 0xFA, 0xAD, 0xA7, 0x17, 0x9E, 0x84,
    0xF3, 0xB9, 0xCA, 0xC2, 0xFC, 0x63, 0x25, 0x51
};
const TestVector_ECC_Curve_Rec_t ECurve_NIST_P256 =
{
    TV_ECC_FAMILY_NIST_P,
    256,
    ECurve_NIST_P256_p,  sizeof(ECurve_NIST_P256_p),
    ECurve_NIST_P256_a,  sizeof(ECurve_NIST_P256_a),
    ECurve_NIST_P256_b,  sizeof(ECurve_NIST_P256_b),
    ECurve_NIST_P256_Gx, sizeof(ECurve_NIST_P256_Gx),
    ECurve_NIST_P256_Gy, sizeof(ECurve_NIST_P256_Gy),
    ECurve_NIST_P256_n,  sizeof(ECurve_NIST_P256_n),
    1
};
TestVector_ECC_Curve_t ECC_Curve_NIST_P256 = &ECurve_NIST_P256;


/* NIST P-384 curve (p = 2^384 - 2^128 - 2^96 + 2^32 - 1) */
static const uint8_t ECurve_NIST_P384_p[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE,
    0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF
};
static const uint8_t ECurve_NIST_P384_a[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE,
    0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFC
};
static const uint8_t ECurve_NIST_P384_b[] =
{
    0xB3, 0x31, 0x2F, 0xA7, 0xE2, 0x3E, 0xE7, 0xE4,
    0x98, 0x8E, 0x05, 0x6B, 0xE3, 0xF8, 0x2D, 0x19,
    0x18, 0x1D, 0x9C, 0x6E, 0xFE, 0x81, 0x41, 0x12,
    0x03, 0x14, 0x08, 0x8F, 0x50, 0x13, 0x87, 0x5A,
    0xC6, 0x56, 0x39, 0x8D, 0x8A, 0x2E, 0xD1, 0x9D,
    0x2A, 0x85, 0xC8, 0xED, 0xD3, 0xEC, 0x2A, 0xEF
};
static const uint8_t ECurve_NIST_P384_Gx[] =
{
    0xAA, 0x87, 0xCA, 0x22, 0xBE, 0x8B, 0x05, 0x37,
    0x8E, 0xB1, 0xC7, 0x1E, 0xF3, 0x20, 0xAD, 0x74,
    0x6E, 0x1D, 0x3B, 0x62, 0x8B, 0xA7, 0x9B, 0x98,
    0x59, 0xF7, 0x41, 0xE0, 0x82, 0x54, 0x2A, 0x38,
    0x55, 0x02, 0xF2, 0x5D, 0xBF, 0x55, 0x29, 0x6C,
    0x3A, 0x54, 0x5E, 0x38, 0x72, 0x76, 0x0A, 0xB7
};
static const uint8_t ECurve_NIST_P384_Gy[] =
{
    0x36, 0x17, 0xDE, 0x4A, 0x96, 0x26, 0x2C, 0x6F,
    0x5D, 0x9E, 0x98, 0xBF, 0x92, 0x92, 0xDC, 0x29,
    0xF8, 0xF4, 0x1D, 0xBD, 0x28, 0x9A, 0x14, 0x7C,
    0xE9, 0xDA, 0x31, 0x13, 0xB5, 0xF0, 0xB8, 0xC0,
    0x0A, 0x60, 0xB1, 0xCE, 0x1D, 0x7E, 0x81, 0x9D,
    0x7A, 0x43, 0x1D, 0x7C, 0x90, 0xEA, 0x0E, 0x5F
};
static const uint8_t ECurve_NIST_P384_n[] =
{
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xC7, 0x63, 0x4D, 0x81, 0xF4, 0x37, 0x2D, 0xDF,
    0x58, 0x1A, 0x0D, 0xB2, 0x48, 0xB0, 0xA7, 0x7A,
    0xEC, 0xEC, 0x19, 0x6A, 0xCC, 0xC5, 0x29, 0x73
};
const TestVector_ECC_Curve_Rec_t ECurve_NIST_P384 =
{
    TV_ECC_FAMILY_NIST_P,
    384,
    ECurve_NIST_P384_p,  sizeof(ECurve_NIST_P384_p),
    ECurve_NIST_P384_a,  sizeof(ECurve_NIST_P384_a),
    ECurve_NIST_P384_b,  sizeof(ECurve_NIST_P384_b),
    ECurve_NIST_P384_Gx, sizeof(ECurve_NIST_P384_Gx),
    ECurve_NIST_P384_Gy, sizeof(ECurve_NIST_P384_Gy),
    ECurve_NIST_P384_n,  sizeof(ECurve_NIST_P384_n),
    1
};
TestVector_ECC_Curve_t ECC_Curve_NIST_P384 = &ECurve_NIST_P384;


/* NIST P-521 curve (p = 2^521 - 1) */
static const uint8_t ECurve_NIST_P521_p[] =
{
    0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF
};
static const uint8_t ECurve_NIST_P521_a[] =
{
    0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFC
};
static const uint8_t ECurve_NIST_P521_b[] =
{
    0x00, 0x51, 0x95, 0x3E, 0xB9, 0x61, 0x8E, 0x1C,
    0x9A, 0x1F, 0x92, 0x9A, 0x21, 0xA0, 0xB6, 0x85,
    0x40, 0xEE, 0xA2, 0xDA, 0x72, 0x5B, 0x99, 0xB3,
    0x15, 0xF3, 0xB8, 0xB4, 0x89, 0x91, 0x8E, 0xF1,
    0x09, 0xE1, 0x56, 0x19, 0x39, 0x51, 0xEC, 0x7E,
    0x93, 0x7B, 0x16, 0x52, 0xC0, 0xBD, 0x3B, 0xB1,
    0xBF, 0x07, 0x35, 0x73, 0xDF, 0x88, 0x3D, 0x2C,
    0x34, 0xF1, 0xEF, 0x45, 0x1F, 0xD4, 0x6B, 0x50,
    0x3F, 0x00
};
static const uint8_t ECurve_NIST_P521_Gx[] =
{
    0x00, 0xC6, 0x85, 0x8E, 0x06, 0xB7, 0x04, 0x04,
    0xE9, 0xCD, 0x9E, 0x3E, 0xCB, 0x66, 0x23, 0x95,
    0xB4, 0x42, 0x9C, 0x64, 0x81, 0x39, 0x05, 0x3F,
    0xB5, 0x21, 0xF8, 0x28, 0xAF, 0x60, 0x6B, 0x4D,
    0x3D, 0xBA, 0xA1, 0x4B, 0x5E, 0x77, 0xEF, 0xE7,
    0x59, 0x28, 0xFE, 0x1D, 0xC1, 0x27, 0xA2, 0xFF,
    0xA8, 0xDE, 0x33, 0x48, 0xB3, 0xC1, 0x85, 0x6A,
    0x42, 0x9B, 0xF9, 0x7E, 0x7E, 0x31, 0xC2, 0xE5,
    0xBD, 0x66
};
static const uint8_t ECurve_NIST_P521_Gy[] =
{
    0x01, 0x18, 0x39, 0x29, 0x6A, 0x78, 0x9A, 0x3B,
    0xC0, 0x04, 0x5C, 0x8A, 0x5F, 0xB4, 0x2C, 0x7D,
    0x1B, 0xD9, 0x98, 0xF5, 0x44, 0x49, 0x57, 0x9B,
    0x44, 0x68, 0x17, 0xAF, 0xBD, 0x17, 0x27, 0x3E,
    0x66, 0x2C, 0x97, 0xEE, 0x72, 0x99, 0x5E, 0xF4,
    0x26, 0x40, 0xC5, 0x50, 0xB9, 0x01, 0x3F, 0xAD,
    0x07, 0x61, 0x35, 0x3C, 0x70, 0x86, 0xA2, 0x72,
    0xC2, 0x40, 0x88, 0xBE, 0x94, 0x76, 0x9F, 0xD1,
    0x66, 0x50
};
static const uint8_t ECurve_NIST_P521_n[] =
{
    0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFA, 0x51, 0x86, 0x87, 0x83, 0xBF, 0x2F,
    0x96, 0x6B, 0x7F, 0xCC, 0x01, 0x48, 0xF7, 0x09,
    0xA5, 0xD0, 0x3B, 0xB5, 0xC9, 0xB8, 0x89, 0x9C,
    0x47, 0xAE, 0xBB, 0x6F, 0xB7, 0x1E, 0x91, 0x38,
    0x64, 0x09
};
const TestVector_ECC_Curve_Rec_t ECurve_NIST_P521 =
{
    TV_ECC_FAMILY_NIST_P,
    521,
    ECurve_NIST_P521_p,  sizeof(ECurve_NIST_P521_p),
    ECurve_NIST_P521_a,  sizeof(ECurve_NIST_P521_a),
    ECurve_NIST_P521_b,  sizeof(ECurve_NIST_P521_b),
    ECurve_NIST_P521_Gx, sizeof(ECurve_NIST_P521_Gx),
    ECurve_NIST_P521_Gy, sizeof(ECurve_NIST_P521_Gy),
    ECurve_NIST_P521_n,  sizeof(ECurve_NIST_P521_n),
    1
};
TestVector_ECC_Curve_t ECC_Curve_NIST_P521 = &ECurve_NIST_P521;


#ifdef EIP130_ENABLE_ASYM_ALGO_CURVE25519
/* Curve25519
 * Montgomery elliptic curves are defined by:
 *
 *     b*y^2 = x^3 + ax^2 + x (mod p), where b is 1 and p is prime.
 *
 * See http://cr.yp.to/ecdh.html for Bernstein's papers/talks on this curve.
 * Also see RFC7748 that specifies values for Gx and Gy (called U(P) and V(P)),
 * but note that there is an errata that corrects the Gy (V(P)) value.
 *
 * For Ed25519 in particular with:
 * p  = 2^255 - 19
 * a  = 486662
 * b  = 1
 * n  = 2^252 + 0x14def9dea2f79cd65812631a5cf5d3ed
 * Gx = 9
 * Gy = 0x5f51e65e475f794b1fe122d388b72eb36dc2b28192839e4dd6163a5d81312c14
 * h  = 8
 */
static const uint8_t ECurve_25519_p[] =
{
    0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xED
};
static const uint8_t ECurve_25519_a[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x6D, 0x06
};
static const uint8_t ECurve_25519_b[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
};
static const uint8_t ECurve_25519_Gx[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09
};
static const uint8_t ECurve_25519_Gy[] =
{
    0x5F, 0x51, 0xE6, 0x5E, 0x47, 0x5F, 0x79, 0x4B, 0x1F, 0xE1, 0x22, 0xD3, 0x88, 0xB7, 0x2E, 0xB3,
    0x6D, 0xC2, 0xB2, 0x81, 0x92, 0x83, 0x9E, 0x4D, 0xD6, 0x16, 0x3A, 0x5D, 0x81, 0x31, 0x2C, 0x14
};
static const uint8_t ECurve_25519_n[] =
{
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x14, 0xDE, 0xF9, 0xDE, 0xA2, 0xF7, 0x9C, 0xD6, 0x58, 0x12, 0x63, 0x1A, 0x5C, 0xF5, 0xD3, 0xED
};
const TestVector_ECC_Curve_Rec_t ECurve_25519 =
{
    TV_ECC_FAMILY_MONTGOMERY,
    255,
    ECurve_25519_p,  sizeof(ECurve_25519_p),
    ECurve_25519_a,  sizeof(ECurve_25519_a),
    ECurve_25519_b,  sizeof(ECurve_25519_b),
    ECurve_25519_Gx, sizeof(ECurve_25519_Gx),
    ECurve_25519_Gy, sizeof(ECurve_25519_Gy),
    ECurve_25519_n,  sizeof(ECurve_25519_n),
    8
};
TestVector_ECC_Curve_t ECC_Curve_25519 = &ECurve_25519;
#endif


#ifdef EIP130_ENABLE_ASYM_ALGO_EDDSA
/* CurveEd25519
 * The twisted Edwards curves are defined by:
 *
 *     -x^2 + y^2 = 1 + d*x^2*y^2
 *
 * See ed25519.cr.yp.to/papers.html for Bernstein's papers on this curve/EdDSA
 * system. Also see https://datatracker.ietf.org/doc/draft-irtf-cfrg-eddsa/ for
 * background.
 *
 * Note that the curve parameters are tweaked a little to fit them in the
 * EIP-130 structure.
 *
 * For Ed25519 in particular with:
 * p         = 2^255 - 19
 * a = d     = -121665/121666
 * b = const = 0x0f26edf460a006bbd27b08dc03fc4f7ec5a1d3d14b7d1a82cc6e04aaff457e06
 * n = L     = 2^252 + 0x14def9dea2f79cd65812631a5cf5d3ed
 * Gx        = 0x216936d3cd6e53fec0a4e231fdd6dc5c692cc7609525a7b2c9562d608f25d51a
 * Gy        = 0x6666666666666666666666666666666666666666666666666666666666666658
 * h         = 8
 */
static const uint8_t ECurve_Ed25519_a[] =
{
    0x52, 0x03, 0x6C, 0xEE, 0x2B, 0x6F, 0xFE, 0x73, 0x8C, 0xC7, 0x40, 0x79, 0x77, 0x79, 0xE8, 0x98,
    0x00, 0x70, 0x0A, 0x4D, 0x41, 0x41, 0xD8, 0xAB, 0x75, 0xEB, 0x4D, 0xCA, 0x13, 0x59, 0x78, 0xA3
};
static const uint8_t ECurve_Ed25519_b[] =
{
    0x0F, 0x26, 0xED, 0xF4, 0x60, 0xA0, 0x06, 0xBB, 0xD2, 0x7B, 0x08, 0xDC, 0x03, 0xFC, 0x4F, 0x7E,
    0xC5, 0xA1, 0xD3, 0xD1, 0x4B, 0x7D, 0x1A, 0x82, 0xCC, 0x6E, 0x04, 0xAA, 0xFF, 0x45, 0x7E, 0x06
};
static const uint8_t ECurve_Ed25519_Gx[] =
{
    0x21, 0x69, 0x36, 0xD3, 0xCD, 0x6E, 0x53, 0xFE, 0xC0, 0xA4, 0xE2, 0x31, 0xFD, 0xD6, 0xDC, 0x5C,
    0x69, 0x2C, 0xC7, 0x60, 0x95, 0x25, 0xA7, 0xB2, 0xC9, 0x56, 0x2D, 0x60, 0x8F, 0x25, 0xD5, 0x1A
};
static const uint8_t ECurve_Ed25519_Gy[] =
{
    0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66,
    0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x58
};
const TestVector_ECC_Curve_Rec_t ECurve_Ed25519 =
{
    TV_ECC_FAMILY_TWISTED_EDWARDS,
    255,
    ECurve_25519_p,    sizeof(ECurve_25519_p),
    ECurve_Ed25519_a,  sizeof(ECurve_Ed25519_a),
    ECurve_Ed25519_b,  sizeof(ECurve_Ed25519_b),
    ECurve_Ed25519_Gx, sizeof(ECurve_Ed25519_Gx),
    ECurve_Ed25519_Gy, sizeof(ECurve_Ed25519_Gy),
    ECurve_25519_n,    sizeof(ECurve_25519_n),
    8
};
TestVector_ECC_Curve_t ECC_Curve_Ed25519 = &ECurve_Ed25519;
#endif


#ifdef EIP130_ENABLE_ASYM_ALGO_SM2
/* SM2 ECC curve */
static const uint8_t ECurve_SM2_Fp_256_p [] =
{
    0x85, 0x42, 0xD6, 0x9E, 0x4C, 0x04, 0x4F, 0x18,
    0xE8, 0xB9, 0x24, 0x35, 0xBF, 0x6F, 0xF7, 0xDE,
    0x45, 0x72, 0x83, 0x91, 0x5C, 0x45, 0x51, 0x7D,
    0x72, 0x2E, 0xDB, 0x8B, 0x08, 0xF1, 0xDF, 0xC3
};
static const uint8_t ECurve_SM2_Fp_256_a [] =
{
    0x78, 0x79, 0x68, 0xB4, 0xFA, 0x32, 0xC3, 0xFD,
    0x24, 0x17, 0x84, 0x2E, 0x73, 0xBB, 0xFE, 0xFF,
    0x2F, 0x3C, 0x84, 0x8B, 0x68, 0x31, 0xD7, 0xE0,
    0xEC, 0x65, 0x22, 0x8B, 0x39, 0x37, 0xE4, 0x98
};
static const uint8_t ECurve_SM2_Fp_256_b [] =
{
    0x63, 0xE4, 0xC6, 0xD3, 0xB2, 0x3B, 0x0C, 0x84,
    0x9C, 0xF8, 0x42, 0x41, 0x48, 0x4B, 0xFE, 0x48,
    0xF6, 0x1D, 0x59, 0xA5, 0xB1, 0x6B, 0xA0, 0x6E,
    0x6E, 0x12, 0xD1, 0xDA, 0x27, 0xC5, 0x24, 0x9A
};
static const uint8_t ECurve_SM2_Fp_256_Gx [] =
{
    0x42, 0x1D, 0xEB, 0xD6, 0x1B, 0x62, 0xEA, 0xB6,
    0x74, 0x64, 0x34, 0xEB, 0xC3, 0xCC, 0x31, 0x5E,
    0x32, 0x22, 0x0B, 0x3B, 0xAD, 0xD5, 0x0B, 0xDC,
    0x4C, 0x4E, 0x6C, 0x14, 0x7F, 0xED, 0xD4, 0x3D
};
static const uint8_t ECurve_SM2_Fp_256_Gy [] =
{
    0x06, 0x80, 0x51, 0x2B, 0xCB, 0xB4, 0x2C, 0x07,
    0xD4, 0x73, 0x49, 0xD2, 0x15, 0x3B, 0x70, 0xC4,
    0xE5, 0xD7, 0xFD, 0xFC, 0xBF, 0xA3, 0x6E, 0xA1,
    0xA8, 0x58, 0x41, 0xB9, 0xE4, 0x6E, 0x09, 0xA2
};
static const uint8_t ECurve_SM2_Fp_256_n [] =
{
    0x85, 0x42, 0xD6, 0x9E, 0x4C, 0x04, 0x4F, 0x18,
    0xE8, 0xB9, 0x24, 0x35, 0xBF, 0x6F, 0xF7, 0xDD,
    0x29, 0x77, 0x20, 0x63, 0x04, 0x85, 0x62, 0x8D,
    0x5A, 0xE7, 0x4E, 0xE7, 0xC3, 0x2E, 0x79, 0xB7
};
const TestVector_ECC_Curve_Rec_t ECurve_SM2_Fp_256 =
{
    TV_ECC_FAMILY_SM2,
    256,
    ECurve_SM2_Fp_256_p,  sizeof(ECurve_SM2_Fp_256_p),
    ECurve_SM2_Fp_256_a,  sizeof(ECurve_SM2_Fp_256_a),
    ECurve_SM2_Fp_256_b,  sizeof(ECurve_SM2_Fp_256_b),
    ECurve_SM2_Fp_256_Gx, sizeof(ECurve_SM2_Fp_256_Gx),
    ECurve_SM2_Fp_256_Gy, sizeof(ECurve_SM2_Fp_256_Gy),
    ECurve_SM2_Fp_256_n,  sizeof(ECurve_SM2_Fp_256_n),
    1
};
TestVector_ECC_Curve_t ECC_Curve_SM2_Fp_256 = &ECurve_SM2_Fp_256;


static const uint8_t ECurve_SM2_Fp_192_p[] = {
    0xBD, 0xB6, 0xF4, 0xFE, 0x3E, 0x8B, 0x1D, 0x9E,
    0x0D, 0xA8, 0xC0, 0xD4, 0x6F, 0x4C, 0x31, 0x8C,
    0xEF, 0xE4, 0xAF, 0xE3, 0xB6, 0xB8, 0x55, 0x1F
};
static const uint8_t ECurve_SM2_Fp_192_a[] = {
    0xBB, 0x8E, 0x5E, 0x8F, 0xBC, 0x11, 0x5E, 0x13,
    0x9F, 0xE6, 0xA8, 0x14, 0xFE, 0x48, 0xAA, 0xA6,
    0xF0, 0xAD, 0xA1, 0xAA, 0x5D, 0xF9, 0x19, 0x85
};
static const uint8_t ECurve_SM2_Fp_192_b[] = {
    0x18, 0x54, 0xBE, 0xBD, 0xC3, 0x1B, 0x21, 0xB7,
    0xAE, 0xFC, 0x80, 0xAB, 0x0E, 0xCD, 0x10, 0xD5,
    0xB1, 0xB3, 0x30, 0x8E, 0x6D, 0xBF, 0x11, 0xC1
};
static const uint8_t ECurve_SM2_Fp_192_Gx[] = {
    0x4A, 0xD5, 0xF7, 0x04, 0x8D, 0xE7, 0x09, 0xAD,
    0x51, 0x23, 0x6D, 0xE6, 0x5E, 0x4D, 0x4B, 0x48,
    0x2C, 0x83, 0x6D, 0xC6, 0xE4, 0x10, 0x66, 0x40
};
static const uint8_t ECurve_SM2_Fp_192_Gy[] = {
    0x02, 0xBB, 0x3A, 0x02, 0xD4, 0xAA, 0xAD, 0xAC,
    0xAE, 0x24, 0x81, 0x7A, 0x4C, 0xA3, 0xA1, 0xB0,
    0x14, 0xB5, 0x27, 0x04, 0x32, 0xDB, 0x27, 0xD2
};
static const uint8_t ECurve_SM2_Fp_192_n[] = {
    0xBD, 0xB6, 0xF4, 0xFE, 0x3E, 0x8B, 0x1D, 0x9E,
    0x0D, 0xA8, 0xC0, 0xD4, 0x0F, 0xC9, 0x62, 0x19,
    0x5D, 0xFA, 0xE7, 0x6F, 0x56, 0x56, 0x46, 0x77
};
const TestVector_ECC_Curve_Rec_t ECurve_SM2_Fp_192 =
{
    TV_ECC_FAMILY_SM2,
    192,
    ECurve_SM2_Fp_192_p,  sizeof(ECurve_SM2_Fp_192_p),
    ECurve_SM2_Fp_192_a,  sizeof(ECurve_SM2_Fp_192_a),
    ECurve_SM2_Fp_192_b,  sizeof(ECurve_SM2_Fp_192_b),
    ECurve_SM2_Fp_192_Gx, sizeof(ECurve_SM2_Fp_192_Gx),
    ECurve_SM2_Fp_192_Gy, sizeof(ECurve_SM2_Fp_192_Gy),
    ECurve_SM2_Fp_192_n,  sizeof(ECurve_SM2_Fp_192_n),
    1
};
TestVector_ECC_Curve_t ECC_Curve_SM2_Fp_192 = &ECurve_SM2_Fp_192;
#endif


/* end of file testvectors_ecc_curves.c */
