/*
 * Copyright (c) 2020-2024, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *   its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* Standard Includes */
#include <ti/drivers/ITM.h>
#include <ti/drivers/dpl/HwiP.h>
#include <ti/drivers/itm/hw_cpu_dwt.h>
#include <ti/drivers/itm/hw_cpu_scs.h>
#include <ti/drivers/itm/hw_cpu_tpiu.h>
#include <ti/drivers/itm/hw_cpu_itm.h>

#include <ti/devices/DeviceFamily.h>
#include DeviceFamily_constructPath(inc/hw_types.h)

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

/* Declare the ITM_Object type */
typedef struct
{
    uint8_t numberOfClients;   /* Number of ITM_open calls */
    uint8_t numDwtComparators; /* Number of DWT comparators available */
    uint32_t dwtCtrlRegister;  /* Shadow of the DWT CTRL register */
    uint32_t fullFIFOInCycles; /* Number of cycles needed to drain TPIU FIFO */
} ITM_Object;

/* Create an instance of the object and initialize it */
static ITM_Object object = {
    .numberOfClients   = 0,
    .numDwtComparators = 0,
    .dwtCtrlRegister   = 0,
    .fullFIFOInCycles  = 0,
};

/* This enables the device specific hwAttrs to be case as the generic ones
 * The hwAttrs will be generated by syscfg
 */
extern void *itmHwAttrs;

/**
 * This function initializes the ITM hardware.
 */
static void ITM_initHw(void);

/**
 * This function will setup the mux to route the TPIU output to the pin(s) that
 * the user has selected in the hwAttrs.
 *
 */
extern bool ITM_applyPinMux(void);
/**
 * This function will remove any pin mux settings applied by ITM_applyPinMux.
 */
extern void ITM_clearPinMux(void);

/*
 *  ======== ITM_open ========
 */
bool ITM_open(void)
{
    ITM_HWAttrs *hwAttrs  = (ITM_HWAttrs *)itmHwAttrs;
    uint32_t tpiuFifoSize = 0;
    bool pinMuxStatus     = false;
    uintptr_t hwiKey;

    hwiKey = HwiP_disable();

    if (object.numberOfClients == 0)
    {
        /* Mux out the pins that will be used by the TPIU */
        pinMuxStatus = ITM_applyPinMux();

        /* If we couldn't acquire the pins, give up now */
        if (pinMuxStatus == false)
        {
            HwiP_restore(hwiKey);
            return (bool)false;
        }

        /* Initialize ITM Hardware */
        ITM_initHw();

        /* Store the number of DWT comparators */
        object.numDwtComparators = (HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) & CPU_DWT_CTRL_NUMCOMP_M) >>
                                    CPU_DWT_CTRL_NUMCOMP_S;

        /* Store the FIFO size of the TPIU, this will be used for flush */
        tpiuFifoSize = (HWREG(ITM_TPIU_BASE_ADDR + CPU_TPIU_O_DEVID) & CPU_TPIU_DEVID_FIFO_SIZE_M) >>
                        CPU_TPIU_DEVID_FIFO_SIZE_S;

        /* FIFO size stored in the register as 2^FIFO_SIZE in bytes
         * 1. Calculate the power of 2 using left shifts.
         * 2. Subtract 2 from the number of positions to shift to convert bytes
         *    into 32-bit words
         */
        tpiuFifoSize = (1 << (tpiuFifoSize - 2));

        /* Then multiply this by the length of the FIFO */
        object.fullFIFOInCycles = tpiuFifoSize * (hwAttrs->fullPacketInCycles);
    }

    /* Increment is open counter to allow multiple open calls */
    object.numberOfClients++;

    HwiP_restore(hwiKey);

    return (bool)true;
}
/*
 *  ======== ITM_close ========
 */
void ITM_close(void)
{
    uintptr_t hwiKey;
    hwiKey = HwiP_disable();

    /* Protect against unbalanced number of open/close calls */
    if (object.numberOfClients <= 0)
    {
        HwiP_restore(hwiKey);
        return;
    }

    /* If this is the last close call, clean up */
    if (object.numberOfClients == 1)
    {
        ITM_disableExceptionTrace();
        ITM_disablePCAndEventSampling();

        /* Disable DWT stimulus. This routes DWT packets to the TPIU */
        HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) &= ~(CPU_ITM_TCR_DWTENA_M);

        /* Disable cycle counter, required to ensure proper flush */
        HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) &= ~CPU_DWT_CTRL_CYCCNTENA_M;
        HWREG(ITM_SCS_BASE_ADDR  + CPU_SCS_O_DEMCR) &= (~CPU_SCS_DEMCR_TRCENA_M);
        HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) &= ~((CPU_DWT_CTRL_PCSAMPLEENA_M) | (CPU_DWT_CTRL_EXCTRCENA_M));
        HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) &= ~CPU_ITM_TCR_ITMENA;

        /* Flush out any remaining packets to be sent */
        ITM_flush();

        /* Unmux pins that were in use by the TPIU. This is device specific
         * and will reside in the device specific file
         */
        ITM_clearPinMux();
    }

    /* Decrement is open counter */
    object.numberOfClients--;

    HwiP_restore(hwiKey);
}

/*
 *  ======== ITM_send32Atomic ========
 */
void ITM_send32Atomic(uint8_t port, uint32_t value)
{
    uint32_t key;
    key = HwiP_disable();
    ITM_send32Polling(port, value);
    HwiP_restore(key);
}

/*
 *  ======== ITM_send16Atomic ========
 */
void ITM_send16Atomic(uint8_t port, uint16_t value)
{
    uint32_t key;
    key = HwiP_disable();
    ITM_send16Polling(port, value);
    HwiP_restore(key);
}

/*
 *  ======== ITM_send8Atomic ========
 */
void ITM_send8Atomic(uint8_t port, uint8_t value)
{
    uint32_t key;
    key = HwiP_disable();
    ITM_send8Polling(port, value);
    HwiP_restore(key);
}

/*
 *  ======== ITM_sendBufferAtomic ========
 */
void ITM_sendBufferAtomic(const uint8_t port, const char *msg, size_t length)
{
    uint32_t key;
    key = HwiP_disable();
    /* Unroll the sending of the data to use the optimal port size
     * This has a slightly higher flash footprint, but has better performance
     * for large buffers. Users who are concerned about flash can consider
     * replacing this with a single for loop. Ensure msg is word-aligned before
     * proceeding.
     */
    while ((length > 0) && ((uint32_t)msg & 0x03))
    {
        ITM_send8Polling(port, *msg++);
        length--;
    }
    while (length > 3)
    {
        /* The cast from uint8_t* to uint32_t* below is safe, due to the loop above */
        ITM_send32Polling(port, *((uint32_t *)msg));
        length -= 4;
        msg += 4;
    }
    while (length > 1)
    {
        /* The cast from uint8_t* to uint16_t* below is safe, due to the loop above */
        ITM_send16Polling(port, *((uint16_t *)msg));
        length -= 2;
        msg += 2;
    }
    if (length > 0)
    {
        ITM_send8Polling(port, *msg);
    }

    HwiP_restore(key);
}

/*
 *  ======== ITM_enableExceptionTrace ========
 */
void ITM_enableExceptionTrace(void)
{
    /* Enable interrupt event tracing by the DWT */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) |= CPU_DWT_CTRL_EXCTRCENA_M;
}

/*
 *  ======== ITM_disableExceptionTrace ========
 */
void ITM_disableExceptionTrace(void)
{
    /* Disable interrupt event tracing by the DWT */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) &= ~(CPU_DWT_CTRL_EXCTRCENA_M);
}

/*
 *  ======== ITM_enablePCSampling ========
 */
void ITM_enablePCSampling(bool prescale1024, uint8_t postReset)
{
    /* Disable before setup */
    ITM_disablePCAndEventSampling();

    /* Setup sampling interval by setting reload down counter value */
    if (prescale1024 == (bool)true)
    {
        /* Tap at bit 10 of system clock, this results in a prescale by 1024 */
        HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) |= CPU_DWT_CTRL_CYCTAP_BIT10;
    }

    /* The postreset value is the reload value for the counter */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) |= ((postReset & 0x0F) << CPU_DWT_CTRL_POSTPRESET_S);

    /* Enable DWT packets in the ITM and cycle counting */
    HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) |= CPU_ITM_TCR_DWTENA_M;

    /* Enable PC sampling event */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) |= CPU_DWT_CTRL_PCSAMPLEENA_M;
}

/*
 *  ======== ITM_enableEventCounter ========
 */
void ITM_enableEventCounter(bool prescale1024, uint8_t postReset)
{
    /* Clear the PC Sampling and Cycle Event bits */
    ITM_disablePCAndEventSampling();
    /* Setup sampling interval by setting reload down counter value */
    if (prescale1024 == (bool)true)
    {
        /* Tap at bit 10 of system clock, this results in a prescale by 1024 */
        HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) |= CPU_DWT_CTRL_CYCTAP_BIT10;
    }

    /* The postreset value is the reload value for the counter */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) |= ((postReset & 0x0F) << CPU_DWT_CTRL_POSTPRESET_S);

    HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) |= CPU_ITM_TCR_DWTENA_M;

    /* Enable Cycle Count event */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) |= CPU_DWT_CTRL_CYCEVTENA_M;
}

/*
 *  ======== ITM_disablePCAndEventSampling ========
 */
void ITM_disablePCAndEventSampling(void)
{
    /* Clear the PC Sampling and Cycle Event bits */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) &= ~(CPU_DWT_CTRL_PCSAMPLEENA_M | CPU_DWT_CTRL_CYCEVTENA_M);
}

/*
 *  ======== ITM_enableWatchpoint ========
 */
bool ITM_enableWatchpoint(ITM_WatchpointAction function, const uintptr_t address)
{

    uint8_t dwtIndex  = 0;
    bool dwtAvailable = false;
    for (dwtIndex = 0; dwtIndex < object.numDwtComparators; dwtIndex++)
    {
        uint32_t offset = 16 * dwtIndex;
        if (0 == (HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_FUNCTION0 + offset) & 0x7FFFFFF))
        {
            HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_COMP0 + offset)     = address;
            HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_MASK0 + offset)     = 0;
            HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_FUNCTION0 + offset) = function;
            dwtAvailable                                            = true;
        }
    }
    return dwtAvailable;
}

/*
 *  ======== ITM_enableTiming ========
 */
void ITM_enableTimestamps(ITM_TimeStampPrescaler tsPrescale, bool asyncMode)
{
    /* Set timestamp prescale value and enable timestamp packet generation */
    HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) |= ((tsPrescale << CPU_ITM_TCR_TSPRESCALE_S) & CPU_ITM_TCR_TSPRESCALE_M);
    HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) |= CPU_ITM_TCR_TSENA_M;

    if (asyncMode == (bool)true)
    {
        /* Asynchronous vs synchronous mode is controlled by the SWOENA bit */
        HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) |= CPU_ITM_TCR_SWOENA_M;
    }
}

/*
 *  ======== ITM_enableSyncPackets ========
 */
void ITM_enableSyncPackets(ITM_SyncPacketRate syncPacketRate)
{
    /* Clear the synchronous packet rate field */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) &= ~(CPU_DWT_CTRL_SYNCTAP_M);
    /* Set sync packet rate */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) |= ((syncPacketRate << CPU_DWT_CTRL_SYNCTAP_S) & CPU_DWT_CTRL_SYNCTAP_M);
    /* Enable sync packet generation */
    HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) |= CPU_ITM_TCR_SYNCENA_M;
}

/*
 *  ======== ITM_flush ========
 */
void __attribute__((weak)) ITM_flush(void)
{
    /* This function is intentionally blank to save overhead in the power
     * policy. When ITM is enabled, the function will be strongly defined by
     * syscfg
     */

    /* When populated by syscfg, this function will execute:
     *  1. ITM_commonFlush() - This is the common implementation
     *  2. Device specific flush e.g. ITMCC26XX_commonFlush
     */
}

/*
 *  ======== ITM_restore ========
 */
void __attribute__((weak)) ITM_restore(void)
{
    /* This function is intentionally blank to save overhead in the power
     * policy. When ITM is enabled, the function will be strongly defined by
     * syscfg
     */
}

/*
 *  ======== ITM_commonFlush ========
 */
void ITM_commonFlush(void)
{
    if (object.numberOfClients != 0)
    {
        /* Cache the DWT CTRL register so it can be restored on wake */
        object.dwtCtrlRegister = HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL);

        /* Disable PC sampling so the buffer doesn't fill up */
        ITM_disablePCAndEventSampling();

        /* Write dummy value to "flush" the FIFO */
        uint32_t dummy = 0xAAAAAAAA;
        ITM_send32Atomic(0, dummy);

        /* Wait until the ITM events has drained */
        while (HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) & (1 << 23)) {}

        /* Flush TPIU FIFO
         * Now, the ITM is flushed, but the TPIU also has a FIFO to be empty
         * There is no flush bit available in the TPIU register set,
         * so here we calculate the time needed to drain the FIFO using
         * using its size and baudrate
         */
        uint32_t ticksNow = HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CYCCNT);
        int32_t numTicks  = 0;
        do
        {
            int32_t tmp = HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CYCCNT) - ticksNow;

            /* Consider the case where the DWT timer has wrapped */
            if (tmp < 0)
            {
                tmp += 0xFFFFFFFF;
            }

            /* Increase the accumulated ticks */
            numTicks += tmp;

            /* Update the current time */
            ticksNow = HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CYCCNT);
        } while ((numTicks < object.fullFIFOInCycles));

        /* Disable ITM */
        HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) &= ~CPU_ITM_TCR_ITMENA;
    }
}

/*
 *  ======== ITM_commonRestore ========
 */
void ITM_commonRestore(void)
{
    /* Only if currently used ... */
    if (object.numberOfClients != 0)
    {

#if (DeviceFamily_PARENT == DeviceFamily_PARENT_CC27XX)
        /* For CC27XX devices, the CPU sub-system is reset in standby, and the
         * ITM must thus be re-initialized.
         */
        ITM_initHw();
#endif

        /* Enable ITM */
        HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) |= CPU_ITM_TCR_ITMENA;

        /* Restore the DWT CTRL register */
        HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) |= object.dwtCtrlRegister;
    }
}

/*
 *  ======== ITM_initHw ========
 */
static void ITM_initHw(void)
{
    ITM_HWAttrs *hwAttrs = (ITM_HWAttrs *)itmHwAttrs;

    /* Disable ITM and trace hardware as we're about to set it up again */
    HWREG(ITM_SCS_BASE_ADDR  + CPU_SCS_O_DEMCR) &= (~CPU_SCS_DEMCR_TRCENA_M);
    HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) = 0x00000000;

    /* Enable trace */
    HWREG(ITM_SCS_BASE_ADDR  + CPU_SCS_O_DEMCR) |= CPU_SCS_DEMCR_TRCENA_M;

    /* Unlock and Setup TPIU for SWO UART mode */
    HWREG(ITM_TPIU_BASE_ADDR + CPU_TPIU_O_LAR)   = ITM_LAR_UNLOCK;
    HWREG(ITM_TPIU_BASE_ADDR + CPU_TPIU_O_SPPR)  = hwAttrs->format;
    HWREG(ITM_TPIU_BASE_ADDR + CPU_TPIU_O_CSPSR) = CPU_TPIU_CSPSR_ONE;

    /* Unlock and enable all ITM stimulus ports with default settings */
    HWREG(ITM_BASE_ADDR + CPU_ITM_O_LAR) = ITM_LAR_UNLOCK;
    HWREG(ITM_BASE_ADDR + CPU_ITM_O_TER) = hwAttrs->traceEnable;
    HWREG(ITM_BASE_ADDR + CPU_ITM_O_TPR) = 0x0000000F;

    /* Program prescaler value which is calculated from SysConfig */
    HWREG(ITM_TPIU_BASE_ADDR + CPU_TPIU_O_ACPR) = hwAttrs->tpiuPrescaler;

    /* Disable formatter */
    HWREG(ITM_TPIU_BASE_ADDR + CPU_TPIU_O_FFCR) = 0;

    /* Unlock DWT */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_LAR) = ITM_LAR_UNLOCK;

    /* Enable cycle counter, required to ensure proper flush */
    HWREG(ITM_DWT_BASE_ADDR  + CPU_DWT_O_CTRL) |= CPU_DWT_CTRL_CYCCNTENA_M;

    /* Enable ITM */
    HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) |= CPU_ITM_TCR_ITMENA;

    /* Enable DWT stimulus. This routes DWT packets to the TPIU */
    HWREG(ITM_BASE_ADDR + CPU_ITM_O_TCR) |= CPU_ITM_TCR_DWTENA_M;
}