/*
 * Copyright (c) 2020-2022, Arm Limited. All rights reserved.
 * Copyright (c) 2022 Cypress Semiconductor Corporation (an Infineon
 * company) or an affiliate of Cypress Semiconductor Corporation. All rights
 * reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 */

#include <arm_cmse.h>
#include "cmsis.h"
#include "mpu_armv8m_drv.h"
#include "region.h"
#include "spu.h"
#include "target_cfg.h"
#include "tfm_hal_isolation.h"
#include "mmio_defs.h"
#include "array.h"
#include "load/spm_load_api.h"

#define PROT_BOUNDARY_VAL \
    ((1U << HANDLE_ATTR_PRIV_POS) & HANDLE_ATTR_PRIV_MASK)

REGION_DECLARE(Image$$, TFM_UNPRIV_CODE_START, $$RO$$Base);
REGION_DECLARE(Image$$, TFM_UNPRIV_CODE_END, $$RO$$Limit);
REGION_DECLARE(Image$$, TFM_APP_CODE_START, $$Base);
REGION_DECLARE(Image$$, TFM_APP_CODE_END, $$Base);
REGION_DECLARE(Image$$, TFM_APP_RW_STACK_START, $$Base);
REGION_DECLARE(Image$$, TFM_APP_RW_STACK_END, $$Base);

#ifdef CONFIG_TFM_PARTITION_META
REGION_DECLARE(Image$$, TFM_SP_META_PTR, $$ZI$$Base);
REGION_DECLARE(Image$$, TFM_SP_META_PTR, $$ZI$$Limit);
#endif /* CONFIG_TFM_PARTITION_META */

/* Get address of memory regions to configure MPU */
extern const struct memory_region_limits memory_regions;

struct mpu_armv8m_dev_t dev_mpu_s = { MPU_BASE };

// We assume we are the only consumer of MPU regions and we use this
// variable to keep track of what the next available region is.
static uint32_t n_configured_regions = 0;
enum tfm_hal_status_t mpu_init_cfg(void);

enum tfm_hal_status_t tfm_hal_set_up_static_boundaries(
                                            uintptr_t *p_spm_boundary)
{
    /* Set up isolation boundaries between SPE and NSPE */
    sau_and_idau_cfg();

    if (spu_init_cfg() != TFM_PLAT_ERR_SUCCESS) {
        return TFM_HAL_ERROR_GENERIC;
    }

    if (spu_periph_init_cfg() != TFM_PLAT_ERR_SUCCESS) {
        return TFM_HAL_ERROR_GENERIC;
    }

    if (mpu_init_cfg() != TFM_HAL_SUCCESS) {
        return TFM_HAL_ERROR_GENERIC;
    }

    *p_spm_boundary = (uintptr_t)PROT_BOUNDARY_VAL;

    return TFM_HAL_SUCCESS;
}

enum tfm_hal_status_t
tfm_hal_bind_boundary(const struct partition_load_info_t *p_ldinf,
                        uintptr_t *p_boundary)
{
    if (!p_ldinf || !p_boundary) {
        return TFM_HAL_ERROR_GENERIC;
    }

    bool privileged;
    bool ns_agent;
    uint32_t partition_attrs = 0;

#if TFM_ISOLATION_LEVEL == 1
    privileged = true;
#else
    privileged = IS_PSA_ROT(p_ldinf);
#endif

    ns_agent = IS_NS_AGENT(p_ldinf);
    partition_attrs = ((uint32_t)privileged << HANDLE_ATTR_PRIV_POS) &
                    HANDLE_ATTR_PRIV_MASK;
    partition_attrs |= ((uint32_t)ns_agent << HANDLE_ATTR_NS_POS) &
                        HANDLE_ATTR_NS_MASK;
    *p_boundary = (uintptr_t)partition_attrs;

    for (uint32_t i = 0; i < p_ldinf->nassets; i++) {
        const struct asset_desc_t *p_asset =
                LOAD_INFO_ASSET(p_ldinf);

        if (!(p_asset[i].attr & ASSET_ATTR_NAMED_MMIO)) {
            // Skip numbered MMIO. NB: Need to add validation if it
            // becomes supported. Should we return an error instead?
            continue;
        }

        bool found = false;
        for (uint32_t j = 0; j < ARRAY_SIZE(partition_named_mmio_list); j++) {
            if (partition_named_mmio_list[j] == p_asset[i].dev.dev_ref) {
                found = true;
                break;
            }
        }

        if (!found) {
            /* The MMIO asset is not in the allowed list of platform. */
            return TFM_HAL_ERROR_GENERIC;
        }

        /* Assume PPC & MPC settings are required even under level 1 */
        struct platform_data_t *plat_data_ptr =
                (struct platform_data_t *)p_asset[i].dev.dev_ref;

        if (plat_data_ptr->periph_start == 0) {
            // Should we return an error instead?
            continue;
        }

        spu_periph_configure_to_secure(plat_data_ptr->periph_start);

        /*
         * Static boundaries are set. Set up MPU region for MMIO.
         * Setup regions for unprivileged assets only.
         */
        if (!privileged) {
            struct mpu_armv8m_region_cfg_t localcfg;

            localcfg.region_nr = n_configured_regions++;

            localcfg.region_base = plat_data_ptr->periph_start;
            localcfg.region_limit = plat_data_ptr->periph_limit;
            localcfg.region_attridx = MPU_ARMV8M_MAIR_ATTR_DEVICE_IDX;
            localcfg.attr_access = MPU_ARMV8M_AP_RW_PRIV_UNPRIV;
            localcfg.attr_sh = MPU_ARMV8M_SH_NONE;
            localcfg.attr_exec = MPU_ARMV8M_XN_EXEC_NEVER;

            enum mpu_armv8m_error_t err =
                    mpu_armv8m_region_enable(&dev_mpu_s, &localcfg);

            if (err != MPU_ARMV8M_OK) {
                return TFM_HAL_ERROR_GENERIC;
            }
        }
    }

    return TFM_HAL_SUCCESS;
}

enum tfm_hal_status_t
tfm_hal_activate_boundary(const struct partition_load_info_t *p_ldinf,
                          uintptr_t boundary)
{
    /* Privileged level is required to be set always */
    CONTROL_Type ctrl;
    ctrl.w = __get_CONTROL();

    ctrl.b.nPRIV = ((uint32_t)boundary & HANDLE_ATTR_PRIV_MASK) ? 0 : 1;

    __set_CONTROL(ctrl.w);

    return TFM_HAL_SUCCESS;
}

#if !defined(__SAUREGION_PRESENT) || (__SAUREGION_PRESENT == 0)
static bool accessible_to_region(const void *p, size_t s, int flags)
{
    cmse_address_info_t tt_base = cmse_TT((void *)p);
    cmse_address_info_t tt_last = cmse_TT((void *)((uint32_t)p + s - 1));

    uint32_t base_spu_id = tt_base.flags.idau_region;
    uint32_t last_spu_id = tt_last.flags.idau_region;

    size_t size;
    uint32_t p_start = (uint32_t)p;
    int i;

    if ((base_spu_id >= spu_regions_flash_get_start_id()) &&
        (last_spu_id <= spu_regions_flash_get_last_id())) {

        size = spu_regions_flash_get_last_address_in_region(base_spu_id) + 1
                                                                      - p_start;

        if (cmse_check_address_range((void *)p_start, size, flags) == 0) {
            return false;
        }

        for (i = base_spu_id + 1; i < last_spu_id; i++) {
            p_start = spu_regions_flash_get_base_address_in_region(i);
            if (cmse_check_address_range((void *)p_start,
                spu_regions_flash_get_region_size(), flags) == 0) {
                return false;
            }
        }

        p_start = spu_regions_flash_get_base_address_in_region(last_spu_id);
        size = (uint32_t)p + s - p_start;
        if (cmse_check_address_range((void *)p_start, size, flags) == 0) {
            return false;
        }


    } else if ((base_spu_id >= spu_regions_sram_get_start_id()) &&
        (last_spu_id <= spu_regions_sram_get_last_id())) {

        size = spu_regions_sram_get_last_address_in_region(base_spu_id) + 1
                                                                      - p_start;
        if (cmse_check_address_range((void *)p_start, size, flags) == 0) {
            return false;
        }

        for (i = base_spu_id + 1; i < last_spu_id; i++) {
            p_start = spu_regions_sram_get_base_address_in_region(i);
            if (cmse_check_address_range((void *)p_start,
                spu_regions_sram_get_region_size(), flags) == 0) {
                return false;
            }
        }

        p_start = spu_regions_sram_get_base_address_in_region(last_spu_id);
        size = (uint32_t)p + s - p_start;
        if (cmse_check_address_range((void *)p_start, size, flags) == 0) {
            return false;
        }
    } else {
        return false;
    }

    return true;
}
#endif /* !defined(__SAUREGION_PRESENT) || (__SAUREGION_PRESENT == 0) */

enum tfm_hal_status_t tfm_hal_memory_check(uintptr_t boundary, uintptr_t base,
                                           size_t size, uint32_t access_type)
{
    int flags = 0;
    int32_t range_access_allowed_by_mpu;

    /* If size is zero, this indicates an empty buffer, we can ignore base. */
    if (size == 0) {
        return TFM_HAL_SUCCESS;
    }

    if (!((uint32_t)boundary & HANDLE_ATTR_PRIV_MASK)) {
        flags |= CMSE_MPU_UNPRIV;
    }

    if ((uint32_t)boundary & HANDLE_ATTR_NS_MASK) {
        CONTROL_Type ctrl;
        ctrl.w = __TZ_get_CONTROL_NS();
        if (ctrl.b.nPRIV == 1) {
            flags |= CMSE_MPU_UNPRIV;
        } else {
            flags &= ~CMSE_MPU_UNPRIV;
        }
        flags |= CMSE_NONSECURE;
    }

    if ((access_type & TFM_HAL_ACCESS_READWRITE) == TFM_HAL_ACCESS_READWRITE) {
        flags |= CMSE_MPU_READWRITE;
    } else if (access_type & TFM_HAL_ACCESS_READABLE) {
        flags |= CMSE_MPU_READ;
    } else {
        return TFM_HAL_ERROR_INVALID_INPUT;
    }

    /* Use the TT instruction to check access to the partition's regions*/
    range_access_allowed_by_mpu =
            cmse_check_address_range((void *)base, size, flags) != NULL;

#if !defined(__SAUREGION_PRESENT) || (__SAUREGION_PRESENT == 0)
    if (!range_access_allowed_by_mpu) {
        /*
         * Verification failure may be due to address range crossing
         * one or multiple IDAU boundaries. In this case request a
         * platform-specific check for access permissions.
         */
        cmse_address_info_t addr_info_base = cmse_TT((void *)base);
        cmse_address_info_t addr_info_last =
                cmse_TT((void *)((uint32_t)base + size - 1));

        if ((addr_info_base.flags.idau_region_valid != 0) &&
            (addr_info_last.flags.idau_region_valid != 0) &&
            (addr_info_base.flags.idau_region !=
             addr_info_last.flags.idau_region)) {
            range_access_allowed_by_mpu =
                                accessible_to_region((void *)base, size, flags);
        }
    }
#endif

    if (range_access_allowed_by_mpu) {
        return TFM_HAL_SUCCESS;
    } else {
        return TFM_HAL_ERROR_MEM_FAULT;
    }
}

enum tfm_hal_status_t mpu_init_cfg(void)
{
    struct mpu_armv8m_region_cfg_t region_cfg;
    enum mpu_armv8m_error_t err;

    mpu_armv8m_clean(&dev_mpu_s);

    /* Veneer region */
    region_cfg.region_nr = n_configured_regions++;

    region_cfg.region_base = memory_regions.veneer_base;
    region_cfg.region_limit = memory_regions.veneer_limit;
    region_cfg.region_attridx = MPU_ARMV8M_MAIR_ATTR_CODE_IDX;
    region_cfg.attr_access = MPU_ARMV8M_AP_RO_PRIV_UNPRIV;
    region_cfg.attr_sh = MPU_ARMV8M_SH_NONE;
    region_cfg.attr_exec = MPU_ARMV8M_XN_EXEC_OK;

    err = mpu_armv8m_region_enable(&dev_mpu_s, &region_cfg);

    if (err != MPU_ARMV8M_OK) {
        return TFM_HAL_ERROR_GENERIC;
    }

    /* TFM Core unprivileged code region */
    region_cfg.region_nr = n_configured_regions++;

    region_cfg.region_base =
            (uint32_t)&REGION_NAME(Image$$, TFM_UNPRIV_CODE_START, $$RO$$Base);
    region_cfg.region_limit =
            (uint32_t)&REGION_NAME(Image$$, TFM_UNPRIV_CODE_END, $$RO$$Limit);
    region_cfg.region_attridx = MPU_ARMV8M_MAIR_ATTR_CODE_IDX;
    region_cfg.attr_access = MPU_ARMV8M_AP_RO_PRIV_UNPRIV;
    region_cfg.attr_sh = MPU_ARMV8M_SH_NONE;
    region_cfg.attr_exec = MPU_ARMV8M_XN_EXEC_OK;

    err = mpu_armv8m_region_enable(&dev_mpu_s, &region_cfg);

    if (err != MPU_ARMV8M_OK) {
        return TFM_HAL_ERROR_GENERIC;
    }

    /* RO region */
    region_cfg.region_nr = n_configured_regions++;

    region_cfg.region_base =
            (uint32_t)&REGION_NAME(Image$$, TFM_APP_CODE_START, $$Base);
    region_cfg.region_limit =
            (uint32_t)&REGION_NAME(Image$$, TFM_APP_CODE_END, $$Base);
    region_cfg.region_attridx = MPU_ARMV8M_MAIR_ATTR_CODE_IDX;
    region_cfg.attr_access = MPU_ARMV8M_AP_RO_PRIV_UNPRIV;
    region_cfg.attr_sh = MPU_ARMV8M_SH_NONE;
    region_cfg.attr_exec = MPU_ARMV8M_XN_EXEC_OK;

    err = mpu_armv8m_region_enable(&dev_mpu_s, &region_cfg);

    if (err != MPU_ARMV8M_OK) {
        return TFM_HAL_ERROR_GENERIC;
    }

    /* RW, ZI and stack as one region */
    region_cfg.region_nr = n_configured_regions++;

    region_cfg.region_base =
            (uint32_t)&REGION_NAME(Image$$, TFM_APP_RW_STACK_START, $$Base);
    region_cfg.region_limit =
            (uint32_t)&REGION_NAME(Image$$, TFM_APP_RW_STACK_END, $$Base);
    region_cfg.region_attridx = MPU_ARMV8M_MAIR_ATTR_DATA_IDX;
    region_cfg.attr_access = MPU_ARMV8M_AP_RW_PRIV_UNPRIV;
    region_cfg.attr_sh = MPU_ARMV8M_SH_NONE;
    region_cfg.attr_exec = MPU_ARMV8M_XN_EXEC_NEVER;

    err = mpu_armv8m_region_enable(&dev_mpu_s, &region_cfg);

    if (err != MPU_ARMV8M_OK) {
        return TFM_HAL_ERROR_GENERIC;
    }

#ifdef CONFIG_TFM_PARTITION_META
    /* TFM partition metadata poniter region */
    region_cfg.region_nr = n_configured_regions++;

    region_cfg.region_base =
            (uint32_t)&REGION_NAME(Image$$, TFM_SP_META_PTR, $$ZI$$Base);
    region_cfg.region_limit =
            (uint32_t)&REGION_NAME(Image$$, TFM_SP_META_PTR, $$ZI$$Limit);
    region_cfg.region_attridx = MPU_ARMV8M_MAIR_ATTR_DATA_IDX;
    region_cfg.attr_access = MPU_ARMV8M_AP_RW_PRIV_UNPRIV;
    region_cfg.attr_sh = MPU_ARMV8M_SH_NONE;
    region_cfg.attr_exec = MPU_ARMV8M_XN_EXEC_NEVER;

    err = mpu_armv8m_region_enable(&dev_mpu_s, &region_cfg);

    if (err != MPU_ARMV8M_OK) {
        return TFM_HAL_ERROR_GENERIC;
    }
#endif /* CONFIG_TFM_PARTITION_META */

#ifdef NULL_POINTER_EXCEPTION_DETECTION
	uint32_t num_regions =
		((MPU->TYPE & MPU_TYPE_DREGION_Msk) >> MPU_TYPE_DREGION_Pos);

	if ((num_regions - n_configured_regions) < 2) {
		// We have enabled null pointer detection, but we don't have
		// enough regions for it.
		//
	    // NB: Enabling null-pointer detection can also
		// cause tfm_hal_bind_boundary to return an error due to
		// insufficient memory regions
		return TFM_HAL_ERROR_GENERIC;
	}

	// The armv8m MPU can not be configured to protect a memory region
	// from priviliged reads. However, it is invalid to have two
	// overlapping memory regions and when a memory access is made to
	// such an overlapping area we will get a MemFault. We exploit
	// this undefined behaviour to block priviliged reads to the first
	// 256 bytes. The first 350 bytes on nRF platforms are used for
	// the vector table but luckily the armv8m MPU does not affect
	// exception vector fetches so these two regions we configure will
	// not accidentally disturb any valid memory access.
	for(int i = 0; i < 2; i++) {
		region_cfg.region_nr = n_configured_regions++;

		region_cfg.region_base = 0;
		region_cfg.region_limit = 256 - 32; // The last protected address is limit + 31

		// The region_attridx, attr_access, attr_sh and attr_exec
		// have no effect when memory regions overlap as any
		// access will trigger a MemFault so we just use the
		// previously configured attributes.

		err = mpu_armv8m_region_enable(&dev_mpu_s, &region_cfg);

		if (err != MPU_ARMV8M_OK) {
			return TFM_HAL_ERROR_GENERIC;
		}
	}
#endif /* NULL_POINTER_EXCEPTION_DETECTION */

    mpu_armv8m_enable(&dev_mpu_s, PRIVILEGED_DEFAULT_ENABLE,
                      HARDFAULT_NMI_ENABLE);

    return TFM_HAL_SUCCESS;
}

bool tfm_hal_boundary_need_switch(uintptr_t boundary_from,
                                  uintptr_t boundary_to)
{
    if (boundary_from == boundary_to) {
        return false;
    }

    if (((uint32_t)boundary_from & HANDLE_ATTR_PRIV_MASK) &&
        ((uint32_t)boundary_to & HANDLE_ATTR_PRIV_MASK)) {
        return false;
    }
    return true;
}
