/*
 * Copyright (c) 2001-2019, Arm Limited and Contributors. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */


#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include <limits.h>

/* mbedtls lib */
#include "mbedtls/ccm.h"
#include "mbedtls/cipher.h"
#include "mbedtls/timing.h"

#include "mbedtls_cc_mng.h"
#include "mbedtls_cc_sbrt.h"
#include "secureboot_defs.h"
#include "bootimagesverifier_def.h"

/* local */
#include "run_integration_pal_log.h"
#include "run_integration_test.h"
#include "run_integration_helper.h"
#include "run_integration_flash.h"
#include "run_integration_otp.h"

#ifdef RUNIT_PIE_ENABLED
/* include sbrom data file to determine whether we are running system flows */
#include "bsv_integration_data_def.h"
#endif /* RUNIT_PIE_ENABLED */

/************************************************************
 *
 * defines
 *
 ************************************************************/
#define RUNIT_FLASH_SIZE            0x7000
#define RUNIT_MAX_CHAIN_LENGTH      3

#ifndef ARRAY_SIZE
#define ARRAY_SIZE(x)    (uint32_t)(sizeof(x)/sizeof(x[0]))
#endif

#ifndef RUNIT_SECURE_BOOT_SKIP_TEST
#define RUNIT_SECURE_BOOT_SKIP_TEST 0
#endif

#define RUNIT_SECURE_BOOT__KEY_PKG_1_FLASH_ADDRESS          0x00000000UL
#define RUNIT_SECURE_BOOT__KEY_PKG_2_FLASH_ADDRESS          0x00001000UL
#define RUNIT_SECURE_BOOT__CONTENT_PKG_1_FLASH_ADDRESS      0x00002000UL
#define RUNIT_SECURE_BOOT__IMAGE_PKG_1_FLASH_ADDRESS        0x00003000UL
#define RUNIT_SECURE_BOOT__IMAGE_PKG_2_FLASH_ADDRESS        0x00004000UL
#define RUNIT_SECURE_BOOT__IMAGE_PKG_3_FLASH_ADDRESS        0x00005000UL
#define RUNIT_SECURE_BOOT__IMAGE_PKG_3_NEW_FLASH_ADDRESS    0x00006000UL

#if defined(CC_SB_X509_CERT_SUPPORTED)
#define IS_X509 "yes"
#else
#define IS_X509 "no"
#endif

#if ! defined(RUNIT_PIE_ENABLED)
#if defined(DX_PLAT_ZYNQ7000)
#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__LOAD_ADDRESS /* threeChains,loadAndVerify,se,hbk1,encrypted */ 0x2FF00000
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__LOAD_ADDRESS /* threeChains,loadAndVerify,se,hbk1,encrypted */ 0x2FF01000
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__LOAD_ADDRESS /* threeChains,loadAndVerify,se,hbk1,encrypted */ 0x2FF02000
#elif defined(DX_PLAT_MPS2_PLUS)
#if defined(CORTEX_M33_FPGA)
#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__LOAD_ADDRESS /* default */ 0x38200000
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__LOAD_ADDRESS /* default */ 0x38201000
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__LOAD_ADDRESS /* default */ 0x38202000
#else
#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__LOAD_ADDRESS /* default */ 0x21000000
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__LOAD_ADDRESS /* default */ 0x21001000
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__LOAD_ADDRESS /* default */ 0x21002000
#endif /* defined(CORTEX_M33_FPGA) */
#else
#error not supported
#endif

#if defined(DX_PLAT_ZYNQ7000)
#if defined(CC_SB_X509_CERT_SUPPORTED)
/************************************/
/*          ZYNQ X509               */
/************************************/
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA /* extended_example/sb_key1_cert_test_10.bin */ 0x30, 0x82, 0x04, 0x6f, 0x30, 0x82, 0x02, 0xa7, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x05, 0x00, 0xc1, 0x43, 0xcd, 0x6d, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x38, 0x30, 0x31, 0x32, 0x35, 0x30, 0x38, 0x34, 0x32, 0x31, 0x33, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x35, 0x34, 0x30, 0x33, 0x30, 0x33, 0x31, 0x35, 0x31, 0x30, 0x32, 0x37, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x07, 0x4b, 0x65, 0x79, 0x43, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x8f, 0x00, 0x30, 0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x6e, 0x30, 0x6c, 0x30, 0x1a, 0x06, 0x03, 0x64, 0x01, 0x01, 0x01, 0x01, 0xff, 0x04, 0x10, 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x30, 0x1e, 0x06, 0x03, 0x64, 0x01, 0x02, 0x01, 0x01, 0xff, 0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x30, 0x2e, 0x06, 0x03, 0x64, 0x01, 0x03, 0x01, 0x01, 0xff, 0x04, 0x24, 0x03, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0x5e, 0x6a, 0x73, 0x83, 0x1b, 0xa5, 0x23, 0xae, 0x88, 0x1d, 0xb5, 0x09, 0xc2, 0x50, 0xec, 0xa1, 0x37, 0x35, 0xfd, 0xb3, 0x4b, 0x65, 0x95, 0x32, 0x88, 0xaa, 0x16, 0xc5, 0x28, 0x2c, 0x16, 0x26, 0xf5, 0xf0, 0xce, 0x4e, 0x8a, 0xd0, 0xf5, 0x00, 0x95, 0xa1, 0xeb, 0x60, 0xca, 0x82, 0x08, 0x09, 0x49, 0x7e, 0x52, 0x14, 0x4e, 0xb8, 0x4f, 0x88, 0x0f, 0x6e, 0xbe, 0xf9, 0xe7, 0xee, 0x36, 0x3a, 0x41, 0xe1, 0xb1, 0x53, 0x37, 0x26, 0xa7, 0x98, 0x89, 0x5b, 0x68, 0x1a, 0x78, 0xc9, 0x19, 0x39, 0x95, 0xf5, 0xad, 0xbe, 0xb3, 0x97, 0x51, 0x02, 0xcd, 0xce, 0x69, 0x7e, 0x36, 0x22, 0xfe, 0x49, 0x9a, 0x31, 0xcf, 0xe3, 0xea, 0xdb, 0x0e, 0x95, 0x7d, 0x03, 0x24, 0x15, 0xd0, 0xa3, 0x34, 0x83, 0xa2, 0x0c, 0xa6, 0x98, 0xc0, 0xef, 0x42, 0xe3, 0x47, 0x49, 0x77, 0x03, 0x47, 0xec, 0xa1, 0x5f, 0x91, 0x6f, 0xe0, 0x45, 0x56, 0x6c, 0xb1, 0xfa, 0x25, 0x76, 0x98, 0xf9, 0x8f, 0xbb, 0xa7, 0x84, 0x97, 0x73, 0x39, 0x89, 0x75, 0x2c, 0x85, 0x95, 0xd9, 0xc4, 0x6d, 0x54, 0x0b, 0x98, 0xdb, 0x8a, 0x58, 0x4a, 0xa9, 0x96, 0xb9, 0x60, 0x84, 0x39, 0x39, 0xcf, 0xd9, 0x5c, 0x58, 0xc6, 0x8e, 0xeb, 0x75, 0xb4, 0xfe, 0xb6, 0x6c, 0x33, 0xe5, 0x14, 0xed, 0x11, 0xe2, 0xe1, 0xcb, 0x14, 0x21, 0x59, 0xa8, 0x24, 0xf1, 0x71, 0x1f, 0x98, 0xb5, 0xdf, 0x88, 0x30, 0x35, 0x46, 0x50, 0xde, 0xb0, 0x91, 0xd2, 0x3d, 0x1d, 0x95, 0xf2, 0x6f, 0x5b, 0x04, 0x10, 0xa1, 0x92, 0x92, 0x9d, 0xb4, 0x52, 0xa2, 0x7b, 0xaa, 0x91, 0xc4, 0x9d, 0xb3, 0x75, 0x71, 0xb1, 0x34, 0x2e, 0x4d, 0x06, 0x5d, 0x1f, 0x0d, 0x3f, 0xf1, 0xf8, 0x8d, 0xf7, 0x37, 0x82, 0x43, 0xfa, 0xe8, 0x0a, 0xa3, 0x4e, 0xca, 0x94, 0x77, 0x6d, 0x9f, 0x8b, 0xe5, 0x0c, 0xb5, 0x19, 0x75, 0x7d, 0xac, 0x22, 0x89, 0x1b, 0x22, 0x73, 0xe5, 0xe0, 0xd3, 0x14, 0x92, 0x72, 0xd3, 0x1c, 0xda, 0xf5, 0xd7, 0x7a, 0xb5, 0xfc, 0xe7, 0x6f, 0x28, 0x59, 0x15, 0xd7, 0xab, 0xb2, 0xa4, 0xfb, 0xf8, 0x7e, 0x4d, 0x33, 0x5d, 0x86, 0xc9, 0xba, 0xd2, 0x84, 0x5c, 0x5b, 0x4e, 0xc2, 0x3f, 0xe8, 0xeb, 0x91, 0x21, 0x62, 0xf1, 0xa8, 0xb5, 0xbb, 0x7c, 0xae, 0x37, 0x8f, 0x80, 0xb9, 0x97, 0x28, 0x1a, 0x2d, 0x4d, 0x5c, 0x8e, 0x18, 0x80, 0x64, 0x4b, 0x1c, 0x75, 0x7b, 0x45, 0xd2, 0x13, 0x92, 0x66, 0x22, 0x59, 0x99, 0x6f, 0xc2, 0x5d, 0x62, 0x9f, 0xab, 0xd2, 0xa5, 0xce, 0x7f, 0xa1, 0xbc, 0x92, 0x69, 0x7b, 0x2d, 0x8a, 0xa6, 0x5c, 0x18, 0xc5, 0xcd, 0xdf, 0x95, 0x2d, 0xa3, 0x94, 0x28, 0x51, 0xc2, 0x1c, 0x04, 0x66, 0x3c, 0x52, 0x67, 0x46,
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA /* extended_example/sb_key2_cert_test_10.bin */ 0x30, 0x82, 0x04, 0x6f, 0x30, 0x82, 0x02, 0xa7, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x05, 0x00, 0xe3, 0x1c, 0x48, 0x45, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x38, 0x30, 0x31, 0x32, 0x35, 0x30, 0x38, 0x34, 0x32, 0x31, 0x33, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x35, 0x34, 0x30, 0x33, 0x30, 0x33, 0x31, 0x35, 0x31, 0x30, 0x32, 0x37, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x07, 0x4b, 0x65, 0x79, 0x43, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x8f, 0x00, 0x30, 0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x6e, 0x30, 0x6c, 0x30, 0x1a, 0x06, 0x03, 0x64, 0x01, 0x01, 0x01, 0x01, 0xff, 0x04, 0x10, 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x30, 0x1e, 0x06, 0x03, 0x64, 0x01, 0x02, 0x01, 0x01, 0xff, 0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x30, 0x2e, 0x06, 0x03, 0x64, 0x01, 0x03, 0x01, 0x01, 0xff, 0x04, 0x24, 0x03, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0x69, 0x05, 0x93, 0x68, 0x65, 0xc3, 0x00, 0x14, 0xc0, 0x00, 0x41, 0x07, 0xfd, 0x65, 0x61, 0x65, 0x76, 0x1c, 0x63, 0xb5, 0x2f, 0xe3, 0x2d, 0x66, 0x7a, 0x42, 0x02, 0x87, 0x58, 0x3a, 0xd5, 0x2a, 0xd3, 0x8b, 0x01, 0x16, 0xf8, 0x64, 0x71, 0xd6, 0x17, 0x21, 0x88, 0xd8, 0x1b, 0xee, 0xd7, 0xdf, 0x78, 0xc2, 0xfe, 0x9b, 0x93, 0xcd, 0x43, 0x6d, 0x50, 0xea, 0x66, 0xe7, 0x3e, 0x90, 0xf9, 0xb7, 0x3a, 0x68, 0x51, 0x5a, 0x7c, 0x21, 0xc8, 0x5a, 0x9d, 0xe1, 0x79, 0x5c, 0x29, 0xf2, 0x06, 0x24, 0x31, 0xaf, 0x81, 0x57, 0x79, 0xd0, 0xa0, 0x8c, 0xc9, 0xc2, 0x1b, 0x08, 0x63, 0xf2, 0xc7, 0xee, 0x06, 0x3e, 0x80, 0x7d, 0x86, 0x93, 0x53, 0x0c, 0xee, 0x7d, 0x65, 0x1a, 0x0f, 0x3f, 0x1c, 0x84, 0x2a, 0x1a, 0x34, 0x2b, 0x6a, 0xf0, 0x9a, 0xfb, 0xff, 0x91, 0x51, 0xa6, 0xba, 0xa8, 0x69, 0x80, 0xa7, 0xe9, 0xc6, 0x37, 0x8d, 0xa0, 0x4b, 0x2a, 0xd5, 0x43, 0x76, 0x4e, 0x0e, 0x19, 0x25, 0xa7, 0xa5, 0x58, 0x03, 0x32, 0x4b, 0x05, 0x04, 0xfc, 0xd8, 0xf8, 0x6c, 0x3f, 0xda, 0x6a, 0xa8, 0xc4, 0xd3, 0x55, 0x9f, 0x68, 0x27, 0x96, 0xb9, 0x96, 0x5a, 0xcb, 0xc7, 0x08, 0x40, 0x33, 0x24, 0x98, 0x3a, 0xac, 0xd4, 0x2e, 0xf6, 0xc5, 0xbb, 0x8a, 0x22, 0xdf, 0x69, 0xbc, 0x85, 0x16, 0x13, 0xcc, 0x8b, 0xa6, 0x6d, 0xf3, 0x8b, 0x12, 0xc1, 0x03, 0x06, 0x40, 0x66, 0xa0, 0x66, 0xcc, 0x14, 0x4b, 0xd7, 0xb7, 0x05, 0x73, 0x60, 0xdb, 0xc5, 0xb2, 0x42, 0xa5, 0x99, 0x79, 0x26, 0x04, 0x2c, 0x94, 0x15, 0xb9, 0x01, 0x9e, 0x0f, 0x98, 0x36, 0xc1, 0xe5, 0xe4, 0x5e, 0x1e, 0x62, 0xea, 0x6a, 0x49, 0x0d, 0x6f, 0x34, 0x88, 0x82, 0x39, 0x6e, 0x63, 0xed, 0x08, 0x9a, 0x47, 0x49, 0x07, 0x62, 0xed, 0x9b, 0x84, 0x23, 0x9c, 0xee, 0x3d, 0x82, 0xc2, 0x6a, 0x65, 0xc8, 0x91, 0x9d, 0x3e, 0x11, 0x0b, 0xa8, 0x57, 0x1d, 0xbf, 0x89, 0x01, 0x0f, 0x0d, 0x7d, 0x8e, 0xde, 0x10, 0x6b, 0x87, 0xbe, 0x61, 0x03, 0x0b, 0xff, 0x8f, 0xc5, 0x03, 0x0e, 0x59, 0x5b, 0xb1, 0x9f, 0x21, 0xc8, 0x76, 0x85, 0x45, 0xb4, 0xd9, 0x72, 0x81, 0x50, 0x02, 0x27, 0xe7, 0x26, 0x07, 0xb9, 0x05, 0x30, 0xe2, 0x8e, 0x66, 0x51, 0x80, 0x99, 0x24, 0x6f, 0x92, 0x45, 0xba, 0xd2, 0x4d, 0x2f, 0xb8, 0x61, 0x45, 0x6e, 0x9c, 0x09, 0x5c, 0x01, 0x13, 0x6d, 0xb1, 0x7b, 0xb5, 0x14, 0x0e, 0x73, 0xce, 0x4c, 0xe5, 0x12, 0xd7, 0x10, 0xdc, 0x27, 0xfb, 0xa5, 0x88, 0xa0, 0xd9, 0xf5, 0x86, 0x0b, 0x47, 0xc5, 0xac, 0x81, 0x9a, 0xff, 0xe8, 0xee, 0x69, 0xa0, 0xc4, 0xa4, 0x0d, 0xfb, 0x94, 0x36, 0x32, 0xd7, 0x52, 0xe0, 0x36,
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA /* extended_example/sb_content_cert_test_10.bin */ 0x30, 0x82, 0x04, 0xde, 0x30, 0x82, 0x03, 0x16, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x04, 0x4a, 0x46, 0xac, 0xf4, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x38, 0x30, 0x31, 0x32, 0x35, 0x30, 0x38, 0x34, 0x32, 0x31, 0x33, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x35, 0x34, 0x30, 0x33, 0x30, 0x33, 0x31, 0x35, 0x31, 0x30, 0x32, 0x37, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x07, 0x43, 0x6e, 0x74, 0x43, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x8f, 0x00, 0x30, 0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xdd, 0x30, 0x81, 0xda, 0x30, 0x1a, 0x06, 0x03, 0x64, 0x02, 0x01, 0x01, 0x01, 0xff, 0x04, 0x10, 0x63, 0x63, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x03, 0x00, 0x30, 0x1e, 0x06, 0x03, 0x64, 0x02, 0x02, 0x01, 0x01, 0xff, 0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x30, 0x81, 0x9b, 0x06, 0x03, 0x64, 0x02, 0x04, 0x01, 0x01, 0xff, 0x04, 0x81, 0x90, 0x03, 0x00, 0x00, 0x00, 0xf5, 0x63, 0xad, 0x57, 0xcb, 0xc1, 0xef, 0xa3, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x00, 0xf0, 0x2f, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x10, 0xf0, 0x2f, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x20, 0xf0, 0x2f, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0xc4, 0x55, 0xb2, 0xdc, 0x4a, 0x70, 0x71, 0xe4, 0xfe, 0x1c, 0xab, 0xec, 0xe9, 0xf8, 0xab, 0xb3, 0xdb, 0x43, 0x53, 0x0f, 0xd9, 0x49, 0x92, 0x4b, 0xa7, 0xd3, 0xda, 0xdd, 0xfc, 0x55, 0x65, 0x65, 0x41, 0xb0, 0x24, 0x39, 0x2b, 0x20, 0x30, 0xde, 0x5e, 0xb1, 0xa4, 0x3d, 0xa7, 0x49, 0x85, 0x75, 0xe0, 0x44, 0xd4, 0xfa, 0xb2, 0x23, 0x30, 0x7d, 0xce, 0xce, 0xf6, 0x39, 0x2f, 0x3d, 0x26, 0xf9, 0x4f, 0x60, 0xf1, 0xea, 0x76, 0x1b, 0xc4, 0xdf, 0x90, 0x0c, 0xac, 0xed, 0x13, 0x5c, 0xce, 0x54, 0x1c, 0x76, 0x67, 0x6a, 0x94, 0x6a, 0x54, 0x84, 0xa3, 0xfe, 0x77, 0xce, 0xb3, 0xc9, 0x75, 0xce, 0xd2, 0xad, 0xa3, 0x47, 0x00, 0x10, 0xb2, 0xc6, 0x07, 0x80, 0x6d, 0xfa, 0xaa, 0x85, 0x9d, 0x35, 0x1d, 0xe4, 0x22, 0x61, 0x6b, 0xb7, 0xca, 0xd7, 0xca, 0x09, 0xaa, 0xfe, 0xb7, 0x82, 0x54, 0xb4, 0x00, 0xe0, 0x35, 0x91, 0x66, 0x00, 0x3b, 0xcb, 0x17, 0x16, 0x0d, 0x58, 0x12, 0x34, 0x28, 0xe4, 0xde, 0x1b, 0xb1, 0x6c, 0x24, 0x12, 0x4d, 0x6b, 0x78, 0xfc, 0xa5, 0xc9, 0xd5, 0xeb, 0x62, 0x1a, 0x73, 0x81, 0x7b, 0x41, 0x96, 0x58, 0x48, 0x05, 0x4d, 0xc0, 0xa0, 0x68, 0x59, 0x6c, 0x26, 0x09, 0xfb, 0x76, 0x15, 0x84, 0xfe, 0x8f, 0x26, 0x95, 0xf9, 0x42, 0xa7, 0xdc, 0x4a, 0x75, 0xf3, 0x57, 0xbd, 0xfc, 0x7c, 0x65, 0x40, 0x28, 0x44, 0xf5, 0x78, 0x51, 0x95, 0x20, 0xfb, 0xee, 0xf9, 0x74, 0x99, 0x95, 0x6a, 0xf9, 0x78, 0xb7, 0xd5, 0xe6, 0xee, 0xcf, 0x69, 0xf2, 0x53, 0xf8, 0xf3, 0xb3, 0x3c, 0x30, 0xd1, 0xbe, 0x04, 0x10, 0x6e, 0xcf, 0x44, 0xea, 0x31, 0x95, 0x30, 0x51, 0x9a, 0x78, 0x2e, 0x87, 0x32, 0x57, 0x03, 0x00, 0xf8, 0xf9, 0xd8, 0x3d, 0x3a, 0xbd, 0xc9, 0x65, 0x46, 0x7d, 0xa0, 0x58, 0x37, 0xa8, 0x77, 0xcb, 0x99, 0x3b, 0xc2, 0x54, 0x33, 0x3f, 0x29, 0x7b, 0x2a, 0xe6, 0x33, 0x04, 0x0d, 0x42, 0x3f, 0x10, 0x04, 0x6a, 0xb5, 0x57, 0x33, 0x05, 0x0d, 0xd9, 0xa4, 0x3e, 0xf6, 0x95, 0xfc, 0xc0, 0x97, 0x7f, 0x40, 0x3f, 0x10, 0x7f, 0x31, 0x60, 0xf7, 0x96, 0x52, 0x02, 0xce, 0x23, 0x82, 0xa5, 0xcd, 0xce, 0x1e, 0x8a, 0x94, 0x32, 0xc1, 0x1d, 0xc4, 0x86, 0x19, 0xd8, 0xee, 0x83, 0xab, 0x8d, 0x3a, 0x1a, 0x3d, 0x8c, 0xa3, 0x13, 0x5d, 0xe4, 0xe7, 0x43, 0xf4, 0xd4, 0x2d, 0x53, 0xef, 0x91, 0x7c, 0xf2, 0xe4, 0xbd, 0x43, 0x6b, 0xed, 0xa0, 0xe7, 0xbd, 0x04, 0xb5, 0x41, 0x17, 0x1d, 0xe4, 0x6f, 0x09, 0xc9, 0xf6, 0xb1, 0x1a, 0x97, 0x72, 0x48, 0x45, 0x54, 0x70, 0x5d, 0x21, 0xfa, 0x1b, 0x56, 0xff, 0x2c, 0x4b, 0xec, 0xaa, 0x45, 0xd3, 0x94, 0x49, 0x03, 0x86, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00,
#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__DATA /* extended_example/sb_image1_enc.bin */ 0x1f, 0xef, 0x8c, 0xa0, 0xb6, 0x85, 0xfb, 0x56, 0x36, 0xcf, 0x03, 0xf8, 0x84, 0xbd, 0x1f, 0x3b, 0x24, 0x9c, 0x56, 0xa9, 0xe5, 0x77, 0x07, 0x44, 0xfe, 0x9e, 0x83, 0x93, 0xbe, 0x35, 0xb6, 0x2b, 0x5e, 0x03, 0x1d, 0x53, 0xf5, 0x32, 0x52, 0xc1, 0x5c, 0x1a, 0xc7, 0x03, 0x30, 0x95, 0x1f, 0x32, 0x87, 0x79, 0x24, 0x5e, 0x96, 0x79, 0xbc, 0xa8, 0xda, 0x5f, 0x21, 0x85, 0xe7, 0xec, 0x21, 0x7b, 0x70, 0x24, 0x18, 0xc2, 0xb2, 0xcc, 0xfe, 0x71, 0x0f, 0x51, 0xb4, 0x9a, 0xf1, 0x39, 0x20, 0x1f, 0x6a, 0x0f, 0x33, 0xab, 0x7c, 0x13, 0x8a, 0x83, 0x5c, 0x77, 0x0b, 0xef, 0x1a, 0x58, 0x97, 0x88, 0x27, 0xcb, 0x54, 0x13, 0xd8, 0x6d, 0x66, 0x23, 0xfa, 0xe1, 0xa5, 0xc4, 0x69, 0x99, 0xa8, 0xdc,
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__DATA /* extended_example/sb_image2_enc.bin */ 0x7b, 0xdd, 0x83, 0xdc, 0x2c, 0xb4, 0x51, 0x21, 0xa6, 0xfb, 0xd7, 0xe3, 0x8d, 0x14, 0xe8, 0xb7, 0xba, 0x4c, 0x45, 0x0a, 0x99, 0x18, 0x59, 0x2c, 0x1e, 0x97, 0x65, 0x09, 0xb6, 0x54, 0x8c, 0x77, 0x5b, 0x33, 0xc3, 0x0a, 0x73, 0x24, 0xe4, 0x7e, 0x2a, 0xfe, 0xd9, 0x50, 0x0b, 0x5e, 0x7c, 0xe3, 0x16, 0x72, 0xbb, 0xf1, 0x28, 0xe0, 0x06, 0x32, 0x9d, 0xa5, 0x56, 0x55, 0xd7, 0x01, 0x75, 0xe6, 0xa6, 0xc0, 0x03, 0x7b, 0xde, 0xda, 0xc5, 0xc3, 0xfb, 0x1e, 0x63, 0x10, 0x7b, 0x1c, 0x54, 0xd4, 0x86, 0x1a, 0xf7, 0x99, 0x0d, 0x8c, 0xf3, 0x9f, 0x39, 0xe1, 0x61, 0xba, 0x9e, 0x4f, 0x1b, 0x08, 0xe6, 0x86, 0x44, 0x93, 0x19, 0x50, 0xe1, 0x88, 0x32, 0xfe, 0xb4, 0x88, 0x2d, 0x84, 0x0a, 0xb1,
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA /* extended_example/sb_image3_enc.bin */ 0xc5, 0x6a, 0x57, 0x9c, 0x84, 0x98, 0x7e, 0xdb, 0x5a, 0x90, 0xf8, 0x34, 0xde, 0xd4, 0x03, 0x7a, 0x6a, 0xdb, 0x35, 0x2a, 0xf0, 0xa1, 0x14, 0xdd, 0xee, 0x40, 0x7c, 0x27, 0xdf, 0x96, 0xb5, 0x96, 0xf8, 0x9b, 0xaa, 0x90, 0x7a, 0x50, 0x60, 0x01, 0xf8, 0x93, 0xaf, 0x9c, 0xb1, 0x35, 0x33, 0xc1, 0xc3, 0x11, 0xdb, 0x5a, 0xe9, 0x57, 0x08, 0x86, 0x03, 0xc3, 0xe4, 0x5f, 0x37, 0xb3, 0x74, 0xf9, 0xa0, 0x65, 0x4b, 0x15, 0xae, 0x7e, 0x2c, 0xe2, 0x88, 0x47, 0x01, 0x5d, 0x70, 0x63, 0x44, 0x77, 0xe3, 0x5e, 0x74, 0x39, 0x22, 0x3a, 0xc8, 0xa9, 0x11, 0x8a, 0x43, 0xc0, 0xe9, 0xc3, 0x34, 0x63, 0xaf, 0x5d, 0x3d, 0x39, 0xe3, 0xbc, 0xc0, 0x77, 0xf8, 0x3f, 0xed, 0x27, 0x85, 0xd5, 0x43, 0xfb,
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA_LENGTH /* sb_key1_cert_test_10.bin */ 1139
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA_LENGTH /* sb_key2_cert_test_10.bin */ 1139
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA_LENGTH /* sb_content_cert_test_10.bin */ 1276
#else
/************************************/
/*        ZYNQ NOT_X509             */
/************************************/
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA /* extended_example/sb_key1_cert_test_10.bin */ 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x03, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0x0b, 0xb1, 0xcd, 0x21, 0x43, 0xf5, 0x9a, 0x39, 0xb8, 0xd3, 0xe4, 0xdc, 0x7a, 0x82, 0xc6, 0xdc, 0x72, 0xd5, 0x6b, 0x13, 0xf3, 0x80, 0x1a, 0x00, 0x77, 0x17, 0xce, 0xc6, 0x98, 0x6b, 0x06, 0xc4, 0x4f, 0x74, 0xa2, 0xfa, 0x97, 0xe8, 0x14, 0x3d, 0x9e, 0xab, 0xa1, 0xaf, 0x5a, 0x3f, 0x38, 0x14, 0x5d, 0x10, 0x40, 0xc1, 0xdf, 0x58, 0xcd, 0xa6, 0xf5, 0xcf, 0x7b, 0xd9, 0x8d, 0xae, 0xd5, 0xe4, 0x13, 0x3e, 0x0b, 0x89, 0x21, 0x06, 0xc9, 0xd4, 0xe9, 0xf5, 0xe5, 0x74, 0x2d, 0x81, 0x45, 0x0d, 0x16, 0xe0, 0x7f, 0xf6, 0xd2, 0x63, 0xfc, 0xf9, 0x95, 0xf4, 0xf9, 0x18, 0x3f, 0x16, 0x1c, 0x69, 0xac, 0x45, 0x22, 0x40, 0x01, 0x88, 0xc3, 0xe3, 0x42, 0x01, 0x7c, 0x0a, 0xd1, 0xeb, 0x15, 0x22, 0x0d, 0xad, 0xd6, 0xc6, 0x03, 0xe6, 0x72, 0x26, 0xba, 0x30, 0xb4, 0x1e, 0xad, 0x93, 0xcf, 0x5e, 0x64, 0xd4, 0xf2, 0x03, 0x7c, 0x79, 0x98, 0x71, 0x60, 0xee, 0xd7, 0xa4, 0xe4, 0x77, 0xe3, 0x6b, 0xdb, 0x06, 0xcc, 0xd4, 0xe6, 0x1b, 0x08, 0x2a, 0xaa, 0xd6, 0x26, 0x21, 0x77, 0x1f, 0x22, 0xae, 0xa8, 0x7a, 0x92, 0xa4, 0x09, 0x54, 0xff, 0x5a, 0x53, 0xb5, 0x50, 0xd6, 0xf5, 0xd4, 0x50, 0xd4, 0x37, 0xd3, 0xcf, 0x1f, 0x8f, 0x93, 0xf7, 0x29, 0xfc, 0x83, 0xda, 0x3d, 0xa1, 0x0a, 0x59, 0xc8, 0x0d, 0xe2, 0xb7, 0x75, 0x9c, 0x18, 0x95, 0x4a, 0x3b, 0xa3, 0x33, 0x52, 0x99, 0x01, 0x31, 0x43, 0xd5, 0x95, 0x6f, 0x8f, 0xe6, 0x58, 0x0d, 0xfd, 0xdd, 0x05, 0xb8, 0x36, 0xea, 0xae, 0x2d, 0x21, 0xbc, 0x47, 0x51, 0x27, 0xe8, 0x96, 0x92, 0x7f, 0xec, 0x0a, 0x3a, 0x32, 0x62, 0x32, 0xfd, 0x92, 0xb4, 0x23, 0x43, 0x14, 0xd1, 0x2b, 0xe1, 0x37, 0xa0, 0x83, 0x02, 0xe4, 0xa8, 0x59, 0x97, 0x45, 0xc6, 0xd2, 0xd1, 0x73, 0x7e, 0x9e, 0x68, 0x33, 0xf3, 0xe0, 0x4d, 0xd0, 0x68, 0xf4, 0x16, 0x6e, 0x49, 0x60, 0xe7, 0xb2, 0xcf, 0x34, 0xce, 0x4b, 0xdf, 0x55, 0x02, 0xe1, 0xa5, 0xe9, 0x65, 0xfc, 0xc3, 0x9e, 0xb7, 0x98, 0x95, 0x5d, 0x1e, 0xf1, 0x18, 0x7b, 0xdb, 0x70, 0x0c, 0xd4, 0x9e, 0x93, 0xdb, 0x8f, 0x78, 0x28, 0xf6, 0xd6, 0xbd, 0x8e, 0x01, 0xd1, 0x58, 0xcd, 0xb5, 0xb5, 0x55, 0x49, 0x95, 0x12, 0x58, 0x61, 0x0b, 0x15, 0x40, 0xb6, 0xcf, 0x6e, 0x20, 0x88, 0xe9, 0x1f, 0x45, 0xf5, 0xed, 0x06, 0xa3, 0xa1, 0x4b, 0xd8, 0xc1, 0xca, 0x55, 0xed, 0x40, 0x2a, 0xb4, 0x74, 0x5a, 0x3b, 0xb4, 0x9d, 0xe2, 0xd8, 0xcf, 0x14, 0x13, 0xa3, 0xb4, 0xd2, 0x0f, 0x96, 0x04, 0x2a, 0xce, 0xf4, 0xe7, 0x72, 0x28, 0x63, 0x6d, 0x59, 0x24, 0x8a, 0x0f, 0x23, 0x54, 0x72, 0x12, 0x84, 0xf5, 0x5f,
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA /* extended_example/sb_key2_cert_test_10.bin */ 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x03, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0xa1, 0x5c, 0xd2, 0xb6, 0x0c, 0x8d, 0x36, 0xef, 0x1b, 0x68, 0x8e, 0x00, 0x82, 0x7e, 0x14, 0x0c, 0x13, 0x76, 0x3c, 0xa9, 0xc7, 0x02, 0x66, 0x72, 0x18, 0xb2, 0xfe, 0xd5, 0x05, 0x69, 0xe8, 0x2d, 0x37, 0x01, 0x83, 0x4d, 0x8d, 0x41, 0x66, 0x5d, 0x2b, 0xd6, 0x23, 0x1d, 0x9b, 0x12, 0x38, 0x62, 0xa1, 0x07, 0x5c, 0xa2, 0x1f, 0xb6, 0xce, 0x35, 0x9d, 0x6a, 0x6e, 0x8c, 0x1c, 0xa9, 0x5c, 0xa5, 0x3b, 0x86, 0x78, 0xac, 0xa8, 0x9f, 0x5b, 0x2e, 0x19, 0x52, 0x4d, 0xc5, 0xaa, 0x47, 0xe7, 0x90, 0xf4, 0xcf, 0x6b, 0x23, 0x3f, 0xc7, 0x05, 0xc1, 0xf9, 0x52, 0x9b, 0x60, 0x74, 0x07, 0x8a, 0xec, 0x73, 0x65, 0x51, 0x1f, 0x99, 0x2c, 0xa0, 0x1d, 0xb0, 0x29, 0x57, 0xf8, 0x9a, 0x8d, 0x80, 0x80, 0x7f, 0x6e, 0xc6, 0x44, 0x69, 0x61, 0x6b, 0x61, 0x10, 0xd5, 0xe8, 0xa0, 0x55, 0x36, 0x11, 0x5e, 0xad, 0x36, 0x8f, 0x92, 0x46, 0x73, 0x57, 0xdf, 0xdc, 0x47, 0xf5, 0xfd, 0xe9, 0x25, 0x2b, 0xec, 0x71, 0xf6, 0x66, 0x96, 0x67, 0xbd, 0xcf, 0x32, 0xd7, 0x46, 0x69, 0x05, 0xb2, 0xb0, 0xd3, 0xb3, 0xe7, 0xef, 0x58, 0x4a, 0x5e, 0xda, 0xe5, 0x0e, 0x61, 0xa3, 0x5a, 0x65, 0xce, 0xe0, 0x23, 0x0f, 0x86, 0x31, 0x2a, 0x97, 0xc3, 0xd5, 0x45, 0xb6, 0x4a, 0x5b, 0xc0, 0x17, 0x36, 0xb2, 0x11, 0xa8, 0x08, 0xcf, 0x26, 0xc8, 0x82, 0x08, 0x7c, 0xdc, 0xa2, 0x61, 0x57, 0x83, 0xa6, 0x35, 0xe7, 0x0a, 0xb2, 0x0c, 0x62, 0xa9, 0x0e, 0xcb, 0xfc, 0x46, 0x13, 0xf6, 0xae, 0x88, 0xc8, 0xc8, 0x8a, 0xb0, 0xee, 0xd4, 0xbc, 0xb3, 0xf8, 0x68, 0x62, 0x40, 0x2f, 0x6b, 0xd4, 0x45, 0x1f, 0xf7, 0x37, 0x95, 0xaf, 0x5b, 0xa9, 0x80, 0xf8, 0x8a, 0xe2, 0xf5, 0xd4, 0x5e, 0xeb, 0xf0, 0xe9, 0x5e, 0x19, 0x9d, 0x6c, 0xc4, 0x70, 0x71, 0x11, 0xb3, 0x63, 0x3d, 0x64, 0xf3, 0x8b, 0x5a, 0x8a, 0x2a, 0xa4, 0xb9, 0x2b, 0x32, 0xf0, 0xcf, 0x1c, 0xf3, 0x00, 0xf8, 0xa8, 0x25, 0x03, 0xf0, 0xa1, 0xed, 0x35, 0x99, 0xa4, 0xb7, 0x86, 0xd1, 0x3d, 0x88, 0xae, 0xa5, 0x0c, 0xd8, 0x3d, 0xf7, 0xae, 0x13, 0x2c, 0x25, 0xfc, 0x1d, 0xb9, 0x6e, 0x55, 0x70, 0x1c, 0x12, 0x46, 0x78, 0x8c, 0xdf, 0xbc, 0xc1, 0x14, 0xca, 0xc3, 0x04, 0xd6, 0x85, 0x60, 0x45, 0x10, 0x02, 0xf3, 0xd2, 0x7c, 0x5b, 0x7c, 0x28, 0x2d, 0xc7, 0x10, 0x9a, 0xe3, 0x73, 0x20, 0x89, 0x7e, 0x4f, 0x51, 0x49, 0x1a, 0x0e, 0xd5, 0xcb, 0x66, 0xf5, 0x59, 0x44, 0xa9, 0x93, 0x81, 0x39, 0xdd, 0x7d, 0xff, 0x5c, 0x25, 0xd9, 0x20, 0x6b, 0x9b, 0x00, 0x48, 0x60, 0x59, 0x87, 0xe6, 0xd6, 0x0a, 0xc2, 0x6b, 0xa9, 0x46, 0x06, 0x0c, 0xd5, 0xec, 0x35,
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA /* extended_example/sb_content_cert_test_10.bin */ 0x63, 0x63, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x03, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x03, 0x00, 0x00, 0x00, 0xf4, 0xad, 0x09, 0x7c, 0x21, 0x20, 0x32, 0x1d, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x00, 0xf0, 0x2f, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x10, 0xf0, 0x2f, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x20, 0xf0, 0x2f, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x73, 0x59, 0x89, 0x34, 0x16, 0x0f, 0x98, 0x59, 0xe3, 0x9c, 0xa8, 0x0f, 0xa3, 0x82, 0x91, 0xe3, 0x34, 0x8e, 0x05, 0x0c, 0xd9, 0x0d, 0x70, 0x18, 0xea, 0x2d, 0xa0, 0xcd, 0x21, 0xda, 0x3f, 0xc3, 0x34, 0x0f, 0x08, 0x05, 0xad, 0x39, 0x0c, 0x3e, 0xb0, 0xfb, 0x09, 0x4b, 0xde, 0xc0, 0x96, 0xa7, 0x36, 0x11, 0xd8, 0x90, 0xbf, 0x56, 0x16, 0xe3, 0x2b, 0x82, 0xd0, 0x79, 0x2a, 0x94, 0xaf, 0xee, 0xb7, 0xff, 0x7e, 0xb0, 0x4e, 0xbf, 0xaf, 0x07, 0x6c, 0x0e, 0x4f, 0xc5, 0xa4, 0xf2, 0xcc, 0x06, 0xb1, 0xde, 0xef, 0x16, 0xec, 0x73, 0xc4, 0x1d, 0x22, 0xa3, 0x22, 0xa1, 0x7b, 0xe0, 0x7c, 0xd2, 0x94, 0x25, 0x26, 0xf8, 0xab, 0x14, 0x52, 0xa0, 0xa4, 0x30, 0x06, 0xb4, 0x0d, 0x79, 0xa3, 0x92, 0xdb, 0x35, 0xca, 0xfd, 0xfb, 0xa6, 0xc5, 0xb1, 0x04, 0xae, 0xd6, 0x08, 0x1e, 0x32, 0xc1, 0xcf, 0xfd, 0x97, 0x9b, 0xf1, 0x3f, 0x6a, 0xdb, 0x6d, 0xec, 0x22, 0x1c, 0xbf, 0x18, 0xd5, 0xa1, 0x35, 0xbd, 0x08, 0x08, 0x58, 0x20, 0xd3, 0xd0, 0x0f, 0x5a, 0xa7, 0x78, 0x05, 0xcf, 0x75, 0x79, 0x58, 0x5e, 0x98, 0xd8, 0x8f, 0xe6, 0x26, 0x20, 0x6a, 0x69, 0xd3, 0x80, 0x03, 0x33, 0x81, 0xff, 0xd9, 0x06, 0x79, 0x71, 0x0d, 0xa5, 0xa7, 0x87, 0x77, 0x18, 0x7d, 0x05, 0x7a, 0xe6, 0xca, 0x8d, 0x4b, 0x63, 0xf8, 0x60, 0x6c, 0x92, 0x17, 0x80, 0x79, 0x0c, 0xcb, 0x7e, 0x30, 0x39, 0x71, 0x96, 0x06, 0xf6, 0x12, 0x90, 0x45, 0x4a, 0x9c, 0xdf, 0x74, 0x75, 0xbb, 0x98, 0xd4, 0x9d, 0x41, 0x35, 0x38, 0xd8, 0xa0, 0x69, 0x92, 0xb7, 0xc7, 0xd7, 0x09, 0xd1, 0x81, 0xb3, 0xf4, 0xc0, 0x0f, 0x10, 0xe0, 0x8a, 0xb3, 0xe6, 0xf1, 0x46, 0x62, 0xcf, 0xbc, 0xdd, 0x02, 0xe0, 0x1b, 0x72, 0x99, 0xd8, 0xae, 0x11, 0xb8, 0x42, 0x01, 0xd4, 0xe2, 0x32, 0x52, 0xa9, 0x9d, 0x6f, 0x2b, 0xf1, 0x68, 0xbd, 0x3e, 0xca, 0xfa, 0x02, 0xb7, 0xfd, 0x5e, 0x8c, 0x47, 0x26, 0x3a, 0x42, 0x90, 0xe8, 0xe6, 0xcf, 0x1f, 0x8b, 0xb3, 0x7d, 0xdf, 0xb8, 0x67, 0x07, 0x85, 0xac, 0xfa, 0x2c, 0x17, 0xec, 0xa1, 0x74, 0xfe, 0x91, 0xa3, 0xa8, 0xc9, 0xc7, 0x1a, 0x49, 0x59, 0xe6, 0x08, 0xbf, 0x8a, 0xd5, 0x9c, 0x5d, 0x9a, 0xc8, 0xa9, 0x46, 0x04, 0xbf, 0xfc, 0x42, 0x83, 0xaf, 0xa2, 0xed, 0x3a, 0x1d, 0xfe, 0xc2, 0xf2, 0xe1, 0x0d, 0x5c, 0x2d, 0xff, 0x5e, 0xcf, 0xbe, 0x26, 0x61, 0x6f, 0x4f, 0x0c, 0xd7, 0x9c, 0x53, 0x3c, 0x03, 0x05, 0x56, 0xe5, 0xf4, 0xc0, 0x5e, 0xdd, 0xb9, 0x8d, 0x08, 0xfc, 0x2b, 0x78, 0x02, 0xc9, 0x63, 0x80, 0xb3, 0x3e, 0x19, 0x89, 0x86, 0x80, 0x99, 0xe2, 0xea, 0x2d, 0x9f, 0xf4, 0x63, 0x00, 0x30, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00,
#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__DATA /* extended_example/sb_image1_enc.bin */ 0xad, 0xea, 0xc5, 0xdf, 0x69, 0xc6, 0x66, 0xf7, 0xef, 0x5c, 0xf4, 0x01, 0xf3, 0x21, 0x62, 0x14, 0xc6, 0x44, 0xaf, 0xb7, 0xd9, 0x65, 0xa1, 0xfe, 0xb0, 0x43, 0x75, 0x34, 0xec, 0x29, 0xd5, 0xef, 0x2e, 0xa7, 0x00, 0x90, 0x33, 0xf8, 0x95, 0xa8, 0x2d, 0xe5, 0x50, 0x1a, 0xa9, 0x8a, 0x17, 0xf9, 0xe1, 0x8e, 0xfb, 0x41, 0xa0, 0x13, 0xd1, 0x08, 0x27, 0xec, 0xa1, 0xbd, 0x43, 0xf4, 0xf1, 0xae, 0x40, 0x49, 0x17, 0x8e, 0x63, 0x19, 0x1e, 0xa6, 0xa8, 0x75, 0xc5, 0xd4, 0x9c, 0x54, 0x83, 0x81, 0x48, 0xe7, 0xc4, 0xbf, 0x94, 0x2e, 0xb6, 0x74, 0x66, 0x07, 0xc8, 0x57, 0xd2, 0x22, 0x4e, 0x5d, 0x9a, 0xa2, 0x1b, 0xfe, 0xa7, 0x0c, 0x42, 0xa9, 0xb2, 0x25, 0x01, 0x68, 0x0e, 0xc1, 0x88, 0x92,
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__DATA /* extended_example/sb_image2_enc.bin */ 0x1b, 0xd9, 0x01, 0x59, 0x53, 0xec, 0xa0, 0xd1, 0x77, 0x9c, 0x12, 0x14, 0xd1, 0x14, 0xaf, 0x24, 0x7e, 0x5f, 0xcf, 0x68, 0x43, 0x75, 0x81, 0x2b, 0x69, 0xbd, 0xb4, 0x59, 0x78, 0x1d, 0xe0, 0x83, 0xe9, 0xce, 0x99, 0x61, 0x08, 0x98, 0x9b, 0x72, 0x7e, 0x89, 0x17, 0x44, 0xb1, 0xb4, 0x5a, 0xb3, 0x42, 0x69, 0x8f, 0x61, 0x96, 0x83, 0x4b, 0x5c, 0x10, 0x90, 0xfc, 0xdd, 0xa4, 0xc8, 0x85, 0xf6, 0x79, 0x1a, 0x96, 0x12, 0x93, 0x0d, 0xa1, 0x9e, 0x9a, 0x4a, 0xfc, 0xa4, 0x9a, 0xfa, 0xb4, 0xf9, 0x78, 0xd2, 0x92, 0x8f, 0xc6, 0x72, 0xa6, 0x6d, 0x7d, 0xe6, 0xf5, 0x20, 0x4c, 0x95, 0x78, 0x63, 0xc8, 0xdb, 0x82, 0x91, 0x2c, 0x48, 0xf8, 0x94, 0xb0, 0x63, 0x51, 0xc6, 0x89, 0xc3, 0xba, 0x33,
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA /* extended_example/sb_image3_enc.bin */ 0x54, 0x84, 0x91, 0xbd, 0x10, 0xa2, 0x58, 0x05, 0xbf, 0x42, 0x3c, 0xac, 0x6f, 0x32, 0x42, 0x43, 0xb1, 0x2f, 0xb0, 0x9e, 0x1e, 0x70, 0xd8, 0x66, 0xfc, 0xdd, 0x0f, 0xc0, 0x44, 0x50, 0xe1, 0x7a, 0xcd, 0xa7, 0x47, 0x99, 0x8a, 0xf0, 0x03, 0xca, 0x28, 0xe2, 0x18, 0x39, 0x55, 0xa6, 0xfd, 0x0e, 0x05, 0x63, 0x93, 0xc3, 0x86, 0x81, 0x7a, 0xe5, 0xfa, 0xb7, 0xaa, 0xeb, 0x07, 0x8e, 0x8c, 0x5e, 0x07, 0x5c, 0x8e, 0xc7, 0x40, 0xa2, 0xde, 0x3c, 0xad, 0xe8, 0xe2, 0xbd, 0x8f, 0xec, 0xe5, 0xc5, 0x9d, 0xe7, 0xe5, 0x76, 0x86, 0x41, 0xfa, 0xf0, 0x3d, 0x13, 0xa2, 0x8e, 0x85, 0xae, 0x34, 0x59, 0x4d, 0x59, 0x48, 0xd8, 0x42, 0x0b, 0xc8, 0x7c, 0xff, 0x91, 0x55, 0x40, 0xef, 0x47, 0x13, 0xb0,
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA_LENGTH /* sb_key1_cert_test_10.bin */ 840
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA_LENGTH /* sb_key2_cert_test_10.bin */ 840
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA_LENGTH /* sb_content_cert_test_10.bin */ 972
#endif /* defined(CC_SB_X509_CERT_SUPPORTED) */
#elif defined(DX_PLAT_MPS2_PLUS)
#if defined(CORTEX_M33_FPGA)
#if defined(CC_SB_X509_CERT_SUPPORTED)
/************************************/
/*          MPS2 CM33 X509          */
/************************************/
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA /* example/sb_key_cert1_hbk0.bin */ 0x30, 0x82, 0x04, 0x6e, 0x30, 0x82, 0x02, 0xa6, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x04, 0x02, 0x3c, 0x84, 0x70, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x39, 0x30, 0x32, 0x30, 0x35, 0x31, 0x34, 0x31, 0x31, 0x34, 0x32, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x35, 0x35, 0x30, 0x33, 0x31, 0x34, 0x32, 0x30, 0x33, 0x39, 0x35, 0x36, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x07, 0x4b, 0x65, 0x79, 0x43, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x8f, 0x00, 0x30, 0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xc0, 0x68, 0x9a, 0xdf, 0x7a, 0x7f, 0x80, 0xad, 0xe1, 0xf1, 0x7d, 0xed, 0xbd, 0x3a, 0x23, 0xc6, 0x80, 0x87, 0x8d, 0x10, 0xf9, 0x99, 0x82, 0x6b, 0xbc, 0xec, 0x41, 0xaa, 0x94, 0xba, 0xcb, 0xbb, 0x6a, 0x59, 0xb2, 0x34, 0x41, 0x6b, 0x26, 0xf2, 0xb7, 0x5f, 0x96, 0xb0, 0x59, 0x93, 0xa9, 0x69, 0x8b, 0xac, 0x5c, 0x14, 0xb2, 0x51, 0xa8, 0x45, 0x2c, 0xd4, 0xe2, 0xfc, 0x37, 0xaf, 0x48, 0x16, 0x9f, 0x7e, 0xad, 0x37, 0x17, 0x0a, 0x2b, 0xc9, 0x2f, 0xf1, 0x5b, 0xdc, 0x78, 0xcf, 0x80, 0x92, 0xe4, 0xd8, 0x68, 0xb2, 0xc7, 0xd0, 0x4e, 0x9b, 0x66, 0x6e, 0x0a, 0x7a, 0xf7, 0x53, 0x60, 0x2e, 0x5e, 0xc6, 0x42, 0xa5, 0xdb, 0x94, 0xd1, 0x67, 0xde, 0x1f, 0x5d, 0xe3, 0x5e, 0x4b, 0x9d, 0xb8, 0x23, 0x3e, 0xdd, 0x79, 0xd8, 0xac, 0xf9, 0xd4, 0xbe, 0x77, 0xf1, 0xae, 0x8b, 0xbb, 0x94, 0xc7, 0xf8, 0xe8, 0xcc, 0x3a, 0xdb, 0x76, 0x8b, 0xb8, 0x5d, 0x99, 0x83, 0x29, 0x11, 0xe2, 0xfc, 0x38, 0x0d, 0x65, 0x95, 0x41, 0xbb, 0xd2, 0x8c, 0x77, 0x1b, 0x31, 0xb4, 0xb4, 0xb9, 0x4a, 0x59, 0x70, 0x72, 0x65, 0xb0, 0x32, 0xb8, 0x34, 0xd0, 0x4e, 0x62, 0x65, 0x9e, 0xfb, 0x49, 0x76, 0x64, 0x2f, 0x48, 0xe2, 0xe5, 0x00, 0xd7, 0xd1, 0xd4, 0xf4, 0xf6, 0x58, 0xc0, 0x04, 0x93, 0xee, 0x60, 0xd6, 0x94, 0xcd, 0xc5, 0x01, 0xc5, 0x77, 0xf7, 0xa0, 0xc7, 0xc3, 0x10, 0x5b, 0xdd, 0x36, 0x1f, 0x37, 0x07, 0xb1, 0x60, 0xef, 0x32, 0x7c, 0x86, 0xc3, 0xd7, 0x2a, 0x19, 0xfd, 0xac, 0xea, 0x2c, 0x34, 0x40, 0x8d, 0xdf, 0x78, 0x7e, 0xcf, 0xc7, 0xeb, 0x7d, 0x41, 0x25, 0xd3, 0x0d, 0x14, 0x5a, 0xdf, 0x90, 0xea, 0x8b, 0x68, 0xc3, 0x32, 0xaf, 0x53, 0xcf, 0x11, 0x24, 0x64, 0xbf, 0xf6, 0xf8, 0xdb, 0x09, 0x4e, 0xae, 0xbb, 0x32, 0x13, 0x9b, 0xc4, 0x90, 0xa7, 0x46, 0xc0, 0xe3, 0x88, 0xb7, 0x36, 0x78, 0x02, 0xab, 0x67, 0x61, 0x34, 0x17, 0x01, 0xc0, 0xe3, 0xc6, 0x2d, 0xfb, 0x3e, 0x5d, 0xf8, 0xd8, 0x43, 0x57, 0x4c, 0x84, 0x40, 0x21, 0x97, 0xc5, 0xe4, 0x06, 0x74, 0x3f, 0xa3, 0x22, 0x35, 0x48, 0x87, 0x33, 0xd8, 0xc5, 0x53, 0x76, 0x98, 0x0d, 0x02, 0x07, 0x48, 0x15, 0xf5, 0xf6, 0xb8, 0x56, 0x53, 0x08, 0x1c, 0xfe, 0xb4, 0x77, 0x4d, 0x9c, 0xa6, 0x1c, 0x99, 0xbb, 0x2b, 0x03, 0x03, 0xf3, 0x0c, 0x32, 0x84, 0x0f, 0x96, 0x67, 0x1a, 0x0d, 0x73, 0x13, 0xbc, 0x0c, 0xe4, 0x34, 0x41, 0x18, 0xb8, 0x09, 0x55, 0x2d, 0xe5, 0x75, 0x47, 0x03, 0xa3, 0xb2, 0xf2, 0x59, 0x94, 0x81, 0x27, 0x85, 0x98, 0x83, 0x94, 0x81, 0xc4, 0x8d, 0xc8, 0x80, 0xd6, 0xa9, 0xae, 0x02, 0x44, 0x45, 0x93, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x6e, 0x30, 0x6c, 0x30, 0x1a, 0x06, 0x03, 0x64, 0x01, 0x01, 0x01, 0x01, 0xff, 0x04, 0x10, 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x1e, 0x06, 0x03, 0x64, 0x01, 0x02, 0x01, 0x01, 0xff, 0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaa, 0x4d, 0xe1, 0xc6, 0x58, 0x5c, 0x3d, 0x03, 0x7a, 0x30, 0x2e, 0x06, 0x03, 0x64, 0x01, 0x03, 0x01, 0x01, 0xff, 0x04, 0x24, 0x05, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0x83, 0x84, 0x0e, 0x48, 0xa1, 0x75, 0x24, 0xa5, 0x0e, 0x89, 0xca, 0xc6, 0x84, 0xf8, 0xf3, 0x8e, 0xb5, 0xc3, 0x9c, 0x1c, 0x3c, 0xa3, 0x46, 0xbf, 0x59, 0xc1, 0x8a, 0xd9, 0x6d, 0xfa, 0x77, 0xf0, 0x4b, 0xfb, 0x27, 0x92, 0x97, 0xdc, 0x1e, 0x89, 0x37, 0xce, 0xcf, 0x65, 0xf5, 0x00, 0x6f, 0xdf, 0xcf, 0x16, 0x39, 0xbe, 0x84, 0x38, 0xf2, 0x54, 0x68, 0xbf, 0xf3, 0x5f, 0x59, 0x4f, 0x33, 0x31, 0x25, 0x92, 0xef, 0x52, 0xdc, 0x05, 0x36, 0x19, 0x5b, 0xc3, 0xe5, 0xf4, 0xe5, 0xb8, 0xd5, 0xa2, 0x58, 0xdb, 0x3d, 0x76, 0x2c, 0x6a, 0x9f, 0x7c, 0xf7, 0x34, 0xec, 0x54, 0x4c, 0x33, 0x62, 0xee, 0x09, 0x0b, 0x84, 0x80, 0xe4, 0x7c, 0x7b, 0x11, 0x1c, 0x72, 0xea, 0xca, 0xb5, 0xb8, 0x37, 0xfa, 0x61, 0xe3, 0xcc, 0xda, 0xdd, 0x3c, 0xbb, 0x4d, 0xcb, 0x1e, 0xa9, 0xfa, 0x23, 0x3d, 0x22, 0xe1, 0xe7, 0xc4, 0xc3, 0xb1, 0x5e, 0x35, 0xf2, 0x9a, 0x26, 0x93, 0x77, 0x50, 0x87, 0x40, 0xad, 0x68, 0xe5, 0xa2, 0x37, 0x3b, 0x53, 0x5b, 0xcd, 0x02, 0xd7, 0x3a, 0x79, 0x97, 0x1c, 0xf7, 0x92, 0x33, 0xdc, 0xe6, 0xdb, 0xa0, 0x77, 0xdd, 0xea, 0xec, 0x17, 0x83, 0x5a, 0xb1, 0x0c, 0x11, 0xac, 0xc4, 0x49, 0x6b, 0xe8, 0xe5, 0x86, 0xaf, 0x1c, 0x1b, 0xff, 0x5c, 0xfd, 0x8a, 0x2c, 0x0a, 0x72, 0x0b, 0x58, 0x11, 0x3b, 0x1e, 0x00, 0x57, 0xc7, 0x83, 0x99, 0x89, 0x2e, 0xd1, 0x0f, 0x7e, 0x7a, 0xbe, 0x2a, 0x80, 0x21, 0x98, 0x21, 0xee, 0x62, 0x45, 0x82, 0x77, 0xd8, 0x54, 0xc2, 0x61, 0xd4, 0x11, 0x95, 0x35, 0xaa, 0xa2, 0xdb, 0x4d, 0x0f, 0x05, 0x70, 0x4a, 0x88, 0x5b, 0x1b, 0x7e, 0xc3, 0x42, 0x5e, 0xba, 0x41, 0x4e, 0x76, 0xa3, 0x75, 0x5a, 0x97, 0x1b, 0xc0, 0x11, 0xab, 0x0b, 0x86, 0x96, 0x15, 0x2e, 0x13, 0xba, 0x53, 0x90, 0xd4, 0xe2, 0xcd, 0x76, 0x5c, 0x6d, 0xb0, 0x1b, 0x2c, 0xc8, 0x1f, 0x1c, 0x97, 0x49, 0x67, 0x55, 0xe5, 0xcd, 0x9d, 0x8e, 0x9d, 0xe5, 0x3a, 0x84, 0x37, 0xdc, 0xc5, 0xb5, 0xda, 0xf0, 0xce, 0x91, 0xc1, 0x6b, 0x4d, 0x6e, 0x23, 0x07, 0x2a, 0xc5, 0xcd, 0x9f, 0x30, 0xad, 0x7e, 0xd6, 0x5c, 0x6f, 0x8e, 0x75, 0x30, 0xc9, 0x58, 0x70, 0xa4, 0xc7, 0x02, 0xac, 0xe2, 0x48, 0x75, 0x64, 0xb7, 0x62, 0xf1, 0x62, 0xb6, 0x74, 0xc9, 0xa5, 0x52, 0x4f, 0x03, 0xf1, 0x40, 0x74, 0x3b, 0x95, 0xec, 0x1b, 0x8e, 0x46, 0x2f, 0xe4, 0x73, 0xed, 0xbf, 0x09, 0xab, 0x1c, 0xf5, 0xa3, 0x70, 0xab, 0x32, 0x9a, 0xcb, 0x87, 0x79, 0x56, 0x3e, 0xf8, 0x62, 0x25, 0xdb, 0x25, 0x41, 0x48, 0x87, 0x0b, 0x94, 0x1f, 0xd6, 0xa1, 0x88, 0x8b, 0x46, 0xa1, 0x95, 0x80, 0x11, 0xc7,
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA /* example/sb_key_cert2_hbk0.bin */ 0x30, 0x82, 0x04, 0x6f, 0x30, 0x82, 0x02, 0xa7, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x05, 0x00, 0xfa, 0xd7, 0x58, 0x63, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x39, 0x30, 0x32, 0x30, 0x35, 0x31, 0x34, 0x31, 0x31, 0x34, 0x32, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x35, 0x35, 0x30, 0x33, 0x31, 0x34, 0x32, 0x30, 0x33, 0x39, 0x35, 0x36, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x07, 0x4b, 0x65, 0x79, 0x43, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x8f, 0x00, 0x30, 0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x6e, 0x30, 0x6c, 0x30, 0x1a, 0x06, 0x03, 0x64, 0x01, 0x01, 0x01, 0x01, 0xff, 0x04, 0x10, 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x1e, 0x06, 0x03, 0x64, 0x01, 0x02, 0x01, 0x01, 0xff, 0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x30, 0x2e, 0x06, 0x03, 0x64, 0x01, 0x03, 0x01, 0x01, 0xff, 0x04, 0x24, 0x05, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0x90, 0x42, 0xa1, 0x9f, 0x68, 0x30, 0x4d, 0x40, 0xf2, 0x73, 0xb5, 0x6d, 0x37, 0x9c, 0x5f, 0xac, 0x57, 0xb8, 0x22, 0x0c, 0x4f, 0x2b, 0xa3, 0xd6, 0xc4, 0xc0, 0x23, 0xfc, 0xe5, 0xea, 0xe0, 0xa3, 0x15, 0x65, 0x6f, 0x13, 0xf6, 0xdb, 0xdf, 0xec, 0xad, 0x8e, 0x0e, 0x68, 0x22, 0xf7, 0x37, 0x1e, 0xfc, 0x0c, 0xc3, 0xcb, 0x84, 0x06, 0x4d, 0xc8, 0xa3, 0x2f, 0xbd, 0x21, 0x74, 0x1e, 0x8d, 0x80, 0x3c, 0xf2, 0x05, 0x7a, 0x2d, 0x14, 0xde, 0x41, 0x32, 0x58, 0x3c, 0x3e, 0x95, 0xe4, 0x6f, 0xfd, 0x25, 0x99, 0x08, 0x8d, 0x56, 0x0b, 0x73, 0x60, 0x59, 0x20, 0x38, 0x24, 0xd1, 0xb1, 0xf1, 0x8e, 0x88, 0x90, 0xbb, 0x4b, 0x82, 0x88, 0xfd, 0xe5, 0x53, 0x92, 0x7a, 0xf4, 0xac, 0xdc, 0x11, 0x20, 0x9d, 0x99, 0x79, 0xa7, 0x8e, 0xdf, 0x85, 0xf8, 0x67, 0x78, 0x4f, 0x89, 0x26, 0xf9, 0x93, 0xbf, 0xa2, 0xe1, 0xeb, 0xb8, 0x54, 0xb1, 0x0c, 0x99, 0x5b, 0x30, 0xea, 0x58, 0xd5, 0x94, 0xea, 0x2b, 0x97, 0x90, 0x3d, 0x0e, 0xc9, 0xe9, 0xff, 0x38, 0x71, 0x3a, 0xe5, 0xbe, 0xb9, 0x0d, 0x39, 0x13, 0xd8, 0x3c, 0x7c, 0x9e, 0xb5, 0x8a, 0x37, 0x2b, 0xbf, 0x3a, 0x6e, 0xf1, 0x4a, 0xb1, 0x57, 0xdc, 0x62, 0x18, 0xff, 0x56, 0xf6, 0x4f, 0x61, 0x0e, 0x7d, 0x64, 0x24, 0x36, 0xf1, 0xc7, 0x82, 0x45, 0x85, 0x9c, 0xde, 0xd2, 0x29, 0xd1, 0xba, 0xb1, 0xc9, 0xa0, 0x7b, 0xa9, 0xdf, 0x26, 0x29, 0xe9, 0xb1, 0x48, 0x24, 0x64, 0x8e, 0xad, 0xe2, 0x01, 0x74, 0x95, 0x15, 0x0c, 0x09, 0xd5, 0x02, 0xc4, 0x7f, 0x59, 0x64, 0x4e, 0x3c, 0x7a, 0x24, 0xc0, 0xaa, 0xb5, 0xd5, 0x5c, 0xf4, 0x83, 0x7a, 0xed, 0xda, 0x5f, 0x28, 0x37, 0xeb, 0x24, 0x3b, 0x91, 0xb0, 0x77, 0x29, 0x28, 0x14, 0x9d, 0xec, 0x2b, 0x94, 0x72, 0xa7, 0xdb, 0x5b, 0x06, 0xff, 0x3d, 0xef, 0x06, 0xa9, 0x09, 0x64, 0x99, 0x1a, 0x9f, 0xf0, 0xad, 0xd3, 0x5c, 0x34, 0x1c, 0xa0, 0x88, 0x0d, 0x00, 0x6f, 0xb0, 0x16, 0xb8, 0xd8, 0xdb, 0xfa, 0xa1, 0xc4, 0x14, 0xfd, 0x4c, 0x89, 0x6d, 0xea, 0x07, 0xd8, 0xf5, 0x14, 0x6f, 0xae, 0x4d, 0x51, 0x34, 0xa0, 0x9e, 0x15, 0xc3, 0xa4, 0x9b, 0xf7, 0xd8, 0x1c, 0x82, 0x33, 0xdb, 0xa2, 0x86, 0x8b, 0xe2, 0x29, 0x00, 0x8a, 0x22, 0x32, 0xf0, 0xa0, 0x52, 0x6f, 0xd1, 0xce, 0x28, 0xfe, 0x2f, 0x62, 0x96, 0x66, 0x8d, 0xe4, 0xe0, 0x47, 0x0b, 0x47, 0xc6, 0xa5, 0xe1, 0xf7, 0xd5, 0x03, 0x65, 0x6a, 0x20, 0xcf, 0xce, 0xda, 0x90, 0x6e, 0x2b, 0xa4, 0xa1, 0xb5, 0x67, 0xc4, 0xed, 0xa2, 0xfc, 0xa1, 0x20, 0x59, 0xeb, 0x08, 0xf8, 0x3b, 0xf1, 0x74, 0x3b, 0x80, 0x9e, 0x47, 0xcd, 0x59, 0x85,
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA /* example/sb_content_cert.bin */ 0x30, 0x82, 0x04, 0xde, 0x30, 0x82, 0x03, 0x16, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x04, 0x78, 0x07, 0xcc, 0xe8, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x39, 0x30, 0x32, 0x30, 0x35, 0x31, 0x34, 0x31, 0x31, 0x34, 0x32, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x35, 0x35, 0x30, 0x33, 0x31, 0x34, 0x32, 0x30, 0x33, 0x39, 0x35, 0x36, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x07, 0x43, 0x6e, 0x74, 0x43, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x8f, 0x00, 0x30, 0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xdd, 0x30, 0x81, 0xda, 0x30, 0x1a, 0x06, 0x03, 0x64, 0x02, 0x01, 0x01, 0x01, 0xff, 0x04, 0x10, 0x63, 0x63, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x03, 0x00, 0x30, 0x1e, 0x06, 0x03, 0x64, 0x02, 0x02, 0x01, 0x01, 0xff, 0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x30, 0x81, 0x9b, 0x06, 0x03, 0x64, 0x02, 0x04, 0x01, 0x01, 0xff, 0x04, 0x81, 0x90, 0x05, 0x00, 0x00, 0x00, 0x43, 0x62, 0x36, 0x5b, 0xac, 0xdb, 0x25, 0x94, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x00, 0x20, 0x38, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x10, 0x20, 0x38, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x20, 0x20, 0x38, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0x68, 0x77, 0x40, 0xb7, 0xe3, 0x03, 0xef, 0xf3, 0xc9, 0xf0, 0xbd, 0xe0, 0x2d, 0x45, 0xc7, 0x83, 0xff, 0xd3, 0x0a, 0xa0, 0xfe, 0x4d, 0xde, 0xc6, 0xa0, 0x61, 0x8e, 0xba, 0xe5, 0x7b, 0x74, 0x22, 0xdd, 0xbe, 0x53, 0x78, 0x9c, 0x28, 0x02, 0x3a, 0x56, 0xdf, 0x7a, 0x4f, 0x65, 0x57, 0x43, 0x7d, 0x99, 0xa5, 0xe1, 0x23, 0xef, 0x20, 0x7b, 0xb8, 0x96, 0x31, 0x3f, 0xec, 0xb3, 0xb3, 0x61, 0x6e, 0xa6, 0x6c, 0x71, 0x8f, 0x8a, 0x0e, 0x6f, 0xde, 0xaf, 0x6d, 0x20, 0xfb, 0xfb, 0x32, 0xde, 0x30, 0xfe, 0xd5, 0x3a, 0x5b, 0xd5, 0x0b, 0xae, 0xa4, 0x17, 0xb9, 0xee, 0x29, 0xb0, 0x6f, 0xbd, 0x8e, 0xd8, 0x52, 0x9a, 0x2a, 0xd1, 0x28, 0x6b, 0xbf, 0x8f, 0x06, 0x21, 0x3f, 0x4b, 0x12, 0x4e, 0x8f, 0x27, 0xfb, 0x83, 0xe3, 0x3d, 0x95, 0xcf, 0x17, 0x00, 0xdc, 0xcc, 0xf9, 0x91, 0x8f, 0x04, 0xbc, 0x2e, 0xb7, 0x30, 0x70, 0x54, 0x2d, 0xe9, 0xe1, 0x55, 0xae, 0x99, 0x83, 0x81, 0x9c, 0xc6, 0x0c, 0x04, 0x7d, 0x0d, 0x44, 0x4a, 0xb8, 0xe0, 0x67, 0x97, 0x3a, 0xf9, 0x59, 0x33, 0x45, 0x3f, 0x20, 0x0b, 0x5c, 0xc6, 0xcc, 0x1d, 0x0e, 0x71, 0x33, 0xa3, 0x4b, 0x06, 0x65, 0xfc, 0xc2, 0xfb, 0xca, 0x2b, 0x0d, 0xdc, 0x67, 0xb8, 0x9b, 0x28, 0x9b, 0xf0, 0x74, 0xf3, 0x25, 0x5f, 0xd4, 0x27, 0xcb, 0x4b, 0x99, 0x95, 0xb8, 0xe2, 0xb4, 0xec, 0xc3, 0xca, 0x55, 0x7c, 0x96, 0x69, 0x7a, 0x36, 0x20, 0x73, 0x6f, 0x74, 0xdc, 0x47, 0x3e, 0x23, 0xd5, 0x1c, 0xc2, 0x84, 0x43, 0x71, 0xfc, 0xb4, 0x75, 0xb4, 0x68, 0x83, 0x28, 0x5b, 0x1a, 0xd1, 0x6d, 0x44, 0x7a, 0xb0, 0x8f, 0x82, 0xe0, 0x1d, 0x4c, 0x87, 0x74, 0xa4, 0xcf, 0x65, 0xf6, 0x57, 0x11, 0x46, 0xba, 0xb6, 0x70, 0x32, 0xe8, 0x68, 0x9b, 0x8c, 0x86, 0x95, 0x0e, 0xc9, 0xac, 0x5e, 0xcb, 0xbd, 0x9b, 0xe8, 0x5d, 0xde, 0xcf, 0x04, 0x7d, 0xd7, 0x17, 0xb7, 0xa0, 0x4f, 0x3e, 0xef, 0x69, 0xaa, 0x96, 0x31, 0x57, 0x55, 0x07, 0x02, 0x8c, 0xc1, 0xeb, 0x9e, 0xf0, 0xdc, 0x3c, 0x56, 0x0c, 0x17, 0xd0, 0x05, 0x32, 0xf3, 0xd6, 0x59, 0xf3, 0xcc, 0x87, 0xbc, 0x6d, 0x42, 0x58, 0x43, 0x9e, 0x4b, 0xd5, 0x11, 0x17, 0x99, 0xba, 0x57, 0x80, 0xc5, 0xb0, 0x68, 0x56, 0xe1, 0x83, 0x5f, 0x33, 0x65, 0x66, 0x74, 0xe7, 0xb6, 0xfc, 0xe4, 0x6c, 0xaf, 0xf9, 0x12, 0x0d, 0xc9, 0x39, 0x7d, 0x29, 0x4e, 0xa9, 0x19, 0xd2, 0xe1, 0x03, 0xd6, 0x94, 0x04, 0x55, 0x43, 0x7e, 0x4d, 0xce, 0x05, 0x5b, 0xe8, 0x99, 0x26, 0x76, 0x37, 0xa6, 0x3f, 0x59, 0x77, 0x53, 0xd3, 0x23, 0xbc, 0x5e, 0x7e, 0x43, 0xb4, 0x2a, 0x3a, 0x16, 0x20, 0x70, 0x3b, 0x6f, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00,
#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__DATA /* example/sb_image1_enc.bin */ 0x8d, 0xe5, 0xa3, 0x63, 0xfc, 0x4e, 0x84, 0x0e, 0xc2, 0x58, 0x36, 0x41, 0x70, 0x76, 0x7c, 0xde, 0xa0, 0x5f, 0x52, 0xcc, 0xd3, 0x52, 0xf9, 0xdc, 0xed, 0xef, 0x62, 0x1b, 0x23, 0x62, 0x87, 0xb0, 0xdc, 0x59, 0x58, 0x10, 0x97, 0x90, 0xb1, 0xbc, 0x76, 0xf3, 0xad, 0x0c, 0xb0, 0xd2, 0xdc, 0x18, 0x42, 0x7f, 0xbc, 0x66, 0xe7, 0x77, 0xf6, 0xcd, 0xc9, 0x94, 0x04, 0x0a, 0x3f, 0x84, 0x98, 0x05, 0x6e, 0x0e, 0x11, 0xc4, 0xbb, 0x06, 0xb2, 0xc9, 0x7f, 0x12, 0x4b, 0xd9, 0xce, 0x4d, 0x66, 0x3b, 0x8b, 0x26, 0x24, 0xe7, 0x85, 0xe2, 0xc5, 0xf2, 0xdf, 0xd1, 0x75, 0xf2, 0x94, 0xea, 0x75, 0x3b, 0xaa, 0x6d, 0x15, 0x68, 0xe8, 0x35, 0x18, 0x83, 0x02, 0x6c, 0x96, 0x4b, 0xb2, 0x8d, 0x17, 0xa6,
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__DATA /* example/sb_image2_enc.bin */ 0x9f, 0x5e, 0xa3, 0xf1, 0x8e, 0x5d, 0x0f, 0x23, 0x4d, 0x3d, 0x57, 0x33, 0xc9, 0x4c, 0x03, 0x58, 0xaf, 0xc8, 0x45, 0x90, 0xa6, 0xf1, 0x5c, 0x06, 0xee, 0x3a, 0xa6, 0x27, 0x05, 0xfb, 0x34, 0xa9, 0x4e, 0xb3, 0xbc, 0xc4, 0x8f, 0xb9, 0xdf, 0xdd, 0x84, 0x76, 0x05, 0xc1, 0x7b, 0x82, 0x4d, 0x63, 0xe0, 0x66, 0xd6, 0xde, 0x10, 0x3b, 0xcb, 0x72, 0xac, 0xa0, 0x74, 0x0f, 0xaf, 0x2f, 0xf2, 0xbc, 0xcb, 0x6e, 0x41, 0xab, 0x1e, 0xd8, 0x4b, 0xa9, 0xdd, 0x67, 0x03, 0x54, 0x83, 0xdb, 0xd7, 0xe8, 0x74, 0x3b, 0xe3, 0x77, 0xc2, 0xce, 0x66, 0x0e, 0x2d, 0x72, 0xcf, 0x2d, 0x91, 0x90, 0xe1, 0xd5, 0x27, 0xca, 0x37, 0xae, 0x94, 0xe1, 0x76, 0xac, 0x36, 0xcb, 0x66, 0xe7, 0xa6, 0x50, 0x8f, 0x92,
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA /* example/sb_image3_enc.bin */ 0x69, 0x92, 0xf9, 0xf3, 0x31, 0x75, 0x0a, 0xf1, 0x62, 0x3e, 0xb9, 0x33, 0x84, 0x25, 0xea, 0x5c, 0x9c, 0x66, 0x35, 0x3b, 0x33, 0x01, 0x30, 0xc9, 0x7a, 0x85, 0xac, 0x63, 0xee, 0x49, 0x39, 0xd4, 0x75, 0xd6, 0xde, 0xe0, 0xc7, 0x79, 0x4d, 0x01, 0xf4, 0x46, 0xff, 0x92, 0x0d, 0xec, 0x75, 0xa5, 0x15, 0x28, 0xab, 0x3d, 0x53, 0x7e, 0xa8, 0xa3, 0xbd, 0xf1, 0x93, 0x7e, 0x8a, 0xa1, 0xbe, 0x06, 0xaf, 0x2f, 0x2d, 0x9f, 0xd0, 0x2a, 0x7c, 0x60, 0x43, 0x5b, 0xe7, 0x05, 0x57, 0x15, 0xf1, 0xdf, 0x19, 0xc4, 0x22, 0x2d, 0x71, 0x14, 0x58, 0x52, 0x74, 0x28, 0xc0, 0x44, 0x46, 0xf2, 0xee, 0x95, 0x0a, 0x17, 0x80, 0x2f, 0xd1, 0x36, 0xf6, 0xd5, 0xd2, 0xb7, 0x14, 0x33, 0x3a, 0xef, 0xce, 0xc7,
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA_LENGTH /* sb_key_cert1_hbk0.bin */ 1138
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA_LENGTH /* sb_key_cert2_hbk0.bin */ 1139
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA_LENGTH /* sb_content_cert.bin */ 1276
#else
/************************************/
/*          MPS2 CM33 NOT_X509      */
/************************************/
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA /* example/sb_key_cert1_hbk0.bin */ 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x68, 0x9a, 0xdf, 0x7a, 0x7f, 0x80, 0xad, 0xe1, 0xf1, 0x7d, 0xed, 0xbd, 0x3a, 0x23, 0xc6, 0x80, 0x87, 0x8d, 0x10, 0xf9, 0x99, 0x82, 0x6b, 0xbc, 0xec, 0x41, 0xaa, 0x94, 0xba, 0xcb, 0xbb, 0x6a, 0x59, 0xb2, 0x34, 0x41, 0x6b, 0x26, 0xf2, 0xb7, 0x5f, 0x96, 0xb0, 0x59, 0x93, 0xa9, 0x69, 0x8b, 0xac, 0x5c, 0x14, 0xb2, 0x51, 0xa8, 0x45, 0x2c, 0xd4, 0xe2, 0xfc, 0x37, 0xaf, 0x48, 0x16, 0x9f, 0x7e, 0xad, 0x37, 0x17, 0x0a, 0x2b, 0xc9, 0x2f, 0xf1, 0x5b, 0xdc, 0x78, 0xcf, 0x80, 0x92, 0xe4, 0xd8, 0x68, 0xb2, 0xc7, 0xd0, 0x4e, 0x9b, 0x66, 0x6e, 0x0a, 0x7a, 0xf7, 0x53, 0x60, 0x2e, 0x5e, 0xc6, 0x42, 0xa5, 0xdb, 0x94, 0xd1, 0x67, 0xde, 0x1f, 0x5d, 0xe3, 0x5e, 0x4b, 0x9d, 0xb8, 0x23, 0x3e, 0xdd, 0x79, 0xd8, 0xac, 0xf9, 0xd4, 0xbe, 0x77, 0xf1, 0xae, 0x8b, 0xbb, 0x94, 0xc7, 0xf8, 0xe8, 0xcc, 0x3a, 0xdb, 0x76, 0x8b, 0xb8, 0x5d, 0x99, 0x83, 0x29, 0x11, 0xe2, 0xfc, 0x38, 0x0d, 0x65, 0x95, 0x41, 0xbb, 0xd2, 0x8c, 0x77, 0x1b, 0x31, 0xb4, 0xb4, 0xb9, 0x4a, 0x59, 0x70, 0x72, 0x65, 0xb0, 0x32, 0xb8, 0x34, 0xd0, 0x4e, 0x62, 0x65, 0x9e, 0xfb, 0x49, 0x76, 0x64, 0x2f, 0x48, 0xe2, 0xe5, 0x00, 0xd7, 0xd1, 0xd4, 0xf4, 0xf6, 0x58, 0xc0, 0x04, 0x93, 0xee, 0x60, 0xd6, 0x94, 0xcd, 0xc5, 0x01, 0xc5, 0x77, 0xf7, 0xa0, 0xc7, 0xc3, 0x10, 0x5b, 0xdd, 0x36, 0x1f, 0x37, 0x07, 0xb1, 0x60, 0xef, 0x32, 0x7c, 0x86, 0xc3, 0xd7, 0x2a, 0x19, 0xfd, 0xac, 0xea, 0x2c, 0x34, 0x40, 0x8d, 0xdf, 0x78, 0x7e, 0xcf, 0xc7, 0xeb, 0x7d, 0x41, 0x25, 0xd3, 0x0d, 0x14, 0x5a, 0xdf, 0x90, 0xea, 0x8b, 0x68, 0xc3, 0x32, 0xaf, 0x53, 0xcf, 0x11, 0x24, 0x64, 0xbf, 0xf6, 0xf8, 0xdb, 0x09, 0x4e, 0xae, 0xbb, 0x32, 0x13, 0x9b, 0xc4, 0x90, 0xa7, 0x46, 0xc0, 0xe3, 0x88, 0xb7, 0x36, 0x78, 0x02, 0xab, 0x67, 0x61, 0x34, 0x17, 0x01, 0xc0, 0xe3, 0xc6, 0x2d, 0xfb, 0x3e, 0x5d, 0xf8, 0xd8, 0x43, 0x57, 0x4c, 0x84, 0x40, 0x21, 0x97, 0xc5, 0xe4, 0x06, 0x74, 0x3f, 0xa3, 0x22, 0x35, 0x48, 0x87, 0x33, 0xd8, 0xc5, 0x53, 0x76, 0x98, 0x0d, 0x02, 0x07, 0x48, 0x15, 0xf5, 0xf6, 0xb8, 0x56, 0x53, 0x08, 0x1c, 0xfe, 0xb4, 0x77, 0x4d, 0x9c, 0xa6, 0x1c, 0x99, 0xbb, 0x2b, 0x03, 0x03, 0xf3, 0x0c, 0x32, 0x84, 0x0f, 0x96, 0x67, 0x1a, 0x0d, 0x73, 0x13, 0xbc, 0x0c, 0xe4, 0x34, 0x41, 0x18, 0xb8, 0x09, 0x55, 0x2d, 0xe5, 0x75, 0x47, 0x03, 0xa3, 0xb2, 0xf2, 0x59, 0x94, 0x81, 0x27, 0x85, 0x98, 0x83, 0x94, 0x81, 0xc4, 0x8d, 0xc8, 0x80, 0xd6, 0xa9, 0xae, 0x02, 0x44, 0x45, 0x93, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaa, 0x4d, 0xe1, 0xc6, 0x58, 0x5c, 0x3d, 0x03, 0x7a, 0x05, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0x27, 0x57, 0xa3, 0x16, 0x94, 0xa1, 0x75, 0x74, 0xec, 0x9f, 0x91, 0x7a, 0xdf, 0x09, 0xe9, 0x52, 0x1e, 0xce, 0x17, 0x3a, 0xd4, 0x1c, 0xfc, 0x75, 0x6e, 0x87, 0x09, 0x77, 0x53, 0x31, 0x79, 0x11, 0xb0, 0xc1, 0xed, 0x38, 0x8d, 0x81, 0x15, 0x4d, 0x14, 0xf4, 0xd7, 0x9e, 0x8b, 0x4f, 0x0b, 0x62, 0x48, 0x5f, 0x8c, 0x78, 0x71, 0x9f, 0x1b, 0xca, 0x89, 0xb6, 0x3b, 0x62, 0x21, 0xeb, 0x3a, 0xd5, 0x8f, 0xee, 0x34, 0xdc, 0x38, 0xf9, 0xfc, 0x6f, 0x8c, 0xa4, 0xf9, 0xaa, 0x85, 0x30, 0x09, 0x3e, 0x31, 0x21, 0x1a, 0xc7, 0xb4, 0x1c, 0xfa, 0x68, 0x3e, 0xa2, 0xee, 0x68, 0xc3, 0xb7, 0x40, 0xe5, 0xb0, 0x73, 0xb8, 0x20, 0x4c, 0xa8, 0xb3, 0x31, 0x05, 0x18, 0x7e, 0x59, 0x31, 0xdf, 0x2b, 0xa7, 0xfb, 0xdd, 0x8a, 0x6e, 0x6d, 0x0a, 0x43, 0x76, 0x76, 0xa1, 0x52, 0xf1, 0xbe, 0x6a, 0x59, 0x33, 0xef, 0x44, 0x5a, 0x1a, 0xe3, 0x02, 0xe6, 0x60, 0xfc, 0xae, 0xee, 0xe6, 0x09, 0xb7, 0xbb, 0xce, 0xe0, 0x18, 0xba, 0x68, 0xd1, 0x04, 0xd3, 0x21, 0x9f, 0xe9, 0x12, 0xe0, 0x99, 0x5b, 0x76, 0xc2, 0x4d, 0x91, 0x6a, 0x70, 0xe0, 0x4d, 0x5d, 0x09, 0x0a, 0x8b, 0x75, 0x1e, 0x5c, 0xc9, 0x99, 0x23, 0x08, 0xe6, 0xb5, 0xc4, 0x56, 0x46, 0x5d, 0x9f, 0x51, 0x82, 0xa6, 0x0d, 0x91, 0x3a, 0xc2, 0x2f, 0xa1, 0xa3, 0x5a, 0x3d, 0xe9, 0x65, 0x0f, 0x04, 0x3d, 0x9c, 0xcc, 0x67, 0x90, 0xab, 0x6a, 0xba, 0x58, 0xfa, 0x49, 0x0d, 0x68, 0xd6, 0x69, 0x60, 0x1a, 0x9a, 0xa6, 0x03, 0xe2, 0xcc, 0x89, 0xae, 0xca, 0xb5, 0x27, 0x7b, 0x37, 0x00, 0x03, 0x9c, 0x83, 0x1a, 0x79, 0xbd, 0xc3, 0x8c, 0x31, 0xfd, 0x11, 0x25, 0xfc, 0x25, 0x53, 0x53, 0xfe, 0x3b, 0x58, 0xab, 0x60, 0xd6, 0x71, 0xb3, 0x1a, 0x62, 0x19, 0xe7, 0x56, 0x3c, 0x15, 0xbc, 0xef, 0x03, 0x09, 0x71, 0x57, 0xbb, 0x0e, 0xe9, 0x1b, 0xfe, 0x16, 0xdb, 0xf9, 0x31, 0x0e, 0x95, 0xa6, 0x46, 0x8d, 0xdd, 0x32, 0x9f, 0x53, 0xb9, 0x91, 0xd9, 0xa1, 0xc8, 0x02, 0xaa, 0x06, 0x89, 0xd5, 0x7f, 0x2b, 0x5e, 0x53, 0xd8, 0x02, 0x98, 0xee, 0xfc, 0x82, 0xe1, 0x0c, 0xd5, 0x10, 0xb1, 0xe3, 0x64, 0x1d, 0xe6, 0xcb, 0xd1, 0xc9, 0x50, 0xf3, 0x53, 0x4e, 0x45, 0x2f, 0x97, 0x37, 0x43, 0x8d, 0x62, 0x0b, 0xc9, 0x16, 0x58, 0x66, 0x73, 0x43, 0x14, 0xc1, 0x90, 0xc1, 0x3e, 0xe6, 0xb6, 0x75, 0xc3, 0xb6, 0x6b, 0x27, 0x9b, 0x3a, 0x0d, 0x9a, 0xfb, 0x10, 0x7d, 0x3b, 0x96, 0x6d, 0x74, 0x4b, 0x71, 0x55, 0x3b, 0x9b, 0xa6, 0xaa, 0xb9, 0x3b, 0x09, 0x4d, 0x5b, 0x73, 0x5c, 0x6d, 0xee, 0xb6, 0xb4, 0x9e, 0x84, 0x68, 0x85, 0x70, 0x2d, 0xbc, 0x15,
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA /* example/sb_key_cert2_hbk0.bin */ 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x05, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0x21, 0xa9, 0x33, 0x43, 0xa6, 0x82, 0x14, 0xea, 0x98, 0x86, 0x40, 0x71, 0x08, 0x47, 0x19, 0x3e, 0xba, 0x21, 0x64, 0xe5, 0xd2, 0x74, 0xa2, 0xdc, 0x93, 0xa8, 0x2f, 0x14, 0x71, 0xee, 0x7d, 0x7b, 0x1b, 0xba, 0x69, 0x93, 0xde, 0xf7, 0xa9, 0xbd, 0xfe, 0xc1, 0x0c, 0xcc, 0x29, 0xf7, 0x36, 0xa1, 0x03, 0x7e, 0xdb, 0xb1, 0x56, 0x67, 0xa0, 0xec, 0xaf, 0x05, 0x64, 0xcb, 0x15, 0x0f, 0xd1, 0xdb, 0xd5, 0x58, 0xcd, 0x38, 0x71, 0xed, 0xd8, 0x66, 0xa0, 0xb6, 0x45, 0xaf, 0x21, 0x94, 0x0d, 0xdc, 0xbf, 0x98, 0x8e, 0x6b, 0x94, 0xf6, 0xd5, 0x90, 0xe5, 0xd0, 0xa7, 0xcd, 0x78, 0x0a, 0x2c, 0xf7, 0x00, 0xb7, 0x1c, 0x5c, 0xc1, 0x7e, 0x5f, 0x0f, 0x94, 0xec, 0xac, 0xbc, 0xf0, 0x25, 0x6a, 0xd3, 0xc9, 0x0d, 0x4e, 0xf3, 0xee, 0x9d, 0xdd, 0xab, 0x12, 0x71, 0x5b, 0xed, 0x3c, 0xcb, 0x04, 0x64, 0x0a, 0x42, 0xf2, 0x10, 0xf8, 0x0d, 0x7e, 0xfe, 0x7f, 0x9a, 0xf7, 0x4e, 0xc8, 0x43, 0x54, 0xbe, 0x2c, 0x2e, 0xc7, 0x66, 0x04, 0xa7, 0xc9, 0xad, 0x71, 0xc4, 0x19, 0x96, 0x96, 0x7b, 0xc3, 0xec, 0xc5, 0x01, 0xf6, 0x32, 0xe0, 0x34, 0x60, 0x0b, 0x53, 0xbd, 0x96, 0x21, 0x3a, 0xb9, 0x47, 0xe2, 0x1f, 0x4b, 0xad, 0xbd, 0x92, 0xc7, 0xd8, 0x12, 0xaa, 0x65, 0x7c, 0xea, 0xae, 0x7b, 0xc9, 0x32, 0xb1, 0x5c, 0x48, 0x6c, 0xe9, 0xec, 0xc4, 0xa2, 0x64, 0xfe, 0x7a, 0x49, 0xd6, 0x8c, 0xbe, 0xb3, 0x8b, 0xac, 0x47, 0xd2, 0xc9, 0xfc, 0xd6, 0xbb, 0x11, 0x13, 0x99, 0x1d, 0x82, 0xee, 0xe2, 0x26, 0xf0, 0x9c, 0xcf, 0x64, 0x9a, 0x5b, 0x8b, 0x1b, 0x9a, 0x7e, 0x1c, 0xbd, 0x70, 0xac, 0x47, 0x23, 0x15, 0xdf, 0x07, 0x1f, 0x48, 0x09, 0x49, 0xc1, 0x5d, 0x2a, 0x15, 0x96, 0x62, 0xb4, 0x2d, 0x9c, 0x87, 0x7a, 0x60, 0x4e, 0x7c, 0x96, 0xf7, 0x99, 0x49, 0xe1, 0x5a, 0x2f, 0x07, 0x10, 0x7f, 0xda, 0x75, 0xf9, 0x77, 0x4e, 0x6f, 0xe7, 0x92, 0x0e, 0xc7, 0xae, 0x95, 0x3f, 0xb6, 0xfa, 0x8a, 0x0f, 0xe2, 0xef, 0xd8, 0xe5, 0xd5, 0x8d, 0x37, 0xd0, 0x11, 0x8d, 0x70, 0x69, 0x64, 0xa2, 0x79, 0x83, 0xde, 0xf1, 0xce, 0x93, 0x13, 0x72, 0xdb, 0x63, 0x4f, 0xdb, 0x8d, 0x08, 0x4f, 0x15, 0xef, 0xc8, 0x84, 0x00, 0xf9, 0x77, 0x33, 0x2d, 0x17, 0xa5, 0x9a, 0x35, 0xcb, 0x55, 0xa1, 0x8f, 0xc7, 0x60, 0xe3, 0x4e, 0xa7, 0x8b, 0xad, 0xef, 0xf6, 0x68, 0x5f, 0x3e, 0x5d, 0x35, 0xb2, 0x61, 0x20, 0x54, 0x9a, 0xda, 0x08, 0x39, 0x16, 0x3b, 0x34, 0x83, 0x10, 0x02, 0x36, 0x54, 0xa1, 0x4d, 0x4d, 0x5f, 0x9e, 0x78, 0x23, 0x1f, 0x2d, 0x8d, 0xb1, 0x6e, 0x41, 0x8e, 0xc9, 0x62, 0x99, 0x01, 0x55, 0x91,
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA /* example/sb_content_cert.bin */ 0x63, 0x63, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x03, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x05, 0x00, 0x00, 0x00, 0xef, 0xcc, 0x12, 0xe8, 0x10, 0x6e, 0x6c, 0x48, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x00, 0x20, 0x38, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x10, 0x20, 0x38, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x20, 0x20, 0x38, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x41, 0xef, 0x38, 0x53, 0xaf, 0x5c, 0xd2, 0xbb, 0x25, 0x87, 0x0e, 0xc6, 0xb4, 0x87, 0x9b, 0xc1, 0x02, 0x83, 0xee, 0x36, 0x4c, 0xb0, 0x11, 0x98, 0xe5, 0x23, 0x5d, 0xaf, 0xa8, 0xda, 0x25, 0x65, 0xd4, 0x8f, 0xd8, 0x74, 0xc2, 0x2c, 0x99, 0x93, 0xfa, 0x78, 0x73, 0x2f, 0x6b, 0x09, 0xce, 0x23, 0xf1, 0x64, 0xc8, 0x7b, 0x0a, 0x70, 0x54, 0x90, 0x43, 0x5b, 0xa7, 0x20, 0xa4, 0xa4, 0xf6, 0x1d, 0x97, 0x92, 0xcf, 0x24, 0xb7, 0x2d, 0x83, 0xdd, 0x5f, 0xc8, 0x3d, 0xeb, 0x5e, 0x24, 0xb7, 0x46, 0xd8, 0xd8, 0xd8, 0xaa, 0xeb, 0xe6, 0x92, 0x12, 0x48, 0xcf, 0x9c, 0x6e, 0x70, 0xaa, 0x1b, 0x6c, 0xc8, 0x41, 0xb2, 0xae, 0x87, 0xf2, 0x20, 0x70, 0x44, 0x05, 0x3e, 0xf2, 0x0a, 0x88, 0x90, 0xe8, 0xc8, 0xf9, 0x7c, 0xa4, 0x63, 0xe0, 0xa3, 0x83, 0x69, 0x33, 0xb3, 0x0b, 0xad, 0x4f, 0x72, 0xf6, 0xe5, 0x78, 0xfb, 0x73, 0xe3, 0xb3, 0xf9, 0x2c, 0x0f, 0xf4, 0x2c, 0xf6, 0xd5, 0x10, 0x12, 0xa3, 0x0d, 0x78, 0xe7, 0x75, 0x73, 0x11, 0x31, 0x3e, 0xab, 0xdd, 0xd6, 0x8b, 0x88, 0x90, 0x58, 0x2e, 0xc1, 0x1f, 0x49, 0x6a, 0x5d, 0xf1, 0xb1, 0xb8, 0x5f, 0x55, 0x77, 0x80, 0x66, 0x4b, 0xf2, 0x05, 0xa5, 0x15, 0x2d, 0x47, 0x23, 0x41, 0xab, 0x69, 0x3b, 0x11, 0xbe, 0x89, 0xea, 0x6e, 0x5f, 0x78, 0x60, 0x01, 0x9b, 0xca, 0xbc, 0xfd, 0x65, 0xf7, 0xa3, 0x71, 0x52, 0x53, 0xc5, 0xb0, 0xe3, 0x45, 0xce, 0x16, 0x08, 0xe0, 0xa5, 0xca, 0x5d, 0xe1, 0x8d, 0x4b, 0xc9, 0x6c, 0x04, 0xda, 0x90, 0x65, 0xb7, 0xfc, 0x2a, 0x85, 0xef, 0xbd, 0x13, 0x03, 0xc8, 0xff, 0x73, 0x55, 0x75, 0x89, 0x61, 0x2d, 0xab, 0x9a, 0xb3, 0x3c, 0x16, 0x4f, 0x79, 0x65, 0xf9, 0x1b, 0xc1, 0x96, 0x95, 0xc3, 0xe0, 0xa3, 0xb6, 0xd6, 0x05, 0xd3, 0x24, 0x7c, 0x8b, 0x5d, 0x72, 0x7f, 0x01, 0xdf, 0x84, 0xdb, 0x06, 0x81, 0xb9, 0x53, 0x2e, 0x9e, 0xc7, 0x7f, 0x73, 0xda, 0xe9, 0xa0, 0x3f, 0x03, 0x44, 0xf5, 0x43, 0x0d, 0x6c, 0xa3, 0x60, 0x72, 0x9e, 0x8a, 0xaa, 0x57, 0x21, 0x27, 0x36, 0xae, 0xe2, 0x02, 0x96, 0xd2, 0x6f, 0x97, 0x7f, 0x0b, 0x70, 0x98, 0xa3, 0xc2, 0xae, 0xbb, 0x07, 0xc5, 0x30, 0xea, 0xf5, 0x82, 0x07, 0x70, 0xc6, 0xa8, 0x92, 0xeb, 0x0a, 0x1e, 0x73, 0x5c, 0x65, 0x60, 0xef, 0x44, 0x22, 0x6a, 0xb2, 0x7a, 0x19, 0x77, 0xc8, 0xf4, 0xea, 0xb1, 0x00, 0x6c, 0x5a, 0x31, 0x9b, 0xa7, 0x25, 0x94, 0xaa, 0x4e, 0xa4, 0x52, 0xd5, 0x51, 0x6d, 0x7d, 0x16, 0xfe, 0x84, 0xd2, 0x8d, 0x92, 0x12, 0x4a, 0x76, 0xd7, 0x14, 0xed, 0xf3, 0x15, 0xb2, 0xab, 0x0b, 0xec, 0x29, 0x67, 0x51, 0xda, 0x7e, 0xa3, 0x00, 0x30, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00,
#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__DATA /* example/sb_image1_enc.bin */ 0x0f, 0xa7, 0xd2, 0xa6, 0x8f, 0x5f, 0x7f, 0x82, 0x86, 0xa8, 0xa2, 0x02, 0x37, 0xd0, 0x38, 0xc5, 0x94, 0x5e, 0x44, 0xb2, 0x37, 0x3a, 0xd4, 0x47, 0xcd, 0x23, 0x57, 0xc7, 0x70, 0x57, 0xb2, 0x0e, 0x26, 0x4c, 0x59, 0x8f, 0x5c, 0xf4, 0xf3, 0xc5, 0xcb, 0x4e, 0xbb, 0x57, 0x54, 0x23, 0x6b, 0xab, 0x11, 0x53, 0x26, 0x1d, 0x88, 0x1f, 0xd9, 0x51, 0x9c, 0x91, 0x6a, 0xf6, 0xaa, 0x78, 0xde, 0x15, 0x06, 0x99, 0x65, 0xbf, 0x75, 0xab, 0x32, 0xdc, 0xde, 0x37, 0x5f, 0x67, 0x34, 0x93, 0xc7, 0x61, 0xa6, 0x89, 0x4d, 0xe9, 0x6d, 0x87, 0x36, 0xb4, 0x2f, 0xd3, 0x9a, 0xae, 0x3f, 0x43, 0x6a, 0x70, 0x72, 0x9f, 0xf2, 0x13, 0xc9, 0xc3, 0x9f, 0x85, 0xd5, 0x45, 0xbc, 0x03, 0xa4, 0xd4, 0x58, 0xac,
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__DATA /* example/sb_image2_enc.bin */ 0xbf, 0x8b, 0x7d, 0xed, 0x8a, 0x6b, 0x00, 0x12, 0x00, 0xe1, 0x0b, 0xb6, 0xd1, 0xc3, 0x01, 0x1d, 0x5f, 0x7d, 0x00, 0xf8, 0xf3, 0x45, 0x42, 0xc0, 0xab, 0x0b, 0xd3, 0x53, 0x9c, 0x1f, 0xfb, 0xed, 0x33, 0x90, 0x19, 0xe5, 0xd7, 0x1a, 0x6c, 0x62, 0x49, 0xdb, 0xf4, 0x11, 0x5d, 0xa4, 0x8d, 0x19, 0xf4, 0xc3, 0xe7, 0xa6, 0x23, 0x0e, 0x9a, 0x0e, 0x11, 0x91, 0x8c, 0xa0, 0x7d, 0x8c, 0x06, 0x23, 0x10, 0x7d, 0xca, 0x38, 0x00, 0x18, 0xfe, 0x88, 0x38, 0xb1, 0xa0, 0x15, 0x63, 0x52, 0x7f, 0xf4, 0x28, 0x1c, 0xe5, 0x3b, 0x23, 0x7a, 0x30, 0x40, 0xee, 0xb2, 0xbb, 0x13, 0x9f, 0x5b, 0x05, 0x38, 0x98, 0xa1, 0x50, 0x46, 0x5c, 0x3f, 0x42, 0xde, 0x61, 0xd6, 0x4c, 0x3f, 0xf3, 0x26, 0x57, 0x6c,
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA /* example/sb_image3_enc.bin */ 0xbc, 0xa9, 0xa1, 0x6e, 0x2c, 0x58, 0xb7, 0x6e, 0x2b, 0x70, 0x39, 0x5c, 0xd6, 0xdc, 0x5f, 0x55, 0xa8, 0xe3, 0x58, 0x98, 0xaf, 0x38, 0xa8, 0xcf, 0x28, 0x0b, 0x34, 0xb4, 0x5e, 0xd6, 0x35, 0x62, 0x25, 0x07, 0xd8, 0x69, 0x33, 0x67, 0x9d, 0xdc, 0x5b, 0xfc, 0xe8, 0xb1, 0xc6, 0xea, 0x77, 0x96, 0x6a, 0x85, 0xec, 0xc7, 0x23, 0x3e, 0xd9, 0x1a, 0x8c, 0xb3, 0x8c, 0xe3, 0xe9, 0x8b, 0x27, 0xe7, 0x75, 0x50, 0x01, 0x69, 0x04, 0xcf, 0xf3, 0xc7, 0x03, 0xb5, 0x31, 0x6b, 0x1d, 0x1a, 0xfb, 0xf0, 0x5f, 0xda, 0xcf, 0xaf, 0xfc, 0x62, 0x91, 0xad, 0xea, 0x1c, 0x6b, 0x10, 0xef, 0x33, 0xad, 0xce, 0x12, 0x6b, 0x88, 0xb1, 0x47, 0xfd, 0x4e, 0x40, 0x53, 0x88, 0x51, 0x02, 0xeb, 0xbd, 0x4e, 0xf0,
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA_LENGTH /* sb_key_cert1_hbk0.bin */ 840
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA_LENGTH /* sb_key_cert2_hbk0.bin */ 840
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA_LENGTH /* sb_content_cert.bin */ 972
#endif /* defined(CC_SB_X509_CERT_SUPPORTED) */
#else
#if defined(CC_SB_X509_CERT_SUPPORTED)
/************************************/
/*          MPS2 CM3 X509           */
/************************************/
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA /* extended_example/sb_key1_cert_test_10.bin */ 0x30, 0x82, 0x04, 0x6e, 0x30, 0x82, 0x02, 0xa6, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x04, 0x32, 0x53, 0x49, 0x84, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x38, 0x30, 0x31, 0x32, 0x35, 0x30, 0x38, 0x35, 0x33, 0x33, 0x36, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x35, 0x34, 0x30, 0x33, 0x30, 0x33, 0x31, 0x35, 0x32, 0x31, 0x35, 0x30, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x07, 0x4b, 0x65, 0x79, 0x43, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x8f, 0x00, 0x30, 0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x6e, 0x30, 0x6c, 0x30, 0x1a, 0x06, 0x03, 0x64, 0x01, 0x01, 0x01, 0x01, 0xff, 0x04, 0x10, 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x30, 0x1e, 0x06, 0x03, 0x64, 0x01, 0x02, 0x01, 0x01, 0xff, 0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x30, 0x2e, 0x06, 0x03, 0x64, 0x01, 0x03, 0x01, 0x01, 0xff, 0x04, 0x24, 0x03, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0xb3, 0x4b, 0xa0, 0xf2, 0x93, 0x2e, 0x2f, 0xa3, 0xd7, 0x14, 0x3f, 0xd4, 0x96, 0xab, 0xc7, 0x5f, 0x1d, 0xad, 0x9a, 0x15, 0xf6, 0x9b, 0x2e, 0xca, 0x6e, 0x01, 0xce, 0xc3, 0xb4, 0x60, 0xfb, 0x53, 0x7c, 0xf6, 0xf0, 0x78, 0x14, 0xd0, 0x7b, 0xb8, 0x10, 0x19, 0x7c, 0x4e, 0x77, 0xb9, 0xd0, 0x6e, 0xb4, 0x78, 0x1a, 0xe2, 0x6a, 0x2d, 0x0b, 0x11, 0xd2, 0x74, 0xa2, 0x60, 0xbf, 0x37, 0x1f, 0x48, 0x9c, 0xaa, 0xc4, 0x60, 0x64, 0x43, 0xe1, 0x54, 0xb0, 0xa5, 0x5c, 0xf9, 0xaa, 0x15, 0xd0, 0xac, 0x67, 0x11, 0x54, 0xae, 0x07, 0x10, 0x93, 0xea, 0x88, 0x9a, 0x98, 0xfd, 0x8d, 0xcb, 0xaf, 0x21, 0x41, 0x9c, 0xcc, 0x0c, 0x21, 0x89, 0xaa, 0xc2, 0x89, 0xa9, 0x58, 0x28, 0x04, 0x10, 0xdb, 0x14, 0x32, 0x97, 0x86, 0xc9, 0x24, 0x65, 0xe0, 0x92, 0x40, 0x28, 0x29, 0xf5, 0x95, 0xe7, 0x59, 0x19, 0x58, 0x08, 0xf8, 0x32, 0x76, 0x35, 0x0b, 0xcb, 0xed, 0xbc, 0xd6, 0xef, 0x26, 0x8c, 0x7c, 0x2e, 0x1f, 0xc5, 0xe8, 0xd1, 0xd8, 0x68, 0xc6, 0x15, 0x22, 0x4b, 0xe8, 0x58, 0x63, 0x32, 0x92, 0xe3, 0x68, 0xfc, 0x1b, 0xb9, 0xf6, 0x23, 0x3a, 0xce, 0xaf, 0x1c, 0x46, 0x49, 0x24, 0xdd, 0xaa, 0xc9, 0x7f, 0x47, 0x78, 0x0c, 0x6d, 0xe8, 0x99, 0xbe, 0x31, 0x31, 0x6b, 0x6d, 0xfa, 0x2b, 0x97, 0xc5, 0xd9, 0xe0, 0xda, 0x74, 0x35, 0xff, 0x83, 0x23, 0xbe, 0xc2, 0xc0, 0x5e, 0x89, 0xab, 0x22, 0xd9, 0x5c, 0xdb, 0x50, 0x00, 0x7e, 0x6b, 0x15, 0x57, 0xd8, 0xc6, 0xeb, 0xe7, 0x10, 0x94, 0x07, 0x9b, 0x42, 0x4f, 0xe5, 0x3b, 0x9b, 0x66, 0x21, 0xa9, 0xe6, 0xf8, 0x40, 0xc1, 0x7e, 0x11, 0xef, 0xdb, 0x00, 0x2b, 0x8a, 0x26, 0xbc, 0x9b, 0xbc, 0x57, 0x18, 0xb0, 0x8e, 0xb2, 0x63, 0xd0, 0x0a, 0x65, 0x9e, 0xd9, 0xba, 0x56, 0x91, 0xdf, 0x35, 0xc7, 0x8a, 0xd9, 0x3a, 0xaf, 0x7a, 0xad, 0x1a, 0x24, 0x87, 0xb2, 0x8c, 0x37, 0x5e, 0xa3, 0xb2, 0x60, 0xb9, 0x42, 0x02, 0xc8, 0x11, 0x48, 0xcb, 0xe9, 0x26, 0xfe, 0x3f, 0xe1, 0x81, 0xed, 0x83, 0x46, 0x5d, 0x1b, 0xcd, 0x95, 0xc8, 0x6d, 0x5c, 0xa6, 0x93, 0x7d, 0xb6, 0x11, 0x95, 0xc7, 0x94, 0x20, 0x82, 0x0c, 0xd7, 0xc8, 0x33, 0x47, 0x6d, 0xa6, 0x40, 0xb3, 0xa0, 0x7f, 0x69, 0x41, 0x7b, 0xce, 0xd0, 0x17, 0xa9, 0x8c, 0x27, 0xbb, 0x02, 0x4c, 0x3c, 0x56, 0x3c, 0xb4, 0x33, 0x3a, 0xe0, 0x22, 0x7d, 0xb8, 0xa0, 0x95, 0xe5, 0x44, 0x20, 0xb4, 0xc1, 0xea, 0x92, 0xfb, 0x26, 0x27, 0x4f, 0x2e, 0x2a, 0xac, 0x97, 0xe7, 0x81, 0x0e, 0x41, 0x84, 0xee, 0x26, 0x46, 0xc3, 0x4f, 0x34, 0x7d, 0x44, 0x67, 0x53, 0x23, 0x50, 0x64, 0x26, 0x7b, 0x6b,
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA /* extended_example/sb_key2_cert_test_10.bin */ 0x30, 0x82, 0x04, 0x6f, 0x30, 0x82, 0x02, 0xa7, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x05, 0x00, 0xd5, 0x96, 0x2d, 0xc6, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x38, 0x30, 0x31, 0x32, 0x35, 0x30, 0x38, 0x35, 0x33, 0x33, 0x36, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x35, 0x34, 0x30, 0x33, 0x30, 0x33, 0x31, 0x35, 0x32, 0x31, 0x35, 0x30, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x07, 0x4b, 0x65, 0x79, 0x43, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x8f, 0x00, 0x30, 0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x6e, 0x30, 0x6c, 0x30, 0x1a, 0x06, 0x03, 0x64, 0x01, 0x01, 0x01, 0x01, 0xff, 0x04, 0x10, 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x30, 0x1e, 0x06, 0x03, 0x64, 0x01, 0x02, 0x01, 0x01, 0xff, 0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x30, 0x2e, 0x06, 0x03, 0x64, 0x01, 0x03, 0x01, 0x01, 0xff, 0x04, 0x24, 0x03, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0x3d, 0x5a, 0x36, 0x91, 0xc5, 0x87, 0x72, 0x89, 0x35, 0xff, 0x9f, 0xf2, 0x63, 0x59, 0xc5, 0xc6, 0xc8, 0xb5, 0xef, 0x07, 0xf6, 0xe1, 0x05, 0xdf, 0x30, 0x1b, 0xfe, 0x6e, 0x0f, 0xb5, 0x53, 0x71, 0xd1, 0xf7, 0x51, 0xfc, 0xc2, 0xf5, 0xfc, 0x22, 0x41, 0xd1, 0xb0, 0x56, 0x22, 0xf2, 0xdd, 0x96, 0xaf, 0xc7, 0xf8, 0x80, 0xc5, 0xd4, 0x55, 0xe6, 0xa2, 0x05, 0x91, 0x05, 0x64, 0xfe, 0xbe, 0xce, 0xa2, 0x2f, 0x18, 0x4a, 0x94, 0xe2, 0x24, 0x5a, 0xc0, 0xe8, 0x4a, 0x90, 0x18, 0x68, 0x5a, 0x94, 0x8b, 0x54, 0x38, 0xd6, 0x52, 0xbe, 0x84, 0x91, 0xc2, 0xb5, 0xc2, 0x9e, 0x25, 0x06, 0x70, 0xb7, 0xb5, 0xb8, 0xc5, 0x54, 0x48, 0x1b, 0x9a, 0x39, 0xb2, 0x33, 0xfc, 0xdc, 0x9d, 0xe7, 0x09, 0x06, 0xf1, 0x42, 0x8f, 0x9d, 0x14, 0xf4, 0x46, 0xc1, 0xba, 0xee, 0x7e, 0xa2, 0x4a, 0x84, 0xe6, 0x71, 0x61, 0x66, 0x04, 0xe8, 0xb3, 0xf8, 0xcb, 0x43, 0x22, 0xec, 0xe8, 0x0e, 0x61, 0xb8, 0x9f, 0xd9, 0x16, 0xae, 0xa5, 0xca, 0xe0, 0x07, 0xf7, 0x98, 0x65, 0xf6, 0xad, 0xb4, 0x4e, 0x6f, 0xa7, 0xd5, 0xfc, 0xfa, 0xdf, 0x31, 0xe5, 0x03, 0x83, 0xd6, 0xfa, 0xf8, 0x79, 0x27, 0xa4, 0x25, 0xb1, 0xf7, 0x75, 0x04, 0x96, 0x6c, 0x5d, 0xde, 0x9f, 0x53, 0xa6, 0x74, 0xa1, 0x51, 0x13, 0xb2, 0xc6, 0xc7, 0x2f, 0x8b, 0x51, 0x07, 0x91, 0x45, 0xaa, 0x9d, 0x97, 0x1d, 0x4f, 0x6e, 0x55, 0x12, 0x83, 0x24, 0xa8, 0x43, 0x0a, 0xbb, 0x9a, 0x76, 0xb5, 0xec, 0x21, 0x02, 0x6b, 0xde, 0x04, 0xe7, 0x57, 0x84, 0xc9, 0x98, 0x36, 0x2c, 0xa6, 0x7f, 0x71, 0xb0, 0xe9, 0x5e, 0x37, 0xde, 0xf3, 0x5f, 0x5d, 0xfd, 0x38, 0xc3, 0xec, 0x1d, 0x64, 0x31, 0xb9, 0xe5, 0x76, 0x89, 0x36, 0xcd, 0xd0, 0x86, 0xbd, 0x00, 0x0d, 0xe7, 0x37, 0xc0, 0x94, 0xe8, 0x13, 0x82, 0xf6, 0xd5, 0x05, 0xa7, 0xfe, 0x5e, 0x7a, 0x2f, 0x7f, 0x85, 0x85, 0xd9, 0x87, 0x0c, 0x37, 0x57, 0x76, 0x24, 0x12, 0x6e, 0x4b, 0x30, 0x07, 0x46, 0x02, 0xff, 0xf5, 0xbf, 0x37, 0x00, 0xf3, 0x69, 0x15, 0x76, 0x4a, 0xed, 0x2e, 0x63, 0x78, 0xfc, 0x25, 0xaa, 0x76, 0xa7, 0xd9, 0x34, 0x88, 0x3c, 0xc3, 0x9e, 0x51, 0x79, 0xba, 0x2a, 0x48, 0x5e, 0xe4, 0xda, 0x2d, 0xa4, 0xf1, 0x44, 0xdf, 0x4e, 0x63, 0x24, 0x3a, 0x16, 0xd4, 0x83, 0x27, 0xe4, 0x50, 0xfe, 0x7d, 0x3d, 0xea, 0xcc, 0x38, 0xf4, 0x66, 0xb0, 0x88, 0x3a, 0xe6, 0xc6, 0xde, 0xd3, 0x9e, 0x4f, 0x53, 0xa7, 0x01, 0x45, 0xc0, 0x54, 0xc6, 0x07, 0xce, 0x04, 0x69, 0x7f, 0x91, 0x91, 0xb7, 0x9c, 0xff, 0x60, 0xa6, 0xc2, 0xc8, 0x51, 0x91, 0xef, 0x81, 0xd7, 0xe4, 0x12, 0x83, 0xf2,
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA /* extended_example/sb_content_cert_test_10.bin */ 0x30, 0x82, 0x04, 0xde, 0x30, 0x82, 0x03, 0x16, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x04, 0x62, 0xd8, 0x68, 0xcd, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x03, 0x41, 0x52, 0x4d, 0x30, 0x20, 0x17, 0x0d, 0x31, 0x38, 0x30, 0x31, 0x32, 0x35, 0x30, 0x38, 0x35, 0x33, 0x33, 0x36, 0x5a, 0x18, 0x0f, 0x32, 0x31, 0x35, 0x34, 0x30, 0x33, 0x30, 0x33, 0x31, 0x35, 0x32, 0x31, 0x35, 0x30, 0x5a, 0x30, 0x12, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x07, 0x43, 0x6e, 0x74, 0x43, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x8f, 0x00, 0x30, 0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xdd, 0x30, 0x81, 0xda, 0x30, 0x1a, 0x06, 0x03, 0x64, 0x02, 0x01, 0x01, 0x01, 0xff, 0x04, 0x10, 0x63, 0x63, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x03, 0x00, 0x30, 0x1e, 0x06, 0x03, 0x64, 0x02, 0x02, 0x01, 0x01, 0xff, 0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x30, 0x81, 0x9b, 0x06, 0x03, 0x64, 0x02, 0x04, 0x01, 0x01, 0xff, 0x04, 0x81, 0x90, 0x03, 0x00, 0x00, 0x00, 0xa9, 0xf7, 0x0a, 0xfe, 0xc3, 0x0c, 0x61, 0x67, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x00, 0x00, 0x21, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x10, 0x00, 0x21, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x20, 0x00, 0x21, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x30, 0x3d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x30, 0xa0, 0x0d, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x03, 0x82, 0x01, 0x81, 0x00, 0x51, 0x72, 0x17, 0xc2, 0xa7, 0x44, 0x51, 0xe9, 0xf0, 0x3e, 0x8a, 0x03, 0x59, 0x08, 0x14, 0xf7, 0x3b, 0x61, 0x33, 0x41, 0x2b, 0x98, 0xba, 0xf7, 0x1d, 0xe2, 0xdc, 0x27, 0x43, 0x46, 0x2d, 0xd7, 0x2f, 0x9a, 0x17, 0xb5, 0xec, 0xc1, 0xcd, 0x49, 0x63, 0x56, 0x9f, 0x87, 0x72, 0xa1, 0xba, 0xb9, 0xc0, 0x38, 0x42, 0x7a, 0xb1, 0x98, 0xce, 0xf3, 0x63, 0xfc, 0x05, 0x82, 0xd9, 0xa5, 0x23, 0x8d, 0xc8, 0x4c, 0x50, 0xf7, 0xb9, 0x4c, 0x8b, 0x71, 0x57, 0x7e, 0xf5, 0xeb, 0x85, 0xb5, 0x56, 0xdb, 0xb8, 0xf7, 0xcc, 0x29, 0x52, 0x56, 0x37, 0x82, 0x91, 0x66, 0xe7, 0x98, 0x68, 0x8d, 0x3a, 0x49, 0xba, 0xc6, 0x7a, 0x22, 0xff, 0x68, 0x56, 0x17, 0x05, 0x96, 0x60, 0x29, 0x8d, 0x04, 0x88, 0xd1, 0xe6, 0x20, 0x00, 0xb2, 0x1c, 0x6c, 0xe4, 0x11, 0xbc, 0xa8, 0xf7, 0x70, 0xd7, 0x18, 0x43, 0xe2, 0xe2, 0xe4, 0x22, 0x43, 0xbc, 0x9d, 0xcb, 0x2a, 0x35, 0x74, 0xdf, 0x70, 0x25, 0xce, 0xe8, 0x1b, 0x24, 0xac, 0x12, 0x76, 0x4c, 0xc6, 0x14, 0xf4, 0x24, 0xa0, 0x67, 0x6c, 0x79, 0x9e, 0xd5, 0xb4, 0x76, 0x6e, 0x11, 0x8e, 0x75, 0xdc, 0x0f, 0xac, 0x02, 0x53, 0xe8, 0x8c, 0x84, 0x6d, 0x55, 0x00, 0x84, 0x97, 0xd3, 0x8d, 0x24, 0xd9, 0x77, 0x1e, 0x38, 0x89, 0x18, 0xb0, 0x0c, 0xe5, 0xf2, 0x61, 0x86, 0x5a, 0xec, 0x42, 0x4e, 0xd6, 0x46, 0xe3, 0x61, 0x65, 0x82, 0x51, 0x39, 0x74, 0xd2, 0xd9, 0x32, 0x96, 0xed, 0x88, 0x1f, 0xcb, 0xf2, 0x8c, 0x40, 0xe0, 0xf7, 0xb8, 0x07, 0x7d, 0xad, 0x5d, 0xd5, 0xc9, 0x67, 0x8f, 0x6c, 0xe6, 0x46, 0xb2, 0x0c, 0x1a, 0x29, 0xa9, 0x8d, 0xb5, 0x1c, 0x96, 0x01, 0xef, 0xcb, 0x19, 0xd1, 0x6d, 0x57, 0xac, 0x67, 0x9b, 0x08, 0x53, 0xd8, 0x9d, 0x95, 0x17, 0x05, 0x16, 0x0d, 0xb6, 0xa5, 0x88, 0x0e, 0xcf, 0xd9, 0x7b, 0x62, 0xfe, 0x7d, 0xdb, 0x7a, 0x95, 0xdc, 0xb5, 0x6d, 0x9e, 0x01, 0x8f, 0xb0, 0x0d, 0xfa, 0x8d, 0x4b, 0xe0, 0x02, 0xb7, 0x82, 0x6e, 0x51, 0xf4, 0x12, 0x73, 0x55, 0x23, 0xff, 0x16, 0x54, 0xf5, 0x50, 0x6a, 0xab, 0xac, 0x2e, 0xa9, 0x80, 0x15, 0x20, 0x59, 0xc9, 0xf1, 0xcc, 0x78, 0x85, 0xf8, 0x02, 0x33, 0xa7, 0x49, 0x0e, 0x1b, 0xce, 0x2f, 0x66, 0xf7, 0x05, 0x5c, 0xee, 0xb1, 0x68, 0x1e, 0xe5, 0x2a, 0x6c, 0x22, 0x2e, 0x8a, 0xdf, 0xe4, 0x6c, 0x80, 0xcd, 0x5e, 0x6d, 0xe6, 0xed, 0x95, 0x87, 0xfb, 0x9d, 0x7b, 0x21, 0x9b, 0x36, 0x0c, 0x07, 0xa2, 0x0c, 0x32, 0x8d, 0x2a, 0xc3, 0xfe, 0x01, 0xfd, 0xd5, 0x1c, 0xec, 0xca, 0x73, 0xd5, 0xf3, 0xc9, 0xd8, 0x63, 0xd6, 0x97, 0xa2, 0xaf, 0x8a, 0xea, 0x41, 0x62, 0x07, 0xd9, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00,
#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__DATA /* extended_example/sb_image1_enc.bin */ 0x55, 0x23, 0x80, 0xfe, 0x51, 0x6a, 0x55, 0x15, 0x6f, 0xb5, 0x2d, 0x98, 0x21, 0xd6, 0x6a, 0xa7, 0x06, 0x53, 0x67, 0x46, 0xb1, 0x88, 0x61, 0x21, 0x23, 0x81, 0xd6, 0xc1, 0x81, 0xdb, 0x1a, 0x17, 0x1c, 0xe5, 0x06, 0xaf, 0x37, 0x50, 0x2f, 0xd0, 0xac, 0x7b, 0xb1, 0xb7, 0x21, 0xf5, 0x82, 0x94, 0xca, 0x72, 0x57, 0x6e, 0x00, 0x4e, 0xe6, 0x5e, 0x1b, 0x81, 0xc6, 0x0a, 0xe7, 0x3c, 0x52, 0x84, 0x53, 0x97, 0x02, 0xe5, 0xf3, 0x74, 0x7a, 0x6c, 0xe7, 0x8c, 0x92, 0xf0, 0x38, 0x56, 0xa3, 0x0d, 0x82, 0xe9, 0xbe, 0x8e, 0x24, 0xae, 0x7e, 0x47, 0xb8, 0x88, 0x47, 0x6a, 0x34, 0xa8, 0x99, 0x4f, 0x6b, 0x03, 0x6e, 0xa1, 0x47, 0x9d, 0x38, 0x4d, 0xd0, 0x6d, 0x69, 0x35, 0xd5, 0x55, 0x68, 0xe1,
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__DATA /* extended_example/sb_image2_enc.bin */ 0xf9, 0xa5, 0x94, 0x60, 0x00, 0x2a, 0xf9, 0xe0, 0x0f, 0x90, 0xf1, 0xde, 0x2b, 0x1a, 0xf0, 0x35, 0xa3, 0x70, 0x61, 0xdb, 0x6c, 0xd6, 0x44, 0x68, 0xe0, 0xb7, 0x5a, 0x95, 0xe1, 0x4a, 0x6a, 0x74, 0x6e, 0x9e, 0xd5, 0x4e, 0xb9, 0xc2, 0xe2, 0xac, 0x95, 0xbc, 0x57, 0x77, 0x3c, 0xf9, 0xf8, 0xb1, 0x78, 0xbf, 0x5b, 0x2c, 0x17, 0x38, 0x53, 0x32, 0x76, 0xf2, 0x02, 0x47, 0x3c, 0x9d, 0x88, 0x4e, 0x46, 0xf3, 0x5a, 0x70, 0x39, 0xa4, 0x3a, 0x67, 0x72, 0x7d, 0x97, 0x0b, 0x9c, 0xa8, 0x04, 0x4d, 0xb3, 0x32, 0xa2, 0x1b, 0x1c, 0xd7, 0xa0, 0x19, 0x73, 0x70, 0xc8, 0xbb, 0x34, 0x7b, 0x91, 0xc3, 0xaf, 0x13, 0xe5, 0xb8, 0xf0, 0x53, 0x3d, 0x09, 0xb2, 0x1f, 0xbc, 0x59, 0x39, 0x95, 0xed, 0xb7,
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA /* extended_example/sb_image3_enc.bin */ 0x19, 0x59, 0xa4, 0x2b, 0x3d, 0x3a, 0xab, 0xa9, 0x8f, 0x45, 0x37, 0xaa, 0xc0, 0x5b, 0x74, 0x87, 0x0c, 0x70, 0x88, 0xae, 0x3f, 0xaf, 0x82, 0xaf, 0xdc, 0x4d, 0x12, 0x81, 0xde, 0x06, 0xcb, 0x45, 0xb2, 0x16, 0xa0, 0x03, 0xf3, 0x6e, 0x3d, 0x27, 0xd9, 0x2a, 0x8d, 0x4b, 0x74, 0x9d, 0x31, 0x03, 0xd5, 0xd2, 0xdc, 0xe5, 0x8b, 0xc7, 0x84, 0x72, 0x3f, 0x93, 0xca, 0x85, 0x85, 0x72, 0x5a, 0x87, 0xaf, 0xb7, 0x9b, 0x83, 0x3f, 0x47, 0x7a, 0x24, 0x42, 0x56, 0xd6, 0x19, 0x87, 0xb4, 0xdb, 0xbb, 0x40, 0x88, 0x92, 0x67, 0x4c, 0x28, 0x21, 0x85, 0x21, 0xd7, 0x6a, 0xde, 0x28, 0xba, 0x88, 0xa7, 0x1b, 0xbd, 0xc8, 0x58, 0xb9, 0x68, 0xf8, 0x6e, 0x5a, 0x91, 0xa9, 0x7d, 0xf1, 0xa6, 0x5d, 0x44,
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA_LENGTH /* sb_key1_cert_test_10.bin */ 1138
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA_LENGTH /* sb_key2_cert_test_10.bin */ 1139
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA_LENGTH /* sb_content_cert_test_10.bin */ 1276
#else
/************************************/
/*          MPS2 CM3 NOT_X509       */
/************************************/
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA /* extended_example/sb_key1_cert_test_10.bin */ 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x03, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0x45, 0x8b, 0x30, 0x9a, 0xd0, 0xa0, 0xb0, 0xb6, 0xe3, 0xd4, 0x5f, 0xbb, 0xb8, 0x69, 0x19, 0xf7, 0x2c, 0xdb, 0xaa, 0x1c, 0x8c, 0x1e, 0x4e, 0x3f, 0xa6, 0x44, 0x03, 0xd5, 0x0d, 0xb0, 0x21, 0xa4, 0xaa, 0x21, 0x18, 0xfc, 0x2a, 0xae, 0x05, 0xb8, 0x01, 0x4d, 0xc1, 0x46, 0x79, 0x36, 0xdf, 0x76, 0x52, 0xa9, 0xf4, 0xb4, 0xa7, 0x81, 0x3e, 0x37, 0x8d, 0xab, 0x00, 0x26, 0xf4, 0x2b, 0x05, 0xbd, 0xe5, 0xf8, 0x30, 0x8c, 0x2a, 0x7c, 0xef, 0xab, 0xaf, 0x67, 0xe7, 0x2a, 0xb6, 0xa0, 0x6b, 0xe3, 0x1b, 0xef, 0x98, 0x35, 0xd6, 0x3b, 0x92, 0x8b, 0x30, 0x01, 0xc4, 0x84, 0xbc, 0x3b, 0x4e, 0x0a, 0xa5, 0x41, 0x48, 0x17, 0x6c, 0x3c, 0x8b, 0x7e, 0xda, 0xcb, 0x66, 0x8b, 0xe8, 0x2b, 0x74, 0x5e, 0x0a, 0x6f, 0xa7, 0xaf, 0x5c, 0xce, 0xbc, 0xbe, 0xe4, 0x1f, 0x78, 0x11, 0x2a, 0x11, 0x25, 0x7a, 0x45, 0xb5, 0x5f, 0xb8, 0xae, 0xc0, 0x59, 0x45, 0x33, 0xb3, 0x59, 0xf9, 0xe3, 0xde, 0x39, 0x73, 0x18, 0x18, 0x8a, 0xe8, 0xb6, 0x0e, 0x19, 0xc9, 0x5a, 0x7a, 0xb9, 0x57, 0xfc, 0x98, 0x60, 0x6c, 0xf3, 0x67, 0xec, 0x62, 0xc0, 0x24, 0x1e, 0x0b, 0x91, 0x41, 0x97, 0x43, 0xfc, 0xfc, 0xf2, 0x60, 0x3f, 0xce, 0xa5, 0x92, 0x7d, 0x9b, 0xc1, 0x09, 0x1e, 0x48, 0xef, 0xdb, 0x7e, 0x1f, 0x50, 0xa2, 0xed, 0x0f, 0xd4, 0x60, 0x8b, 0x52, 0x28, 0x2d, 0x42, 0x95, 0x07, 0xda, 0x6c, 0x93, 0x6e, 0xba, 0xa8, 0x5c, 0x5f, 0x05, 0x63, 0x6b, 0xb9, 0xdb, 0x57, 0xf4, 0x2b, 0xc4, 0x7d, 0x6b, 0xd3, 0x04, 0x1a, 0xc5, 0x9b, 0xab, 0x6c, 0x58, 0x5b, 0xf8, 0x43, 0x18, 0xc1, 0x12, 0x72, 0xe5, 0x52, 0x93, 0x98, 0xab, 0x7e, 0xed, 0xd0, 0x2f, 0x7f, 0x1c, 0x32, 0x6f, 0xc7, 0x83, 0xd1, 0x8c, 0x55, 0xf9, 0x85, 0x64, 0x35, 0x4b, 0xdb, 0x9e, 0x3e, 0x42, 0xbb, 0xf4, 0x4b, 0x4e, 0x5e, 0xe9, 0x7e, 0x85, 0x6a, 0x42, 0xe1, 0x38, 0xcc, 0xf0, 0xaf, 0xfd, 0xda, 0x10, 0xb5, 0x7d, 0x68, 0x3a, 0x6c, 0xbc, 0xb5, 0x1f, 0x2f, 0x54, 0x1d, 0xa1, 0xe9, 0x8f, 0x81, 0x0d, 0xe5, 0x7b, 0xa7, 0x31, 0x43, 0x78, 0x8e, 0xc9, 0x3b, 0x8b, 0xad, 0xdf, 0xdd, 0xa7, 0x4d, 0x6a, 0x33, 0x55, 0xfd, 0x88, 0x28, 0x05, 0xce, 0x2f, 0xf8, 0xcd, 0x2c, 0x1e, 0x82, 0x95, 0xf9, 0x03, 0x8a, 0x17, 0xf0, 0x30, 0xcb, 0x01, 0xbb, 0xe5, 0xcc, 0x1f, 0x9f, 0x65, 0xa5, 0x51, 0xab, 0x50, 0x49, 0x33, 0x76, 0x1f, 0xb7, 0x5c, 0x4b, 0xa9, 0xeb, 0x75, 0x6d, 0x6f, 0xf8, 0xae, 0xd7, 0xbc, 0x9e, 0x18, 0xcd, 0x3d, 0x2a, 0x2c, 0x27, 0xfe, 0x2b, 0xb0, 0xd4, 0x7f, 0x9e, 0x9c, 0xcb, 0x83, 0x6c, 0xbd, 0xd3, 0x39, 0xcc, 0xaa,
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA /* extended_example/sb_key2_cert_test_10.bin */ 0x63, 0x6b, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x72, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x03, 0x00, 0x00, 0x00, 0xda, 0x7b, 0xf1, 0x2f, 0x06, 0xb5, 0x3e, 0xe9, 0x0b, 0x4f, 0xa5, 0x9a, 0x2b, 0x75, 0xde, 0x4a, 0x85, 0x5e, 0x84, 0xaf, 0x59, 0xb3, 0x46, 0x9b, 0x0d, 0x88, 0x4e, 0xf8, 0xf7, 0x0a, 0x72, 0x1b, 0xf0, 0x9b, 0x52, 0x8e, 0xc7, 0xf3, 0x3d, 0x23, 0x42, 0xcd, 0xb6, 0x81, 0xf6, 0x24, 0xc4, 0xce, 0xb5, 0xb9, 0x4b, 0x0c, 0x83, 0xd6, 0x7d, 0x11, 0xdb, 0x4e, 0xca, 0xb1, 0xe4, 0xbf, 0xb8, 0x97, 0xa8, 0x93, 0xa1, 0x5b, 0x1d, 0xbd, 0xea, 0x04, 0x6d, 0xb2, 0x40, 0x00, 0x7d, 0x05, 0x7d, 0xe0, 0x6f, 0x51, 0xc8, 0xb5, 0xb5, 0x32, 0x9f, 0x70, 0x0f, 0xd8, 0xaa, 0x18, 0x90, 0x84, 0x96, 0x7c, 0x98, 0x08, 0x21, 0xd5, 0xfa, 0x93, 0xc8, 0x75, 0xb6, 0xf6, 0x65, 0x76, 0x34, 0x15, 0xd9, 0x41, 0x07, 0x0e, 0x3b, 0xf8, 0x47, 0xcf, 0xc1, 0xfe, 0x7b, 0xaa, 0x38, 0x70, 0xec, 0xe9, 0x93, 0x41, 0x7b, 0x72, 0x9a, 0x0e, 0x28, 0xe2, 0x1d, 0x26, 0x3b, 0xd6, 0xc8, 0xe5, 0xd7, 0x7a, 0xfe, 0xa7, 0xcb, 0x87, 0xd0, 0x93, 0x18, 0x61, 0xfc, 0xca, 0xe5, 0x57, 0xa0, 0x43, 0x8d, 0xa2, 0x70, 0x26, 0xf3, 0x44, 0xbe, 0x7e, 0xfc, 0x63, 0x8f, 0xd6, 0xd7, 0xbe, 0xed, 0x88, 0x0d, 0xe3, 0x54, 0x9d, 0x3d, 0xd5, 0xcc, 0xec, 0x83, 0x2a, 0x16, 0x1d, 0xc5, 0xda, 0xa6, 0xe1, 0x89, 0x62, 0x57, 0xdf, 0x47, 0x74, 0x89, 0x39, 0x70, 0x09, 0x11, 0x65, 0xec, 0xf6, 0x9e, 0xee, 0xc6, 0xc7, 0x59, 0x22, 0x65, 0x1a, 0xdc, 0xae, 0x9e, 0x90, 0x7b, 0x1c, 0xfc, 0x5d, 0xfc, 0xc7, 0x49, 0xc1, 0x0a, 0xae, 0x4c, 0xc1, 0xc9, 0x6f, 0xe1, 0x07, 0xef, 0xf5, 0x54, 0x4e, 0x77, 0x5c, 0x7c, 0xa5, 0xc1, 0xd5, 0x15, 0x70, 0xc5, 0xa5, 0xd4, 0x88, 0x52, 0xae, 0xbc, 0x37, 0x8a, 0x9d, 0x2a, 0x83, 0xc5, 0xe4, 0xe3, 0xa0, 0x5c, 0xed, 0xc8, 0x0c, 0x80, 0xc7, 0x53, 0xb6, 0x00, 0x53, 0x00, 0xa9, 0xce, 0x1d, 0xce, 0x8e, 0xa3, 0x73, 0xd7, 0x31, 0xd7, 0x8f, 0x48, 0x38, 0x91, 0x20, 0x38, 0xca, 0x64, 0x61, 0x6b, 0x16, 0xbb, 0xc4, 0x7e, 0xd5, 0xac, 0xca, 0x53, 0xb3, 0x7d, 0x4b, 0x04, 0xb6, 0xe3, 0xd9, 0x00, 0x55, 0xb4, 0x86, 0x17, 0xec, 0x64, 0x9e, 0x63, 0xf3, 0x53, 0x9b, 0x11, 0x38, 0x66, 0xfe, 0xd7, 0xd8, 0xe6, 0x6a, 0x3e, 0x2b, 0x56, 0xdd, 0xc5, 0x2d, 0xf9, 0xf9, 0x83, 0x6c, 0x82, 0x93, 0xad, 0xeb, 0x97, 0x10, 0x2e, 0x85, 0xce, 0xb5, 0xba, 0x95, 0x91, 0x92, 0x12, 0x8e, 0x29, 0x46, 0x9a, 0x63, 0x68, 0x59, 0xdc, 0x8d, 0xde, 0x06, 0x7a, 0x51, 0x86, 0xff, 0xd3, 0x6c, 0xea, 0x06, 0x18, 0x98, 0xe3, 0x19, 0x0f, 0x96, 0x07, 0x1b, 0x07, 0x28, 0xe6, 0x93, 0xaa, 0x6c, 0x3b, 0x35, 0x6a, 0x68, 0xb7, 0xe4, 0xa5, 0x7a, 0xbf, 0x43, 0xa2, 0x9a, 0xf9, 0x36, 0x8c, 0x22, 0x30, 0x34, 0x03, 0xde, 0xff, 0x1a, 0x20, 0xd2, 0xfb, 0x1f, 0x62, 0xe9, 0xf8, 0x17, 0x42, 0x3b, 0x3e, 0xa0,
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA /* extended_example/sb_content_cert_test_10.bin */ 0x63, 0x63, 0x42, 0x53, 0x00, 0x00, 0x01, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x03, 0x00, 0xc7, 0xad, 0x9d, 0x16, 0xab, 0x03, 0xd3, 0x71, 0x19, 0x96, 0x86, 0xa4, 0x7c, 0x53, 0x51, 0x93, 0x22, 0xc7, 0x8d, 0xa9, 0x78, 0x92, 0x5e, 0x0b, 0x9d, 0x55, 0x0a, 0x89, 0x8c, 0x56, 0x0b, 0xe0, 0x5f, 0x00, 0xaf, 0xac, 0x32, 0xee, 0x46, 0x83, 0x51, 0xfc, 0x8a, 0x9e, 0x30, 0x38, 0xe1, 0xf4, 0xe1, 0xfb, 0x49, 0x70, 0x17, 0x5c, 0x8f, 0xba, 0xb0, 0xe6, 0x48, 0xe3, 0x6c, 0x22, 0x80, 0x25, 0xd9, 0xb8, 0xfc, 0x7c, 0x53, 0x9e, 0x76, 0xcf, 0xb5, 0xc9, 0x2f, 0xe3, 0x67, 0x25, 0xce, 0xd7, 0x29, 0xf3, 0x02, 0x2d, 0x29, 0x2e, 0x42, 0x2c, 0xc6, 0xdc, 0x36, 0xe7, 0x25, 0x1c, 0xa6, 0x9a, 0x60, 0x69, 0xe2, 0x7c, 0x80, 0xe3, 0xc0, 0x37, 0x94, 0x59, 0x77, 0x00, 0xd5, 0x61, 0x5f, 0x07, 0x70, 0x04, 0x87, 0x1e, 0xc5, 0xb3, 0x94, 0xa8, 0xa4, 0xa0, 0xc4, 0xeb, 0x9d, 0xfc, 0x4c, 0x02, 0xb3, 0xc8, 0x81, 0xcd, 0x0a, 0xd9, 0x06, 0xce, 0x4c, 0xf9, 0x6a, 0x62, 0x6b, 0x36, 0x57, 0x17, 0xe4, 0xab, 0x75, 0x5c, 0x37, 0xfa, 0x50, 0x22, 0x69, 0x38, 0x92, 0xe8, 0x91, 0x27, 0x30, 0xd1, 0x57, 0xa0, 0x4f, 0x21, 0x0a, 0x8d, 0x17, 0xee, 0xe4, 0xa4, 0x8c, 0xdb, 0xc7, 0x2f, 0xb3, 0xc6, 0xdd, 0x0b, 0x54, 0x81, 0xcd, 0x14, 0x08, 0x1b, 0x50, 0x7b, 0x30, 0xd3, 0x5d, 0x97, 0xa3, 0x82, 0x15, 0x9a, 0x96, 0x69, 0xaf, 0x8f, 0x6b, 0xf2, 0x5b, 0x80, 0x1a, 0x21, 0x67, 0x37, 0x42, 0x56, 0xd9, 0x25, 0xb5, 0x72, 0xf6, 0x83, 0x25, 0x0f, 0x04, 0x12, 0x50, 0x7e, 0x74, 0xd6, 0xba, 0xe6, 0x51, 0x55, 0x69, 0xb2, 0xc7, 0xe0, 0x78, 0x5d, 0xe3, 0x31, 0xfb, 0x49, 0xed, 0x17, 0x4a, 0xd0, 0x70, 0xe2, 0x2c, 0x71, 0x7a, 0x3f, 0x15, 0x16, 0x68, 0x42, 0xac, 0x84, 0x51, 0x3c, 0x4e, 0x67, 0xfe, 0x13, 0x41, 0x80, 0x9c, 0x61, 0xd6, 0xd4, 0x84, 0xc8, 0x95, 0x66, 0x6d, 0x0a, 0x4f, 0x21, 0xee, 0x97, 0xb9, 0x02, 0x0e, 0x1b, 0x95, 0x73, 0xa2, 0xf0, 0x54, 0x31, 0x6a, 0xe9, 0xa6, 0xfa, 0x57, 0xad, 0x59, 0xc0, 0xf1, 0x50, 0xf9, 0xda, 0x4a, 0x13, 0xa9, 0x8a, 0x6f, 0x52, 0x12, 0x5c, 0x8f, 0x01, 0x91, 0xef, 0x1d, 0xe1, 0x16, 0xc3, 0xc3, 0x3d, 0xaf, 0xcc, 0xce, 0x5b, 0x92, 0x7d, 0xa5, 0xae, 0x70, 0x10, 0xa0, 0xac, 0x1d, 0x59, 0x8d, 0xb1, 0xf3, 0x14, 0x16, 0x49, 0x05, 0xeb, 0xf2, 0x99, 0x71, 0xb2, 0x59, 0x97, 0x43, 0x29, 0x1c, 0x82, 0x9d, 0x77, 0x00, 0x94, 0x7e, 0x77, 0x3a, 0x7e, 0x1f, 0x9a, 0x45, 0x61, 0x37, 0x2d, 0x6f, 0xcd, 0x26, 0x25, 0xec, 0xfa, 0x58, 0x87, 0x7a, 0x5d, 0xfd, 0xc8, 0x6a, 0x15, 0x62, 0x37, 0xcb, 0xb0, 0xdf, 0xaa, 0x63, 0x6c, 0x97, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x1a, 0xa3, 0xc2, 0xd9, 0x60, 0xd8, 0x72, 0xa7, 0x03, 0x00, 0x00, 0x00, 0xd0, 0x8c, 0x7c, 0xac, 0x88, 0xc6, 0x3a, 0x2e, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x00, 0x00, 0x21, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x10, 0x00, 0x21, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x69, 0x4f, 0xfb, 0xfa, 0xa6, 0xfe, 0x6e, 0x79, 0xcf, 0x4a, 0xa9, 0x6c, 0xe3, 0x5f, 0x97, 0xb4, 0x52, 0xb5, 0x22, 0x2c, 0x3c, 0x9a, 0x32, 0x3e, 0x00, 0x06, 0x19, 0xd8, 0x56, 0xee, 0x19, 0x00, 0x20, 0x00, 0x21, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xd2, 0x97, 0x56, 0xd4, 0xa0, 0xf5, 0xd7, 0x04, 0x24, 0x22, 0x17, 0xe0, 0x32, 0xa2, 0xb2, 0x09, 0x08, 0x3c, 0xe6, 0x9a, 0xea, 0x38, 0xe4, 0x5a, 0x94, 0x45, 0xc9, 0x86, 0xba, 0xc7, 0x9c, 0x42, 0x61, 0xb7, 0xc7, 0x87, 0x1b, 0x9d, 0x2f, 0x3b, 0x75, 0xfb, 0xe3, 0x10, 0xd8, 0x35, 0x94, 0xec, 0xf9, 0xb3, 0xb8, 0xaa, 0x1b, 0x06, 0x17, 0x4c, 0x5b, 0x87, 0xb8, 0x7a, 0x3a, 0xbc, 0x64, 0x38, 0x52, 0x25, 0x79, 0xc0, 0xfc, 0xd8, 0x7c, 0x49, 0x71, 0xa0, 0x43, 0xac, 0x6c, 0x44, 0xd9, 0x30, 0x23, 0xdf, 0x12, 0xdd, 0x9d, 0x73, 0x9a, 0x41, 0x13, 0x95, 0x80, 0xc3, 0xdd, 0x6c, 0x79, 0x9a, 0x0e, 0x9b, 0xa3, 0x9a, 0xee, 0xea, 0x17, 0x63, 0xac, 0x72, 0x83, 0x78, 0x83, 0xf9, 0x04, 0x46, 0x32, 0x79, 0xcf, 0xc0, 0x80, 0x69, 0xbd, 0x5b, 0x8b, 0xf0, 0x29, 0x5b, 0x4f, 0x0d, 0x94, 0x9f, 0xf3, 0xd0, 0xf9, 0x5a, 0x60, 0x88, 0x29, 0x47, 0x44, 0x70, 0x6e, 0x5f, 0xb4, 0xb9, 0x6c, 0x17, 0x2b, 0x79, 0xba, 0xa9, 0xaf, 0xbc, 0xc2, 0x6f, 0xaa, 0xac, 0xad, 0xdd, 0x20, 0x5c, 0x2c, 0x56, 0xc6, 0x5d, 0xe9, 0x0c, 0xe5, 0x63, 0x3d, 0xd0, 0xa1, 0xb1, 0xd7, 0x53, 0x27, 0xa8, 0x47, 0x1b, 0xa7, 0x6f, 0xd9, 0xd1, 0xe8, 0xd6, 0x5f, 0x3d, 0x9e, 0x03, 0xd9, 0x4a, 0x5a, 0x85, 0x58, 0xd2, 0xb2, 0x1f, 0x43, 0x9b, 0x8d, 0x43, 0x88, 0xe5, 0x1d, 0xa2, 0x66, 0xf6, 0xa7, 0x3f, 0x3f, 0xc6, 0x47, 0x55, 0x0f, 0x39, 0xc1, 0x4e, 0xda, 0x8c, 0x57, 0x20, 0xbe, 0x1a, 0x93, 0xa9, 0x74, 0x38, 0x0b, 0xf3, 0xe8, 0x54, 0x73, 0xed, 0xe5, 0x62, 0x3e, 0xdb, 0xd1, 0x3d, 0x1d, 0x1f, 0x52, 0x01, 0x25, 0xe9, 0xdd, 0xdd, 0xdb, 0x91, 0x36, 0xaf, 0x5a, 0xf7, 0x90, 0xea, 0xf4, 0x88, 0x55, 0xfd, 0x04, 0x4d, 0x9e, 0xfa, 0x06, 0xd0, 0x4b, 0xe5, 0xc0, 0x21, 0x0e, 0xbd, 0xad, 0x72, 0xe8, 0x72, 0x29, 0x2a, 0xf7, 0x5b, 0x70, 0x31, 0xec, 0xd3, 0x51, 0x38, 0x35, 0x93, 0x80, 0xf2, 0xa3, 0xe0, 0x14, 0x59, 0x59, 0xfe, 0x9b, 0x36, 0xa2, 0xb7, 0xcd, 0xb9, 0x41, 0x3a, 0x45, 0x23, 0xd2, 0xa9, 0xe0, 0x4f, 0xe2, 0x6b, 0xa3, 0xcf, 0xf8, 0xec, 0xc8, 0x15, 0xed, 0x5b, 0x65, 0x96, 0x04, 0x63, 0x8c, 0x53, 0x5e, 0xe4, 0xc1, 0x1e, 0x21, 0xa9, 0xa4, 0x2b, 0xdb, 0xa8, 0x0f, 0x33, 0xc0, 0x80, 0x69, 0x8c, 0xe7, 0xfb, 0x26, 0x32, 0x1d, 0xf8, 0x74, 0x1a, 0xad, 0x4f, 0x60, 0xdf, 0x0a, 0x57, 0xd7, 0xc5, 0x82, 0x1a, 0x44, 0x61, 0xa5, 0x9e, 0xd3, 0x7c, 0x16, 0xf9, 0x61, 0xb2, 0x1d, 0xc4, 0xbf, 0x0f, 0x89, 0x22, 0xc7, 0xd3, 0x39, 0x9f, 0xff, 0xfc, 0xb9, 0xaf, 0x78, 0x9f, 0xf4, 0x35, 0x00, 0x30, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00,
#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__DATA /* extended_example/sb_image1_enc.bin */ 0xd2, 0x8c, 0x25, 0x4d, 0x27, 0x70, 0xa6, 0x70, 0x31, 0xcc, 0x7f, 0x02, 0x9f, 0x06, 0x51, 0x0b, 0x5c, 0x18, 0xe3, 0x5e, 0x35, 0x16, 0x98, 0x84, 0x84, 0xe7, 0x84, 0xb5, 0xea, 0xf7, 0x44, 0xd7, 0x22, 0xe9, 0xd5, 0x40, 0x63, 0x8b, 0xb9, 0xc2, 0xbf, 0x43, 0xae, 0xff, 0xb6, 0xe8, 0xbf, 0xcd, 0xb7, 0x0f, 0xcc, 0x99, 0x3d, 0x50, 0x4b, 0x55, 0x5b, 0xbb, 0x0a, 0xbc, 0x97, 0x95, 0x0a, 0x06, 0x3e, 0x71, 0x66, 0x50, 0x23, 0x65, 0x64, 0x46, 0xca, 0xc8, 0xa3, 0x38, 0xd6, 0x23, 0xae, 0xdc, 0x32, 0x37, 0xc7, 0xdd, 0x8f, 0x38, 0x51, 0x74, 0xb3, 0xf3, 0x1f, 0x97, 0xa7, 0x72, 0x65, 0x00, 0x96, 0x38, 0x30, 0x71, 0xb3, 0x34, 0xad, 0xe9, 0xa8, 0x83, 0x26, 0x2e, 0x6c, 0xd1, 0x80, 0xad,
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__DATA /* extended_example/sb_image2_enc.bin */ 0x59, 0x61, 0xfc, 0xe7, 0xce, 0xed, 0x25, 0xbb, 0xea, 0x60, 0x29, 0xb6, 0x57, 0xce, 0x36, 0x0c, 0x80, 0x05, 0x54, 0xa3, 0xd9, 0x47, 0x3a, 0x8c, 0x6a, 0x9d, 0x06, 0x62, 0x82, 0x89, 0x5a, 0xc1, 0xc8, 0x27, 0xe5, 0x89, 0xc9, 0xf2, 0x37, 0x24, 0x4f, 0x1b, 0xf7, 0x7c, 0x13, 0xc1, 0xb9, 0xd5, 0x04, 0x86, 0x12, 0xd4, 0xe0, 0x66, 0xc9, 0xe1, 0x5b, 0x70, 0x3c, 0x0b, 0xde, 0x01, 0x77, 0xce, 0x06, 0x32, 0x01, 0xf5, 0x3e, 0x76, 0x92, 0x8c, 0x23, 0x91, 0x48, 0xc0, 0x97, 0x7b, 0x40, 0x07, 0x55, 0x6a, 0xde, 0x31, 0xe9, 0xff, 0x88, 0xe6, 0x98, 0x03, 0x82, 0x6d, 0xe8, 0x27, 0xff, 0xa6, 0xc1, 0x46, 0xb4, 0x17, 0x87, 0x7b, 0x91, 0x15, 0xd2, 0x6c, 0x83, 0x5a, 0xd1, 0x5d, 0x7c, 0xee,
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA /* extended_example/sb_image3_enc.bin */ 0x37, 0x80, 0x22, 0xcf, 0xba, 0x00, 0xe3, 0x19, 0xc9, 0xb2, 0x3e, 0x27, 0x71, 0x06, 0xd1, 0xa0, 0xa4, 0xc8, 0x72, 0x55, 0x4a, 0xd1, 0x76, 0xff, 0xc0, 0x47, 0xbc, 0x70, 0x25, 0xb6, 0x3f, 0x52, 0x15, 0x3d, 0xaa, 0x70, 0x4e, 0x79, 0xef, 0x24, 0x60, 0xa2, 0x17, 0x78, 0x40, 0x14, 0x94, 0x24, 0x5b, 0x90, 0x68, 0x50, 0x60, 0x65, 0x19, 0xa3, 0x90, 0x33, 0xb6, 0x09, 0x9c, 0xeb, 0xd9, 0x0a, 0x61, 0xe5, 0xcd, 0x58, 0x34, 0x21, 0x7d, 0x6d, 0x5e, 0xab, 0x73, 0x4f, 0x58, 0x3f, 0x55, 0xaa, 0xcb, 0xf9, 0xf2, 0xc8, 0x0f, 0x24, 0x4d, 0x29, 0x6e, 0x9e, 0x04, 0xe8, 0x81, 0xc0, 0x88, 0xac, 0xb8, 0x0b, 0xd2, 0x5e, 0xd8, 0x6e, 0x7b, 0x9c, 0xa5, 0x08, 0x90, 0x83, 0x70, 0xc7, 0x73, 0x6a,
#define RUNIT_SECURE_BOOT__CERT_1_BIN__DATA_LENGTH /* sb_key1_cert_test_10.bin */ 840
#define RUNIT_SECURE_BOOT__CERT_2_BIN__DATA_LENGTH /* sb_key2_cert_test_10.bin */ 840
#define RUNIT_SECURE_BOOT__CERT_3_BIN__DATA_LENGTH /* sb_content_cert_test_10.bin */ 972
#endif /* defined(CC_SB_X509_CERT_SUPPORTED) */
#endif /* defined(CORTEX_M33_FPGA) */
#else
#error "not supported"

#endif /* defined(DX_PLAT_ZYNQ7000) */

#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__PLAIN_TEXT_DATA /* extended_example/sb_image1.bin */ 0xc3, 0x1b, 0x35, 0xfc, 0xc8, 0xd4, 0x8a, 0x28, 0x55, 0x99, 0x3e, 0x74, 0x04, 0x03, 0x04, 0x79, 0x3e, 0x7f, 0x7b, 0x52, 0x97, 0x53, 0x28, 0x7b, 0x54, 0x2c, 0x8c, 0x5c, 0x34, 0x94, 0x5e, 0x39, 0x94, 0x4c, 0x24, 0x8c, 0x5b, 0x3a, 0x8e, 0x54, 0x2c, 0x98, 0x33, 0x0b, 0x7d, 0x5c, 0x3c, 0x9b, 0x55, 0x3a, 0x8a, 0x6e, 0x54, 0x9d, 0x3e, 0x1c, 0x84, 0x43, 0x24, 0x84, 0x4c, 0x2c, 0x8c, 0x54, 0x34, 0x94, 0x44, 0x24, 0x8c, 0x4c, 0x2c, 0x99, 0x56, 0x3a, 0x94, 0x57, 0x3c, 0x97, 0x5c, 0x40, 0x9c, 0x5c, 0x44, 0x94, 0x46, 0x35, 0x6e, 0x64, 0x4d, 0x9c, 0x74, 0x60, 0xa1, 0x4c, 0x34, 0x8c, 0x54, 0x3c, 0x94, 0x55, 0x3d, 0x95, 0x57, 0x3f, 0x97, 0x5c, 0x42, 0x9c, 0x37, 0x28, 0x5e, 0x5b,
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__PLAIN_TEXT_DATA /* extended_example/sb_image2.bin */ 0xc3, 0x1b, 0x35, 0xfc, 0xc8, 0xd4, 0x8a, 0x28, 0x55, 0x99, 0x3e, 0x74, 0x04, 0x03, 0x04, 0x79, 0x3e, 0x7f, 0x7b, 0x52, 0x97, 0x53, 0x28, 0x7b, 0x54, 0x2c, 0x8c, 0x5c, 0x34, 0x94, 0x5e, 0x39, 0x94, 0x4c, 0x24, 0x8c, 0x5b, 0x3a, 0x8e, 0x54, 0x2c, 0x98, 0x33, 0x0b, 0x7d, 0x5c, 0x3c, 0x9b, 0x55, 0x3a, 0x8a, 0x6e, 0x54, 0x9d, 0x3e, 0x1c, 0x84, 0x43, 0x24, 0x84, 0x4c, 0x2c, 0x8c, 0x54, 0x34, 0x94, 0x44, 0x24, 0x8c, 0x4c, 0x2c, 0x99, 0x56, 0x3a, 0x94, 0x57, 0x3c, 0x97, 0x5c, 0x40, 0x9c, 0x5c, 0x44, 0x94, 0x46, 0x35, 0x6e, 0x64, 0x4d, 0x9c, 0x74, 0x60, 0xa1, 0x4c, 0x34, 0x8c, 0x54, 0x3c, 0x94, 0x55, 0x3d, 0x95, 0x57, 0x3f, 0x97, 0x5c, 0x42, 0x9c, 0x37, 0x28, 0x5e, 0x5b,
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__PLAIN_TEXT_DATA /* extended_example/sb_image3.bin */ 0xc3, 0x1b, 0x35, 0xfc, 0xc8, 0xd4, 0x8a, 0x28, 0x55, 0x99, 0x3e, 0x74, 0x04, 0x03, 0x04, 0x79, 0x3e, 0x7f, 0x7b, 0x52, 0x97, 0x53, 0x28, 0x7b, 0x54, 0x2c, 0x8c, 0x5c, 0x34, 0x94, 0x5e, 0x39, 0x94, 0x4c, 0x24, 0x8c, 0x5b, 0x3a, 0x8e, 0x54, 0x2c, 0x98, 0x33, 0x0b, 0x7d, 0x5c, 0x3c, 0x9b, 0x55, 0x3a, 0x8a, 0x6e, 0x54, 0x9d, 0x3e, 0x1c, 0x84, 0x43, 0x24, 0x84, 0x4c, 0x2c, 0x8c, 0x54, 0x34, 0x94, 0x44, 0x24, 0x8c, 0x4c, 0x2c, 0x99, 0x56, 0x3a, 0x94, 0x57, 0x3c, 0x97, 0x5c, 0x40, 0x9c, 0x5c, 0x44, 0x94, 0x46, 0x35, 0x6e, 0x64, 0x4d, 0x9c, 0x74, 0x60, 0xa1, 0x4c, 0x34, 0x8c, 0x54, 0x3c, 0x94, 0x55, 0x3d, 0x95, 0x57, 0x3f, 0x97, 0x5c, 0x42, 0x9c, 0x37, 0x28, 0x5e, 0x5b,

#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__DATA_LENGTH /* sb_image1_enc.bin */ 112
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__DATA_LENGTH /* sb_image2_enc.bin */ 112
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA_LENGTH /* sb_image3_enc.bin */ 112
#define RUNIT_SECURE_BOOT__IMAGE_1_BIN__PLAIN_TEXT_DATA_LENGTH /* sb_image1.bin */ 112
#define RUNIT_SECURE_BOOT__IMAGE_2_BIN__PLAIN_TEXT_DATA_LENGTH /* sb_image2.bin */ 112
#define RUNIT_SECURE_BOOT__IMAGE_3_BIN__PLAIN_TEXT_DATA_LENGTH /* sb_image3.bin */ 112

#endif /* RUNIT_SECURE_BOOT_SKIP_TEST */

#define RUNIT_EXPECTED_SW_VERSION 5

#if !RUNIT_SECURE_BOOT_SKIP_TEST
/************************************************************
 *
 * types
 *
 ************************************************************/
typedef struct RunItBinaryFile_t
{
    /** mainly for debug */
    const char* name;
    /** length of data */
    uint32_t length;
    /** the binary data */
    uint8_t* data;
} RunItBinaryFile_t;

typedef struct RunItSecureBoot_SwCompEnc_t
{
    /** offset of image in flash */
    uint32_t flashAddr;
    /** address where to load the image in host's address space */
    uint32_t virtualHostOffset;
    /** binary file itself */
    RunItBinaryFile_t* swCompBin;
    /** plain text binary file used to compare result */
    RunItBinaryFile_t* swCompPlainTextBin;
} RunItSecureBoot_SwCompEnc_t;

typedef struct RunItSecureBootCertInfo_t
{
    RunItBinaryFile_t* cert;
    uint32_t storeFlashAddress;
} RunItSecureBootCertInfo_t;

typedef struct RunItSecureBoot_CertChain_t
{
    /** An Arrays of Maximum number of certificates */
    RunItSecureBootCertInfo_t certArr[RUNIT_MAX_CHAIN_LENGTH];
    /** a list of images in this certificate */
    RunItSecureBoot_SwCompEnc_t *cntCompsInfo;
    /** amount of images is the list */
    uint8_t numOfSwCompImg;
    uint32_t expSwVersion;
}RunItSecureBoot_CertChain_t;

/************************************************************
 *
 * static function prototypes
 *
 ************************************************************/
static RunItError_t runIt_loadParams(RunItBinaryFile_t* pCertBin,
                                     uint32_t flashAddr,
                                     RunItSecureBoot_SwCompEnc_t *pListOfSWComps,
                                     uint32_t numOfSwComps);
static RunItError_t runIt_flashLoadSecureBootCertificates(void);
static RunItError_t runIt_prepareFlash(void);

/************************************************************
 *
 * variables
 *
 ************************************************************/
static uint32_t OTP_SECURE_2_HBK128_VALUES[] = {

    0x01020408,
    0x01010101,
    0x01020408,
    0x02020202,
    0x01020408,
    0x04040404,
    0x01020408,
    0x08080808,
    0x04030201,
    0x08070605,
    0x0c0b0a09,
    0x000f0e0d,
    0xa68dc2f8,
    0xb7ae0c04,
    0x0bd8d969,
    0xb7cb523d,
    0x3c3f60e0,
    0x21f7b953,
    0xfe56a24d,
    0x4e3ec4ef,
    0xc21f3c40,
    0x2ff17bda,
    0xe93eb506,
    0x9aa54f0b,
    0x4ade752b,
    0x00000000,
    0x00000000,
    0x00000000,
    0x00000000,
    0xa68dc2f8,
    0xb7ae0c04,
    0x0bd8d969,
    0xb7cb523d,
    0x003f0038,
    0x00000001,
    0x00000000,
    0x00000007,
    0x00000000,
    0x00000000,
    0x00000000,
    0x00000000,
    0x00000000,
    0x00000000,
    0x00000000,

//    /*  [0x00-0x07]: 256bit Device root key (HUK) */
//    0x01020408, 0x01010101, 0x01020408, 0x02020202, 0x01020408, 0x04040404, 0x01020408, 0x08080808,
//    /*  [0x08-0x0B]: 128bit ICV Provosioning secret (Kpicv) */
//    0xEEEEAAAA, 0x11115555, 0x11115555, 0xEEEEAAAA,
//    /*  [0x0C-0x0F]: 128bit ICV Provosioning secret (Kceicv) */
//    0x00000000, 0x00000000, 0x00000000, 0x00000000,
//    /*  [0x10]: manufacturer programmed flag */
//    0x3C0040E0,
//    /*  [0x11-0x14]:  HBK0 */
//    0x21F7B953, 0xFE56A24D, 0x4E3EC4EF, 0xC21F3C40,
//    /*  [0x15-0x18]: HBK1 */
//    0x2FF17BDA, 0xE93EB506, 0x9AA54F0B, 0x4ADE752B,
//    /*  [0x19-0x1C]: 128bit OEM provisioning secret (Kcp) */
//    0x0000AAAA,  0x11115555, 0x11115555, 0x0000AAAA,
//    /*  [0x1D-0x20]: 128bit OEM code encryption secret (Kce) */
//    0x00000000, 0x00000000, 0x00000000, 0x00000000,
//    /*  [0x21]: OEM programmed flag */
//    0x00005838,
//    /*  [0x22-0x23]: Hbk0 trusted FW min version */
//    0x00000001, 0x00000000,
//    /*  [0x24-0x26]: Hbk1 trusted FW min version */
//    0x00000003, 0x00000000, 0x00000000,
//    /*  [0x27]: general purpose configuration flag */
//    0x00000000,
//    /*  [0x28-0x2B] - 128b DCU lock mask */
//    0x55551111, 0xAAAAEEEE, 0x55551111, 0xAAAAEEEE,

};

uint8_t runItsecureBoot_Cert_1_Bin_data[] = { RUNIT_SECURE_BOOT__CERT_1_BIN__DATA };

uint8_t runItsecureBoot_Cert_2_Bin_data[] = { RUNIT_SECURE_BOOT__CERT_2_BIN__DATA };

uint8_t runItsecureBoot_Cert_3_Bin_data[] = { RUNIT_SECURE_BOOT__CERT_3_BIN__DATA };

uint8_t runItsecureBoot_image1bin_data[] = { RUNIT_SECURE_BOOT__IMAGE_1_BIN__DATA };

uint8_t runItsecureBoot_image2bin_data[] = { RUNIT_SECURE_BOOT__IMAGE_2_BIN__DATA };

uint8_t runItsecureBoot_image3bin_data[] = { RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA };

uint8_t runItsecureBoot_image1bin_plainTextData[] = { RUNIT_SECURE_BOOT__IMAGE_1_BIN__PLAIN_TEXT_DATA };

uint8_t runItsecureBoot_image2bin_plainTextData[] = { RUNIT_SECURE_BOOT__IMAGE_2_BIN__PLAIN_TEXT_DATA };

uint8_t runItsecureBoot_image3bin_plainTextData[] = { RUNIT_SECURE_BOOT__IMAGE_3_BIN__PLAIN_TEXT_DATA };

RunItBinaryFile_t runItsecureBoot_key1CertBin =
{
    "KeyCert1" ,
    RUNIT_SECURE_BOOT__CERT_1_BIN__DATA_LENGTH,
    runItsecureBoot_Cert_1_Bin_data
};

RunItBinaryFile_t runItsecureBoot_key2CertBin =
{
    "KeyCert2" ,
    RUNIT_SECURE_BOOT__CERT_2_BIN__DATA_LENGTH,
    runItsecureBoot_Cert_2_Bin_data
};

RunItBinaryFile_t runItsecureBoot_contentEncCertBin =
{
    "ContentCert" ,
    RUNIT_SECURE_BOOT__CERT_3_BIN__DATA_LENGTH,
    runItsecureBoot_Cert_3_Bin_data
};

RunItBinaryFile_t runItsecureBoot_image1bin =
{
    "Image1" ,
    RUNIT_SECURE_BOOT__IMAGE_1_BIN__DATA_LENGTH,
    runItsecureBoot_image1bin_data,

};

RunItBinaryFile_t runItsecureBoot_image2bin =
{
    "Image2" ,
    RUNIT_SECURE_BOOT__IMAGE_2_BIN__DATA_LENGTH,
    runItsecureBoot_image2bin_data
};

RunItBinaryFile_t runItsecureBoot_image3bin =
{
    "Image3" ,
    RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA_LENGTH,
    runItsecureBoot_image3bin_data
};

RunItBinaryFile_t runItsecureBoot_image1PlainTextbin =
{
    "plainTextImage1" ,
    RUNIT_SECURE_BOOT__IMAGE_1_BIN__PLAIN_TEXT_DATA_LENGTH,
    runItsecureBoot_image1bin_plainTextData
};

RunItBinaryFile_t runItsecureBoot_image2PlainTextbin =
{
    "plainTextImage2" ,
    RUNIT_SECURE_BOOT__IMAGE_2_BIN__PLAIN_TEXT_DATA_LENGTH,
    runItsecureBoot_image2bin_plainTextData
};

RunItBinaryFile_t runItsecureBoot_image3PlainTextbin =
{
    "plainTextImage3" ,
    RUNIT_SECURE_BOOT__IMAGE_3_BIN__PLAIN_TEXT_DATA_LENGTH,
    runItsecureBoot_image3bin_plainTextData
};

RunItSecureBoot_SwCompEnc_t runItsecureBoot_cntCertEnc[] =
{
    {
        RUNIT_SECURE_BOOT__IMAGE_PKG_1_FLASH_ADDRESS,
        RUNIT_SECURE_BOOT__IMAGE_1_BIN__LOAD_ADDRESS,
        &runItsecureBoot_image1bin,
        &runItsecureBoot_image1PlainTextbin,
    },
    {
        RUNIT_SECURE_BOOT__IMAGE_PKG_2_FLASH_ADDRESS,
        RUNIT_SECURE_BOOT__IMAGE_2_BIN__LOAD_ADDRESS,
        &runItsecureBoot_image2bin,
        &runItsecureBoot_image2PlainTextbin,
    },
    {
        RUNIT_SECURE_BOOT__IMAGE_PKG_3_FLASH_ADDRESS,
        RUNIT_SECURE_BOOT__IMAGE_3_BIN__LOAD_ADDRESS,
        &runItsecureBoot_image3bin,
        &runItsecureBoot_image3PlainTextbin,
    }
};

const RunItSecureBoot_CertChain_t runIt_secureBoot_CertChain = {
    {
        /* 0 length in binary file indicates that certificate is not in use */
        { &runItsecureBoot_key1CertBin, RUNIT_SECURE_BOOT__KEY_PKG_1_FLASH_ADDRESS},
        { &runItsecureBoot_key2CertBin, RUNIT_SECURE_BOOT__KEY_PKG_2_FLASH_ADDRESS},
        { &runItsecureBoot_contentEncCertBin, RUNIT_SECURE_BOOT__CONTENT_PKG_1_FLASH_ADDRESS}
    },
    runItsecureBoot_cntCertEnc,
    ARRAY_SIZE(runItsecureBoot_cntCertEnc),
    RUNIT_EXPECTED_SW_VERSION
};

/************************************************************
 *
 * static functions
 *
 ************************************************************/
/**
 * @brief                   Load a certificate and a set of images to the flash as part of preparation towards the test.
 *
 * @param pCertBin          [input]  The binary to load to flash
 * @param flashAddr         [input] The address to load to
 * @param pListOfSWComps    [input] list of images to load
 * @param numOfSwComps      [input] number of images in list
 *
 * @return                  RUNIT_ERROR__OK on success, RUNIT_ERROR__FAIL otherwise
 */
static RunItError_t runIt_loadParams(RunItBinaryFile_t* pCertBin,
                                     uint32_t flashAddr,
                                     RunItSecureBoot_SwCompEnc_t *pListOfSWComps,
                                     uint32_t numOfSwComps)
{
    RunItError_t rc = RUNIT_ERROR__OK;
    uint32_t imageIndex;

    RUNIT_PRINT_DBG("Loading cert[%s] to flash at offset[%#"PRIxPTR"]\n", pCertBin->name, (uintptr_t)flashAddr);

    rc = runIt_flashWrite(flashAddr, pCertBin->data, pCertBin->length);
    if (rc != RUNIT_ERROR__OK)
    {
        RUNIT_PRINT_ERROR("failed to load cert[%s] to flash at offset[%#"PRIxPTR"]\n", pCertBin->name, (uintptr_t)flashAddr);
    }

    for (imageIndex = 0; imageIndex < numOfSwComps; ++imageIndex)
    {
        uint32_t imageFlashAddr = pListOfSWComps[imageIndex].flashAddr;
        RunItBinaryFile_t* pSwCompBin = pListOfSWComps[imageIndex].swCompBin;

        RUNIT_PRINT_DBG("Loading image[%s] to flash at offset[%#"PRIxPTR"]\n", pSwCompBin->name, (uintptr_t)imageFlashAddr);

        RUNIT_ASSERT(runIt_flashWrite(imageFlashAddr, pSwCompBin->data, pSwCompBin->length) == RUNIT_ERROR__OK);
    }

bail:
    return rc;
}

/**
 * @brief               Load the relevant boot certificates to Flash
 *
 * @return              RUNIT_ERROR__OK on success, RUNIT_ERROR__FAIL otherwise
 */
static RunItError_t runIt_flashLoadSecureBootCertificates(void)
{
    RunItError_t rc = RUNIT_ERROR__OK;

    const RunItSecureBoot_CertChain_t* pCertChain = &runIt_secureBoot_CertChain;
    uint8_t chainIndex = 0;

    const RunItSecureBootCertInfo_t *pCertInfo = NULL;

    /* load Key certificate to Flash file */
    pCertInfo = &pCertChain->certArr[chainIndex++];
    RUNIT_ASSERT(runIt_loadParams(pCertInfo->cert,
                                  pCertInfo->storeFlashAddress,
                                  NULL,
                                  0) == RUNIT_ERROR__OK);


    /* load Key certificate to Flash file */
    pCertInfo = &pCertChain->certArr[chainIndex++];
    RUNIT_ASSERT(runIt_loadParams(pCertInfo->cert,
                                  pCertInfo->storeFlashAddress,
                                  NULL,
                                  0) == RUNIT_ERROR__OK);


    /* load Key certificate to Flash file */
    pCertInfo = &pCertChain->certArr[chainIndex++];
    RUNIT_ASSERT(runIt_loadParams(pCertInfo->cert,
                                  pCertInfo->storeFlashAddress,
                                  pCertChain->cntCompsInfo,
                                  pCertChain->numOfSwCompImg) == RUNIT_ERROR__OK);

bail:
    return rc;
}

static RunItError_t runIt_prepareOtp(void)
{
    RunItError_t rc = RUNIT_ERROR__OK;

    RUNIT_ASSERT(runIt_burnOtp((unsigned int *)OTP_SECURE_2_HBK128_VALUES, CC_MNG_LCS_SEC_ENABLED) == RUNIT_ERROR__OK);

bail:
    return rc;
}

static RunItError_t runIt_prepareFlash(void)
{
    RunItError_t rc = RUNIT_ERROR__OK;

    RUNIT_ASSERT(runIt_flashInit(RUNIT_FLASH_SIZE) == RUNIT_ERROR__OK);
    RUNIT_ASSERT(runIt_flashLoadSecureBootCertificates() == RUNIT_ERROR__OK);

bail:
    return rc;
}

RunItError_t runIt_secureBoot(void)
{
    RunItError_t rc = RUNIT_ERROR__OK;

    const uint32_t LAST_IMAGE = 2;

    CCSbCertInfo_t sbCertInfoCtx;

    uint32_t *pWspace = NULL;
    uint32_t *pImage3Ptr = NULL;
    uint32_t *pCntCertPtr = NULL;

    RunItPtr wspacePtr;
    RunItPtr image3Ptr;
    RunItPtr cntCertPtr;

    const char* TEST_NAME = "SecureBoot";
    RUNIT_SUB_TEST_START(TEST_NAME);

    memset((uint8_t*) &sbCertInfoCtx, 0x0, sizeof(sbCertInfoCtx));

    ALLOC32(wspacePtr, pWspace, CC_SB_MIN_WORKSPACE_SIZE_IN_BYTES);
    ALLOC32(cntCertPtr, pCntCertPtr, CC_SB_MIN_WORKSPACE_SIZE_IN_BYTES);
    ALLOC32(image3Ptr, pImage3Ptr, CC_SB_MIN_WORKSPACE_SIZE_IN_BYTES);

    /* read modify write certificate */
    runIt_flashRead(RUNIT_SECURE_BOOT__CONTENT_PKG_1_FLASH_ADDRESS, (uint8_t*)pCntCertPtr, RUNIT_SECURE_BOOT__CERT_3_BIN__DATA_LENGTH);

    RUNIT_ASSERT_API(mbedtls_sb_sw_image_store_address_change(pCntCertPtr,
                                                              RUNIT_SECURE_BOOT__CERT_3_BIN__DATA_LENGTH,
                                                              (CCAddr_t) RUNIT_SECURE_BOOT__IMAGE_PKG_3_NEW_FLASH_ADDRESS,
                                                              LAST_IMAGE) == CC_OK);

    runIt_flashWrite(RUNIT_SECURE_BOOT__CONTENT_PKG_1_FLASH_ADDRESS, (uint8_t*)pCntCertPtr, RUNIT_SECURE_BOOT__CERT_3_BIN__DATA_LENGTH);

    /* move image 3 to new location */
    runIt_flashRead(RUNIT_SECURE_BOOT__IMAGE_PKG_3_FLASH_ADDRESS, (uint8_t*)pImage3Ptr, RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA_LENGTH);
    runIt_flashWrite(RUNIT_SECURE_BOOT__IMAGE_PKG_3_NEW_FLASH_ADDRESS, (uint8_t*)pImage3Ptr, RUNIT_SECURE_BOOT__IMAGE_3_BIN__DATA_LENGTH);

    /* verify images and load them */
    RUNIT_ASSERT_API(mbedtls_sb_cert_chain_cerification_init(&sbCertInfoCtx) == CC_OK);

    RUNIT_ASSERT_API(mbedtls_sb_cert_verify_single((CCSbFlashReadFunc)runIt_flashReadWrap,
                                                   NULL,
                                                   RUNIT_SECURE_BOOT__KEY_PKG_1_FLASH_ADDRESS,
                                                   &sbCertInfoCtx,
                                                   NULL,
                                                   0,
                                                   pWspace,
                                                   CC_SB_MIN_WORKSPACE_SIZE_IN_BYTES) == CC_OK);

    RUNIT_ASSERT_API(mbedtls_sb_cert_verify_single((CCSbFlashReadFunc)runIt_flashReadWrap,
                                                   NULL,
                                                   RUNIT_SECURE_BOOT__KEY_PKG_2_FLASH_ADDRESS,
                                                   &sbCertInfoCtx,
                                                   NULL,
                                                   0,
                                                   pWspace,
                                                   CC_SB_MIN_WORKSPACE_SIZE_IN_BYTES) == CC_OK);

    RUNIT_ASSERT_API(mbedtls_sb_cert_verify_single((CCSbFlashReadFunc)runIt_flashReadWrap,
                                                   NULL,
                                                   RUNIT_SECURE_BOOT__CONTENT_PKG_1_FLASH_ADDRESS,
                                                   &sbCertInfoCtx,
                                                   NULL,
                                                   0,
                                                   pWspace,
                                                   CC_SB_MIN_WORKSPACE_SIZE_IN_BYTES) == CC_OK);

    /* no need to verify images were loaded succesfully since we do not allow loading in runtime */

bail:

    FREE_IF_NOT_NULL(wspacePtr);
    FREE_IF_NOT_NULL(cntCertPtr);
    FREE_IF_NOT_NULL(image3Ptr);

    RUNIT_SUB_TEST_RESULT_W_PARAMS(TEST_NAME, "IS_X509[%s]", IS_X509);
    return rc;
}

#endif /* !RUNIT_SECURE_BOOT_SKIP_TEST */
/************************************************************
 *
 * public functions
 *
 ************************************************************/
RunItError_t runIt_secureBootTest(void)
{
    RunItError_t rc = RUNIT_ERROR__OK;

#if !RUNIT_SECURE_BOOT_SKIP_TEST
    const char* TEST_NAME = "SecureBoot";
    RUNIT_TEST_START(TEST_NAME);

    RUNIT_ASSERT(runIt_prepareFlash() == RUNIT_ERROR__OK);
    RUNIT_ASSERT(runIt_prepareOtp() == RUNIT_ERROR__OK);
    RUNIT_ASSERT(runIt_secureBoot() == RUNIT_ERROR__OK);

bail:

    runIt_flashFinalize();

    RUNIT_TEST_RESULT(TEST_NAME);
#endif /* !RUNIT_SECURE_BOOT_SKIP_TEST */
    return rc;
}
