/******************************************************************************

@file  app_ranging_client.c

@brief This file contains the implementation of the Ranging Requester
       (RREQ) APIs and functionality.

Group: WCS, BTS
Target Device: cc23xx

******************************************************************************

 Copyright (c) 2025, Texas Instruments Incorporated
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 *  Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

 *  Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

 *  Neither the name of Texas Instruments Incorporated nor the names of
    its contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

******************************************************************************


*****************************************************************************/

#ifdef RANGING_CLIENT
//*****************************************************************************
//! Includes
//*****************************************************************************
#include "ti_ble_config.h"
#include "ti/ble/app_util/framework/bleapputil_api.h"
#include "ti/ble/app_util/framework/bleapputil_timers.h"
#include "app_ranging_client_api.h"


//*****************************************************************************
//! Defines
//*****************************************************************************

//*****************************************************************************
//! Prototypes
//*****************************************************************************
static void AppRREQ_dataReadyHandler(uint16_t connHandle, uint16_t rangingCount);
static void AppRREQ_statusHandle(uint16_t connHandle, uint8_t statusCode, uint8_t statusDataLen, uint8_t* statusData);
static void AppRREQ_completeEventHandler(uint16_t connHandle, uint16_t rangingCount, uint8_t status, uint16_t dataLen, uint8_t* pData);
static void AppRREQ_timerCB(BLEAppUtil_timerHandle timerHandle, BLEAppUtil_timerTermReason_e reason, void *pData);
static bool AppRREQ_extEvtHandler(AppRREQEventType_e event, BLEAppUtil_msgHdr_t *pMsgData);

// Contains the inputs for the enable API to be passed to the timer
typedef struct
{
  uint16_t connHandle;
  RREQEnableModeType_e enableMode;
}AppRREQ_enableData_t;

//*****************************************************************************
//! Globals
//*****************************************************************************

// The configuration of the Ranging Requester
const RREQConfig_t reqConfig =
{
  .subConfig.onDemandSubType = RREQ_PREFER_NOTIFY,        // Notification for on-demand data
  .subConfig.controlPointSubType = RREQ_PREFER_NOTIFY,    // Notification for control point commands
  .subConfig.dataReadySubType = RREQ_PREFER_NOTIFY,       // Notification for data ready events
  .subConfig.overwrittenSubType = RREQ_PREFER_NOTIFY,     // Notification for data ready events
  .timeOutDataReady  = RREQ_TIMEOUT_DATA_READY_MS,        // Timeout for data ready event in milliseconds
  .timeOutNextSegment = RREQ_TIMEOUT_SUBEVENT_DATA_MS,    // Timeout for next segment event in milliseconds
  .timeOutCompleteEvent = RREQ_TIMEOUT_COMPLETE_EVENT_MS  // Timeout for complete event in milliseconds
};

// The call back of the Ranging Requester
const RREQCallbacks_t gRREQDataReadyCB =
{
  .pDataReadyCallback = AppRREQ_dataReadyHandler,              // Callback for data ready events
  .pDataCompleteEventCallback = AppRREQ_completeEventHandler,  // Callback for complete events
  .pStatusCallback = AppRREQ_statusHandle,                     // Callback for status events
};

// The external event handler for RREQ events
static AppRREQ_eventHandler_t gExtEvtHandler = NULL;

// The external event handler for data complete events
static AppRREQ_dataEventHandler_t gDataCompleteEvtHandler = NULL;

// Timer handle for the RREQ Enable
BLEAppUtil_timerHandle gEnableTimerHandle = BLEAPPUTIL_TIMER_INVALID_HANDLE;

//*****************************************************************************
//! Functions
//*****************************************************************************

/*********************************************************************
 * @fn      AppRREQ_start
 *
 * @brief   This function is called to start the RREQ module.
 *
 * input parameters
 *
 * @param   none
 *
 * output parameters
 *
 * @param   None
 *
 * @return  SUCCESS or an error status indicating the failure reason.
 */
uint8_t AppRREQ_start(void)
{
  uint8_t status = USUCCESS;

  // Start the ranging requester
  status = RREQ_Start(&gRREQDataReadyCB ,&reqConfig);

  return status;
}

/*********************************************************************
 * @fn      AppRREQ_enable
 *
 * @brief   Enables the RREQ process.
 *          This function start the RREQ process by discovering
 *          the RAS (Ranging Service) service on the specified
 *          connection handle
 *
 * input parameters
 *
 * @param   connHandle - The connection handle for the RAS service
 * @param   enableMode - The mode to enable (RREQEnableModeType_e)
 *
 * output parameters
 *
 * @param   None
 *
 * @return  SUCCESS - if the RREQ process was successfully enabled.
 *          blePending - if the operation is pending and will be completed later.
 *          bleMemAllocError - if memory allocation failed.
 *          INVALIDPARAMETER - if the connection handle is invalid or the enable mode is not supported.
 */
uint8_t AppRREQ_enable(uint16_t connHandle, RREQEnableModeType_e enableMode)
{
  uint8_t status = SUCCESS;

  // Call the ranging client enable API
  status = RREQ_Enable(connHandle, enableMode);
  if(status == blePending)
  {
    linkDBInfo_t connInfo = {0};
    // Get the connection information
    status = linkDB_GetInfo(connHandle, &connInfo);

    // Check if the connection is valid
    if ( (status != bleTimeout) && (status != bleNotConnected))
    {
      // Allocate memory for the enable data
      AppRREQ_enableData_t *pEnableData = (AppRREQ_enableData_t *)BLEAppUtil_malloc(sizeof(AppRREQ_enableData_t));
      if(pEnableData != NULL)
      {
        // Fill the enable data structure
        pEnableData->connHandle = connHandle;
        pEnableData->enableMode = enableMode;

        if (gEnableTimerHandle == BLEAPPUTIL_TIMER_INVALID_HANDLE)
        {
          // Start the timer for the enable process
          gEnableTimerHandle = BLEAppUtil_startTimer(AppRREQ_timerCB, connInfo.connInterval, false, pEnableData);
        }

        // Update the status to pending
        status = blePending;
      }
      else
      {
        // Failed to allocate memory, handle the error
        status = bleMemAllocError;
      }
    }
  }

  return status;
}

/*********************************************************************
 * @fn      AppRREQ_disable
 *
 * @brief Disable the RREQ process.
 *        This function start the RREQ process by discovering
 *        the RAS (Ranging Service) service on the specified
 *        connection handle
 *
 * input parameters
 *
 * @param connHandle - Connection handle.
 *
 * output parameters
 *
 * @param   None
 *
 * @return SUCCESS - if the RREQ process was successfully disabled.
 *         INVALIDPARAMETER - if the connection handle is invalid.
 */
uint8_t AppRREQ_disable(uint16_t connHandle)
{
  return RREQ_Disable(connHandle);
}

/*********************************************************************
 * @fn      AppRREQ_abort
 *
 * @brief abort the RREQ process.
 *        This function abort the RREQ process.
 *
 * input parameters
 *
 * @param connHandle - Connection handle.
 *
 * output parameters
 *
 * @param   None
 *
 * @return SUCCESS - if the RREQ process was successfully aborted.
 *         INVALIDPARAMETER - if the connection handle is invalid.
 */
uint8_t AppRREQ_abort(uint16_t connHandle)
{
  return RREQ_Abort(connHandle);
}

/*********************************************************************
 * @fn      AppRREQ_GetRangingData
 *
 * @brief   Starts the process of reading data for a ranging request.
 *          This function initiates the data reading process for a specified
 *          connection handle and ranging count.
 *
 * input parameters
 *
 * @param   connHandle - Connection handle.
 * @param   RangingCount - CS procedure counter.
 *
 * output parameters
 *
 * @param   None
 *
 * @return return SUCCESS or an error status indicating the failure reason.
 */
uint8_t AppRREQ_GetRangingData(uint16_t connHandle, uint16_t rangingCount)
{
  return RREQ_GetRangingData(connHandle, rangingCount);
}

/*********************************************************************
 * @fn      AppRREQ_extEvtHandler
 *
 * @brief   The purpose of this function is to forward the event to the external
 *          event handler that registered to handle the events.
 *
 * input parameters
 *
 * @param   event     - RREQ message event.
 * @param   pMsgData  - pointer to message data.
 *
 * output parameters
 *
 * @param   None
 *
 * @return  true if the event was forwarded to the external handler, false otherwise.
 */
static bool AppRREQ_extEvtHandler(AppRREQEventType_e event, BLEAppUtil_msgHdr_t *pMsgData)
{
  bool reVal = false;

  // Send the event to the upper layer if its handle exists
  if (gExtEvtHandler != NULL)
  {
    gExtEvtHandler( event, pMsgData);
    reVal = true;
  }

  return reVal;
}

/*********************************************************************
 * @fn      AppRREQ_registerEvtHandler
 *
 * @brief   This function is called to register the external event handler
 *          function.
 *
 * input parameters
 *
 * @param   fEventHandler - pointer to the event handler function.
 *
 * output parameters
 *
 * @param   None
 *
 * @return  None
 */
void AppRREQ_registerEvtHandler(AppRREQ_eventHandler_t fEventHandler)
{
  gExtEvtHandler = fEventHandler;
}

/*********************************************************************
 * @fn      AppRREQ_registerDataCompleteEvtHandler
 *
 * @brief   This function is called to register the external event handler
 *          function for data complete events.
 *
 * input parameters
 *
 * @param   pDataEventHandler - pointer to the event handler function.
 *
 * output parameters
 *
 * @param   None
 *
 * @return  None
 */
void AppRREQ_registerDataEvtHandler(AppRREQ_dataEventHandler_t pDataEventHandler)
{
  gDataCompleteEvtHandler = pDataEventHandler;
}

/*********************************************************************
 * @fn      AppRREQ_dataReadyHandler
 *
 * @brief   Callback function triggered when the RAS server is ready to send data.
 *          This function is called when the RAS server indicates that it has data
 *          available for the client. It triggers the RREQ module to start reading
 *          the data from the server.
 *
 * input parameters
 *
 * @param connHandle -  The connection handle associated with the data.
 * @param rangingCount  - CS procedure counter.
 *
 * output parameters
 *
 * @param   None
 *
 * @return None
 */
static void AppRREQ_dataReadyHandler(uint16_t connHandle, uint16_t rangingCount)
{
  AppRREQDataReady_t dataReadyInfo;
  dataReadyInfo.connHandle = connHandle;
  dataReadyInfo.rangingCount = rangingCount;
  bool isEventForwarded;

  isEventForwarded = AppRREQ_extEvtHandler(APP_RREQ_EVENT_DATA_READY, (BLEAppUtil_msgHdr_t *)&dataReadyInfo);

  if (isEventForwarded == false)
  {
    // if the event was not forwarded, trigger the data reading process internally as default behavior
    RREQ_GetRangingData(connHandle, rangingCount);
  }
}

/*********************************************************************
 * @fn      AppRREQ_statusHandle
 *
 * @brief   Callback function triggered when an error occurs in the RAS client.
 *          This function is called when an error occurs during the RAS client operation.
 *          It currently does not perform any specific action but can be extended in the future.
 *
 * input parameters
 *
 * @param   connHandle - The connection handle associated with the error.
 * @param   statusCode - The error code indicating the type of error.
 * @param   statusDatalen - length of statusData
 * @param   statusData - Additional data related to the error, if any.
 *
 * output parameters
 *
 * @param    None
 *
 * @return   None
 */
static void AppRREQ_statusHandle(uint16_t connHandle, uint8_t statusCode, uint8_t statusDataLen, uint8_t* statusData)
{
  AppRREQStatus_t AppRREQStatus;

  // Prepare the error status structure
  AppRREQStatus.connHandle    = connHandle;
  AppRREQStatus.statusCode    = statusCode;
  AppRREQStatus.statusData    = statusData;
  AppRREQStatus.statusDataLen = statusDataLen;

  // Send the event to the upper layer
  AppRREQ_extEvtHandler(APP_RREQ_EVENT_STATUS, (BLEAppUtil_msgHdr_t *)&AppRREQStatus);
}

/*********************************************************************
 * @fn      AppRREQ_completeEventHandler
 *
 * @brief   Handles the completion of an application request event.
 *          This function is triggered when an application request event is completed.
 *          It performs necessary operations to measure distance.
 *
 * input parameters
 *
 * @param connHandle - The connection handle associated with the event.
 * @param rangingCount - The CS procedure counter of ranging data available.
 * @param status - The status of the event (SUCCESS or error).
 * @param peerDataLen - The length of the data received from the peer device.
 * @param peerData - Pointer to the data received from the peer device.
 *
 * output parameters
 *
 * @param   None
 *
 * @return   None
 */
static void AppRREQ_completeEventHandler(uint16_t connHandle, uint16_t rangingCount, uint8_t status, uint16_t dataLen, uint8_t* pData)
{
  // Send the event to the upper layer if its handle exists
  if (gDataCompleteEvtHandler != NULL)
  {
    gDataCompleteEvtHandler(connHandle, status, dataLen, pData);
  }
}

/*********************************************************************
 * @fn      AppRREQ_timerCB
 *
 * @brief   Timer callback function for the RREQ enable process.
 *          This function is called when the timer expires.
 *          It checks the reason for the timer expiration and
 *          performs the necessary actions.
 *
 * input parameters
 *
 * @param   timerHandle - The handle of the timer that expired.
 * @param   reason - The reason for the timer expiration.
 * @param   pData - Pointer to the data associated with the timer.
 *
 * output parameters
 *
 * @param   None
 *
 * @return None
 */
static void AppRREQ_timerCB(BLEAppUtil_timerHandle timerHandle, BLEAppUtil_timerTermReason_e reason, void *pData)
{
  if (reason == BLEAPPUTIL_TIMER_TIMEOUT)
  {
    if (pData != NULL)
    {
      // Fill the enable data structure
      AppRREQ_enableData_t enableData;
      enableData.connHandle = ((AppRREQ_enableData_t *)pData)->connHandle;
      enableData.enableMode = ((AppRREQ_enableData_t *)pData)->enableMode;

      // Free the allocated memory for the enable data
      BLEAppUtil_free(pData);

      // Reset the timer handle
      gEnableTimerHandle = BLEAPPUTIL_TIMER_INVALID_HANDLE;

      // Call the RREQ enable function
      AppRREQ_enable(enableData.connHandle, enableData.enableMode);
    }
  }
  else
  {
    // Reset the timer handle
    gEnableTimerHandle = BLEAPPUTIL_TIMER_INVALID_HANDLE;
  }
}

#endif // RANGING_CLIENT
