/*
 * Copyright (c) 2019-2023,Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms,with or without
 * modification,are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice,this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice,this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES,INCLUDING,BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT,INDIRECT,INCIDENTAL,SPECIAL,
 * EXEMPLARY,OR CONSEQUENTIAL DAMAGES (INCLUDING,BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,DATA,OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT,STRICT LIABILITY,OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* Includes */
#include <stdint.h>

#include <ti/drivers/ECDSA.h>

#include <ti/drivers/cryptoutils/hsm/HSMLPF3Utility.h>

#include "ecdsaTestVector.h"

ECDSA_SignVector signVectorsP256[] = {
    {
        .privateKey =
            {
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F,
            },
        .publicKey =
            {
                /* clang-format off */
                0x04,
                /* X */
                0xF0, 0x45, 0x4D, 0xC6, 0x97, 0x1A, 0xBA, 0xE7, 0xAD, 0xFB, 0x37, 0x89, 0x99, 0x88, 0x82, 0x65,
                0xAE, 0x03, 0xAF, 0x92, 0xDE, 0x3A, 0x0E, 0xF1, 0x63, 0x66, 0x8C, 0x63, 0xE5, 0x9B, 0x9D, 0x5F,
                /* Y */
                0xB5, 0xB9, 0x3E, 0xE3, 0x59, 0x2E, 0x2D, 0x1F, 0x4E, 0x65, 0x94, 0xE5, 0x1F, 0x96, 0x43, 0xE6,
                0x2A, 0x3B, 0x21, 0xCE, 0x75, 0xB5, 0xFA, 0x3F, 0x47, 0xE5, 0x9C, 0xDE, 0x0D, 0x03, 0x4F, 0x36,
                /* clang-format on */
            },
        .hash =
            {
                0xA4, 0x1A, 0x41, 0xA1, 0x2A, 0x79, 0x95, 0x48, 0x21, 0x1C, 0x41, 0x0C, 0x65, 0xD8, 0x13, 0x3A,
                0xFD, 0xE3, 0x4D, 0x28, 0xBD, 0xD5, 0x42, 0xE4, 0xB6, 0x80, 0xCF, 0x28, 0x99, 0xC8, 0xA8, 0xC4,
            },
        .r           = {0},
        .s           = {0},
        .curveLength = BITS_TO_BYTES(ECDSA_CURVE_LENGTH_256),
        .curveType   = ECDSA_TYPE_SEC_P_256_R1,
    },
    {
        /* http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/ECDSA_Prime.pdf */
        .privateKey =
            {
                0xC4, 0x77, 0xF9, 0xF6, 0x5C, 0x22, 0xCC, 0xE2, 0x06, 0x57, 0xFA, 0xA5, 0xB2, 0xD1, 0xD8, 0x12,
                0x23, 0x36, 0xF8, 0x51, 0xA5, 0x08, 0xA1, 0xED, 0x04, 0xE4, 0x79, 0xC3, 0x49, 0x85, 0xBF, 0x96,
            },
        .publicKey =
            {
                /* clang-format off */
                0x04,
                /* X */
                0xB7, 0xE0, 0x8A, 0xFD, 0xFE, 0x94, 0xBA, 0xD3, 0xF1, 0xDC, 0x8C, 0x73, 0x47, 0x98, 0xBA, 0x1C,
                0x62, 0xB3, 0xA0, 0xAD, 0x1E, 0x9E, 0xA2, 0xA3, 0x82, 0x01, 0xCD, 0x08, 0x89, 0xBC, 0x7A, 0x19,
                /* Y */
                0x36, 0x03, 0xF7, 0x47, 0x95, 0x9D, 0xBF, 0x7A, 0x4B, 0xB2, 0x26, 0xE4, 0x19, 0x28, 0x72, 0x90,
                0x63, 0xAD, 0xC7, 0xAE, 0x43, 0x52, 0x9E, 0x61, 0xB5, 0x63, 0xBB, 0xC6, 0x06, 0xCC, 0x5E, 0x09,
                /* clang-format on */
            },
        .hash =
            {
                0xA4, 0x1A, 0x41, 0xA1, 0x2A, 0x79, 0x95, 0x48, 0x21, 0x1C, 0x41, 0x0C, 0x65, 0xD8, 0x13, 0x3A,
                0xFD, 0xE3, 0x4D, 0x28, 0xBD, 0xD5, 0x42, 0xE4, 0xB6, 0x80, 0xCF, 0x28, 0x99, 0xC8, 0xA8, 0xC4,
            },
        .r           = {0},
        .s           = {0},
        .curveLength = BITS_TO_BYTES(ECDSA_CURVE_LENGTH_256),
        .curveType   = ECDSA_TYPE_SEC_P_256_R1,
    },
};

ECDSA_SignVector signVectorsP224[] = {{
    /* http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/ECDSA_Prime.pdf */
    .privateKey =
        {
            0x3F, 0x0C, 0x48, 0x8E, 0x98, 0x7C, 0x80, 0xBE, 0x0F, 0xEE, 0x52, 0x1F, 0x8D, 0x90,
            0xBE, 0x60, 0x34, 0xEC, 0x69, 0xAE, 0x11, 0xCA, 0x72, 0xAA, 0x77, 0x74, 0x81, 0xE8,
        },
    .publicKey =
        {
            /* clang-format off */
            0x04,
            /* X */
            0xE8, 0x4F, 0xB0, 0xB8, 0xE7, 0x00, 0x0C, 0xB6, 0x57, 0xD7, 0x97, 0x3C, 0xF6, 0xB4,
            0x2E, 0xD7, 0x8B, 0x30, 0x16, 0x74, 0x27, 0x6D, 0xF7, 0x44, 0xAF, 0x13, 0x0B, 0x3E,
            /* Y */
            0x43, 0x76, 0x67, 0x5C, 0x6F, 0xC5, 0x61, 0x2C, 0x21, 0xA0, 0xFF, 0x2D, 0x2A, 0x89,
            0xD2, 0x98, 0x7D, 0xF7, 0xA2, 0xBC, 0x52, 0x18, 0x3B, 0x59, 0x82, 0x29, 0x85, 0x55,
            /* clang-format on */
        },
    .hash =
        {
            0x1F, 0x1E, 0x1C, 0xF8, 0x92, 0x92, 0x6C, 0xFC, 0xCF, 0xC5, 0xA2, 0x8F, 0xEE, 0xF3,
            0xD8, 0x07, 0xD2, 0x3F, 0x77, 0x80, 0x08, 0xDB, 0xA4, 0xB3, 0x5F, 0x04, 0xB2, 0xFD,
        },
    .r =
        {
            0
            /* C3A3F5B8 27125320 04C6F6D1 DB672F55 D931C340 9EA1216D 0BE77380 */
        },
    .s =
        {
            0
            /* C5AA1EAE 6095DEA3 4C9BD84D A3852CCA 41A8BD9D 5548F36D ABDF6617 */
        },
    .curveLength = BITS_TO_BYTES(ECDSA_CURVE_LENGTH_224),
    .curveType   = ECDSA_TYPE_SEC_P_224_R1,
}};

ECDSA_SignVector signVectorsP384[] = {{
    /* http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/ECDSA_Prime.pdf */
    .privateKey =
        {
            0x20, 0x1B, 0x43, 0x2D, 0x8D, 0xF1, 0x43, 0x24, 0x18, 0x2D, 0x62, 0x61, 0xDB, 0x3E, 0x4B, 0x3F,
            0x46, 0xA8, 0x28, 0x44, 0x82, 0xD5, 0x2E, 0x37, 0x0D, 0xA4, 0x1E, 0x6C, 0xBD, 0xF4, 0x5E, 0xC2,
            0x95, 0x2F, 0x5D, 0xB7, 0xCC, 0xBC, 0xE3, 0xBC, 0x29, 0x44, 0x9F, 0x4F, 0xB0, 0x80, 0xAC, 0x97,
        },
    .publicKey =
        {
            /* clang-format off */
            0x04,
            /* X */
            0xC2, 0xB4, 0x79, 0x44, 0xFB, 0x5D, 0xE3, 0x42, 0xD0, 0x32, 0x85, 0x88, 0x01, 0x77, 0xCA, 0x5F,
            0x7D, 0x0F, 0x2F, 0xCA, 0xD7, 0x67, 0x8C, 0xCE, 0x42, 0x29, 0xD6, 0xE1, 0x93, 0x2F, 0xCA, 0xC1,
            0x1B, 0xFC, 0x3C, 0x3E, 0x97, 0xD9, 0x42, 0xA3, 0xC5, 0x6B, 0xF3, 0x41, 0x23, 0x01, 0x3D, 0xBF,
            /* Y */
            0x37, 0x25, 0x79, 0x06, 0xA8, 0x22, 0x38, 0x66, 0xED, 0xA0, 0x74, 0x3C, 0x51, 0x96, 0x16, 0xA7,
            0x6A, 0x75, 0x8A, 0xE5, 0x8A, 0xEE, 0x81, 0xC5, 0xFD, 0x35, 0xFB, 0xF3, 0xA8, 0x55, 0xB7, 0x75,
            0x4A, 0x36, 0xD4, 0xA0, 0x67, 0x2D, 0xF9, 0x5D, 0x6C, 0x44, 0xA8, 0x1C, 0xF7, 0x62, 0x0C, 0x2D
            /* clang-format on */
        },
    .hash =
        {
            /* Original Message
            0x6B, 0x45, 0xD8, 0x80, 0x37, 0x39, 0x2E, 0x13, 0x71, 0xD9, 0xFD, 0x1C, 0xD1, 0x74, 0xE9, 0xC1,
            0x83, 0x8D, 0x11, 0xC3, 0xD6, 0x13, 0x3D, 0xC1, 0x7E, 0x65, 0xFA, 0x0C, 0x48, 0x5D, 0xCC, 0xA9,
            0xF5, 0x2D, 0x41, 0xB6, 0x01, 0x61, 0x24, 0x60, 0x39, 0xE4, 0x2E, 0xC7, 0x84, 0xD4, 0x94, 0x00,
            0xBF, 0xFD, 0xB5, 0x14, 0x59, 0xF5, 0xDE, 0x65, 0x40, 0x91, 0x30, 0x1A, 0x09, 0x37, 0x8F, 0x93,
            0x46, 0x4D, 0x52, 0x11, 0x8B, 0x48, 0xD4, 0x4B, 0x30, 0xD7, 0x81, 0xEB, 0x1D, 0xBE, 0xD0, 0x9D,
            0xA1, 0x1F, 0xB4, 0xC8, 0x18, 0xDB, 0xD4, 0x42, 0xD1, 0x61, 0xAB, 0xA4, 0xB9, 0xED, 0xC7, 0x9F,
            0x05, 0xE4, 0xB7, 0xE4, 0x01, 0x65, 0x13, 0x95, 0xB5, 0x3B, 0xD8, 0xB5, 0xBD, 0x3F, 0x2A, 0xAA,
            0x6A, 0x00, 0x87, 0x7F, 0xA9, 0xB4, 0x5C, 0xAD, 0xB8, 0xE6, 0x48, 0x55, 0x0B, 0x4C, 0x6C, 0xBE,
            */
            /* Original Message's Hash */
            0x31, 0xa4, 0x52, 0xd6, 0x16, 0x4d, 0x90, 0x4b, 0xb5, 0x72, 0x4c, 0x87, 0x82, 0x80, 0x23, 0x1e,
            0xae, 0x70, 0x5c, 0x29, 0xce, 0x9d, 0x4b, 0xc7, 0xd5, 0x8e, 0x02, 0x0e, 0x10, 0x85, 0xf1, 0x7e,
            0xeb, 0xcc, 0x1a, 0x38, 0xf0, 0xed, 0x0b, 0xf2, 0xb3, 0x44, 0xd8, 0x1f, 0xbd, 0x89, 0x68, 0x25,
        },
    .r =
        {
            0
            /*
            0x50, 0x83, 0x5A, 0x92, 0x51, 0xBA, 0xD0, 0x08, 0x10, 0x61, 0x77, 0xEF, 0x00, 0x4B, 0x09, 0x1A,
            0x1E, 0x42, 0x35, 0xCD, 0x0D, 0xA8, 0x4F, 0xFF, 0x54, 0x54, 0x2B, 0x0E, 0xD7, 0x55, 0xC1, 0xD6,
            0xF2, 0x51, 0x60, 0x9D, 0x14, 0xEC, 0xF1, 0x8F, 0x9E, 0x1D, 0xDF, 0xE6, 0x9B, 0x94, 0x6E, 0x32,
            */
        },
    .s =
        {
            0
            /*
            0x04, 0x75, 0xF3, 0xD3, 0x0C, 0x64, 0x63, 0xB6, 0x46, 0xE8, 0xD3, 0xBF, 0x24, 0x55, 0x83, 0x03,
            0x14, 0x61, 0x1C, 0xBD, 0xE4, 0x04, 0xBE, 0x51, 0x8B, 0x14, 0x46, 0x4F, 0xDB, 0x19, 0x5F, 0xDC,
            0xC9, 0x2E, 0xB2, 0x22, 0xE6, 0x1F, 0x42, 0x6A, 0x4A, 0x59, 0x2C, 0x00, 0xA6, 0xA8, 0x97, 0x21,
            */
        },
    .curveLength = BITS_TO_BYTES(ECDSA_CURVE_LENGTH_384),
    .curveType   = ECDSA_TYPE_SEC_P_384_R1,
}};

ECDSA_SignVector signVectorsP521[] = {{
    /* http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/ECDSA_Prime.pdf */
    .privateKey =
        {
            0x00, 0xF7, 0x49, 0xD3, 0x27, 0x04, 0xBC, 0x53, 0x3C, 0xA8, 0x2C, 0xEF, 0x0A, 0xCF, 0x10, 0x3D, 0x8F,
            0x4F, 0xBA, 0x67, 0xF0, 0x8D, 0x26, 0x78, 0xE5, 0x15, 0xED, 0x7D, 0xB8, 0x86, 0x26, 0x7F, 0xFA, 0xF0,
            0x2F, 0xAB, 0x00, 0x80, 0xDC, 0xA2, 0x35, 0x9B, 0x72, 0xF5, 0x74, 0xCC, 0xC2, 0x9A, 0x0F, 0x21, 0x8C,
            0x86, 0x55, 0xC0, 0xCC, 0xCF, 0x9F, 0xEE, 0x6C, 0x5E, 0x56, 0x7A, 0xA1, 0x4C, 0xB9, 0x26,
        },
    .publicKey =
        {
            /* clang-format off */
            0x04,
            /* X */
            0x00, 0x61, 0x38, 0x7F, 0xD6, 0xB9, 0x59, 0x14, 0xE8, 0x85, 0xF9, 0x12, 0xED, 0xFB, 0xB5, 0xFB,
            0x27, 0x46, 0x55, 0x02, 0x7F, 0x21, 0x6C, 0x40, 0x91, 0xCA, 0x83, 0xE1, 0x93, 0x36, 0x74, 0x0F,
            0xD8, 0x1A, 0xED, 0xFE, 0x04, 0x7F, 0x51, 0xB4, 0x2B, 0xDF, 0x68, 0x16, 0x11, 0x21, 0x01, 0x3E,
            0x0D, 0x55, 0xB1, 0x17, 0xA1, 0x4E, 0x43, 0x03, 0xF9, 0x26, 0xC8, 0xDE, 0xBB, 0x77, 0xA7, 0xFD,
            0xAA, 0xD1,
            /* Y */
            0x00, 0xE7, 0xD0, 0xC7, 0x5C, 0x38, 0x62, 0x6E, 0x89, 0x5C, 0xA2, 0x15, 0x26, 0xB9, 0xF9, 0xFD,
            0xF8, 0x4D, 0xCE, 0xCB, 0x93, 0xF2, 0xB2, 0x33, 0x39, 0x05, 0x50, 0xD2, 0xB1, 0x46, 0x3B, 0x7E,
            0xE3, 0xF5, 0x8D, 0xF7, 0x34, 0x64, 0x35, 0xFF, 0x04, 0x34, 0x19, 0x95, 0x83, 0xC9, 0x7C, 0x66,
            0x5A, 0x97, 0xF1, 0x2F, 0x70, 0x6F, 0x23, 0x57, 0xDA, 0x4B, 0x40, 0x28, 0x8D, 0xEF, 0x88, 0x8E,
            0x59, 0xE6
            /* clang-format on */
        },
    .hash =
        {
            /* Original Message
            0x9E, 0xCD, 0x50, 0x0C, 0x60, 0xE7, 0x01, 0x40, 0x49, 0x22, 0xE5, 0x8A, 0xB2, 0x0C, 0xC0, 0x02,
            0x65, 0x1F, 0xDE, 0xE7, 0xCB, 0xC9, 0x33, 0x6A, 0xDD, 0xA3, 0x3E, 0x4C, 0x10, 0x88, 0xFA, 0xB1,
            0x96, 0x4E, 0xCB, 0x79, 0x04, 0xDC, 0x68, 0x56, 0x86, 0x5D, 0x6C, 0x8E, 0x15, 0x04, 0x1C, 0xCF,
            0x2D, 0x5A, 0xC3, 0x02, 0xE9, 0x9D, 0x34, 0x6F, 0xF2, 0xF6, 0x86, 0x53, 0x1D, 0x25, 0x52, 0x16,
            0x78, 0xD4, 0xFD, 0x3F, 0x76, 0xBB, 0xF2, 0xC8, 0x93, 0xD2, 0x46, 0xCB, 0x4D, 0x76, 0x93, 0x79,
            0x2F, 0xE1, 0x81, 0x72, 0x10, 0x81, 0x46, 0x85, 0x31, 0x03, 0xA5, 0x1F, 0x82, 0x4A, 0xCC, 0x62,
            0x1C, 0xB7, 0x31, 0x1D, 0x24, 0x63, 0xC3, 0x36, 0x1E, 0xA7, 0x07, 0x25, 0x4F, 0x2B, 0x05, 0x2B,
            0xC2, 0x2C, 0xB8, 0x01, 0x28, 0x73, 0xDC, 0xBB, 0x95, 0xBF, 0x1A, 0x5C, 0xC5, 0x3A, 0xB8, 0x9F,
            */
            /* Original Message's Hash */
            0x65, 0xf8, 0x34, 0x08, 0x09, 0x22, 0x61, 0xbd, 0xa5, 0x99, 0x38, 0x9d, 0xf0, 0x33, 0x82, 0xc5,
            0xbe, 0x01, 0xa8, 0x1f, 0xe0, 0x0a, 0x36, 0xf3, 0xf4, 0xbb, 0x65, 0x41, 0x26, 0x3f, 0x80, 0x16,
            0x27, 0xc4, 0x40, 0xe5, 0x08, 0x09, 0x71, 0x2b, 0x0c, 0xac, 0xe7, 0xc2, 0x17, 0xe6, 0xe5, 0x05,
            0x1a, 0xf8, 0x1d, 0xe9, 0xbf, 0xec, 0x32, 0x04, 0xdc, 0xd6, 0x3c, 0x4f, 0x9a, 0x74, 0x10, 0x47,
        },
    .r =
        {
            0
            /*
            0x00, 0x4D, 0xE8, 0x26, 0xEA, 0x70, 0x4A, 0xD1, 0x0B, 0xC0, 0xF7, 0x53, 0x8A, 0xF8, 0xA3, 0x84,
            0x3F, 0x28, 0x4F, 0x55, 0xC8, 0xB9, 0x46, 0xAF, 0x92, 0x35, 0xAF, 0x5A, 0xF7, 0x4F, 0x2B, 0x76,
            0xE0, 0x99, 0xE4, 0xBC, 0x72, 0xFD, 0x79, 0xD2, 0x8A, 0x38, 0x0F, 0x8D, 0x4B, 0x4C, 0x91, 0x9A,
            0xC2, 0x90, 0xD2, 0x48, 0xC3, 0x79, 0x83, 0xBA, 0x05, 0xAE, 0xA4, 0x2E, 0x2D, 0xD7, 0x9F, 0xDD,
            0x33, 0xE8,
            */
        },
    .s =
        {
            0
            /*
            0x00, 0x87, 0x48, 0x8C, 0x85, 0x9A, 0x96, 0xFE, 0xA2, 0x66, 0xEA, 0x13, 0xBF, 0x6D, 0x11, 0x4C,
            0x42, 0x9B, 0x16, 0x3B, 0xE9, 0x7A, 0x57, 0x55, 0x90, 0x86, 0xED, 0xB6, 0x4A, 0xED, 0x4A, 0x18,
            0x59, 0x4B, 0x46, 0xFB, 0x9E, 0xFC, 0x7F, 0xD2, 0x5D, 0x8B, 0x2D, 0xE8, 0xF0, 0x9C, 0xA0, 0x58,
            0x7F, 0x54, 0xBD, 0x28, 0x72, 0x99, 0xF4, 0x7B, 0x2F, 0xF1, 0x24, 0xAA, 0xC5, 0x66, 0xE8, 0xEE,
            0x3B, 0x43,
            */
        },
    .curveLength = BITS_TO_BYTES(ECDSA_CURVE_LENGTH_521),
    .curveType   = ECDSA_TYPE_SEC_P_521_R1,
}};

ECDSA_SignVector signVectorsBRP256[] = {{
    .privateKey =
        {
            0x21, 0x77, 0x4E, 0x4D, 0x01, 0x4F, 0x8B, 0x44, 0x1A, 0x37, 0x38, 0x18, 0x55, 0x30, 0xF3, 0x37,
            0xDA, 0x42, 0xD6, 0x73, 0x93, 0x52, 0x7A, 0xBA, 0xFA, 0x89, 0x40, 0x5E, 0xFB, 0x4F, 0x7F, 0xE3,
        },
    .publicKey =
        {
            /* clang-format off */
            0x04,
            /* X */
            0x68, 0x6B, 0xFA, 0xBE, 0x7C, 0x94, 0x8B, 0x04, 0xB5, 0xD5, 0xFB, 0x73, 0xC5, 0x2D, 0x06, 0x3E,
            0x39, 0xCF, 0x60, 0x23, 0x4E, 0x99, 0x04, 0x8A, 0x22, 0xF5, 0x19, 0x06, 0xF3, 0x74, 0x76, 0x6A,
            /* Y */
            0x0D, 0xA0, 0xD0, 0x43, 0xF1, 0x9E, 0x81, 0x2D, 0x96, 0xCE, 0x12, 0x38, 0x2B, 0x0C, 0x35, 0x9F,
            0xE5, 0x3F, 0xDA, 0x65, 0xBE, 0xBA, 0x8F, 0x90, 0xA6, 0x66, 0xF5, 0x71, 0xFF, 0x6D, 0xF3, 0x60,
            /* clang-format on */
        },
    .hash =
        {
            0xA4, 0x1A, 0x41, 0xA1, 0x2A, 0x79, 0x95, 0x48, 0x21, 0x1C, 0x41, 0x0C, 0x65, 0xD8, 0x13, 0x3A,
            0xFD, 0xE3, 0x4D, 0x28, 0xBD, 0xD5, 0x42, 0xE4, 0xB6, 0x80, 0xCF, 0x28, 0x99, 0xC8, 0xA8, 0xC4,
        },
    .r           = {0},
    .s           = {0},
    .curveLength = BITS_TO_BYTES(ECDSA_CURVE_LENGTH_256),
    .curveType   = ECDSA_TYPE_BRP_P_256_R1,
}};

ECDSA_SignVector signVectorsBRP384[] = {{
    /* http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/ECDSA_Prime.pdf */
    .privateKey =
        {
            0x5F, 0x0F, 0xDE, 0x21, 0xB0, 0x3D, 0xA0, 0x5F, 0xE2, 0xF7, 0x96, 0xDF, 0xEA, 0x91, 0x11, 0xE6,
            0xB7, 0xC7, 0x73, 0x5E, 0xD5, 0x42, 0x86, 0x37, 0x18, 0x82, 0x5B, 0xE3, 0x70, 0xD7, 0x77, 0x37,
            0x28, 0x94, 0x5F, 0x8E, 0x9F, 0x3E, 0xF5, 0x2A, 0x47, 0x6E, 0x6B, 0xF8, 0xA2, 0xA7, 0x3E, 0x49,
        },
    .publicKey =
        {
            /* clang-format off */
            0x04,
            /* X */
            0x6C, 0xF2, 0xAD, 0xE1, 0x19, 0x41, 0xAD, 0x49, 0x92, 0xE7, 0xBD, 0x22, 0x57, 0x39, 0xC6, 0x76,
            0xDA, 0x52, 0x07, 0x58, 0x65, 0x29, 0x17, 0x77, 0x68, 0x95, 0xA6, 0xCE, 0x59, 0xB2, 0x91, 0x32,
            0xD8, 0xF5, 0x31, 0x02, 0x67, 0x0E, 0xDF, 0xAE, 0xC1, 0x63, 0x0D, 0x2D, 0x71, 0x69, 0x7A, 0x9C,
            /* Y */
            0x60, 0x1D, 0x09, 0x47, 0x70, 0x97, 0xB6, 0x0A, 0x6A, 0x5F, 0x0F, 0x5E, 0x4B, 0x05, 0x9B, 0xAD,
            0xF7, 0x00, 0xBB, 0xF6, 0xD1, 0x75, 0x57, 0x85, 0xF7, 0x58, 0x07, 0x24, 0x59, 0x6A, 0x3C, 0xEE,
            0x88, 0xD4, 0x62, 0x79, 0xA1, 0x32, 0x4A, 0xE2, 0x54, 0x35, 0x76, 0x25, 0x04, 0x0C, 0x84, 0x19,
            /* clang-format on */
        },
    .hash =
        {
            /* Original Message
            0x6B, 0x45, 0xD8, 0x80, 0x37, 0x39, 0x2E, 0x13, 0x71, 0xD9, 0xFD, 0x1C, 0xD1, 0x74, 0xE9, 0xC1,
            0x83, 0x8D, 0x11, 0xC3, 0xD6, 0x13, 0x3D, 0xC1, 0x7E, 0x65, 0xFA, 0x0C, 0x48, 0x5D, 0xCC, 0xA9,
            0xF5, 0x2D, 0x41, 0xB6, 0x01, 0x61, 0x24, 0x60, 0x39, 0xE4, 0x2E, 0xC7, 0x84, 0xD4, 0x94, 0x00,
            0xBF, 0xFD, 0xB5, 0x14, 0x59, 0xF5, 0xDE, 0x65, 0x40, 0x91, 0x30, 0x1A, 0x09, 0x37, 0x8F, 0x93,
            0x46, 0x4D, 0x52, 0x11, 0x8B, 0x48, 0xD4, 0x4B, 0x30, 0xD7, 0x81, 0xEB, 0x1D, 0xBE, 0xD0, 0x9D,
            0xA1, 0x1F, 0xB4, 0xC8, 0x18, 0xDB, 0xD4, 0x42, 0xD1, 0x61, 0xAB, 0xA4, 0xB9, 0xED, 0xC7, 0x9F,
            0x05, 0xE4, 0xB7, 0xE4, 0x01, 0x65, 0x13, 0x95, 0xB5, 0x3B, 0xD8, 0xB5, 0xBD, 0x3F, 0x2A, 0xAA,
            0x6A, 0x00, 0x87, 0x7F, 0xA9, 0xB4, 0x5C, 0xAD, 0xB8, 0xE6, 0x48, 0x55, 0x0B, 0x4C, 0x6C, 0xBE,
            */
            /* Original Message's Hash */
            0x31, 0xa4, 0x52, 0xd6, 0x16, 0x4d, 0x90, 0x4b, 0xb5, 0x72, 0x4c, 0x87, 0x82, 0x80, 0x23, 0x1e,
            0xae, 0x70, 0x5c, 0x29, 0xce, 0x9d, 0x4b, 0xc7, 0xd5, 0x8e, 0x02, 0x0e, 0x10, 0x85, 0xf1, 0x7e,
            0xeb, 0xcc, 0x1a, 0x38, 0xf0, 0xed, 0x0b, 0xf2, 0xb3, 0x44, 0xd8, 0x1f, 0xbd, 0x89, 0x68, 0x25,
        },
    .r           = {0},
    .s           = {0},
    .curveLength = BITS_TO_BYTES(ECDSA_CURVE_LENGTH_384),
    .curveType   = ECDSA_TYPE_BRP_P_384_R1,
}};

ECDSA_SignVector signVectorsBRP512[] = {{
    /* http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/ECDSA_Prime.pdf */
    .privateKey =
        {
            0x6C, 0x8E, 0x6A, 0x65, 0x15, 0x1A, 0x09, 0xB7, 0xE0, 0x92, 0xB5, 0xCD, 0x88, 0x0D, 0xCC, 0x96,
            0x57, 0xC8, 0xFC, 0x9A, 0xB6, 0x8F, 0x00, 0x11, 0xEA, 0xC5, 0xB4, 0x68, 0xCE, 0x53, 0x60, 0x17,
            0x79, 0x22, 0xD2, 0xE7, 0xB1, 0x62, 0x37, 0x0F, 0xDF, 0x00, 0xB0, 0x06, 0x0B, 0x78, 0x9E, 0x4B,
            0x66, 0x5E, 0xF8, 0xCE, 0x91, 0x22, 0x5D, 0x76, 0x69, 0xAE, 0xF4, 0x5D, 0xA9, 0x16, 0x78, 0xD5,
        },
    .publicKey =
        {
            /* clang-format off */
            0x04,
            /* X */
            0x65, 0x50, 0x30, 0x66, 0x32, 0x1B, 0x67, 0xE1, 0xB8, 0x5E, 0xF5, 0x1F, 0x6C, 0xFA, 0x95, 0xAD,
            0x23, 0x24, 0x33, 0xEC, 0x67, 0x9F, 0x87, 0x7A, 0xC7, 0xA7, 0xF6, 0x19, 0x0A, 0xF9, 0x59, 0xD9,
            0xEF, 0x37, 0xF2, 0xBF, 0x0E, 0xA1, 0x22, 0xCA, 0x60, 0x8C, 0xEF, 0x69, 0x0E, 0x79, 0xF6, 0xDE,
            0xBD, 0x43, 0x40, 0x8B, 0x07, 0x41, 0x73, 0xBF, 0x8D, 0x7E, 0x90, 0x59, 0xDB, 0x6F, 0x74, 0xCA,
            /* Y */
            0x5C, 0x89, 0xA1, 0x48, 0xCC, 0x66, 0x51, 0xD8, 0x29, 0xB3, 0x35, 0x85, 0xA1, 0xC6, 0xAB, 0x07,
            0x6E, 0xEA, 0x8C, 0x01, 0x16, 0x77, 0x98, 0xEB, 0xC2, 0xB0, 0xC3, 0x2F, 0xBB, 0xB6, 0xF5, 0xA1,
            0xC7, 0x8E, 0xD5, 0x5E, 0xF4, 0x31, 0xE7, 0x08, 0x45, 0xDE, 0x09, 0x82, 0xB2, 0xD0, 0xE4, 0xAA,
            0xBF, 0xA4, 0xF1, 0xC9, 0x4B, 0x69, 0x01, 0x4A, 0x84, 0x60, 0xBF, 0x11, 0x25, 0xD3, 0x11, 0xE8,
            /* clang-format on */
        },
    .hash =
        {
            /* Original Message
            0x9E, 0xCD, 0x50, 0x0C, 0x60, 0xE7, 0x01, 0x40, 0x49, 0x22, 0xE5, 0x8A, 0xB2, 0x0C, 0xC0, 0x02,
            0x65, 0x1F, 0xDE, 0xE7, 0xCB, 0xC9, 0x33, 0x6A, 0xDD, 0xA3, 0x3E, 0x4C, 0x10, 0x88, 0xFA, 0xB1,
            0x96, 0x4E, 0xCB, 0x79, 0x04, 0xDC, 0x68, 0x56, 0x86, 0x5D, 0x6C, 0x8E, 0x15, 0x04, 0x1C, 0xCF,
            0x2D, 0x5A, 0xC3, 0x02, 0xE9, 0x9D, 0x34, 0x6F, 0xF2, 0xF6, 0x86, 0x53, 0x1D, 0x25, 0x52, 0x16,
            0x78, 0xD4, 0xFD, 0x3F, 0x76, 0xBB, 0xF2, 0xC8, 0x93, 0xD2, 0x46, 0xCB, 0x4D, 0x76, 0x93, 0x79,
            0x2F, 0xE1, 0x81, 0x72, 0x10, 0x81, 0x46, 0x85, 0x31, 0x03, 0xA5, 0x1F, 0x82, 0x4A, 0xCC, 0x62,
            0x1C, 0xB7, 0x31, 0x1D, 0x24, 0x63, 0xC3, 0x36, 0x1E, 0xA7, 0x07, 0x25, 0x4F, 0x2B, 0x05, 0x2B,
            0xC2, 0x2C, 0xB8, 0x01, 0x28, 0x73, 0xDC, 0xBB, 0x95, 0xBF, 0x1A, 0x5C, 0xC5, 0x3A, 0xB8, 0x9F,
            */
            /* Original Message's Hash */
            0x65, 0xf8, 0x34, 0x08, 0x09, 0x22, 0x61, 0xbd, 0xa5, 0x99, 0x38, 0x9d, 0xf0, 0x33, 0x82, 0xc5,
            0xbe, 0x01, 0xa8, 0x1f, 0xe0, 0x0a, 0x36, 0xf3, 0xf4, 0xbb, 0x65, 0x41, 0x26, 0x3f, 0x80, 0x16,
            0x27, 0xc4, 0x40, 0xe5, 0x08, 0x09, 0x71, 0x2b, 0x0c, 0xac, 0xe7, 0xc2, 0x17, 0xe6, 0xe5, 0x05,
            0x1a, 0xf8, 0x1d, 0xe9, 0xbf, 0xec, 0x32, 0x04, 0xdc, 0xd6, 0x3c, 0x4f, 0x9a, 0x74, 0x10, 0x47,
        },
    .r           = {0},
    .s           = {0},
    .curveLength = BITS_TO_BYTES(ECDSA_CURVE_LENGTH_512),
    .curveType   = ECDSA_TYPE_BRP_P_512_R1,
}};

ECDSA_SignVector *signVectors[] = {signVectorsP256,
                                   signVectorsP224,
                                   signVectorsP384,
                                   signVectorsP521,
                                   signVectorsBRP256,
                                   signVectorsBRP384,
                                   signVectorsBRP512};
uint8_t signVectorCountTable[]  = {sizeof(signVectorsP256) / sizeof(ECDSA_SignVector),
                                   sizeof(signVectorsP224) / sizeof(ECDSA_SignVector),
                                   sizeof(signVectorsP384) / sizeof(ECDSA_SignVector),
                                   sizeof(signVectorsP521) / sizeof(ECDSA_SignVector),
                                   sizeof(signVectorsBRP256) / sizeof(ECDSA_SignVector),
                                   sizeof(signVectorsBRP384) / sizeof(ECDSA_SignVector),
                                   sizeof(signVectorsBRP512) / sizeof(ECDSA_SignVector)};