/*
 * Copyright (c) 2025, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== hsmFwUpdate.c ========
 */

/* For usleep() */
#include <unistd.h>
#include <stdint.h>
#include <stddef.h>

/* Driver Header files */
#include <ti/drivers/GPIO.h>
#include <ti/devices/DeviceFamily.h>
#include DeviceFamily_constructPath(driverlib/hapi.h)

/* Driver configuration */
#include "ti_drivers_config.h"

/* HSM FW Update Image */
#if defined(DeviceFamily_CC27XX)
    #include <cc27xxx10_hsm_fw_v3.1.1.h>
#else
    #error "Example not supported for this device yet!"
#endif

#define ID_NONE           0
#define ID_HSM_FW_UPDATE  1
#define ID_BOOT_SSB       2
#define ID_UPDATE_TRIGGER 3

#define STA_NA               0
#define STA_UPDATE_SUCCESS   1
#define STA_FIH_ERROR        2
#define STA_IMG_REJECTED     3
#define STA_IMG_VERIF_FAILED 4
#define STA_UPDATE_FAILED    5
#define STA_INVALID_CONFIG   6
#define STA_HSM_BOOT_FAILURE 7

#define SECTOR_OUT_OF_BOUNDS    0xc401c425
#define SECTOR_ADDR_NOT_ALIGNED 0x49b63027
#define SECTOR_SET_SUCCESS      0x17c2e67c

#define ID_INVALID     0xbea664db
#define ID_SET_SUCCESS 0x2e3a28fc

uint32_t performUpdateSequence(uint32_t hsmFwAddr)
{
    uint32_t result = 0;

    /* 1. Inform the device ROM where the HSM FW image resides */
    result = HapiSbSetUpdateImageAddress(hsmFwAddr);

    /* 2. Inform the device ROM what to expect at that addr */
    if (SECTOR_SET_SUCCESS == result)
    {
        if (ID_SET_SUCCESS != HapiSbSetId(ID_HSM_FW_UPDATE))
        {
            result = 0xFF;
        }
        else
        {
            /* 3. Reset the device so ROM can complete the HSM FW update */
            HapiResetDevice(); // Expected not to return
        }
    }

    return result;
}

/*
 *  ======== mainThread ========
 */
void *mainThread(void *arg0)
{
    /* 0.5 second delay */
    uint32_t time = 500000;

    /* Call driver init functions */
    GPIO_init();

    /* Configure the LED pin */
    GPIO_setConfig(CONFIG_GPIO_LED_GREEN, GPIO_CFG_OUT_STD | GPIO_CFG_OUT_LOW);
    GPIO_setConfig(CONFIG_GPIO_LED_RED, GPIO_CFG_OUT_STD | GPIO_CFG_OUT_LOW);

    /* Get the current status */
    uint32_t updateStatus = HapiSbGetStatus();
    if (updateStatus == STA_IMG_VERIF_FAILED)
    {
        /*
         * If a previous attempt has been made to update the HSM FW but the
         * result was a failure there's no need to keep trying
         */
        while (1)
        {
            usleep(time);
            GPIO_toggle(CONFIG_GPIO_LED_RED);
        }
    }
    else if (updateStatus != STA_UPDATE_SUCCESS)
    {
        /*
         * If it appears no update attempt has been made since the last PoR then
         * perform the update sequence
         */
        uint32_t hsmFwAddr = (uint32_t)hsmFwBinary;
        updateStatus       = performUpdateSequence(hsmFwAddr);

        /* Only an error reaches this point */
        while (1)
        {
            usleep(time);
            GPIO_toggle(CONFIG_GPIO_LED_RED);
        }
    }
    else
    {
        /* Only a success reaches this point */
        while (1)
        {
            usleep(time);
            GPIO_toggle(CONFIG_GPIO_LED_GREEN);
        }
    }
}
