/*
 * Copyright (c) 2024, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "HSMBareMetalECCUtility.h"

#if defined(__GNUC__) || defined(__clang__)
    #define CRYPTOUTILS_NOINLINE __attribute__((noinline))
#else
    #define CRYPTOUTILS_NOINLINE
#endif

/* NIST curves */

/* NIST_P_224 = SEC_P_224_R1 */
static const uint8_t HSMBareMetal_SEC224r1[] =
    {0xE0, 0x00, 0x00, 0x07, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE0, 0x00, 0x01, 0x07, 0xFE, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE0, 0x00, 0x02, 0x07, 0xB4, 0xFF, 0x55, 0x23, 0x43, 0x39, 0x0B, 0x27,
     0xBA, 0xD8, 0xBF, 0xD7, 0xB7, 0xB0, 0x44, 0x50, 0x56, 0x32, 0x41, 0xF5, 0xAB, 0xB3, 0x04, 0x0C, 0x85, 0x0A, 0x05,
     0xB4, 0xE0, 0x00, 0x03, 0x07, 0x3D, 0x2A, 0x5C, 0x5C, 0x45, 0x29, 0xDD, 0x13, 0x3E, 0xF0, 0xB8, 0xE0, 0xA2, 0x16,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE0, 0x00, 0x04, 0x07, 0x21,
     0x1D, 0x5C, 0x11, 0xD6, 0x80, 0x32, 0x34, 0x22, 0x11, 0xC2, 0x56, 0xD3, 0xC1, 0x03, 0x4A, 0xB9, 0x90, 0x13, 0x32,
     0x7F, 0xBF, 0xB4, 0x6B, 0xBD, 0x0C, 0x0E, 0xB7, 0xE0, 0x00, 0x05, 0x07, 0x34, 0x7E, 0x00, 0x85, 0x99, 0x81, 0xD5,
     0x44, 0x64, 0x47, 0x07, 0x5A, 0xA0, 0x75, 0x43, 0xCD, 0xE6, 0xDF, 0x22, 0x4C, 0xFB, 0x23, 0xF7, 0xB5, 0x88, 0x63,
     0x37, 0xBD, 0x01, 0x00, 0x06, 0x07, 0x01, 0x00, 0x00, 0x00};

/* NIST_P_256 = SEC_P_256_R1 */
static const uint8_t HSMBareMetal_SEC256r1[] =
    {0x00, 0x01, 0x00, 0x07, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x01,
     0x01, 0x07, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x01, 0x02, 0x07,
     0x4B, 0x60, 0xD2, 0x27, 0x3E, 0x3C, 0xCE, 0x3B, 0xF6, 0xB0, 0x53, 0xCC, 0xB0, 0x06, 0x1D, 0x65, 0xBC, 0x86, 0x98,
     0x76, 0x55, 0xBD, 0xEB, 0xB3, 0xE7, 0x93, 0x3A, 0xAA, 0xD8, 0x35, 0xC6, 0x5A, 0x00, 0x01, 0x03, 0x07, 0x51, 0x25,
     0x63, 0xFC, 0xC2, 0xCA, 0xB9, 0xF3, 0x84, 0x9E, 0x17, 0xA7, 0xAD, 0xFA, 0xE6, 0xBC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x01, 0x04, 0x07, 0x96, 0xC2, 0x98, 0xD8,
     0x45, 0x39, 0xA1, 0xF4, 0xA0, 0x33, 0xEB, 0x2D, 0x81, 0x7D, 0x03, 0x77, 0xF2, 0x40, 0xA4, 0x63, 0xE5, 0xE6, 0xBC,
     0xF8, 0x47, 0x42, 0x2C, 0xE1, 0xF2, 0xD1, 0x17, 0x6B, 0x00, 0x01, 0x05, 0x07, 0xF5, 0x51, 0xBF, 0x37, 0x68, 0x40,
     0xB6, 0xCB, 0xCE, 0x5E, 0x31, 0x6B, 0x57, 0x33, 0xCE, 0x2B, 0x16, 0x9E, 0x0F, 0x7C, 0x4A, 0xEB, 0xE7, 0x8E, 0x9B,
     0x7F, 0x1A, 0xFE, 0xE2, 0x42, 0xE3, 0x4F, 0x01, 0x00, 0x06, 0x07, 0x01, 0x00, 0x00, 0x00};

/* NIST_P_384 = SEC_P_384_R1 */
static const uint8_t HSMBareMetal_SEC384r1[] =
    {0x80, 0x01, 0x00, 0x07, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x80, 0x01, 0x01, 0x07, 0xFC,
     0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x80, 0x01, 0x02, 0x07, 0xEF, 0x2A, 0xEC, 0xD3, 0xED, 0xC8,
     0x85, 0x2A, 0x9D, 0xD1, 0x2E, 0x8A, 0x8D, 0x39, 0x56, 0xC6, 0x5A, 0x87, 0x13, 0x50, 0x8F, 0x08, 0x14, 0x03, 0x12,
     0x41, 0x81, 0xFE, 0x6E, 0x9C, 0x1D, 0x18, 0x19, 0x2D, 0xF8, 0xE3, 0x6B, 0x05, 0x8E, 0x98, 0xE4, 0xE7, 0x3E, 0xE2,
     0xA7, 0x2F, 0x31, 0xB3, 0x80, 0x01, 0x03, 0x07, 0x73, 0x29, 0xC5, 0xCC, 0x6A, 0x19, 0xEC, 0xEC, 0x7A, 0xA7, 0xB0,
     0x48, 0xB2, 0x0D, 0x1A, 0x58, 0xDF, 0x2D, 0x37, 0xF4, 0x81, 0x4D, 0x63, 0xC7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x80,
     0x01, 0x04, 0x07, 0xB7, 0x0A, 0x76, 0x72, 0x38, 0x5E, 0x54, 0x3A, 0x6C, 0x29, 0x55, 0xBF, 0x5D, 0xF2, 0x02, 0x55,
     0x38, 0x2A, 0x54, 0x82, 0xE0, 0x41, 0xF7, 0x59, 0x98, 0x9B, 0xA7, 0x8B, 0x62, 0x3B, 0x1D, 0x6E, 0x74, 0xAD, 0x20,
     0xF3, 0x1E, 0xC7, 0xB1, 0x8E, 0x37, 0x05, 0x8B, 0xBE, 0x22, 0xCA, 0x87, 0xAA, 0x80, 0x01, 0x05, 0x07, 0x5F, 0x0E,
     0xEA, 0x90, 0x7C, 0x1D, 0x43, 0x7A, 0x9D, 0x81, 0x7E, 0x1D, 0xCE, 0xB1, 0x60, 0x0A, 0xC0, 0xB8, 0xF0, 0xB5, 0x13,
     0x31, 0xDA, 0xE9, 0x7C, 0x14, 0x9A, 0x28, 0xBD, 0x1D, 0xF4, 0xF8, 0x29, 0xDC, 0x92, 0x92, 0xBF, 0x98, 0x9E, 0x5D,
     0x6F, 0x2C, 0x26, 0x96, 0x4A, 0xDE, 0x17, 0x36, 0x01, 0x00, 0x06, 0x07, 0x01, 0x00, 0x00, 0x00};

/* NIST_P_521 = SEC_P_521_R1 */
static const uint8_t HSMBareMetal_SEC521r1[] =
    {0x09, 0x02, 0x00, 0x07, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x01, 0x00, 0x00, 0x09, 0x02, 0x01, 0x07,
     0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x01, 0x00, 0x00, 0x09, 0x02, 0x02, 0x07, 0x00, 0x3F, 0x50, 0x6B,
     0xD4, 0x1F, 0x45, 0xEF, 0xF1, 0x34, 0x2C, 0x3D, 0x88, 0xDF, 0x73, 0x35, 0x07, 0xBF, 0xB1, 0x3B, 0xBD, 0xC0, 0x52,
     0x16, 0x7B, 0x93, 0x7E, 0xEC, 0x51, 0x39, 0x19, 0x56, 0xE1, 0x09, 0xF1, 0x8E, 0x91, 0x89, 0xB4, 0xB8, 0xF3, 0x15,
     0xB3, 0x99, 0x5B, 0x72, 0xDA, 0xA2, 0xEE, 0x40, 0x85, 0xB6, 0xA0, 0x21, 0x9A, 0x92, 0x1F, 0x9A, 0x1C, 0x8E, 0x61,
     0xB9, 0x3E, 0x95, 0x51, 0x00, 0x00, 0x00, 0x09, 0x02, 0x03, 0x07, 0x09, 0x64, 0x38, 0x91, 0x1E, 0xB7, 0x6F, 0xBB,
     0xAE, 0x47, 0x9C, 0x89, 0xB8, 0xC9, 0xB5, 0x3B, 0xD0, 0xA5, 0x09, 0xF7, 0x48, 0x01, 0xCC, 0x7F, 0x6B, 0x96, 0x2F,
     0xBF, 0x83, 0x87, 0x86, 0x51, 0xFA, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0x01, 0x00, 0x00, 0x09, 0x02, 0x04, 0x07, 0x66, 0xBD, 0xE5, 0xC2, 0x31, 0x7E, 0x7E, 0xF9, 0x9B, 0x42, 0x6A, 0x85,
     0xC1, 0xB3, 0x48, 0x33, 0xDE, 0xA8, 0xFF, 0xA2, 0x27, 0xC1, 0x1D, 0xFE, 0x28, 0x59, 0xE7, 0xEF, 0x77, 0x5E, 0x4B,
     0xA1, 0xBA, 0x3D, 0x4D, 0x6B, 0x60, 0xAF, 0x28, 0xF8, 0x21, 0xB5, 0x3F, 0x05, 0x39, 0x81, 0x64, 0x9C, 0x42, 0xB4,
     0x95, 0x23, 0x66, 0xCB, 0x3E, 0x9E, 0xCD, 0xE9, 0x04, 0x04, 0xB7, 0x06, 0x8E, 0x85, 0xC6, 0x00, 0x00, 0x00, 0x09,
     0x02, 0x05, 0x07, 0x50, 0x66, 0xD1, 0x9F, 0x76, 0x94, 0xBE, 0x88, 0x40, 0xC2, 0x72, 0xA2, 0x86, 0x70, 0x3C, 0x35,
     0x61, 0x07, 0xAD, 0x3F, 0x01, 0xB9, 0x50, 0xC5, 0x40, 0x26, 0xF4, 0x5E, 0x99, 0x72, 0xEE, 0x97, 0x2C, 0x66, 0x3E,
     0x27, 0x17, 0xBD, 0xAF, 0x17, 0x68, 0x44, 0x9B, 0x57, 0x49, 0x44, 0xF5, 0x98, 0xD9, 0x1B, 0x7D, 0x2C, 0xB4, 0x5F,
     0x8A, 0x5C, 0x04, 0xC0, 0x3B, 0x9A, 0x78, 0x6A, 0x29, 0x39, 0x18, 0x01, 0x00, 0x00, 0x01, 0x00, 0x06, 0x07, 0x01,
     0x00, 0x00, 0x00};

/* Brainpool curves */

/* BRP_P_256_R1 */
static const uint8_t HSMBareMetal_BRP256r1[] =
    {0x00, 0x01, 0x00, 0x17, 0x77, 0x53, 0x6E, 0x1F, 0x1D, 0x48, 0x13, 0x20, 0x28, 0x20, 0x26, 0xD5, 0x23, 0xF6, 0x3B,
     0x6E, 0x72, 0x8D, 0x83, 0x9D, 0x90, 0x0A, 0x66, 0x3E, 0xBC, 0xA9, 0xEE, 0xA1, 0xDB, 0x57, 0xFB, 0xA9, 0x00, 0x01,
     0x01, 0x07, 0xD9, 0xB5, 0x30, 0xF3, 0x44, 0x4B, 0x4A, 0xE9, 0x6C, 0x5C, 0xDC, 0x26, 0xC1, 0x55, 0x80, 0xFB, 0xE7,
     0xFF, 0x7A, 0x41, 0x30, 0x75, 0xF6, 0xEE, 0x57, 0x30, 0x2C, 0xFC, 0x75, 0x09, 0x5A, 0x7D, 0x00, 0x01, 0x02, 0x07,
     0xB6, 0x07, 0x8C, 0xFF, 0x18, 0xDC, 0xCC, 0x6B, 0xCE, 0xE1, 0xF7, 0x5C, 0x29, 0x16, 0x84, 0x95, 0xBF, 0x7C, 0xD7,
     0xBB, 0xD9, 0xB5, 0x30, 0xF3, 0x44, 0x4B, 0x4A, 0xE9, 0x6C, 0x5C, 0xDC, 0x26, 0x00, 0x01, 0x03, 0x07, 0xA7, 0x56,
     0x48, 0x97, 0x82, 0x0E, 0x1E, 0x90, 0xF7, 0xA6, 0x61, 0xB5, 0xA3, 0x7A, 0x39, 0x8C, 0x71, 0x8D, 0x83, 0x9D, 0x90,
     0x0A, 0x66, 0x3E, 0xBC, 0xA9, 0xEE, 0xA1, 0xDB, 0x57, 0xFB, 0xA9, 0x00, 0x01, 0x04, 0x07, 0x62, 0x32, 0xCE, 0x9A,
     0xBD, 0x53, 0x44, 0x3A, 0xC2, 0x23, 0xBD, 0xE3, 0xE1, 0x27, 0xDE, 0xB9, 0xAF, 0xB7, 0x81, 0xFC, 0x2F, 0x48, 0x4B,
     0x2C, 0xCB, 0x57, 0x7E, 0xCB, 0xB9, 0xAE, 0xD2, 0x8B, 0x00, 0x01, 0x05, 0x07, 0x97, 0x69, 0x04, 0x2F, 0xC7, 0x54,
     0x1D, 0x5C, 0x54, 0x8E, 0xED, 0x2D, 0x13, 0x45, 0x77, 0xC2, 0xC9, 0x1D, 0x61, 0x14, 0x1A, 0x46, 0xF8, 0x97, 0xFD,
     0xC4, 0xDA, 0xC3, 0x35, 0xF8, 0x7E, 0x54, 0x01, 0x00, 0x06, 0x07, 0x01, 0x00, 0x00, 0x00};

/* BRP_P_384_R1 */
static const uint8_t HSMBareMetal_BRP384r1[] =
    {0x80, 0x01, 0x00, 0x17, 0x53, 0xEC, 0x07, 0x31, 0x13, 0x00, 0x47, 0x87, 0x71, 0x1A, 0x1D, 0x90, 0x29, 0xA7, 0xD3,
     0xAC, 0x23, 0x11, 0xB7, 0x7F, 0x19, 0xDA, 0xB1, 0x12, 0xB4, 0x56, 0x54, 0xED, 0x09, 0x71, 0x2F, 0x15, 0xDF, 0x41,
     0xE6, 0x50, 0x7E, 0x6F, 0x5D, 0x0F, 0x28, 0x6D, 0x38, 0xA3, 0x82, 0x1E, 0xB9, 0x8C, 0x80, 0x01, 0x01, 0x07, 0x26,
     0x28, 0xCE, 0x22, 0xDD, 0xC7, 0xA8, 0x04, 0xEB, 0xD4, 0x3A, 0x50, 0x4A, 0x81, 0xA5, 0x8A, 0x0F, 0xF9, 0x91, 0xBA,
     0xEF, 0x65, 0x91, 0x13, 0x87, 0x27, 0xB2, 0x4F, 0x8E, 0xA2, 0xBE, 0xC2, 0xA0, 0xAF, 0x05, 0xCE, 0x0A, 0x08, 0x72,
     0x3C, 0x0C, 0x15, 0x8C, 0x3D, 0xC6, 0x82, 0xC3, 0x7B, 0x80, 0x01, 0x02, 0x07, 0x11, 0x4C, 0x50, 0xFA, 0x96, 0x86,
     0xB7, 0x3A, 0x94, 0xC9, 0xDB, 0x95, 0x02, 0x39, 0xB4, 0x7C, 0xD5, 0x62, 0xEB, 0x3E, 0xA5, 0x0E, 0x88, 0x2E, 0xA6,
     0xD2, 0xDC, 0x07, 0xE1, 0x7D, 0xB7, 0x2F, 0x7C, 0x44, 0xF0, 0x16, 0x54, 0xB5, 0x39, 0x8B, 0x26, 0x28, 0xCE, 0x22,
     0xDD, 0xC7, 0xA8, 0x04, 0x80, 0x01, 0x03, 0x07, 0x65, 0x65, 0x04, 0xE9, 0x02, 0x32, 0x88, 0x3B, 0x10, 0xC3, 0x7F,
     0x6B, 0xAF, 0xB6, 0x3A, 0xCF, 0xA7, 0x25, 0x04, 0xAC, 0x6C, 0x6E, 0x16, 0x1F, 0xB3, 0x56, 0x54, 0xED, 0x09, 0x71,
     0x2F, 0x15, 0xDF, 0x41, 0xE6, 0x50, 0x7E, 0x6F, 0x5D, 0x0F, 0x28, 0x6D, 0x38, 0xA3, 0x82, 0x1E, 0xB9, 0x8C, 0x80,
     0x01, 0x04, 0x07, 0x1E, 0xAF, 0xD4, 0x47, 0xE2, 0xB2, 0x87, 0xEF, 0xAA, 0x46, 0xD6, 0x36, 0x34, 0xE0, 0x26, 0xE8,
     0xE8, 0x10, 0xBD, 0x0C, 0xFE, 0xCA, 0x7F, 0xDB, 0xE3, 0x4F, 0xF1, 0x7E, 0xE7, 0xA3, 0x47, 0x88, 0x6B, 0x3F, 0xC1,
     0xB7, 0x81, 0x3A, 0xA6, 0xA2, 0xFF, 0x45, 0xCF, 0x68, 0xF0, 0x64, 0x1C, 0x1D, 0x80, 0x01, 0x05, 0x07, 0x15, 0x53,
     0x3C, 0x26, 0x41, 0x03, 0x82, 0x42, 0x11, 0x81, 0x91, 0x77, 0x21, 0x46, 0x46, 0x0E, 0x28, 0x29, 0x91, 0xF9, 0x4F,
     0x05, 0x9C, 0xE1, 0x64, 0x58, 0xEC, 0xFE, 0x29, 0x0B, 0xB7, 0x62, 0x52, 0xD5, 0xCF, 0x95, 0x8E, 0xEB, 0xB1, 0x5C,
     0xA4, 0xC2, 0xF9, 0x20, 0x75, 0x1D, 0xBE, 0x8A, 0x01, 0x00, 0x06, 0x07, 0x01, 0x00, 0x00, 0x00};

/* BRP_P_512_R1 */
static const uint8_t HSMBareMetal_BRP512r1[] =
    {0x00, 0x02, 0x00, 0x17, 0xF3, 0x48, 0x3A, 0x58, 0x56, 0x60, 0xAA, 0x28, 0x85, 0xC6, 0x82, 0x2D, 0x2F, 0xFF, 0x81,
     0x28, 0xE6, 0x80, 0xA3, 0xE6, 0x2A, 0xA1, 0xCD, 0xAE, 0x42, 0x68, 0xC6, 0x9B, 0x00, 0x9B, 0x4D, 0x7D, 0x71, 0x08,
     0x33, 0x70, 0xCA, 0x9C, 0x63, 0xD6, 0x0E, 0xD2, 0xC9, 0xB3, 0xB3, 0x8D, 0x30, 0xCB, 0x07, 0xFC, 0xC9, 0x33, 0xAE,
     0xE6, 0xD4, 0x3F, 0x8B, 0xC4, 0xE9, 0xDB, 0xB8, 0x9D, 0xDD, 0xAA, 0x00, 0x02, 0x01, 0x07, 0xCA, 0x94, 0xFC, 0x77,
     0x4D, 0xAC, 0xC1, 0xE7, 0xB9, 0xC7, 0xF2, 0x2B, 0xA7, 0x17, 0x11, 0x7F, 0xB5, 0xC8, 0x9A, 0x8B, 0xC9, 0xF1, 0x2E,
     0x0A, 0xA1, 0x3A, 0x25, 0xA8, 0x5A, 0x5D, 0xED, 0x2D, 0xBC, 0x63, 0x98, 0xEA, 0xCA, 0x41, 0x34, 0xA8, 0x10, 0x16,
     0xF9, 0x3D, 0x8D, 0xDD, 0xCB, 0x94, 0xC5, 0x4C, 0x23, 0xAC, 0x45, 0x71, 0x32, 0xE2, 0x89, 0x3B, 0x60, 0x8B, 0x31,
     0xA3, 0x30, 0x78, 0x00, 0x02, 0x02, 0x07, 0x23, 0xF7, 0x16, 0x80, 0x63, 0xBD, 0x09, 0x28, 0xDD, 0xE5, 0xBA, 0x5E,
     0xB7, 0x50, 0x40, 0x98, 0x67, 0x3E, 0x08, 0xDC, 0xCA, 0x94, 0xFC, 0x77, 0x4D, 0xAC, 0xC1, 0xE7, 0xB9, 0xC7, 0xF2,
     0x2B, 0xA7, 0x17, 0x11, 0x7F, 0xB5, 0xC8, 0x9A, 0x8B, 0xC9, 0xF1, 0x2E, 0x0A, 0xA1, 0x3A, 0x25, 0xA8, 0x5A, 0x5D,
     0xED, 0x2D, 0xBC, 0x63, 0x98, 0xEA, 0xCA, 0x41, 0x34, 0xA8, 0x10, 0x16, 0xF9, 0x3D, 0x00, 0x02, 0x03, 0x07, 0x69,
     0x00, 0xA9, 0x9C, 0x82, 0x96, 0x87, 0xB5, 0xDD, 0xDA, 0x5D, 0x08, 0x81, 0xD3, 0xB1, 0x1D, 0x47, 0x10, 0xAC, 0x7F,
     0x19, 0x61, 0x86, 0x41, 0x19, 0x26, 0xA9, 0x4C, 0x41, 0x5C, 0x3E, 0x55, 0x70, 0x08, 0x33, 0x70, 0xCA, 0x9C, 0x63,
     0xD6, 0x0E, 0xD2, 0xC9, 0xB3, 0xB3, 0x8D, 0x30, 0xCB, 0x07, 0xFC, 0xC9, 0x33, 0xAE, 0xE6, 0xD4, 0x3F, 0x8B, 0xC4,
     0xE9, 0xDB, 0xB8, 0x9D, 0xDD, 0xAA, 0x00, 0x02, 0x04, 0x07, 0x22, 0xF8, 0xB9, 0xBC, 0x09, 0x22, 0x35, 0x8B, 0x68,
     0x5E, 0x6A, 0x40, 0x47, 0x50, 0x6D, 0x7C, 0x5F, 0x7D, 0xB9, 0x93, 0x7B, 0x68, 0xD1, 0x50, 0x8D, 0xD4, 0xD0, 0xE2,
     0x78, 0x1F, 0x3B, 0xFF, 0x8E, 0x09, 0xD0, 0xF4, 0xEE, 0x62, 0x3B, 0xB4, 0xC1, 0x16, 0xD9, 0xB5, 0x70, 0x9F, 0xED,
     0x85, 0x93, 0x6A, 0x4C, 0x9C, 0x2E, 0x32, 0x21, 0x5A, 0x64, 0xD9, 0x2E, 0xD8, 0xBD, 0xE4, 0xAE, 0x81, 0x00, 0x02,
     0x05, 0x07, 0x92, 0x08, 0xD8, 0x3A, 0x0F, 0x1E, 0xCD, 0x78, 0x06, 0x54, 0xF0, 0xA8, 0x2F, 0x2B, 0xCA, 0xD1, 0xAE,
     0x63, 0x27, 0x8A, 0xD8, 0x4B, 0xCA, 0x5B, 0x5E, 0x48, 0x5F, 0x4A, 0x49, 0xDE, 0xDC, 0xB2, 0x11, 0x81, 0x1F, 0x88,
     0x5B, 0xC5, 0x00, 0xA0, 0x1A, 0x7B, 0xA5, 0x24, 0x00, 0xF7, 0x09, 0xF2, 0xFD, 0x22, 0x78, 0xCF, 0xA9, 0xBF, 0xEA,
     0xC0, 0xEC, 0x32, 0x63, 0x56, 0x5D, 0x38, 0xDE, 0x7D, 0x01, 0x00, 0x06, 0x07, 0x01, 0x00, 0x00, 0x00};

/* Bernstein curves */
/* curve25519 */
static const uint8_t HSMBareMetal_curve25519[] =
    {0xFF, 0x00, 0x00, 0x07, 0xED, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
     0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F, 0xFF, 0x00,
     0x01, 0x07, 0x06, 0x6D, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x02, 0x07,
     0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x03, 0x07, 0xED, 0xD3,
     0xF5, 0x5C, 0x1A, 0x63, 0x12, 0x58, 0xD6, 0x9C, 0xF7, 0xA2, 0xDE, 0xF9, 0xDE, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0xFF, 0x00, 0x04, 0x07, 0x09, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x05, 0x07, 0x14, 0x2C, 0x31, 0x81, 0x5D, 0x3A,
     0x16, 0xD6, 0x4D, 0x9E, 0x83, 0x92, 0x81, 0xB2, 0xC2, 0x6D, 0xB3, 0x2E, 0xB7, 0x88, 0xD3, 0x22, 0xE1, 0x1F, 0x4B,
     0x79, 0x5F, 0x47, 0x5E, 0xE6, 0x51, 0x5F, 0x04, 0x00, 0x06, 0x07, 0x08, 0x00, 0x00, 0x00};

/*
 *  ======== HSMBareMetalECCParams_getCurveParameters ========
 */
int_fast16_t HSMBareMetalECCParams_getCurveParameters(uint8_t curveType,
                                                      const uint8_t **curveParamsPtr,
                                                      uint32_t *curveParamsSize)
{
    int_fast16_t status = 0U;

    switch (curveType)
    {
        case 1U: /* SEC_P_224_R1 */
            *curveParamsPtr  = HSMBareMetal_SEC224r1;
            *curveParamsSize = sizeof(HSMBareMetal_SEC224r1);
            break;

        case 2U: /* SEC_P_256_R1 */
            *curveParamsPtr  = HSMBareMetal_SEC256r1;
            *curveParamsSize = sizeof(HSMBareMetal_SEC256r1);
            break;

        case 3U: /* SEC_P_384_R1 */
            *curveParamsPtr  = HSMBareMetal_SEC384r1;
            *curveParamsSize = sizeof(HSMBareMetal_SEC384r1);
            break;

        case 4U: /* SEC_P_521_R1 */
            *curveParamsPtr  = HSMBareMetal_SEC521r1;
            *curveParamsSize = sizeof(HSMBareMetal_SEC521r1);
            break;

        case 5U: /* BRP_P_256_R1 */
            *curveParamsPtr  = HSMBareMetal_BRP256r1;
            *curveParamsSize = sizeof(HSMBareMetal_BRP256r1);
            break;

        case 6U: /* BRP_P_384_R1 */
            *curveParamsPtr  = HSMBareMetal_BRP384r1;
            *curveParamsSize = sizeof(HSMBareMetal_BRP384r1);
            break;

        case 7U: /* BRP_P_512_R1 */
            *curveParamsPtr  = HSMBareMetal_BRP512r1;
            *curveParamsSize = sizeof(HSMBareMetal_BRP512r1);
            break;

        case 8U: /* CURVE25519 */
            *curveParamsPtr  = HSMBareMetal_curve25519;
            *curveParamsSize = sizeof(HSMBareMetal_curve25519);
            break;

        default:
            *curveParamsPtr  = NULL;
            *curveParamsSize = 0U;
            status           = -1U;
            break;
    }

    return status;
}

/*
 *  ======== HSMBareMetalECCParams_reverseMemCpy ========
 */
void *HSMBareMetalECCParams_reverseMemCpy(void *dest, const void *src, size_t size)
{
    uint8_t *dp       = (uint8_t *)dest;
    const uint8_t *sp = (const uint8_t *)src;
    size_t tempSize   = size;

    sp = &sp[(tempSize - 1U)];
    while (tempSize > 0U)
    {
        *dp = *sp;
        dp++;
        sp--;
        tempSize--;
    }

    return dest;
}

/*
 *  ======== HSMBareMetalECCParams_getUncompressedGeneratorPoint ========
 */
int_fast16_t HSMBareMetalECCParams_getUncompressedGeneratorPoint(uint8_t *generatorX,
                                                                 uint8_t *generatorY,
                                                                 uint8_t *buffer,
                                                                 size_t curveLength,
                                                                 size_t length)
{
    size_t pointLength = (BITS_TO_BYTES(curveLength) * 2) + 1;

    if (length < pointLength)
    {
        return -1;
    }

    /* Reverse and concatenate x and y */
    HSMBareMetalECCParams_reverseMemCpy(buffer + 1, generatorX, BITS_TO_BYTES(curveLength));

    HSMBareMetalECCParams_reverseMemCpy(buffer + 1 + BITS_TO_BYTES(curveLength),
                                        generatorY,
                                        BITS_TO_BYTES(curveLength));

    buffer[0] = 0x04;

    memset(buffer + pointLength, 0, length - pointLength);

    return 0;
}

/*
 *  ======== HSMBareMetalECCParams_asymVectorHeaderFormat ========
 */
void HSMBareMetalECCParams_asymVectorHeaderFormat(const size_t modulusSizeBits,
                                                  const uint8_t itemsLength,
                                                  const uint8_t itemIdx,
                                                  const uint8_t domainId,
                                                  uint32_t *blob)
{
    *blob = ((domainId << 28) | (itemsLength << 24) | (itemIdx << 16) | (modulusSizeBits));
}

/*
 *  ================ Helper APIs to handle ECDH related operation ================
 */

/*
 *  ======== HSMBareMetalECCParams_asymDHPriKeyToHW ========
 */
void HSMBareMetalECCParams_asymDHPriKeyToHW(uint8_t *in,
                                            const size_t modulusSizeBits,
                                            const uint8_t domainId,
                                            uint8_t *blob)
{
    HSMBareMetalECCParams_asymVectorHeaderFormat(modulusSizeBits, 1, 0, domainId, (uint32_t *)blob);

    HSMBareMetalECCParams_reverseMemCpy(&blob[HSM_ASYM_DATA_VHEADER], &in[0], BITS_TO_BYTES(modulusSizeBits));
}

/*
 *  ======== HSMBareMetalECCParams_asymBEDHPubKeyToHW ========
 */
static void HSMBareMetalECCParams_asymBEDHPubKeyToHW(uint8_t *in,
                                                     const size_t modulusSizeBits,
                                                     const uint8_t itemsLength,
                                                     const uint8_t domainId,
                                                     uint8_t *blob)
{
    HSMBareMetalECCParams_asymVectorHeaderFormat(modulusSizeBits, itemsLength, 0, domainId, (uint32_t *)blob);

    HSMBareMetalECCParams_reverseMemCpy(&blob[HSM_ASYM_DATA_VHEADER], &in[0], BITS_TO_BYTES(modulusSizeBits));

    /* In the case of public key, curve25519 based public keys only offer one component pubkey.u.
     * All other curves have two components pubkey.x and pubkey.y.
     */
    if (itemsLength == HSM_ASYM_ECC_PUB_KEY_VCOUNT)
    {
        HSMBareMetalECCParams_asymVectorHeaderFormat(modulusSizeBits,
                                                     itemsLength,
                                                     1,
                                                     0,
                                                     (uint32_t *)&blob[HSM_ASYM_DATA_SIZE_VWB(modulusSizeBits)]);

        HSMBareMetalECCParams_reverseMemCpy(&blob[HSM_ASYM_DATA_SIZE_VWB(modulusSizeBits) + HSM_ASYM_DATA_VHEADER],
                                            &in[BITS_TO_BYTES(modulusSizeBits)],
                                            BITS_TO_BYTES(modulusSizeBits));
    }
}

/*
 *  ======== HSMBareMetalECCParams_asymDHPubKeyToHW ========
 */
void HSMBareMetalECCParams_asymDHPubKeyToHW(uint8_t *in,
                                            const size_t modulusSizeBits,
                                            const uint8_t itemsLength,
                                            const uint8_t domainId,
                                            uint8_t *blob)
{
    uint8_t *key = in + HSM_ASYM_ECC_PUB_KEY_UNCOMP_ENC_LENG;

    HSMBareMetalECCParams_asymBEDHPubKeyToHW(key, modulusSizeBits, itemsLength, domainId, blob);
}

/*
 *  ======== HSMBareMetalECCParams_asymRsaSignatureFromHw ========
 */
static void HSMBareMetalECCParams_asymRsaSignatureFromHw(const uint8_t *const in,
                                                         const size_t modulusSizeBits,
                                                         uint8_t *out)
{
    uint32_t size = (uint32_t)BITS_TO_BYTES(modulusSizeBits);

    /* Convert big integer from HW to application format */
    HSMBareMetalECCParams_reverseMemCpy(out, &in[HSM_ASYM_DATA_VHEADER], size);
}

/*
 *  ======== HSMBareMetalECCParams_asymBEDHPubKeyFromHW ========
 */
static void HSMBareMetalECCParams_asymBEDHPubKeyFromHW(const uint8_t *const in,
                                                       const size_t modulusSizeBits,
                                                       const uint8_t itemsLength,
                                                       uint8_t *out_pubKey)
{
    HSMBareMetalECCParams_asymRsaSignatureFromHw(in, modulusSizeBits, out_pubKey);

    /* In the case of public key, curve25519 based public keys only offer one component pubkey.u.
     * All other curves have two components pubkey.x and pubkey.y.
     */
    if (itemsLength == HSM_ASYM_ECC_PUB_KEY_VCOUNT)
    {
        HSMBareMetalECCParams_asymRsaSignatureFromHw(&in[HSM_ASYM_DATA_SIZE_VWB(modulusSizeBits)],
                                                     modulusSizeBits,
                                                     &out_pubKey[BITS_TO_BYTES(modulusSizeBits)]);
    }
}

/*
 *  ======== HSMBareMetalECCParams_asymDHPubKeyFromHW ========
 */
void HSMBareMetalECCParams_asymDHPubKeyFromHW(const uint8_t *const in,
                                              const size_t modulusSizeBits,
                                              const uint8_t itemsLength,
                                              uint8_t *out_pubKey)
{
    uint8_t *key = out_pubKey;

    *key = HSM_ASYM_ECC_UNCOMP_ENC_VALUE;
    key  = key + HSM_ASYM_ECC_PUB_KEY_UNCOMP_ENC_LENG;

    HSMBareMetalECCParams_asymBEDHPubKeyFromHW(in, modulusSizeBits, itemsLength, key);
}

/*
 *  ================ Helper APIs to handle DSA related operation ================
 */

/*
 *  ======== HSMBareMetalECCParams_asymDsaSignatureFromHW ========
 */
void HSMBareMetalECCParams_asymDsaSignatureFromHW(const uint8_t *const in,
                                                  const size_t modulusSizeBits,
                                                  uint8_t *outR,
                                                  uint8_t *outS)
{
    /* Convert Signature from HW to application format */
    HSMBareMetalECCParams_asymRsaSignatureFromHw(in, modulusSizeBits, outR);
    HSMBareMetalECCParams_asymRsaSignatureFromHw(&in[HSM_ASYM_DATA_SIZE_VWB(modulusSizeBits)], modulusSizeBits, outS);
}

/*
 *  ======== HSMBareMetalECCParams_asymDsaSignatureToHW ========
 */
void HSMBareMetalECCParams_asymDsaSignatureToHW(const uint8_t *const signatureR,
                                                const uint8_t *const signatureS,
                                                const size_t modulusSizeBits,
                                                uint8_t *const blob)
{
    HSMBareMetalECCParams_asymVectorHeaderFormat(modulusSizeBits, 2, 0, 0, (uint32_t *)blob);

    HSMBareMetalECCParams_reverseMemCpy(&blob[HSM_ASYM_DATA_VHEADER], &signatureR[0], BITS_TO_BYTES(modulusSizeBits));

    /* In the case of public key, curve25519 based public keys only offer one component pubkey.u.
     * All other curves have two components pubkey.x and pubkey.y.
     */
    HSMBareMetalECCParams_asymVectorHeaderFormat(modulusSizeBits,
                                                 2,
                                                 1,
                                                 0,
                                                 (uint32_t *)&blob[HSM_ASYM_DATA_SIZE_VWB(modulusSizeBits)]);

    HSMBareMetalECCParams_reverseMemCpy(&blob[HSM_ASYM_DATA_SIZE_VWB(modulusSizeBits) + HSM_ASYM_DATA_VHEADER],
                                        &signatureS[0],
                                        BITS_TO_BYTES(modulusSizeBits));
}

/*
 *  ======== HSMBareMetalECCParams_bufferMatch ========
 */
#if defined(__IAR_SYSTEMS_ICC__)
    #pragma inline = never
#elif defined(__TI_COMPILER_VERSION__) && !defined(__cplusplus)
    #pragma FUNC_CANNOT_INLINE(CryptoUtils_buffersMatch)
#elif defined(__TI_COMPILER_VERSION__)
    #pragma FUNC_CANNOT_INLINE
#endif
CRYPTOUTILS_NOINLINE bool HSMBareMetalECCParams_bufferMatch(const volatile uint8_t *const buffer1,
                                                            const volatile uint8_t *const buffer2,
                                                            const size_t size)
{
    volatile uint8_t tempResult = 0;
    uint8_t byte1;
    uint8_t byte2;
    size_t i;

    /* XOR each byte of the buffer together and OR the results.
     * If the OR'd result is non-zero, the buffers do not match.
     * There is no branch based on the content of the buffers here to avoid
     * timing attacks.
     */
    for (i = 0; i < size; i++)
    {
        byte1 = ((uint8_t *)buffer1)[i];
        byte2 = ((uint8_t *)buffer2)[i];

        tempResult |= byte1 ^ byte2;
    }

    return tempResult == 0;
}
