\page ieee_rxtx_handler IEEE 802.15.4 Rx and Tx Command Handler

# Introduction

The IEEE 802.15.4 Rx and Tx command allows to receive frames (packets) at a specific RF frequency (corresponding to a channel defined in the 15.4 standard), or to transmit frames at the same frequency. The following sections describe how the command can be configured and used, its life cycle, and how it fits into the RCL architecture.

In order to submit an IEEE 802.15.4 Rx and Tx command, the following steps must have taken place:

1. RCL has been initialized (See ::RCL_init) and a handle must exist (See ::RCL_open).

2. The ::RCL_CMD_IEEE_RX_TX_t command has been initialized and configured.

3. To do Rx, the following steps are also needed:

    a. The ::RCL_CmdIeee_RxAction_t struct has been initialized and configured

    b. A [Multibuffer](rcl_buffer_management.html) has been initialized and set up.

4. To do Tx, the following steps are also needed:

    a. The ::RCL_CmdIeee_TxAction_t struct has been initialized and configured

    b. A data entry of the frame to transmit has been initialized and set up.

Once these steps have been completed, ::RCL_Command_submit and ::RCL_Command_pend are called to effectively send and then wait for the command to conclude. Once this has happened, the callback and the command status can be used for error checking and the application can proceed according to its specification.

# Usage

The IEEE 802.15.4 Rx and Tx command can be used for all aspects of supporting the IEEE 802.15.4 MAC and PHY. The command can be used to receive frames, transmit frames after clear-channel assessment (CCA), and transmit and receive ACKs. It can also be used for energy scanning.

In addition to the common fields, the command has fields for frequency, [Tx power](rcl_tx_power_output.html), and [coexistence](rcl_coex.html). The frequency is given in Hz; the macro ::RCL_CMD_IEEE_CHANNEL_FREQUENCY can be used to convert from an IEEE 802.15.4 channel number. The command also contains pointers to an Rx action, a Tx action, and statistics of the type ::RCL_STATS_IEEE_t, which is updated if non-NULL. It is required that a Tx action, an Rx action, or both are provided (non-NULL).

The statistics structure `stats` contains counters and RSSI measurements for the command. The structure is updated when the command ends; if `stats->config.activeUpdate` is 1, it is also updated every time a frame is received. If `stats->config.accumulate` is 0, all the counters are reset when the command starts. Otherwise, existing values are used as the starting point and updated as the new command runs.

The various counters give the number of times the given category of frame has been sent or received. The `stats->lastTimestamp` field gives a timestamp of the start of the last successfully received frame and is valid if `stats->timestampValid` is 1. The `stats->lastRssi` field gives the RSSI of the last received frame or ::LRF_RSSI_INVALID if no valid RSSi is available. Note that this value only gives the RSSI of a frame; to read the RSSI at any time including sync search, use the function ::RCL_readRssi. The `stats->maxRssi` field gives the maximum RSSI observed any time during the receiver operation. Note that if `stats->config.accumulate`, the value of `stats->maxRssi` when the command starts is used as the initial maximum, and only RSSI values greater than that will cause an update to occur.

## Receive Action

The command is set up as a receiver by setting the `rxAction` field to a non-NULL pointer to a ::RCL_CmdIeee_RxAction_t struct. When the command is started, the radio will be set up as a receiver looking for IEEE 802.15.4 frames. If a PHY header is found, the frame is received and the frame checksum (FCF) according to the MAC definition (that is, CRC), is checked at the end of the frame. If frame filtering is enabled, additional checks are done on the MAC header.

If a frame is received and the CRC is correct, it is stored to the first multibuffer in the provided receive queue, following the [internal packet format](rcl_command_handlers.html). The CRC is not stored, but a link quality indicator (LQI), received signal strength indicator (RSSI) and timestamp are appended at the end. By modifying the configuration, appended fields can be changed. The format of received frames is shown below.

![IEEE 802.15.4 Rx frame entry example](docs/rcl/source/images/internal_packet_format_ieee.png)

The appended fields may be extracted from a data entry using the functions ::RCL_IEEE_getRxRssi, ::RCL_IEEE_getRxLqi, and ::RCL_IEEE_getRxTimestamp.

After receiving a frame, the command will go back to Rx and listen for more frames. The command will run until one of the timeouts (if any) set up in the command expires, the stop API is used, a Tx action with `endCmdWhenDone` set to 1 is finished, or an error occurs (for instance running out of Rx buffer).

Below is an example of starting a receiver with normal frame filtering and auto-ACK.

\snippet source/snippets/ieee_example/ieee_example.c ieee_rx_code_snippet

### Frame Filtering

If frame filtering is enabled by setting `numPan` of the Rx action to 1, each received frame is subject to frame filtering. This means that the MAC header is checked against the filtering rules found in the IEEE 802.15.4 spec. In particular, this means checking the destination address and destination PAN ID fields against the local address and PAN ID fields found in the PAN configuration (::RCL_CmdIeee_PanConfig). If a received frame does not meet the frame filtering requirements, it is marked as ignored and discarded at the end of reception. If `frameFiltStop` is 1, frame reception is stopped immediately if the frame does not meet the frame filtering requirements, and the receiver will return to sync search (in this case, the missed frame will not be counted as ignored). If a frame passes frame filtering, the event ::LRF_EventRxCtrl is raised after reception of the MAC header; this event may be used in a callback.

If `alwaysStoreAck` of the Rx action is 0, all received ACK frames are ignored except if received as part of the ACK reception of a Tx action, see [Transmit Action](#transmit-action) below.

If `numPan` is 0, the receiver is set in promiscuous mode, which means that all frames are accepted as long as they have CRC OK.

### Source Matching

The source addresses of a received frame can be compared against a list of known addresses. The main purpose of this is to provide the wanted frame pending bit in transmitted ACKs. The current version only supports source matching of short addresses. The source matching table is given in `sourceMatchingTableShort` of the PAN configuration; if it is `NULL`, source matching is not done. The source matching table may contain up to 64 short address-PAN ID pairs. The table has the type ::RCL_CmdIeee_SourceMatchingTableShort. In the table, `shortEntry` is an array of `numEntries` short address-PAN ID pairs of type ::RCL_CmdIeee_PanIdAddr.

The source matching table also contains two arrays called `entryEnable` and `framePending`. For each entry, one bit from these arrays applies, so for the first entry (index 0), the least significant bit of `entryEnable[0]` and `framePending[0]` apply, for the entry with index 15, the most significant bit of `entryEnable[0]` and `framePending[0]` apply, for the entry with index 16, the least significant bit of `entryEnable[1]` and `framePending[1]` apply, and so on.

If the `entryEnable` bit corresponding to an entry is 0, the entry is ignored and not used for source matching; otherwise the short address and PAN ID of that entry is compared to the received source short address and source PAN ID. If a match is found, the corresponding `framePending` bit may be used for sending an ACK; see the next section.

### Auto-ACK

If auto-ack is enabled by setting `autoAckMode` in the PAN configuration different from ::RCL_CmdIeee_AutoAck_Off, a received frame will be ACK'ed if the following conditions are met:
* The frame passed frame filtering
* The CRC of the frame is OK
* The frame is a Data or MAC command frame with AR = 1 in the frame control field of the MAC header
* The destination address is not broadcast

In this case, the radio will start the transmitter and transmit an Imm-ACK frame after an interframe spacing of 192 us after the received frame ended. The radio will construct the Imm-ACK according to the IEEE 802.15.4 specification. The sequence number is set to the same as the sequence number of the received frame. The Frame pending bit in the transmitted ACK depends on the `autoAckMode` setting, the `defaultPend` setting, the received frame, and source matching results. The receiver can be set up to give different results for a MAC command frame where the command type is a data request. The table below shows the different situations.

|`autoAckMode`                               | Received frame type | Source matching result | Transmitted Frame pending bit           |
|--------------------------------------------|---------------------|------------------------|-----------------------------------------|
|::RCL_CmdIeee_AutoAck_ImmAckNoAutoPend      | Any                 | Any                    | `defaultPend`                           |
|::RCL_CmdIeee_AutoAck_ImmAckAutoPendAll     | Any                 | No match               | `defaultPend`                           |
|::RCL_CmdIeee_AutoAck_ImmAckAutoPendAll     | Any                 | Match found            | Bit of `framePending` for matched entry |
|::RCL_CmdIeee_AutoAck_ImmAckAutoPendDataReq | Data request        | No match               | `defaultPend`                           |
|::RCL_CmdIeee_AutoAck_ImmAckAutoPendDataReq | Data request        | Match found            | Bit of `framePending` for matched entry |
|::RCL_CmdIeee_AutoAck_ImmAckAutoPendDataReq | Other               | Any                    | 0                                       |

If `autoAckMode` is ::RCL_CmdIeee_AutoAck_ImmAckProvidedFrame, the treatment of Imm-ACKs is as for Enh-ACKs, described in the next section.

### Enhanced ACKs and Caller Provided Immediate ACKs

If `autoAckMode` is not ::RCL_CmdIeee_AutoAck_Off, the RCL will respond to an accepted frame of frame version 2 with an Enh-ACK, under the same conditions as above. In this case, the system needs to use the function ::RCL_IEEE_enterAck to provide the ACK payload. This can be done after a callback of the ::LRF_EventRxCtrlAck event has happened. If the received frame has CRC error, the ACK will not go out, even if it is provided. If `autoAckMode` is ::RCL_CmdIeee_AutoAck_ImmAckProvidedFrame, the same system is used in order for the caller to provide an Imm-ACK to be transmitted in response to an accepted frame of frame version 0 or 1.

The payload provided to ::RCL_IEEE_enterAck should be provided in the `ackData` parameter as a pointer to a ::RCL_Buffer_DataEntry_s struct (a cast is needed). It is not necessary to provide all the payload at once, but the `length` field of the ::RCL_Buffer_DataEntry_s struct must be set as for the total length of the ACK. If the `numWords` parameter is non-zero, this indicates the number of 32-bit words to be provided. Note that if the number of bytes available is not a multiple of 4, the remaining bytes must be saved for the next data provision. To provide more data, a new call to ::RCL_IEEE_enterAck should be made; this time, `ackData` should point to the next word of payload that was not provided in the previous call. At the last call to the function, `numWords` must give the number of words missing to fill the frame with the length indicated in the `length` field provided in the first call; if the number of bytes in the ACK frame is not a multiple of 4, padding bytes should be provided in the end. The return value of ::RCL_IEEE_enterAck indicates if the operation was successful and if more data is expected in this ACK.

If any part of the ACK payload is not provided in time for it to go on the air, the transmission will be stopped so that the CRC on the receive side will not be OK. This is signaled by the ::LRF_EventTxCtrl event. If ::RCL_IEEE_enterAck is called after this, it will return ::RCL_IEEE_AckNotExpected.

#### Cancelling ACKs

It is possible to cancel an outgoing ACK at any time from the ::LRF_EventRxCtrlAck is received until the ACK is finished transmitting. The ACK will be stopped as soon as possible, but some energy may have gone on the air. At least, the CRC will not be transmitted, so the CRC at the receive side will be wrong. Cancelling an ACK is done with the ::RCL_IEEE_cancelAck function. The return value indicates if the cancellation was successful; otherwise the ACK may have been sent. In order to allow cancelling Imm-ACKs, `autoAckMode`  must be ::RCL_CmdIeee_AutoAck_ImmAckProvidedFrame.

### Examining a Partially Received Frame

A frame under reception may be inspected by using the ::RCL_IEEE_readPartialFrame function. This is typically called after a ::LRF_EventRxCtrlAck or ::LRF_EventRxCtrl event is received; at this time, at least all the addressing fields of the MAC header are available. This is typically used to start preparing an Enh-ACK while receiving the frame to be ACK-ed.

In order to use the function to read a frame, the Rx and Tx command must still be running and reception of a new frame must not have been started. The caller provides a pointer to an ::RCL_Buffer_DataEntry_s struct (`dataEntry`) where the received data will be stored. The size of the struct must also be provided, and the number of bytes stored is limited upwards by the number of bytes received at the time of calling the function and by the size of the struct. When the function returns, `dataEntry->length` will indicate the number of bytes available, so it will not match the received PHY header (unless the packet reception is complete).

### Dynamically Updating Receiver Parameters

Parameters for a running command with an Rx action can be changed during operation. To do so, the relevant parameters in the ::RCL_CmdIeee_RxAction_t struct can be updated. The change will not take effect until the ::RCL_IEEE_updateRxAction API is called. Until then, the original parameters apply. If the command is running, the call will return ::RCL_IEEE_UpdatePending, meaning that the update is in progress. The RCL will then generate the event ::RCL_EventCmdUpdateDone once the update is finished. In the meantime, the receiver will be turned off for a short time, but this will not be done until any frame reception or Rx action is finished.

If the updated parameters contain an error, the running command may end with ::RCL_CommandStatus_Error_Param as the status.

All fields of the ::RCL_CmdIeee_RxAction_t can be changed this way except `rxBuffers`. Updates to the receive buffers should be done using the normal APIs from RCL_Buffer.h. Note that the `rxAction` pointer in the command structure should not be modified while the command is running.

The fields of the Rx action should not be modified between calling ::RCL_IEEE_updateRxAction and getting the ::RCL_EventCmdUpdateDone event or between command submission and the first ::RCL_EventCmdStarted event.

### Dynamically Updating Source Matching Table

An entry in a source matching table can be updated while a command is running by using the API ::RCL_IEEE_updateSourceMatchingTableShort. This is done without stopping the receiver. When the function is called while the command is running, it will return ::RCL_IEEE_UpdatePending, meaning that the update is in progress. The RCL will then generate the event ::RCL_EventCmdUpdateDone once the update is finished. At that time, the source matching table has been updated by the RCL to show the new state.

A call to ::RCL_IEEE_updateSourceMatchingTableShort cannot be done while another such update or an update through ::RCL_IEEE_updateRxAction is pending; if attempted, the call will return ::RCL_IEEE_UpdateCmdError. Additionally, ::RCL_IEEE_updateSourceMatchingTableShort should not be called if the ::RCL_CmdIeee_RxAction_t struct is being updated, even if ::RCL_IEEE_updateRxAction has not yet been called.

The operations that can be done are the following:
* Enable or disable a current entry in the table
* Change the frame pending bit of a current entry in the table
* Set a new PAN ID and address in an entry of the table. In this case, the entry will be enabled (regardless of the previous state) and the frame pending bit is set as requested

During an update, a received frame may be treated as belonging to either the old or the new state of the table entry, but never a mix of the two.

If a large part of the table should be changed, it may be faster to provide the new table by updating `rxAction->panConfig[0].sourceMatchingTableShort` or its contents and then calling ::RCL_IEEE_updateRxAction than using ::RCL_IEEE_updateSourceMatchingTableShort for each entry.

### Dynamically Updating Tx Power

The [Tx power](rcl_tx_power_output.html) set in the command may be updated using the function ::RCL_IEEE_updateTxPower. If this function is called while the command is running, the new Tx power will be used for the next transmitted frame onwards, affecting both ACKs and frames provided using a Tx action. A frame that is under transmission when the function is called will not be impacted. If the provided Tx power is invalid, the function will return an error and the command will continue without changing Tx power. Otherwise, the `txPower` field in the command structure is changed to the new value.

If the command is not running, the `txPower` field in the command structure is updated, but the radio is not notified until the command starts. In this case, it is not checked whether the new Tx power is valid, and in case of an error, this will be seen when the command starts.

### Energy Detection Scanning

In order to perform energy detection scanning, the normal ::RCL_CMD_IEEE_RX_TX_t command can be used with only an Rx action. The `common.timing.relGracefulStopTime` or `common.timing.relHardStopTime` should be used to set the duration of the scan, or alternatively, the caller may cancel the command at the appropriate time. If IEEE 802.15.4 frames do not need to be received or sensed except for their energy, `rxAction->disableSync` should be set to 1. When the command ends, `stats->maxRssi` can be read to see the maximum RSSI observed on the channel. Note that `stats->config.accumulate` should be 0 unless this is meant as a continuation of a previous scan.

## Transmit Action

The command is set up as a transmitter by setting the `txAction` field to a non-NULL pointer to a ::RCL_CmdIeee_TxAction_t struct.

The frame to send is provided as an ::RCL_Buffer_DataEntry_s struct. No list is used, since only one frame is sent.

If `rxAction` is NULL, the command will act as a simple transmitter and end after the frame has been transmitted. In this case, the transmission must be done without CCA and ACK reception by setting `ccaMode` to ::RCL_CmdIeee_NoCca and `expectImmAck` and `expectEnhAck` to 0; otherwise the command will end immediately with the status ::RCL_CommandStatus_Error_Param. In order to use the command's start time (if provided) as the transmit time of the frame, use the default settings for the timing inside the Tx action.

The Tx action may be set up to start when the command is submitted, or it may be added later. To add a Tx action to an already running command, the API ::RCL_IEEE_Tx_submit can be used, as long as no other Tx action is running and pending. A running or pending Tx action may be stopped using the API ::RCL_IEEE_Tx_stop. When a Tx action finishes, the RCL event ::RCL_EventCmdStepDone is raised, and it may be processed in a callback. At this time, the status of the Tx action can be read from the `txStatus` field.

The Tx action may be set up to perform Clear-Channel Assessment (CCA) before Tx to ensure the channel is free before transmitting. If the channel is seen as busy, the Tx action will end without transmitting, with `txStatus` ::RCL_CommandStatus_ChannelBusy. If CCA is used, an Rx action must be active at the same time. If the channel is busy, it is up to the caller to set up a wait time according to the CSMA-CA algorithm of IEEE 802.15.4 before making another try.

The Tx action can start directly, or be set up with a start time given by `absCcaStartTime`, depending on `ccaScheduling`. If `ccaScheduling` is ::RCL_Schedule_AbsTime, the time provided is the time that the CCA check is performed, otherwise the CCA check happens as soon as possible. If `allowDelay` is 1, the check is done as soon as possible if `absCcaStartTime` is in the past or has too little margin, otherwise the Tx action ends prematurely with ::RCL_CommandStatus_Error_StartTooLate as `txStatus`. The time from the provided `absCcaStartTime` or the start of the operation to a transmission is started is given by `relTxStartTime`. If `allowTxDelay` is 1, the transmission will happen even if the time is too short to achieve, otherwise the Tx action ends prematurely with ::RCL_CommandStatus_Error_StartTooLate as `txStatus`. To start the transmission as soon as possible after the CCA, set `relTxStartTime` to 0 and `allowTxDelay` to 1.

If a frame is transmitted, the actual start time of the frame, taking all the above operations and allowed delays into account, can be read from
`txTimeStamp` after the Tx action has finished.

The Tx action can be set up to listen for an ACK after the transmission happens by setting either `expectImmAck` or `expectEnhAck` to 1. This option should only be used when sending a frame for which an ACK is expected. If the transmitted frame has frame version 2 and an Enh-ACK is thus expected, `expectEnhAck` should be 1, while when an Imm-ACK is expected `expectImmAck` should be 1. If this is enabled, the receiver will look for an ACK following the transmitted frame. In order for the ACK to be accepted when `expectImmAck` is 1, it has to be a correctly formed Imm-ACK where the sequence number is the same as in the frame just transmitted. When `expectEnhAck` is 1, the ACK is accepted as long as the frame type and frame version indicate an Enh-ACK and the CRC is correct. The rest of the check must be done by examining the received ACK frame. The Tx action will end after receiving the ACK or if no such ACK is found within the time given by `ackTimeout`; the result is indicated by `txStatus`. The status will also indicate if the received ACK frame had the Frame Pending (FP) bit of the MAC header set to 1.

If `endCmdWhenDone` is 1 or if no Rx action is provided, the Rx and Tx command will end when the Tx action ends. In this case, the command's `common.status` will be the same as the `txStatus` of the Tx action. Otherwise, the receiver will continue as normal.

The statuses of a Tx action are summarized below:

| Status                                 | Description                                                                           |
|----------------------------------------|---------------------------------------------------------------------------------------|
| ::RCL_CommandStatus_Idle               | Tx action is not yet submitted                                                        |
| ::RCL_CommandStatus_Scheduled          | Tx action is waiting for start time                                                   |
| ::RCL_CommandStatus_Active             | Tx action is running                                                                  |
| ::RCL_CommandStatus_Finished           | Frame transmitted, ACK received successfully (FP = 0) or not expected                 |
| ::RCL_CommandStatus_ChannelBusy        | CCA gave busy channel as result                                                       |
| ::RCL_CommandStatus_NoSync             | No frame was found while looking for ACK                                              |
| ::RCL_CommandStatus_RxErr              | The frame received after Tx was not the expected ACK or had CRC error                 |
| ::RCL_CommandStatus_FramePending       | Frame transmitted, ACK received successfully (FP = 1)                                 |
| ::RCL_CommandStatus_CoexNoGrant        | Frame was not fully transmitted as the [coexistence](rcl_coex.html) did not get grant |
| ::RCL_CommandStatus_Error_Param        | A parameter in the Tx action was not permitted                                        |
| ::RCL_CommandStatus_Error_StartTooLate | Start time of CCA or Tx was in the past or had too little margin                      |

In addition, a status indicating stop may be given if the Tx action or the Rx and Tx command ends from an API call, by the scheduler or an end time.

Below is an example of submitting a Tx action to a running command.

\snippet source/snippets/ieee_example/ieee_example.c ieee_tx_action_code_snippet

### CCA Operation

If `ccaMode` is different from ::RCL_CmdIeee_NoCca, some kind of checking is done before sending. The checks are following the modes specified in the IEEE 802.15.4 standard. All modes doing CCA will check if the receiver is currently receiving a frame or if an ACK is being transmitted or is pending. In this case, the channel will be seen as busy in all CCA modes. For ::RCL_CmdIeee_CcaMode4Aloha, this is the only check made. The remaining checks described below are made only if no frame is being received or ACK transmitted.

CCA mode 1 is implemented if `ccaMode` is set to ::RCL_CmdIeee_CcaMode1Energy. In this case, the received signal strength (RSSI) is compared to `rssiLimit`. If the observed RSSI is greater than or equal to the limit, the channel is seen as busy. The RSSI of the IEEE 802.15.4 receiver is calculated over a window of 128 us. If the receiver has not run long enough for an RSSI value to be available, a new assessment is made when an RSSI becomes available.

CCA mode 2 is implemented if `ccaMode` is set to ::RCL_CmdIeee_CcaMode2Signal. In this case, the received signal is checked for correlation against the DSSS (direct sequence spread-spectrum) sequence used in the PHY. The number of correlation tops over a 128 us period ahead of the CCA time is evaluated; this number can be between 0 and 8. If this number of correlation tops is greater than the threshold `ccaCorrThresh`, the channel is seen as busy.

CCA mode 3 is implemented if `ccaMode` is ::RCL_CmdIeee_CcaMode3EnergyOrSignal or ::RCL_CmdIeee_CcaMode3EnergyAndSignal. This gives a combination of CCA mode 1 and 2. The IEEE 802.15.4 standard allows combining these two modes in any way. ::RCL_CmdIeee_CcaMode3EnergyOrSignal means that the channel will be declared busy if the conditions for a busy channel by either CCA mode 1, CCA mode 2, or both, are fulfilled. ::RCL_CmdIeee_CcaMode3EnergyAndSignal means that the channel will be declared busy if the conditions for a busy channel by both CCA mode 1 and CCA mode 2 are fulfilled.

Note that the detection by correlation tops will only detect other IEEE 802.15.4 transmitters. If the CCA is intended to protect against collisions with other PHYs, such as WiFi or Bluetooth, ::RCL_CmdIeee_CcaMode1Energy or ::RCL_CmdIeee_CcaMode3EnergyOrSignal should be used. The other modes will declare the channel as idle as long as no IEEE 802.15.4 signal is observed.

# Architecture

## Events

The following tables list the events handled.

| RCL Event (In)              | Description                                     |
|-----------------------------|-------------------------------------------------|
| ::RCL_EventSetup            | Setup has been performed                        |
| ::RCL_EventTimerStart       | Timer-based start signalled                     |
| ::RCL_EventRxBufferUpdate   | Rx buffer has been updated                      |
| ::RCL_EventHandlerCmdUpdate | Tx action has been submitted or stopped         |
| ::RCL_EventStopTimesUpdated | Scheduler has updated active command stop times |

| RCL Event (Out)             | Description                                                                             |
|-----------------------------|-----------------------------------------------------------------------------------------|
| ::RCL_EventCmdStarted       | Command handler has accepted and started executing a command                            |
| ::RCL_EventLastCmdDone      | The RCL is finished with the command                                                    |
| ::RCL_EventRxEntryAvail     | An Rx entry has been made available                                                     |
| ::RCL_EventRxBufferFinished | An Rx multi-buffer is finished                                                          |
| ::RCL_EventCmdStepDone      | A Tx action is finished                                                                 |
| ::RCL_EventCmdUpdateDone    | The update of Rx parameters or a source matching table is done                          |
| ::RCL_EventCoexNoTx         | An ACK was not fully transmitted as the [coexistence](rcl_coex.html) did not get grant  |

| LRF Event                   | Description                                                                             |
|-----------------------------|-----------------------------------------------------------------------------------------|
| ::LRF_EventOpDone           | The PBE operation has finished (not recommended for callback)                           |
| ::LRF_EventOpError          | The PBE operation has finished with error or hard stop (not recommended for callback)   |
| ::LRF_EventRxCtrlAck        | Received MAC header indicates that CPU must provide ACK payload                         |
| ::LRF_EventRxOk             | Frame received with CRC OK and passed frame filtering                                   |
| ::LRF_EventRxNok            | Frame received with CRC error                                                           |
| ::LRF_EventRxIgnored        | Frame received with CRC OK, but failed frame filtering                                  |
| ::LRF_EventTxDone           | Frame transmitted by Tx action                                                          |
| ::LRF_EventTxAck            | ACK transmitted in response to received frame                                           |
| ::LRF_EventTxCtrl           | ACK transmission failed because data arrived too late                                   |
| ::LRF_EventRxCtrl           | Received MAC header passed frame filtering                                              |
| ::LRF_EventRxEmpty          | Treatment of ACK reception as part of Tx action done                                    |
