/* testvectors_asn1_der_data.h
 *
 * Description: Test vectors for ASN1 DER objects.
 */

/* -------------------------------------------------------------------------- */
/*                                                                            */
/*   Module        : DDK-130_bsd                                              */
/*   Version       : 4.1.1                                                    */
/*   Configuration : DDK_EIP130_BSD                                           */
/*                                                                            */
/*   Date          : 2023-Mar-10                                              */
/*                                                                            */
/* Copyright (c) 2014-2023 by Rambus, Inc. and/or its subsidiaries.           */
/*                                                                            */
/* Redistribution and use in source and binary forms, with or without         */
/* modification, are permitted provided that the following conditions are     */
/* met:                                                                       */
/*                                                                            */
/* 1. Redistributions of source code must retain the above copyright          */
/* notice, this list of conditions and the following disclaimer.              */
/*                                                                            */
/* 2. Redistributions in binary form must reproduce the above copyright       */
/* notice, this list of conditions and the following disclaimer in the        */
/* documentation and/or other materials provided with the distribution.       */
/*                                                                            */
/* 3. Neither the name of the copyright holder nor the names of its           */
/* contributors may be used to endorse or promote products derived from       */
/* this software without specific prior written permission.                   */
/*                                                                            */
/* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS        */
/* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT          */
/* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR      */
/* A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT       */
/* HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,     */
/* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT           */
/* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,      */
/* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY      */
/* THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT        */
/* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE      */
/* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.       */
/* -------------------------------------------------------------------------- */

#include <third_party/hsmddk/include/Config/cs_eip130.h>

#include <third_party/hsmddk/include/Kit/DriverFramework/Basic_Defs_API/incl/basic_defs.h>

/*
The testvectors are generated with the following openssl commands:

# NIST
openssl ecparam -genkey -name secp224r1 -noout -outform DER -out prvkey_N224.der
openssl pkey -inform DER -in prvkey_N224.der -pubout -outform DER -out pubkey_N224.der
openssl ecparam -genkey -name prime256v1 -noout -outform DER -out prvkey_N256.der
openssl ec -inform DER -in prvkey_N256.der -pubout -outform DER -out pubkey_N256.der
openssl ecparam -genkey -name secp384r1 -noout -outform DER -out prvkey_N384.der
openssl ec -inform DER -in prvkey_N384.der -pubout -outform DER -out pubkey_N384.der
openssl ecparam -genkey -name secp521r1 -noout -outform DER -out prvkey_N521.der
openssl ec -inform DER -in prvkey_N521.der -pubout -outform DER -out pubkey_N521.der

# Brainpool
openssl ecparam -genkey -name brainpoolP224r1 -noout -outform DER -out prvkey_B224.der
openssl ec -inform DER -in prvkey_B224.der -pubout -outform DER -out pubkey_B224.der
openssl ecparam -genkey -name brainpoolP256r1 -noout -outform DER -out prvkey_B256.der
openssl ec -inform DER -in prvkey_B256.der -pubout -outform DER -out pubkey_B256.der
openssl ecparam -genkey -name brainpoolP384r1 -noout -outform DER -out prvkey_B384.der
openssl ec -inform DER -in prvkey_B384.der -pubout -outform DER -out pubkey_B384.der
openssl ecparam -genkey -name brainpoolP512r1 -noout -outform DER -out prvkey_B512.der
openssl ec -inform DER -in prvkey_B512.der -pubout -outform DER -out pubkey_B512.der

# curve25519
openssl genpkey -algorithm x25519 -outform DER -out prvkey_x25519.der
openssl pkey -inform DER -in prvkey_x25519.der -pubout -outform DER -out pubkey_x25519.der

# Edwards-25519
openssl genpkey -algorithm ed25519 -outform DER -out prvkey_ed25519.der
openssl pkey -inform DER -in prvkey_ed25519.der -pubout -outform DER -out pubkey_ed25519.der

# RSASSA-PKCS#1v1.5
openssl genpkey -algorithm rsa -pkeyopt rsa_keygen_bits:2048 -outform DER -out prvkey_rsa_pkcs1_2k.der
openssl pkey -inform DER -in prvkey_rsa_pkcs1_2k.der -pubout -outform DER -out pubkey_rsa_pkcs1_2k.der

# RSASSA-PSS
openssl genpkey -algorithm rsa-pss -pkeyopt rsa_keygen_bits:2048 -outform DER -out prvkey_rsa_pss_2k.der
openssl pkey -inform DER -in prvkey_rsa_pss_2k.der -pubout -outform DER -out pubkey_rsa_pss_2k.der

*/

/* NIST */
static const uint8_t Asn1Der_ECC_N224_PrvKey[] =
{
    0x30, 0x68, 0x02, 0x01, 0x01, 0x04, 0x1C, 0xBA, 0x92, 0x1A, 0xDE, 0xEA, 0xBF, 0x25, 0x87, 0x3E,
    0x6C, 0x3E, 0xBF, 0x01, 0x09, 0x5C, 0x2C, 0x25, 0x85, 0x31, 0x8F, 0x98, 0xEB, 0x33, 0x16, 0xF6,
    0x43, 0x22, 0x3D, 0xA0, 0x07, 0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x21, 0xA1, 0x3C, 0x03, 0x3A,
    0x00, 0x04, 0xBC, 0xF8, 0xAE, 0x26, 0x1F, 0x30, 0xA2, 0x8C, 0x64, 0x76, 0x6E, 0xE6, 0x58, 0x2D,
    0xAF, 0xB5, 0xB1, 0x15, 0x91, 0x40, 0x5A, 0xE4, 0x74, 0x9B, 0xB6, 0x26, 0xC7, 0xE9, 0xA0, 0x1C,
    0x0D, 0x45, 0x3C, 0xFA, 0x7A, 0x0A, 0x77, 0xB6, 0xC1, 0x2F, 0x32, 0x28, 0xF0, 0xF0, 0xFC, 0x61,
    0xD5, 0xBB, 0x8A, 0xAF, 0xD1, 0x80, 0x32, 0x74, 0x1A, 0xF0
};
static const uint8_t Asn1Der_ECC_N224_PubKey[] =
{
    0x30, 0x4E, 0x30, 0x10, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x05, 0x2B,
    0x81, 0x04, 0x00, 0x21, 0x03, 0x3A, 0x00, 0x04, 0xBC, 0xF8, 0xAE, 0x26, 0x1F, 0x30, 0xA2, 0x8C,
    0x64, 0x76, 0x6E, 0xE6, 0x58, 0x2D, 0xAF, 0xB5, 0xB1, 0x15, 0x91, 0x40, 0x5A, 0xE4, 0x74, 0x9B,
    0xB6, 0x26, 0xC7, 0xE9, 0xA0, 0x1C, 0x0D, 0x45, 0x3C, 0xFA, 0x7A, 0x0A, 0x77, 0xB6, 0xC1, 0x2F,
    0x32, 0x28, 0xF0, 0xF0, 0xFC, 0x61, 0xD5, 0xBB, 0x8A, 0xAF, 0xD1, 0x80, 0x32, 0x74, 0x1A, 0xF0
};

static const uint8_t Asn1Der_ECC_N256_PrvKey[] =
{
    0x30, 0x77, 0x02, 0x01, 0x01, 0x04, 0x20, 0xB1, 0xD9, 0x73, 0xBA, 0xB8, 0x41, 0x36, 0x6E, 0xFF,
    0x19, 0x40, 0x0C, 0xDE, 0xBE, 0xF1, 0x13, 0xB0, 0x94, 0x04, 0xE3, 0xAB, 0xB7, 0xB8, 0x39, 0x98,
    0xF2, 0xCF, 0xAE, 0xFC, 0x1E, 0xBA, 0x34, 0xA0, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
    0x03, 0x01, 0x07, 0xA1, 0x44, 0x03, 0x42, 0x00, 0x04, 0xA5, 0x36, 0x33, 0x24, 0xEE, 0x51, 0x5C,
    0xDE, 0xD7, 0x8D, 0x4D, 0x7A, 0x42, 0xD2, 0x8B, 0xD3, 0x1E, 0x8F, 0xFD, 0xDA, 0xAE, 0x7C, 0x50,
    0x60, 0x37, 0xB2, 0x86, 0x40, 0x18, 0xB2, 0xAD, 0x85, 0xA7, 0xFE, 0xDD, 0x7E, 0x9C, 0x0F, 0xFE,
    0x04, 0xD3, 0x2F, 0x10, 0x19, 0xFE, 0x2E, 0xCD, 0x60, 0x43, 0x9B, 0x84, 0xB6, 0x44, 0xA7, 0x9D,
    0x09, 0xF2, 0x64, 0xFB, 0xD5, 0x4E, 0xCC, 0x58, 0xDB
};
static const uint8_t Asn1Der_ECC_N256_PubKey[] =
{
    0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A,
    0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xA5, 0x36, 0x33, 0x24, 0xEE,
    0x51, 0x5C, 0xDE, 0xD7, 0x8D, 0x4D, 0x7A, 0x42, 0xD2, 0x8B, 0xD3, 0x1E, 0x8F, 0xFD, 0xDA, 0xAE,
    0x7C, 0x50, 0x60, 0x37, 0xB2, 0x86, 0x40, 0x18, 0xB2, 0xAD, 0x85, 0xA7, 0xFE, 0xDD, 0x7E, 0x9C,
    0x0F, 0xFE, 0x04, 0xD3, 0x2F, 0x10, 0x19, 0xFE, 0x2E, 0xCD, 0x60, 0x43, 0x9B, 0x84, 0xB6, 0x44,
    0xA7, 0x9D, 0x09, 0xF2, 0x64, 0xFB, 0xD5, 0x4E, 0xCC, 0x58, 0xDB
};

static const uint8_t Asn1Der_ECC_N384_PrvKey[] =
{
    0x30, 0x81, 0xA4, 0x02, 0x01, 0x01, 0x04, 0x30, 0xF4, 0x58, 0xF7, 0x82, 0xFE, 0x73, 0xF2, 0x9D,
    0xBC, 0xAA, 0xAA, 0x4E, 0xD5, 0x09, 0x29, 0x64, 0xE6, 0x41, 0x16, 0x70, 0xFF, 0x28, 0x8A, 0xC9,
    0xF6, 0x33, 0xC2, 0x03, 0x4E, 0xBC, 0xAD, 0x0A, 0x0F, 0x58, 0xAB, 0x96, 0x86, 0xF7, 0x66, 0x9F,
    0xC4, 0x28, 0xBB, 0xF7, 0xA0, 0xF5, 0x9C, 0x69, 0xA0, 0x07, 0x06, 0x05, 0x2B, 0x81, 0x04, 0x00,
    0x22, 0xA1, 0x64, 0x03, 0x62, 0x00, 0x04, 0x2C, 0x3D, 0x96, 0x17, 0xD2, 0x20, 0x9D, 0xAC, 0xC3,
    0x45, 0x1C, 0xF8, 0x57, 0x6F, 0x7B, 0xF1, 0x97, 0x34, 0x8D, 0x27, 0x7D, 0xAE, 0xA0, 0x8B, 0xDD,
    0xCC, 0x8D, 0xA8, 0xC0, 0x4E, 0xF6, 0x9A, 0x77, 0x51, 0x5F, 0x64, 0xFB, 0xE9, 0x07, 0x6B, 0xD3,
    0x21, 0xD4, 0x6F, 0xA0, 0x10, 0x3C, 0xE1, 0xAF, 0xFE, 0x46, 0xAC, 0xF6, 0xD6, 0x94, 0xF6, 0xF8,
    0x32, 0x27, 0x23, 0xD6, 0x08, 0x19, 0x92, 0x7B, 0xCC, 0x55, 0xDC, 0xCC, 0x25, 0x19, 0x1B, 0xA1,
    0x03, 0xC4, 0x07, 0x6F, 0xAB, 0xE6, 0xA4, 0xDB, 0x5A, 0x44, 0x9E, 0x36, 0x09, 0x22, 0xDA, 0x43,
    0x87, 0x64, 0xE2, 0xBF, 0x3C, 0x28, 0x20
};
static const uint8_t Asn1Der_ECC_N384_PubKey[] =
{
    0x30, 0x76, 0x30, 0x10, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x05, 0x2B,
    0x81, 0x04, 0x00, 0x22, 0x03, 0x62, 0x00, 0x04, 0x2C, 0x3D, 0x96, 0x17, 0xD2, 0x20, 0x9D, 0xAC,
    0xC3, 0x45, 0x1C, 0xF8, 0x57, 0x6F, 0x7B, 0xF1, 0x97, 0x34, 0x8D, 0x27, 0x7D, 0xAE, 0xA0, 0x8B,
    0xDD, 0xCC, 0x8D, 0xA8, 0xC0, 0x4E, 0xF6, 0x9A, 0x77, 0x51, 0x5F, 0x64, 0xFB, 0xE9, 0x07, 0x6B,
    0xD3, 0x21, 0xD4, 0x6F, 0xA0, 0x10, 0x3C, 0xE1, 0xAF, 0xFE, 0x46, 0xAC, 0xF6, 0xD6, 0x94, 0xF6,
    0xF8, 0x32, 0x27, 0x23, 0xD6, 0x08, 0x19, 0x92, 0x7B, 0xCC, 0x55, 0xDC, 0xCC, 0x25, 0x19, 0x1B,
    0xA1, 0x03, 0xC4, 0x07, 0x6F, 0xAB, 0xE6, 0xA4, 0xDB, 0x5A, 0x44, 0x9E, 0x36, 0x09, 0x22, 0xDA,
    0x43, 0x87, 0x64, 0xE2, 0xBF, 0x3C, 0x28, 0x20
};

static const uint8_t Asn1Der_ECC_N521_PrvKey[] =
{
    0x30, 0x81, 0xDC, 0x02, 0x01, 0x01, 0x04, 0x42, 0x01, 0x0E, 0x84, 0x59, 0xC3, 0x78, 0x1D, 0x0E,
    0xA9, 0xF2, 0x87, 0xA8, 0xB7, 0x4B, 0x5A, 0xA7, 0x2B, 0x33, 0xD0, 0x93, 0xE0, 0x91, 0xAC, 0x32,
    0x68, 0xAA, 0x02, 0xFB, 0xF1, 0x30, 0x55, 0xC8, 0x70, 0xAE, 0x09, 0xD9, 0x83, 0x36, 0xCE, 0xD4,
    0x5A, 0xFC, 0xB8, 0xD7, 0x58, 0x16, 0xD6, 0x94, 0x6C, 0x98, 0x60, 0x92, 0xA5, 0xBE, 0x92, 0x37,
    0x74, 0x29, 0xB5, 0x23, 0x3E, 0xF3, 0xE7, 0x8D, 0x2B, 0x77, 0xA0, 0x07, 0x06, 0x05, 0x2B, 0x81,
    0x04, 0x00, 0x23, 0xA1, 0x81, 0x89, 0x03, 0x81, 0x86, 0x00, 0x04, 0x01, 0x9E, 0x9C, 0xC7, 0x09,
    0x6D, 0x38, 0x0C, 0x50, 0x1E, 0x1B, 0x97, 0x3D, 0xCB, 0xC4, 0x78, 0x09, 0xF6, 0xD7, 0x78, 0x47,
    0x14, 0x70, 0x3B, 0x5B, 0x86, 0xB5, 0xD9, 0xDC, 0x9E, 0xAB, 0x57, 0x49, 0xBA, 0x20, 0xE5, 0x9C,
    0x72, 0x37, 0xCC, 0x08, 0xAE, 0x70, 0x6E, 0xCF, 0x5F, 0x85, 0x8A, 0xB1, 0xC7, 0x4B, 0xB5, 0x74,
    0xB1, 0x78, 0x69, 0x99, 0xDB, 0xEE, 0x5E, 0xAF, 0x7B, 0xDF, 0x11, 0x68, 0xF0, 0x01, 0x25, 0x65,
    0x79, 0x4D, 0xAE, 0x62, 0xE6, 0x86, 0xBA, 0xB6, 0x9B, 0xFF, 0xC3, 0xD0, 0xB3, 0xC4, 0x45, 0x7A,
    0x2B, 0xD0, 0x06, 0x24, 0xFF, 0xAE, 0xD2, 0xBA, 0xBB, 0xA8, 0x2A, 0x34, 0xAF, 0x0A, 0xDD, 0x06,
    0xF0, 0x4F, 0x95, 0x19, 0x8E, 0x17, 0xE8, 0x16, 0x39, 0xA5, 0x7B, 0x2B, 0x46, 0xB2, 0x89, 0xBA,
    0x62, 0x4B, 0x3E, 0x3D, 0xCE, 0xE6, 0x64, 0x4E, 0x03, 0x0F, 0x96, 0xE9, 0xD2, 0xE2, 0x9E
};
static const uint8_t Asn1Der_ECC_N521_PubKey[] =
{
    0x30, 0x81, 0x9B, 0x30, 0x10, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x05,
    0x2B, 0x81, 0x04, 0x00, 0x23, 0x03, 0x81, 0x86, 0x00, 0x04, 0x01, 0x9E, 0x9C, 0xC7, 0x09, 0x6D,
    0x38, 0x0C, 0x50, 0x1E, 0x1B, 0x97, 0x3D, 0xCB, 0xC4, 0x78, 0x09, 0xF6, 0xD7, 0x78, 0x47, 0x14,
    0x70, 0x3B, 0x5B, 0x86, 0xB5, 0xD9, 0xDC, 0x9E, 0xAB, 0x57, 0x49, 0xBA, 0x20, 0xE5, 0x9C, 0x72,
    0x37, 0xCC, 0x08, 0xAE, 0x70, 0x6E, 0xCF, 0x5F, 0x85, 0x8A, 0xB1, 0xC7, 0x4B, 0xB5, 0x74, 0xB1,
    0x78, 0x69, 0x99, 0xDB, 0xEE, 0x5E, 0xAF, 0x7B, 0xDF, 0x11, 0x68, 0xF0, 0x01, 0x25, 0x65, 0x79,
    0x4D, 0xAE, 0x62, 0xE6, 0x86, 0xBA, 0xB6, 0x9B, 0xFF, 0xC3, 0xD0, 0xB3, 0xC4, 0x45, 0x7A, 0x2B,
    0xD0, 0x06, 0x24, 0xFF, 0xAE, 0xD2, 0xBA, 0xBB, 0xA8, 0x2A, 0x34, 0xAF, 0x0A, 0xDD, 0x06, 0xF0,
    0x4F, 0x95, 0x19, 0x8E, 0x17, 0xE8, 0x16, 0x39, 0xA5, 0x7B, 0x2B, 0x46, 0xB2, 0x89, 0xBA, 0x62,
    0x4B, 0x3E, 0x3D, 0xCE, 0xE6, 0x64, 0x4E, 0x03, 0x0F, 0x96, 0xE9, 0xD2, 0xE2, 0x9E
};

/* Brainpool */
static const uint8_t Asn1Der_ECC_B224_PrvKey[] =
{
    0x30, 0x6C, 0x02, 0x01, 0x01, 0x04, 0x1C, 0x58, 0x98, 0x6A, 0x06, 0x36, 0x06, 0x1A, 0x04, 0x22,
    0x14, 0x44, 0x46, 0x94, 0x48, 0xA6, 0x4A, 0x5B, 0x01, 0x19, 0x10, 0x36, 0xCB, 0xB6, 0xB0, 0x3D,
    0x77, 0x70, 0x64, 0xA0, 0x0B, 0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x05,
    0xA1, 0x3C, 0x03, 0x3A, 0x00, 0x04, 0xD5, 0x38, 0xE6, 0xA1, 0x93, 0x79, 0xCF, 0x77, 0x9A, 0x54,
    0xD2, 0xE1, 0x9C, 0xFB, 0x6D, 0xB4, 0x6C, 0x8C, 0x77, 0x27, 0x12, 0x64, 0x76, 0xCA, 0x6D, 0x04,
    0xF2, 0x9F, 0xBA, 0xB1, 0x61, 0xB0, 0x7E, 0x2C, 0x50, 0xC1, 0x0B, 0x81, 0x88, 0x7F, 0x63, 0x76,
    0x12, 0x36, 0x51, 0xAC, 0x80, 0x43, 0x4E, 0x3B, 0x45, 0xFA, 0x1B, 0x7E, 0x5A, 0xD8
};
static const uint8_t Asn1Der_ECC_B224_PubKey[] =
{
    0x30, 0x52, 0x30, 0x14, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x09, 0x2B,
    0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x05, 0x03, 0x3A, 0x00, 0x04, 0xD5, 0x38, 0xE6, 0xA1,
    0x93, 0x79, 0xCF, 0x77, 0x9A, 0x54, 0xD2, 0xE1, 0x9C, 0xFB, 0x6D, 0xB4, 0x6C, 0x8C, 0x77, 0x27,
    0x12, 0x64, 0x76, 0xCA, 0x6D, 0x04, 0xF2, 0x9F, 0xBA, 0xB1, 0x61, 0xB0, 0x7E, 0x2C, 0x50, 0xC1,
    0x0B, 0x81, 0x88, 0x7F, 0x63, 0x76, 0x12, 0x36, 0x51, 0xAC, 0x80, 0x43, 0x4E, 0x3B, 0x45, 0xFA,
    0x1B, 0x7E, 0x5A, 0xD8
};

static const uint8_t Asn1Der_ECC_B256_PrvKey[] =
{
    0x30, 0x78, 0x02, 0x01, 0x01, 0x04, 0x20, 0x56, 0xBB, 0x2F, 0xB4, 0x6A, 0xB4, 0x22, 0x63, 0x1B,
    0xD3, 0x4D, 0x94, 0x8B, 0x15, 0x30, 0x8B, 0x85, 0x1C, 0x87, 0x43, 0x4D, 0x59, 0xBD, 0x7F, 0xDA,
    0x6C, 0xEC, 0xFA, 0xD2, 0x82, 0xBB, 0x25, 0xA0, 0x0B, 0x06, 0x09, 0x2B, 0x24, 0x03, 0x03, 0x02,
    0x08, 0x01, 0x01, 0x07, 0xA1, 0x44, 0x03, 0x42, 0x00, 0x04, 0x0C, 0xE8, 0x5A, 0xCB, 0x63, 0x84,
    0x2D, 0xB9, 0x26, 0x98, 0x60, 0xE7, 0x9B, 0x90, 0xB1, 0x8E, 0x75, 0x0A, 0x00, 0x5D, 0x80, 0x6F,
    0x33, 0x6D, 0x13, 0xD3, 0xD0, 0x69, 0xE7, 0x93, 0xCE, 0x23, 0x28, 0x64, 0xEF, 0xB4, 0xA4, 0xEE,
    0x41, 0xEA, 0xFD, 0xBA, 0x8D, 0x52, 0x7A, 0x95, 0x47, 0xB0, 0x99, 0xFB, 0xA5, 0x70, 0x24, 0x41,
    0x17, 0xE3, 0x1F, 0x05, 0x01, 0x42, 0xE2, 0xB2, 0xEC, 0x28
};
static const uint8_t Asn1Der_ECC_B256_PubKey[] =
{
    0x30, 0x5A, 0x30, 0x14, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x09, 0x2B,
    0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x0C, 0xE8, 0x5A, 0xCB,
    0x63, 0x84, 0x2D, 0xB9, 0x26, 0x98, 0x60, 0xE7, 0x9B, 0x90, 0xB1, 0x8E, 0x75, 0x0A, 0x00, 0x5D,
    0x80, 0x6F, 0x33, 0x6D, 0x13, 0xD3, 0xD0, 0x69, 0xE7, 0x93, 0xCE, 0x23, 0x28, 0x64, 0xEF, 0xB4,
    0xA4, 0xEE, 0x41, 0xEA, 0xFD, 0xBA, 0x8D, 0x52, 0x7A, 0x95, 0x47, 0xB0, 0x99, 0xFB, 0xA5, 0x70,
    0x24, 0x41, 0x17, 0xE3, 0x1F, 0x05, 0x01, 0x42, 0xE2, 0xB2, 0xEC, 0x28
};

static const uint8_t Asn1Der_ECC_B384_PrvKey[] =
{
    0x30, 0x81, 0xA8, 0x02, 0x01, 0x01, 0x04, 0x30, 0x09, 0x6A, 0x32, 0x73, 0x8E, 0x26, 0x7C, 0xCA,
    0xA9, 0x57, 0x62, 0xAB, 0x0E, 0x7B, 0x12, 0x81, 0x6A, 0x2A, 0xD1, 0x35, 0xFF, 0xC9, 0x92, 0x99,
    0x56, 0x85, 0x61, 0x2D, 0xCA, 0x7C, 0x78, 0x04, 0x0C, 0xFF, 0x89, 0xF2, 0x3D, 0x54, 0x7E, 0xA6,
    0x3B, 0xD9, 0xB1, 0x09, 0x72, 0x6E, 0x4A, 0x20, 0xA0, 0x0B, 0x06, 0x09, 0x2B, 0x24, 0x03, 0x03,
    0x02, 0x08, 0x01, 0x01, 0x0B, 0xA1, 0x64, 0x03, 0x62, 0x00, 0x04, 0x7A, 0x59, 0x7D, 0xA4, 0x15,
    0x95, 0xF8, 0xCE, 0x32, 0x56, 0x4C, 0x82, 0x72, 0x91, 0x7F, 0xC2, 0xCD, 0xB2, 0x33, 0x86, 0x83,
    0x11, 0xBC, 0xC6, 0x70, 0xE6, 0x0E, 0x1E, 0x18, 0x30, 0x87, 0xA9, 0xD6, 0x83, 0xD4, 0xBC, 0x54,
    0xD9, 0x81, 0xD0, 0x83, 0x0D, 0xFD, 0xEE, 0x6A, 0x2C, 0x53, 0x30, 0x05, 0x2B, 0x72, 0xF8, 0xC9,
    0x5F, 0x11, 0x2B, 0xC8, 0x5E, 0x03, 0x77, 0xF1, 0x6A, 0x3F, 0x0B, 0x07, 0xF8, 0xC3, 0xF4, 0x17,
    0x0D, 0x66, 0x37, 0xF2, 0xD8, 0x78, 0x9D, 0x90, 0xC3, 0x80, 0x00, 0xE9, 0x87, 0xC6, 0x22, 0xA2,
    0x80, 0x6D, 0xC2, 0x02, 0x44, 0x87, 0xC8, 0x0A, 0x89, 0x33, 0x09
};
static const uint8_t Asn1Der_ECC_B384_PubKey[] =
{
    0x30, 0x7A, 0x30, 0x14, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x09, 0x2B,
    0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x0B, 0x03, 0x62, 0x00, 0x04, 0x7A, 0x59, 0x7D, 0xA4,
    0x15, 0x95, 0xF8, 0xCE, 0x32, 0x56, 0x4C, 0x82, 0x72, 0x91, 0x7F, 0xC2, 0xCD, 0xB2, 0x33, 0x86,
    0x83, 0x11, 0xBC, 0xC6, 0x70, 0xE6, 0x0E, 0x1E, 0x18, 0x30, 0x87, 0xA9, 0xD6, 0x83, 0xD4, 0xBC,
    0x54, 0xD9, 0x81, 0xD0, 0x83, 0x0D, 0xFD, 0xEE, 0x6A, 0x2C, 0x53, 0x30, 0x05, 0x2B, 0x72, 0xF8,
    0xC9, 0x5F, 0x11, 0x2B, 0xC8, 0x5E, 0x03, 0x77, 0xF1, 0x6A, 0x3F, 0x0B, 0x07, 0xF8, 0xC3, 0xF4,
    0x17, 0x0D, 0x66, 0x37, 0xF2, 0xD8, 0x78, 0x9D, 0x90, 0xC3, 0x80, 0x00, 0xE9, 0x87, 0xC6, 0x22,
    0xA2, 0x80, 0x6D, 0xC2, 0x02, 0x44, 0x87, 0xC8, 0x0A, 0x89, 0x33, 0x09
};

static const uint8_t Asn1Der_ECC_B512_PrvKey[] =
{
    0x30, 0x81, 0xDA, 0x02, 0x01, 0x01, 0x04, 0x40, 0x65, 0x4E, 0xC3, 0x34, 0x90, 0x57, 0xC6, 0xC1,
    0xB6, 0xF6, 0x56, 0xB7, 0xD0, 0x1A, 0x9D, 0xFB, 0x43, 0xB4, 0x53, 0x21, 0xE7, 0x8F, 0x38, 0xCE,
    0xE0, 0x98, 0x89, 0x16, 0xD4, 0x7B, 0x2A, 0x1F, 0x11, 0x75, 0xFF, 0x62, 0xEE, 0x57, 0x15, 0xF4,
    0xD0, 0x67, 0x1B, 0xD4, 0xAE, 0xD3, 0x3B, 0xBC, 0x58, 0xEA, 0xB5, 0xFC, 0x5B, 0xDC, 0x5D, 0x67,
    0x17, 0xF8, 0x88, 0x21, 0x93, 0x4D, 0xB9, 0xF0, 0xA0, 0x0B, 0x06, 0x09, 0x2B, 0x24, 0x03, 0x03,
    0x02, 0x08, 0x01, 0x01, 0x0D, 0xA1, 0x81, 0x85, 0x03, 0x81, 0x82, 0x00, 0x04, 0x6D, 0x3A, 0x34,
    0xC0, 0xC7, 0x29, 0x88, 0xED, 0xD7, 0xFF, 0x09, 0xFC, 0xAB, 0x91, 0x54, 0x68, 0x6F, 0x41, 0xB3,
    0xAA, 0x98, 0x15, 0xE6, 0xB1, 0xAE, 0x8E, 0x2F, 0x14, 0x7A, 0xAF, 0xF2, 0x56, 0xA9, 0x82, 0x6F,
    0x53, 0x10, 0x22, 0x29, 0x48, 0x6C, 0xA5, 0x3E, 0xB7, 0xD5, 0x66, 0x0F, 0x35, 0xB6, 0x02, 0xEE,
    0x5B, 0xBA, 0x36, 0x2A, 0x89, 0x01, 0xB5, 0xFF, 0x50, 0xD0, 0x0E, 0xF6, 0xB6, 0x39, 0x4F, 0xDA,
    0x1C, 0x86, 0xC6, 0xFF, 0xFE, 0x47, 0x85, 0x99, 0x00, 0x4C, 0x72, 0xEE, 0x9C, 0x4A, 0xBF, 0x4B,
    0x3D, 0x22, 0x03, 0xC1, 0x05, 0x7F, 0x5A, 0x39, 0xC2, 0x28, 0xB7, 0x0E, 0xA9, 0xCE, 0x51, 0x37,
    0x45, 0x54, 0x6C, 0x1F, 0x3C, 0x45, 0x2C, 0x0F, 0xC4, 0x01, 0xB2, 0x24, 0x09, 0x09, 0xD9, 0x08,
    0x25, 0xE3, 0x96, 0x13, 0xEE, 0x46, 0x31, 0x9A, 0x56, 0x3B, 0x38, 0x93, 0x78
};
static const uint8_t Asn1Der_ECC_B512_PubKey[] =
{
    0x30, 0x81, 0x9B, 0x30, 0x14, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x09,
    0x2B, 0x24, 0x03, 0x03, 0x02, 0x08, 0x01, 0x01, 0x0D, 0x03, 0x81, 0x82, 0x00, 0x04, 0x6D, 0x3A,
    0x34, 0xC0, 0xC7, 0x29, 0x88, 0xED, 0xD7, 0xFF, 0x09, 0xFC, 0xAB, 0x91, 0x54, 0x68, 0x6F, 0x41,
    0xB3, 0xAA, 0x98, 0x15, 0xE6, 0xB1, 0xAE, 0x8E, 0x2F, 0x14, 0x7A, 0xAF, 0xF2, 0x56, 0xA9, 0x82,
    0x6F, 0x53, 0x10, 0x22, 0x29, 0x48, 0x6C, 0xA5, 0x3E, 0xB7, 0xD5, 0x66, 0x0F, 0x35, 0xB6, 0x02,
    0xEE, 0x5B, 0xBA, 0x36, 0x2A, 0x89, 0x01, 0xB5, 0xFF, 0x50, 0xD0, 0x0E, 0xF6, 0xB6, 0x39, 0x4F,
    0xDA, 0x1C, 0x86, 0xC6, 0xFF, 0xFE, 0x47, 0x85, 0x99, 0x00, 0x4C, 0x72, 0xEE, 0x9C, 0x4A, 0xBF,
    0x4B, 0x3D, 0x22, 0x03, 0xC1, 0x05, 0x7F, 0x5A, 0x39, 0xC2, 0x28, 0xB7, 0x0E, 0xA9, 0xCE, 0x51,
    0x37, 0x45, 0x54, 0x6C, 0x1F, 0x3C, 0x45, 0x2C, 0x0F, 0xC4, 0x01, 0xB2, 0x24, 0x09, 0x09, 0xD9,
    0x08, 0x25, 0xE3, 0x96, 0x13, 0xEE, 0x46, 0x31, 0x9A, 0x56, 0x3B, 0x38, 0x93, 0x78
};

/* Special public keys for ECDH tests */
static const uint8_t Asn1Der_ECC_N256_PubKey_ECDH[] =
{
    0x30, 0x57, 0x30, 0x11, 0x06, 0x05, 0x2B, 0x81, 0x04, 0x01, 0x0C, 0x06, 0x08, 0x2A, 0x86, 0x48,
    0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xA5, 0x36, 0x33, 0x24, 0xEE, 0x51, 0x5C,
    0xDE, 0xD7, 0x8D, 0x4D, 0x7A, 0x42, 0xD2, 0x8B, 0xD3, 0x1E, 0x8F, 0xFD, 0xDA, 0xAE, 0x7C, 0x50,
    0x60, 0x37, 0xB2, 0x86, 0x40, 0x18, 0xB2, 0xAD, 0x85, 0xA7, 0xFE, 0xDD, 0x7E, 0x9C, 0x0F, 0xFE,
    0x04, 0xD3, 0x2F, 0x10, 0x19, 0xFE, 0x2E, 0xCD, 0x60, 0x43, 0x9B, 0x84, 0xB6, 0x44, 0xA7, 0x9D,
    0x09, 0xF2, 0x64, 0xFB, 0xD5, 0x4E, 0xCC, 0x58, 0xDB
};

static const uint8_t Asn1Der_ECC_B256_PubKey_ECDH[] =
{
    0x30, 0x58, 0x30, 0x12, 0x06, 0x05, 0x2B, 0x81, 0x04, 0x01, 0x0C, 0x06, 0x09, 0x2B, 0x24, 0x03,
    0x03, 0x02, 0x08, 0x01, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x0C, 0xE8, 0x5A, 0xCB, 0x63, 0x84,
    0x2D, 0xB9, 0x26, 0x98, 0x60, 0xE7, 0x9B, 0x90, 0xB1, 0x8E, 0x75, 0x0A, 0x00, 0x5D, 0x80, 0x6F,
    0x33, 0x6D, 0x13, 0xD3, 0xD0, 0x69, 0xE7, 0x93, 0xCE, 0x23, 0x28, 0x64, 0xEF, 0xB4, 0xA4, 0xEE,
    0x41, 0xEA, 0xFD, 0xBA, 0x8D, 0x52, 0x7A, 0x95, 0x47, 0xB0, 0x99, 0xFB, 0xA5, 0x70, 0x24, 0x41,
    0x17, 0xE3, 0x1F, 0x05, 0x01, 0x42, 0xE2, 0xB2, 0xEC, 0x28
};

#ifdef EIP130_ENABLE_ASYM_ALGO_CURVE25519
/* curve25519 */
static const uint8_t Asn1Der_X25519_PrvKey[] =
{
    0x30, 0x2E, 0x02, 0x01, 0x00, 0x30, 0x05, 0x06, 0x03, 0x2B, 0x65, 0x6E, 0x04, 0x22, 0x04, 0x20,
    0x60, 0x0F, 0x10, 0x99, 0xCD, 0x5F, 0x5B, 0x4E, 0xD1, 0x70, 0x45, 0x6F, 0x55, 0x4B, 0x4E, 0xDB,
    0xEF, 0x1F, 0x85, 0xD5, 0x26, 0xC0, 0x76, 0x38, 0x15, 0xA9, 0xDA, 0xD9, 0xE8, 0x3E, 0x2D, 0x76
};
static const uint8_t Asn1Der_X25519_PubKey[] =
{
    0x30, 0x2A, 0x30, 0x05, 0x06, 0x03, 0x2B, 0x65, 0x6E, 0x03, 0x21, 0x00, 0xC6, 0x0D, 0x45, 0xF7,
    0x84, 0x24, 0x11, 0x1E, 0x51, 0x18, 0x34, 0xF3, 0xFE, 0xC6, 0x6B, 0xB5, 0xDB, 0xBE, 0x6D, 0xD4,
    0x0B, 0x6D, 0x9C, 0x79, 0x6B, 0x2E, 0xBF, 0xD5, 0xFD, 0x8F, 0x9D, 0x74
};
#endif

#ifdef EIP130_ENABLE_ASYM_ALGO_EDDSA
/* Edwards-25519 */
static const uint8_t Asn1Der_Ed25519_PrvKey[] =
{
    0x30, 0x2E, 0x02, 0x01, 0x00, 0x30, 0x05, 0x06, 0x03, 0x2B, 0x65, 0x70, 0x04, 0x22, 0x04, 0x20,
    0x20, 0xEE, 0x55, 0x8B, 0x71, 0x62, 0x14, 0x34, 0xF9, 0xC1, 0x5B, 0x72, 0x81, 0x3B, 0x2F, 0x44,
    0x7A, 0xF3, 0x44, 0x40, 0x49, 0x26, 0x43, 0xE4, 0xA1, 0x67, 0xCE, 0xB2, 0x67, 0x6E, 0x4D, 0xBA
};
static const uint8_t Asn1Der_Ed25519_PubKey[] =
{
    0x30, 0x2A, 0x30, 0x05, 0x06, 0x03, 0x2B, 0x65, 0x70, 0x03, 0x21, 0x00, 0xCC, 0xC6, 0xE9, 0x87,
    0x03, 0x01, 0xCA, 0x09, 0x68, 0xD1, 0xAD, 0x29, 0x7F, 0x8E, 0x1E, 0xA4, 0x51, 0xE3, 0x67, 0xAD,
    0xA5, 0xFD, 0xA5, 0x75, 0xB6, 0x6D, 0x4B, 0xF8, 0x76, 0xA0, 0x3F, 0xF7
};
#endif

#ifdef EIP130_ENABLE_ASYM_ALGO_RSA
/* RSASSA-PKCS#1v1.5 */
static const uint8_t Asn1Der_RSASSA_PKCS1v15_PrvKey[] =
{
    0x30, 0x82, 0x04, 0xA3, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01, 0x01, 0x00, 0xE4, 0xA2, 0x16, 0xD2,
    0x8D, 0xEC, 0x9A, 0xBE, 0x24, 0xF8, 0xC8, 0xCE, 0xC0, 0xDB, 0x07, 0x2E, 0x36, 0x1F, 0xD3, 0x4B,
    0xC5, 0xC3, 0x1D, 0x63, 0x12, 0xAD, 0x7A, 0xD6, 0x8B, 0x41, 0xD9, 0x20, 0x45, 0xA9, 0xF4, 0x50,
    0xA5, 0x8C, 0xB7, 0x55, 0xA9, 0xAA, 0x63, 0xD3, 0xA1, 0x6E, 0xCF, 0x58, 0x4C, 0x8F, 0x21, 0xC7,
    0x45, 0xED, 0xAA, 0x51, 0x72, 0xBF, 0x40, 0x36, 0x74, 0x4A, 0xD9, 0xBF, 0xFC, 0xC1, 0x31, 0x0A,
    0x6C, 0x04, 0x76, 0xC8, 0x8A, 0x14, 0xFC, 0x51, 0xDA, 0x69, 0xEA, 0x55, 0x1A, 0x3D, 0x00, 0x98,
    0x66, 0x2E, 0xEF, 0x28, 0x50, 0x7E, 0x84, 0x46, 0x6A, 0x1D, 0x05, 0x1B, 0x44, 0xDB, 0x1F, 0x8F,
    0xB8, 0x27, 0x72, 0xC6, 0x1F, 0xF5, 0x5B, 0xAC, 0x78, 0x3F, 0x42, 0x8F, 0xF5, 0x5A, 0xA2, 0xE6,
    0x76, 0xA3, 0x67, 0xA8, 0xBE, 0x91, 0x50, 0xEE, 0x9E, 0xD0, 0xF3, 0x42, 0x83, 0x4E, 0x05, 0x1B,
    0x8B, 0xF1, 0x92, 0x7E, 0xCE, 0xF8, 0x88, 0xCA, 0xEF, 0x7B, 0x56, 0xD3, 0xCE, 0x70, 0x77, 0x05,
    0x63, 0x16, 0x47, 0x19, 0x16, 0xBF, 0x3E, 0x90, 0xF0, 0xAA, 0x77, 0x85, 0x98, 0x9C, 0x30, 0x70,
    0x30, 0x44, 0xC3, 0xEE, 0x11, 0xAF, 0x59, 0x14, 0xE6, 0x37, 0x90, 0xA3, 0x41, 0x48, 0x55, 0xA2,
    0x8D, 0x7E, 0xE2, 0xF3, 0x78, 0x93, 0xA7, 0xA9, 0xAE, 0x6F, 0x07, 0x73, 0xB6, 0x93, 0x9B, 0x51,
    0xD1, 0x69, 0x28, 0x0C, 0xEA, 0x76, 0x51, 0x87, 0x61, 0xE8, 0x62, 0xAF, 0x1D, 0x03, 0xB3, 0x19,
    0x6F, 0x56, 0xA3, 0xB8, 0x25, 0x77, 0x59, 0xEF, 0xEE, 0x2C, 0xA7, 0x22, 0xFB, 0x70, 0x9D, 0xD6,
    0x89, 0x3F, 0x2E, 0x4C, 0xC7, 0x96, 0x8D, 0xE7, 0xDE, 0xFD, 0xDB, 0x9B, 0x90, 0x0B, 0x01, 0x24,
    0xEC, 0x50, 0xFB, 0x09, 0xD2, 0xAE, 0xE0, 0xB8, 0x5E, 0xB1, 0xB5, 0x21, 0x02, 0x03, 0x01, 0x00,
    0x01, 0x02, 0x82, 0x01, 0x00, 0x38, 0x94, 0xF0, 0xA6, 0xFD, 0x1E, 0xB0, 0x11, 0x1C, 0x9F, 0x41,
    0x3B, 0x01, 0xB9, 0x72, 0xE6, 0x79, 0xD8, 0x98, 0x77, 0x29, 0x23, 0x27, 0x73, 0xAC, 0x5C, 0xA8,
    0x88, 0xCD, 0xFF, 0xE6, 0xD7, 0x38, 0xC6, 0x5D, 0xC3, 0x53, 0xA5, 0x26, 0x7E, 0x19, 0xB6, 0x24,
    0x6C, 0x6C, 0x88, 0x3A, 0xE2, 0x34, 0x58, 0x3B, 0xD8, 0xD0, 0x32, 0x88, 0x33, 0x0D, 0xFD, 0xCC,
    0xCB, 0x6D, 0x2E, 0x55, 0x03, 0x51, 0x6F, 0x12, 0x4E, 0xEC, 0xFA, 0x91, 0xD7, 0x01, 0x6D, 0x3E,
    0x20, 0x97, 0xB8, 0x1C, 0x99, 0x21, 0x1C, 0x35, 0xB8, 0x81, 0x99, 0x5D, 0xF0, 0x91, 0x33, 0x2E,
    0x53, 0x2A, 0xD1, 0xB1, 0xC0, 0xD4, 0xE2, 0x91, 0xC6, 0x04, 0xA2, 0x26, 0xA2, 0x67, 0xA8, 0x56,
    0x1E, 0xB1, 0xC7, 0x9E, 0x16, 0x78, 0x77, 0x0C, 0xD7, 0x92, 0x46, 0xD0, 0x51, 0x8F, 0x8F, 0xF1,
    0x02, 0x8C, 0xE5, 0x53, 0x3E, 0x0B, 0xE0, 0x48, 0x25, 0x8E, 0xE9, 0x3C, 0xD3, 0xB1, 0x8A, 0x68,
    0xF8, 0x26, 0xD4, 0x9A, 0x8D, 0x21, 0x12, 0x87, 0x90, 0x00, 0xC1, 0x97, 0xF9, 0x48, 0xFC, 0x7F,
    0x50, 0x24, 0x23, 0x79, 0x4C, 0xA4, 0x0A, 0x7C, 0x85, 0xD6, 0xB1, 0x1D, 0x37, 0x89, 0x14, 0xFA,
    0xA5, 0x16, 0x15, 0x16, 0x84, 0x1D, 0xF7, 0x3E, 0x0C, 0xA7, 0x19, 0x6A, 0xF2, 0xCF, 0xBF, 0xB4,
    0x7B, 0x12, 0x26, 0xB4, 0xE6, 0xCA, 0x74, 0xD2, 0xBF, 0x05, 0x59, 0xC0, 0x49, 0x8F, 0x00, 0x29,
    0xA2, 0x58, 0xCC, 0xD5, 0x96, 0xB1, 0x65, 0xE1, 0x1E, 0x34, 0xF5, 0x32, 0xA7, 0xB0, 0x73, 0x8B,
    0x46, 0x65, 0xDC, 0x7C, 0x5D, 0xEB, 0x51, 0xCF, 0xD5, 0x28, 0x15, 0xDC, 0xA4, 0x9C, 0x2D, 0x0D,
    0xBE, 0xDE, 0x66, 0x04, 0x39, 0x37, 0xCC, 0x6E, 0xFE, 0x87, 0x7A, 0x60, 0x47, 0xED, 0xBA, 0xCD,
    0x0F, 0xE6, 0x6D, 0x90, 0x1D, 0x02, 0x81, 0x81, 0x00, 0xFB, 0x90, 0x96, 0x7A, 0xF9, 0x29, 0x98,
    0x09, 0x47, 0xC9, 0xCC, 0xB8, 0x6F, 0xE9, 0xED, 0x93, 0xEB, 0xF4, 0x63, 0x53, 0xDB, 0xB5, 0x4B,
    0xC3, 0x7A, 0x23, 0x17, 0x56, 0x51, 0x03, 0x17, 0x60, 0xC8, 0x44, 0x57, 0xF3, 0x97, 0xCD, 0xAE,
    0x88, 0xFF, 0xB3, 0x57, 0x94, 0x7E, 0xCB, 0xBB, 0xAB, 0x63, 0x85, 0x76, 0x4D, 0x2A, 0x91, 0x82,
    0xB9, 0x7E, 0x5E, 0x73, 0x0D, 0x21, 0x9E, 0xC3, 0xF8, 0xA1, 0xC5, 0x51, 0x49, 0xA0, 0x56, 0x16,
    0xBA, 0x44, 0x79, 0x82, 0x83, 0xBC, 0xE8, 0x2A, 0xF9, 0xA9, 0xDB, 0x9F, 0x80, 0x65, 0x38, 0x15,
    0x77, 0x01, 0x5E, 0x4D, 0x63, 0xA6, 0x1B, 0x02, 0x04, 0x2C, 0xAE, 0x89, 0xB4, 0x01, 0xA4, 0x46,
    0x86, 0xC4, 0x33, 0x20, 0xA1, 0x3B, 0x34, 0x5A, 0xDB, 0x23, 0xB2, 0x13, 0x90, 0x8F, 0x55, 0x60,
    0xF3, 0x90, 0x6A, 0x6C, 0xA5, 0xB3, 0xE7, 0xE1, 0xE3, 0x02, 0x81, 0x81, 0x00, 0xE8, 0xAA, 0x00,
    0x71, 0x24, 0x43, 0xDE, 0xEA, 0xCC, 0x0B, 0xDC, 0x1E, 0xE0, 0x47, 0xB9, 0xC0, 0x7F, 0x92, 0x6E,
    0xCA, 0x41, 0xDB, 0x5B, 0xEB, 0x1A, 0x2C, 0xBE, 0xCE, 0xDC, 0x04, 0x8A, 0x95, 0xC6, 0x35, 0x34,
    0xED, 0x9F, 0x00, 0x1F, 0x0F, 0x63, 0xE5, 0x1F, 0x99, 0xBD, 0xD5, 0xEC, 0x16, 0xA7, 0x34, 0xC8,
    0xF4, 0x1B, 0x89, 0x09, 0x59, 0x13, 0x98, 0x97, 0xED, 0x7E, 0xF9, 0x79, 0x84, 0x67, 0x68, 0xCB,
    0x8E, 0xBF, 0x97, 0x67, 0xF4, 0xC2, 0xC8, 0x28, 0xBC, 0xEC, 0xF0, 0xF6, 0xD4, 0x12, 0x27, 0xB4,
    0x93, 0xB8, 0x19, 0x7A, 0xB7, 0x38, 0x17, 0xAA, 0x12, 0xA1, 0xE5, 0xB2, 0xE8, 0xBC, 0x9C, 0x5E,
    0x4D, 0xDF, 0xAD, 0xE5, 0xD3, 0x19, 0xDA, 0xB1, 0x11, 0xBA, 0xCD, 0x4C, 0x0B, 0x9A, 0x14, 0x39,
    0x47, 0xAD, 0x63, 0x46, 0x55, 0x1C, 0x43, 0x8E, 0x98, 0x3D, 0x31, 0x6C, 0x2B, 0x02, 0x81, 0x80,
    0x7B, 0x69, 0x80, 0x59, 0x4D, 0xD8, 0xF8, 0xEC, 0x3C, 0x9A, 0xA4, 0x63, 0x2B, 0x0C, 0x10, 0xF0,
    0x4B, 0x86, 0x36, 0x20, 0x52, 0x9D, 0xCB, 0x50, 0x57, 0x9F, 0xD5, 0x78, 0x14, 0x00, 0x37, 0x3C,
    0xD4, 0xA2, 0xAA, 0x9F, 0x27, 0x2E, 0xD1, 0x5C, 0x06, 0x4D, 0x5E, 0x3E, 0xCC, 0x41, 0x78, 0x8C,
    0xE9, 0x22, 0xF5, 0x59, 0x7F, 0xF3, 0xA8, 0xB5, 0x05, 0x68, 0xA2, 0x7A, 0xA3, 0xAE, 0xB6, 0xA7,
    0x47, 0xFD, 0x7C, 0x16, 0x69, 0xA1, 0x32, 0x3F, 0x29, 0x0E, 0x8F, 0x57, 0xBF, 0x9A, 0x0A, 0x53,
    0x36, 0xE0, 0x36, 0x8C, 0x8F, 0x9B, 0x86, 0xF4, 0x3D, 0xE8, 0x9C, 0x80, 0xB1, 0x61, 0x78, 0x88,
    0xC1, 0xBE, 0xD2, 0x71, 0x4B, 0x88, 0xE2, 0xCC, 0x84, 0x7E, 0x2C, 0x98, 0xCC, 0x69, 0xAF, 0xD4,
    0xE7, 0xDB, 0x19, 0xBC, 0x06, 0x85, 0x35, 0x2D, 0xE8, 0x00, 0x3D, 0xD2, 0xC0, 0xC2, 0x96, 0x4F,
    0x02, 0x81, 0x81, 0x00, 0xE8, 0x00, 0xF5, 0x95, 0xDC, 0xEA, 0xCC, 0xF3, 0xE7, 0x6A, 0x80, 0x22,
    0x4D, 0xD5, 0x58, 0x15, 0xEB, 0x60, 0x52, 0x97, 0xDF, 0x19, 0xE5, 0x4B, 0xC7, 0xFB, 0x0C, 0x7D,
    0xEC, 0x09, 0x9F, 0x44, 0x00, 0xB4, 0xF7, 0xC5, 0x67, 0x82, 0xD0, 0x76, 0x21, 0x4F, 0xCD, 0xB6,
    0x1A, 0x05, 0xA2, 0x79, 0x68, 0x5C, 0x8B, 0x72, 0x57, 0xAE, 0xCB, 0xA2, 0x86, 0x96, 0xE8, 0x10,
    0xCC, 0x47, 0xEA, 0xF3, 0xAB, 0xC6, 0x65, 0x0D, 0x40, 0x61, 0xB3, 0x28, 0xEF, 0xBE, 0x76, 0x59,
    0x06, 0x10, 0xB6, 0xA0, 0xFC, 0x45, 0x99, 0x9C, 0x9F, 0xBC, 0xD0, 0x90, 0xB9, 0x1F, 0x67, 0xD9,
    0x52, 0x95, 0x61, 0x55, 0x11, 0x17, 0x00, 0x36, 0xB9, 0x30, 0x87, 0xF7, 0x7C, 0x55, 0x85, 0xB2,
    0x3F, 0x4B, 0x0E, 0x64, 0x97, 0x4D, 0x49, 0x5C, 0xFF, 0x39, 0xFA, 0x5D, 0x4E, 0x64, 0xD3, 0x61,
    0x5D, 0xCB, 0xD5, 0xA7, 0x02, 0x81, 0x80, 0x4B, 0x37, 0xCB, 0xC3, 0xF2, 0x7C, 0x06, 0x0A, 0xBE,
    0x23, 0xDB, 0xEA, 0x1F, 0xC4, 0xBE, 0x8D, 0xBA, 0xDE, 0xD2, 0xB3, 0x5A, 0xF8, 0x1C, 0xF3, 0xC1,
    0x0F, 0xA7, 0x15, 0x55, 0x8B, 0x18, 0x86, 0x84, 0x47, 0x92, 0xA1, 0x33, 0xB4, 0x28, 0x61, 0x04,
    0xDE, 0xF3, 0x17, 0xE5, 0x91, 0x35, 0xCC, 0x60, 0x66, 0xC8, 0xE8, 0x4F, 0x50, 0x9C, 0xCE, 0x48,
    0x7E, 0x5E, 0x0E, 0x02, 0xF0, 0xE9, 0xEE, 0x63, 0xEF, 0xBA, 0xE9, 0xDB, 0x08, 0x9B, 0x68, 0x48,
    0xA4, 0xD3, 0x4D, 0x45, 0xF5, 0x17, 0x22, 0x52, 0xB6, 0x4D, 0x50, 0x86, 0xF7, 0x71, 0x2A, 0xA9,
    0xE1, 0x38, 0x6B, 0x1B, 0x10, 0x5D, 0xCD, 0x56, 0x05, 0xDE, 0x77, 0x04, 0x36, 0x91, 0xAA, 0x05,
    0x95, 0xC9, 0x7C, 0x3D, 0xB0, 0x1B, 0x01, 0x57, 0x8C, 0x4B, 0xC4, 0x17, 0xD5, 0xDA, 0x38, 0x10,
    0x6D, 0x40, 0xC4, 0x88, 0x1D, 0x53, 0xCF
};
static const uint8_t Asn1Der_RSASSA_PKCS1v15_PubKey[] =
{
    0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01,
    0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01,
    0x00, 0xE4, 0xA2, 0x16, 0xD2, 0x8D, 0xEC, 0x9A, 0xBE, 0x24, 0xF8, 0xC8, 0xCE, 0xC0, 0xDB, 0x07,
    0x2E, 0x36, 0x1F, 0xD3, 0x4B, 0xC5, 0xC3, 0x1D, 0x63, 0x12, 0xAD, 0x7A, 0xD6, 0x8B, 0x41, 0xD9,
    0x20, 0x45, 0xA9, 0xF4, 0x50, 0xA5, 0x8C, 0xB7, 0x55, 0xA9, 0xAA, 0x63, 0xD3, 0xA1, 0x6E, 0xCF,
    0x58, 0x4C, 0x8F, 0x21, 0xC7, 0x45, 0xED, 0xAA, 0x51, 0x72, 0xBF, 0x40, 0x36, 0x74, 0x4A, 0xD9,
    0xBF, 0xFC, 0xC1, 0x31, 0x0A, 0x6C, 0x04, 0x76, 0xC8, 0x8A, 0x14, 0xFC, 0x51, 0xDA, 0x69, 0xEA,
    0x55, 0x1A, 0x3D, 0x00, 0x98, 0x66, 0x2E, 0xEF, 0x28, 0x50, 0x7E, 0x84, 0x46, 0x6A, 0x1D, 0x05,
    0x1B, 0x44, 0xDB, 0x1F, 0x8F, 0xB8, 0x27, 0x72, 0xC6, 0x1F, 0xF5, 0x5B, 0xAC, 0x78, 0x3F, 0x42,
    0x8F, 0xF5, 0x5A, 0xA2, 0xE6, 0x76, 0xA3, 0x67, 0xA8, 0xBE, 0x91, 0x50, 0xEE, 0x9E, 0xD0, 0xF3,
    0x42, 0x83, 0x4E, 0x05, 0x1B, 0x8B, 0xF1, 0x92, 0x7E, 0xCE, 0xF8, 0x88, 0xCA, 0xEF, 0x7B, 0x56,
    0xD3, 0xCE, 0x70, 0x77, 0x05, 0x63, 0x16, 0x47, 0x19, 0x16, 0xBF, 0x3E, 0x90, 0xF0, 0xAA, 0x77,
    0x85, 0x98, 0x9C, 0x30, 0x70, 0x30, 0x44, 0xC3, 0xEE, 0x11, 0xAF, 0x59, 0x14, 0xE6, 0x37, 0x90,
    0xA3, 0x41, 0x48, 0x55, 0xA2, 0x8D, 0x7E, 0xE2, 0xF3, 0x78, 0x93, 0xA7, 0xA9, 0xAE, 0x6F, 0x07,
    0x73, 0xB6, 0x93, 0x9B, 0x51, 0xD1, 0x69, 0x28, 0x0C, 0xEA, 0x76, 0x51, 0x87, 0x61, 0xE8, 0x62,
    0xAF, 0x1D, 0x03, 0xB3, 0x19, 0x6F, 0x56, 0xA3, 0xB8, 0x25, 0x77, 0x59, 0xEF, 0xEE, 0x2C, 0xA7,
    0x22, 0xFB, 0x70, 0x9D, 0xD6, 0x89, 0x3F, 0x2E, 0x4C, 0xC7, 0x96, 0x8D, 0xE7, 0xDE, 0xFD, 0xDB,
    0x9B, 0x90, 0x0B, 0x01, 0x24, 0xEC, 0x50, 0xFB, 0x09, 0xD2, 0xAE, 0xE0, 0xB8, 0x5E, 0xB1, 0xB5,
    0x21, 0x02, 0x03, 0x01, 0x00, 0x01
};

/* RSASSA-PSS */
static const uint8_t Asn1Der_RSASSA_PSS_PrvKey[] =
{
    0x30, 0x82, 0x04, 0xBC, 0x02, 0x01, 0x00, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
    0x0D, 0x01, 0x01, 0x0A, 0x04, 0x82, 0x04, 0xA8, 0x30, 0x82, 0x04, 0xA4, 0x02, 0x01, 0x00, 0x02,
    0x82, 0x01, 0x01, 0x00, 0xAF, 0x28, 0xAC, 0x53, 0xFD, 0xD4, 0x2F, 0x21, 0x0F, 0xCE, 0x77, 0x2B,
    0x4D, 0x6D, 0x9C, 0x42, 0x1B, 0xBF, 0x2A, 0x2D, 0x0C, 0xB3, 0x38, 0x76, 0x3A, 0xF2, 0x72, 0xE0,
    0x85, 0x95, 0xE0, 0x4D, 0x3D, 0x31, 0xDE, 0x5B, 0x6D, 0x7F, 0xEB, 0x64, 0x0F, 0x32, 0x4A, 0x22,
    0x55, 0x49, 0xFA, 0x1C, 0x65, 0x3F, 0x43, 0x60, 0xE5, 0xD8, 0x20, 0xE8, 0x80, 0x3A, 0x52, 0x99,
    0xB7, 0x62, 0xE9, 0x3A, 0x2A, 0xEA, 0x8E, 0x18, 0xF8, 0xC0, 0x9D, 0x0D, 0x75, 0xA7, 0xC3, 0x94,
    0x48, 0x02, 0xC6, 0x08, 0xB8, 0xC0, 0xA2, 0x1D, 0xE0, 0xB7, 0x32, 0x51, 0x03, 0x50, 0x39, 0x16,
    0x83, 0x33, 0xDF, 0xC6, 0x28, 0x88, 0x39, 0xDD, 0x26, 0xE2, 0xAB, 0xF6, 0xFD, 0x92, 0xA8, 0xC4,
    0xA9, 0x75, 0x41, 0x61, 0x39, 0xB0, 0x66, 0x01, 0xD9, 0xFF, 0xB5, 0x8D, 0x2D, 0xF0, 0x97, 0x5C,
    0x48, 0x92, 0x87, 0xA2, 0x17, 0x84, 0x9A, 0xEA, 0x8A, 0x52, 0x4F, 0x0F, 0x78, 0x67, 0x3C, 0x5F,
    0xB2, 0xC5, 0x6A, 0x8F, 0xD6, 0x3A, 0x59, 0x3B, 0x7F, 0x4D, 0xF4, 0x3D, 0x75, 0xDE, 0x74, 0x97,
    0xD9, 0x0F, 0x85, 0x47, 0x40, 0xBA, 0x8A, 0x5F, 0x90, 0xCB, 0xE8, 0xDA, 0x49, 0xD0, 0xEB, 0xE6,
    0xD7, 0x02, 0xFC, 0x2F, 0xFF, 0xCE, 0x34, 0x1C, 0xE1, 0x82, 0x28, 0xAE, 0xB3, 0x38, 0xA7, 0xE6,
    0xE2, 0xF0, 0x84, 0xC7, 0x42, 0xF4, 0x9B, 0xBF, 0x60, 0x14, 0x61, 0x27, 0x35, 0xA8, 0x1C, 0x34,
    0x45, 0x71, 0x1E, 0xF8, 0x94, 0xDD, 0x60, 0xEC, 0x0E, 0xBB, 0x63, 0xDA, 0xFD, 0x93, 0x1A, 0xD3,
    0x6E, 0xDA, 0x3D, 0x5E, 0x0D, 0x69, 0xAC, 0xDE, 0xB1, 0xAB, 0x82, 0x6D, 0x83, 0xA5, 0xCC, 0x73,
    0xD2, 0xFA, 0x5F, 0x6B, 0x3A, 0x4D, 0xB6, 0x99, 0x26, 0xA7, 0xC2, 0x8D, 0x8E, 0x80, 0x58, 0x79,
    0x7D, 0x4E, 0xBF, 0xD1, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02, 0x82, 0x01, 0x01, 0x00, 0xAC, 0x04,
    0xBB, 0xC8, 0xF8, 0xDD, 0x2B, 0x33, 0xC2, 0xFC, 0x37, 0x58, 0x95, 0x7F, 0xF2, 0x17, 0x92, 0x7C,
    0x59, 0x05, 0x49, 0x15, 0xB1, 0x7B, 0x9B, 0x44, 0x83, 0xC1, 0x30, 0x30, 0x12, 0xCA, 0x88, 0x30,
    0xB3, 0x79, 0x02, 0xBE, 0xC0, 0x03, 0xD2, 0x50, 0xC7, 0x31, 0x94, 0xA5, 0x44, 0x36, 0x22, 0xD0,
    0xD0, 0x71, 0x17, 0x08, 0xFC, 0xB8, 0x99, 0x45, 0xAD, 0xDD, 0x3C, 0x96, 0xAD, 0x09, 0xED, 0x1F,
    0xD1, 0x51, 0x80, 0xEA, 0x58, 0xD2, 0xE1, 0x1B, 0x53, 0x1E, 0x0A, 0xE2, 0x7E, 0x6E, 0x0E, 0x53,
    0x50, 0x63, 0x1B, 0x4F, 0x8E, 0xE0, 0xB4, 0xEA, 0x52, 0x46, 0x63, 0xAF, 0xCC, 0x32, 0x25, 0x98,
    0x33, 0x8B, 0x97, 0x26, 0xC1, 0x4F, 0x9D, 0x94, 0x2C, 0x9E, 0x86, 0xF5, 0x9A, 0xB1, 0xCB, 0x20,
    0x0B, 0xEE, 0x01, 0x93, 0xF9, 0x4E, 0x9F, 0x31, 0x4A, 0xD4, 0xDC, 0x88, 0xDB, 0x1D, 0x99, 0x77,
    0x6A, 0x08, 0xF2, 0xD5, 0x11, 0x79, 0x52, 0xB3, 0x53, 0x07, 0xAA, 0xE6, 0x7A, 0x3D, 0xEE, 0xEB,
    0x37, 0x06, 0x41, 0x7D, 0x54, 0x87, 0x04, 0x62, 0x17, 0xFD, 0xFE, 0x09, 0x9A, 0x59, 0xF4, 0xB5,
    0x6E, 0x64, 0xF9, 0x4F, 0x83, 0x16, 0x50, 0x42, 0x3D, 0x96, 0xD8, 0xF4, 0x5A, 0x10, 0x2F, 0x1C,
    0x96, 0x49, 0x6A, 0xED, 0x79, 0x80, 0x1D, 0xF9, 0xCB, 0xCF, 0x51, 0x68, 0xB5, 0x72, 0x52, 0x93,
    0x13, 0xA6, 0x48, 0x72, 0x35, 0x71, 0x47, 0x3F, 0x30, 0xBC, 0xFE, 0x76, 0x01, 0xC9, 0x0E, 0x74,
    0x9F, 0xA3, 0xDC, 0x5E, 0x02, 0x20, 0xE5, 0x33, 0x2E, 0xA1, 0x6D, 0x7A, 0xBF, 0xA1, 0xEC, 0x95,
    0x16, 0x37, 0x51, 0xAA, 0xEA, 0x73, 0x88, 0x9C, 0xA7, 0x94, 0x39, 0xC4, 0x1F, 0xB5, 0x92, 0xD6,
    0x13, 0x97, 0x2B, 0x2B, 0x62, 0x32, 0x29, 0x1C, 0x75, 0xBA, 0x1D, 0xC0, 0x15, 0xE9, 0x02, 0x81,
    0x81, 0x00, 0xE7, 0x95, 0x5C, 0xD4, 0x5E, 0xB7, 0x9E, 0x91, 0x11, 0x15, 0x69, 0x49, 0x84, 0xD5,
    0xD4, 0xAA, 0xE8, 0x62, 0xB2, 0x9A, 0xCC, 0xBF, 0x3F, 0x29, 0xD8, 0x7B, 0xA6, 0xB3, 0xCF, 0x4A,
    0x06, 0x6B, 0x60, 0x13, 0x5B, 0x44, 0x3A, 0x8E, 0x0B, 0x91, 0xD3, 0x76, 0xC7, 0xA6, 0xC6, 0xCD,
    0x06, 0x7D, 0x01, 0x1A, 0x52, 0xA8, 0xF7, 0xBB, 0x1E, 0x0F, 0x9C, 0xB0, 0x30, 0xE7, 0x0D, 0x8F,
    0x8B, 0x8C, 0x8D, 0xCC, 0x2A, 0x84, 0xD9, 0x18, 0x8D, 0x51, 0xEF, 0x6C, 0x9F, 0xC0, 0x6E, 0xB6,
    0xBF, 0xEF, 0x7C, 0x19, 0x0F, 0x64, 0x05, 0x05, 0xA5, 0x23, 0x76, 0x72, 0x52, 0x99, 0x74, 0x44,
    0x28, 0x73, 0x52, 0x0C, 0x97, 0x8C, 0xB0, 0x74, 0xFC, 0xBD, 0x80, 0x26, 0x0F, 0x79, 0x70, 0x0A,
    0xEC, 0x51, 0xE9, 0x24, 0x66, 0xD7, 0x99, 0xA6, 0x74, 0xF4, 0x42, 0x13, 0x95, 0xCC, 0xBD, 0x92,
    0xBF, 0x7F, 0x02, 0x81, 0x81, 0x00, 0xC1, 0xA0, 0x5C, 0xD8, 0xDF, 0xA2, 0x46, 0xFC, 0x27, 0xD6,
    0x0B, 0xB6, 0xE0, 0xF6, 0x72, 0x36, 0xE4, 0x8F, 0xF1, 0xD8, 0x2A, 0x6A, 0xCA, 0x58, 0xCF, 0xCE,
    0xED, 0x5C, 0x45, 0x1E, 0x0E, 0x9B, 0x8A, 0x8C, 0xEF, 0xA1, 0xD7, 0x5D, 0x88, 0x1C, 0x09, 0x45,
    0x59, 0x0B, 0x14, 0xB1, 0x50, 0x60, 0x4C, 0x48, 0xE5, 0x6E, 0x26, 0x20, 0xEA, 0x59, 0x7F, 0xED,
    0xE4, 0x5A, 0x34, 0x99, 0xA0, 0x2D, 0x49, 0x97, 0xC1, 0x7B, 0x33, 0x7A, 0x62, 0x53, 0x3F, 0x3E,
    0xFA, 0x7B, 0x9F, 0x38, 0xD2, 0xB0, 0x69, 0xD4, 0x63, 0x20, 0x3D, 0xB2, 0x48, 0x57, 0xC4, 0xA6,
    0xD9, 0x37, 0x48, 0x1B, 0x69, 0x3E, 0xC8, 0x9A, 0xCB, 0x87, 0x31, 0x3E, 0x8E, 0x69, 0x93, 0x69,
    0x86, 0xED, 0xA3, 0x1F, 0x75, 0xCB, 0xFE, 0x02, 0x54, 0x90, 0x5A, 0xA1, 0xBD, 0xFF, 0x34, 0x74,
    0x36, 0x01, 0x10, 0xEE, 0x28, 0xAF, 0x02, 0x81, 0x81, 0x00, 0xE0, 0xBD, 0x22, 0x89, 0x36, 0x22,
    0x1B, 0xCF, 0xCC, 0x34, 0x6B, 0x28, 0xDD, 0xB0, 0x65, 0xE2, 0x36, 0x03, 0x12, 0x31, 0x07, 0xE1,
    0xD0, 0xFA, 0x5B, 0x0C, 0x24, 0x8A, 0xFA, 0xEF, 0x6B, 0xAB, 0x68, 0x61, 0x60, 0x80, 0xA6, 0x12,
    0x93, 0x47, 0xC1, 0x5F, 0xD4, 0x01, 0x15, 0x8C, 0x82, 0x6F, 0x6D, 0x52, 0x8E, 0x45, 0x4B, 0xA4,
    0x5B, 0x57, 0xEB, 0x47, 0x51, 0x2D, 0xE9, 0x0B, 0x14, 0xF2, 0x21, 0xE9, 0xB0, 0x99, 0x9B, 0xD5,
    0x27, 0xD0, 0x0B, 0xF9, 0xEE, 0x4E, 0x61, 0x23, 0x18, 0xC0, 0xE0, 0x3D, 0xB1, 0xA1, 0xEE, 0xFD,
    0xBB, 0x74, 0xDE, 0xCD, 0xB3, 0x3B, 0x18, 0x89, 0xCC, 0x77, 0x9D, 0x78, 0x7A, 0xDA, 0x1D, 0xA4,
    0x6A, 0x61, 0xBC, 0xAB, 0x32, 0xF9, 0x36, 0xB2, 0x63, 0x04, 0x5A, 0xAC, 0xE8, 0xE0, 0xF2, 0x4B,
    0x87, 0xD4, 0x43, 0x0B, 0x9E, 0xA4, 0x42, 0x94, 0xD7, 0x21, 0x02, 0x81, 0x80, 0x51, 0xAA, 0xCB,
    0xBE, 0x85, 0x84, 0xB2, 0xA0, 0x04, 0x45, 0x59, 0xA7, 0x86, 0xF7, 0x2B, 0x71, 0x4F, 0xB2, 0x0E,
    0xF5, 0x77, 0xDE, 0x96, 0xFE, 0x22, 0x98, 0x4A, 0xB1, 0x12, 0x4D, 0xFD, 0x27, 0x49, 0xA6, 0xEC,
    0x23, 0xBC, 0xEC, 0xAC, 0x4D, 0xF6, 0xF3, 0x12, 0xFD, 0x83, 0x7A, 0xCE, 0x45, 0xEF, 0xA7, 0x07,
    0x6E, 0x10, 0x12, 0xEE, 0x05, 0x55, 0xDA, 0x06, 0xE8, 0xAD, 0x2A, 0xE6, 0xCE, 0x73, 0x2D, 0xD8,
    0xE3, 0x97, 0xF0, 0xD4, 0xAB, 0xFC, 0x0F, 0x1A, 0x23, 0x2E, 0x23, 0x0F, 0x6A, 0x6D, 0x18, 0xB1,
    0xBE, 0x95, 0x87, 0x2E, 0xD0, 0x26, 0x17, 0xA1, 0xA2, 0x99, 0x7D, 0x46, 0x44, 0xBE, 0xE3, 0xDB,
    0x58, 0x68, 0x42, 0xD4, 0x5A, 0x5F, 0x5C, 0xB1, 0x16, 0x66, 0x29, 0x0C, 0xCC, 0x3F, 0x94, 0xC8,
    0x0A, 0x22, 0xCF, 0xF9, 0x8D, 0x3C, 0x98, 0x8F, 0x76, 0x9F, 0xA7, 0x73, 0xED, 0x02, 0x81, 0x80,
    0x27, 0x4D, 0xAC, 0x00, 0xC1, 0xE4, 0x08, 0xFC, 0xCC, 0xCE, 0x75, 0x43, 0x46, 0x35, 0xC2, 0xA2,
    0x6B, 0x73, 0xD9, 0x80, 0x5D, 0x12, 0xD5, 0x90, 0xF8, 0x32, 0xEF, 0x93, 0x76, 0xCA, 0x10, 0x6E,
    0xC7, 0x56, 0x94, 0x33, 0x93, 0xE8, 0x6F, 0x1B, 0x05, 0x49, 0x72, 0xC1, 0x0E, 0xA9, 0x7A, 0x59,
    0xBE, 0xCB, 0x73, 0x12, 0xBC, 0x51, 0xB6, 0xC3, 0x3B, 0xDB, 0x54, 0xB3, 0x61, 0x5E, 0x49, 0x9C,
    0x8F, 0x8E, 0xBB, 0xB1, 0x6B, 0x52, 0x10, 0x22, 0x44, 0xCB, 0xF2, 0xE7, 0x82, 0x43, 0xAB, 0x5B,
    0x5D, 0x75, 0x09, 0xDF, 0x7B, 0xD1, 0xA8, 0xAC, 0x07, 0xE4, 0xB6, 0x57, 0x02, 0x47, 0xCD, 0x5E,
    0x67, 0x95, 0x7B, 0xEB, 0x3C, 0xAD, 0x57, 0x7A, 0x56, 0xBE, 0xEC, 0xED, 0xEE, 0x1B, 0xCC, 0xDF,
    0xA9, 0xF5, 0x5A, 0xF2, 0xEE, 0xA8, 0xAB, 0x74, 0x40, 0x40, 0x37, 0xE7, 0x15, 0xD7, 0x7B, 0x30
};
static const uint8_t Asn1Der_RSASSA_PSS_PubKey[] =
{
    0x30, 0x82, 0x01, 0x20, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01,
    0x0A, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xAF,
    0x28, 0xAC, 0x53, 0xFD, 0xD4, 0x2F, 0x21, 0x0F, 0xCE, 0x77, 0x2B, 0x4D, 0x6D, 0x9C, 0x42, 0x1B,
    0xBF, 0x2A, 0x2D, 0x0C, 0xB3, 0x38, 0x76, 0x3A, 0xF2, 0x72, 0xE0, 0x85, 0x95, 0xE0, 0x4D, 0x3D,
    0x31, 0xDE, 0x5B, 0x6D, 0x7F, 0xEB, 0x64, 0x0F, 0x32, 0x4A, 0x22, 0x55, 0x49, 0xFA, 0x1C, 0x65,
    0x3F, 0x43, 0x60, 0xE5, 0xD8, 0x20, 0xE8, 0x80, 0x3A, 0x52, 0x99, 0xB7, 0x62, 0xE9, 0x3A, 0x2A,
    0xEA, 0x8E, 0x18, 0xF8, 0xC0, 0x9D, 0x0D, 0x75, 0xA7, 0xC3, 0x94, 0x48, 0x02, 0xC6, 0x08, 0xB8,
    0xC0, 0xA2, 0x1D, 0xE0, 0xB7, 0x32, 0x51, 0x03, 0x50, 0x39, 0x16, 0x83, 0x33, 0xDF, 0xC6, 0x28,
    0x88, 0x39, 0xDD, 0x26, 0xE2, 0xAB, 0xF6, 0xFD, 0x92, 0xA8, 0xC4, 0xA9, 0x75, 0x41, 0x61, 0x39,
    0xB0, 0x66, 0x01, 0xD9, 0xFF, 0xB5, 0x8D, 0x2D, 0xF0, 0x97, 0x5C, 0x48, 0x92, 0x87, 0xA2, 0x17,
    0x84, 0x9A, 0xEA, 0x8A, 0x52, 0x4F, 0x0F, 0x78, 0x67, 0x3C, 0x5F, 0xB2, 0xC5, 0x6A, 0x8F, 0xD6,
    0x3A, 0x59, 0x3B, 0x7F, 0x4D, 0xF4, 0x3D, 0x75, 0xDE, 0x74, 0x97, 0xD9, 0x0F, 0x85, 0x47, 0x40,
    0xBA, 0x8A, 0x5F, 0x90, 0xCB, 0xE8, 0xDA, 0x49, 0xD0, 0xEB, 0xE6, 0xD7, 0x02, 0xFC, 0x2F, 0xFF,
    0xCE, 0x34, 0x1C, 0xE1, 0x82, 0x28, 0xAE, 0xB3, 0x38, 0xA7, 0xE6, 0xE2, 0xF0, 0x84, 0xC7, 0x42,
    0xF4, 0x9B, 0xBF, 0x60, 0x14, 0x61, 0x27, 0x35, 0xA8, 0x1C, 0x34, 0x45, 0x71, 0x1E, 0xF8, 0x94,
    0xDD, 0x60, 0xEC, 0x0E, 0xBB, 0x63, 0xDA, 0xFD, 0x93, 0x1A, 0xD3, 0x6E, 0xDA, 0x3D, 0x5E, 0x0D,
    0x69, 0xAC, 0xDE, 0xB1, 0xAB, 0x82, 0x6D, 0x83, 0xA5, 0xCC, 0x73, 0xD2, 0xFA, 0x5F, 0x6B, 0x3A,
    0x4D, 0xB6, 0x99, 0x26, 0xA7, 0xC2, 0x8D, 0x8E, 0x80, 0x58, 0x79, 0x7D, 0x4E, 0xBF, 0xD1, 0x02,
    0x03, 0x01, 0x00, 0x01
};
#endif


/* end of file testvectors_asn1_der_data.h */
