/******************************************************************************
 Group: CMCU LPRF
 Target Device: cc23xx

 ******************************************************************************
 
 Copyright (c) 2024-2025, Texas Instruments Incorporated
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 *  Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

 *  Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

 *  Neither the name of Texas Instruments Incorporated nor the names of
    its contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ******************************************************************************
 
 
 *****************************************************************************/

#ifndef JDLLC_H
#define JDLLC_H 1

#include <ti/drivers/dpl/ClockP.h>

// In order to have the zb_uint types defined, include zb_types.h (correctly done by including zboss_api.h or zb_common.h; zb_common.h will already include zboss_api.h)
#include "zb_common.h"
#include "translate.h"
#include "mac_disassociate.h"

/*! sensor minimum poll time = 10 ms */
#define SENSOR_MIN_POLL_TIME        10

/*! Event ID - Process successful association */
#define JDLLC_JOIN_EVT           0x0004
/*! Event ID - State change event */
#define JDLLC_STATE_CHANGE_EVT   0x0008
/*! Event ID - Poll event */
#define JDLLC_POLL_EVT           0x0010
/*! Event ID - Associate Request */
#define JDLLC_ASSOCIATE_REQ_EVT  0x0020
/*! Event ID - Process coordinator realignment */
#define JDLLC_COORD_REALIGN      0x0040
/*! Event ID - Scan backoff , silent period before resuming scanning */
#define JDLLC_SCAN_BACKOFF       0x0100

/*!
 Joining Device State Values (changed via updateState)
 */
typedef enum
{
    /*! Powered up, not started and waiting for user to start */
    Jdllc_states_initWaiting,
    /*! Starting device: scanning and selecting the best network to join */
    Jdllc_states_joining,
    /*! Powered up, found network information, and restoring device in network*/
    Jdllc_states_initRestoring,
    /*! Device is operating in network */
    Jdllc_states_joined,
    /*! Device is restored as device in the network */
    Jdllc_states_rejoined,
    /*! Device is orphaned */
    Jdllc_states_orphan,
    /*! Device join access denied or PAN at capacity */
    Jdllc_states_accessDenied
} Jdllc_states_t;

/*!
 Joining Device starting State Values (changed via switchState)
 */
typedef enum
{
    /*! Joining Device is performing a active scan  */
    Jdllc_deviceStates_scanActive,
    /*! Joining Device is performing a passive scan  */
    Jdllc_deviceStates_scanPassive,
    /*! Sync request for beacon enabled device*/
    Jdllc_deviceStates_syncReq,
    /*! Orphan scan */
    Jdllc_deviceStates_scanOrphan,
    /*! Scan backoff state */
    Jdllc_deviceStates_scanBackoff,
    /*! Device stops joining as PAN reaches Capacity */
    Jdllc_deviceStates_accessDenied
} Jdllc_device_states_t;

/*! Device Descriptor (creating this because MAC only has zb_addr_u and the ZBOSS equivalent is an APS bind table entry, but we aren't using the APS layer, only the MAC) */
typedef struct zb_device_descriptor_s
{
    /*! The 16-bit PAN identifier of the device */
    zb_uint16_t panID;
    /*! The 16-bit short address of the device */
    zb_uint16_t shortAddress;
    /*!
     The 64-bit IEEE extended address of the device. This element is also
     used inunsecuring operations on incoming frames.
     */
    zb_ieee_addr_t extAddress;
} zb_device_descriptor_t;

/*! Network Information */
typedef struct
{
    /* Address information */
    zb_device_descriptor_t devInfo;
    /*! Channel - non FH */
    zb_uint8_t channel;
} Llc_netInfo_t;

extern zb_uint16_t Jdllc_events;

/******************************************************************************
 Structures - Building blocks for the JDLLC
 *****************************************************************************/

/*!
 Device joined callback - passes the device information and its parent's
  information to app
 */
typedef void (*Jdllc_networkJoinedFp_t)(zb_device_descriptor_t *pDevInfo, Llc_netInfo_t  *pParentInfo);

/*!
  Disassociation indication callback - passes reason for device disassociation
  or parent disassociating from device to app
 */
typedef void (*Jdllc_networkDisassocIndFp_t)(zb_ieee_addr_t *extAddress, zb_disassociate_reason_t reason);

/*!
  Disassociation confirm callback - passes status of device's disassociation
  request to app
 */
typedef void (*Jdllc_networkDisassocCnfFp_t)(zb_ieee_addr_t *extAddress, zb_mac_status_t status);

/*!
 State Changed callback - This indication says that the JDLLC's state
 has changed.
 */
typedef void (*Jdllc_stateChangedFp_t)(Jdllc_states_t state);

typedef struct _Jdllc_callbacks_t
{
    /*! Network Joined Indication callback */
    Jdllc_networkJoinedFp_t pJoinedCb;
    /* Disassociation Indication callback */
    Jdllc_networkDisassocIndFp_t pDisassocIndCb;
    /*! Disassociation Confirm callback */
    Jdllc_networkDisassocCnfFp_t pDisassocCnfCb;
    /*! State Changed indication callback */
    Jdllc_stateChangedFp_t pStateChangeCb;
} Jdllc_callbacks_t;

/******************************************************************************
 Function Prototypes
 *****************************************************************************/

/*!
 * @brief       Initialize this module and update the callback structure.
 *              <BR>
 *              This function will take the application's MAC callback
 *              structure and override some callbacks with its own callbacks.
 *              The application callbacks that are overridden are saved so that
 *              they can be chain called when processing the callbacks.
 *
 * @param       pMacCbs - pointer to the API MAC callback structure
 * @param       pJdllcCbs - pointer to the JDLLC callback structure
 */
extern void Jdllc_init(zb_mac_callbacks_t *pMacCbs,
                   Jdllc_callbacks_t *pJdllcCbs);

/*!
 * @brief Application task processing.
 */
extern void Jdllc_process(void);

/*!
 * @brief       set the PAN ID to join.
 *
 * @param       panId - PAN ID to set
 */
extern void Jdllc_setJoiningPanId(zb_uint16_t panId);

/*!
 * @brief       get the PAN ID to join.
 *
 * @param       panId - Pointer to current PAN ID in jdllc
 */
extern void Jdllc_getJoiningPanId(zb_uint16_t *pPanId);

/*!
 * @brief       set the channel mask.
 *
 * @param       chanMask - channel mask to set
 */
extern void Jdllc_setChanMask(zb_uint32_t chanMask);

/*!
 * @brief       get the channel mask.
 *
 * @param       chanMask - Pointer to current channel mask in jdllc
 */
extern void Jdllc_getChanMask(zb_uint8_t *chanMask);

#ifdef FEATURE_MAC_SECURITY
/*!
 * @brief       set the default security key.
 *
 * @param       key - default key to set
 */
extern void Jdllc_setDefaultKey(zb_uint8_t *key);

/*!
 * @brief       get the default security key.
 *
 * @param       key - default key to get
 */
extern void Jdllc_getDefaultKey(zb_uint8_t *key);
#endif

/*!
 * @brief       Get the collector (Full Function Device - FFD) address.
 *
 * @param       addr - FFD address from jdllc
 */
extern void Jdllc_getFfdAddr(zb_ieee_addr_t *addr);

/*!
 * @brief       Get the current PHY ID of the sensor device.
 *
 * @return       frequency - Current MAC PHY ID
 */
extern zb_uint8_t Jdllc_getFreq(void);

/*!
 * @brief       Get the current channel of the sensor device
 *
 * @return      channel - Current MAC channel
 */
extern zb_uint8_t Jdllc_getChan(void);

/*!
 * @brief       Get the current state of the sensor device.
 *
 * @return       state - Sensor state
 */
extern zb_uint8_t Jdllc_getProvState(void);

/*!
 * @brief       Get the previous state of the sensor device.
 *
 * @return       state - Sensor state
 */
extern zb_uint8_t Jdllc_getPrevProvState(void);

/* @brief       Start the joining process.
 *              <BR>
 *              The application will call this function to start the process
 *              of joining for this device a new network.
 *              <BR>
 *              This module will automatically determine the best channel(s)
 *              and PANID, before starting.
 */
extern void Jdllc_join(void);

/*!
 * @brief       Restore the device in the network.
 *              <BR>
 *              The application will call this function to restore the
 *              device in the network by passing all the network
 *              information needed to restore the device.
 *              <BR>
 *              This module will configure the MAC with all the network
 *              information then start the device without scanning.
 *
 * @param       pDevInfo - pointer to device descriptor for the end device
 * @param       pParentInfo - pointer to network information for the device's
 *              parent
 */
extern void Jdllc_rejoin(zb_device_descriptor_t *pDevInfo,
                         Llc_netInfo_t *pParentInfo);

/*!
 * @brief       API for app to set the poll interval.
 *              <BR>
 *              The application will call this function to set the poll
 *              interval received in config message
 *
 * @param       pollInterval - poll interval in milliseconds
 */
extern void Jdllc_setPollRate(zb_uint32_t pollInterval);

/*!
 * @brief       API for app to set disassociation request.
 */
extern void Jdllc_sendDisassociationRequest(void);

/*!
 * @brief       Fill in the security fields in an outgoing message
 *
 * @param       pSecurityLevel - pointer to security level
 * @param       pKeyIdMode - pointer to the key ID mode
 * @param       pKeySource - pointer to the key source
 * @param       pKeyIndex - pointer to the key index
 */
extern void Jdllc_securityFill(zb_uint8_t *pSecurityLevel,
                               zb_uint8_t *pKeyIdMode,
                               zb_uint8_t *pKeySource,
                               zb_uint8_t *pKeyIndex);   
/*!
 * @brief       Check the security level of an incoming message against expected level
 *
 * @param       securityLevel - the security level of the incoming message received by the device
 *
 * @return      true is matches expected security level, false if not
 */
extern zb_bool_t Jdllc_securityCheck(zb_uint8_t securityLevel);

/*!
 * @brief      Add a device to the MAC security device table.
 *
 * @param      panID - PAN ID
 * @param      shortAddr - short address of the device
 * @param      pExtAddr - pointer to the extended address
 * @param      frameCounter - starting frame counter
 *
 * @return     status returned by ApiMac_secAddDevice()
 */
extern void Jdllc_addSecDevice(zb_uint16_t panID,
                               zb_uint16_t shortAddr,
                               zb_ieee_addr_t *pExtAddr,
                               zb_uint32_t frameCounter);


/******************************************************************************
 ClockP Utility Function Prototypes. 15.4 stack had a separate layer for all these
 utilities, but I am putting them in the JDLLC layer to reduce dependencies.
 *****************************************************************************/

/*!
 * @brief   Initialize a TIRTOS Clock instance.
 *
 * @param   pClock        - pointer to clock instance structure.
 * @param   clockCB       - callback function upon clock expiration.
 * @param   clockDuration - longevity of clock timer in milliseconds
 * @param   clockPeriod   - duration of a periodic clock, used continuously
 *                          after clockDuration expires.
 * @param   startFlag     - TRUE to start immediately, FALSE to wait.
 * @param   arg           - argument passed to callback function.
 *
 * @return  Clock_Handle  - a handle to the clock instance.
 */
extern ClockP_Handle UtilTimer_construct(ClockP_Struct *pClock,
                                         ClockP_Fxn clockCB,
                                        zb_uint32_t clockDuration,
                                        zb_uint32_t clockPeriod,
                                        zb_uint8_t startFlag,
                                        uintptr_t arg);

/*!
 * @brief   Destruct the clock
 *
 * @param   pClock - clock struct
 */
extern void UtilTimer_destruct(ClockP_Struct *pClock);

/*!
 * @brief   Start a Timer/Clock.
 *
 * @param   pClock - pointer to clock struct
 */
extern void UtilTimer_start(ClockP_Struct *pClock);

/*!
 * @brief   Determine if a clock is currently active.
 *
 * @param   pClock - pointer to clock struct
 *
 * @return  TRUE or FALSE
 */
extern bool UtilTimer_isActive(ClockP_Struct *pClock);

/*!
 * @brief   Stop a Timer/Clock.
 *
 * @param   pClock - pointer to clock struct
 */
extern void UtilTimer_stop(ClockP_Struct *pClock);

/*!
 * @brief   Set a Timer/Clock timeout.
 *
 * @param   timeOut - Timeout value in milliseconds
 */
extern void UtilTimer_setTimeout(ClockP_Handle handle, zb_uint32_t timeout);

/*!
 * @brief   Get a Timer/Clock timeout.
 *
 * @param   handle - clock handle
 *
 * @return   timeOut - Timeout value in milliseconds
 */
extern uint32_t UtilTimer_getTimeout(ClockP_Handle handle);

/*!
 * @brief       Set the poll timer clock
 *
 * @param       pollTime - duration of poll timer in milliseconds
 */
extern void setPollClock(zb_uint32_t pollTime);

/*!
 * @brief       Set the scan backoff timer clock
 *
 * @param       scanBackoffTime - duration of scan backoff timer in milliseconds
 */
extern void setScanBackoffClock(zb_uint32_t scanBackoffTime);

//*****************************************************************************
//*****************************************************************************


#endif /* JDLLC_H */