/*
 * Copyright (c) 2022-2025 Texas Instruments Incorporated - http://www.ti.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== LogModule.syscfg.js ========
 */

"use strict";

const topModuleDisplayName = "Logging";
const topModuleDescription = "TI Log Configuration";

/*
 *  ======== objectValuesToArray ========
 *  Recursively iterate all values in an object and return them as an array
 */
function objectValuesToArray(obj, stack) {
    for (var property in obj) {
        if (obj.hasOwnProperty(property)) {
            if (typeof obj[property] == "object") {
                objectValuesToArray(obj[property], stack);
            } else {
                stack.push(obj[property]);
            }
        }
    }
}

/*
 *  ======== getLogSinks ========
 *  Get all LogSinks that exist in all products. The LogSink must exist
 *  with the format "/LogSink[A-Z]..."
 */
function getLogSinks() {
    let logSinks = []
    // Get all components in all products
    for (var i = 0; i < system.getProducts().length; i++){
        let product = system.getProducts()[i];
        if ('components' in product) {
            let components = product.components;
            for (var j = 0; j < components.length; j++) {
                let temp = system.getScript(components[j]);
                // Get all modules in component topModule
                let modules = []
                if ('topModules' in temp) {
                    objectValuesToArray(temp.topModules, modules);
                    for (var k = 0; k < modules.length; k++) {
                        if (typeof modules[k] === 'string' && modules[k].match(/\/LogSink[A-Z]/)) {
                            logSinks.push({ name: modules[k] });
                        }
                    }
                }
            }
        }
    }
    return logSinks;
}

/*
 *  ======== sinksToTopModule ========
 *  This function can be called from a components.js file to add a LogSink
 *  as a topmodule to the product, to make sure it appears in the correct place
 *  under "Logging". See log.component.js for example usage.
 */
function sinksToTopModule(sinks) {
    let loggingTopModule = {
        displayName: topModuleDisplayName,
        description: topModuleDescription,
        categories: [
            {
                displayName: "Log Sinks",
                modules: sinks
            }
        ]
    }
    return loggingTopModule;
}

let logLevels = [
    {
        name: "enable_DEBUG",
        displayName: "Enable Level DEBUG",
        default: true
    },
    {
        name: "enable_VERBOSE",
        displayName: "Enable Level VERBOSE",
        default: true
    },
    {
        name: "enable_INFO",
        displayName: "Enable Level INFO",
        default: true
    },
    {
        name: "enable_WARNING",
        displayName: "Enable Level WARNING",
        default: true
    },
    {
        name: "enable_ERROR",
        displayName: "Enable Level ERROR",
        default: true
    }
];

/*
 *  ======== config_instance ========
 *  Define the config params of the module instance
 */
let config_instance = [
    {
        name: "enableModule",
        displayName: "Enable Module",
        description: "Enable or disable module. If disabled, no logs will be generated by this module",
        default: true
    },
    {
        name: "logLevelGroup",
        displayName: "Log Level Configuration",
        description: "Set a filter for which levels this module should generate logs",
        collapsed: false,
        config: logLevels
    },
    {
        name: "loggerSink",
        displayName: "Sink",
        description: "Choose which sink the log output should be handled by",
        default: "/ti/log/LogSinkBuf",
        options: getLogSinks
    }
];

/*
 *  ======== static_config ========
 *  Define the global config params
 */
let static_config = [
    {
        name: "enableGlobal",
        displayName: "Global Enable Module",
        description: "Enable or disable all modules. If disabled, no logs will be generated",
        default: true
    },
    {
        name: "globalLogLevelGroup",
        displayName: "Global Log Level Configuration",
        description: "Set which log levels should generate logs (this is applied globally for all modules)",
        collapsed: false,
        config: logLevels
    },
    {
        name: "enableDynamicModuleLogLevels",
        displayName: "Enable Dynamic Module Log Levels",
        description: `Enable runtime checking for each log module's log level bitmask and allow
        each log module's log level bitmask to be dynamically changed at runtime. If disabled,
        a log module's log level will be static and no runtime checks or dynamic changes can occur.`,
        default: false
    }
];

let module_static = {
    collapsed: false,
    config: static_config,
    validate: validateStatic
};

/*
 *  ======== sharedModuleInstances ========
 *  Express dependencies for shared instances of other modules
 */
function sharedModuleInstances(inst) {
    let modules = new Array();

    modules.push({
        name: "logger",
        displayName: "Logger Sink instance",
        moduleName: inst.loggerSink
    });

    return (modules);
}


function getLibs(mod) {
    let libGroup = {
        name: "/ti/log",
        vers: "1.0.0.0",
        deps: [],
        libs: []
    };

    /* Get device information from GenLibs */
    let GenLibs = system.getScript("/ti/utils/build/GenLibs");
    let libPath = GenLibs.libPath;

    /* get library name from DriverLib */
    var DriverLib = system.getScript("/ti/devices/DriverLib");
    let devId = system.deviceData.deviceId;
    let libFamilyName = DriverLib.getAttrs(devId).libName;

    /* add the log library to libGroup's libs */
    libGroup.libs.push(libPath("ti/log", "log_" + libFamilyName + ".a"));

    /* add dependency on /ti/drivers (if needed) */
    let needDrivers = false;
    for (let i = 0; i < mod.$instances.length; i++) {
        let inst = mod.$instances[i];
        if (inst.loggerSink != "/ti/log/LogSinkBuf") {
            needDrivers = true;
            break;
        }
    }
    libGroup.deps = needDrivers ? ["/ti/drivers"] : [];

    return (libGroup);
}

function getOpts() {
    let options = [];

    let logModule = system.modules["/ti/log/LogModule"];

    /* Push the module-level enable define for each active module */
    for (let i = 0; i < logModule.$instances.length; i++) {
        let inst = logModule.$instances[i];
        if (inst.enableModule === true) {
            options.push("-Dti_log_Log_ENABLE_" + inst.$name + "=1");
        }
    }

    /* Push the global enable define that enables all logs */
    options.push("-Dti_log_Log_ENABLE")

    /* Push these defines to enable logs in application-compiled code */
    return options;
}

function validateStatic(inst, validation) {
    let logModule = system.modules["/ti/log/LogModule"];

    if (logModule.$instances.length == 0) {
        validation.logWarning(`Global Log module enabled without any instances! This is caused by altered global
settings. Click \"REMOVE ALL\" next to Log Modules to remove application-compiled log sites.`,
            inst,
            "enableGlobal");
    }
}

/*
 *  ======== base ========
 *  Module definition object
 */
let base = {
    displayName: "Log Modules",
    description: "Log Module Configuration",
    longDescription: `
The [__Log framework__][1] provides an API for generating formatted debug output,
which can be either stored on target, or transported to a host-side tool.
It is meant to be an easy-to-use and low-overhead tool for debugging embedded applications.
This module relies on Log sinks, which can be found a subsection to this SysConfig module.

* [Log API][2]
* [Log Tools][3]

[1]: /drivers/doxygen/html/index.html#log "Log framework"
[2]: /tiutils/html/group__ti__log__LOG.html "Log API"
[3]: /../tools/log/tiutils/Readme.html "Log Tools"
`,
    config: config_instance,
    moduleStatic: module_static,
    defaultInstanceName: "LogModule",
    sharedModuleInstances: sharedModuleInstances,
    templates: {
        /* contribute libraries to linker command file */
        "/ti/utils/build/GenLibs.cmd.xdt":
            { modName: "/ti/log/LogModule", getLibs: getLibs },
        "/ti/utils/build/GenOpts.opt.xdt":
            { modName: "/ti/log/LogModule", getOpts: getOpts },
        "/ti/log/templates/ti_log_config.c.xdt":
            "/ti/log/templates/LogModule.Config.c.xdt",
        "/ti/log/templates/ti_log_config.h.xdt":
            "/ti/log/templates/LogModule.Config.h.xdt"
    },
    sinksToTopModule: sinksToTopModule,
    topModuleDisplayName: topModuleDisplayName,
    topModuleDescription: topModuleDescription
};

/* export the module */
exports = base;
