/*
 * Copyright (c) 2023-2024, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*!*****************************************************************************
 *  @file       PWMTimerLPF3.h
 *  @brief      PWM driver implementation for Low Power F3 devices
 *
 *  # Overview #
 * The general PWM API should be used in application code, i.e. PWM_open()
 * should be used instead of PWMTimerLPF3_open(). The board file will define the device
 * specific config, and casting in the general API will ensure that the correct
 * device specific functions are called.
 *
 * # General Behavior #
 * Before using PWM on LPF3:
 *   - The Timer HW is configured and system dependencies (for example IOs,
 *     power, etc.) are set by calling PWM_open().
 *
 * # Error handling #
 * If unsupported arguments are provided to an API returning an error code, the
 * PWM configuration will *not* be updated and PWM will stay in the mode it
 * was already configured to.
 *
 * # Power Management #
 * The SimpleLink power management framework will try to put the device into the most
 * power efficient mode whenever possible. Please see the technical reference
 * manual for further details on each power mode.
 *
 *  The PWMTimerLPF3.h driver is not explicitly setting a power constraint when the
 *  PWM is running to prevent standby as this is assumed to be done in the
 *  underlying LGPTimer driver.
 *  The following statements are valid:
 *    - After PWM_open(): The device is still allowed to enter Standby. When the
 *                        device is active the underlying LGPTimer peripheral will
 *                        be enabled and clocked.
 *    - After PWM_start(): The device can only go to Idle power mode since the
 *                         high-frequency clock (CLKSVT) is needed for PWM operation:
 *    - After PWM_stop():  Conditions are equal as for after PWM_open
 *    - After PWM_close(): The underlying LGPTimer is turned off and the device
 *                         is allowed to go to standby.
 *
 *  # Accuracy #
 *  The PWM output period and duty cycle are limited by the underlying timer.
 *  The LGPT peripherals have the following counter widths:
 *    - LGPT0: 16 bits
 *    - LGPT1: 16 bits
 *    - LGPT2: 16 bits
 *    - LGPT3: 24 bits
 *
 *  Note that some LPF3 device types do not support all four LGPT peripherals.
 *
 *  The frequency of the underlying timer counter can be divided by configuring
 *  the preScalerDivision element of the @ref PWMTimerLPF3_HwAttrs struct.
 *  This configuration can be used to extend the PWM signal period but will
 *  decrease the accuracy.
 *  The minimum obtainable PWM signal frequency is dependent on the width of
 *  the counter for the LGPT peripheral used and the configured preScalerDivision
 *  value. When using the LGPT3 peripheral with the default preScalerDivision
 *  value of 1, the minimal frequency is  48MHz / (2^24-1) = 2.86Hz (349.525ms).
 *
 *  When using high output frequencies the duty cycle resolution is reduced
 *  correspondingly. For a 24MHz PWM only a 0%/50%/100% duty is available as
 *  the timer uses only counts 0 and 1.
 *  Similarly for a 12MHz period the duty cycle will be limited to a 12.5%
 *  resolution.
 *
 *  @note The PWM signals are generated by the LGPT peripheral using the
 *  timer clock which is dependent on the high frequency clock (CLKSVT).
 *
 *  # Limitations #
 *  - The PWM output can currently not be synchronized with other PWM outputs.
 *  - The PWM driver does not support updating duty and period using DMA.
 *  - Attempts to change both the period and duty cycle at the same time
 *    for an active PWM signal, can cause pulses to be too long or short if
 *    the change is applied close to the end of the current counter cycle.
 *    This does not apply to changing only the period or only the duty cycle,
 *    as any of these separate changes will take effect on the next counter cycle.
 *  # PWM usage #
 *
 *  ## Basic PWM output ##
 *  The below example will output a 8MHz PWM signal with 50% duty cycle.
 *  @code
 *  PWM_Handle pwmHandle;
 *  PWM_Params params;
 *
 *  PWM_Params_init(&params);
 *  params.idleLevel      = PWM_IDLE_LOW;
 *  params.periodUnits    = PWM_PERIOD_HZ;
 *  params.periodValue    = 8e6;
 *  params.dutyUnits      = PWM_DUTY_FRACTION;
 *  params.dutyValue      = PWM_DUTY_FRACTION_MAX / 2;
 *
 *  pwmHandle = PWM_open(CONFIG_PWM0, &params);
 *  if(pwmHandle == NULL) {
 *    Log_error0("Failed to open PWM");
 *    Task_exit();
 *  }
 *  PWM_start(pwmHandle);
 *  @endcode
 *
 *
 *******************************************************************************
 */
#ifndef ti_drivers_pwm__PWMTimerLPF3_include
#define ti_drivers_pwm__PWMTimerLPF3_include

#include <stdint.h>
#include <stdbool.h>

#include <ti/drivers/PWM.h>
#include <ti/drivers/timer/LGPTimerLPF3.h>

#ifdef __cplusplus
extern "C" {
#endif

/*!
 *  @name PWMTimerLPF3 specific control commands and arguments
 *  @{
 */

/*! Timer debug stall mode (stop PWM output debugger halts CPU)
    When enabled, PWM output will be HIGH when CPU is halted
 */
#define PWMTimerLPF3_CMD_DEBUG_STALL (PWM_CMD_RESERVED + 0) /*!< @hideinitializer */
/*!
 *  @name Arguments for PWMTimerLPF3_CMD_DEBUG_STALL
 *  @{
 */
#define CMD_ARG_DEBUG_STALL_OFF      ((uint32_t)LGPTimerLPF3_DEBUG_STALL_OFF)       /*!< @hideinitializer */
#define CMD_ARG_DEBUG_STALL_ON       ((uint32_t)LGPTimerLPF3_DEBUG_STALL_IMMEDIATE) /*!< @hideinitializer */
/*  @} */

/*  @} */

/* PWM function table pointer */
extern const PWM_FxnTable PWMTimerLPF3_fxnTable;

/*!
 *  @brief  PWMTimerLPF3 Hardware attributes
 *
 *  These fields are used by the driver to set up the underlying LGPTimer
 *  driver statically.
 */
typedef struct PWMTimerLPF3_HwAttrs
{
    uint8_t lgpTimerInstance;   /*!< LGPTimer unit index (0, 1, 2..) */
    uint16_t preScalerDivision; /*!< LGPTimer prescaler divider */
} PWMTimerLPF3_HwAttrs;

/*!
 *  @brief  PWMTimerLPF3 Object
 *
 * These fields are used by the driver to store and modify PWM configuration
 * during run-time.
 * The application must not edit any member variables of this structure.
 * Applications should also not access member variables of this structure
 * as backwards compatibility is not guaranteed.
 */
typedef struct PWMTimerLPF3_Object
{
    uint32_t periodValue;            /*!< Current period value in unit */
    uint32_t periodCounts;           /*!< Current period in raw timer counts */
    uint32_t dutyValue;              /*!< Current duty cycle value in unit */
    uint32_t dutyCounts;             /*!< Current duty in raw timer counts */
    LGPTimerLPF3_Handle hTimer;      /*!< Handle to underlying LGPTimer peripheral */
    PWM_Period_Units periodUnit;     /*!< Current period unit */
    PWM_Duty_Units dutyUnit;         /*!< Current duty cycle unit */
    LGPTimerLPF3_ChannelNo chNumber; /*!< LGPTimer peripheral channel number selected for PWM output signal */
    PWM_IdleLevel idleLevel;         /*!< PWM idle level when stopped / not started */
    bool isOpen;                     /*!< open flag used to check if PWM is opened */
    bool isRunning;                  /*!< running flag, set if the output is active */
} PWMTimerLPF3_Object;

#ifdef __cplusplus
}
#endif
#endif /* ti_driver_pwm_PWMTimerLPF3_include */
