/* testvectors_aria_basic_data.h
 *
 * Description: ARIA test vector data for ECB, CBC, CTR and ICM
 */

/* -------------------------------------------------------------------------- */
/*                                                                            */
/*   Module        : DDK-130_bsd                                              */
/*   Version       : 4.1.1                                                    */
/*   Configuration : DDK_EIP130_BSD                                           */
/*                                                                            */
/*   Date          : 2023-Mar-10                                              */
/*                                                                            */
/* Copyright (c) 2014-2023 by Rambus, Inc. and/or its subsidiaries.           */
/*                                                                            */
/* Redistribution and use in source and binary forms, with or without         */
/* modification, are permitted provided that the following conditions are     */
/* met:                                                                       */
/*                                                                            */
/* 1. Redistributions of source code must retain the above copyright          */
/* notice, this list of conditions and the following disclaimer.              */
/*                                                                            */
/* 2. Redistributions in binary form must reproduce the above copyright       */
/* notice, this list of conditions and the following disclaimer in the        */
/* documentation and/or other materials provided with the distribution.       */
/*                                                                            */
/* 3. Neither the name of the copyright holder nor the names of its           */
/* contributors may be used to endorse or promote products derived from       */
/* this software without specific prior written permission.                   */
/*                                                                            */
/* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS        */
/* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT          */
/* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR      */
/* A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT       */
/* HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,     */
/* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT           */
/* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,      */
/* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY      */
/* THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT        */
/* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE      */
/* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.       */
/* -------------------------------------------------------------------------- */

#ifndef INCLUDE_GUARD_TEST_VECTORS_ARIA_BASIC_DATA_H
#define INCLUDE_GUARD_TEST_VECTORS_ARIA_BASIC_DATA_H

#include <third_party/hsmddk/include/Kit/DriverFramework/Basic_Defs_API/incl/basic_defs.h>

/* ECB */
/* - ARIA128(ECB)MMT */
static const uint8_t ARIA_ECB_128_1_key[] =
{
    0x5D, 0x9D, 0x01, 0xD8, 0x00, 0x75, 0x7F, 0x94, 0x6B, 0x93, 0x45, 0x20, 0x6B, 0xE3, 0xA2, 0x60
};
static const uint8_t ARIA_ECB_128_1_pt[] =
{
    0xE7, 0x01, 0xD2, 0x71, 0x13, 0x9F, 0xD0, 0x31, 0xE7, 0x28, 0x6B, 0x01, 0xD0, 0xA2, 0x9C, 0xC9
};
static const uint8_t ARIA_ECB_128_1_ct[] =
{
    0x16, 0xE4, 0xFB, 0xE5, 0x23, 0x7E, 0x38, 0xCB, 0x47, 0x5F, 0xA9, 0x13, 0x99, 0xB7, 0xD5, 0x3F
};

static const uint8_t ARIA_ECB_128_2_key[] =
{
    0xF8, 0x72, 0x90, 0x02, 0xF8, 0x25, 0x9F, 0xA0, 0x03, 0x99, 0xC8, 0xD6, 0x8A, 0xC9, 0x29, 0x20
};
static const uint8_t ARIA_ECB_128_2_pt[] =
{
    0xA8, 0xB4, 0xD8, 0x07, 0x58, 0xBB, 0xBF, 0xA0, 0xE6, 0x4A, 0xAA, 0x55, 0xA3, 0xBA, 0x9E, 0xE7,
    0xDD, 0x76, 0x6D, 0xB4, 0x0E, 0x5D, 0x87, 0x36, 0x54, 0x68, 0x13, 0xCD, 0x21, 0x8D, 0x3E, 0xBE,
    0x34, 0xE3, 0xD3, 0x7B, 0xC8, 0x06, 0x0D, 0x25, 0x0B, 0xBD, 0xA9, 0x4F, 0xD1, 0xDA, 0x3E, 0x05
};
static const uint8_t ARIA_ECB_128_2_ct[] =
{
    0x3B, 0x86, 0xD9, 0x34, 0x78, 0xDF, 0x7C, 0x90, 0xF6, 0x71, 0xA5, 0x98, 0xFD, 0x1E, 0x38, 0x82,
    0xFA, 0x33, 0x14, 0xD7, 0x5F, 0x43, 0xF7, 0x92, 0x7C, 0x83, 0x2D, 0x75, 0x14, 0x8B, 0x31, 0x7F,
    0x0F, 0xDC, 0x6B, 0xA7, 0x8F, 0xB3, 0x2B, 0x6D, 0xD1, 0xE4, 0x6C, 0x49, 0x0C, 0xCF, 0xC2, 0x9F
};

/* - ARIA192(ECB)MMT */
static const uint8_t ARIA_ECB_192_1_key[] =
{
    0x5D, 0x9D, 0x01, 0xD8, 0x00, 0x75, 0x7F, 0x94, 0x6B, 0x93, 0x45, 0x20, 0x6B, 0xE3, 0xA2, 0x60,
    0xE7, 0x01, 0xD2, 0x71, 0x13, 0x9F, 0xD0, 0x31
};
static const uint8_t ARIA_ECB_192_1_pt[] =
{
    0xE7, 0x28, 0x6B, 0x01, 0xD0, 0xA2, 0x9C, 0xC9, 0xA5, 0xD9, 0x17, 0x3D, 0xDC, 0xBA, 0x20, 0x5F
};
static const uint8_t ARIA_ECB_192_1_ct[] =
{
    0x9D, 0x9C, 0x8B, 0x8E, 0x41, 0x3D, 0x33, 0x20, 0x82, 0x2B, 0x1F, 0xCC, 0x71, 0xBE, 0xB3, 0xF3
};

static const uint8_t ARIA_ECB_192_2_key[] =
{
    0x86, 0x6E, 0x2E, 0xF5, 0x3A, 0x89, 0x72, 0xA7, 0xE7, 0x7E, 0xE6, 0x93, 0x8A, 0xA0, 0xA0, 0x49,
    0x04, 0x9C, 0xA8, 0x7C, 0xCE, 0x5D, 0x50, 0x05
};
static const uint8_t ARIA_ECB_192_2_pt[] =
{
    0xE8, 0x2C, 0xC8, 0x2A, 0x12, 0x1C, 0xED, 0x21, 0x3F, 0xA5, 0x62, 0xD0, 0x47, 0x8F, 0xE2, 0xD7,
    0xDC, 0x11, 0xD4, 0xD0, 0xBF, 0x3B, 0x20, 0xD5, 0x33, 0x91, 0x42, 0x43, 0xB3, 0xF9, 0x9D, 0xBE,
    0xE0, 0xD7, 0x11, 0x76, 0xBD, 0x76, 0xCF, 0xA8, 0x1E, 0xAA, 0x6C, 0x67, 0x5E, 0xB3, 0x35, 0x9C,
    0x51, 0x65, 0xC3, 0x4C, 0x79, 0x86, 0xCF, 0x19, 0x7D, 0x78, 0x47, 0x0C, 0xD6, 0x17, 0xA0, 0x91
};
static const uint8_t ARIA_ECB_192_2_ct[] =
{
    0xAA, 0x8F, 0x46, 0x2D, 0x59, 0x4D, 0x87, 0xD8, 0xF9, 0x4B, 0x04, 0xE5, 0x55, 0xDA, 0x89, 0xA1,
    0x36, 0x26, 0x10, 0x5E, 0xD8, 0xCC, 0xFF, 0x02, 0x85, 0xA1, 0xA4, 0xC3, 0xA1, 0x01, 0xC8, 0xF9,
    0xC6, 0x1E, 0x5C, 0x29, 0xE4, 0xEE, 0xC0, 0xDD, 0x37, 0x72, 0xDD, 0x7B, 0xD6, 0xC1, 0x69, 0xB5,
    0x12, 0xBB, 0xA5, 0x78, 0x5C, 0x42, 0x44, 0xAC, 0xC1, 0x0E, 0xFB, 0x4C, 0xB8, 0xFE, 0xC4, 0x7D
};

/* - ARIA256(ECB)MMT */
static const uint8_t ARIA_ECB_256_1_key[] =
{
    0x5D, 0x9D, 0x01, 0xD8, 0x00, 0x75, 0x7F, 0x94, 0x6B, 0x93, 0x45, 0x20, 0x6B, 0xE3, 0xA2, 0x60,
    0xE7, 0x01, 0xD2, 0x71, 0x13, 0x9F, 0xD0, 0x31, 0xE7, 0x28, 0x6B, 0x01, 0xD0, 0xA2, 0x9C, 0xC9
};
static const uint8_t ARIA_ECB_256_1_pt[] =
{
    0xA5, 0xD9, 0x17, 0x3D, 0xDC, 0xBA, 0x20, 0x5F, 0xFE, 0xFD, 0xA0, 0x45, 0x58, 0x07, 0x78, 0x1D
};
static const uint8_t ARIA_ECB_256_1_ct[] =
{
    0xF5, 0x62, 0xEF, 0x72, 0x17, 0x08, 0x2A, 0x03, 0xB3, 0x94, 0x6A, 0x26, 0x74, 0x01, 0x34, 0xD7
};

static const uint8_t ARIA_ECB_256_2_key[] =
{
    0x94, 0x9F, 0x09, 0x9B, 0x26, 0xB0, 0xC1, 0xF0, 0xEF, 0x3C, 0xA5, 0xEA, 0x4E, 0x79, 0x26, 0x06,
    0x5B, 0x6B, 0x41, 0x1D, 0x1D, 0x4F, 0x54, 0xE7, 0xFE, 0x3A, 0x30, 0xBD, 0x7B, 0xFE, 0x6F, 0xA3
};
static const uint8_t ARIA_ECB_256_2_pt[] =
{
    0x20, 0x4D, 0x08, 0x85, 0xB4, 0x21, 0x20, 0xC1, 0xB1, 0x5A, 0x23, 0xE6, 0xF7, 0xA9, 0x14, 0xBB,
    0xC0, 0xAE, 0x21, 0x84, 0xD7, 0x5D, 0x7B, 0x80, 0x05, 0xA6, 0x64, 0xB6, 0xCC, 0x55, 0x8A, 0xF2,
    0x58, 0x37, 0x93, 0x0A, 0xB3, 0x7E, 0xF9, 0x65, 0x38, 0x66, 0x18, 0xBE, 0x4A, 0x19, 0x85, 0x26,
    0x46, 0xD0, 0xA2, 0x48, 0xB5, 0x3C, 0x6F, 0xF1, 0xC7, 0x24, 0xA3, 0xD0, 0xFD, 0x4F, 0xFC, 0x7A,
    0x25, 0xA4, 0xD3, 0xB0, 0x8B, 0x90, 0xF3, 0xE5, 0x6E, 0xA8, 0xAB, 0xFB, 0xB1, 0x90, 0x22, 0x4E
};
static const uint8_t ARIA_ECB_256_2_ct[] =
{
    0x6A, 0x56, 0x5E, 0xB7, 0xE4, 0x1A, 0xD1, 0xC5, 0x88, 0xB3, 0xFB, 0x5A, 0xB7, 0x4C, 0xC7, 0x0D,
    0x54, 0x35, 0xCF, 0x08, 0x54, 0x5B, 0x3E, 0xC5, 0xBD, 0xF5, 0xB9, 0xB5, 0x75, 0x53, 0x55, 0x60,
    0x92, 0x4B, 0xF1, 0xA7, 0xFC, 0x85, 0x5F, 0x05, 0x51, 0x53, 0xA8, 0x97, 0x49, 0xF0, 0x8A, 0xE1,
    0x9E, 0xFA, 0x4D, 0x4C, 0x75, 0x5B, 0x57, 0xD1, 0x90, 0x46, 0x59, 0x7F, 0xF6, 0x1C, 0x2F, 0xD1,
    0x07, 0x77, 0x64, 0xE7, 0x31, 0x7F, 0xD8, 0xF5, 0x0D, 0x74, 0xC6, 0x9C, 0x31, 0xCA, 0xDC, 0x50
};


/* CBC */
/* - ARIA128(CBC)MMT */
static const uint8_t ARIA_CBC_128_1_key[] =
{
    0x60, 0x9A, 0xC9, 0xD7, 0x4A, 0xCB, 0xA7, 0x25, 0xEA, 0x8F, 0xB1, 0x4C, 0xE6, 0x85, 0x52, 0x1A
};
static const uint8_t ARIA_CBC_128_1_iv[] =
{
    0xA9, 0xFB, 0xF9, 0x55, 0xAB, 0x18, 0x76, 0x97, 0xF2, 0x5A, 0xA8, 0x5C, 0xD9, 0x44, 0x55, 0x5A
};
static const uint8_t ARIA_CBC_128_1_pt[] =
{
    0x10, 0x50, 0x59, 0x11, 0xE4, 0x2B, 0x03, 0xA2, 0xC0, 0x21, 0xBC, 0xE4, 0xF4, 0xD6, 0xE9, 0xB5
};
static const uint8_t ARIA_CBC_128_1_ct[] =
{
    0x6A, 0xBA, 0xB3, 0x88, 0x71, 0x86, 0xDB, 0xDF, 0xF1, 0xBB, 0x7F, 0xE9, 0xCC, 0xA2, 0xF8, 0x75
};

static const uint8_t ARIA_CBC_128_2_key[] =
{
    0xB6, 0x38, 0x13, 0x8E, 0x66, 0xB1, 0x45, 0x5B, 0x17, 0xA1, 0xB6, 0x89, 0x4A, 0x0B, 0xA8, 0x60
};
static const uint8_t ARIA_CBC_128_2_iv[] =
{
    0xFC, 0x8F, 0x90, 0x93, 0xE4, 0x98, 0x77, 0x15, 0xF6, 0xD6, 0xA0, 0x2F, 0x2B, 0xEF, 0x6B, 0xD0
};
static const uint8_t ARIA_CBC_128_2_pt[] =
{
    0x84, 0x71, 0x79, 0x24, 0x4D, 0x0C, 0x11, 0x67, 0x9F, 0xFE, 0xC3, 0xFC, 0x28, 0xCE, 0x4B, 0xB9,
    0x2E, 0x3D, 0xB8, 0x86, 0xD2, 0x7A, 0xCD, 0x26, 0x92, 0x95, 0xA7, 0x54, 0x12, 0x36, 0xA0, 0x12
};
static const uint8_t ARIA_CBC_128_2_ct[] =
{
    0x2F, 0xA2, 0xAF, 0x56, 0x77, 0x06, 0xAD, 0xD9, 0xB8, 0xA8, 0xC9, 0x0D, 0x56, 0xA9, 0x08, 0x26,
    0xDE, 0x96, 0x5A, 0x5F, 0xBB, 0xAB, 0x46, 0x5D, 0x83, 0x8A, 0xA2, 0xBC, 0x66, 0xFA, 0x36, 0x51
};

/* - ARIA192(CBC)MMT */
static const uint8_t ARIA_CBC_192_1_key[] =
{
    0x60, 0x9A, 0xC9, 0xD7, 0x4A, 0xCB, 0xA7, 0x25, 0xEA, 0x8F, 0xB1, 0x4C, 0xE6, 0x85, 0x52, 0x1A,
    0xA9, 0xFB, 0xF9, 0x55, 0xAB, 0x18, 0x76, 0x97
};
static const uint8_t ARIA_CBC_192_1_pt[] =
{
    0xF2, 0x5A, 0xA8, 0x5C, 0xD9, 0x44, 0x55, 0x5A, 0x10, 0x50, 0x59, 0x11, 0xE4, 0x2B, 0x03, 0xA2
};
static const uint8_t ARIA_CBC_192_1_iv[] =
{
    0xC0, 0x21, 0xBC, 0xE4, 0xF4, 0xD6, 0xE9, 0xB5, 0xB6, 0x38, 0x13, 0x8E, 0x66, 0xB1, 0x45, 0x5B
};
static const uint8_t ARIA_CBC_192_1_ct[] =
{
    0x2E, 0x2F, 0xA3, 0xB5, 0xA5, 0x06, 0xB1, 0xD1, 0xA5, 0x3C, 0x6B, 0x08, 0x60, 0xFA, 0xF8, 0xDF
};

static const uint8_t ARIA_CBC_192_2_key[] =
{
    0xAE, 0x43, 0x17, 0x11, 0x37, 0x38, 0xCB, 0x7C, 0x9E, 0x45, 0x17, 0x7C, 0x32, 0x14, 0x50, 0x24,
    0x85, 0x78, 0x4A, 0x04, 0xB8, 0xA3, 0xC0, 0xFE
};
static const uint8_t ARIA_CBC_192_2_iv[] =
{
    0xF8, 0x99, 0xF5, 0x55, 0x49, 0xE5, 0x9C, 0x07, 0x82, 0x8A, 0xF5, 0x5D, 0x99, 0x3C, 0x3A, 0xC0
};
static const uint8_t ARIA_CBC_192_2_pt[] =
{
    0x21, 0xD0, 0x39, 0xCE, 0x11, 0xF2, 0x42, 0xAD, 0xC7, 0x16, 0x43, 0x9F, 0x4D, 0xB1, 0x32, 0xD7,
    0xDA, 0xA8, 0xAB, 0x8C, 0x9C, 0x08, 0xDB, 0x4B, 0xB4, 0xF9, 0x9C, 0x91, 0x83, 0xED, 0xE1, 0x99,
    0x24, 0x1D, 0x55, 0x73, 0x39, 0x36, 0x3F, 0x56, 0xEB, 0x4F, 0xA9, 0x37, 0x2C, 0x1F, 0xC1, 0x9A
};
static const uint8_t ARIA_CBC_192_2_ct[] =
{
    0x2F, 0xF0, 0x82, 0x07, 0x83, 0xDF, 0x8C, 0x5B, 0x3A, 0xBD, 0x19, 0x9C, 0x9C, 0x11, 0x05, 0xC4,
    0xF3, 0x04, 0xB4, 0x43, 0x24, 0xCE, 0x35, 0x39, 0x4C, 0x9A, 0xC5, 0x1D, 0x61, 0x50, 0x5A, 0xE0,
    0xD2, 0x6E, 0x2D, 0x82, 0x1D, 0x95, 0x5B, 0x8E, 0xEF, 0xB6, 0x4E, 0x44, 0x28, 0xC4, 0x13, 0xA9
};

/* - ARIA256(CBC)MMT */
static const uint8_t ARIA_CBC_256_1_key[] =
{
    0x60, 0x9A, 0xC9, 0xD7, 0x4A, 0xCB, 0xA7, 0x25, 0xEA, 0x8F, 0xB1, 0x4C, 0xE6, 0x85, 0x52, 0x1A,
    0xA9, 0xFB, 0xF9, 0x55, 0xAB, 0x18, 0x76, 0x97, 0xF2, 0x5A, 0xA8, 0x5C, 0xD9, 0x44, 0x55, 0x5A
};
static const uint8_t ARIA_CBC_256_1_iv[] =
{
    0x10, 0x50, 0x59, 0x11, 0xE4, 0x2B, 0x03, 0xA2, 0xC0, 0x21, 0xBC, 0xE4, 0xF4, 0xD6, 0xE9, 0xB5
};
static const uint8_t ARIA_CBC_256_1_pt[] =
{
    0xB6, 0x38, 0x13, 0x8E, 0x66, 0xB1, 0x45, 0x5B, 0x17, 0xA1, 0xB6, 0x89, 0x4A, 0x0B, 0xA8, 0x60
};
static const uint8_t ARIA_CBC_256_1_ct[] =
{
    0x6D, 0x8D, 0x63, 0x16, 0x52, 0xE1, 0x83, 0x84, 0xEF, 0x7D, 0xA0, 0xC4, 0x75, 0x5E, 0x90, 0x91
};

static const uint8_t ARIA_CBC_256_2_key[] =
{
    0xF8, 0x0D, 0xC2, 0xF3, 0xDE, 0x56, 0x35, 0xEF, 0x30, 0xD0, 0xB9, 0xD4, 0x6E, 0x89, 0x6D, 0x56,
    0x50, 0xEE, 0x61, 0x54, 0x08, 0xF6, 0xFB, 0xAC, 0xC7, 0x9D, 0x9A, 0x54, 0x82, 0x94, 0x7A, 0xA7
};
static const uint8_t ARIA_CBC_256_2_iv[] =
{
    0xF6, 0x37, 0xD8, 0xFC, 0xEA, 0xBF, 0x56, 0x7D, 0xB4, 0xB9, 0xA0, 0x37, 0xFE, 0xB7, 0x50, 0x6D
};
static const uint8_t ARIA_CBC_256_2_pt[] =
{
    0xCA, 0x45, 0x0F, 0x32, 0xB4, 0x1E, 0xFD, 0x37, 0x77, 0xA0, 0x54, 0xE1, 0xB3, 0x7E, 0x48, 0x9E,
    0xEE, 0xB5, 0x31, 0x7A, 0xD8, 0xC1, 0xEB, 0xED, 0xD3, 0x10, 0x7D, 0xF7, 0xB2, 0xB7, 0xFA, 0x6F,
    0xC1, 0x64, 0xA5, 0x99, 0x06, 0x93, 0x58, 0xF6, 0xC7, 0x05, 0x27, 0x60, 0x4D, 0x6E, 0x40, 0x56,
    0xE7, 0x6F, 0x15, 0x94, 0x31, 0xC3, 0xBE, 0xE6, 0x95, 0xBD, 0x98, 0x3F, 0x13, 0xF2, 0x33, 0x05
};
static const uint8_t ARIA_CBC_256_2_ct[] =
{
    0x55, 0xE7, 0xAC, 0xAB, 0xD3, 0x8F, 0x91, 0x79, 0x4F, 0xF7, 0x7C, 0x35, 0x5C, 0x13, 0x0F, 0xF9,
    0xA9, 0xC8, 0x2C, 0xFF, 0xD5, 0xC8, 0x48, 0xD8, 0x40, 0x2E, 0x83, 0x74, 0xA5, 0xF9, 0x72, 0x90,
    0xDC, 0x12, 0x7F, 0xAA, 0x72, 0x37, 0xFD, 0x12, 0x46, 0x14, 0xE0, 0x9D, 0xBA, 0xB7, 0xA8, 0xC7,
    0x78, 0x30, 0x05, 0x3A, 0x91, 0x64, 0x2C, 0x64, 0xB9, 0x21, 0xD4, 0xDE, 0x62, 0x84, 0x56, 0xFC
};


/* CTR */
/* - ARIA128(CTR)MMT */
static const uint8_t ARIA_CTR_128_1_key[] =
{
    0x64, 0x96, 0x91, 0xD6, 0x95, 0x22, 0xD0, 0xB5, 0x6A, 0x8B, 0x1D, 0x78, 0x61, 0x28, 0x02, 0xD4
};
static const uint8_t ARIA_CTR_128_1_iv[] =
{
    0x6B, 0xF4, 0x1F, 0x3A, 0x42, 0x91, 0x1D, 0xFE, 0xFD, 0x8D, 0xE4, 0xB8, 0xE2, 0xE5, 0x0E, 0xEB
};
static const uint8_t ARIA_CTR_128_1_pt[] =
{
    0x7B, 0xC7, 0x9B, 0xE4, 0xEC, 0x9C, 0xE6, 0xE5, 0x83, 0x46, 0xD7, 0x83, 0x91, 0xA6, 0x5B, 0x4D
};
static const uint8_t ARIA_CTR_128_1_ct[] =
{
    0x60, 0x5D, 0x10, 0xA5, 0x83, 0xE5, 0xA0, 0x92, 0x91, 0xCF, 0xA0, 0x1D, 0x58, 0x8C, 0x96, 0x10
};

static const uint8_t ARIA_CTR_128_2_key[] =
{
    0x81, 0xC5, 0x71, 0xF8, 0xF3, 0xBE, 0xD6, 0x11, 0x09, 0x95, 0x48, 0x16, 0x2C, 0xDD, 0x9C, 0xD5
};
static const uint8_t ARIA_CTR_128_2_iv[] =
{
    0xFA, 0x8D, 0x7E, 0x9E, 0x99, 0x03, 0x62, 0x5F, 0x4B, 0x49, 0xBD, 0x0E, 0x43, 0x96, 0x74, 0x47
};
static const uint8_t ARIA_CTR_128_2_pt[] =
{
    0xB8, 0x18, 0x94, 0x37, 0x37, 0xF0, 0x45, 0xB8, 0x3D, 0x7F, 0xF9, 0xBD, 0x05, 0xE0, 0xE9, 0xF1,
    0x9F, 0xB5, 0xC2, 0x9B, 0x42, 0x27, 0x9B, 0xC5, 0xE3, 0x28, 0x2B, 0x9D, 0x06, 0xFB, 0x39, 0x1B,
    0xD5, 0xF7, 0x53, 0xE4, 0x6F, 0x88, 0xC3, 0x6F, 0x82, 0x76, 0xBF, 0x66, 0x1C, 0x6A, 0xE0, 0x4F,
    0xBE, 0xAE, 0xD6, 0x6B, 0xB3, 0x34, 0x59, 0xDE, 0x9F, 0xDA, 0x62, 0x11, 0x5B, 0xEC, 0x9C, 0x55
};
static const uint8_t ARIA_CTR_128_2_ct[] =
{
    0x97, 0x61, 0x80, 0xCF, 0x66, 0xC6, 0xC5, 0xA7, 0x3E, 0xF1, 0x0A, 0x1B, 0x62, 0x4E, 0xCD, 0x02,
    0x93, 0x89, 0x03, 0x25, 0xB2, 0x46, 0xD6, 0xE5, 0x3F, 0x15, 0x56, 0x10, 0x08, 0xFA, 0x18, 0xED,
    0x3F, 0xD1, 0xC8, 0x7D, 0x26, 0x83, 0xA2, 0xB4, 0x5A, 0x17, 0xC7, 0x7A, 0x9C, 0xB0, 0x65, 0xBE,
    0x62, 0x49, 0xDB, 0xF1, 0x27, 0xB8, 0xF6, 0x24, 0x03, 0x9A, 0x5B, 0x9E, 0x70, 0x3F, 0x0C, 0xA0
};

/* - ARIA192(CTR)MMT */
static const uint8_t ARIA_CTR_192_1_key[] =
{
    0x64, 0x96, 0x91, 0xD6, 0x95, 0x22, 0xD0, 0xB5, 0x6A, 0x8B, 0x1D, 0x78, 0x61, 0x28, 0x02, 0xD4,
    0x6B, 0xF4, 0x1F, 0x3A, 0x42, 0x91, 0x1D, 0xFE
};
static const uint8_t ARIA_CTR_192_1_iv[] =
{
    0xFD, 0x8D, 0xE4, 0xB8, 0xE2, 0xE5, 0x0E, 0xEB, 0x7B, 0xC7, 0x9B, 0xE4, 0xEC, 0x9C, 0xE6, 0xE5
};
static const uint8_t ARIA_CTR_192_1_pt[] =
{
    0x83, 0x46, 0xD7, 0x83, 0x91, 0xA6, 0x5B, 0x4D, 0x78, 0x62, 0x11, 0xAE, 0x06, 0xE5, 0x47, 0x15
};
static const uint8_t ARIA_CTR_192_1_ct[] =
{
    0xC9, 0xD8, 0x90, 0x58, 0x8E, 0xE4, 0x1B, 0xF5, 0x1D, 0xD0, 0x70, 0xEC, 0x74, 0x6B, 0x15, 0x2E
};

static const uint8_t ARIA_CTR_192_2_key[] =
{
    0x28, 0xA4, 0x5A, 0xA7, 0xA5, 0x6A, 0x89, 0xD4, 0x32, 0xCE, 0x86, 0xA9, 0x92, 0x4E, 0x61, 0x43,
    0x80, 0x8E, 0xC2, 0xFC, 0xFE, 0x58, 0xDB, 0x90
};
static const uint8_t ARIA_CTR_192_2_iv[] =
{
    0xEA, 0xE1, 0x9B, 0x7E, 0xD2, 0x4E, 0x24, 0xB1, 0x09, 0x76, 0x80, 0x60, 0xFA, 0xEA, 0x8F, 0x33
};
static const uint8_t ARIA_CTR_192_2_pt[] =
{
    0xBD, 0x33, 0x44, 0xA7, 0xE8, 0x5A, 0x11, 0xB3, 0xA8, 0xAE, 0xA0, 0xAC, 0x10, 0xC1, 0xC4, 0x66,
    0xB0, 0xB4, 0xAF, 0x9C, 0x6A, 0xB4, 0x65, 0x91, 0x81, 0xC5, 0x71, 0xF8, 0xF3, 0xBE, 0xD6, 0x11,
    0x09, 0x95, 0x48, 0x16, 0x2C, 0xDD, 0x9C, 0xD5, 0xFA, 0x8D, 0x7E, 0x9E, 0x99, 0x03, 0x62, 0x5F
};
static const uint8_t ARIA_CTR_192_2_ct[] =
{
    0x7B, 0x19, 0x3B, 0x7C, 0x88, 0x99, 0xF4, 0x36, 0x94, 0x2C, 0xBB, 0x59, 0xA3, 0x4C, 0x42, 0xED,
    0x6E, 0x45, 0xC3, 0xB1, 0xCB, 0xB2, 0x9E, 0xE3, 0x0E, 0x93, 0x28, 0x07, 0x44, 0x65, 0xC6, 0xB1,
    0xE2, 0x4A, 0xFA, 0x4A, 0xA1, 0x67, 0x47, 0x72, 0xA2, 0x28, 0x95, 0xB9, 0x13, 0x2B, 0x98, 0x80
};

/* - ARIA256(CTR)MMT */
static const uint8_t ARIA_CTR_256_1_key[] =
{
    0x67, 0x93, 0x5A, 0xD6, 0xDF, 0x79, 0xF8, 0x46, 0xEA, 0x87, 0x89, 0xA4, 0xDC, 0xCA, 0xB3, 0x8E,
    0x2D, 0xED, 0x46, 0x1F, 0xDA, 0x0A, 0xC4, 0x64, 0x08, 0xC0, 0x21, 0x14, 0xEA, 0x86, 0xC7, 0x7B
};
static const uint8_t ARIA_CTR_256_1_iv[] =
{
    0xE6, 0x3E, 0xDD, 0xB8, 0xF3, 0x0E, 0xC9, 0x28, 0x46, 0x6B, 0xF3, 0x22, 0x2D, 0x76, 0xCC, 0xE5
};
static const uint8_t ARIA_CTR_256_1_pt[] =
{
    0x3B, 0x8C, 0x10, 0xCE, 0xA5, 0x18, 0x48, 0xCF, 0xEC, 0xAC, 0x92, 0x1C, 0xBC, 0x51, 0xA4, 0x27
};
static const uint8_t ARIA_CTR_256_1_ct[] =
{
    0xA3, 0xB5, 0x9E, 0x61, 0x32, 0x71, 0x05, 0xF3, 0xA5, 0x92, 0x73, 0xA5, 0x2E, 0xFC, 0x2E, 0x4C
};

static const uint8_t ARIA_CTR_256_2_key[] =
{
    0x15, 0x1B, 0x0F, 0xCF, 0xA8, 0x00, 0xC1, 0xD4, 0x84, 0xE9, 0xCE, 0x8F, 0xF1, 0x0A, 0x70, 0xE0,
    0x9C, 0x6F, 0x4C, 0x67, 0xF5, 0xD8, 0xF6, 0xF7, 0xD6, 0xC7, 0xB8, 0x47, 0x65, 0xD7, 0x8F, 0xEB
};
static const uint8_t ARIA_CTR_256_2_iv[] =
{
    0x3B, 0x4F, 0x77, 0x83, 0xC6, 0xF7, 0xE8, 0x33, 0xEB, 0x2B, 0xA1, 0x53, 0xF0, 0x2D, 0xA4, 0x67
};
static const uint8_t ARIA_CTR_256_2_pt[] =
{
    0x9A, 0xC0, 0x82, 0x51, 0x92, 0xF2, 0xD8, 0xC5, 0x0D, 0x39, 0x1A, 0xFE, 0xAC, 0xC1, 0x8F, 0xAF,
    0xA1, 0x05, 0x9D, 0x3D, 0x14, 0x9D, 0x47, 0x2B, 0xC5, 0x56, 0xF4, 0xD6, 0xF2, 0x88, 0x72, 0x62
};
static const uint8_t ARIA_CTR_256_2_ct[] =
{
    0x26, 0x34, 0x27, 0x9F, 0x30, 0x6C, 0x1D, 0xD3, 0xCA, 0x37, 0xE0, 0x26, 0xF9, 0xE7, 0x8F, 0xF4,
    0x00, 0xB5, 0x5A, 0x2E, 0xCD, 0xAE, 0xB8, 0xFC, 0x55, 0x2D, 0x34, 0xE8, 0x9A, 0x74, 0xE8, 0x12
};

/* ICM */
/* - 128 */
static const uint8_t ARIA_ICM_128_1_key[] =
{
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
};
static const uint8_t ARIA_ICM_128_1_iv[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static const uint8_t ARIA_ICM_128_1_pt[] =
{
    0x11, 0x11, 0x11, 0x11, 0xAA, 0xAA, 0xAA, 0xAA, 0x11, 0x11, 0x11, 0x11, 0xBB, 0xBB, 0xBB, 0xBB,
    0x11, 0x11, 0x11, 0x11, 0xCC, 0xCC, 0xCC, 0xCC, 0x11, 0x11, 0x11, 0x11, 0xDD, 0xDD, 0xDD, 0xDD,
    0x22, 0x22, 0x22, 0x22, 0xAA, 0xAA, 0xAA, 0xAA, 0x22, 0x22, 0x22, 0x22, 0xBB, 0xBB, 0xBB, 0xBB,
    0x22, 0x22, 0x22, 0x22, 0xCC, 0xCC, 0xCC, 0xCC, 0x22, 0x22, 0x22, 0x22, 0xDD, 0xDD, 0xDD, 0xDD,
    0x33, 0x33, 0x33, 0x33, 0xAA, 0xAA, 0xAA, 0xAA, 0x33, 0x33, 0x33, 0x33, 0xBB, 0xBB, 0xBB, 0xBB,
    0x33, 0x33, 0x33, 0x33, 0xCC, 0xCC, 0xCC, 0xCC, 0x33, 0x33, 0x33, 0x33, 0xDD, 0xDD, 0xDD, 0xDD,
    0x44, 0x44, 0x44, 0x44, 0xAA, 0xAA, 0xAA, 0xAA, 0x44, 0x44, 0x44, 0x44, 0xBB, 0xBB, 0xBB, 0xBB,
    0x44, 0x44, 0x44, 0x44, 0xCC, 0xCC, 0xCC, 0xCC, 0x44, 0x44, 0x44, 0x44, 0xDD, 0xDD, 0xDD, 0xDD,
    0x55, 0x55, 0x55, 0x55, 0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55, 0xBB, 0xBB, 0xBB, 0xBB,
    0x55, 0x55, 0x55, 0x55, 0xCC, 0xCC, 0xCC, 0xCC, 0x55, 0x55, 0x55, 0x55, 0xDD, 0xDD, 0xDD, 0xDD
};
static const uint8_t ARIA_ICM_128_1_ct[] =
{
    0xAC, 0x5D, 0x7D, 0xE8, 0x05, 0xA0, 0xBF, 0x1C, 0x57, 0xC8, 0x54, 0x50, 0x1A, 0xF6, 0x0F, 0xA1,
    0x14, 0x97, 0xE2, 0xA3, 0x45, 0x19, 0xDE, 0xA1, 0x56, 0x9E, 0x91, 0xE5, 0xB5, 0xCC, 0xAE, 0x2F,
    0xF3, 0xBF, 0xA1, 0xBF, 0x97, 0x5F, 0x45, 0x71, 0xF4, 0x8B, 0xE1, 0x91, 0x61, 0x35, 0x46, 0xC3,
    0x91, 0x11, 0x63, 0xC0, 0x85, 0xF8, 0x71, 0xF0, 0xE7, 0xAE, 0x5F, 0x2A, 0x08, 0x5B, 0x81, 0x85,
    0x1C, 0x2A, 0x3D, 0xDF, 0x20, 0xEC, 0xB8, 0xFA, 0x51, 0x90, 0x1A, 0xEC, 0x8E, 0xE4, 0xBA, 0x32,
    0xA3, 0x5D, 0xAB, 0x67, 0xBB, 0x72, 0xCD, 0x91, 0x40, 0xAD, 0x18, 0x8A, 0x96, 0x7A, 0xC0, 0xFB,
    0xBD, 0xFA, 0x94, 0xEA, 0x6C, 0xCE, 0x47, 0xDC, 0xF8, 0x52, 0x5A, 0xB5, 0xA8, 0x14, 0xCF, 0xEB,
    0x2B, 0xB6, 0x0E, 0xE2, 0xB1, 0x26, 0xE2, 0xD9, 0xD8, 0x47, 0xC1, 0xA9, 0xE9, 0x6F, 0x90, 0x19,
    0xE3, 0xE6, 0xA7, 0xFE, 0x40, 0xD3, 0x82, 0x9A, 0xFB, 0x73, 0xDB, 0x1C, 0xC2, 0x45, 0x64, 0x6A,
    0xDD, 0xB6, 0x2D, 0x9B, 0x90, 0x7B, 0xAA, 0xAF, 0xBE, 0x46, 0xA7, 0x3D, 0xBC, 0x13, 0x1D, 0x3D
};

/* - 192 */
static const uint8_t ARIA_ICM_192_1_key[] =
{
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77
};
static const uint8_t ARIA_ICM_192_1_iv[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static const uint8_t ARIA_ICM_192_1_pt[] =
{
    0x11, 0x11, 0x11, 0x11, 0xAA, 0xAA, 0xAA, 0xAA, 0x11, 0x11, 0x11, 0x11, 0xBB, 0xBB, 0xBB, 0xBB,
    0x11, 0x11, 0x11, 0x11, 0xCC, 0xCC, 0xCC, 0xCC, 0x11, 0x11, 0x11, 0x11, 0xDD, 0xDD, 0xDD, 0xDD,
    0x22, 0x22, 0x22, 0x22, 0xAA, 0xAA, 0xAA, 0xAA, 0x22, 0x22, 0x22, 0x22, 0xBB, 0xBB, 0xBB, 0xBB,
    0x22, 0x22, 0x22, 0x22, 0xCC, 0xCC, 0xCC, 0xCC, 0x22, 0x22, 0x22, 0x22, 0xDD, 0xDD, 0xDD, 0xDD,
    0x33, 0x33, 0x33, 0x33, 0xAA, 0xAA, 0xAA, 0xAA, 0x33, 0x33, 0x33, 0x33, 0xBB, 0xBB, 0xBB, 0xBB,
    0x33, 0x33, 0x33, 0x33, 0xCC, 0xCC, 0xCC, 0xCC, 0x33, 0x33, 0x33, 0x33, 0xDD, 0xDD, 0xDD, 0xDD,
    0x44, 0x44, 0x44, 0x44, 0xAA, 0xAA, 0xAA, 0xAA, 0x44, 0x44, 0x44, 0x44, 0xBB, 0xBB, 0xBB, 0xBB,
    0x44, 0x44, 0x44, 0x44, 0xCC, 0xCC, 0xCC, 0xCC, 0x44, 0x44, 0x44, 0x44, 0xDD, 0xDD, 0xDD, 0xDD,
    0x55, 0x55, 0x55, 0x55, 0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55, 0xBB, 0xBB, 0xBB, 0xBB,
    0x55, 0x55, 0x55, 0x55, 0xCC, 0xCC, 0xCC, 0xCC, 0x55, 0x55, 0x55, 0x55, 0xDD, 0xDD, 0xDD, 0xDD
};
static const uint8_t ARIA_ICM_192_1_ct[] =
{
    0x08, 0x62, 0x5C, 0xA8, 0xFE, 0x56, 0x9C, 0x19, 0xBA, 0x7A, 0xF3, 0x76, 0x0A, 0x6E, 0xD1, 0xCE,
    0xF4, 0xD1, 0x99, 0x26, 0x3E, 0x99, 0x9D, 0xDE, 0x14, 0x08, 0x2D, 0xBB, 0xA7, 0x56, 0x0B, 0x79,
    0xA4, 0xC6, 0xB4, 0x56, 0xB8, 0x70, 0x7D, 0xCE, 0x75, 0x1F, 0x98, 0x54, 0xF1, 0x88, 0x93, 0xDF,
    0xDB, 0x3F, 0x4E, 0x5A, 0xFA, 0x53, 0x97, 0x33, 0xE6, 0xF1, 0xE7, 0x0B, 0x98, 0xBA, 0x37, 0x89,
    0x1F, 0x8F, 0x81, 0xE9, 0x5D, 0xF8, 0xEF, 0xC2, 0x6C, 0x7C, 0xE0, 0x43, 0x50, 0x4C, 0xB1, 0x89,
    0x58, 0xB8, 0x65, 0xE4, 0xE3, 0x16, 0xCD, 0x2A, 0xA1, 0xC9, 0x7F, 0x31, 0xBF, 0x23, 0xDC, 0x04,
    0x6E, 0xF3, 0x26, 0xB9, 0x5A, 0x69, 0x2A, 0x19, 0x1B, 0xA0, 0xF2, 0xA4, 0x1C, 0x5F, 0xE9, 0xAE,
    0x07, 0x0F, 0x23, 0x6F, 0xF7, 0x07, 0x8E, 0x70, 0x3B, 0x42, 0x66, 0x6C, 0xAA, 0xFB, 0xDD, 0x20,
    0xBA, 0xD7, 0x4A, 0xC4, 0xC2, 0x0C, 0x0F, 0x46, 0xC7, 0xCA, 0x24, 0xC1, 0x51, 0x71, 0x65, 0x75,
    0xC9, 0x47, 0xDA, 0x16, 0xC9, 0x0C, 0xFE, 0x1B, 0xF2, 0x17, 0xA4, 0x1C, 0xFE, 0xBE, 0x75, 0x31
};

/* - 256 */
static const uint8_t ARIA_ICM_256_1_key[] =
{
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
};
static const uint8_t ARIA_ICM_256_1_iv[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static const uint8_t ARIA_ICM_256_1_pt[] =
{
    0x11, 0x11, 0x11, 0x11, 0xAA, 0xAA, 0xAA, 0xAA, 0x11, 0x11, 0x11, 0x11, 0xBB, 0xBB, 0xBB, 0xBB,
    0x11, 0x11, 0x11, 0x11, 0xCC, 0xCC, 0xCC, 0xCC, 0x11, 0x11, 0x11, 0x11, 0xDD, 0xDD, 0xDD, 0xDD,
    0x22, 0x22, 0x22, 0x22, 0xAA, 0xAA, 0xAA, 0xAA, 0x22, 0x22, 0x22, 0x22, 0xBB, 0xBB, 0xBB, 0xBB,
    0x22, 0x22, 0x22, 0x22, 0xCC, 0xCC, 0xCC, 0xCC, 0x22, 0x22, 0x22, 0x22, 0xDD, 0xDD, 0xDD, 0xDD,
    0x33, 0x33, 0x33, 0x33, 0xAA, 0xAA, 0xAA, 0xAA, 0x33, 0x33, 0x33, 0x33, 0xBB, 0xBB, 0xBB, 0xBB,
    0x33, 0x33, 0x33, 0x33, 0xCC, 0xCC, 0xCC, 0xCC, 0x33, 0x33, 0x33, 0x33, 0xDD, 0xDD, 0xDD, 0xDD,
    0x44, 0x44, 0x44, 0x44, 0xAA, 0xAA, 0xAA, 0xAA, 0x44, 0x44, 0x44, 0x44, 0xBB, 0xBB, 0xBB, 0xBB,
    0x44, 0x44, 0x44, 0x44, 0xCC, 0xCC, 0xCC, 0xCC, 0x44, 0x44, 0x44, 0x44, 0xDD, 0xDD, 0xDD, 0xDD,
    0x55, 0x55, 0x55, 0x55, 0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55, 0xBB, 0xBB, 0xBB, 0xBB,
    0x55, 0x55, 0x55, 0x55, 0xCC, 0xCC, 0xCC, 0xCC, 0x55, 0x55, 0x55, 0x55, 0xDD, 0xDD, 0xDD, 0xDD
};
static const uint8_t ARIA_ICM_256_1_ct[] =
{
    0x30, 0x02, 0x6C, 0x32, 0x96, 0x66, 0x14, 0x17, 0x21, 0x17, 0x8B, 0x99, 0xC0, 0xA1, 0xF1, 0xB2,
    0xF0, 0x69, 0x40, 0x25, 0x3F, 0x7B, 0x30, 0x89, 0xE2, 0xA3, 0x0E, 0xA8, 0x6A, 0xA3, 0xC8, 0x8F,
    0x59, 0x40, 0xF0, 0x5A, 0xD7, 0xEE, 0x41, 0xD7, 0x13, 0x47, 0xBB, 0x72, 0x61, 0xE3, 0x48, 0xF1,
    0x83, 0x60, 0x47, 0x3F, 0xDF, 0x7D, 0x4E, 0x77, 0x23, 0xBF, 0xFB, 0x44, 0x11, 0xCC, 0x13, 0xF6,
    0xCD, 0xD8, 0x9F, 0x3B, 0xC7, 0xB9, 0xC7, 0x68, 0x14, 0x50, 0x22, 0xC7, 0xA7, 0x4F, 0x14, 0xD7,
    0xC3, 0x05, 0xCD, 0x01, 0x2A, 0x10, 0xF1, 0x60, 0x50, 0xC2, 0x3F, 0x1A, 0xE5, 0xC2, 0x3F, 0x45,
    0x99, 0x8D, 0x13, 0xFB, 0xAA, 0x04, 0x1E, 0x51, 0x61, 0x95, 0x77, 0xE0, 0x77, 0x27, 0x64, 0x89,
    0x6A, 0x5D, 0x45, 0x16, 0xD8, 0xFF, 0xCE, 0xB3, 0xBF, 0x7E, 0x05, 0xF6, 0x13, 0xED, 0xD9, 0xA6,
    0x0C, 0xDC, 0xED, 0xAF, 0xF9, 0xCF, 0xCA, 0xF4, 0xE0, 0x0D, 0x44, 0x5A, 0x54, 0x33, 0x4F, 0x73,
    0xAB, 0x2C, 0xAD, 0x94, 0x4E, 0x51, 0xD2, 0x66, 0x54, 0x8E, 0x61, 0xC6, 0xEB, 0x0A, 0xA1, 0xCD
};


#endif /* INCLUDE_GUARD_TEST_VECTORS_AES_BASIC_DATA_H */

/* end of file testvectors_aria_basic_data.h */

