\page periodic_scan_handler Periodic Scanner Command Handler


# Introduction

In [Bluetooth&reg; Low Energy (BLE)](https://www.bluetooth.com/learn-about-bluetooth/tech-overview/), periodic advertising (without responses) refers to a feature in which an advertiser can broadcast data at regular intervals without expecting or allowing a response from receiving devices. This advertising method relies on the use of [extended advertising](rcl_glossary.html##extended-advertising) PDUs, and is particularly useful for applications requiring consistent data transmission over time, such as broadcasting sensor data, location beacons, or public information in a power-efficient manner. By not waiting for acknowledgments or replies, periodic advertising minimizes power consumption and reduces the complexity of maintaining a connection.

Periodic Advertising with Responses (PAwR) was introduced in Bluetooth Core Specification 5.4 to enable bi-directional communication between an advertiser and multiple scanners within a structured time frame. Unlike standard periodic advertising, PAwR allows devices to send responses within pre-defined response slots, enabling efficient communication while maintaining low power consumption.

A key feature of PAwR is that AUX_CONNECT_REQ PDUs are allowed, meaning that advertisers can act as initiators and trigger a transition from periodic advertising to an active connection with a particular scanner when needed. This enables both connectionless and connection-oriented communication models.

| PDU Type |        PDU Name       | Physical Channel       | LE 1M | LE 2M | LE Coded | Currently Supported | Used in            |
|:--------:|:---------------------:|------------------------|:-----:|:-----:|:--------:|:-------------------:|:-------------------:
|  0b0101  |    AUX_CONNECT_REQ    | Secondary              |   *   |   *   |     *    |          *          |  PAwR              |
|  0b0111  |      ADV_EXT_IND      | Primary                |   *   |       |     *    |          *          |  PA                |
|  0b0111  |      AUX_ADV_IND      | Secondary              |   *   |   *   |     *    |          *          |  PA                |
|  0b0111  |      AUX_SYNC_IND     | Periodic               |   *   |   *   |     *    |          *          |  PA                |
|  0b0111  |     AUX_CHAIN_IND     | Secondary and Periodic |   *   |   *   |     *    |          *          |  PA                |
|  0b0111  | AUX_SYNC_SUBEVENT_IND | Periodic               |   *   |   *   |     *    |          *          |  PAwR              |
|  0b0111  | AUX_SYNC_SUBEVENT_RSP | Periodic               |   *   |   *   |     *    |          *          |  PAwR              |
|  0b1000  |    AUX_CONNECT_RSP    | Secondary              |   *   |   *   |     *    |          *          |  PAwR              |

# Usage

The periodic scanner command handler is tailored for both periodic advertising with responses and periodic advertising without responses. It can receive indication PDUs as detailed in the table of supported PDU types above. Unlike the general scanner command handler, it does not send scan requests (SCAN_REQ or AUX_SCAN_REQ) or receive scan responses (SCAN_RSP or AUX_SCAN_RSP).

## Periodic Advertising Establishment
To receive periodic advertisements, a scanner device must perform a procedure known as periodic advertising establishment. During this procedure, the advertiser device sends an ADV_EXT_IND PDU pointing to an AUX_ADV_IND that contains the SyncInfo needed by the scanner to synchronize with the periodic advertising train.

The AUX_ADV_IND may include an [ACAD](rcl_glossary.html##acad) field with an [AD Structure](rcl_glossary.html##ad-structure) corresponding to the [Periodic Advertising Response Timing Information](rcl_glossary.html##response-timing-info). The scanner can use this information to determine whether it is periodic advertising with responses.

To perform periodic advertising establishment, follow these basic steps:
1. Initialize the RCL (See ::RCL_init) and provide a handle (See ::RCL_open).
2. Initialize and configure a ::RCL_CMD_BLE5_SCANNER_t command for periodic advertising establishment.
3. Provide the necessary Multibuffer(s) such that they have been initialized and configured to receive the necessary syncInfo needed for the periodic advertising.
4. Submit the command with ::RCL_Command_submit, and either use ::RCL_Command_pend or a callback to wait for the command's conclusion and proceed with any additional tasks such as post-processing or the submitting of new commands.

## Periodic Advertising without Responses
When configured to do Periodic Advertising (without responses), the periodic scanner command handler can only receive AUX_SYNC_IND and AUX_CHAIN_IND PDUs and if needed follow the AuxPtrs in them.

![Example of periodic advertising events from the same advertising set](docs/rcl/source/images/periodic_scanning_example.png)

If the execution of the ::RCL_CMD_BLE5_SCANNER_t is successful, it can be assumed that the periodic advertisement establishment is done, and the user can proceed with the reception of the periodic advertising events.
5. Initialize and configure a ::RCL_CMD_BLE5_PER_SCANNER_t command considering the received [SyncInfo](rcl_glossary.html##sync-info).
6. Use the reception time of the AUX_ADV_IND to calculate the [syncPacketWindowOffset](rcl_glossary.html##sync-packet-window-offset) and set up the start time of the ::RCL_CMD_BLE5_PER_SCANNER_t command.
7. Provide the necessary Multibuffer(s) such that they have been initialized and configured to receive the advertising data.
8. Submit the command ::RCL_Command_submit, and either use ::RCL_Command_pend or a callback to wait for the command's conclusion and proceed with any additional tasks such as post-processing or the submitting of new commands.
9. Rely on the start time of the previous ::RCL_CMD_BLE5_PER_SCANNER_t command and the received periodic advertising interval to schedule new periodic advertising commands.

The following code snippet shows how to establish a periodic advertising as a scanner:

\snippet source/snippets/ble_example/ble_example.c periodicScanner_code_snippet

For demonstration purposes, a helper function that inspects the extended header flags of the received packet and stores it in a C struct is used to extract the information contained in the [SyncInfo](rcl_glossary.html##sync-info) field.

\snippet source/snippets/ble_example/ble_example.c getSyncInfoData_code_snippet

The scanning device can then use this information and the timestamp of the received AUX_ADV_IND (using the ::RCL_BLE5_getRxTimestamp API) to calculate the [syncPacketWindowOffset](rcl_glossary.html##sync-packet-window-offset) and setup the start time of the periodic scanner command.

It's worth mentioning that the PHY to be used for the periodic advertising event corresponds to the PHY that the advertiser used when it sent the AUX_ADV_IND. For this reason, the example also relies on the use of the ::RCL_BLE5_getRxStatus API to return the status byte of the received packet.

## Periodic Advertising With Responses
When configured to do periodic advertising with responses, the periodic scanner command handler can only receive AUX_SYNC_SUBEVENT_IND and AUX_CONNECT_REQ PDUs.

If an AUX_SYNC_SUBEVENT_IND is received, the caller of the command must process the received PDU and if needed use the ::RCL_CMD_BLE5_GENERIC_TX_t command to send the appropriate AUX_SYNC_SUBEVENT_RSP.

If an AUX_CONNECT_REQ is received, the command handler will automatically determine if it is the target of the connect request, and respond with an AUX_CONNECT_RSP if needed. In such case, the command handler will return a specific command status to indicate that a connection has been established.

![Example of periodic advertising with responses events from the same advertising set](docs/rcl/source/images/pawr_scanning_example.png)

If the execution of the ::RCL_CMD_BLE5_SCANNER_t is successful, it can be assumed that the periodic advertisement establishment is done, and the user can proceed with the reception of the periodic advertising events.
5. Initialize and configure a ::RCL_CMD_BLE5_PER_SCANNER_t command considering the received [SyncInfo](rcl_glossary.html##sync-info) and [ACAD](rcl_glossary.html##acad).
6. Use the reception time of the AUX_ADV_IND to calculate the [syncPacketWindowOffset](rcl_glossary.html##sync-packet-window-offset) and set up the start time of the ::RCL_CMD_BLE5_PER_SCANNER_t command.
7. Ensure the TX buffer is properly configured to contain an AUX_CONNECT_RSP PDU in case an AUX_CONNECT_REQ is received.
8. Provide the necessary Multibuffer(s) such that they have been initialized and configured to receive the advertising data.
9. Submit the command ::RCL_Command_submit, and either use ::RCL_Command_pend or a callback to wait for the command's conclusion and proceed with any additional tasks such as post-processing or the submitting of new commands.
10. If the ::RCL_CMD_BLE5_PER_SCANNER_t command concludes with ::RCL_CommandStatus_Connect, schedule the connection events with the advertiser.
11. If the ::RCL_CMD_BLE5_PER_SCANNER_t command concludes with ::RCL_CommandStatus_Finished and the received packet is an AUX_SYNC_SUBEVENT_IND, use the reception time of the AUX_SYNC_SUBEVENT_IND and the designated subevent and response slot to initialize and configure a ::RCL_CMD_BLE5_GENERIC_TX_t command to send the appropriate AUX_SYNC_SUBEVENT_RSP.
12. Use the reception time of the AUX_SYNC_SUBEVENT_IND, the periodic advertising subevent interval and the periodic advertising interval to schedule the listening for the next subevents.

The following code snippet shows how to establish a connection by using the PAwR scanner:

\snippet source/snippets/ble_example/ble_example.c pawrScannerConn_code_snippet

To enable PAwR, ensure the `perAdvType` field is set as demonstrated in the code snippet. The PAwR scanner requires a Tx buffer for the AUX_CONNECT_RSP (in case an AUX_CONNECT_REQ is received) and a Multibuffer for receiving AUX_SYNC_SUBEVENT_IND or AUX_CONNECT_REQ PDUs. Note that the AdvA in AUX_CONNECT_RSP corresponds to the scanner's device address, while the TargetA corresponds to the advertiser's device address.

The code snippet includes a helper function that configures the Tx buffer for AUX_CONNECT_RSP.
\snippet source/snippets/ble_example/ble_example.c setAuxConnRspBuffer_code_snippet

The following code snippet demonstrates how to use the periodic scanner command handler to receive AUX_SYNC_SUBEVENT_IND PDUs:
\snippet source/snippets/ble_example/ble_example.c pawrScanner_code_snippet

When the PAwR scanner receives an AUX_SYNC_SUBEVENT_IND, the application can schedule an ::RCL_CMD_BLE5_GENERIC_TX_t command to send an AUX_SYNC_SUBEVENT_RSP at the approprite response slot. The start time for this transmission command is calculated based on the reception time of the AUX_SYNC_SUBEVENT_IND and the [Periodic Advertising Response Timing Information](rcl_glossary.html##response-timing-info) extracted from the [ACAD](rcl_glossary.html##acad) field in the received AUX_ADV_IND. It is important to note that the access address for the AUX_SYNC_SUBEVENT_RSP must match the one contained in the [ACAD](rcl_glossary.html##acad). Additionally, the PHY used for transmitting the AUX_SYNC_SUBEVENT_RSP must be the same as the one used to receive the AUX_ADV_IND.

A helper function is provided to demonstrate how to configure the Tx buffer for AUX_SYNC_SUBEVENT_RSP.
\snippet source/snippets/ble_example/ble_example.c setAuxSyncSubEventRspBuffer_code_snippet

For demonstration purposes, a helper function is provided to inspect the extended header flags of the received packet and store the [response timing information](rcl_glossary.html##response-timing-info) from the [ACAD](rcl_glossary.html##acad) field into a C struct.
\snippet source/snippets/ble_example/ble_example.c subevent_timing_info_code_snippet
\snippet source/snippets/ble_example/ble_example.c getSubEventInfoData_code_snippet

# Architecture
The life cycle of the periodic scanner command handler is influenced by the type of periodic advertising event, whether it involves responses or not.

For periodic advertising without responses, the command handler requires a Multibuffer capable of receiving the entire chain of packets that constitute the periodic advertising event. Upon receiving a packet, the command handler checks for the presence of an AuxPtr and schedules a new listening window to receive subsequent packets if necessary.

For periodic advertising with responses, the command handler needs a TxBuffer for a AUX_CONNECT_RSP (in case an AUX_CONNECT_REQ is received) and a Multibuffer for receiving either AUX_SYNC_SUBEVENT_IND or AUX_CONNECT_REQ PDUs. Depending on the received PDU, the scanner command handler can transition the scanner from the periodic advertising state to a connection state.

It's worth mentioning that only non-connectable/non-scannable extended PDUs can be received with the periodic scanner command, and other PDUs will be rejected and the command will end indicating an Rx buffer corruption.

@startuml
skinparam defaultTextAlignment center
:Start;
:===Configure Radio Operation\n\n(FIFO config, set AA, set CRC Init);
if (//perAdvType// == 1) then (yes)
:===Additional Radio Configuration\n\n(set advCfg, set scanCfg, \nset ownA, set peerA, \nconfigure the handling of InitA, \ndisable filter lists);
if (number of valid Tx buffer == 0?) then (yes)
    :Set //cmdStatus// to //MissingTxBuffer//, raise\n//lastCmdDone// event;
    stop
else (no)
    if (failed to free up finished Tx buffer?) then (yes)
        :Set //cmdStatus// to //TxBufferCorruption//, raise\n//lastCmdDone// event;
        stop
    else (no)
        :Set txBuffer //state// to //Finished//, raise\n//txBufferFinished// event;
    endif
endif
endif
:===Prepare LRF\n\n(set start time, configure PBE regs,\nenable interrupts, etc.);
repeat :Start Rx Operation;
:Wait for LRF events;
switch (LRF event?)
    case ( rxOk )
        if (//perAdvType// == 1 ?) then (yes)
            if ( PDU type is AUX_SYNC_SUBEVENT_IND? ) then (yes)
                :Validate the extended header;
                if ( PDU has valid extended header? ) then (yes)
                    :Set //hasValidExtHdr// to true;
                    :Record the timestamp of reception;
                else (no)
                    :Set //hasValidExtHdr// to false;
                endif
            elseif (PDU type is AUX_CONNECT_REQ? ) then (yes)
                :Set //hasValidExtHdr// to true;
            else (no)
                :Set //hasValidExtHdr// to false;
            endif
        else (no)
            if ( PDU type is Extended? ) then (yes)
                :Extract AuxPtr from the received packet;
                if (//auxPtrPresent// == true ?) then (yes)
                    :Set //followAuxPtr// to true;
                else (no)
                    :Set //followAuxPtr// to false;
                endif
            else (no)
                :Set //hasValidExtHdr// to false;
            endif
        endif
        if (//hasValidExtHdr// == true ?) then (yes)
            :===Handle packet\n\n(commit packet, adjust FIFO size,\nraise //rxEntryAvail// event);
        else (no)
            :===Ignore packet\n\n(set //cmdStatus// to //RxBufferCorruption//,\nraise //lastCmdDone// event);
            stop
        endif
    case ( opError )
        :===Process error\n\n(find //endCause//, set //cmdStatus//,\nraise //lastCmdDone// event);
        :Set //followAuxPtr//\nto **FALSE**;
    case ( opDone )
        if (//perAdvType// == 1 AND //endCause// is CONNECT) then (yes)
            :Record the timestamp of reception;
            :Set //cmdStatus// to\n//Connect//, raise\n//lastCmdDone// event;
        elseif (//endCause// is ENDOK?) then (yes)
            if (//perAdvType// == 1) then (yes)
                :Set //followAuxPtr//\nto **FALSE**;
                :Set //cmdStatus// to\n//Finished//, raise\n//lastCmdDone// event;
            else
                if ( AuxPtr is present? ) then (yes)
                    :Set //followAuxPtr//\nto **TRUE**;
                else
                    :Set //followAuxPtr//\nto **FALSE**;
                    :Set //cmdStatus// to\n//Finished//, raise\n//lastCmdDone// event;
                endif
            endif
        else
            :Set //cmdStatus// to\n//endCause//, raise\n//lastCmdDone// event;
            :Set //followAuxPtr//\nto **FALSE**;
        endif
endswitch
: Update stats;
backward:===Process AuxPtr\n\n(update channel, set new start\ntime, prepare LRF, etc.);
repeat while (//perAdvType// == 0 AND //followAuxPtr//?) is (Yes) not (No)
: Done;
stop
@enduml


| RCL Event (In)              | Description                     |
|-----------------------------|---------------------------------|
| ::RCL_EventSetup            | Setup has been performed        |
| ::RCL_EventTimerStart       | Timer-based start signalled     |
| ::RCL_EventGracefulStop     | Graceful stop has been observed |
| ::RCL_EventHardStop         | Hard stop has been observed     |
| ::RCL_EventRxBufferUpdate   | RX buffer has been updated      |

| RCL Event (Out)             | Description                                                   |
|-----------------------------|---------------------------------------------------------------|
| ::RCL_EventLastCmdDone      | The RCL is finished with the command                          |
| ::RCL_EventCmdStarted       | Command handler has accepted and started executing            |
| ::RCL_EventRxBufferFinished | An RX multi-buffer is finished                                |
| ::RCL_EventRxEntryAvail     | An RX entry has been made available                           |
| ::RCL_EventTxBufferFinished | A TX buffer is finished                                       |


| LRF Event            | Description                                                       |
|----------------------|-------------------------------------------------------------------|
| ::LRF_EventOpDone    | The PBE operation has finished                                    |
| ::LRF_EventOpError   | Something went wrong. Cause located in the PBE ENDCAUSE register  |
| ::LRF_EventRxOk      | Packet received with CRC OK and not to be ignored by the MCU      |
| ::LRF_EventRxNok     | Packet received with CRC error                                    |
| ::LRF_EventRxIgnored | Packet received, but may be ignored by MCU                        |

