/*
 * Copyright (c) 2020-2023 Texas Instruments Incorporated - http://www.ti.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*!*****************************************************************************
 *  @file       LogSinkITM.h
 *  @brief      <b>PRELIMINARY</b> LogSinkITM interface
 *
 *  <b>WARNING</b> These APIs are <b>PRELIMINARY</b>, and subject to
 *  change in the next few months.
 *
 *  The LogSinkITM module is a sink that can be used in conjunction with the
 *  Log.h API. The API defined in this file should not be used directly, but
 *  is made available to the Logging framework and used as a transport layer
 *  for Log.h
 *
 *  To use the ITM sink, ensure that the correct library for your
 *  device is linked in and include this header file as follows:
 *  @code
 *  #include <ti/log/LogSinkITM.h>
 *  @endcode
 *
 *  Additionally, LogSinkITM_init must be called before LogSinkITM can be used.
 *  @code
 *  // Initialize ITM sink
 *  LogSinkITM_init();
 *  @endcode
 *
 *  This module implements two functions that are required by the Log API:
 *   - printf(const Log_Module *handle, uint32_t header, uint32_t index, uint32_t numArgs, ...);
 *   - buf(const Log_Module *handle, uint32_t header, uint32_t index, uint8_t *data, size_t size);
 *
 *  Whenever a log-statement is invoked, that uses LogSinkITM as its sink, the functions above are ultimately invoked.
 *
 *  @anchor ti_log_LogSinkITM_Overview
 *  # Overview
 *  LogSinkITM is a sink/transport layer that is able to output log-statements over ITM.
 *  It uses the ITM driver to stream data out onto a user-selectable pin, which can be received and processed
 *  by a host-side tool. For more information about the host-side tool, see tools/log/tiutils.
 *  For more information about the Log API, see the @ref log "Log documentation"
 *
 *  @anchor ti_log_LogSinkITM_Channels
 *  # ITM Channels
 *  LogSinkITM uses certain channels for different purposes. Seperate channels are used for data transfer,
 *  time synchronization, etc. For a complete overview, refer to LogSinkITM_StimulusPorts
 *
 *  @anchor ti_log_LogSinkITM_Timestamps
 *  # ITM Timestamps
 *  Timestamps are automatically generated by the ITM hardware, and a synchronization timestamp is sent
 *  over the time-sync channel when the ITM sink is initiated. The timestamp is taken from the RTC, but the format
 *  depends on the device. Refer to TimestampP.h for implementation details
 *
 *  ============================================================================
 */

#ifndef ti_log_LogSinkITM__include
#define ti_log_LogSinkITM__include

#include <stdint.h>
#include <ti/log/Log.h>

#if defined(__cplusplus)
extern "C" {
#endif

#define Log_TI_LOG_SINK_ITM_VERSION 0.1.0

/*
 * ======== LogSinkITM_StimulusPorts ========
 */
typedef enum
{
    LogSinkITM_STIM_RAW0 = 0, //!< Port 0. Reserved for future use
    LogSinkITM_STIM_RAW1,     //!< Port 1. Reserved for future use
    LogSinkITM_STIM_RAW2,     //!< Port 2. Reserved for future use
    LogSinkITM_STIM_RAW3,     //!< Port 3. Reserved for future use
    LogSinkITM_STIM_RAW4,     //!< Port 4. Reserved for future use
    LogSinkITM_STIM_RAW5,     //!< Port 5. Reserved for future use
    LogSinkITM_STIM_RAW6,     //!< Port 6. Reserved for future use
    LogSinkITM_STIM_RAW7,     //!< Port 7. Reserved for future use
    LogSinkITM_STIM_RAW8,     //!< Port 8. Reserved for future use
    LogSinkITM_STIM_RAW9,     //!< Port 9. Reserved for future use
    LogSinkITM_STIM_RAW10,    //!< Port 10. Reserved for future use
    LogSinkITM_STIM_RAW11,    //!< Port 11. Reserved for logger time sync
    LogSinkITM_STIM_RAW12,    //!< Port 12. Used to notify logger states
    LogSinkITM_STIM_RAW13,    //!< Port 13. Reserved for future use
    LogSinkITM_STIM_RAW14,    //!< Port 14. Reserved for logger header messages
    LogSinkITM_STIM_RAW15,    //!< Port 15. Reserved for logger main data transfer

    /* Ports 16-31 reserved for LogSinkITM and dependents  */
    LogSinkITM_STIM_RESV16 = 16, //!< Port 16. Reserved for raw ITM data
    LogSinkITM_STIM_RESV17,      //!< Port 17. Reserved for raw ITM data
    LogSinkITM_STIM_RESV18,      //!< Port 18. Reserved for raw ITM data
    LogSinkITM_STIM_RESV19,      //!< Port 19. Reserved for raw ITM data
    LogSinkITM_STIM_RESV20,      //!< Port 20. Reserved for raw ITM data
    LogSinkITM_STIM_RESV21,      //!< Port 21. Reserved for raw ITM data
    LogSinkITM_STIM_RESV22,      //!< Port 22. Reserved for raw ITM data
    LogSinkITM_STIM_RESV23,      //!< Port 23. Reserved for raw ITM data
    LogSinkITM_STIM_RESV24,      //!< Port 24. Reserved for raw ITM data
    LogSinkITM_STIM_RESV25,      //!< Port 25. Reserved for raw ITM data
    LogSinkITM_STIM_RESV26,      //!< Port 26. Reserved for raw ITM data
    LogSinkITM_STIM_RESV27,      //!< Port 27. Reserved for raw ITM data
    LogSinkITM_STIM_TRACE,       //!< Port 28. Reserved for logger main data transfer
    LogSinkITM_STIM_HEADER,      //!< Port 29. Reserved for logger header messages
    LogSinkITM_STIM_TIME_SYNC,   //!< Port 30. Reserved for logger time sync
    LogSinkITM_STIM_INFO         //!< Port 31. Reserved for logger control/info packets
} LogSinkITM_StimulusPorts;

/*
 * ======== LogSinkITM_ControlWord ========
 */
typedef enum LogSinkITM_ControlWord
{
    LogSinkITM_Info_Timing = 3,
} LogSinkITM_ControlWord;

/*
 *  ======== LogSinkITM_Instance ========
 */
typedef struct LogSinkITM_Instance
{
    uint32_t serial;
} LogSinkITM_Instance;

/*
 *  ======== LogSinkITM_Handle ========
 */
typedef LogSinkITM_Instance *LogSinkITM_Handle;

/*
 *  ======== LogSinkITM_init ========
 */
extern void LogSinkITM_init(void);

/*
 *  ======== LogSinkITM_finalize ========
 */
extern void LogSinkITM_finalize(void);

/*
 *  ======== LogSinkITM_printf ========
 */
extern void ti_log_LogSinkITM_printf(const Log_Module *handle, uint32_t header, uint32_t index, uint32_t numArgs, ...);

/*
 *  ======== LogSinkITM_buf ========
 */
extern void ti_log_LogSinkITM_buf(const Log_Module *handle,
                                  uint32_t header,
                                  uint32_t index,
                                  uint8_t *data,
                                  size_t size);

/*
 * Helpers to define/use instance. ITM is a singleton so no arguments are taken.
 */
#define Log_SINK_ITM_DEFINE() LogSinkITM_Instance LogSinkITM_singletonConfig = {.serial = 0}
#define Log_SINK_ITM_USE()    extern LogSinkITM_Instance LogSinkITM_singletonConfig
#define Log_MODULE_INIT_SINK_ITM(name, _levels)                                                                      \
    {                                                                                                                \
        .sinkConfig = &LogSinkITM_singletonConfig, .printf = ti_log_LogSinkITM_printf, .buf = ti_log_LogSinkITM_buf, \
        .levels = _levels,                                                                                           \
    }

_Log_DEFINE_LOG_VERSION(LogSinkITM, Log_TI_LOG_SINK_ITM_VERSION);

#if defined(__cplusplus)
}
#endif

#endif /* ti_log_LogSinkITM__include */
