/*
 * Copyright (c) 2018-2022, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*!***************************************************************************
 *  @file       AESCBC.h
 *
 *  @brief      AESCBC driver header
 *
 *  @anchor ti_drivers_AESCBC_Overview
 *  # Overview #
 *  The Cipher Block Chaining (CBC) mode of operation is a generic
 *  block cipher mode of operation. It can be used with any block cipher
 *  including AES.
 *
 *  CBC mode encrypts messages of any practical length that have a length
 *  evenly divisible by the block size. Unlike ECB, it guarantees
 *  confidentiality of the entire message when the message is larger than
 *  one block.
 *
 *  ## Operation #
 *  In CBC encryption, the initialization vector (IV) is XOR'd with a block of
 *  plaintext and then encrypted. The output ciphertext block is then XOR'd with
 *  the next plaintext block and the result is encrypted. This process is repeated
 *  until the final block of plaintext has been encrypted.
 *
 *  To decrypt the message, decrypt the first block of ciphertext and XOR the result
 *  with the IV. The result is the first plaintext block. For subsequent ciphertext
 *  blocks, decrypt each block and XOR the previous block of the encrypted message
 *  into the result.
 *
 *  ## Padding #
 *  CBC operates on entire blocks of ciphertext and plaintext at a time. This
 *  means that message lengths must be a multiple of the block cipher block size.
 *  AES has a block size of 16 bytes no matter the key size. Since messages do
 *  not necessarily always have a length that is a multiple of 16 bytes, it may
 *  be necessary to pad the message to a 16-byte boundary. Padding requires
 *  the sender and receiver to implicitly agree on the padding convention.
 *  Improperly designed or implemented padding schemes may leak information
 *  to an attacker through a padding oracle attack for example.
 *
 *  ## Initialization Vectors #
 *  The IV is generated by the party performing the encryption operation.
 *  Within the scope of any encryption key, the IV value must be unique.
 *  The IV does not need to be kept secret and is usually transmitted together
 *  with the ciphertext to the decrypting party.
 *  In CBC mode, the IVs must not be predictable. Two recommended ways to
 *  generate IVs is to either:
 *
 *  - Apply the block cipher (AESCBC), using the same key used with CBC,
 *    to a nonce. This nonce must be unique for each key-message pair.
 *    A counter will usually suffice. If the same symmetric key is used
 *    by both parties to encrypt messages, they should agree to use a
 *    nonce scheme that avoids generating the same nonce and thus IV twice.
 *    Incrementing the counter by two and making one party use even numbers
 *    and the other odd numbers is a common method to avoid such collisions.
 *  - Use a TRNG (True Random Number Generator) or PRNG
 *    (Pseudo-Random Number Generator) to generate a random number for use
 *    as IV.
 *
 *  ## Drawbacks #
 *  CBC mode has several drawbacks. Unless interfacing with legacy devices,
 *  it is recommended to use an AEAD (Authenticated Encryption with Associated Data)
 *  mode such as CCM or GCM. Below is a non-exhaustive list of reasons to use
 *  a different block cipher mode of operation.
 *
 *  - CBC mode does not offer authentication or integrity guarantees. In practice,
 *    this means that attackers can intercept the encrypted message and manipulate
 *    the ciphertext before sending the message on to the receiver. While this
 *    does not break confidentiality and reveal the plaintext, it has enabled several
 *    attacks in the past. This is especially problematic given that changing the
 *    ciphertext of a block will only corrupt the block itself and the subsequent
 *    block of resultant plaintext. This property may be used to manipulate only
 *    certain parts of the message.
 *
 *  - CBC mode requires message lengths to be evenly divisible by the block size.
 *    This necessitates a padding scheme. Improperly implemented padding schemes
 *    may lead to vulnerabilities that can be exploited by attackers. It often
 *    makes more sense to use a dedicated stream cipher such as CTR (Counter) that
 *    does not have this restriction. CCM and GCM both use CTR for encryption.
 *
 *  @anchor ti_drivers_AESCBC_Usage
 *  # Usage #
 *  ## Before starting a CBC operation #
 *
 *  Before starting a CBC operation, the application must do the following:
 *      - Call #AESCBC_init() to initialize the driver
 *      - Call #AESCBC_Params_init() to initialize the #AESCBC_Params to default values.
 *      - Modify the #AESCBC_Params as desired
 *      - Call #AESCBC_open() to open an instance of the driver
 *      - Initialize a CryptoKey. These opaque data structures are representations
 *        of keying material and its storage. Depending on how the keying material
 *        is stored (RAM or flash, key store), the CryptoKey must be
 *        initialized differently. The AESCBC API can handle all types of CryptoKey.
 *        However, not all device-specific implementations support all types of CryptoKey.
 *        Devices without a key store will not support CryptoKeys with keying material
 *        stored in a key store for example.
 *        All devices support plaintext CryptoKeys.
 *      - Initialize the appropriate AESCBC operation struct using the relevant
 *        operation init functions and set all fields. For example, one-step (one-shot
 *        or single call) operations should initialize AESCBC_Operation or
 *        AESCBC_OneStepOperation using AESCBC_Operation_init() or
 *        AESCBC_OneStepOperation_init(). For multi-step (segmented or multiple call)
 *        operations, AESCBC_SegmentedOperation must be initialized and set.
 *
 *  ## Starting a CBC operation #
 *
 *  The #AESCBC_oneStepEncrypt and #AESCBC_oneStepDecrypt functions perform a CBC operation
 *  in a single call. They will always be the most highly optimized routines with the
 *  least overhead and the fastest runtime. However, they require all plaintext
 *  or ciphertext to be available to the function at the start of the call.
 *  All devices support single call operations.
 *
 *  ## After the CBC operation completes #
 *
 *  After the CBC operation completes, the application should either start
 *  another operation or close the driver by calling #AESCBC_close().
 *
 *  @anchor ti_drivers_AESCBC_Synopsis
 *  ## Synopsis
 *  @anchor ti_drivers_AESCBC_Synopsis_Code
 *  @code
 *  // Import AESCBC Driver definitions
 *  #include <ti/drivers/AESCBC.h>
 *
 *  // Define name for AESCBC channel index
 *  #define AESCBC_INSTANCE 0
 *
 *  AESCBC_init();
 *
 *  handle = AESCBC_open(AESCBC_INSTANCE, NULL);
 *
 *  // Initialize symmetric key
 *  CryptoKeyPlaintext_initKey(&cryptoKey, keyingMaterial, sizeof(keyingMaterial));
 *
 *  // Set up AESCBC_Operation
 *  AESCBC_OneStepOperation operation;
 *  AESCBC_OneStepOperation_init(&operation);
 *  operation.key               = &cryptoKey;
 *  operation.input             = plaintext;
 *  operation.output            = ciphertext;
 *  operation.inputLength       = sizeof(plaintext);
 *  operation.iv                = iv;
 *
 *  encryptionResult = AESCBC_oneStepEncrypt(handle, &operation);
 *
 *  AESCBC_close(handle);
 *  @endcode
 *
 *  @anchor ti_drivers_AESCBC_Examples
 *  ## Examples
 *
 *  ### Single call CBC encryption with plaintext CryptoKey in blocking return mode #
 *  @code
 *
 *  #include <ti/drivers/AESCBC.h>
 *  #include <ti/drivers/cryptoutils/cryptokey/CryptoKeyPlaintext.h>
 *
 *  ...
 *
 *  AESCBC_Handle handle;
 *  CryptoKey cryptoKey;
 *  int_fast16_t encryptionResult;
 *
 *  // For example purposes only. Generate IVs in a non-static way in practice.
 *  // Test vector 0 from NIST CAPV set CBCMMT128
 *  uint8_t iv[16] =                {0x2f, 0xe2, 0xb3, 0x33, 0xce, 0xda, 0x8f, 0x98,
 *                                   0xf4, 0xa9, 0x9b, 0x40, 0xd2, 0xcd, 0x34, 0xa8};
 *  uint8_t plaintext[16] =         {0x45, 0xcf, 0x12, 0x96, 0x4f, 0xc8, 0x24, 0xab,
 *                                   0x76, 0x61, 0x6a, 0xe2, 0xf4, 0xbf, 0x08, 0x22};
 *  uint8_t ciphertext[sizeof(plaintext)];
 *  uint8_t keyingMaterial[16] =    {0x1f, 0x8e, 0x49, 0x73, 0x95, 0x3f, 0x3f, 0xb0,
 *                                   0xbd, 0x6b, 0x16, 0x66, 0x2e, 0x9a, 0x3c, 0x17};
 *
 *  // The ciphertext should be the following after the encryption operation:
 *  //  0x0f, 0x61, 0xc4, 0xd4, 0x4c, 0x51, 0x47, 0xc0
 *  //  0x3c, 0x19, 0x5a, 0xd7, 0xe2, 0xcc, 0x12, 0xb2
 *
 *
 *  handle = AESCBC_open(0, NULL);
 *
 *  if (handle == NULL) {
 *      // handle error
 *  }
 *
 *  CryptoKeyPlaintext_initKey(&cryptoKey, keyingMaterial, sizeof(keyingMaterial));
 *
 *  AESCBC_OneStepOperation operation;
 *  AESCBC_OneStepOperation_init(&operation);
 *
 *  operation.key               = &cryptoKey;
 *  operation.input             = plaintext;
 *  operation.output            = ciphertext;
 *  operation.inputLength       = sizeof(plaintext);
 *  operation.iv                = iv;
 *
 *  encryptionResult = AESCBC_oneStepEncrypt(handle, &operation);
 *
 *  if (encryptionResult != AESCBC_STATUS_SUCCESS) {
 *      // handle error
 *  }
 *
 *  AESCBC_close(handle);
 *
 *  @endcode
 *
 *  ### Single call CBC decryption with plaintext CryptoKey in callback return mode #
 *
 *  @note The following code example presented uses a 256-bit key. However,
 *        CC13x1/CC26x1 and CC23x0 only support a maximum key size of 128-bits,
 *        so reduction of keyingMaterial would be required.
 *
 *  @code
 *
 *  #include <ti/drivers/AESCBC.h>
 *  #include <ti/drivers/cryptoutils/cryptokey/CryptoKeyPlaintext.h>
 *
 *  ...
 *
 *  // Test vector 0 from NIST CAPV set CBCMMT256
 *
 *  uint8_t iv[16] =                {0xdd, 0xbb, 0xb0, 0x17, 0x3f, 0x1e, 0x2d, 0xeb,
 *                                   0x23, 0x94, 0xa6, 0x2a, 0xa2, 0xa0, 0x24, 0x0e};
 *  uint8_t ciphertext[16] =        {0xd5, 0x1d, 0x19, 0xde, 0xd5, 0xca, 0x4a, 0xe1,
 *                                   0x4b, 0x2b, 0x20, 0xb0, 0x27, 0xff, 0xb0, 0x20};
 *  uint8_t keyingMaterial[32] =    {0x43, 0xe9, 0x53, 0xb2, 0xae, 0xa0, 0x8a, 0x3a,
 *                                   0xd5, 0x2d, 0x18, 0x2f, 0x58, 0xc7, 0x2b, 0x9c,
 *                                   0x60, 0xfb, 0xe4, 0xa9, 0xca, 0x46, 0xa3, 0xcb,
 *                                   0x89, 0xe3, 0x86, 0x38, 0x45, 0xe2, 0x2c, 0x9e};
 *  uint8_t plaintext[sizeof(ciphertext)];
 *
 *  // The plaintext should be the following after the decryption operation:
 *  //  0x07, 0x27, 0x0d, 0x0e, 0x63, 0xaa, 0x36, 0xda
 *  //  0xed, 0x8c, 0x6a, 0xde, 0x13, 0xac, 0x1a, 0xf1
 *
 *
 *  void cbcCallback(AESCBC_Handle handle,
 *                   int_fast16_t returnValue,
 *                   AESCBC_OperationUnion *operation,
 *                   AESCBC_OperationType operationType) {
 *
 *      if (returnValue != AESCBC_STATUS_SUCCESS) {
 *          // handle error
 *      }
 *
 *      if (operationType == AESCBC_OPERATION_TYPE_DECRYPT ||
 *          operationType == AESCBC_OPERATION_TYPE_ENCRYPT) {
 *          // do something with operation->oneStepOperation
 *      } else {
 *          // do something with operation->segmentedOperation
 *      }
 *  }
 *
 *  AESCBC_OneStepOperation operation;
 *
 *  void cbcStartFunction(void) {
 *      AESCBC_Handle handle;
 *      AESCBC_Params params;
 *      CryptoKey cryptoKey;
 *      int_fast16_t decryptionResult;
 *
 *      AESCBC_Params_init(&params);
 *      params.returnBehavior = AESCBC_RETURN_BEHAVIOR_CALLBACK;
 *      params.callbackFxn = cbcCallback;
 *
 *      handle = AESCBC_open(0, &params);
 *
 *      if (handle == NULL) {
 *          // handle error
 *      }
 *
 *      CryptoKeyPlaintext_initKey(&cryptoKey, keyingMaterial, sizeof(keyingMaterial));
 *
 *      AESCBC_OneStepOperation_init(&operation);
 *
 *      operation.key               = &cryptoKey;
 *      operation.input             = ciphertext;
 *      operation.output            = plaintext;
 *      operation.inputLength       = sizeof(ciphertext);
 *      operation.iv                = iv;
 *
 *      decryptionResult = AESCBC_oneStepDecrypt(handle, &operation);
 *
 *      if (decryptionResult != AESCBC_STATUS_SUCCESS) {
 *          // handle error
 *      }
 *
 *      // do other things while CBC operation completes in the background
 *  }
 *
 *  @endcode
 *
 *  ### Multi-step CBC encryption with plaintext CryptoKey in blocking return mode #
 *  @code
 *
 *  #include <ti/drivers/AESCBC.h>
 *  #include <ti/drivers/cryptoutils/cryptokey/CryptoKeyPlaintext.h>
 *
 *  ...
 *
 *  #define AES_BLOCK_SIZE      16 // bytes
 *
 *  AESCBC_Handle handle;
 *  CryptoKey cryptoKey;
 *  int_fast16_t encryptionResult;
 *
 *  // For example purposes only. Generate IVs in a non-static way in practice.
 *  // Test vector 0 from NIST CAPV set CBCMMT128
 *  uint8_t iv[16] =                {0x2f, 0xe2, 0xb3, 0x33, 0xce, 0xda, 0x8f, 0x98,
 *                                   0xf4, 0xa9, 0x9b, 0x40, 0xd2, 0xcd, 0x34, 0xa8};
 *  uint8_t plaintext[16] =         {0x45, 0xcf, 0x12, 0x96, 0x4f, 0xc8, 0x24, 0xab,
 *                                   0x76, 0x61, 0x6a, 0xe2, 0xf4, 0xbf, 0x08, 0x22};
 *  uint8_t ciphertext[sizeof(plaintext)];
 *  uint8_t keyingMaterial[16] =    {0x1f, 0x8e, 0x49, 0x73, 0x95, 0x3f, 0x3f, 0xb0,
 *                                   0xbd, 0x6b, 0x16, 0x66, 0x2e, 0x9a, 0x3c, 0x17};
 *
 *  // The ciphertext should be the following after the encryption operation:
 *  //  0x0f, 0x61, 0xc4, 0xd4, 0x4c, 0x51, 0x47, 0xc0
 *  //  0x3c, 0x19, 0x5a, 0xd7, 0xe2, 0xcc, 0x12, 0xb2
 *
 *
 *  handle = AESCBC_open(0, NULL);
 *
 *  if (handle == NULL) {
 *      // handle error
 *  }
 *
 *  CryptoKeyPlaintext_initKey(&cryptoKey, keyingMaterial, sizeof(keyingMaterial));
 *
 *  AESCBC_SegmentedOperation operation;
 *  AESCBC_SegmentedOperation_init(&operation);
 *
 *  operation.input             = plaintext;
 *  operation.output            = ciphertext;
 *  operation.inputLength       = sizeof(plaintext);
 *
 *  encryptionResult = AESCBC_setupEncrypt(handle, &cryptoKey);
 *  if (encryptionResult != AESCBC_STATUS_SUCCESS) {
 *      // handle error
 *  }
 *
 *  encryptionResult = AESCBC_setIV(handle, iv, AES_BLOCK_SIZE);
 *  if (encryptionResult != AESCBC_STATUS_SUCCESS) {
 *      // handle error
 *  }
 *
 *  encryptionResult = AESCBC_addData(handle, &operation);
 *  if (encryptionResult != AESCBC_STATUS_SUCCESS) {
 *      // handle error
 *  }
 *
 *  operation.inputLength = 0;
 *  encryptionResult = AESCBC_finalize(handle, &operation);
 *  if (encryptionResult != AESCBC_STATUS_SUCCESS) {
 *      // handle error
 *  }
 *
 *  AESCBC_close(handle);
 *
 *  @endcode
 *
 *  ### Multi-step CBC decryption with plaintext CryptoKey in callback return mode #
 *
 *  @note The following code example presented uses a 256-bit key. However,
 *        CC13x1/CC26x1 and CC23x0 only support a maximum key size of 128-bits,
 *        so reduction of keyingMaterial would be required.
 *
 *  @code
 *
 *  #include <ti/drivers/AESCBC.h>
 *  #include <ti/drivers/cryptoutils/cryptokey/CryptoKeyPlaintext.h>
 *
 *  ...
 *
 *  #define AES_BLOCK_SIZE      16 // bytes
 *
 *  // Test vector 0 from NIST CAPV set CBCMMT256
 *
 *  uint8_t iv[16] =                {0xdd, 0xbb, 0xb0, 0x17, 0x3f, 0x1e, 0x2d, 0xeb,
 *                                   0x23, 0x94, 0xa6, 0x2a, 0xa2, 0xa0, 0x24, 0x0e};
 *  uint8_t ciphertext[16] =        {0xd5, 0x1d, 0x19, 0xde, 0xd5, 0xca, 0x4a, 0xe1,
 *                                   0x4b, 0x2b, 0x20, 0xb0, 0x27, 0xff, 0xb0, 0x20};
 *  uint8_t keyingMaterial[32] =    {0x43, 0xe9, 0x53, 0xb2, 0xae, 0xa0, 0x8a, 0x3a,
 *                                   0xd5, 0x2d, 0x18, 0x2f, 0x58, 0xc7, 0x2b, 0x9c,
 *                                   0x60, 0xfb, 0xe4, 0xa9, 0xca, 0x46, 0xa3, 0xcb,
 *                                   0x89, 0xe3, 0x86, 0x38, 0x45, 0xe2, 0x2c, 0x9e};
 *  uint8_t plaintext[sizeof(ciphertext)];
 *
 *  // The plaintext should be the following after the decryption operation:
 *  //  0x07, 0x27, 0x0d, 0x0e, 0x63, 0xaa, 0x36, 0xda
 *  //  0xed, 0x8c, 0x6a, 0xde, 0x13, 0xac, 0x1a, 0xf1
 *
 *
 *  void cbcCallback(AESCBC_Handle handle,
 *                   int_fast16_t returnValue,
 *                   AESCBC_OperationUnion *operation,
 *                   AESCBC_OperationType operationType) {
 *
 *      if (returnValue != AESCBC_STATUS_SUCCESS) {
 *          // handle error
 *      }
 *
 *      if (operationType == AESCBC_OPERATION_TYPE_DECRYPT ||
 *          operationType == AESCBC_OPERATION_TYPE_ENCRYPT) {
 *          // do something with operation->oneStepOperation
 *      } else {
 *          // do something with operation->segmentedOperation
 *      }
 *  }
 *
 *  AESCBC_SegmentedOperation operation;
 *
 *  void cbcStartFunction(void) {
 *      AESCBC_Handle handle;
 *      AESCBC_Params params;
 *      CryptoKey cryptoKey;
 *      int_fast16_t decryptionResult;
 *
 *      AESCBC_Params_init(&params);
 *      params.returnBehavior = AESCBC_RETURN_BEHAVIOR_CALLBACK;
 *      params.callbackFxn = cbcCallback;
 *
 *      handle = AESCBC_open(0, &params);
 *
 *      if (handle == NULL) {
 *          // handle error
 *      }
 *
 *      CryptoKeyPlaintext_initKey(&cryptoKey, keyingMaterial, sizeof(keyingMaterial));
 *
 *      AESCBC_SegmentedOperation_init(&operation);
 *
 *      operation.input             = ciphertext;
 *      operation.output            = plaintext;
 *      operation.inputLength       = sizeof(ciphertext);
 *
 *      decryptionResult = AESCBC_setupDecrypt(handle, &cryptoKey);
 *      if (decryptionResult != AESCBC_STATUS_SUCCESS) {
 *          // handle error
 *      }
 *
 *      decryptionResult = AESCBC_setIV(handle, iv, AES_BLOCK_SIZE);
 *      if (decryptionResult != AESCBC_STATUS_SUCCESS) {
 *          // handle error
 *      }
 *
 *      decryptionResult = AESCBC_addData(handle, &operation);
 *      if (decryptionResult != AESCBC_STATUS_SUCCESS) {
 *          // handle error
 *      }
 *
 *      // do other things while CBC operation completes in the background
 *
 *      operation.inputLength = 0;
 *      decryptionResult = AESCBC_finalize(handle, &operation);
 *      if (decryptionResult != AESCBC_STATUS_SUCCESS) {
 *          // handle error
 *      }
 *
 *  }
 *
 *  @endcode
 *
 *  ### Multi-step CBC encryption with plaintext CryptoKey and non-empty finalize in blocking return mode #
 *  @code
 *
 *  #include <ti/drivers/AESCBC.h>
 *  #include <ti/drivers/cryptoutils/cryptokey/CryptoKeyPlaintext.h>
 *
 *  ...
 *
 *  #define AES_BLOCK_SIZE      16 // bytes
 *
 *  AESCBC_Handle handle;
 *  CryptoKey cryptoKey;
 *  int_fast16_t encryptionResult;
 *
 *  // For example purposes only. Generate IVs in a non-static way in practice.
 *  // Test vector 1 from NIST CAPV set CBCMMT128
 *  uint8_t iv[16] =                {0xaa, 0xd1, 0x58, 0x3c, 0xd9, 0x13, 0x65, 0xe3,
 *                                   0xbb, 0x2f, 0x0c, 0x34, 0x30, 0xd0, 0x65, 0xbb};
 *  uint8_t plaintext[32] =         {0x06, 0x8b, 0x25, 0xc7, 0xbf, 0xb1, 0xf8, 0xbd,
 *                                   0xd4, 0xcf, 0xc9, 0x08, 0xf6, 0x9d, 0xff, 0xc5,
 *                                   0xdd, 0xc7, 0x26, 0xa1, 0x97, 0xf0, 0xe5, 0xf7,
 *                                   0x20, 0xf7, 0x30, 0x39, 0x32, 0x79, 0xbe, 0x91};
 *  uint8_t ciphertext[sizeof(plaintext)];
 *  uint8_t keyingMaterial[16] =    {0x07, 0x00, 0xd6, 0x03, 0xa1, 0xc5, 0x14, 0xe4,
 *                                   0x6b, 0x61, 0x91, 0xba, 0x43, 0x0a, 0x3a, 0x0c};
 *
 *  // The ciphertext should be the following after the encryption operation:
 *  //  0xc4, 0xdc, 0x61, 0xd9, 0x72, 0x59, 0x67, 0xa3
 *  //  0x02, 0x01, 0x04, 0xa9, 0x73, 0x8f, 0x23, 0x86
 *  //  0x85, 0x27, 0xce, 0x83, 0x9a, 0xab, 0x17, 0x52
 *  //  0xfd, 0x8b, 0xdb, 0x95, 0xa8, 0x2c, 0x4d, 0x00
 *
 *
 *  handle = AESCBC_open(0, NULL);
 *
 *  if (handle == NULL) {
 *      // handle error
 *  }
 *
 *  CryptoKeyPlaintext_initKey(&cryptoKey, keyingMaterial, sizeof(keyingMaterial));
 *
 *  AESCBC_SegmentedOperation operation;
 *  AESCBC_SegmentedOperation_init(&operation);
 *
 *  operation.input             = plaintext;
 *  operation.output            = ciphertext;
 *  // One should pass in data that is a block-sized multiple length (16 bytes)
 *  operation.inputLength       = AES_BLOCK_SIZE;
 *
 *  encryptionResult = AESCBC_setupEncrypt(handle, &cryptoKey);
 *  if (encryptionResult != AESCBC_STATUS_SUCCESS) {
 *      // handle error
 *  }
 *
 *  encryptionResult = AESCBC_setIV(handle, iv, AES_BLOCK_SIZE);
 *  if (encryptionResult != AESCBC_STATUS_SUCCESS) {
 *      // handle error
 *  }
 *
 *  encryptionResult = AESCBC_addData(handle, &operation);
 *  if (encryptionResult != AESCBC_STATUS_SUCCESS) {
 *      // handle error
 *  }
 *
 *  operation.input             = plaintext + AES_BLOCK_SIZE;
 *  operation.output            = ciphertext + AES_BLOCK_SIZE;
 *
 *  // You can also finalize with more data (non-zero inputLength)
 *  operation.inputLength       = sizeof(plaintext) - AES_BLOCK_SIZE;
 *  encryptionResult = AESCBC_finalize(handle, &operation);
 *  if (encryptionResult != AESCBC_STATUS_SUCCESS) {
 *      // handle error
 *  }
 *
 *  AESCBC_close(handle);
 *
 *  @endcode
 */

#ifndef ti_drivers_AESCBC__include
#define ti_drivers_AESCBC__include

#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>

#include <ti/drivers/AESCommon.h>
#include <ti/drivers/cryptoutils/cryptokey/CryptoKey.h>

#ifdef __cplusplus
extern "C" {
#endif

/*!
 * Common AESCBC status code reservation offset.
 * AESCBC driver implementations should offset status codes with
 * #AESCBC_STATUS_RESERVED growing negatively.
 *
 * Example implementation specific status codes:
 * @code
 * #define AESCBCXYZ_STATUS_ERROR0    AESCBC_STATUS_RESERVED - 0
 * #define AESCBCXYZ_STATUS_ERROR1    AESCBC_STATUS_RESERVED - 1
 * #define AESCBCXYZ_STATUS_ERROR2    AESCBC_STATUS_RESERVED - 2
 * @endcode
 */
#define AESCBC_STATUS_RESERVED AES_STATUS_RESERVED

/*!
 * @brief   Successful status code.
 *
 * Functions return #AESCBC_STATUS_SUCCESS if the function was executed
 * successfully.
 */
#define AESCBC_STATUS_SUCCESS AES_STATUS_SUCCESS

/*!
 * @brief   Generic error status code.
 *
 * Functions return #AESCBC_STATUS_ERROR if the function was not executed
 * successfully and no more pertinent error code could be returned.
 */
#define AESCBC_STATUS_ERROR AES_STATUS_ERROR

/*!
 * @brief   An error status code returned if the hardware or software resource
 * is currently unavailable.
 *
 * AESCBC driver implementations may have hardware or software limitations on how
 * many clients can simultaneously perform operations. This status code is returned
 * if the mutual exclusion mechanism signals that an operation cannot currently be performed.
 */
#define AESCBC_STATUS_RESOURCE_UNAVAILABLE AES_STATUS_RESOURCE_UNAVAILABLE

/*!
 *  @brief  The ongoing operation was canceled.
 */
#define AESCBC_STATUS_CANCELED AES_STATUS_CANCELED

/*!
 *  @brief  The operation requested is not supported for now.
 *
 *  This code is returned by AESCBC_generateIV(), and internal
 *  generation of IVs isn't supported for now. Eventually, this
 *  function will make the TRNG generate the IV.
 */
#define AESCBC_STATUS_FEATURE_NOT_SUPPORTED AES_STATUS_FEATURE_NOT_SUPPORTED

/*!
 *  @brief  The operation tried to load a key from the keystore using an invalid key ID.
 */
#define AESCBC_STATUS_KEYSTORE_INVALID_ID AES_STATUS_KEYSTORE_INVALID_ID

/*!
 *  @brief  The key store module returned a generic error. See key store documentation
 *  for additional details.
 */
#define AESCBC_STATUS_KEYSTORE_GENERIC_ERROR AES_STATUS_KEYSTORE_GENERIC_ERROR

/*!
 * @brief   The operation does not support non-word-aligned input and/or output.
 *
 * AESCBC driver implementations may have restrictions on the alignment of
 * input/output data due to performance limitations of the hardware.
 */
#define AESCBC_STATUS_UNALIGNED_IO_NOT_SUPPORTED AES_STATUS_UNALIGNED_IO_NOT_SUPPORTED

/*!
 *  @brief AESCBC Global configuration
 *
 *  The #AESCBC_Config structure contains a set of pointers used to characterize
 *  the AESCBC driver implementation.
 *
 *  This structure needs to be defined before calling #AESCBC_init() and it must
 *  not be changed thereafter.
 *
 *  @sa     #AESCBC_init()
 */
typedef AESCommon_Config AESCBC_Config;

/*!
 *  @brief  A handle that is returned from an #AESCBC_open() call.
 */
typedef AESCBC_Config *AESCBC_Handle;

/*!
 * @brief   The way in which CBC function calls return after performing an
 * encryption or decryption operation.
 *
 * Not all CBC operations exhibit the specified return behavior. Functions that do not
 * require significant computation and cannot offload that computation to a background thread
 * behave like regular functions. Which functions exhibit the specified return behavior is not
 * implementation dependent. Specifically, a software-backed implementation run on the same
 * CPU as the application will emulate the return behavior while not actually offloading
 * the computation to the background thread.
 *
 * AESCBC functions exhibiting the specified return behavior have restrictions on the
 * context from which they may be called.
 *
 * |                                | Task  | Hwi   | Swi   |
 * |--------------------------------|-------|-------|-------|
 * |AESCBC_RETURN_BEHAVIOR_CALLBACK | X     | X     | X     |
 * |AESCBC_RETURN_BEHAVIOR_BLOCKING | X     |       |       |
 * |AESCBC_RETURN_BEHAVIOR_POLLING  | X     | X     | X     |
 *
 */
typedef enum
{
    AESCBC_RETURN_BEHAVIOR_CALLBACK = AES_RETURN_BEHAVIOR_CALLBACK,
    /*!< The function call will return immediately while the
     *   CBC operation goes on in the background. The registered
     *   callback function is called after the operation completes.
     *   The context the callback function is called (task, HWI, SWI)
     *   is implementation-dependent.
     */
    AESCBC_RETURN_BEHAVIOR_BLOCKING = AES_RETURN_BEHAVIOR_BLOCKING,
    /*!< The function call will block while the CBC operation goes
     *   on in the background. CBC operation results are available
     *   after the function returns.
     */
    AESCBC_RETURN_BEHAVIOR_POLLING  = AES_RETURN_BEHAVIOR_POLLING,
    /*!< The function call will continuously poll a flag while CBC
     *   operation goes on in the background. CBC operation results
     *   are available after the function returns.
     */
} AESCBC_ReturnBehavior;

/*!
 *  @brief  Enum for the direction of the CBC operation.
 */
typedef enum
{
    AESCBC_MODE_ENCRYPT = 1,
    AESCBC_MODE_DECRYPT = 2
} AESCBC_Mode;

/*!
 *  @brief  Struct containing the parameters required for encrypting/decrypting
 *          a message in a single-step operation.
 */
typedef struct
{
    CryptoKey *key;     /*!< Pointer to a previously initialized CryptoKey. */
    uint8_t *input;     /*!<
                         *   - Encryption: The plaintext buffer to be
                         *     encrypted in the CBC operation.
                         *   - Decryption: The ciphertext to be decrypted.
                         */
    uint8_t *output;    /*!<
                         *   - Encryption: The output ciphertext buffer that
                         *     the encrypted plaintext is copied to.
                         *   - Decryption: The plaintext derived from the
                         *     decrypted ciphertext is copied here.
                         */
    uint8_t *iv;        /*!< A buffer containing an IV. IVs must be unique to
                         *   each CBC operation and may not be reused. If
                         *   ivInternallyGenerated is set, the IV will be
                         *   generated by #AESCBC_oneStepEncrypt() and copied to
                         *   this buffer.
                         */
    size_t inputLength; /*!< Length of the input buffer in bytes for one-step
                         *   AES CBC operations. Must be a multiple
                         *   of the AES block size (16 bytes). Also, the
                         *   output buffer must be large enough to receive
                         *   the same number of bytes. The user or application
                         *   should take care of necessary padding.
                         *   Max length supported may be limited depending on the return behavior.
                         */

    bool ivInternallyGenerated; /*!< When true, the IV buffer passed into #AESCBC_oneStepEncrypt()
                                 *   will be overwritten with a randomly generated IV.
                                 *   Not supported by all implementations.
                                 */
} AESCBC_OneStepOperation;

/*!
 *  @brief  Struct containing the parameters required for encrypting/decrypting
 *          a message in a segmented operation. Must be updated for steps of a segmented
 *          operation where data is processed (addData() and finalize()).
 */
typedef struct
{
    uint8_t *input;     /*!<
                         *   - Encryption: The plaintext buffer to be
                         *     encrypted in the CBC operation.
                         *   - Decryption: The ciphertext to be decrypted.
                         */
    uint8_t *output;    /*!<
                         *   - Encryption: The output ciphertext buffer that
                         *     the encrypted plaintext is copied to.
                         *   - Decryption: The plaintext derived from the
                         *     decrypted ciphertext is copied here.
                         */
    size_t inputLength; /*!< Length of the input buffer in bytes for segmented
                         *   AES CBC operations. Must be a multiple
                         *   of the AES block size (16 bytes) unless finalizing
                         *   without new data. In that case, this value can be 0.
                         *   Also, the output buffer must be large enough to receive
                         *   the same number of bytes.
                         *   Max length supported may be limited depending on the return behavior.
                         */
} AESCBC_SegmentedOperation;

/**
 * @deprecated
 * Define a typedef for deprecated operation AESCBC_Operation.
 * Existing code should be refactored to use AESCBC_OneStepOperation.
 * This reference may be removed at some point in the future
 *
 */
typedef AESCBC_OneStepOperation AESCBC_Operation;

/*!
 *  @brief Union  containing a reference to a one step or
 *  segmented operation
 */
typedef union AESCBC_OperationUnion
{
    AESCBC_OneStepOperation oneStepOperation;     /* One-step operation element of the operation union */
    AESCBC_SegmentedOperation segmentedOperation; /* Segmented operation element of the operation union */
} AESCBC_OperationUnion;

/*!
 *  @brief  Enum for the operation types supported by the driver.
 */
typedef enum
{
    AESCBC_OPERATION_TYPE_ENCRYPT    = 1, /* Fields 1 and 2 are for backward compatibility */
    AESCBC_OPERATION_TYPE_DECRYPT    = 2,
    AESCBC_OP_TYPE_ONESTEP_ENCRYPT   = 1, /* Names changed to _OP_TYPE_ to avoid MISRA deviation from first 31 chars not
                                             being unique */
    AESCBC_OP_TYPE_ONESTEP_DECRYPT   = 2,
    AESCBC_OP_TYPE_ENCRYPT_SEGMENTED = 3,
    AESCBC_OP_TYPE_DECRYPT_SEGMENTED = 4,
    AESCBC_OP_TYPE_FINALIZE_ENCRYPT_SEGMENTED = 5,
    AESCBC_OP_TYPE_FINALIZE_DECRYPT_SEGMENTED = 6
} AESCBC_OperationType;

/*!
 *  @brief  The definition of a callback function used by the AESCBC driver
 *          when used in ::AESCBC_RETURN_BEHAVIOR_CALLBACK
 *
 *  @param  handle        Handle of the client that started the CBC operation.
 *
 *  @param  returnValue   The result of the CBC operation. May contain an error code.
 *                        Informs the application of why the callback function was
 *                        called.
 *
 *  @param  operation     A pointer to an operation union.
 *
 *  @param  operationType This parameter determines which operation the
 *                        callback refers to.
 */
typedef void (*AESCBC_CallbackFxn)(AESCBC_Handle handle,
                                   int_fast16_t returnValue,
                                   AESCBC_OperationUnion *operation,
                                   AESCBC_OperationType operationType);

/*!
 *  @brief  CBC Parameters
 *
 *  CBC Parameters used with the #AESCBC_open() call. Default values for
 *  these parameters are set using #AESCBC_Params_init().
 *
 *  @sa     #AESCBC_Params_init()
 */
typedef struct
{
    AESCBC_ReturnBehavior returnBehavior; /*!< Blocking, callback, or polling return behavior */
    AESCBC_CallbackFxn callbackFxn;       /*!< Callback function pointer */
    uint32_t timeout;                     /*!< Timeout before the driver returns an error in
                                           *   ::AESCBC_RETURN_BEHAVIOR_BLOCKING
                                           */
    void *custom;                         /*!< Custom argument used by driver
                                           *   implementation
                                           */
} AESCBC_Params;

/*!
 *  @brief Default #AESCBC_Params structure
 *
 *  @sa     #AESCBC_Params_init()
 */
extern const AESCBC_Params AESCBC_defaultParams;

/*!
 *  @brief  This function initializes the CBC module.
 *
 *  @pre    The AESCBC_config structure must exist and be persistent before this
 *          function can be called. This function must also be called before
 *          any other CBC driver APIs. This function call does not modify any
 *          peripheral registers.
 */
void AESCBC_init(void);

/*!
 *  @brief  Function to initialize the #AESCBC_Params struct to its defaults
 *
 *  @param  params      An pointer to #AESCBC_Params structure for
 *                      initialization
 *
 *  Defaults values are:
 *      returnBehavior              = AESCBC_RETURN_BEHAVIOR_BLOCKING
 *      callbackFxn                 = NULL
 *      timeout                     = SemaphoreP_WAIT_FOREVER
 *      custom                      = NULL
 */
void AESCBC_Params_init(AESCBC_Params *params);

/*!
 *  @brief  This function opens a given CBC peripheral.
 *
 *  @pre    CBC controller has been initialized using #AESCBC_init()
 *
 *  @param  [in] index    Logical peripheral number for the CBC indexed into
 *                        the AESCBC_config table
 *
 *  @param  [in] params   Pointer to an parameter block, if NULL it will use
 *                        default values.
 *
 *  @return An #AESCBC_Handle on success or a NULL on an error or if it has
 *          been opened already.
 *
 *  @sa     #AESCBC_init()
 *  @sa     #AESCBC_close()
 */
AESCBC_Handle AESCBC_open(uint_least8_t index, const AESCBC_Params *params);

/*!
 *  @brief  Function to close a CBC peripheral specified by the CBC handle
 *
 *  @pre    #AESCBC_open() or #AESCBC_construct()
 *
 *  @param  [in] handle A CBC handle
 *
 *  @sa     #AESCBC_open()
 */
void AESCBC_close(AESCBC_Handle handle);

/*!
 *  @brief  Function to initialize an #AESCBC_Operation struct to its defaults
 *
 *  @deprecated  This function should be replaced by calls to operation-specific init
 *               functions.
 *
 *  @param      [in] operationStruct     A pointer to an #AESCBC_Operation structure for
 *                                       initialization
 *
 *  Defaults values are all zeros.
 */
void AESCBC_Operation_init(AESCBC_Operation *operationStruct);

/*!
 *  @brief  Function to initialize an #AESCBC_OneStepOperation struct to its defaults
 *
 *  @param  [in] operationStruct     A pointer to an #AESCBC_OneStepOperation structure for
 *                                   initialization
 *
 *  Defaults values are all zeros.
 */
void AESCBC_OneStepOperation_init(AESCBC_OneStepOperation *operationStruct);

/*!
 *  @brief  Function to initialize an #AESCBC_SegmentedOperation struct to its defaults
 *
 *  @param  [in] operationStruct     A pointer to an #AESCBC_SegmentedOperation structure for
 *                                   initialization
 *
 *  Defaults values are all zeros.
 */
void AESCBC_SegmentedOperation_init(AESCBC_SegmentedOperation *operationStruct);

/*!
 *  @brief  Function to perform an AESCBC encryption operation in one call.
 *
 *  @note   None of the buffers provided as arguments may be altered by the application during an ongoing operation.
 *          Doing so can yield corrupted ciphertext.
 *
 *  @pre    #AESCBC_open() or #AESCBC_construct(), and #AESCBC_OneStepOperation_init() must be called first.
 *
 *  @param  [in] handle                 A CBC handle returned from #AESCBC_open() or #AESCBC_construct()
 *
 *  @param  [in] operationStruct        A pointer to a struct containing the parameters required to perform the
 * operation.
 *
 *  @retval #AESCBC_STATUS_SUCCESS               The operation succeeded.
 *  @retval #AESCBC_STATUS_ERROR                 The operation failed.
 *  @retval #AESCBC_STATUS_RESOURCE_UNAVAILABLE  The required hardware resource was not available. Try again later.
 *  @retval #AESCBC_STATUS_UNALIGNED_IO_NOT_SUPPORTED  The input and/or output buffer were not word-aligned.
 *
 *  @sa     #AESCBC_oneStepDecrypt()
 */
int_fast16_t AESCBC_oneStepEncrypt(AESCBC_Handle handle, AESCBC_OneStepOperation *operationStruct);

/*!
 *  @brief  Function to perform an AESCBC decryption operation in one call.
 *
 *  @note   None of the buffers provided as arguments may be altered by the application during an ongoing operation.
 *          Doing so can yield corrupted plaintext.
 *
 *  @pre    #AESCBC_open() or #AESCBC_construct(), and #AESCBC_OneStepOperation_init() must be called first.
 *
 *  @param  [in] handle                 A CBC handle returned from #AESCBC_open() or #AESCBC_construct()
 *
 *  @param  [in] operationStruct        A pointer to a struct containing the parameters required to perform the
 * operation.
 *
 *  @retval #AESCBC_STATUS_SUCCESS               The operation succeeded.
 *  @retval #AESCBC_STATUS_ERROR                 The operation failed.
 *  @retval #AESCBC_STATUS_RESOURCE_UNAVAILABLE  The required hardware resource was not available. Try again later.
 *  @retval #AESCBC_STATUS_UNALIGNED_IO_NOT_SUPPORTED  The input and/or output buffer were not word-aligned.
 *
 *  @sa     AESCBC_oneStepEncrypt()
 */
int_fast16_t AESCBC_oneStepDecrypt(AESCBC_Handle handle, AESCBC_OneStepOperation *operationStruct);

/*!
 *  @brief  Function to prepare a segmented AESCBC encryption operation.
 *
 *  This function sets up a segmented AESCBC encryption operation.
 *
 *  @pre    #AESCBC_open() or #AESCBC_construct()
 *
 *  @param  [in] handle     A CBC handle returned from #AESCBC_open()
 *                          or #AESCBC_construct()
 *
 *  @param  [in] key        Pointer to a previously initialized CryptoKey.
 *
 *  @retval #AESCBC_STATUS_SUCCESS                  The operation succeeded.
 *  @retval #AESCBC_STATUS_ERROR                    The operation failed.
 *
 *  @post   #AESCBC_setIV() or #AESCBC_generateIV()
 */
int_fast16_t AESCBC_setupEncrypt(AESCBC_Handle handle, const CryptoKey *key);

/*!
 *  @brief  Function to prepare a segmented AESCBC decryption operation.
 *
 *  This function sets up a segmented AESCBC decryption operation.
 *
 *  @pre    #AESCBC_open() or #AESCBC_construct()
 *
 *  @param  [in] handle     A CBC handle returned from #AESCBC_open()
 *                          or #AESCBC_construct()
 *
 *  @param  [in] key        Pointer to a previously initialized CryptoKey.
 *
 *  @retval #AESCBC_STATUS_SUCCESS                  The operation succeeded.
 *  @retval #AESCBC_STATUS_ERROR                    The operation failed.
 *
 *  @post   #AESCBC_setIV()
 */
int_fast16_t AESCBC_setupDecrypt(AESCBC_Handle handle, const CryptoKey *key);

/*!
 *  @brief  Function to set an initialization vector for an AES CBC
 *  segmented operation.
 *
 *  This function sets up a segmented AESCBC encryption or decryption operation.
 *
 *  @pre    #AESCBC_setupEncrypt() or #AESCBC_setupDecrypt()
 *
 *  @param  [in] handle    A CBC handle returned from #AESCBC_open()
 *                         or #AESCBC_construct()
 *
 *  @param  [in] iv        Pointer to the buffer containing the initialization vector
 *
 *  @param  [in] ivLength  The length of the initialization vector. The length of the
 *                         IV buffer must be 16 bytes.
 *
 *  @retval #AESCBC_STATUS_SUCCESS                  The operation succeeded.
 *  @retval #AESCBC_STATUS_ERROR                    The operation failed.
 *
 *  @post   #AESCBC_addData()
 */
int_fast16_t AESCBC_setIV(AESCBC_Handle handle, const uint8_t *iv, size_t ivLength);

/*!
 *  @brief  Function to generate an initialization vector for an AES CBC
 *  segmented encryption operation.
 *
 *  This function sets up a segmented AESCBC encryption operation.
 *
 *  @pre    #AESCBC_setupEncrypt()
 *
 *  @param  [in] handle    A CBC handle returned from #AESCBC_open()
 *                         or #AESCBC_construct()
 *
 *  @param  [in] iv        Pointer to a buffer to write the generated initialization vector to
 *
 *  @param  [in] ivSize    Size of the buffer pointed to by iv. This buffer must be
 *                          16 bytes long for AES CBC mode.
 *
 *  @param  [out] ivLength  The length of the initialization vector actually written if
 *                           the operation was successful.
 *
 *  @retval #AESCBC_STATUS_SUCCESS                  The operation succeeded.
 *  @retval #AESCBC_STATUS_ERROR                    The operation failed.
 *  @retval #AESCBC_STATUS_FEATURE_NOT_SUPPORTED    The operation is not supported
 *                                                  in this device.
 *
 *  @post   #AESCBC_addData()
 */
int_fast16_t AESCBC_generateIV(AESCBC_Handle handle, uint8_t *iv, size_t ivSize, size_t *ivLength);

/*!
 *  @brief  Encrypts or decrypts a segment of data defined by the AESCBC_SegmentedOperation struct
 *
 *  #AESCBC_addData() may be called an arbitrary number times before finishing the operation with
 *  #AESCBC_finalize().
 *
 *  This function returns according to the return behavior set when opening the driver.
 *
 *  @pre    #AESCBC_setupEncrypt() or #AESCBC_setupDecrypt()
 *
 *  @param  [in] handle         A CBC handle returned from #AESCBC_open() or #AESCBC_construct()
 *
 *  @param  [in] operation      Pointer to a segmented CBC operation structure
 *
 *  @retval #AESCBC_STATUS_SUCCESS                  The operation succeeded.
 *  @retval #AESCBC_STATUS_ERROR                    The operation failed.
 *  @retval #AESCBC_STATUS_RESOURCE_UNAVAILABLE     The required hardware
 *                                                  resource was not available.
 *                                                  Try again later.
 *  @retval #AESCBC_STATUS_UNALIGNED_IO_NOT_SUPPORTED  The input and/or output buffer were not word-aligned.
 *
 *  @post   #AESCBC_addData() or #AESCBC_finalize()
 */
int_fast16_t AESCBC_addData(AESCBC_Handle handle, AESCBC_SegmentedOperation *operation);

/*!
 *  @brief  Finalize the AES operation. If new data needs to be added,
 *  @c inputLength will be used to govern how many bytes will be written.
 *  New data must be a multiple of a block length, 16 bytes, or 0 if no
 *  new data needs to be added.
 *
 *  @pre    #AESCBC_addData()
 *
 *  @param  [in] handle         A CBC handle returned from #AESCBC_open() or #AESCBC_construct()
 *
 *  @param  [in] operation      Pointer to a segmented CBC operation structure
 *
 *  @retval #AESCBC_STATUS_SUCCESS                  In ::AESCBC_RETURN_BEHAVIOR_BLOCKING and
 *                                                  ::AESCBC_RETURN_BEHAVIOR_POLLING, this means the CBC
 *                                                  was generated successfully. In ::AESCBC_RETURN_BEHAVIOR_CALLBACK,
 *                                                  this means the operation started successfully.
 *  @retval #AESCBC_STATUS_ERROR                    The operation failed.
 *  @retval #AESCBC_STATUS_RESOURCE_UNAVAILABLE     The required hardware
 *                                                  resource was not available.
 *                                                  Try again later.
 *  @retval #AESCBC_STATUS_UNALIGNED_IO_NOT_SUPPORTED  The input and/or output buffer were not word-aligned.
 */
int_fast16_t AESCBC_finalize(AESCBC_Handle handle, AESCBC_SegmentedOperation *operation);

/*!
 *  @brief Cancels an ongoing AESCBC operation.
 *
 *  Asynchronously cancels an AESCBC operation. Only available when using
 *  AESCBC_RETURN_BEHAVIOR_CALLBACK.
 *  The operation will terminate as though an error occurred. The
 *  return status code of the operation will be AESCBC_STATUS_CANCELED.
 *
 *  @param  [in] handle Handle of the operation to cancel
 *
 *  @retval #AESCBC_STATUS_SUCCESS               The operation was canceled or the operation had already completed.
 *  @retval #AESCBC_STATUS_ERROR                 The IV buffer was not properly cleared.
 */
int_fast16_t AESCBC_cancelOperation(AESCBC_Handle handle);

/**
 *  @brief  Constructs a new AESCBC object
 *
 *  Unlike #AESCBC_open(), #AESCBC_construct() does not require the hwAttrs and
 *  object to be allocated in a #AESCBC_Config array that is indexed into.
 *  Instead, the #AESCBC_Config, hwAttrs, and object can be allocated at any
 *  location. This allows for relatively simple run-time allocation of temporary
 *  driver instances on the stack or the heap.
 *  The drawback is that this makes it more difficult to write device-agnostic
 *  code. If you use an ifdef with DeviceFamily, you can choose the correct
 *  object and hwAttrs to allocate. That compilation unit will be tied to the
 *  device it was compiled for at this point. To change devices, recompilation
 *  of the application with a different DeviceFamily setting is necessary.
 *
 *  @param [in] config #AESCBC_Config describing the location of the object and hwAttrs.
 *
 *  @param [in] params #AESCBC_Params to configure the driver instance.
 *
 *  @return     Returns a #AESCBC_Handle on success or NULL on failure.
 *
 *  @pre    The object struct @c config points to must be zeroed out prior to
 *          calling this function. Otherwise, unexpected behavior may ensue.
 */
AESCBC_Handle AESCBC_construct(AESCBC_Config *config, const AESCBC_Params *params);

#ifdef __cplusplus
}
#endif

#endif /* ti_drivers_AESCBC__include */
