\page rcl_command_handlers Command Handlers

As previously mentioned, an important part of the RCL command life-cycle is what is known as the command handler. The command handler is a self-contained finite state machine that takes care of the command parameters, the payload, and the interrupt handling needed during the execution of the command (i.e. the interrupts coming from the radio and the SYSTIM). In simple terms, it is the piece of code takes care of protocol specific operations (e.g. BLE advertising), and interacts directly with the LRF.

The following section provides an introduction to their usage and common characteristics. For more protocol specific information, see the following pages:

- \subpage ble_command_handlers
- \subpage generic_command_handlers
- \subpage nesb_command_handlers

# Usage

Command handlers are configured at the application level through the use of command structs. The user will normally first declare a command of a specific type (e.g. ::RCL_CMD_GENERIC_TX_t or ::RCL_CMD_BLE5_ADV), and then configure the command handler by updating the fields in the command struct.

As expected, the command struct will mostly include protocol specific configuration. Nevertheless, there are some configuration fields which are common to all command handlers (::RCL_Command_s).

```c
/* Declare command */
RCL_CmdGenericTx cmd;
cmd = RCL_CmdGenericTx_DefaultRuntime();

/* Common configuration */
cmd.common.scheduling = RCL_Schedule_AbsTime;
cmd.common.runtime.callback = defaultCallback;
cmd.common.runtime.lrfCallbackMask.value = LRF_EventRxOk.value;
cmd.common.runtime.rclCallbackMask = RCL_EventLastCmdDone.value | RCL_EventRxEntryAvail.value;;

/* Command-specific configuration */
cmd.rfFrequency = 2450000000;
cmd.syncWord = 0xAB0B51DE;
cmd.config.fsOff = 0;
```

Part of the configuration fields common to all command handlers is the runtime information (RCL_CommandRuntime_s) and the timing information (RCL_CommandTiming_s).

The runtime information among other things allows to enable callbacks for events that come directly from the LRF or for events generated by the RCL. That is, the callback will be executed only when the appropriate event is set.

The timing information on the other hand allows to configure the start time (unless ::RCL_Schedule_Now is used), and also stop times (graceful and hard) of the command.


# Architecture

Regardless of the protocol, the interaction between the RCL and the command handler requires four elements.

- RCL_Events_u going into the command handler: Signals that the RCL might want to send to the command handler (for example a graceful stop or that the radio setup has been performed). This allows the command handler to determine whether it should proceed with the radio operation, reject it, or stop it.
- RCL_Events_u coming out of the command handler: Signals that the command handler might want ot send to the RCL (for example that an Rx entry has been made available or that the command handler has accepted and started executing the command).
- LRF_Events_u: Radio specific events (for example that a packet has been transmitted or that a packet was received with a CRC error) used by both the RCL and the command handler.
- RCL_Command_s common configuration: Provides the command handler with information about scheduling, the command ID, graceful stop times, conflict policy, etc.

The following diagram illustrates in a simplified way how the state machine in a command handler might look:

\startuml "Basic command handler state machine"
[*] --> Idle
Idle --> Scheduled
Scheduled : Check conditions for radio operation\nTrigger operation
Scheduled --> Running: Timer-based start signalled
Scheduled --> Error : Check for possible error reasons
Running --> Error : LRF event\nwas OpError
Running : Wait for RCL input events\nand/or LRF events
Error : Set end status
Error --> Done
Running --> Success : LRF event\nwas OpDone
Success : Set end status
Success --> Done
Done : Disable radio
Done --> [*]
\enduml

## Internal packet format {#internal-packet-format}

The internal packet format refers to how packets are stored in the LRF FIFOs, and it's particularly important since regardless of the communication protocol used, the structure for both transmitted and received packet entries must be compatible with the format.

![LRF Internal packet format](docs/rcl/source/images/internal_packet_format.png)

The following rules apply to the internal packet format:

- Word 0 shall always be aligned to a 32 bit address
- Padding can be used by application to e.g. word align header, word align the payload or other points in the packet.
- Required fields are: FIFO length (Bytes 0-1), padding length (Byte 2), and payload.
- The optional padding must be continuous.
- When it comes to the order in the FIFO, the optional padding is always followed by the header (if any), and the header is always followed by the payload.
- In the case of fixed length or that the header fits in byte 3, then word 1 can be omitted.
- The length field to be transmitted over the air must be embedded in the header written to the FIFO
- Configurable padding (based on LRF settings) at the end of the entry which is emptied from the TX FIFO by the LRF in TX, and entered by the LRF into RX FIFO in RX.

As an example, a correct packet entry for a BLE packet for transmission (or reception with no extra bytes) will be:

![BLE packet entry example](docs/rcl/source/images/internal_packet_format_ble.png)

