/*
 * Copyright (c) 2025, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef ti_edgeai_DAP_core__include
#define ti_edgeai_DAP_core__include

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include <ti/ai/edge_ai/dap/DAP.h>

#ifdef __cplusplus
extern "C" {
#endif

#define DAP_MAX_FRAME_SIZE_BYTES (200)

typedef struct
{
    /*! Pointer to buffer */
    uint8_t buffer[DAP_MAX_FRAME_SIZE_BYTES];

    /*! Length in bytes*/
    size_t length;

    /*! Buffer Index */
    uint8_t index;
} DAP_Frame;

/*!
 *  @brief      DAP error types
 */
typedef enum
{
    /*! No error */
    DAP_ERROR_TYPE_NONE                 = 0x00,
    /*! Mismatch Start byte error */
    DAP_ERROR_TYPE_MISMATCH_START_BYTE  = 0xE1,
    /*! Mismatch end byte error */
    DAP_ERROR_TYPE_MISMATCH_END_BYTE    = 0xE2,
    /*! Error in Address range */
    DAP_ERROR_TYPE_ADDR_RANGE           = 0xE3,
    /*! Mismatch CRC error */
    DAP_ERROR_TYPE_MISMATCH_CRC         = 0xE4,
    /*! Mismatch target address error*/
    DAP_ERROR_TYPE_MISMATCH_TARGET_ADDR = 0xE5
} DAP_ErrorType;

/*!
 *  @brief      DAP command types
 *
 *  This enum defines the commands that can be sent between the host and the
 *  target.
 */
typedef enum
{
    /*! Sent from target to host to indicate there has been an error. */
    DAP_ERROR_COMMAND                   = 0x00,
    /*! Get software capabilities and metadata. */
    DAP_GET_CAPABILITIES_COMMAND        = 0x01,
    /*! List all available sensors. */
    DAP_LIST_SENSORS_COMMAND            = 0x02,
    /*! Configure the application. */
    DAP_CONFIGURE_PIPELINE_COMMAND      = 0x03,
    /*! List all available models. */
    DAP_LIST_MODELS_COMMAND             = 0x04,
    /*! Remove one of the models. */
    DAP_REMOVE_MODEL_COMMAND            = 0x05,
    /*! Signal the target that the host will start sending data. */
    DAP_START_MODEL_UPLOAD_COMMAND      = 0x06,
    /*! Signal the target that the host is done sending data. */
    DAP_END_MODEL_UPLOAD_COMMAND        = 0x07,
    /*! Signal the target to start sending data to the host. */
    DAP_START_STREAMING_COMMAND         = 0x08,
    /*! Signal the target to stop sending data to the host. */
    DAP_STOP_STREAMING_COMMAND          = 0x09,
    /*! List all inference values. */
    DAP_LIST_INFERENCING_VALUES_COMMAND = 0x0a,
    // reserved1 = 0x0b,
    /*! Read a sampling property. */
    DAP_READ_PROPERTY_COMMAND           = 0x0c,
    /*! Write to a sampling property, */
    DAP_WRITE_PROPERTY_COMMAND          = 0x0d,
    /*! List all sampling properties. */
    DAP_LIST_PROPERTIES_COMMAND         = 0x0e,
    /*! Data comming from the host to the target (model upload). */
    DAP_SEND_DATA_COMMAND               = 0x10,
    /*! Data comming from the target to the host (streaming). */
    DAP_RECEIVE_DATA_COMMAND            = 0x10,
} DAP_CommandType;

/*!
 *  @brief      Minimum payload length of commands sent from the host to the
 *              target
 */
typedef enum
{
    /*! No fixed payload */
    DAP_GET_CAPABILITIES_CMD_FIXED_LENGHT        = 0,
    /*! No fixed payload */
    DAP_LIST_SENSORS_FIXED_CMD_LENGHT            = 0,
    /*! Mode, Model Index */
    DAP_CONFIGURE_PIPELINE_CMD_FIXED_LENGHT      = 2,
    /*! No fixed payload */
    DAP_LIST_MODELS_CMD_FIXED_LENGHT             = 0,
    /*! Model Index */
    DAP_REMOVE_MODEL_CMD_FIXED_LENGTH            = 1,
    /*! No fixed payload */
    DAP_START_MODEL_UPLOAD_CMD_FIXED_LENGTH      = 0,
    /*! Checksum */
    DAP_END_MODEL_UPLOAD_CMD_FIXED_LENGTH        = 4,
    /*! No fixed payload */
    DAP_START_STREAMING_CMD_FIXED_LENGHT         = 0,
    /*! No fixed payload */
    DAP_STOP_STREAMING_CMD_FIXED_LENGHT          = 0,
    /*! No fixed payload */
    DAP_LIST_INFERENCING_VALUES_CMD_FIXED_LENGTH = 0,
    /*! Property Index */
    DAP_READ_PROPERTY_CMD_FIXED_LENGHT           = 1,
    /*! Property Index */
    DAP_WRITE_PROPERTY_CMD_FIXED_LENGHT          = 1,
    /*! No fixed payload */
    DAP_LIST_PROPERTIES_CMD_FIXED_LENGHT         = 0,
    /*! No fixed payload */
    DAP_DATA_CMD_FIXED_LENGTH                    = 0
} DAP_CommandPayloadFixedLength;

/*!
 *  @brief      Minimum payload length of commands sent from the target to the
 *              host
 */
typedef enum
{
    /*! Command, Error Number */
    DAP_ERROR_FIXED_LENGHT                   = 2,
    /*! API Versio, Capability Flag, SDK Major Version, SDK Minor Version */
    DAP_GET_CAPABILITIES_FIXED_LENGHT        = 4,
    /*! Sensor Count, Sensor Index */
    DAP_LIST_SENSORS_FIXED_LENGHT            = 2,
    /*! No fixed payload */
    DAP_CONFIGURE_PIPELINE_FIXED_LENGHT      = 0,
    /*! Model Count, Model Index */
    DAP_LIST_MODELS_FIXED_LENGHT             = 2,
    /*! No fixed payload */
    DAP_REMOVE_MODEL_FIXED_LENGTH            = 0,
    /*! No fixed payload */
    DAP_START_MODEL_UPLOAD_FIXED_LENGTH      = 0,
    /*! No fixed payload */
    DAP_END_MODEL_UPLOAD_FIXED_LENGTH        = 0,
    /*! No fixed payload */
    DAP_START_STREAMING_FIXED_LENGHT         = 0,
    /*! No fixed payload */
    DAP_STOP_STREAMING_FIXED_LENGHT          = 0,
    /*! Inference Count, Inference Index, Inference Data Format */
    DAP_LIST_INFERENCING_VALUES_FIXED_LENGTH = 3,
    /*! Property Index */
    DAP_READ_PROPERTY_FIXED_LENGHT           = 1,
    /*! Property Index */
    DAP_WRITE_PROPERTY_FIXED_LENGHT          = 1,
    /*! Property Count, Property Index, Property Data Format */
    DAP_LIST_PROPERTIES_FIXED_LENGHT         = 3,
    /*! No fixed payload */
    DAP_DATA_FIXED_LENGTH                    = 0
} DAP_ResponsePayloadFixedLength;

/*!
 *  @brief      Data Types for the inferences and properties
 */
typedef enum
{
    /*! 8-bit signed integer */
    DAP_DATA_TYPE_INT8    = 0x01,
    /*! 16-bit signed integer */
    DAP_DATA_TYPE_INT16   = 0x02,
    /*! 32-bit signed integer */
    DAP_DATA_TYPE_INT32   = 0x03,
    /*! 8-bit unsigned integer */
    DAP_DATA_TYPE_UINT8   = 0x04,
    /*! 16-bit unsigned integer */
    DAP_DATA_TYPE_UINT16  = 0x05,
    /*! 32-bit unsigned integer */
    DAP_DATA_TYPE_UINT32  = 0x06,
    /*! 16-bit half precision floating point value */
    DAP_DATA_TYPE_FLOAT16 = 0x07,
    /*! 32-bit normal precision floating point value */
    DAP_DATA_TYPE_FLOAT32 = 0x08,
    /*! 64-bit double precision floating point value */
    DAP_DATA_TYPE_FLOAT64 = 0x09,
} DAP_DataTypes;

/*!
 *  @brief      Data Types for the inferences and properties
 */
typedef struct
{
    const char *name;
    DAP_DataTypes dataFormat;
    union
    {
        uint8_t u8;
        uint16_t u16;
        uint32_t u32;
        int8_t i8;
        int16_t i16;
        int32_t i32;
        float f16;
        float f32;
        double f64;
    } value;
} DAP_Property;

typedef struct
{
    const char *info;
    DAP_DataTypes dataFormat;
} DAP_Inference;

/*!
 *  @brief  Verify a frame
 *
 *  This function verify if a frame is well constructed
 *
 *  @param[in]  frame pointer to a frame
 *
 *  @return Returns if there is an error in the frame.
 */
DAP_ErrorType DAP_verifyFrame(DAP_Frame *frame);

/*!
 *  @brief  Service a command sent by the host
 *
 *  This function decodes the command sent by the host and generates a response
 *  packet based on the command received.
 *
 *  @param[in]  dapHandle pointer
 *
 *  @param[in]  commandFrame pointer to the received packet.
 */
void DAP_serviceCommand(DAP_Handle *dapHandle, DAP_Frame *commandFrame);

/*!
 *  @brief  Send error to host
 *
 *  This function sends an error packet to the host. Errors can be synchronous,
 *  in response to a command, or asynchronous, errors that are not sent in
 *  response to any specific command, such as, errors during streaming
 *  operations.
 *
 *  @param[in]  dapHandle pointer
 *
 *  @param[in]  command error type
 *
 *  @param[in]  errorMessage string with the error message
 */
void DAP_sendError(DAP_Handle *dapHandle, DAP_CommandType command, const char errorMessage[]);

/*!
 *  @brief  Send data to host
 *
 *  This function sends a buffer of data to the DAP
 *
 *  @param[in]  dapHandle pointer
 *
 *  @param[in]  sendDataType Type of the data sent
 *
 *  @param[in]  data pointer to a buffer with the data
 *
 *  @param[in]  dataSize size of the data buffer in bytes
 */
void DAP_receiveDataCommand(DAP_Handle *dapHandle, DAP_SendDataType sendDataType, void *data, size_t dataSize);

#ifdef __cplusplus
}
#endif
#endif /* ti_edgeai_DAP_core__include */