/* Copyright (C) 2025 Texas Instruments Incorporated
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 *  ======== DAP.syscfg.js ========
 */

"use strict";

let Common   = system.getScript("/ti/drivers/Common.js");
let logError = Common.logError;

/*
 * ======== getLibs ========
 */
function getLibs(mod)
{
    let GenLibs = system.getScript("/ti/utils/build/GenLibs.syscfg.js");
    var toolchain = GenLibs.getToolchainDir();
    var isa = GenLibs.getDeviceIsa();

    var link_info = {
        name: "edgeai",
        deps: ["/ti/drivers"],
        libs: [
            "ti/ai/edge_ai/lib/" + toolchain + "/" + isa + "/" + "edgeai.a",
        ]
    };

    return link_info;
}

/*
 * ======== moduleInstances ========
 */
function moduleInstances(inst)
{
    let sharedInstance = new Array();

    sharedInstance.push(
        {
            name       : "uart",
            displayName: "UART",
            moduleName : "/ti/drivers/UART2",
            collapsed: true,
            requiredArgs: {
                enableNonblocking: true,
            }
        }
    );

    return (sharedInstance);
}

let config = [

    {
        name        : "baudRate",
        displayName : "Baud Rate",
        default     : 115200
    },
];

/*
 * ======== validate ========
 */
function validate(inst, validation)
{
    if (inst.baudRate > 3e6 || inst.baudRate <= 0) {
        logError(validation, inst, "baudRate", "Baud rate must be greater than"
        + " 0 and less than " + 3e6);
    }
}

/*
 *  ======== base ========
 */
let base = {
    displayName: "DAP",
    longDescription: `
The Device Agent Protocol (DAP) is a serial communication protocol using
packets that allows interaction between Edge AI Studio and the target device.
This protocol is specifically designed to support edge AI applications.
`,
    defaultInstanceName: "CONFIG_DAP_",
    config             : config,
    validate           : validate,
    moduleInstances    : moduleInstances,
    maxInstances: 1,
    templates: {
        "/ti/ai/edge_ai/templates/edgeai_template.h.xdt": true,
        "/ti/ai/edge_ai/templates/edgeai_template.c.xdt": true,
        "/ti/utils/build/GenLibs.cmd.xdt": { modName: "/ti/ai/edge_ai/DAP", getLibs: getLibs }
    }
};

exports = base;
