// SPDX-License-Identifier: GPL-2.0-only
/*
 * Driver for Texas Instruments INA423x power monitor chip
 * Datasheet: https://www.ti.com/product/ina4235
 * @author Mitch Morse <mitchmorse@ti.com>
 * Copyright (C) 2025 Texas Instruments Incorporated
 * 
 * Based off the ina238.c driver for the Texas Instruments INA238 power monitor chip
 * Copyright (C) 2021 Nathan Rossi <nathan.rossi@digi.com>
 *
 */
 
#include <linux/err.h>
#include <linux/hwmon.h>
#include <linux/hwmon-sysfs.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/regmap.h>
#include <linux/mutex.h>
#include <linux/util_macros.h>
#include <linux/bitfield.h>
#include <linux/property.h>

/* INA423x register definitions */
#define INA423x_CONFIG1					0x20
#define INA423x_CONFIG2					0x21
#define INA423x_CALIBRATION_CH(x)		(0x05 + (8 * (x - 1)))
#define INA423x_ALERT_CONFIG(x)			(0x07 + (8 * (x - 1)))
#define INA423x_ALERT_LIMIT(x)			(0x06 + (8 * (x - 1)))
#define INA423x_SHUNT_VOLTAGE_CH(x)		(0x00 + (8 * (x - 1)))
#define INA423x_BUS_VOLTAGE_CH(x)		(0x01 + (8 * (x - 1)))
#define INA423x_CURRENT_CH(x)			(0x02 + (8 * (x - 1)))
#define INA423x_POWER_CH(x)				(0x03 + (8 * (x - 1)))
#define INA423x_ENERGY_CH(x)			(0x04 + (8 * (x - 1)))
#define INA423x_FLAGS					0x22
#define INA423x_MANUFACTURER_ID			0x7E
#define INA423x_DEVICE_ID				0x7F

#define INA423x_CONFIG1_MODE_SHUNT		BIT(0)
#define INA423x_CONFIG1_MODE_BUS		BIT(1)
#define INA423x_CONFIG1_MODE_CONTINUOUS	BIT(2)
#define INA423x_CONFIG1_VSHCT			GENMASK(5, 3)
#define INA423x_CONFIG1_VBUSCT			GENMASK(8, 6)
#define INA423x_CONFIG1_AVG				GENMASK(11, 9)
#define INA423x_CONFIG1_ENABLE_CH(x)	BIT(11 + (x))

#define INA423x_CONFIG2_RANGE(x)		BIT((x) - 1)
#define INA423x_CONFIG2_ALERT_POL		BIT(4)
#define INA423x_CONFIG2_ALERT_LATCH		BIT(5)
#define INA423x_CONFIG2_ENOF_MASK		BIT(6)
#define INA423x_CONFIG2_CNVR_MASK		BIT(7)
#define INA423x_CONFIG2_ACC_RST(x)		BIT((x) + 7)

#define INA423x_ALERT_CONFIG_MASK		GENMASK(2, 0)
#define INA423x_ALERT_CONFIG_CHAN		GENMASK(4, 3)

#define INA423x_FLAGS_CONVERSION_READY	BIT(7)
#define INA423x_FLAGS_ENERGYOF_CH(x)	BIT(7 + (x))
#define INA423x_FLAGS_LIMIT_ALERT(x)	BIT(11 + (x))

#define INA423x_RSHUNT_DEFAULT			10000 /* uOhm */

/* Default configuration of device on reset. */
/* 4 chan enabled, 1 sample averaging, 1100us conversion time, continuous mode */
#define INA423x_CONFIG1_DEFAULT			0xF127
/* Range (all channels) ±81.92mV */
#define INA423x_CONFIG2_DEFAULT			0x0000
/* Alerts disabled and set to channel 1 */
#define INA423x_ALERT_CONFIG_DEFAULT	0x0000
/* limits set to device default of 0 */
#define INA423x_ALERT_LIMIT_DEFAULT		0x0000

/*
 * This driver uses a fixed calibration value in order to scale current/power
 * based on a fixed shunt resistor value. This allows for conversion within the
 * device to avoid integer limits while current/power accuracy is scaled
 * relative to the shunt resistor value within the driver. This is similar to
 * how the ina2xx and ina238 drivers handle current/power scaling.
 *
 * The end result of this is that increasing shunt values (from a fixed 2.5 mOhm
 * shunt) increase the effective current/power accuracy while limiting the
 * range and decreasing shunt values decrease the effective accuracy but
 * increase the range.
 *
 * The maximum shunt voltage is ±81.92 mV (0x7fff, ADC_RANGE = 0, gain = 4).
 *
 * The maximum current register value (for positive current) is 0x7FFF which is
 * 32767 decimal. At 1mA lsb, the maximum current value for the register is 32767 mA. 
 *
 * This puts our fixed shunt value at .08192/32.767 ≈ 2.5 mOhm.
 *
 * SHUNT_CAL = 0.00512 / (current_lsb * Rshunt)
 *
 * With chosen current_lsb of 1mA and fixed shunt of 2.5 mOhm we get:
 *   
 * SHUNT_CAL = 0.00512 / (0.001 * .0025) = 2048 = 0x800
 *
 * This scaling means the resulting values for Current and Power registers need
 * to be scaled by the difference between the fixed shunt resistor and the
 * actual shunt resistor:
 *
 *  Current (mA) = register value * 2500 / rshunt / 4 * gain
 *  Power (mW) = 32 * register value * 2500 / rshunt / 4 * gain
 *  Energy (mJ) = 32 * register value * 2500 / rshunt / 4 * gain
 */
#define INA423x_CALIBRATION_VALUE	2048
#define INA423x_FIXED_SHUNT			2500 /* 2.5 mOhm */
#define INA423x_SHUNT_VOLTAGE_LSB	2500 /* 2500 nV/lsb */
#define INA423x_BUS_VOLTAGE_LSB		1600 /* 1.6 mV/lsb */

/* Lookup table for Bus and Shunt conversion times in usec */
static const u16 ina423x_conv_time[] = {
	140, 204, 332, 588, 1100, 2116, 4156, 8244,
};

/* Lookup table for number of samples using in averaging mode */
static const int ina423x_avg_samples[] = {
	1, 4, 16, 64, 128, 256, 512, 1024,
};

// Define possible alarm modes
enum ina423x_alarm_mode {
	SOL = 1,
	SUL,
	BOL,
	BUL,
	POL,
};
//Defines for custom show/store functions
enum ina423x_chip_rw {
	CHANNEL1_LABEL = 1,
	CHANNEL2_LABEL,
	CHANNEL3_LABEL,
	CHANNEL4_LABEL,
	CHANNEL1_ENABLE,
	CHANNEL2_ENABLE,
	CHANNEL3_ENABLE,
	CHANNEL4_ENABLE,
	ALARM1_CHANNEL,
	ALARM2_CHANNEL,
	ALARM3_CHANNEL,
	ALARM4_CHANNEL,
	ALARM1_MODE,
	ALARM2_MODE,
	ALARM3_MODE,
	ALARM4_MODE,
	ALARM1_LIMIT,
	ALARM2_LIMIT,
	ALARM3_LIMIT,
	ALARM4_LIMIT,
	ALARM1,
	ALARM2,
	ALARM3,
	ALARM4,
	// items not indexed 1-4 must be at the end
	CONVERSION_MODE,
	CONVERSION_ENABLE_BUS,
	CONVERSION_ENABLE_SHUNT,
	CONVERSION_TIME_BUS_US,
	CONVERSION_TIME_SHUNT_US,
	CONVERSION_READY,
	ALERT_PIN_ENABLE_CONVERSION_READY,
	ALERT_PIN_ENABLE_ENERGY_OVERFLOW,
	ALERT_PIN_MODE,
	FLAGS_REGISTER,
};
enum ina423x_energy_ro {
	ENERGY1_INPUT = 1,
	ENERGY2_INPUT,
	ENERGY3_INPUT,
	ENERGY4_INPUT,
	ENERGY1_OVERFLOW,
	ENERGY2_OVERFLOW,
	ENERGY3_OVERFLOW,
	ENERGY4_OVERFLOW,
};
enum ina423x_energy_wo {
	ENERGY1_RESET = 1,
	ENERGY2_RESET,
	ENERGY3_RESET,
	ENERGY4_RESET,
};

#define INA423x_NUM_ALARMS 4
#define INA423x_NUM_CHANNELS 4

static const struct regmap_range ina423x_volatile_ranges[] = {
    /* Measurement data for Channel 1 */
    { .range_min = INA423x_SHUNT_VOLTAGE_CH(1), .range_max = INA423x_ENERGY_CH(1) },
    /* Measurement data for Channel 2 */
    { .range_min = INA423x_SHUNT_VOLTAGE_CH(2), .range_max = INA423x_ENERGY_CH(2) },
    /* Measurement data for Channel 3 */
    { .range_min = INA423x_SHUNT_VOLTAGE_CH(3), .range_max = INA423x_ENERGY_CH(3) },
    /* Measurement data for Channel 4 */
    { .range_min = INA423x_SHUNT_VOLTAGE_CH(4), .range_max = INA423x_ENERGY_CH(4) },
    
    /* System FLAGS register */
    { .range_min = INA423x_FLAGS, .range_max = INA423x_FLAGS },
};

static const struct regmap_access_table ina423x_volatile_table = {
    .yes_ranges = ina423x_volatile_ranges,
    .n_yes_ranges = ARRAY_SIZE(ina423x_volatile_ranges),
};

static const struct regmap_config ina423x_regmap_config = {
	.max_register = INA423x_DEVICE_ID,
	.reg_bits = 8,
	.val_bits = 16,
	.cache_type = REGCACHE_MAPLE, 
    .volatile_table = &ina423x_volatile_table,
};
enum ina423x_ids {ina4235, ina4230, ina423x};
struct ina423x_config {
	u32 power_calculate_factor;	/* fixed parameters for power calculate */
	u16 config1_default;		/* Power-on default state */
	u16 config2_default;		/* Power-on default state */
	int bus_voltage_lsb;		/* use for BUS calculate, uV/lsb */
};
struct ina423x_data {
	const struct ina423x_config *config;
	struct i2c_client *client;
	struct mutex config_lock;
	struct regmap *regmap;
	u32 rshunt[INA423x_NUM_CHANNELS]; 
	u8 alarm_mode[INA423x_NUM_ALARMS]; 
	int gain[INA423x_NUM_CHANNELS];
	const char *labels[INA423x_NUM_CHANNELS];
};
static const struct ina423x_config ina423x_config[] = {
	[ina4235] = {
		.power_calculate_factor = 32,
		.config1_default = INA423x_CONFIG1_DEFAULT,
		.config2_default = INA423x_CONFIG2_DEFAULT,
		.bus_voltage_lsb = INA423x_BUS_VOLTAGE_LSB,
	},
	[ina4230] = {
		.power_calculate_factor = 32,
		.config1_default = INA423x_CONFIG1_DEFAULT,
		.config2_default = INA423x_CONFIG2_DEFAULT,
		.bus_voltage_lsb = INA423x_BUS_VOLTAGE_LSB,
	},
	[ina423x] = {
		.power_calculate_factor = 32,
		.config1_default = INA423x_CONFIG1_DEFAULT,
		.config2_default = INA423x_CONFIG2_DEFAULT,
		.bus_voltage_lsb = INA423x_BUS_VOLTAGE_LSB,
	},
};
static int ina423x_read_reg32(const struct i2c_client *client, u8 reg, u32 *val)
{
	u8 data[4];
	int err;
	/* 32-bit register read */
	err = i2c_smbus_read_i2c_block_data(client, reg, 4, data);
	if (err < 0)
		return err;
	if (err != 4)
		return -EIO;
	*val = (data[0] << 24) | (data[1] << 16) | (data[2] << 8) | data[3];
	return 0;
}

static int ina423x_read_chip(struct device *dev, u32 attr, long *val)
{
	struct ina423x_data *data = dev_get_drvdata(dev);
	int regval;
	int reg, mask;
	int err;

	switch (attr) {
	case hwmon_chip_samples:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_AVG;
		err = regmap_read(data->regmap, reg, &regval);
		if (err < 0)
			return err;
		regval = FIELD_GET(INA423x_CONFIG1_AVG, regval);
		*val = ina423x_avg_samples[regval];
		break;
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}

static int ina423x_write_chip(struct device *dev, u32 attr, long val)
{
	struct ina423x_data *data = dev_get_drvdata(dev);
	int regval;
	int reg, mask;
	int index;
	int err;

	switch (attr) {
	case hwmon_chip_samples:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_AVG;
		index = find_closest(val, ina423x_avg_samples, ARRAY_SIZE(ina423x_avg_samples));
		regval = FIELD_PREP(INA423x_CONFIG1_AVG, index);
		err = regmap_update_bits(data->regmap, reg, mask, regval);
		if (err < 0)
			return err;
		break;
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}

static ssize_t ina423x_chip_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sensor_device_attribute *sd_attr = to_sensor_dev_attr(attr);
	struct ina423x_data *data = dev_get_drvdata(dev);
	int reg, mask;
	int regval;
	long val;
	int mode;
	long power;
	int err;

	unsigned int attr_ind = sd_attr->index;
	unsigned int channel = ((attr_ind - 1) % 4 ) + 1;

switch (attr_ind) {
	case CHANNEL1_LABEL:
	case CHANNEL2_LABEL:
	case CHANNEL3_LABEL:
	case CHANNEL4_LABEL:
		return sysfs_emit(buf, "%s\n", data->labels[channel-1]);
	case CHANNEL1_ENABLE:
	case CHANNEL2_ENABLE:
	case CHANNEL3_ENABLE:
	case CHANNEL4_ENABLE:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_ENABLE_CH(channel);
		break;
	case CONVERSION_MODE:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_MODE_CONTINUOUS;
		break;
	case CONVERSION_ENABLE_BUS:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_MODE_BUS;
		break;
	case CONVERSION_ENABLE_SHUNT:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_MODE_SHUNT;
		break;
	case CONVERSION_TIME_BUS_US:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_VBUSCT;
		break;
	case CONVERSION_TIME_SHUNT_US:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_VSHCT;
		break;
	case CONVERSION_READY:
		reg = INA423x_FLAGS;
		mask = INA423x_FLAGS_CONVERSION_READY;
		break;
	case ALERT_PIN_ENABLE_CONVERSION_READY:
		reg = INA423x_CONFIG2;
		mask = INA423x_CONFIG2_CNVR_MASK;
		break;
	case ALERT_PIN_ENABLE_ENERGY_OVERFLOW:
		reg = INA423x_CONFIG2;
		mask = INA423x_CONFIG2_ENOF_MASK;
		break;
	case ALERT_PIN_MODE:
		reg = INA423x_CONFIG2;
		mask = INA423x_CONFIG2_ALERT_LATCH;
		break;
	case ALARM1_CHANNEL:
	case ALARM2_CHANNEL:
	case ALARM3_CHANNEL:
	case ALARM4_CHANNEL:
		reg = INA423x_ALERT_CONFIG(channel);
		mask = INA423x_ALERT_CONFIG_CHAN;
		break;
	case ALARM1_MODE:
	case ALARM2_MODE:
	case ALARM3_MODE:
	case ALARM4_MODE:
		reg = INA423x_ALERT_CONFIG(channel);
		mask = INA423x_ALERT_CONFIG_MASK;
		break;
	case ALARM1_LIMIT:
	case ALARM2_LIMIT:
	case ALARM3_LIMIT:
	case ALARM4_LIMIT:
		reg = INA423x_ALERT_LIMIT(channel);
		break;
	case ALARM1:
	case ALARM2:
	case ALARM3:
	case ALARM4:
		reg = INA423x_FLAGS;
		mask = INA423x_FLAGS_LIMIT_ALERT(channel);
		break;
	case FLAGS_REGISTER:
		reg = INA423x_FLAGS;
		break;
	default:
		return -EOPNOTSUPP;
	}

	err = regmap_read(data->regmap, reg, &regval);
	if (err < 0)
		return err;

	switch (attr_ind) {
	case CONVERSION_TIME_BUS_US: 
		regval = FIELD_GET(INA423x_CONFIG1_VBUSCT, regval);
		val = ina423x_conv_time[regval];
		break;
	case CONVERSION_TIME_SHUNT_US: 
		regval = FIELD_GET(INA423x_CONFIG1_VSHCT, regval);
		val = ina423x_conv_time[regval];
		break;
	case ALARM1_CHANNEL:
	case ALARM2_CHANNEL:
	case ALARM3_CHANNEL:
	case ALARM4_CHANNEL:
		val = FIELD_GET(INA423x_ALERT_CONFIG_CHAN, regval) + 1; //add one to display channel number
		break;
	case ALARM1_MODE:
	case ALARM2_MODE:
	case ALARM3_MODE:
	case ALARM4_MODE:
		val = (regval & mask); 
		break;
	case ALARM1_LIMIT:
	case ALARM2_LIMIT:
	case ALARM3_LIMIT:
	case ALARM4_LIMIT:
		mode = data->alarm_mode[channel-1];
		switch (mode) {
		case SOL:
		case SUL:
			//convert as shunt limit
			val = (regval * INA423x_SHUNT_VOLTAGE_LSB) * data->gain[channel-1] / (1000 * 4);
			break;
		case BOL:
		case BUL:
			//convert as bus limit
			val = (regval * data->config->bus_voltage_lsb) / 1000;
			break;
		case POL:
			//convert as power limit 
			power = div_u64(regval * 1000ULL * INA423x_FIXED_SHUNT * data->gain[channel-1] * data->config->power_calculate_factor, data->rshunt[channel-1] * 4);
			/* Clamp value to maximum value of long */
			val = clamp_val(power, 0, LONG_MAX);
			break;
		default:
			return -EINVAL;
		}
		break;
	case FLAGS_REGISTER:
		val = regval;
		break;
	case CHANNEL1_ENABLE:
	case CHANNEL2_ENABLE:
	case CHANNEL3_ENABLE:
	case CHANNEL4_ENABLE:
	case CONVERSION_MODE:
	case CONVERSION_ENABLE_BUS:
	case CONVERSION_ENABLE_SHUNT:
	case CONVERSION_READY:
	case ALERT_PIN_ENABLE_CONVERSION_READY:
	case ALERT_PIN_ENABLE_ENERGY_OVERFLOW:
	case ALERT_PIN_MODE:
	case ALARM1:
	case ALARM2:
	case ALARM3:
	case ALARM4:
		val = !!(regval & mask);
		break;
	default:
		return -EOPNOTSUPP;
	}	

	return sysfs_emit(buf, "%ld\n", val);
}

static ssize_t ina423x_chip_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    struct sensor_device_attribute *sd_attr = to_sensor_dev_attr(attr);
    struct ina423x_data *data = dev_get_drvdata(dev);
	int regval;
	int reg, mask;
	long val;
	int mode;
	int index;
	int err;

	unsigned int attr_ind = sd_attr->index;
	unsigned int channel = ((attr_ind - 1) % 4 ) + 1;

    // Safely parse the input string 'buf' into a long integer (base 10)
    err = kstrtol(buf, 10, &val);
    if (err)
		return err;

	switch (attr_ind) {
	case CHANNEL1_ENABLE:
	case CHANNEL2_ENABLE:
	case CHANNEL3_ENABLE:
	case CHANNEL4_ENABLE:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_ENABLE_CH(channel);
		regval = val ? INA423x_CONFIG1_ENABLE_CH(channel) : 0;
		break;
	case CONVERSION_MODE:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_MODE_CONTINUOUS;
		regval = val ? INA423x_CONFIG1_MODE_CONTINUOUS : 0;
		break;
	case CONVERSION_ENABLE_BUS:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_MODE_BUS;
		regval = val ? INA423x_CONFIG1_MODE_BUS : 0;
		break;
	case CONVERSION_ENABLE_SHUNT:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_MODE_SHUNT;
		regval = val ? INA423x_CONFIG1_MODE_SHUNT : 0;
		break;
	case CONVERSION_TIME_BUS_US:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_VBUSCT;
		index = find_closest(val, ina423x_conv_time, ARRAY_SIZE(ina423x_conv_time));
		regval = FIELD_PREP(INA423x_CONFIG1_VBUSCT, index);
		break;
	case CONVERSION_TIME_SHUNT_US:
		reg = INA423x_CONFIG1;
		mask = INA423x_CONFIG1_VSHCT;
		index = find_closest(val, ina423x_conv_time, ARRAY_SIZE(ina423x_conv_time));
		regval = FIELD_PREP(INA423x_CONFIG1_VSHCT, index);
		break;
	case ALERT_PIN_ENABLE_CONVERSION_READY:
		reg = INA423x_CONFIG2;
		mask = INA423x_CONFIG2_CNVR_MASK;
		regval = val ? INA423x_CONFIG2_CNVR_MASK : 0;
		break;
	case ALERT_PIN_ENABLE_ENERGY_OVERFLOW:
		reg = INA423x_CONFIG2;
		mask = INA423x_CONFIG2_ENOF_MASK;
		regval = val ? INA423x_CONFIG2_ENOF_MASK : 0;
		break;
	case ALERT_PIN_MODE:
		reg = INA423x_CONFIG2;
		mask = INA423x_CONFIG2_ALERT_LATCH;
		regval = val ? INA423x_CONFIG2_ALERT_LATCH : 0;
		break;
	case ALARM1_CHANNEL:
	case ALARM2_CHANNEL:
	case ALARM3_CHANNEL:
	case ALARM4_CHANNEL:
		reg = INA423x_ALERT_CONFIG(channel);
		mask = INA423x_ALERT_CONFIG_CHAN;
		if (val < 1 || val > 4)
        	return -EINVAL;
		regval = FIELD_PREP(INA423x_ALERT_CONFIG_CHAN, val-1);
		break;
	case ALARM1_MODE:
	case ALARM2_MODE:
	case ALARM3_MODE:
	case ALARM4_MODE:
		reg = INA423x_ALERT_CONFIG(channel);
		mask = INA423x_ALERT_CONFIG_MASK;
		if (val < 0 || val > 5)
        	return -EINVAL;
		data->alarm_mode[channel-1] = val;
		regval = val;
		break;
	case ALARM1_LIMIT:
	case ALARM2_LIMIT:
	case ALARM3_LIMIT:
	case ALARM4_LIMIT:
		reg = INA423x_ALERT_LIMIT(channel);
		mode = data->alarm_mode[channel-1];
		switch (mode) {
		case SOL:
		case SUL:
			//convert as shunt limit
			regval = val * (1000 * 4) / (INA423x_SHUNT_VOLTAGE_LSB * data->gain[channel-1]);
			break;
		case BOL:
		case BUL:
			//convert as bus limit
			regval = val * 1000 / (data->config->bus_voltage_lsb);
			break;
		case POL:
			//convert as power limit
			regval = val * data->rshunt[channel-1] * 4 / (1000ULL * INA423x_FIXED_SHUNT * data->gain[channel-1] * data->config->power_calculate_factor);
			break; 
		default:
			return -EINVAL;
		}
		break;
	default:
		return -EOPNOTSUPP;
	}
	/* write value to device */	
	switch (attr_ind) {
	case CHANNEL1_ENABLE:
	case CHANNEL2_ENABLE:
	case CHANNEL3_ENABLE:
	case CHANNEL4_ENABLE:
	case CONVERSION_MODE:
	case CONVERSION_ENABLE_BUS:
	case CONVERSION_ENABLE_SHUNT:
	case CONVERSION_TIME_BUS_US:
	case CONVERSION_TIME_SHUNT_US:
	case ALERT_PIN_ENABLE_CONVERSION_READY:
	case ALERT_PIN_ENABLE_ENERGY_OVERFLOW:
	case ALERT_PIN_MODE:
	case ALARM1_CHANNEL:
	case ALARM2_CHANNEL:
	case ALARM3_CHANNEL:
	case ALARM4_CHANNEL:
	case ALARM1_MODE:
	case ALARM2_MODE:
	case ALARM3_MODE:
	case ALARM4_MODE:
		err = regmap_update_bits(data->regmap, reg, mask, regval);
		if (err < 0)
			return err;
		break;
	case ALARM1_LIMIT:
	case ALARM2_LIMIT:
	case ALARM3_LIMIT:
	case ALARM4_LIMIT:
		err = regmap_write(data->regmap, INA423x_ALERT_LIMIT(channel), regval);
		if (err < 0)
			return err;
		break;
	default:
		return -EOPNOTSUPP;
	}	

    // Return the number of bytes successfully processed
    return count;
}

static int ina423x_read_in(struct device *dev, u32 attr, int channel, long *val)
{
	struct ina423x_data *data = dev_get_drvdata(dev);
	int reg;
	int regval;
	int err;
	if (channel < 1 || channel > 8)
		return -EOPNOTSUPP;

	unsigned int chan = ((channel - 1) % 4 ) + 1;

	switch (channel) {
	case 1:
	case 2:
	case 3:
	case 4:
		switch (attr) {
		case hwmon_in_input:
			reg = INA423x_BUS_VOLTAGE_CH(chan);
			break;
		default:
			return -EOPNOTSUPP;
		}
		break;
	case 5:
	case 6:
	case 7:
	case 8:
		switch (attr) {
		case hwmon_in_input:
			reg = INA423x_SHUNT_VOLTAGE_CH(chan);
			break;
		default:
			return -EOPNOTSUPP;
		}
		break;
	default:
		return -EOPNOTSUPP;
	}
	err = regmap_read(data->regmap, reg, &regval);
	if (err < 0)
		return err;
	switch (attr) {
	case hwmon_in_input:
		/* signed register */
		regval = (s16)regval;
		if (channel >= 1 && channel <= 4)
			/* value in mV */
			*val = (regval * data->config->bus_voltage_lsb) / 1000;
		else
			/* value in µV, gain of 1 -> LSB / 4 */
			*val = (regval * INA423x_SHUNT_VOLTAGE_LSB) * data->gain[chan-1] / (1000 * 4);
		break;
	}
	return 0;
}

static int ina423x_read_current(struct device *dev, u32 attr, int channel, long *val)
{
	struct ina423x_data *data = dev_get_drvdata(dev);
	int regval;
	int err;
	
	if (channel < 1 || channel > 4)
		return -EOPNOTSUPP;
	switch (attr) {
	case hwmon_curr_input:
		err = regmap_read(data->regmap, INA423x_CURRENT_CH(channel), &regval);
		if (err < 0)
			return err;
		/* Signed register, fixed 1mA current lsb. result in mA */
		*val = div_s64((s16)regval * INA423x_FIXED_SHUNT * data->gain[channel-1], data->rshunt[channel-1] * 4); 
		break;
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}
static int ina423x_read_power(struct device *dev, u32 attr, int channel, long *val)
{
	struct ina423x_data *data = dev_get_drvdata(dev);
	long power;
	int regval;
	int err;
	if (channel < 1 || channel > 4)
		return -EOPNOTSUPP;
	switch (attr) {
	case hwmon_power_input:
		err = regmap_read(data->regmap, INA423x_POWER_CH(channel), &regval);
		if (err < 0)
			return err;
		/* Fixed 1mA lsb, scaled by 1000000 to have result in uW */
		power = div_u64(regval * 1000ULL * INA423x_FIXED_SHUNT * data->gain[channel-1] * data->config->power_calculate_factor, data->rshunt[channel-1] * 4);
		/* Clamp value to maximum value of long */
		*val = clamp_val(power, 0, LONG_MAX);
		break;
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}

static ssize_t ina423x_energy_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct sensor_device_attribute *sd_attr = to_sensor_dev_attr(attr);
	struct ina423x_data *data = dev_get_drvdata(dev);
	int reg, mask;
	int err;
	u32 regval;
	u64 energy;
	long val;
	unsigned int attr_ind = sd_attr->index;
	unsigned int channel = ((attr_ind - 1) % 4 ) + 1;

	switch (attr_ind) {
	case ENERGY1_INPUT:
	case ENERGY2_INPUT:
	case ENERGY3_INPUT:
	case ENERGY4_INPUT:
		reg = INA423x_ENERGY_CH(channel);
		err = ina423x_read_reg32(data->client, reg, &regval);
		if (err < 0)
			return err;
		/* result in µJ */
		energy = div_u64(regval * 1000ULL * INA423x_FIXED_SHUNT *	data->gain[channel-1] * data->config->power_calculate_factor, data->rshunt[channel-1] * 4);
		return sysfs_emit(buf, "%llu\n", energy);
	case ENERGY1_OVERFLOW:
	case ENERGY2_OVERFLOW:
	case ENERGY3_OVERFLOW:
	case ENERGY4_OVERFLOW:
		reg = INA423x_FLAGS;
		mask = INA423x_FLAGS_ENERGYOF_CH(channel);
		err = regmap_read(data->regmap, reg, &regval);
		if (err < 0)
			return err;
		val = !!(regval & mask);
		return sysfs_emit(buf, "%ld\n", val);
	default:
		return -EOPNOTSUPP;
	}
}

static ssize_t ina423x_energy_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    struct sensor_device_attribute *sd_attr = to_sensor_dev_attr(attr);
    unsigned int channel = sd_attr->index;
    struct ina423x_data *data = dev_get_drvdata(dev);

	if (channel < 1 || channel > 4)
		return -EOPNOTSUPP;

	int regval;
    long val;
    int err;

    // Safely parse the input string 'buf' into a long integer (base 10)
    err = kstrtol(buf, 10, &val);
    if (err)
		return err;

	regval = val ? INA423x_CONFIG2_ACC_RST(channel) : 0;

	err = regmap_update_bits(data->regmap, INA423x_CONFIG2, INA423x_CONFIG2_ACC_RST(channel), regval);
	if (err < 0)
		return err;

    // Return the number of bytes successfully processed
    return count;
}

static int ina423x_read(struct device *dev, enum hwmon_sensor_types type, u32 attr, int channel, long *val)
{
	switch (type) {
	case hwmon_chip:
		return ina423x_read_chip(dev, attr, val);
	case hwmon_in:
		return ina423x_read_in(dev, attr, channel, val);
	case hwmon_curr:
		return ina423x_read_current(dev, attr, channel + 1, val);
	case hwmon_power:
		return ina423x_read_power(dev, attr, channel + 1, val);
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}
static int ina423x_write(struct device *dev, enum hwmon_sensor_types type, u32 attr, int channel, long val)
{
	struct ina423x_data *data = dev_get_drvdata(dev);
	int err;
	
	mutex_lock(&data->config_lock);
	switch (type) {
	case hwmon_chip:
		err = ina423x_write_chip(dev, attr, val);
		break;
	default:
		err = -EOPNOTSUPP;
		break;
	}
	mutex_unlock(&data->config_lock);
	return err;
}
static umode_t ina423x_is_visible(const void *drvdata, enum hwmon_sensor_types type, u32 attr, int channel)
{
	//const struct ina423x_data *data = drvdata;

	switch (type) {
	case hwmon_chip:
		switch (attr) {
		case hwmon_chip_samples:
			return 0644;
		default:
			return 0;
		}
	case hwmon_in:
		/* Ignore in0_ */
		if (channel == 0)
			return 0;

		switch (attr) {
		case hwmon_in_input:
			return 0444;
		default:
			return 0;
		}
	case hwmon_curr:
		switch (attr) {
		case hwmon_curr_input:
			return 0444;
		default:
			return 0;
		}
	case hwmon_power:
		switch (attr) {
		case hwmon_power_input:
			return 0444;
		default:
			return 0;
		}
	default:
		return 0;
	}
}

/* custom chip attributes */
static SENSOR_DEVICE_ATTR_RO(channel1_label, ina423x_chip, CHANNEL1_LABEL);
static SENSOR_DEVICE_ATTR_RO(channel2_label, ina423x_chip, CHANNEL2_LABEL);
static SENSOR_DEVICE_ATTR_RO(channel3_label, ina423x_chip, CHANNEL3_LABEL);
static SENSOR_DEVICE_ATTR_RO(channel4_label, ina423x_chip, CHANNEL4_LABEL);

static SENSOR_DEVICE_ATTR_RW(channel1_enable, ina423x_chip, CHANNEL1_ENABLE);
static SENSOR_DEVICE_ATTR_RW(channel2_enable, ina423x_chip, CHANNEL2_ENABLE);
static SENSOR_DEVICE_ATTR_RW(channel3_enable, ina423x_chip, CHANNEL3_ENABLE);
static SENSOR_DEVICE_ATTR_RW(channel4_enable, ina423x_chip, CHANNEL4_ENABLE);

static SENSOR_DEVICE_ATTR_RW(conversion_mode, ina423x_chip, CONVERSION_MODE);
static SENSOR_DEVICE_ATTR_RW(conversion_enable_bus, ina423x_chip, CONVERSION_ENABLE_BUS);
static SENSOR_DEVICE_ATTR_RW(conversion_enable_shunt, ina423x_chip, CONVERSION_ENABLE_SHUNT);
static SENSOR_DEVICE_ATTR_RW(conversion_time_bus_us, ina423x_chip, CONVERSION_TIME_BUS_US);
static SENSOR_DEVICE_ATTR_RW(conversion_time_shunt_us, ina423x_chip, CONVERSION_TIME_SHUNT_US);

static SENSOR_DEVICE_ATTR_RO(conversion_ready, ina423x_chip, CONVERSION_READY);

static SENSOR_DEVICE_ATTR_RW(alert_pin_enable_conversion_ready, ina423x_chip, ALERT_PIN_ENABLE_CONVERSION_READY);
static SENSOR_DEVICE_ATTR_RW(alert_pin_enable_energy_overflow, ina423x_chip, ALERT_PIN_ENABLE_ENERGY_OVERFLOW);
static SENSOR_DEVICE_ATTR_RW(alert_pin_mode, ina423x_chip, ALERT_PIN_MODE);

static SENSOR_DEVICE_ATTR_RW(alarm1_channel, ina423x_chip, ALARM1_CHANNEL);
static SENSOR_DEVICE_ATTR_RW(alarm2_channel, ina423x_chip, ALARM2_CHANNEL);
static SENSOR_DEVICE_ATTR_RW(alarm3_channel, ina423x_chip, ALARM3_CHANNEL);
static SENSOR_DEVICE_ATTR_RW(alarm4_channel, ina423x_chip, ALARM4_CHANNEL);

static SENSOR_DEVICE_ATTR_RW(alarm1_mode, ina423x_chip, ALARM1_MODE);
static SENSOR_DEVICE_ATTR_RW(alarm2_mode, ina423x_chip, ALARM2_MODE);
static SENSOR_DEVICE_ATTR_RW(alarm3_mode, ina423x_chip, ALARM3_MODE);
static SENSOR_DEVICE_ATTR_RW(alarm4_mode, ina423x_chip, ALARM4_MODE);

static SENSOR_DEVICE_ATTR_RW(alarm1_limit, ina423x_chip, ALARM1_LIMIT);
static SENSOR_DEVICE_ATTR_RW(alarm2_limit, ina423x_chip, ALARM2_LIMIT);
static SENSOR_DEVICE_ATTR_RW(alarm3_limit, ina423x_chip, ALARM3_LIMIT);
static SENSOR_DEVICE_ATTR_RW(alarm4_limit, ina423x_chip, ALARM4_LIMIT);

static SENSOR_DEVICE_ATTR_RO(flags_register, ina423x_chip, FLAGS_REGISTER);

static SENSOR_DEVICE_ATTR_RO(alarm1, ina423x_chip, ALARM1);
static SENSOR_DEVICE_ATTR_RO(alarm2, ina423x_chip, ALARM2);
static SENSOR_DEVICE_ATTR_RO(alarm3, ina423x_chip, ALARM3);
static SENSOR_DEVICE_ATTR_RO(alarm4, ina423x_chip, ALARM4);

/* custom energy attributes. energy input attributes are 4 bytes wide so we need u32 */
static SENSOR_DEVICE_ATTR_RO(energy1_input, ina423x_energy, ENERGY1_INPUT);
static SENSOR_DEVICE_ATTR_RO(energy2_input, ina423x_energy, ENERGY2_INPUT);
static SENSOR_DEVICE_ATTR_RO(energy3_input, ina423x_energy, ENERGY3_INPUT);
static SENSOR_DEVICE_ATTR_RO(energy4_input, ina423x_energy, ENERGY4_INPUT);

static SENSOR_DEVICE_ATTR_RO(energy1_overflow, ina423x_energy, ENERGY1_OVERFLOW);
static SENSOR_DEVICE_ATTR_RO(energy2_overflow, ina423x_energy, ENERGY2_OVERFLOW);
static SENSOR_DEVICE_ATTR_RO(energy3_overflow, ina423x_energy, ENERGY3_OVERFLOW);
static SENSOR_DEVICE_ATTR_RO(energy4_overflow, ina423x_energy, ENERGY4_OVERFLOW);

/* custom energy attributes */
static SENSOR_DEVICE_ATTR_WO(energy1_reset, ina423x_energy, ENERGY1_RESET);
static SENSOR_DEVICE_ATTR_WO(energy2_reset, ina423x_energy, ENERGY2_RESET);
static SENSOR_DEVICE_ATTR_WO(energy3_reset, ina423x_energy, ENERGY3_RESET);
static SENSOR_DEVICE_ATTR_WO(energy4_reset, ina423x_energy, ENERGY4_RESET);

static struct attribute *ina423x_attrs[] = {
	&sensor_dev_attr_channel1_label.dev_attr.attr,
	&sensor_dev_attr_channel2_label.dev_attr.attr,
	&sensor_dev_attr_channel3_label.dev_attr.attr,
	&sensor_dev_attr_channel4_label.dev_attr.attr,
	&sensor_dev_attr_channel1_enable.dev_attr.attr,
	&sensor_dev_attr_channel2_enable.dev_attr.attr,
	&sensor_dev_attr_channel3_enable.dev_attr.attr,
	&sensor_dev_attr_channel4_enable.dev_attr.attr,
	&sensor_dev_attr_conversion_mode.dev_attr.attr,
	&sensor_dev_attr_conversion_enable_bus.dev_attr.attr,
	&sensor_dev_attr_conversion_enable_shunt.dev_attr.attr,
	&sensor_dev_attr_conversion_time_bus_us.dev_attr.attr,
	&sensor_dev_attr_conversion_time_shunt_us.dev_attr.attr,
	&sensor_dev_attr_conversion_ready.dev_attr.attr,
	&sensor_dev_attr_alert_pin_enable_conversion_ready.dev_attr.attr,
	&sensor_dev_attr_alert_pin_enable_energy_overflow.dev_attr.attr,
	&sensor_dev_attr_alert_pin_mode.dev_attr.attr,
	&sensor_dev_attr_alarm1_channel.dev_attr.attr,
	&sensor_dev_attr_alarm2_channel.dev_attr.attr,
	&sensor_dev_attr_alarm3_channel.dev_attr.attr,
	&sensor_dev_attr_alarm4_channel.dev_attr.attr,
	&sensor_dev_attr_alarm1_mode.dev_attr.attr,
	&sensor_dev_attr_alarm2_mode.dev_attr.attr,
	&sensor_dev_attr_alarm3_mode.dev_attr.attr,
	&sensor_dev_attr_alarm4_mode.dev_attr.attr,
	&sensor_dev_attr_alarm1_limit.dev_attr.attr,
	&sensor_dev_attr_alarm2_limit.dev_attr.attr,
	&sensor_dev_attr_alarm3_limit.dev_attr.attr,
	&sensor_dev_attr_alarm4_limit.dev_attr.attr,
	&sensor_dev_attr_alarm1.dev_attr.attr,
	&sensor_dev_attr_alarm2.dev_attr.attr,
	&sensor_dev_attr_alarm3.dev_attr.attr,
	&sensor_dev_attr_alarm4.dev_attr.attr,
	&sensor_dev_attr_energy1_input.dev_attr.attr,
	&sensor_dev_attr_energy2_input.dev_attr.attr,
	&sensor_dev_attr_energy3_input.dev_attr.attr,
	&sensor_dev_attr_energy4_input.dev_attr.attr,
	&sensor_dev_attr_energy1_overflow.dev_attr.attr,
	&sensor_dev_attr_energy2_overflow.dev_attr.attr,
	&sensor_dev_attr_energy3_overflow.dev_attr.attr,
	&sensor_dev_attr_energy4_overflow.dev_attr.attr,
	&sensor_dev_attr_energy1_reset.dev_attr.attr,
	&sensor_dev_attr_energy2_reset.dev_attr.attr,
	&sensor_dev_attr_energy3_reset.dev_attr.attr,
	&sensor_dev_attr_energy4_reset.dev_attr.attr,
	&sensor_dev_attr_flags_register.dev_attr.attr,
	NULL,
};

ATTRIBUTE_GROUPS(ina423x);

static const struct hwmon_channel_info * const ina423x_info[] = {
	HWMON_CHANNEL_INFO(chip,
			   HWMON_C_SAMPLES),
	HWMON_CHANNEL_INFO(in,
			   /* 0: dummy, skipped in is_visible */
			   HWMON_I_INPUT,
			   /* 1-4: bus voltage */
			   HWMON_I_INPUT,
			   HWMON_I_INPUT,
			   HWMON_I_INPUT,
			   HWMON_I_INPUT,
			   /* 5-8: shunt voltage */
			   HWMON_I_INPUT,
			   HWMON_I_INPUT,
			   HWMON_I_INPUT,
			   HWMON_I_INPUT),
	HWMON_CHANNEL_INFO(curr,
			   /* 1-4: current through shunt */
			   HWMON_C_INPUT,
			   HWMON_C_INPUT,
			   HWMON_C_INPUT,
			   HWMON_C_INPUT),
	HWMON_CHANNEL_INFO(power,
			   /* 1-4: power */
			   HWMON_P_INPUT,
			   HWMON_P_INPUT,
			   HWMON_P_INPUT,
			   HWMON_P_INPUT),
	NULL
};

static const struct hwmon_ops ina423x_hwmon_ops = {
	.is_visible = ina423x_is_visible,
	.read = ina423x_read,
	.write = ina423x_write,
};
static const struct hwmon_chip_info ina423x_chip_info = {
	.ops = &ina423x_hwmon_ops,
	.info = ina423x_info,
};

static int ina423x_probe(struct i2c_client *client)
{
	struct device *dev = &client->dev;
	struct device *hwmon_dev;
	struct ina423x_data *data;
	enum ina423x_ids chip;
	int config1;
	int config2;
	const char *temp_labels[INA423x_NUM_CHANNELS];
	int numLabels;
	bool active_high;
	const char *polarity_str;
	int ret;
	int i;

	chip = (uintptr_t)i2c_get_match_data(client);
	data = devm_kzalloc(dev, sizeof(*data), GFP_KERNEL);
	if (!data)
		return -ENOMEM;
	data->client = client;
	/* set the device type */
	data->config = &ina423x_config[chip];
	mutex_init(&data->config_lock);
	data->regmap = devm_regmap_init_i2c(client, &ina423x_regmap_config);
	if (IS_ERR(data->regmap)) {
		dev_err(dev, "failed to allocate register map\n");
		return PTR_ERR(data->regmap);
	}

	/* load channel labels */
	numLabels = device_property_read_string_array(dev, "channel-labels", temp_labels, INA423x_NUM_CHANNELS);
	for (i = 0; i < INA423x_NUM_CHANNELS; i++) {
		if (numLabels > 0 && i < numLabels) {
			data->labels[i] = temp_labels[i];
		} else {
			data->labels[i] = devm_kasprintf(dev, GFP_KERNEL, "Channel %d", i + 1);
			if (!data->labels[i])
				return -ENOMEM;
		}
	}

	/* load shunt values */
    if (device_property_read_u32_array(dev, "shunt-resistor", data->rshunt, INA423x_NUM_CHANNELS) < 0) {
		for (i = 0; i < INA423x_NUM_CHANNELS; i++)
			data->rshunt[i] = INA423x_RSHUNT_DEFAULT;
    }
    /* validate shunt values for each channel */
    for (i = 0; i < INA423x_NUM_CHANNELS; i++) {
        if (data->rshunt[i] == 0) {
            dev_err(dev, "invalid shunt resistor value %u for channel %d\n", data->rshunt[i], i+1);
            return -EINVAL;
        }
    }

	/* load shunt gain values */
    if (device_property_read_u32_array(dev, "shunt-gain", data->gain, INA423x_NUM_CHANNELS) < 0) {
		for (i = 0; i < INA423x_NUM_CHANNELS; i++)
			data->gain[i] = 4; /* Default to ADCRANGE = 0 */
	}

	/* validate gain values for each channel */
	for (i = 0; i < INA423x_NUM_CHANNELS; i++) {
		if (data->gain[i] != 1 && data->gain[i] != 4) {
			dev_err(dev, "invalid shunt gain value %u for channel %d\n", data->gain[i], i+1);
			return -EINVAL;
		}
	}

	/* Setup CONFIG1 register */
	config1 = data->config->config1_default;

	ret = regmap_write(data->regmap, INA423x_CONFIG1, config1);
	if (ret < 0) {
		dev_err(dev, "error configuring the device: %d\n", ret);
		return -ENODEV;
	}

	/* Setup CONFIG2 register */
	config2 = data->config->config2_default;

	for(i = 0; i < INA423x_NUM_CHANNELS; i++) {
		if (data->gain[0] == 1) {
			config2 |= INA423x_CONFIG2_RANGE(i+1); /* ADCRANGE = 1 is /1 */
		}
	}
	active_high = device_property_read_bool(dev, "ti,alert-polarity-active-high");
	config2 = config2 | FIELD_PREP(INA423x_CONFIG2_ALERT_POL, active_high);
	ret = regmap_write(data->regmap, INA423x_CONFIG2, config2);
	if (ret < 0) {
		dev_err(dev, "error configuring the device: %d\n", ret);
		return -ENODEV;
	}
	polarity_str = devm_kasprintf(dev, GFP_KERNEL, "%s", active_high ? "Active High" : "Active Low");

	/* Setup SHUNT_CALIBRATION register with fixed value */
	for(i = 0; i < INA423x_NUM_CHANNELS; i++) {
		ret = regmap_write(data->regmap, INA423x_CALIBRATION_CH(i+1), INA423x_CALIBRATION_VALUE);
		if (ret < 0) {
			dev_err(dev, "error configuring the device for channel %d: %d\n", i+1, ret);
			return -ENODEV;
		}
	}
	
	/* Setup alert config with device default values */
	for(i = 0; i < INA423x_NUM_CHANNELS; i++) {
		ret = regmap_write(data->regmap, INA423x_ALERT_CONFIG(i+1), INA423x_ALERT_CONFIG_DEFAULT);
		if (ret < 0) {
			dev_err(dev, "error configuring the device for channel %d: %d\n", i+1, ret);
			return -ENODEV;
		}
	}
	/* Setup alert limits with device default values */
	for(i = 0; i < INA423x_NUM_CHANNELS; i++) {
		ret = regmap_write(data->regmap, INA423x_ALERT_LIMIT(i+1), INA423x_ALERT_LIMIT_DEFAULT);
		if (ret < 0) {
			dev_err(dev, "error configuring the device for channel %d: %d\n", i+1, ret);
			return -ENODEV;
		}
	}
	
	hwmon_dev = devm_hwmon_device_register_with_info(dev, client->name, data, &ina423x_chip_info, ina423x_groups);
	if (IS_ERR(hwmon_dev))
		return PTR_ERR(hwmon_dev);
	dev_info(dev, "power monitor %s (Channel1_label = %s, Channel2_label = %s, Channel3_label = %s, Channel4_label = %s, Rshunt1 = %u uOhm, Rshunt2 = %u uOhm, Rshunt3 = %u uOhm, Rshunt4 = %u uOhm, gain1 = %u, gain2 = %u, gain3 = %u, gain4 = %u, Alert Polarity: %s)\n",
		 client->name, data->labels[0], data->labels[1], data->labels[2], data->labels[3], data->rshunt[0], data->rshunt[1], data->rshunt[2], data->rshunt[3], data->gain[0], data->gain[1], data->gain[2], data->gain[3], polarity_str);
	return 0;
}
static const struct i2c_device_id ina423x_id[] = {
	{ "ina4235", ina423x },
	{ "ina4230", ina423x },
	{ "ina423x", ina423x },
	{ }
};
MODULE_DEVICE_TABLE(i2c, ina423x_id);
static const struct of_device_id __maybe_unused ina423x_of_match[] = {
	{
		.compatible = "ti,ina4235",
		.data = (void *)ina423x
	},
	{
		.compatible = "ti,ina4230",
		.data = (void *)ina423x
	},
	{
		.compatible = "ti,ina423x",
		.data = (void *)ina423x
	},
	{ }
};
MODULE_DEVICE_TABLE(of, ina423x_of_match);
static struct i2c_driver ina423x_driver = {
	.driver = {
		.name	= "ina423x",
		.of_match_table = of_match_ptr(ina423x_of_match),
	},
	.probe		= ina423x_probe,
	.id_table	= ina423x_id,
};
module_i2c_driver(ina423x_driver);
MODULE_AUTHOR("Mitch Morse <mitchmorse@ti.com>");
MODULE_DESCRIPTION("ina423x driver");
MODULE_LICENSE("GPL");